/* Copyright (C) 2000 Damir Zucic */

/*=============================================================================

				motion_notify.c

Purpose:
	Handle MotionNotify events.

Input:
	(1) Pointer to MolComplexS structure, with macromol. complexes.
	(2) Number of macromolecular complexes.
	(3) Pointer to RuntimeS structure, with some runtime data.
	(4) Pointer to ConfigS structure, with configuration data.
	(5) Pointer to GUIS structure, with GUI data.
	(6) Pointer to NearestAtomS structure.
	(7) The number of pixels in the main window free area.
	(8) Pointer to refreshI.
	(9) Pointer to XMotionEvent structure.

Output:
	(1) Some data related to the atom under  the pointer written to
	    output window.
	(2) Sequence neighborhood  of the residue under the pointer may
	    be written to the main window (if requested).
	(3) Return value.

Return value:
	(1) Positive on success.
	(2) Zero if event is ignored.
	(3) Negative on failure.

========includes:============================================================*/

#include <stdio.h>

#include <string.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>

#include "defines.h"
#include "typedefs.h"

/*======function prototypes:=================================================*/

int		RamaMouse_ (MolComplexS *, int, ConfigS *, GUIS *,
			    NearestAtomS *, size_t, unsigned int,
			    XMotionEvent *);
int		PlotMouse_ (RuntimeS *, GUIS *,
			    NearestAtomS *, size_t, unsigned int,
			    XMotionEvent *);
int		ComparisonMouse_ (RuntimeS *, GUIS *,
				  NearestAtomS *, size_t,
				  XMotionEvent *);

/*======handle MotionNotify events:==========================================*/

int MotionNotify_ (MolComplexS *mol_complexSP, int mol_complexesN,
		   RuntimeS *runtimeSP, ConfigS *configSP, GUIS *guiSP,
		   NearestAtomS *nearest_atomSP, size_t pixelsN,
		   unsigned int refreshI,
		   XMotionEvent *motion_eventSP)
{
size_t			pixelI;
NearestAtomS		*pixelSP;
MolComplexS		*curr_mol_complexSP;
AtomS			*curr_atomSP;
RawAtomS		*raw_atomSP;
static char		stringA[STRINGSIZE];
int			text_line_height;
int			screen_x0, screen_y0;
float			tf;
double			x, y, z;
ResidueS		*curr_residueSP;
double			phi, psi;
int			central_residueI, first_residueI, last_residueI;
int			space_width, space_half_width;
int			residueI;
AtomS			*first_atomSP;
char			*residue_nameP;
char			residue_nameA[RESNAMESIZE];
int			residue_name_length, string_length;
int			text_width;
int			line_x0 = 0, line_x1 = 0;
int			width, height;

/* If main window contains Ramachandran plot: */
if (guiSP->main_window_modeI == 1)
	{
	return RamaMouse_ (mol_complexSP, mol_complexesN,
			   configSP, guiSP,
			   nearest_atomSP, pixelsN,
			   refreshI,
			   motion_eventSP);
	}

/* If main window contains hydrophobicity plots: */
else if (guiSP->main_window_modeI == 4)
	{
	return PlotMouse_ (runtimeSP, guiSP,
			   nearest_atomSP, pixelsN, refreshI,
			   motion_eventSP);
	}

/* If main window contains sequence comparison: */
else if (guiSP->main_window_modeI == 5)
	{
	return ComparisonMouse_ (runtimeSP, guiSP,
				 nearest_atomSP, pixelsN,
				 motion_eventSP);
	}

/* If the event came from child window, return zero: */
if (motion_eventSP->subwindow != None) return 0;

/* Pixel index: */
pixelI = guiSP->main_win_free_area_width * motion_eventSP->y +
	 motion_eventSP->x;

/* Check the pixel index: */
if (pixelI >= pixelsN) return -1;

/* Pointer to NearestAtomS structure assigned to this pixel: */
pixelSP = nearest_atomSP + pixelI;

/* Check the refreshI associated with this pixel: */
if (pixelSP->last_refreshI != refreshI) return 0;

/* Check the style index associated with this pixel; */
/* if only plane  was drawn to this pixel,  there is */
/* no atomic information  associated with the pixel: */
if (pixelSP->styleI == PLANE_STYLE) return 0;

/* Refresh the output window: */
XSetForeground (guiSP->displaySP, guiSP->theGCA[0],
		guiSP->output_winS.bg_colorID);
XFillRectangle(guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
	       0, 0, guiSP->output_winS.width, guiSP->output_winS.height);

/* Pointer to the current macromolecular complex: */
curr_mol_complexSP = mol_complexSP + pixelSP->mol_complexI;

/* Pointer to the current atom: */
curr_atomSP = curr_mol_complexSP->atomSP + pixelSP->atomI;

/* Pointer to raw atomic data: */
raw_atomSP = &curr_atomSP->raw_atomS;

/* Pointer to the current residue: */
curr_residueSP = curr_mol_complexSP->residueSP + curr_atomSP->residue_arrayI;

/* Prepare the text foreground color: */
XSetForeground (guiSP->displaySP, guiSP->theGCA[0],
		guiSP->output_winS.fg_colorID);

/* Text line height: */
text_line_height = guiSP->output_winS.text_line_height;

/* The initial drawing position: */
screen_x0 = TEXTMARGIN;
screen_y0 = guiSP->output_winS.font_height + 5;

/* Draw the unique PDB identifier: */
sprintf (stringA, "(%d) %s",
	 curr_mol_complexSP->mol_complexID,
	 curr_mol_complexSP->unique_PDB_codeA);
stringA[STRINGSIZE - 1] = '\0';
XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
	     screen_x0, screen_y0, stringA, strlen (stringA));

/* Draw residue data: */
screen_y0 += text_line_height;
sprintf (stringA, "%s %d %c",
	 raw_atomSP->residue_nameA,
	 raw_atomSP->residue_sequenceI,
	 raw_atomSP->chainID);
stringA[STRINGSIZE - 1] = '\0';
XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
	     screen_x0, screen_y0, stringA, strlen (stringA));

/* Draw atom name,  alternate location  indicator, */
/* atom serial number and chemical element symbol: */
screen_y0 += text_line_height;
sprintf (stringA, "%s%c %d %s",
	 raw_atomSP->pure_atom_nameA,
	 raw_atomSP->alt_location,
	 raw_atomSP->serialI,
	 raw_atomSP->chemical_symbolA);
stringA[STRINGSIZE - 1] = '\0';
XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
	     screen_x0, screen_y0, stringA, strlen (stringA));

/* Temperature factor: */
screen_y0 += text_line_height;
tf = raw_atomSP->temperature_factor;
if (tf < 99.99) sprintf (stringA, "t.f.: %5.2f", tf);
else sprintf (stringA, "t.f. :%.2f", tf);
stringA[STRINGSIZE - 1] = '\0';
XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
	     screen_x0, screen_y0, stringA, strlen (stringA));

/* The peptide group conformation: */
screen_y0 += text_line_height;
if (curr_residueSP->cis_transF == 1)
	{
	strcpy (stringA, "trans");
	XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
		     screen_x0, screen_y0, stringA, strlen (stringA));
	}
else if (curr_residueSP->cis_transF == 2)
	{
	strcpy (stringA, "cis");
	XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
		     screen_x0, screen_y0, stringA, strlen (stringA));
	}
else
	{
	strcpy (stringA, "bad/undef.");
	XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
		     screen_x0, screen_y0, stringA, strlen (stringA));
	}

/* The angle phi (see dihedral_angles.c for definition): */
screen_y0 += text_line_height;
phi = curr_residueSP->phi;
if (phi == BADDIHEDANGLE)
	{
	strcpy (stringA, "phi: missing");
	XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
		     screen_x0, screen_y0, stringA, strlen (stringA));
	}
else
	{
	phi *= RAD_TO_DEG;
	sprintf (stringA, "phi:%7.2f", phi);
	stringA[STRINGSIZE - 1] = '\0';
	XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
		     screen_x0, screen_y0, stringA, strlen (stringA));
	}

/* The angle psi (see dihedral_angles.c for definition): */
screen_y0 += text_line_height;
psi = curr_residueSP->psi;
if (psi == BADDIHEDANGLE)
	{
	strcpy (stringA, "psi: missing");
	XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
		     screen_x0, screen_y0, stringA, strlen (stringA));
	}
else
	{
	psi *= RAD_TO_DEG;
	sprintf (stringA, "psi:%7.2f", psi);
	stringA[STRINGSIZE - 1] = '\0';
	XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
		     screen_x0, screen_y0, stringA, strlen (stringA));
	}

/* Hydrophobicity: */
screen_y0 += text_line_height;
sprintf (stringA, "phob:%6.2f", raw_atomSP->hydrophobicity);
stringA[STRINGSIZE - 1] = '\0';
XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
	     screen_x0, screen_y0, stringA, strlen (stringA));

/* Selection status: */
screen_y0 += text_line_height;
if (curr_atomSP->selectedF) strcpy (stringA, "selected");
else strcpy (stringA, "not selected");
XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
	     screen_x0, screen_y0, stringA, strlen (stringA));

/* x coordinate: */
screen_y0 += text_line_height;
x = raw_atomSP->x[0];
if (x < 9999.999) sprintf (stringA, "x:  %8.3f", x);
else sprintf (stringA, "x:  %.3f", x);
stringA[STRINGSIZE - 1] = '\0';
XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
	     screen_x0, screen_y0, stringA, strlen (stringA));

/* y coordinate: */
screen_y0 += text_line_height;
y = raw_atomSP->y;
if (y < 9999.999) sprintf (stringA, "y:  %8.3f", y);
else sprintf (stringA, "y:  %.3f", y);
stringA[STRINGSIZE - 1] = '\0';
XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
	     screen_x0, screen_y0, stringA, strlen (stringA));

/* z coordinate: */
screen_y0 += text_line_height;
z = raw_atomSP->z[0];
if (z < 9999.999) sprintf (stringA, "z:  %8.3f", z);
else sprintf (stringA, "z:  %.3f", z);
stringA[STRINGSIZE - 1] = '\0';
XDrawString (guiSP->displaySP, guiSP->output_winS.ID, guiSP->theGCA[0],
	     screen_x0, screen_y0, stringA, strlen (stringA));

/* Write  the sequence neighborhood  of the */
/* residue under the pointer, if requested: */
if (configSP->show_sequence_neighborhoodF)
	{
	/* The central residue index: */
	central_residueI = curr_atomSP->residue_arrayI;

	/* Space half-width: */
	space_width = XTextWidth (guiSP->main_winS.fontSP, " ", 1);
	space_half_width = space_width / 2;

	/* Indices of the first and the last residue in the neighbourhood: */
	first_residueI = central_residueI - NEIGH_HALF_WIDTH;
	if (first_residueI < 0) first_residueI = 0;
	last_residueI = central_residueI + NEIGH_HALF_WIDTH;
	if (last_residueI > curr_mol_complexSP->residuesN)
		{
		last_residueI = curr_mol_complexSP->residuesN;
		}

	/* Prepare the string with the left part of the neighborhood: */
	stringA[0] = '\0';
	residue_name_length = RESNAMESIZE - 1;
	strcat (stringA, " ");
	for (residueI = first_residueI; residueI <= last_residueI; residueI++)
		{
		/* The string width, without the central residue: */
		if (residueI == central_residueI)
			{
			string_length = strlen (stringA);
			line_x0 = XTextWidth (guiSP->main_winS.fontSP,
					      stringA, string_length) -
					      space_half_width;
			}

		/* Pointer to current residue: */
		curr_residueSP = curr_mol_complexSP->residueSP + residueI;

		/* Pointer to the first atom of the current residue: */
		first_atomSP = curr_mol_complexSP->atomSP +
			       curr_residueSP->residue_startI;;

		/* Pointer to residue name: */
		residue_nameP = first_atomSP->raw_atomS.pure_residue_nameA;

		/* Add the residue name to the output string: */
		strncpy (residue_nameA, residue_nameP, residue_name_length);
		residue_nameA[residue_name_length] = '\0';
		strcat (stringA, residue_nameA);
		strcat (stringA, " ");

		/* The string width, including the central residue: */
		if (residueI == central_residueI)
			{
			string_length = strlen (stringA);
			line_x1 = XTextWidth (guiSP->main_winS.fontSP,
					      stringA, string_length) -
					      space_half_width;
			}
		}

	/* String length: */
	string_length = strlen (stringA);

	/* Text width: */
	text_width = XTextWidth (guiSP->main_winS.fontSP,
				 stringA, string_length);

	/* Prepare text background color: */
	XSetForeground (guiSP->displaySP, guiSP->theGCA[0],
			guiSP->main_winS.bg_colorID);

	/* Refresh the string background: */
	screen_x0 = 1;
	screen_y0 = 1;
	width = guiSP->control_winS.x0 - guiSP->main_winS.border_width - 2;
	if (width < 0) width = 0;
	height = guiSP->main_winS.font_height + 4;
	XFillRectangle(guiSP->displaySP, guiSP->main_winS.ID, guiSP->theGCA[0],
		       screen_x0, screen_y0,
		       (unsigned int) width, (unsigned int) height);

        /* Prepare text color: */
	XSetForeground (guiSP->displaySP, guiSP->theGCA[0],
			guiSP->main_winS.fg_colorID);

	/* Draw the bounding rectangle: */
	XDrawRectangle (guiSP->displaySP, guiSP->main_winS.ID,
			guiSP->theGCA[0],
			screen_x0, screen_y0,
			(unsigned int) width, (unsigned int) height);

	/* Draw the string which contains sequence neighborhood: */
	screen_x0 = guiSP->control_winS.x0 / 2 - text_width / 2;
	screen_y0 = guiSP->main_winS.font_height + 1;
	XDrawString (guiSP->displaySP, guiSP->main_winS.ID, guiSP->theGCA[0],
		     screen_x0, screen_y0, stringA, string_length);

	/* Draw rectangle bounding the central residue: */
	line_x0 += guiSP->control_winS.x0 / 2 - text_width / 2;
	line_x1 += guiSP->control_winS.x0 / 2 - text_width / 2;
	width = line_x1 - line_x0 + 1;
	if (width < 0) width = 0;
	height = guiSP->main_winS.font_height - 1;
	if (height < 0) height = 0;
	screen_y0 = height;
	XDrawRectangle (guiSP->displaySP, guiSP->main_winS.ID,
			guiSP->theGCA[0],
			line_x0, 4,
			(unsigned int) width, (unsigned int) height);
	}

/* Return positive value: */
return 1;
}

/*===========================================================================*/


