/* sqlqueryenv.c
 *
 * Copyright (C) 1999 - 2001 Vivien Malerba
 * Copyright (C) 2001 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include "sqlqueryenv.h"
#include "choicecombo.h"

static void sql_query_env_class_init (SqlQueryEnvClass * class);
static void sql_query_env_init (SqlQueryEnv * qe);
static void sql_query_env_destroy (GtkObject * object);

/* TODO FM: check usage of JOIN_CROSS; related to not used links in query */

/*
 *
 * Main functions for the object
 *
 */

/* get a pointer to the parents to be able to call their destructor */
static GtkObject *parent_class = NULL;

enum
{
	QUERY_CHANGED,
	DESCR_CHANGED,
	TYPE_CHANGED,
	LAST_SIGNAL
};

static gint sql_query_env_signals[LAST_SIGNAL] = { 0, 0 };

guint
sql_query_env_get_type (void)
{
	static guint f_type = 0;

	if (!f_type) {
		GtkTypeInfo f_info = {
			"Sql_Query_Env",
			sizeof (SqlQueryEnv),
			sizeof (SqlQueryEnvClass),
			(GtkClassInitFunc) sql_query_env_class_init,
			(GtkObjectInitFunc) sql_query_env_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};

		f_type = gtk_type_unique (gtk_object_get_type (), &f_info);
	}

	return f_type;
}

static void
sql_query_env_class_init (SqlQueryEnvClass * class)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass *) class;

	sql_query_env_signals[QUERY_CHANGED] =
		gtk_signal_new ("query_changed",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryEnvClass,
						   query_changed),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	sql_query_env_signals[DESCR_CHANGED] =
		gtk_signal_new ("descr_changed",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryEnvClass,
						   descr_changed),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	sql_query_env_signals[TYPE_CHANGED] =
		gtk_signal_new ("type_changed",
				GTK_RUN_FIRST,
				object_class->type,
				GTK_SIGNAL_OFFSET (SqlQueryEnvClass,
						   type_changed),
				gtk_signal_default_marshaller, GTK_TYPE_NONE,
				0);

	gtk_object_class_add_signals (object_class, sql_query_env_signals,
				      LAST_SIGNAL);
	class->query_changed = NULL;
	class->descr_changed = NULL;
	class->type_changed = NULL;

	object_class->destroy = sql_query_env_destroy;
	parent_class = gtk_type_class (gtk_object_get_type ());
}

static void
sql_query_env_init (SqlQueryEnv * qe)
{
	qe->q = NULL;
	qe->modif_table = NULL;
	qe->name = NULL;
	qe->descr = NULL;
	qe->actions =
		QUERY_ACTION_FIRST | QUERY_ACTION_LAST | QUERY_ACTION_PREV |
		QUERY_ACTION_NEXT | QUERY_ACTION_REFRESH | QUERY_ACTION_EDIT |
		QUERY_ACTION_VIEWALL;
	qe->form = NULL;
	qe->form_is_default = FALSE;
	qe->edit_dlg = NULL;
	qe->execs_list = NULL;
	qe->no_insert = TRUE;
}


static void sql_query_env_elts_changed_cb (SqlQuery * q, QueryElement * elt,
					   SqlQueryEnv * qe);
GtkObject *
sql_query_env_new (SqlQuery * q)
{
	GtkObject *obj;
	SqlQueryEnv *qe;

	g_return_val_if_fail (q != NULL, NULL);
	g_return_val_if_fail (IS_SQL_QUERY (q), NULL);

	obj = gtk_type_new (sql_query_env_get_type ());
	qe = SQL_QUERY_ENV (obj);
	qe->q = q;
	q->envs = g_slist_append (q->envs, qe);

	/* connection to signals from SqlDb */
	gtk_signal_connect_while_alive (GTK_OBJECT (q), "elt_created",
					GTK_SIGNAL_FUNC
					(sql_query_env_elts_changed_cb), qe,
					obj);
	gtk_signal_connect_while_alive (GTK_OBJECT (q), "elt_dropped",
					GTK_SIGNAL_FUNC
					(sql_query_env_elts_changed_cb), qe,
					obj);
	return obj;
}

static void update_no_insert_info (SqlQueryEnv * qe);
void
sql_query_env_set_modif_table (SqlQueryEnv * env, SqlMemTable * table)
{
	env->modif_table = table;
	update_no_insert_info (env);
}

enum _ModifTableLevel
{
	LINKS_TO_TABLE = 1 << 3,
	KEY_PRESENT = 1 << 2,
	ALL_FIELDS = 1 << 1,
	ALL_FIELDS_WITH_DEFAULT = 1 << 0
};

struct _ModifTable
{
	SqlMemTable *table;
	enum _ModifTableLevel level;
	gint nbstars;
};


/* returns a list of struct _ModifTable to be freed by the caller */
static GSList *list_potential_modif_tables (SqlQueryEnv * env);
static void
update_no_insert_info (SqlQueryEnv * qe)
{
	GSList *list, *hold;
	gboolean found = FALSE;

	if (!qe->modif_table) {
		qe->no_insert = TRUE;
		return;
	}

	list = list_potential_modif_tables (qe);
	while (list) {
		struct _ModifTable *mt = (struct _ModifTable *) (list->data);
		if (mt->table == qe->modif_table) {
			qe->no_insert =
				(mt->
				 level & ALL_FIELDS_WITH_DEFAULT) ? FALSE :
				TRUE;
			found = TRUE;
		}
		g_free (mt);
		hold = list;
		list = g_slist_remove_link (list, list);
		g_slist_free_1 (hold);
	}
	if (!found)
		qe->no_insert = TRUE;
}

static GtkWidget *get_tables_optionmenu_menu (SqlQueryEnv * env);
static void
sql_query_env_elts_changed_cb (SqlQuery * q, QueryElement * elt,
			       SqlQueryEnv * qe)
{
	if (qe->edit_dlg) {
		GtkWidget *optionmenu, *optionmenu_menu;

		optionmenu =
			gtk_object_get_data (GTK_OBJECT (qe->edit_dlg),
					     "omenu");
		optionmenu_menu = get_tables_optionmenu_menu (qe);
		gtk_object_set_data (GTK_OBJECT (qe->edit_dlg), "omenu_menu",
				     optionmenu_menu);
		gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu),
					  optionmenu_menu);
		gtk_widget_show (optionmenu_menu);
	}
	else {
		GSList *hold, *list = list_potential_modif_tables (qe);
		gboolean found = FALSE;
		while (list) {
			struct _ModifTable *mt =
				(struct _ModifTable *) (list->data);
			if (mt->table == qe->modif_table)
				found = TRUE;
			g_free (mt);
			hold = list;
			list = g_slist_remove_link (list, list);
			g_slist_free_1 (hold);
		}
		if (!found)
			qe->modif_table = NULL;
		update_no_insert_info (qe);
	}
}

static void
sql_query_env_destroy (GtkObject * object)
{
	SqlQueryEnv *qe;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SQL_QUERY_ENV (object));

	qe = SQL_QUERY_ENV (object);

	/* we destroy all the SqlQueryExec objects */
	while (qe->execs_list) {
		/* the list is updated automatically because objects have been 
		   registered with sql_query_env_register_exec_obj() */
		gtk_object_destroy (GTK_OBJECT (qe->execs_list->data));
	}

	if (qe->edit_dlg)
		gtk_object_destroy (GTK_OBJECT (qe->edit_dlg));

	qe->q->envs = g_slist_remove (qe->q->envs, qe);
	if (qe->descr)
		g_free (qe->descr);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

void
sql_query_env_free (SqlQueryEnv * env)
{
	sql_query_env_destroy (GTK_OBJECT (env));
}

static void obj_destroyed_cb (GtkObject * obj, SqlQueryEnv * env);
void
sql_query_env_register_exec_obj (SqlQueryEnv * env, GtkObject * obj)
{
	env->execs_list = g_slist_append (env->execs_list, obj);
	gtk_signal_connect (obj, "destroy",
			    GTK_SIGNAL_FUNC (obj_destroyed_cb), env);
}

static void
obj_destroyed_cb (GtkObject * obj, SqlQueryEnv * env)
{
	env->execs_list = g_slist_remove (env->execs_list, obj);
}


void
sql_query_env_set_name (SqlQueryEnv * env, gchar * name)
{
	g_return_if_fail (env != NULL);
	if (env->name)
		g_free (env->name);

	if (name)
		env->name = g_strdup (name);
	else
		env->name = NULL;
}


void
sql_query_env_set_descr (SqlQueryEnv * env, gchar * descr)
{
	g_return_if_fail (env != NULL);
	if (env->descr)
		g_free (env->descr);

	if (descr)
		env->descr = g_strdup (descr);
	else
		env->descr = NULL;
}

/*
 * Properties edition in a dialog
 */

static GtkWidget *get_action_cb (SqlQueryEnv * env, SqlQueryActions action);
static void edit_dlg_clicked_cb (GnomeDialog * dialog, gint button_number,
				 SqlQueryEnv * env);
static void edit_dlg_destroy_cb (GnomeDialog * dialog, SqlQueryEnv * env);
static void edit_dlg_name_changed_cb (GtkEditable * editable,
				      SqlQueryEnv * env);
static void edit_dlg_descr_changed_cb (GtkEditable * editable,
				       SqlQueryEnv * env);
static void edit_dlg_table_mitem_activate_cb (GtkMenuItem * menu_item,
					      SqlQueryEnv * env);
static void edit_dlg_new_table_cb (SqlDb * db, SqlMemTable * new_table,
				   SqlQueryEnv * env);
static void edit_dlg_del_table_cb (SqlDb * db, SqlMemTable * new_table,
				   SqlQueryEnv * env);
static void edit_dlg_type_mitem_activate_cb (GtkMenuItem * menu_item,
					     SqlQueryEnv * env);
static void edit_layout (GtkButton * wid, SqlQueryEnv * qe);
GtkWidget *
sql_query_env_show_props_dialog (SqlQueryEnv * env)
{
	GtkWidget *dlg;

	g_return_val_if_fail (env != NULL, NULL);
	g_return_val_if_fail (IS_SQL_QUERY_ENV (env), NULL);

	if (env->edit_dlg)
		gdk_window_raise (env->edit_dlg->window);
	else {
		gchar *str;
		GtkWidget *frame, *vbox, *label, *hbb, *button, *table, *cb,
			*entry;
		GtkWidget *optionmenu, *optionmenu_menu, *mitem;
		GSList *tables;
		gint i;

		if (env->descr)
			str = g_strdup_printf (_("Properties for '%s'"),
					       env->descr);
		else
			str = g_strdup (_("Form Properties"));
		dlg = gnome_dialog_new (str, GNOME_STOCK_BUTTON_CLOSE, NULL);
		g_free (str);
		env->edit_dlg = dlg;

		/* dialog termination */
		gtk_signal_connect (GTK_OBJECT (dlg), "clicked",
				    GTK_SIGNAL_FUNC (edit_dlg_clicked_cb),
				    env);
		gtk_signal_connect (GTK_OBJECT (dlg), "destroy",
				    GTK_SIGNAL_FUNC (edit_dlg_destroy_cb),
				    env);

		/* signals from other objects */
		if (!env->q)
			g_error ("sql_query_env_show_props_dialog() env->q is NULL");
		gtk_signal_connect_while_alive (GTK_OBJECT (env->q->conf->db),
						"table_created_filled",
						GTK_SIGNAL_FUNC
						(edit_dlg_new_table_cb), env,
						GTK_OBJECT (dlg));
		gtk_signal_connect_while_alive (GTK_OBJECT (env->q->conf->db),
						"table_dropped",
						GTK_SIGNAL_FUNC
						(edit_dlg_del_table_cb), env,
						GTK_OBJECT (dlg));

		/* Basic properties frame */
		frame = gtk_frame_new (_("Basic properties"));
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox), frame,
				    TRUE, TRUE, 0);
		gtk_widget_show (frame);

		table = gtk_table_new (4, 2, FALSE);
		gtk_container_set_border_width (GTK_CONTAINER (table),
						GNOME_PAD / 2.);
		gtk_table_set_row_spacings (GTK_TABLE (table),
					    GNOME_PAD / 4.);
		gtk_table_set_col_spacings (GTK_TABLE (table),
					    GNOME_PAD / 4.);
		gtk_container_add (GTK_CONTAINER (frame), table);
		gtk_widget_show (table);

		label = gtk_label_new (_("Name:"));
		gtk_table_attach (GTK_TABLE (table), label,
				  0, 1, 0, 1, 0, 0, 0, 0);
		gtk_widget_show (label);

		entry = gtk_entry_new ();
		if (env->name)
			gtk_entry_set_text (GTK_ENTRY (entry), env->name);
		gtk_table_attach_defaults (GTK_TABLE (table), entry,
					   1, 2, 0, 1);
		gtk_widget_show (entry);
		gtk_signal_connect (GTK_OBJECT (entry), "changed",
				    GTK_SIGNAL_FUNC
				    (edit_dlg_name_changed_cb), env);

		label = gtk_label_new (_("Description:"));
		gtk_table_attach (GTK_TABLE (table), label,
				  0, 1, 1, 2, 0, 0, 0, 0);
		gtk_widget_show (label);

		entry = gtk_entry_new ();
		if (env->descr)
			gtk_entry_set_text (GTK_ENTRY (entry), env->descr);
		gtk_table_attach_defaults (GTK_TABLE (table), entry,
					   1, 2, 1, 2);
		gtk_widget_show (entry);
		gtk_signal_connect (GTK_OBJECT (entry), "changed",
				    GTK_SIGNAL_FUNC
				    (edit_dlg_descr_changed_cb), env);

		label = gtk_label_new (_("Modified table:"));
		gtk_table_attach (GTK_TABLE (table), label,
				  0, 1, 2, 3, 0, 0, 0, 0);
		gtk_widget_show (label);

		optionmenu = gtk_option_menu_new ();
		gtk_object_set_data (GTK_OBJECT (dlg), "omenu", optionmenu);
		optionmenu_menu = get_tables_optionmenu_menu (env);
		gtk_object_set_data (GTK_OBJECT (dlg), "omenu_menu",
				     optionmenu_menu);
		gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu),
					  optionmenu_menu);
		gtk_widget_show (optionmenu_menu);
		gtk_table_attach_defaults (GTK_TABLE (table), optionmenu,
					   1, 2, 2, 3);
		gtk_widget_show (optionmenu);


		label = gtk_label_new (_("View type:"));
		gtk_table_attach (GTK_TABLE (table), label,
				  0, 1, 3, 4, 0, 0, 0, 0);
		gtk_widget_show (label);

		optionmenu = gtk_option_menu_new ();
		optionmenu_menu = gtk_menu_new ();
		mitem = gtk_menu_item_new_with_label (_("Form"));
		gtk_menu_append (GTK_MENU (optionmenu_menu), mitem);
		gtk_widget_show (mitem);
		gtk_object_set_data (GTK_OBJECT (mitem), "data",
				     GINT_TO_POINTER (1));
		gtk_signal_connect (GTK_OBJECT (mitem), "activate",
				    GTK_SIGNAL_FUNC
				    (edit_dlg_type_mitem_activate_cb), env);

		mitem = gtk_menu_item_new_with_label (_("Grid"));
		gtk_menu_append (GTK_MENU (optionmenu_menu), mitem);
		gtk_widget_show (mitem);
		gtk_signal_connect (GTK_OBJECT (mitem), "activate",
				    GTK_SIGNAL_FUNC
				    (edit_dlg_type_mitem_activate_cb), env);

		if (!env->form_is_default)
			gtk_menu_set_active (GTK_MENU (optionmenu_menu), 1);

		gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu),
					  optionmenu_menu);
		gtk_widget_show (optionmenu_menu);
		gtk_table_attach_defaults (GTK_TABLE (table), optionmenu,
					   1, 2, 3, 4);
		gtk_widget_show (optionmenu);

		/* Actions frame */
		frame = gtk_frame_new (_("Action buttons"));
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox), frame,
				    TRUE, TRUE, 0);
		gtk_widget_show (frame);

		vbox = gtk_vbox_new (FALSE, GNOME_PAD / 2.);
		gtk_container_add (GTK_CONTAINER (frame), vbox);
		gtk_widget_show (vbox);

		label = gtk_label_new (_
				       ("Action buttons for the grid and form vues:"));
		gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, FALSE, 0);
		gtk_widget_show (label);

		table = gtk_table_new (3, 4, FALSE);
		gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, FALSE, 0);
		gtk_widget_show (table);

		cb = get_action_cb (env, QUERY_ACTION_FIRST);
		gtk_table_attach_defaults (GTK_TABLE (table), cb, 0, 1, 0, 1);
		gtk_widget_show (cb);

		cb = get_action_cb (env, QUERY_ACTION_LAST);
		gtk_table_attach_defaults (GTK_TABLE (table), cb, 0, 1, 1, 2);
		gtk_widget_show (cb);

		cb = get_action_cb (env, QUERY_ACTION_COMMIT);
		gtk_table_attach_defaults (GTK_TABLE (table), cb, 0, 1, 2, 3);
		gtk_widget_show (cb);

		cb = get_action_cb (env, QUERY_ACTION_PREV);
		gtk_table_attach_defaults (GTK_TABLE (table), cb, 1, 2, 0, 1);
		gtk_widget_show (cb);

		cb = get_action_cb (env, QUERY_ACTION_NEXT);
		gtk_table_attach_defaults (GTK_TABLE (table), cb, 1, 2, 1, 2);
		gtk_widget_show (cb);

		cb = get_action_cb (env, QUERY_ACTION_CHOOSE);
		gtk_table_attach_defaults (GTK_TABLE (table), cb, 1, 2, 2, 3);
		gtk_widget_show (cb);

		cb = get_action_cb (env, QUERY_ACTION_INSERT);
		gtk_table_attach_defaults (GTK_TABLE (table), cb, 2, 3, 0, 1);
		gtk_widget_show (cb);

		cb = get_action_cb (env, QUERY_ACTION_EDIT);
		gtk_table_attach_defaults (GTK_TABLE (table), cb, 2, 3, 1, 2);
		gtk_widget_show (cb);

		cb = get_action_cb (env, QUERY_ACTION_DELETE);
		gtk_table_attach_defaults (GTK_TABLE (table), cb, 2, 3, 2, 3);
		gtk_widget_show (cb);

		cb = get_action_cb (env, QUERY_ACTION_REFRESH);
		gtk_table_attach_defaults (GTK_TABLE (table), cb, 0, 1, 3, 4);
		gtk_widget_show (cb);

		cb = get_action_cb (env, QUERY_ACTION_VIEWALL);
		gtk_table_attach_defaults (GTK_TABLE (table), cb, 1, 2, 3, 4);
		gtk_widget_show (cb);

		/* Form layout frame */
		frame = gtk_frame_new (_("Form Layout"));
		gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dlg)->vbox), frame,
				    TRUE, TRUE, 0);
		gtk_widget_show (frame);

		vbox = gtk_vbox_new (FALSE, GNOME_PAD / 2.);
		gtk_container_add (GTK_CONTAINER (frame), vbox);
		gtk_widget_show (vbox);

		if (env->form)
			label = gtk_label_new (_
					       ("A customized form layout is now used"));
		else
			label = gtk_label_new (_
					       ("The default form layout is now used"));
		gtk_object_set_data (GTK_OBJECT (dlg), "layout label", label);
		gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE,
				    GNOME_PAD / 2.);
		gtk_widget_show (label);

		hbb = gtk_hbutton_box_new ();
		gtk_box_pack_start (GTK_BOX (vbox), hbb, FALSE, TRUE,
				    GNOME_PAD / 2.);
		gtk_widget_show (hbb);

		button = gtk_button_new_with_label (_("Edit Layout"));
		gtk_signal_connect (GTK_OBJECT (button), "clicked",
				    GTK_SIGNAL_FUNC (edit_layout), env);
		gtk_container_add (GTK_CONTAINER (hbb), button);
		gtk_widget_show (button);

		gtk_widget_show (dlg);
	}
}

static void edit_dlg_cb_toggled_cb (GtkToggleButton * toggle_button,
				    SqlQueryEnv * env);
static GtkWidget *
get_action_cb (SqlQueryEnv * env, SqlQueryActions action)
{
	GtkWidget *cb;
	gchar *str, *prop;

	switch (action) {
	case QUERY_ACTION_FIRST:
		str = _("First record");
		prop = "fr";
		break;
	case QUERY_ACTION_LAST:
		str = _("Last record");
		prop = "lr";
		break;
	case QUERY_ACTION_PREV:
		str = _("Previous record");
		prop = "pr";
		break;
	case QUERY_ACTION_NEXT:
		str = _("Next record");
		prop = "nr";
		break;
	case QUERY_ACTION_INSERT:
		str = _("New entry");
		prop = "ne";
		break;
	case QUERY_ACTION_COMMIT:
		str = _("Commit");
		prop = "co";
		break;
	case QUERY_ACTION_DELETE:
		str = _("Delete");
		prop = "de";
		break;
	case QUERY_ACTION_CHOOSE:
		str = _("Choose one");
		prop = "ch";
		break;
	case QUERY_ACTION_REFRESH:
		str = _("Refresh");
		prop = "re";
		break;
	case QUERY_ACTION_EDIT:
		str = _("Edit");
		prop = "ed";
		break;
	case QUERY_ACTION_VIEWALL:
		str = _("View all");
		prop = "va";
		break;
	}

	cb = gtk_check_button_new_with_label (str);
	gtk_object_set_data (GTK_OBJECT (cb), "act",
			     GUINT_TO_POINTER (action));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (cb),
				      (env->actions & action));
	g_return_val_if_fail (env->edit_dlg != NULL, cb);	/* just to output errors */
	gtk_object_set_data (GTK_OBJECT (env->edit_dlg), prop, cb);
	gtk_signal_connect (GTK_OBJECT (cb), "toggled",
			    GTK_SIGNAL_FUNC (edit_dlg_cb_toggled_cb), env);

	return cb;
}

static void
edit_dlg_cb_toggled_cb (GtkToggleButton * toggle_button, SqlQueryEnv * env)
{
	SqlQueryActions action;

	action = GPOINTER_TO_UINT (gtk_object_get_data
				   (GTK_OBJECT (toggle_button), "act"));
	if (env->actions & action)
		env->actions = env->actions & (~action);	/* remove the bit */
	else
		env->actions = env->actions | action;	/* set the bit */
}

static void
edit_dlg_clicked_cb (GnomeDialog * dialog, gint button_number,
		     SqlQueryEnv * env)
{
	gtk_object_destroy (GTK_OBJECT (dialog));
}

static void
edit_dlg_destroy_cb (GnomeDialog * dialog, SqlQueryEnv * env)
{
	env->edit_dlg = NULL;
}

static void
edit_dlg_name_changed_cb (GtkEditable * editable, SqlQueryEnv * env)
{
	if (env->name)
		g_free (env->name);
	env->name = g_strdup (gtk_entry_get_text (GTK_ENTRY (editable)));
#ifdef debug_signal
	g_print (">> 'DESCR_CHANGED' "
		 "from sqlqueryenv->edit_dlg_name_changed_cb\n");
#endif
	gtk_signal_emit (GTK_OBJECT (env),
			 sql_query_env_signals[DESCR_CHANGED], NULL);
#ifdef debug_signal
	g_print ("<< 'DESCR_CHANGED' "
		 "from sqlqueryenv->edit_dlg_name_changed_cb\n");
#endif

}

static void
edit_dlg_descr_changed_cb (GtkEditable * editable, SqlQueryEnv * env)
{
	if (env->descr)
		g_free (env->descr);
	env->descr = g_strdup (gtk_entry_get_text (GTK_ENTRY (editable)));
#ifdef debug_signal
	g_print (">> 'DESCR_CHANGED' "
		 "from sqlqueryenv->edit_dlg_descr_changed_cb\n");
#endif
	gtk_signal_emit (GTK_OBJECT (env),
			 sql_query_env_signals[DESCR_CHANGED], NULL);
#ifdef debug_signal
	g_print ("<< 'DESCR_CHANGED' "
		 "from sqlqueryenv->edit_dlg_descr_changed_cb\n");
#endif

}

/* returns the list of tables which can be modified for a given query 
   and sets the modif_table to a default value if not set, or set it to NULL
   if no table can be modified 
*/
static GtkWidget *
get_tables_optionmenu_menu (SqlQueryEnv * env)
{
	GtkWidget *optionmenu_menu, *mitem;
	GSList *tables, *hold;
	gint i;

	tables = list_potential_modif_tables (env);
	optionmenu_menu = gtk_menu_new ();

	i = 0;
	while (tables) {
		gchar *str, *stars;
		struct _ModifTable *mt;

		mt = (struct _ModifTable *) (tables->data);
		switch (mt->nbstars) {
		case 0:
			stars = "";
			break;
		case 1:
			stars = "*";
			break;
		case 2:
			stars = "**";
			break;
		case 3:
			stars = "***";
			break;
		case 4:
			stars = "****";
			break;
		}
		if (mt->table) {
			if (mt->nbstars)
				str = g_strdup_printf ("%s (%s)",
						       mt->table->name,
						       stars);
			else
				str = g_strdup (mt->table->name);
		}
		else
			str = g_strdup (_("NONE"));
		mitem = gtk_menu_item_new_with_label (str);
		g_free (str);
		gtk_signal_connect (GTK_OBJECT (mitem), "activate",
				    GTK_SIGNAL_FUNC
				    (edit_dlg_table_mitem_activate_cb), env);
		gtk_object_set_data (GTK_OBJECT (mitem), "table", mt->table);
		gtk_menu_append (GTK_MENU (optionmenu_menu), mitem);
		if (mt->table == env->modif_table)
			gtk_menu_set_active (GTK_MENU (optionmenu_menu), i);

		gtk_widget_show (mitem);
		i++;

		g_free (tables->data);
		hold = tables;
		tables = g_slist_remove_link (tables, tables);
		g_slist_free_1 (hold);
	}

	update_no_insert_info (env);
	return optionmenu_menu;
}

static GSList *
list_potential_modif_tables (SqlQueryEnv * env)
{
	GSList *list = NULL, *tables;
	struct _ModifTable *mt;
	SqlMemTable *table;

	/* 1st step : list without order */
	tables = env->q->conf->db->tables;
	while (tables) {
		table = SQL_MEM_TABLE (tables->data);
		if (!table->is_view) {
			gboolean listed = FALSE;

			if (sql_query_is_table_concerned (env->q, table)) {
				mt = g_new0 (struct _ModifTable, 1);
				mt->table = table;
				list = g_slist_append (list, mt);
				listed = TRUE;
			}

			if (!listed) {
				GSList *links;
				gint nblinks = 0;
				gboolean toit = TRUE;
				links = env->q->depend_list;
				while (links && toit) {
					QueryLink *ql =
						(QueryLink *) (links->data);
					if ((ql->join_type != JOIN_CROSS) &&
					    (sql_db_find_table_from_field (env->q->conf->db,
									   ql->link->to) == table))
						nblinks += 1;
					if ((ql->join_type != JOIN_CROSS) &&
					    (sql_db_find_table_from_field (env->q->conf->db,
									   ql->link->from) == table))
						toit = FALSE;
					links = g_slist_next (links);
				}

				if (toit && (nblinks >= 2)) {
					mt = g_new0 (struct _ModifTable, 1);
					mt->table = table;
					list = g_slist_append (list, mt);
				}
			}
		}
		tables = g_slist_next (tables);
	}

	/* 2nd step: associate priorities level to selected tables */
	tables = list;
	while (tables) {
		gboolean ltest, ltest2;
		GSList *tlist;
		mt = (struct _ModifTable *) (tables->data);

		/* KEY_PRESENT: is there a key field printed in the query */
		ltest = FALSE;
		tlist = mt->table->fields;
		while (!ltest && tlist) {
			if ((SQL_MEM_FIELD (tlist->data)->is_key) &&
			    sql_query_is_field_printed (env->q,
							SQL_MEM_FIELD (tlist->
								       data)))
				ltest = TRUE;
			tlist = g_slist_next (tlist);
		}
		mt->level = ltest ? KEY_PRESENT : 0;

		/* LINKS_TO_TABLE: are all the links related to the table pointing 
		   TO a field in the table */
		ltest = TRUE;
		tlist = env->q->depend_list;
		while (ltest && tlist) {
			QueryLink *ql = (QueryLink *) (tlist->data);
			if ((ql->join_type != JOIN_CROSS) &&
			    (sql_db_find_table_from_field (env->q->conf->db, ql->link->from) == mt->table))
				ltest = FALSE;
			tlist = g_slist_next (tlist);
		}
		mt->level = ltest ? mt->level | LINKS_TO_TABLE : mt->level;

		/* ALL_FIELDS_WITH_DEFAULT and ALL_FIELDS: are there respectively ALL the fields
		   with a NOT NULL value, and ALL the fields with a NOT NULL value minus the ones
		   with a default value, present in the printed part of a query? */
		ltest = TRUE;
		ltest2 = TRUE;	/* WITH_DEFAULT */
		tlist = mt->table->fields;
		while ((ltest || ltest2) && tlist) {
			if (!(SQL_MEM_FIELD (tlist->data)->null_allowed) &&
			    (sql_query_is_field_printed
			     (env->q, SQL_MEM_FIELD (tlist->data)) == -1)) {
				/* do we have a link we use TO that field */
				gboolean found = FALSE;
				GSList *llist = env->q->depend_list;
				while (llist && !found) {
					QueryLink *ql =
						(QueryLink *) (llist->data);
					if ((ql->join_type != JOIN_CROSS) && 
					    (ql->link->to == SQL_MEM_FIELD (tlist->data)))
						found = TRUE;
					llist = g_slist_next (llist);
				}
				if (!found) {
					ltest = FALSE;
					if (!SQL_MEM_FIELD (tlist->data)->
					    default_val)
						ltest2 = FALSE;
				}
			}
			tlist = g_slist_next (tlist);
		}

		mt->level = ltest ? mt->level | ALL_FIELDS : mt->level;
		mt->level =
			ltest2 ? mt->level | ALL_FIELDS_WITH_DEFAULT : mt->
			level;

		tables = g_slist_next (tables);
	}

	/* 3rd step: prepend a "NONE" entry, and order the tables */
	tables = list;
	while (tables) {
		gboolean ltest, ltest2;
		GSList *tlist;
		mt = (struct _ModifTable *) (tables->data);

		if (mt->level & LINKS_TO_TABLE)
			mt->nbstars = 4;
		if (!mt->nbstars && (mt->level > (int) KEY_PRESENT))
			mt->nbstars = 3;
		if (!mt->nbstars && (mt->level == (int) KEY_PRESENT))
			mt->nbstars = 2;
		if (!mt->nbstars
		    && (mt->level >= (int) ALL_FIELDS_WITH_DEFAULT))
			mt->nbstars = 1;

		tables = g_slist_next (tables);
	}
	/* FIXME: order */

	/* "NONE" entry */
	mt = g_new0 (struct _ModifTable, 1);
	mt->table = NULL;
	list = g_slist_prepend (list, mt);

	return list;
}

static void
edit_dlg_table_mitem_activate_cb (GtkMenuItem * menu_item, SqlQueryEnv * env)
{
	SqlMemTable *table;
	gboolean found = FALSE;

	table = gtk_object_get_data (GTK_OBJECT (menu_item), "table");
	env->modif_table = table;

	/* update the "no_insert" value */
	update_no_insert_info (env);
}

static void
edit_dlg_new_table_cb (SqlDb * db, SqlMemTable * new_table, SqlQueryEnv * env)
{
	GtkWidget *mitem, *optionmenu_menu;
	GList *list;
	gint i;
	gboolean halt;
	SqlMemTable *table;

	mitem = gtk_menu_item_new_with_label (new_table->name);
	gtk_signal_connect (GTK_OBJECT (mitem), "activate",
			    GTK_SIGNAL_FUNC
			    (edit_dlg_table_mitem_activate_cb), env);
	gtk_object_set_data (GTK_OBJECT (mitem), "table", new_table);

	optionmenu_menu =
		gtk_object_get_data (GTK_OBJECT (env->edit_dlg),
				     "omenu_menu");
	list = GTK_MENU_SHELL (optionmenu_menu)->children;
	halt = FALSE;
	i = 0;
	while (list && !halt) {
		table = gtk_object_get_data (GTK_OBJECT (list->data),
					     "table");
		if (strcmp (new_table->name, table->name) < 0)
			halt = TRUE;
		else {
			list = g_list_next (list);
			i++;
		}
	}

	gtk_menu_insert (GTK_MENU (optionmenu_menu), mitem, i);
	gtk_widget_show (mitem);
}

static void
edit_dlg_del_table_cb (SqlDb * db, SqlMemTable * table, SqlQueryEnv * env)
{
	GtkWidget *optionmenu, *optionmenu_menu, *menu;

	optionmenu =
		gtk_object_get_data (GTK_OBJECT (env->edit_dlg), "omenu");
	gtk_option_menu_remove_menu (GTK_OPTION_MENU (optionmenu));

	if (env->modif_table == table)
		env->modif_table = NULL;

	optionmenu_menu = get_tables_optionmenu_menu (env);
	gtk_object_set_data (GTK_OBJECT (env->edit_dlg), "omenu_menu",
			     optionmenu_menu);
	gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu),
				  optionmenu_menu);
	gtk_widget_show (optionmenu_menu);
}

static void
edit_dlg_type_mitem_activate_cb (GtkMenuItem * menu_item, SqlQueryEnv * env)
{
	gpointer data;

	data = gtk_object_get_data (GTK_OBJECT (menu_item), "data");
	env->form_is_default = data ? TRUE : FALSE;
#ifdef debug_signal
	g_print (">> 'TYPE_CHANGED' "
		 "from sqlqueryenv->edit_dlg_type_mitem_activate_cb\n");
#endif
	gtk_signal_emit (GTK_OBJECT (env),
			 sql_query_env_signals[TYPE_CHANGED], NULL);
#ifdef debug_signal
	g_print ("<< 'TYPE_CHANGED' "
		 "from sqlqueryenv->edit_dlg_type_mitem_activate_cb\n");
#endif

}

static void
edit_layout (GtkButton * wid, SqlQueryEnv * qe)
{
	gnome_app_message (GNOME_APP (qe->q->conf->app),
			   _("Not Yet implemented\n"
			     "Here will be the import of Glade designed forms."));
}
