/*
 * Gnome-biff program
 * Version:  2.6
 * Copyright (C) 2000-2001  Nicolas Rougier
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#ifdef USE_GNOME
#  include <gnome.h>
#  include <applet-widget.h>
#endif

#include "gbiff.h"
#include "applet.h"
#include "callbacks.h"
#include "interface.h"
#include "support.h"

#ifdef USE_GNOME
/********************************************************************************
 * Function: Create an applet in the gnome panel
 * Input   : A biff structure and the orientation of the applet
 *           (APPLET_VERTICAL or APPLET_HORIZONTAL)
 * Output  : /
 * Return  : An applet
 ********************************************************************************/
GtkWidget *create_Applet (Biff *biff, int orientation) {
  GtkWidget *Applet;
  GtkWidget *AppletEventBox;
  GtkWidget *AppletTable;
  GtkWidget *TitleLabel;
  GtkWidget *StatusPixmap;
  GtkWidget *MailNumberLabel;
  char *     fontName = "-adobe-times-medium-r-normal-*-*-100-*-*-p-*-iso8859-1";
  GtkStyle * style;
  GdkFont *  font;

  Applet = applet_widget_new (PACKAGE);
  gtk_object_set_data (GTK_OBJECT (Applet), "Applet", Applet);
  if (!Applet)
    g_error("Can't create applet!\n");   
  gtk_widget_realize (Applet);

  biff->applet = Applet;
  gnome_load_session (biff);
  setDefaultParameters (biff);

  AppletEventBox = gtk_event_box_new ();
  gtk_widget_set_name (AppletEventBox, "AppletEventBox");
  gtk_widget_ref (AppletEventBox);
  gtk_object_set_data_full (GTK_OBJECT (Applet), "AppletEventBox", AppletEventBox,
                            (GtkDestroyNotify) gtk_widget_unref);
  applet_widget_add (APPLET_WIDGET (Applet), AppletEventBox);
  gtk_widget_show (AppletEventBox);

  AppletTable = gtk_table_new (3, 3, FALSE);
  gtk_table_set_col_spacings (GTK_TABLE(AppletTable), 3);
  gtk_widget_set_name (AppletTable, "AppletTable");
  gtk_widget_ref (AppletTable);
  gtk_object_set_data_full (GTK_OBJECT (Applet), "AppletTable", AppletTable,
			    (GtkDestroyNotify) gtk_widget_unref);
  gtk_container_add (GTK_CONTAINER (AppletEventBox), AppletTable);
  gtk_widget_show (AppletTable);

  /* Title label */
  TitleLabel = gtk_label_new (biff->title);
  if ((font = gdk_font_load (fontName)) == 0) {
    fprintf (stderr, "Cannot load %s font\n", fontName);
  }
  else {
    style = gtk_style_copy (gtk_widget_get_style (TitleLabel));
    style->font = font;
    gtk_widget_set_style (TitleLabel, style);
  }
  gtk_widget_set_name (TitleLabel, "TitleLabel");
  gtk_widget_ref (TitleLabel);
  gtk_object_set_data_full (GTK_OBJECT (Applet), "TitleLabel", TitleLabel,
			    (GtkDestroyNotify) gtk_widget_unref);

  /* Status pixmap */
  StatusPixmap = create_pixmap (Applet, biff->nomailPixmapFilename);
  gtk_widget_set_name (StatusPixmap, "StatusPixmap");
  gtk_widget_ref (StatusPixmap);
  gtk_object_set_data_full (GTK_OBJECT (Applet), "StatusPixmap", StatusPixmap,
                            (GtkDestroyNotify) gtk_widget_unref);
  
  /* Mail number label */
  MailNumberLabel = gtk_label_new (_("00"));
  gtk_widget_set_name (MailNumberLabel, "MailNumberLabel");
  gtk_widget_ref (MailNumberLabel);
  gtk_object_set_data_full (GTK_OBJECT (Applet), "MailNumberLabel", MailNumberLabel,
                            (GtkDestroyNotify) gtk_widget_unref);
  
  /* Putting widget into table */
  if (orientation == APPLET_VERTICAL) {
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), TitleLabel, 0, 1, 0, 1);
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), StatusPixmap, 0, 1, 1, 2);
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), MailNumberLabel, 0, 1, 2, 3);
  }
  else {
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), TitleLabel, 0, 1, 0, 1);
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), StatusPixmap, 1, 2, 0, 1);
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), MailNumberLabel, 2, 3, 0, 1);
  }
  gtk_widget_show (TitleLabel);
  gtk_widget_show (StatusPixmap);
  gtk_widget_show (MailNumberLabel);

  applet_widget_register_stock_callback (APPLET_WIDGET(Applet),
					 "AppletPropertiesMenu",
					 GNOME_STOCK_MENU_PREF,
					 _("Properties..."),
					 (AppletCallbackFunc) on_Applet_PropertiesMenu_selection,
					 biff);
  applet_widget_register_stock_callback (APPLET_WIDGET(Applet),
					 "AppletAboutMenu",
					 GNOME_STOCK_MENU_ABOUT,
					 _("About..."),
					 (AppletCallbackFunc) on_Applet_AboutMenu_selection,
					 biff);
  gtk_signal_connect (GTK_OBJECT(Applet),"remove",
		      GTK_SIGNAL_FUNC(on_Applet_remove_event),
		      biff);
  gtk_signal_connect (GTK_OBJECT (AppletEventBox), "button_press_event",
		      GTK_SIGNAL_FUNC (on_AppletEventBox_button_press_event),
		      biff);
  gtk_signal_connect (GTK_OBJECT (AppletEventBox), "button_release_event",
		      GTK_SIGNAL_FUNC (on_AppletEventBox_button_release_event),
		      biff);
  gtk_signal_connect (GTK_OBJECT (Applet), "change_orient",
		      GTK_SIGNAL_FUNC (on_Applet_change_orient),
		      biff);
  gtk_signal_connect (GTK_OBJECT (Applet), "save_session", 
		      GTK_SIGNAL_FUNC (gnome_save_session),
		      biff);

#ifdef HAVE_PANEL_PIXEL_SIZE
  gtk_signal_connect(GTK_OBJECT(Applet),"change_pixel_size",
		     GTK_SIGNAL_FUNC (on_Applet_change_pixel_size),
		     biff);
#endif
  gtk_widget_realize (Applet);
  biff->mailGdkPixmap = gdk_pixmap_create_from_xpm (Applet->window, NULL, NULL, biff->mailPixmapFilename);
  biff->nomailGdkPixmap = gdk_pixmap_create_from_xpm (Applet->window, NULL, NULL, biff->nomailPixmapFilename);
  gtk_pixmap_set (GTK_PIXMAP(lookup_widget(biff->applet, "StatusPixmap")), biff->nomailGdkPixmap, 0);

  return Applet;
}

/********************************************************************************
 * Function: Set the size of the applet according to orientation
 * Input   : Biff structure and an orientation
 *           (APPLET_VERTICAL or APPLET_HORIZONTAL)
 * Output  : /
 * Return  : /
 ********************************************************************************/
void applet_change_orientation (Biff *biff, int orientation) {
  GtkWidget *AppletTable = lookup_widget (biff->applet, "AppletTable");
  GtkWidget *TitleLabel = lookup_widget (biff->applet, "TitleLabel");
  GtkWidget *StatusPixmap = lookup_widget (biff->applet, "StatusPixmap");
  GtkWidget *MailNumberLabel = lookup_widget (biff->applet, "MailNumberLabel");

  gtk_container_remove (GTK_CONTAINER (AppletTable), TitleLabel);
  gtk_container_remove (GTK_CONTAINER (AppletTable), StatusPixmap);
  gtk_container_remove (GTK_CONTAINER (AppletTable), MailNumberLabel);
  
  if (orientation == APPLET_VERTICAL) {
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), TitleLabel, 0, 1, 0, 1);
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), StatusPixmap, 0, 1, 1, 2);
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), MailNumberLabel, 0, 1, 2, 3);
  }
  else {
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), TitleLabel, 0, 1, 0, 1);
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), StatusPixmap, 1, 2, 0, 1);
    gtk_table_attach_defaults (GTK_TABLE(AppletTable), MailNumberLabel, 2, 3, 0, 1);    
  }  
}

/********************************************************************************
 * Function: Session saving
 * Input   : Generic Gnome options
 * Ouptut  : Saved configuration
 * Return  : FALSE (see gnome documentation for explanation)
 * Notes   : Added by Neil Muller, March 2001
 *           Modified by Nicolas Rougier, March 2001
 ********************************************************************************/
gint gnome_save_session (GtkWidget *w, const char *privcfgpath, const char *globcfgpath, Biff *biff) {
  /* Set up */
  gnome_config_push_prefix(privcfgpath);

  /* Save data */
  /* PRIVATE */
  gnome_config_private_set_string ("options/address", biff->address);
  gnome_config_private_set_string ("options/user", biff->user);
  /* FIX ME */
  /* > Password SHOULD NOT be saved within session, very insecure    */
  /* > DO NOT UNCOMMENT THIS LINE UNLESS YOU KNOW WHAT YOU ARE DOING */
  /* >                    YOU HAVE BEEN WARNED  !                    */
  /*  gnome_config_private_set_string("options/password",biff->password); */
  /* GLOBAL */
  gnome_config_set_int ("options/protocol", biff->protocol);
  gnome_config_set_string ("options/title", biff->title);
  gnome_config_set_int ("options/polltime", biff->polltime);
  gnome_config_set_int ("options/poptime", biff->poptime);
  gnome_config_set_int ("options/popupX", biff->popupX);
  gnome_config_set_int ("options/popupY", biff->popupY);  
  gnome_config_set_int ("options/noPopup", biff->optNoPopup);
  gnome_config_set_int ("options/noSound", biff->optNoSound);
  gnome_config_set_int ("options/noDate", biff->optNoDate);
  gnome_config_set_int ("options/noTitle", biff->optNoTitle);
  gnome_config_set_int ("options/noConfigure", biff->optNoConfigure);
  gnome_config_set_string ("options/soundfile", biff->sound);
  gnome_config_set_string ("options/font", biff->font);
  gnome_config_set_string ("options/mailPixmap", biff->mailPixmapFilename);
  gnome_config_set_string ("options/nomailPixmap", biff->nomailPixmapFilename);

  /* Clean up */
  gnome_config_pop_prefix();
  gnome_config_sync();
  gnome_config_drop_all();
  return FALSE;
}

/********************************************************************************
 * Function: Load a session from the gnome session manager
 * Input   : Biff structure
 * Ouptut  : Filled biff structure
 * Return  : /
 * Notes   : Added by Neil Muller, March 2001
 *           Modified by Nicolas Rougier, March 2001
 ********************************************************************************/
void gnome_load_session (Biff *biff) {
  char *text;
  int value;
  gboolean bool, def;
 
  /* For each argument, we check if the default value is used and if so, then
     the argument is not set because it will be set later with the 
     setDefautlParameters function.
     If the argument is already set (because of the command line parsing, then
     the argument is not set again (command line options prevails onto gnome session) */

  gnome_config_push_prefix (APPLET_WIDGET(biff->applet)->privcfgpath);
  
  /* PRIVATE */
  /* Address */
  text = gnome_config_private_get_string_with_default ("options/address=address", &def);
  if ((def == 0) && (biff->address == UNSET_ADDRESS)) {
    biff->address = strdup (text);
    g_free (text);
  };
  /* User */
  text = gnome_config_private_get_string_with_default ("options/user=user", &def);
  if ((def == 0) && (biff->user == UNSET_USER)) {
    biff->user = strdup (text);
    g_free (text);
  };
  /* Password */
  text = gnome_config_private_get_string_with_default ("options/password=password", &def);
  if ((def == 0) && (biff->password == UNSET_PASSWORD)) {
    biff->password = strdup (text);
    g_free (text);
  };

  /* GLOBAL */
  /* Title */
  text = gnome_config_get_string_with_default ("options/title=mail", &def);
  if ((def == 0) && (biff->title == UNSET_TITLE)) {
    biff->title = strdup (text);
    g_free (text);
  };
  /* Protocol */
  value = gnome_config_get_int_with_default ("options/protocol=0", &def);
  if ((def == 0) && (biff->protocol == UNSET_PROTOCOL))
    biff->protocol = value;
  /* Polltime */
  value = gnome_config_get_int_with_default ("options/polltime=0", &def);
  if ((def == 0) && (biff->polltime == UNSET_POLLTIME))
    biff->polltime = value;
  /* Poptime */
  value = gnome_config_get_int_with_default ("options/poptime=0", &def);
  if ((def == 0) && (biff->poptime == UNSET_POPTIME))
    biff->poptime = value;
  /* Popup geometry X */
  value = gnome_config_get_int_with_default ("options/popupX=0", &def);
  if ((def == 0) && (biff->popupX == UNSET_GEOM_POP_X))
    biff->popupX = value;
  /* Popup geometry Y */
  value = gnome_config_get_int_with_default ("options/popupY=0", &def);
  if ((def == 0) && (biff->popupX == UNSET_GEOM_POP_Y))
    biff->popupY = value;
  /* No date option */
  bool = gnome_config_get_int_with_default ("options/noDate=0", &def);
  if ((def == 0) && (biff->optNoDate == UNSET_OPT_NO_DATE))
    biff->optNoDate = bool;
  /* No popup option */
  bool = gnome_config_get_int_with_default ("options/noPopup=0", &def);
  if ((def == 0) && (biff->optNoPopup == UNSET_OPT_NO_POPUP))
    biff->optNoPopup = bool;
  /* No sound option */
  bool = gnome_config_get_int_with_default ("options/noSound=0", &def);
  if ((def == 0) && (biff->optNoSound == UNSET_OPT_NO_SOUND))
    biff->optNoSound = bool;
  /* No title option */
  bool = gnome_config_get_int_with_default ("options/noTitle=0", &def);
  if ((def == 0) && (biff->optNoTitle == UNSET_OPT_NO_TITLE))
    biff->optNoTitle = bool;
  /* No configure option */
  bool = gnome_config_get_int_with_default ("options/noConfigure=0", &def);
  if ((def == 0) && (biff->optNoConfigure == UNSET_OPT_NO_CONFIGURE))
    biff->optNoConfigure = bool;
  /* Soundfile */
  text = gnome_config_get_string_with_default ("options/soundfile=soundfile", &def);
  if ((def == 0) && (biff->sound == UNSET_SOUND)) {
    biff->sound = strdup (text);
    g_free (text);
  };
  /* Font */
  text = gnome_config_get_string_with_default ("options/font=font", &def);
  if ((def == 0) && (biff->font == UNSET_FONT)) {
    biff->font = strdup (text);
    g_free (text);
  };
  /* Mail pixmap */
  text = gnome_config_get_string_with_default ("options/mailPixmap=mailPixmap", &def);
  if ((def == 0) && (biff->mailPixmapFilename == UNSET_MAIL_PIXMAP_FILENAME)) {
    biff->mailPixmapFilename = strdup (text);
    g_free (text);
  };
  /* No mail pixmap */
  text = gnome_config_get_string_with_default ("options/nomailPixmap=nomailPixmap", &def);
  if ((def == 0) && (biff->nomailPixmapFilename == UNSET_NOMAIL_PIXMAP_FILENAME)) {
    biff->nomailPixmapFilename = strdup (text);
    g_free (text);
  };

  gnome_config_pop_prefix();
}
#endif  /* ifdef USE_GNOME */
