/* Check arguments for printf, scanf ...
   Copyright 1995, 1996 Tristan Gingold
		  Written December 1995 by Tristan Gingold

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License 
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.

The author may be reached by US/French mail:
		Tristan Gingold 
		8 rue Parmentier
		F-91120 PALAISEAU
		FRANCE
*/

#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <limits.h>
#define NEED_CHKR_VA_ARG_ADDR
#include "machine.h"
#include "check-printf.h"
#include "checker_api.h"

#ifndef chkr_va_arg_addr
#define chkr_va_arg_addr &va_arg
#endif

#define	LONG  1
#define	SHORT 2
#define LONGLONG 4
#define WIDTH    8
#define SUPPRESS 16
#define FORCE_SIGN 32
#define HASH 64
#define GROUP 128

#define CONV_CHAR 1
#define CONV_INT  2
#define CONV_STRING 3
#define CONV_DOUBLE 4
#define CONV_POINTER 5
#define CONV_HEX 6
#define CONV_OCT 7

/* Number of chars needed to write a decimal number with type t.  */
#ifndef CHAR_BIT
#define CHAR_BIT 8
#endif
#define NBR_CHARS_FOR_TYPE(t) (sizeof(t) * 3)
#define NBR_BITS_FOR_TYPE(t) (sizeof(t) * CHAR_BIT)
#define NBR_XDIGITS_FOR_TYPE(t) (NBR_BITS_FOR_TYPE(t) / 4)
#define NBR_ODIGITS_FOR_TYPE(t) ((NBR_BITS_FOR_TYPE(t) + 2) / 3)

/* Returns the length of string S, but not greater than N. */
size_t
strncharlen (const char *s, size_t n)
{
  size_t res = 0;
  while (res < n && *s++ == 0)
    res++;
  return res;
}
  
/* Returns the size of string S, including the null terminator, but
   not greater than N. */
size_t
strnsize (const char *s, size_t  n)
{
  size_t res = 0;
  for (;;)
    {
      if (res == n)
	return res;

      res++;
      if (*s++ == 0)
	return res;
    }
}

/* Convert the string at *S to a number, returning the number, and
   advance *S past the number. */
long
local_atol (const char **s)
{
  long temp = 0;
  while (isdigit (**s))
    {
      temp = temp * 10 + (**s - '0');
      (*s)++;
    }
  return temp;
}

#ifdef CHECK_INCOMING_ARGS
#undef CHECK_INCOMING_ARGS
#define CHECK_INCOMING_ARGS 1
#else
#define CHECK_INCOMING_ARGS 0
#endif

/* TYPE is TYPE_PRINTF.  */
/* Return the number of bytes that will be written.  */
int
check_printf_format (char const *name, char const *fmt, va_list ap, int type, int va)
{
  int num;
  int len;

  num = 0;
  len = 1; /* because of '\0'.  */
  
  /* Scan the format for conversions (`%' character). */
  for (;;)
    {
      int flags;			/* flags as above */
      char c;
      int width;	/* Field width. */
      int prec;		/* Precision. */
      int w;		/* Width of characters to print. */
      const char *start;
      
      while (*fmt && *fmt != '%')
	{
	  len++;
	  fmt++;
	}

      if (*fmt == '\0')
	return len;

      start = fmt++;		/* skip over '%' */

      flags = 0;

      if (type >= 0 && type == num)
	{
	  /* All the args where checked.  */
	  return 0;
	}
      else
        num++;

      for (;;)
	{
	  c = *fmt;
	  if (c == '-' || c == '0')
	    /* doesn't affect output width */;
	  else if (c == '+' || c == ' ')
	    flags |= FORCE_SIGN;
	  else if (c == '#')
	    flags |= HASH;
	  else if (c == '\'')
	    flags |= GROUP; /* FIXME not implemented for integer conversions */
	  else
	    break;

	  fmt++;
	}

      if (isdigit (*fmt))
	width = local_atol (&fmt);
      else if (*fmt == '*')
	{
	  int *addr = chkr_va_arg_addr (ap, int);
	  if (va || CHECK_INCOMING_ARGS)
	    stubs_chkr_check_addr (addr, sizeof (int), CHKR_RO, "%*");
	  width = *addr;
	  
	  fmt++;
	}
      else
	width = 0;

      if (width < 0)
	width = -width;

      prec = -1;
      if (*fmt == '.')
	{
	  fmt++;
	  
	  if (isdigit (*fmt))
	    prec = local_atol (&fmt);
	  else if (*fmt == '*')
	    {
	      int *addr = chkr_va_arg_addr (ap, int);
	      if (va || CHECK_INCOMING_ARGS)
		stubs_chkr_check_addr (addr, sizeof (int), CHKR_RO, "%*");
	      prec = *addr;

	      fmt++;
	    }

	  if (prec < 0)
	    {
	      chkr_report (M_C_FUN_LB_ET);
	      chkr_printf ("%s: negative precision %d in format string\n", prec);
	      chkr_disp_call_chain ();

	      prec = -1;
	    }
	}

      if (*fmt == 'h')
	flags |= SHORT, fmt++;
      else if (*fmt == 'l' && fmt[1] == 'l')
	flags |= LONGLONG, fmt += 2;
      else if (*fmt == 'l' || *fmt == 'D' || *fmt == 'O' || *fmt == 'U')
	flags |= LONG, fmt++;
      else if (*fmt == 'L' || *fmt == 'q')
	flags |= LONGLONG, fmt++;
      else if (*fmt == 'Z')
	flags |= LONG, fmt++; /* actually should be sizeof (size_t) */

      w = 0;
      switch (c = *fmt++)
	{
	case '%':
	  w = 1;
	  break;
	case 'c':
	  {
	    int *addr = chkr_va_arg_addr (ap, int);
	    if (va || CHECK_INCOMING_ARGS)
	      stubs_chkr_check_addr (addr, sizeof (int), CHKR_RO, "%c");

	    w = 1;
	  }
	  break;
	case 'd':
	case 'i':
	case 'o':
	case 'u':
	case 'x':
	case 'X':
	  {
	    long long int value;
	    int n_digits = 0;
	    int base;

	    if (c == 'o')
	      base = 8;
	    else if (c == 'x' || c == 'X')
	      base = 16;
	    else
	      base = 10;

	    if (c == 'd' || c == 'i')
	      {
		if (flags & LONGLONG)
		  {
		    long long *addr = chkr_va_arg_addr (ap, long long);
		    if (va || CHECK_INCOMING_ARGS)
		      stubs_chkr_check_addr (addr, sizeof (long long), CHKR_RO, "%qd");
		    value = *addr;
		  }
		else if (flags & LONG)
		  {
		    long *addr = chkr_va_arg_addr (ap, long);
		    if (va || CHECK_INCOMING_ARGS)
		      stubs_chkr_check_addr (addr, sizeof (long), CHKR_RO, "%ld");
		    value = *addr;
		  }
		else if (flags & SHORT)
		  {
		    short *addr = chkr_va_arg_addr (ap, short);
		    if (va || CHECK_INCOMING_ARGS)
		      stubs_chkr_check_addr (addr, sizeof (short), CHKR_RO, "%hd");
		    value = *addr;
		  }
		else
		  {
		    int *addr = chkr_va_arg_addr (ap, int);
		    if (va || CHECK_INCOMING_ARGS)
		      stubs_chkr_check_addr (addr, sizeof (int), CHKR_RO, "%d");
		    value = *addr;
		  }
	      }
	    else
	      {
		if (flags & LONGLONG)
		  {
		    unsigned long long *addr = chkr_va_arg_addr (ap, unsigned long long);
		    if (va || CHECK_INCOMING_ARGS)
		      stubs_chkr_check_addr (addr, sizeof (unsigned long long), CHKR_RO, "%qd");
		    value = *addr;
		  }
		else if (flags & LONG)
		  {
		    unsigned long *addr = chkr_va_arg_addr (ap, unsigned long);
		    if (va || CHECK_INCOMING_ARGS)
		      stubs_chkr_check_addr (addr, sizeof (unsigned long), CHKR_RO, "%ld");
		    value = *addr;
		  }
		else if (flags & SHORT)
		  {
		    unsigned short *addr = chkr_va_arg_addr (ap, unsigned short);
		    if (va || CHECK_INCOMING_ARGS)
		      stubs_chkr_check_addr (addr, sizeof (unsigned short), CHKR_RO, "%hd");
		    value = *addr;
		  }
		else
		  {
		    unsigned int *addr = chkr_va_arg_addr (ap, unsigned int);
		    if (va || CHECK_INCOMING_ARGS)
		      stubs_chkr_check_addr (addr, sizeof (unsigned int), CHKR_RO, "%d");
		    value = *addr;
		  }
	      }

	    if (value < 0 || (flags & FORCE_SIGN))
	      w++;
	    value = abs (value);

	    if (flags & HASH)
	      {
		if (base == 8)
		  w++;
		else if (base == 16)
		  w += 2;
	      }

	    if (value == 0 && prec == 0)
	      n_digits = 0;
	    else if (value == 0)
	      n_digits = 1;
	    else 
	      while (value > 0)
		{
		  value /= base;
		  n_digits++;
		}

	    if (n_digits < prec)
	      w += prec;
	    else
	      w += n_digits;
	  }
	  break;
	case 'e':
	case 'E':
	case 'f':
	case 'F':
	case 'g':
	case 'G':
	  {
	    char buf[1000];
	    char format[128], *cp;

	    cp = format;
	    *cp++ = '%';
	    if (flags & FORCE_SIGN)
	      *cp++ = '+';
	    if (flags & HASH)
	      *cp++ = '#';
	    if (flags & GROUP)
	      *cp++ = '\'';
	    *cp++ = '.';
	    cp += sprintf (cp, "%d", prec);
	    if (flags & LONGLONG)
	      *cp++ = 'L';
	    *cp++ = 'f';
	    *cp = '\0';

	    if (flags & LONGLONG)
	      {
		long double *addr = chkr_va_arg_addr (ap, long double);
		if (va || CHECK_INCOMING_ARGS)
		  stubs_chkr_check_addr (addr, sizeof (long double), CHKR_RO, "%Lf");
		sprintf (buf, format, *addr);
	      }
	    else 
	      {
		double *addr = chkr_va_arg_addr (ap, double);
		if (va || CHECK_INCOMING_ARGS)
		  stubs_chkr_check_addr (addr, sizeof (double), CHKR_RO, "%f");
		sprintf (buf, format, *addr);
	      }

	    w = strlen (buf);
	  }
	  break;
#if __GLIBC__
	case 'm':
	  {
	    int *addr;
	    char buf[1000]; /* This is the buffer size used by glibc 2.0.6 at least. */
	    
	    addr = chkr_va_arg_addr (ap, int);
	    if (va || CHECK_INCOMING_ARGS)
	      stubs_chkr_check_addr (addr, sizeof (int), CHKR_RO, "%m");
	    strerror_r (*addr, buf, 1000);
	    w = strlen (buf);
	    if (prec != -1 && w > prec)
	      w = prec;
	  }
	  break;
#endif /* __GLIBC__ */
	case 'n':
	  {
	    if (flags & LONG)
	      {
		long **addr = chkr_va_arg_addr (ap, long *);
		if (va || CHECK_INCOMING_ARGS)
		  stubs_chkr_check_addr (addr, sizeof (long *), CHKR_RO, "%ln");
		stubs_chkr_check_addr (*addr, sizeof (long), CHKR_WO, "*%ln");
	      }
	    else if (flags & SHORT)
	      {
		short **addr = chkr_va_arg_addr (ap, short *);
		if (va || CHECK_INCOMING_ARGS)
		  stubs_chkr_check_addr (addr, sizeof (short *), CHKR_RO, "%hn");
		stubs_chkr_check_addr (*addr, sizeof (short), CHKR_WO, "*%hn");
	      }
	    else
	      {
		int **addr = chkr_va_arg_addr (ap, int *);
		if (va || CHECK_INCOMING_ARGS)
		  stubs_chkr_check_addr (addr, sizeof (int *), CHKR_RO, "%n");
		stubs_chkr_check_addr (*addr, sizeof (int), CHKR_WO, "*%hn");
	      }
	    w = 0;
	  }
	  break;
	case 'p':
	  {
	    void **addr;
	    unsigned long p;
	    
	    addr = chkr_va_arg_addr (ap, void *);
	    p = *(unsigned long *) addr;
	    if (va || CHECK_INCOMING_ARGS)
	      stubs_chkr_check_addr (addr, sizeof (void *), CHKR_RO, "%p");
	    if (p == 0)
	      w += strlen ("(nil)");
	    else
	      {
		w += 2;
		while (p != 0)
		  w++, p /= 16;
	      }
	  }
	  break;
	case 's':
	  {
	    char **addr;
	    addr = chkr_va_arg_addr (ap, char *);
	    if (va || CHECK_INCOMING_ARGS)
	      stubs_chkr_check_addr (addr, sizeof (char *), CHKR_RO, "%s");
	    if (*addr)
	      {
		if (prec != -1)
		  {
		    if (prec > 0)
		      {
			w = strncharlen (*addr, prec);
			stubs_chkr_check_addr (*addr, strnsize (*addr, prec), CHKR_RO, "%s");
		      }
		    else 
		      w = 0;
		  }
		else
		  {
		    stubs_chkr_check_str (*addr, CHKR_RO, "%s");
		    w = strlen (*addr);
		  }
	      }
	    else
	      {
		/* printf ("%s", NULL) usually prints (null).  */
		w = 6;
		chkr_report (M_C_FUN_LB_ET);
		chkr_printf ("%s: NULL argument for format `%%s'\n", name);
		chkr_disp_call_chain ();
	      }
	  }
	  break;
	default:
	  chkr_report (M_C_FUN_LB_ET);
	  chkr_printf ("%s: Unknown type '%c'\n", name, c);
	  chkr_disp_call_chain ();
	}
      
      if (w > width)
	len += w;
      else
	len += width;
    }

  return len;
}

/* TYPE is either TYPE_PRESCANF or a positive number of args that were
   written.  */
/* Return the maximum of bytes written.  */
int
check_scanf_format (char const *name, char const *fmt, va_list ap, int type, int va)
{
  int flags;			/* flags as above */
  char c;
  int width;
  int conv;
  int num;
  int len;
  int right;

  if (type == TYPE_PRESCANF)
    right = CHKR_TW;
  else
    right = CHKR_WO;

  num = 0;
  len = 1; /* because of '\0'.  */
  
  /* Scan the format for conversions (`%' character). */
  while (1)
    {
      while (*fmt && *fmt != '%')
	{
	  len++;
	  fmt++;
	}

      if (*fmt == '\0')
	return len;

      fmt++;			/* skip over '%' */

      flags = 0;
      width = 0;
      conv = 0;

      if (type >= 0 && type == num)
	{
	  /* All the args where checked.  */
	  return 0;
	}
      else
        num++;

    rflag:
      switch ((c = *fmt++))
	{
	case '%':
	  len++;
	  continue;	/* ignore this char */
	case ' ':
	case '#':
	case '-':
	case '+':
	  goto rflag;
	case '*':
	  flags |= SUPPRESS;
	  num--;
	  goto rflag;
	case '.':
	case '0':
	  goto rflag;
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	    do
	      {
		width = width * 10 + (c - '0');
		c = *fmt++;
	      }
	    while (isdigit ((unsigned char)c));
	    fmt--;
	    flags |= WIDTH;
	  goto rflag;
 	case 'L':
	  flags |= LONGLONG;
	  goto rflag;
	case 'h':
	  flags |= SHORT;
	  goto rflag;
	case 'l':
	  flags |= LONG;
	  goto rflag;
	case 'c':
	  conv = CONV_CHAR;
	  break;
	case 'D':
	  flags |= LONG;
	case 'd':
	case 'i':
	  conv = CONV_INT;
	  break;
	case 'e':
	case 'E':
	case 'f':
	case 'F':
	case 'g':
	case 'G':
	  conv = CONV_DOUBLE;
	  break;
	case 'n':
	  conv = CONV_INT;
	  break;
	case 'O':
	  flags |= LONG;
	case 'o':
	  conv = CONV_OCT;
	  break;
	case 'p':
	  conv = CONV_POINTER;
	  break;
	case 's':
	  conv = CONV_STRING;
	  break;
	case 'U':
	  flags |= LONG;
	case 'u':
	  conv = CONV_INT;
	  break;
	case 'X':
	case 'x':
	  conv = CONV_HEX;
	  break;
	case '[':
	  if (*fmt++ == '^')
	    fmt++;		/* skip the character after the '^' */
	  while (*fmt && *fmt != ']')
	    fmt++;
	  if (!*fmt)
	    {
	      chkr_report (M_C_FUN_LB_ET);
	      chkr_printf ("%s: Bad format after '['\n", name);
	      chkr_disp_call_chain ();
	    }
	  else
	    {
	      conv = CONV_STRING;
	      fmt++;
	    }
	  break;
	default:
	  chkr_report (M_C_FUN_LB_ET);
	  chkr_printf ("%s: Unknown type '%c'\n", name, c);
	  chkr_disp_call_chain ();
	}

      if (type == TYPE_PRESCANF && !(flags & SUPPRESS))
	switch (conv)
	  {
	  case CONV_CHAR:
	    {
	      char **addr;
	      addr = chkr_va_arg_addr (ap, char*);
	      if (va || CHECK_INCOMING_ARGS)
	        stubs_chkr_check_addr (addr, sizeof (char *), CHKR_RO, "%c");
	      stubs_chkr_check_addr (*addr, width ? width * sizeof(char) : sizeof(char), right, "%c");
	    }
	    break;
	  case CONV_INT:
	    if (flags & LONG)
	      {
		long **addr;
		addr = chkr_va_arg_addr (ap, long*);
		if (va || CHECK_INCOMING_ARGS)
		  stubs_chkr_check_addr (addr, sizeof (long*), CHKR_RO, "%ld");
		stubs_chkr_check_addr (*addr, sizeof (long), right, "%ld");
	      }
	    else if (flags & SHORT)
	      {
		int **addr;
		addr = chkr_va_arg_addr (ap, int*);
		if (va || CHECK_INCOMING_ARGS)
		  stubs_chkr_check_addr (addr, sizeof (short*), CHKR_RO, "%h");
		stubs_chkr_check_addr (*addr, sizeof (short), right, "%h");
	      }
	    else
	      {
		int **addr;
		addr = chkr_va_arg_addr (ap, int*);
		if (va || CHECK_INCOMING_ARGS)
		  stubs_chkr_check_addr (addr, sizeof (int*), CHKR_RO, "%d");
		stubs_chkr_check_addr (*addr, sizeof (int), right, "%d");
	      }
	    break;
	  case CONV_DOUBLE:
	    {
	      double **addr;
	      addr = chkr_va_arg_addr (ap, double*);
	      if (va || CHECK_INCOMING_ARGS)
	        stubs_chkr_check_addr (addr, sizeof (double*), CHKR_RO, "%e");
	      stubs_chkr_check_addr (*addr, sizeof (double), right, "%e");
	    }
	    break;
	  case CONV_POINTER:
	    {
	      void ***addr;
	      addr = chkr_va_arg_addr (ap, void **);
	      if (va || CHECK_INCOMING_ARGS)
	        stubs_chkr_check_addr (addr, sizeof (void **), CHKR_RO, "%p");
	      stubs_chkr_check_addr (*addr, sizeof (void *), right, "%p");
	    }
	    break;
	  case CONV_STRING:
	    {
	      char **addr;
	      addr = chkr_va_arg_addr (ap, char *);
	      if (va || CHECK_INCOMING_ARGS)
	        stubs_chkr_check_addr (addr, sizeof (char *), CHKR_RO, "%s");
	      if (!width)
	        {
	          chkr_report (M_C_FUN_LB_ET);
	          chkr_printf("%s: No width field for string.\n", name);
	          chkr_disp_call_chain ();
	        }
	      else
	        stubs_chkr_check_addr (*addr, width + 1, right, "%s");
	    }
	    break;
	  }
      else if (!(flags & SUPPRESS))
	switch (conv)
	  {
	  case CONV_CHAR:
	    {
	      char **addr;
	      addr = chkr_va_arg_addr (ap, char*);
	      stubs_chkr_set_right (*addr, width ? width * sizeof(char) : sizeof(char), CHKR_RW);
	    }
	    break;
	  case CONV_INT:
	    if (flags & LONG)
	      {
		long **addr;
		addr = chkr_va_arg_addr (ap, long*);
		stubs_chkr_set_right (*addr, sizeof (long), CHKR_RW);
	      }
	    else if (flags & SHORT)
	      {
		int **addr;
		addr = chkr_va_arg_addr (ap, int*);
		stubs_chkr_set_right (*addr, sizeof (short), CHKR_RW);
	      }
	    else
	      {
		int **addr;
		addr = chkr_va_arg_addr (ap, int*);
		stubs_chkr_set_right (*addr, sizeof (int), CHKR_RW);
	      }
	    break;
	  case CONV_DOUBLE:
	    {
	      double **addr;
	      addr = chkr_va_arg_addr (ap, double*);
	      stubs_chkr_set_right (*addr, sizeof (double), CHKR_RW);
	    }
	    break;
	  case CONV_POINTER:
	    {
	      void ***addr;
	      addr = chkr_va_arg_addr (ap, void **);
	      stubs_chkr_set_right (*addr, sizeof (void *), CHKR_RW);
	    }
	    break;
	  case CONV_STRING:
	    {
	      char **addr;
	      int n;
	      addr = chkr_va_arg_addr (ap, char *);
	      n = strlen (*addr) + 1;	/* FIXME */
	      stubs_chkr_set_right (*addr, n * sizeof (char), CHKR_RW);
	    }
	    break;
	  }
    }
}

#ifdef TEST
int
my_printf (char *format,...)
{
  va_list param;
  va_start (param, format);
  check_printf_format ("my_printf", format, param, TYPE_PRINTF);
  return vprintf (format, param);
}

int
my_scanf (char *format,...)
{
  va_list param;
  int n;
  va_start (param, format);
  check_printf_format ("my_scanf", format, param, TYPE_PRESCANF);
  n = vscanf (format, param);
  if (n != EOF)
    check_printf_format ("my_scanf", format, param, n);
  return n;
}

void
stubs_chkr_check_addr (const PTR ptr, int len, int right)
{
  printf ("Check at %p for %d bytes (%d)\n", ptr, len, right);
}

void
stubs_chkr_check_str (const PTR ptr, int right)
{
  printf ("Check string at %p (%d)\n", ptr, right);
}

void
stubs_chkr_set_right (const PTR ptr, int len, int right)
{
  printf ("Set right at %p for %d bytes (%d)\n", ptr, len, right);
}

int
main (int argc, char *argv[])
{
  int n;
  char c[20];
  my_printf ("Hello 100%%\n");
  my_printf ("%d + %d = %d\n", 1, 4, 4 + 1);
  my_printf ("%s: %s\n", argv[0], "Hello");
  my_printf ("%s: %s%n\n", argv[0], "Hello", &n);
  my_printf ("The last line has %d chars\n", n);
  my_printf ("Please enter a small string:\n");
  my_scanf ("%s", c);
  my_scanf ("%20s", c);
  my_printf ("Please enter a number:\n");
  my_scanf ("%d", &n);
  my_printf ("Please enter a number, a string and another number:\n");
  my_scanf ("%d%10s%d", &n, c, &n);
  return 0;
}

#endif /* TEST */
