from utils.Observable import Observable
from utils.TypeConverter import TypeConverter
from utils.datatypes import *
from utils import actionparser
from utils import dialog


#
# Abstract class for data targets.
#
class DataTarget(Observable):

    AUTHORIZED_METHODS = ()

    def __init__(self, parent):

        # the XML tag name of the object
        self.__name = ""

        # the setter and getter methods for properties
        self.__property_handlers = {}

        # the table of properties; you can store your properties in here to
        # avoid having another instance variable
        self.__properties = {}

        # the actions and their associated calls
        self.__actions = {}

        # the index path of this target
        self.__index_path = []

        # the parent of this target
        self.__parent = parent

        # the display of this target
        self.__display = parent._get_display()

        # converter for data types
        self.__type_converter = TypeConverter()

        # list of watch bindings for being able to unbind the sensor again
        # FIXME: remove eventually :)
        self.__watch_bindings = []


        self._register_property("event", TYPE_OBJECT, None, self._getp)
        self._register_property("id", TYPE_STRING, self._setp_id, self._getp)
        self._register_property("index-path", TYPE_INT, None, self._getp)
        self._register_property("watch", TYPE_LIST, self._setp_watch, None)

        if (parent and not self.is_standalone()):
            self.__index = parent.get_next_child_index()
            self.__index_path = parent.get_index_path()
        else:
            self.__index = -1
            self.__index_path = []

        if (not self.is_standalone()):
            self.add_observer(parent.child_observer)

        if (self.__index != -1): self.__index_path.append(self.__index)
        if (self.__index_path):
            self._setp("index-path", self.__index_path[:])
        else:
            self._setp("index", [])



    #
    # Returns the parent of this target.
    #
    def _get_parent(self): return self.__parent



    #
    # Returns whether this target is standalone, i.e. needs no parent.
    #
    def is_standalone(self): return True



    #
    # Tells this target about its XML name.
    #
    def set_name(self, name): self.__name = name


    def delete(self):

        self.drop_observers()
        self.__unbind_sensors()



    #
    # Returns the display of this target.
    #
    def _get_display(self): return self.__display



    #
    # Returns the sensor watch propagator of this target. This is either the
    # display window or an array.
    #
    def _get_propagator(self):

        parent = self._get_parent()
        while (parent and not parent.__class__.__name__ == "TargetArray"):
            parent = parent._get_parent()

        if (not parent): parent = self._get_display()

        return parent



    #
    # Returns the index path of this target.
    #
    def get_index_path(self): return self.__index_path[:]



    #
    # Registers the given action.
    #
    def _register_action(self, action):

        self._register_property("on-" + action, TYPE_STRING,
                                self._setp__action, self._getp)



    #
    # Registers the given property with the given setter and getter methods.
    #
    def _register_property(self, name, datatype, setter, getter):

        self.__type_converter.add_type(name, datatype)
        self.__property_handlers[name] = (setter, getter)



    #
    # Sets the given XML property. This is the same as set_prop() but takes
    # strings as values and converts before setting.
    #
    def set_xml_prop(self, key, value):

        # work around bad outout values from sensors
        # FIXME: always convert eventually :)
        #if (type(value) == str or type(value) == unicode):
        value = self.__type_converter.str2type(key, value)
        self.set_prop(key, value)



    #
    # Sets the given property.
    #
    def set_prop(self, key, value):

        key = key.replace("_", "-")
        try:
            setter = self.__property_handlers[key] [0]
        except KeyError:
            name = self.__name
            dialog.warning(_("No such property: %s") % key,
                           _("The element <b>%(name)s</b> does not have "
                             "the <b>%(key)s</b> property.") % vars(),
                           False)
            return
            #raise KeyError("Error: No such property: %s" % key)

        if (not setter):
            name = self.__name
            dialog.warning(_("Permission Error"),
                           _("The property <b>%s</b> of element <b>%s</b> "
                             "is not writable.") % (key, name), False)
            return
        else:
            setter(key, value)



    #
    # Returns the value of the given property.
    #
    def get_prop(self, key):

        key = key.replace("_", "-")
        try:
            getter = self.__property_handlers[key] [1]
        except KeyError:
            raise KeyError("Error: No such property: %s" % key)

        if (not getter):
            name = self.__name
            dialog.warning(_("Permission Error"),
                           _("The property <b>%s</b> of element <b>%s</b> "
                             "is not readable.") % (key, name), False)
            raise KeyError("Error: property not readable: %s" % key)
        else:
            return getter(key)


    def get_props(self): return self.__property_handlers.keys()


    #
    # Emits the given action if the pointer is over this target.
    #
    def get_action_call(self, action):

        return self.__actions.get(action, None)



    #
    # Handles the given action.
    #
    def handle_action(self, action, px, py, event):

        call = self.get_action_call(action)
        if (call):
            # analyze call to see if it's a legacy call
            # FIXME: remove eventually :)
            import re
            if (re.match("[\w\-]+:.*", call)):
                
                import warnings
                warnings.warn("Please use new style call", DeprecationWarning)
                
                try:
                    legacy_args = event._args
                except:
                    legacy_args = []

                legacy_call = actionparser.parse(call)
                path = self.get_index_path()
                self._get_display().call_sensor(legacy_call, path,
                                                *legacy_args)

            else:
                self._setp("event", event)
                self._get_display().execute_callback_script(call, self)


    #
    # Unbinds this target from the sensors.
    # FIXME: remove eventually :)
    #
    def __unbind_sensors(self):

        for sensorplug, prop in self.__watch_bindings:
            self._get_display().unbind_sensor(sensorplug, self, prop)
        #end for
        self.__watch_bindings = []



    #
    # Generic setter and getter methods for properties.
    #
    def _setp(self, key, value): self.__properties[key] = value
    def _getp(self, key): return self.__properties[key]



    #
    # Action handlers.
    #
    def _setp__action(self, key, value):

        name = key[3:]
        self.__actions[name] = value
        self._setp(key, value)



    #
    # "watch" property.
    #
    def _setp_watch(self, key, value):

        entries = value
        for e in entries:
            prop, sensorplug = e.split("=")
            if (self.__index_path):
                self._get_display().register_array_for_port(
                    self._get_propagator(),
                    sensorplug)
                sensorplug += str(self.__index_path)
            self._get_display().bind_sensor(sensorplug, self, prop)
            self.__watch_bindings.append((sensorplug, prop))
        #end for


    #
    # ID property.
    #
    def _setp_id(self, key, value):

        # FIXME: remove eventually :)
        if (self.__index_path):
            array = self._get_propagator()
            path = self.__index_path[:-1]
        self._setp(key, value)
