/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#include "gdis.h"
#include "sginfo.h"

/* re-entrant surface search step (Ang) */
#define LIFT_STEP 0.01
/* re-entrant surface search sweep (radians) */
#define SWEEP_STEP 0.05

/* main pak structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

/* parameters are adjustable via a dialogue */
gint ix[3] = {0,1,2};   /* indep. variables x,y,z */

/********************************/
/* grid to cartesian conversion */
/********************************/
void grid2cart(gfloat *grid, gfloat *x)
{
/* decode grid scan plane format to x,y,z format */
if (ix[0] == 1)
  {
/* yz */
  *(x+0) = *(grid+2);
  *(x+1) = *(grid+0);
  *(x+2) = *(grid+1);  
  }
else
  {
/* xy */
  if (ix[1] == 1)
    {
    *(x+0) = *(grid+0);
    *(x+1) = *(grid+1);
    *(x+2) = *(grid+2);  
    }
/* xz */
  else
    {
    *(x+0) = *(grid+0);
    *(x+1) = *(grid+2);
    *(x+2) = *(grid+1);  
    }
  }
}

/****************************************************/
/* make a grid representation of a connolly surface */
/****************************************************/
#define DEBUG_CALCCON 0
void calc_surf(struct model_pak *data)
{
gint i, j, k, m, n, ai, bi, num_exp, swap;
gint *atom_exp, grid, touch, flag, flag2;
gfloat da, db, dc, pa, pb, pc, prad, rad, len, height, d2, r2, td2, ta, tb;
gfloat a[3], b[3], g[3], p[3], x[3], pos[3], min[3], maxh, step[3];
gfloat qa, qb, qc;
gfloat angle, area, gamma, atot, psa, gsa, *area_exp, *area_pac;

/* checks */
g_return_if_fail(data != NULL);
if (!sysenv.num_models)
  return;
if (data->fractional)
  {
  show_text("Sorry, can't do this for fractional coordinates yet.");
  return;
  }

/* setup */
prad = data->csurf.prad;
grid = data->csurf.grid;

/* attempt to match atom type with database */
#if DEBUG_CALCCON
printf("Constructing grid (%dx%d) with probe of radius: %f\n",grid,grid,prad);
#endif

/* allocation */
g_free(data->csurf.touch);
g_free(data->csurf.code);
g_free(data->csurf.h);
g_free(data->csurf.pa);
g_free(data->csurf.pb);
g_free(data->csurf.pc);
g_free(data->csurf.px);
g_free(data->csurf.py);
g_free(data->csurf.rx);
g_free(data->csurf.ry);
g_free(data->csurf.rz);
/* CURRENT */
g_free(data->csurf.n[0]);
g_free(data->csurf.n[1]);
g_free(data->csurf.n[2]);
data->csurf.touch = (gint *) g_malloc(grid*grid*sizeof(gint));
data->csurf.code = (gint *) g_malloc(grid*grid*sizeof(gint));
data->csurf.h = (gfloat *) g_malloc(grid*grid*sizeof(gfloat));
data->csurf.pa = (gfloat *) g_malloc(grid*grid*sizeof(gfloat));
data->csurf.pb = (gfloat *) g_malloc(grid*grid*sizeof(gfloat));
data->csurf.pc = (gfloat *) g_malloc(grid*grid*sizeof(gfloat));
data->csurf.px = (gint *) g_malloc(grid*grid*sizeof(gint));
data->csurf.py = (gint *) g_malloc(grid*grid*sizeof(gint));
data->csurf.rx = (gfloat *) g_malloc(grid*grid*sizeof(gfloat));
data->csurf.ry = (gfloat *) g_malloc(grid*grid*sizeof(gfloat));
data->csurf.rz = (gfloat *) g_malloc(grid*grid*sizeof(gfloat));
data->csurf.n[0] = (gfloat *) g_malloc(grid*grid*sizeof(gfloat));
data->csurf.n[1] = (gfloat *) g_malloc(grid*grid*sizeof(gfloat));
data->csurf.n[2] = (gfloat *) g_malloc(grid*grid*sizeof(gfloat));

/* match min/max to displayed pbc cell */
/* start at origin (see make_cell()) */
min[0] = min[1] = min[2] = 0.0;
/* supplement with 'height' scan variable ONLY */
maxh = -999999.999;
min[ix[2]] = 999999.999;
for (n=0 ; n<data->num_atoms ; n++)
  {
  if ((data->atoms+n)->status & (DELETED | HIDDEN))
    continue;
  x[0] = (data->atoms+n)->x;
  x[1] = (data->atoms+n)->y;
  x[2] = (data->atoms+n)->z;
  vecmat(data->latmat, x);

/* 'height' only */
  if (x[ix[2]] < min[ix[2]])
    min[ix[2]] = x[ix[2]];
  if (x[ix[2]] > maxh)
    maxh = x[ix[2]];
  }
/* ensure full 'height' scan by adding a 'vertical' safety zone */
/* (if max is too low - probe will not reach the top of high atoms) */
if (!data->fractional)
  maxh += (prad + 2.0);

/* compute step size required */
step[0] = 1.0 / (grid-1);
step[1] = 1.0 / (grid-1);
step[2] = 1.0 / (grid-1);

/* ab is scan plane - c reps surface 'height' values */
#if DEBUG_CALCCON
printf("a limits: %9.2f (%9.2f)\n",min[ix[0]],step[ix[0]]);
printf("b limits: %9.2f (%9.2f)\n",min[ix[1]],step[ix[1]]);
printf("c limits: %9.2f to %9.2f\n",min[ix[2]],maxh);
#endif

/* scan the required plane */
k=0;
ta=tb=0;
for (i=0 ; i<grid ; i++)
  {
  for (j=0 ; j<grid ; j++)
    {
/* generate actual grid point coordinates */
    p[0] = (gfloat) i*step[ix[0]];
    p[1] = (gfloat) j*step[ix[1]];
    p[2] = 0.0;
    grid2cart(p,g);
    vecmat(data->latmat,g);
    g[0] += min[ix[0]];
    g[1] += min[ix[1]];
    g[2] += min[ix[2]];
/* find highest touched (periodic!) atom at this point */
    touch = REENTRANT;
    len = 1.0;
    for (n=0 ; n<data->num_atoms ; n++)
      {
      if ((data->atoms+n)->status & (DELETED | HIDDEN))
        continue;
/* get atom's data */
      rad = elements[(data->atoms+n)->atom_code].vdw;
/* do periodic images */
      for (ai=-1 ; ai<2 ; ai++)
        {
        for (bi=-1 ; bi<2 ; bi++)
          {
/* if not periodic - only do ai=bi=0 */
          if (!data->periodic)
            if (ai || bi)
              continue;
/* pbc */
          p[0] = (gfloat) ai;
          p[1] = (gfloat) bi;
          p[2] = 0.0;
          grid2cart(p,pos);
/* current atom's location */
          pos[0] += (data->atoms+n)->x;
          pos[1] += (data->atoms+n)->y;
          pos[2] += (data->atoms+n)->z;
/* latmat? */
/* apply lattice vectors */
          vecmat(data->latmat,pos); 
/* compute ab plane projected separation */
          da = g[0] - pos[ix[0]];
          db = g[1] - pos[ix[1]];
          r2 = rad*rad - da*da - db*db;
/* smaller than atom's radius? */
          if (r2 > 0.0)
            {
/* yes, calculate height on atom's surface (NB: assume +ve is UP) */
            height = pos[ix[2]] + sqrt(r2); 
            if (height > g[2])
              {
              g[2] = height;
              touch = n;
              len = rad;
/* save touched atom's translational info!!! */
              ta = (gfloat) ai;
              tb = (gfloat) bi;
              }
            }
          }
        }
      }
/* was an atom found? */
/* FIXME - hack to prevent gcc warning */
    pa=pb=pc=0.0;
    if (touch < 0)
      {
#if DEBUG_CALCCON
printf("Warning: hole found in structure. (%f,%f,%f)\n",g[0],g[1],g[2]);
#endif
      flag=1;
      }
    else
      {
/* pbc translation component of atom touched */
      p[0] = ta;
      p[1] = tb;
      p[2] = 0.0;
      grid2cart(p,pos);
/* calculate probe position require to touch point: pos[] */
      pos[0] += (data->atoms+touch)->x;
      pos[1] += (data->atoms+touch)->y;
      pos[2] += (data->atoms+touch)->z;
/* latmat? */
      vecmat(data->latmat,pos); 
/* vector from touched atom centre to touch pt */
      da = g[0] - pos[ix[0]];
      db = g[1] - pos[ix[1]];
      dc = g[2] - pos[ix[2]];
/* touch point */
      pa = g[0];
      pb = g[1];
      pc = g[2];
/* probe center coords */
      pa += da * prad / len;
      pb += db * prad / len;
      pc += dc * prad / len;
/* does probe intersect any other atoms? */
      flag=0;
      for (n=0 ; n<data->num_atoms ; n++)
        {
        if (n == touch)
          continue; 
        if ((data->atoms+n)->status & (DELETED | HIDDEN))
          continue;
/* check against n & surrounding images of n */
        for (ai=-1 ; ai<2 ; ai++)
          {
          for (bi=-1 ; bi<2 ; bi++)
            {
/* if not periodic - only do ai=bi=0 */
            if (!data->periodic)
              if (ai || bi)
                continue;
/* pbc */
            p[0] = (gfloat) ai;
            p[1] = (gfloat) bi;
            p[2] = 0.0;
            grid2cart(p,pos);
/* current atom's location */
            pos[0] += (data->atoms+n)->x;
            pos[1] += (data->atoms+n)->y;
            pos[2] += (data->atoms+n)->z;
/* latmat? */
/* apply lattice vectors */
            vecmat(data->latmat,pos); 
/* touching required distance (squared) */
            rad = elements[(data->atoms+n)->atom_code].vdw;
            td2 = (rad+prad)*(rad+prad);
/* actual separation */
            da = pa - pos[ix[0]];
            db = pb - pos[ix[1]];
            dc = pc - pos[ix[2]];
            d2 = da*da + db*db + dc*dc;
            if (d2 < td2)
              {
              flag=2;
              goto skip;
              }
            }    /* image scan */
          }      /* image scan */
        }        /* probe intersection scan */
      }          /* if probe touch */

/* was the probe's position ok? */
skip:;
    switch(flag)
      {
      case 0:
/* yes, record actual touching position */
        *(data->csurf.touch+k) = touch; 
        *(data->csurf.code+k) = (data->atoms+touch)->atom_code; 
        *(data->csurf.h+k) = g[2];
/* record probe's center - for post-scan */
        *(data->csurf.pa+k) = pa;
        *(data->csurf.pb+k) = pb;
        *(data->csurf.pc+k) = pc;
        break;
      case 1:
/* no - type hole */
        *(data->csurf.touch+k) = HOLE; 
        *(data->csurf.code+k) = -1;
        *(data->csurf.h+k) = min[ix[2]];
        break;
      case 2:
/* no - type 'normal' re-entrant surface */
        *(data->csurf.touch+k) = REENTRANT;
        *(data->csurf.code+k) = -1;
        break;
      }
/* next grid/surface point */
    k++;
    }  /* grid loop (j) */
  }    /* grid loop (i) */


/***********************/
/* post scan 1 - holes */
/***********************/
/* current grid point index */
k=0;
for (i=0 ; i<grid ; i++)
  {
  for (j=0 ; j<grid ; j++)
    {
/* only interested in holes */
    if (*(data->csurf.touch+k) == HOLE)
      {
      p[0] = (gfloat) i*step[ix[0]];
      p[1] = (gfloat) j*step[ix[1]];
      p[2] = 0.0;
      grid2cart(p,g);
      vecmat(data->latmat,g);
      g[0] += min[ix[0]];
      g[1] += min[ix[1]];
      g[2] += min[ix[2]];
/* scan loop */
      flag=0;
      for (n=0 ; n<data->num_atoms ; n++)
        {
        if ((data->atoms+n)->status & (DELETED | HIDDEN))
          continue;
/* check against n & surrounding images of n */
        for (ai=-1 ; ai<2 ; ai++)
          {
          for (bi=-1 ; bi<2 ; bi++)
            {
/* if not periodic - only do ai=bi=0 */
            if (!data->periodic)
              if (ai || bi)
                continue;
/* pbc */
            p[0] = ai;
            p[1] = bi;
            p[2] = 0.0;
            grid2cart(p,pos);
/* latmat? */
/* apply lattice vectors */
            vecmat(data->latmat,pos); 
/* current atom's location */
            pos[0] += (data->atoms+n)->x;
            pos[1] += (data->atoms+n)->y;
            pos[2] += (data->atoms+n)->z;
/* touching required distance (squared) */
            rad = elements[(data->atoms+n)->atom_code].vdw;
            td2 = (rad+prad)*(rad+prad);
/* dist to grid point */
            da = pos[ix[0]] - g[0];
            db = pos[ix[1]] - g[1];
/* close enough? */
            r2 = td2 - da*da - db*db;
            if (r2 > 0.0)
              {
/* bottom of probe surface */
              height = pos[ix[2]] + sqrt(r2);
/* is it the lowest so far? */
              if (height > g[2])
                {
                g[2] = height; 
                flag++;
                }
              }
            }  /* pbc loop bi */
          }    /* pbc loop ai */
        }      /* search loop n*/
/* record - was a 'proper' height found? */
      if (flag)
        {
/* this is a special hole */
        *(data->csurf.touch+k) = VALLEY;
        *(data->csurf.code+k) = -1;
        *(data->csurf.pa+k) = g[0];
        *(data->csurf.pb+k) = g[1];
        *(data->csurf.pc+k) = g[2];
        *(data->csurf.h+k) = g[2] - prad;
        }
      }        /* if hole */
    k++;
    }          /* grid loop j */
  }            /* grid loop i */

/************************************/
/* post scan 2 - re-entrant surface */
/************************************/
/* current grid point index */
k=0;
num_exp=0;
atom_exp = (gint *) g_malloc(data->num_atoms * sizeof(gint));
qa=qb=qc=0.0;           
for (i=0 ; i<grid ; i++)
  {
  for (j=0 ; j<grid ; j++)
    {
/* if contact */
    if (*(data->csurf.touch+k) > 0)
      {
      flag=0;
      for(m=0 ; m<num_exp ; m++)
        if (*(atom_exp+m) == *(data->csurf.touch+k))
          flag++;
/* already recorded as exposed? */
      if (!flag)
        {
        *(atom_exp+num_exp) = *(data->csurf.touch+k);
        num_exp++;
        }
      }

/* skip if not re-entrant (ie -1) */
    if (*(data->csurf.touch+k) == REENTRANT)
      {
      p[0] = (gfloat) i*step[ix[0]];
      p[1] = (gfloat) j*step[ix[1]];
      p[2] = 0.0;
      grid2cart(p,g);
      vecmat(data->latmat,g);
      g[0] += min[ix[0]];
      g[1] += min[ix[1]];
      g[2] = 10.0 + maxh;
/* scan for contact surfaces close enough to generate a height */
/* and record the lowest of these - n is scan index */
      flag=0;
flag2=0;
      for (n=0 ; n<grid*grid ; n++)
        {
/* only want surfaces with attached probe centers */
        if (*(data->csurf.touch+n) == REENTRANT ||
            *(data->csurf.touch+n) == HOLE)
          continue;
/* check against n & surrounding images of n */
        for (ai=-1 ; ai<2 ; ai++)
          {
          for (bi=-1 ; bi<2 ; bi++)
            {
/* if not periodic - only do ai=bi=0 */
            if (!data->periodic)
              if (ai || bi)
                continue;
/* required translation */
    p[0] = (gfloat) ai; 
    p[1] = (gfloat) bi; 
    p[2] = 0.0;
/* decode to cartesian */
    grid2cart(p,x);
/* apply lattice vectors */
    vecmat(data->latmat,x); 
/* generate PBC xlat'd probe center coords */
    pa = x[ix[0]] + *(data->csurf.pa+n);
    pb = x[ix[1]] + *(data->csurf.pb+n);
    pc = x[ix[2]] + *(data->csurf.pc+n);
/* decode */
/*
    p[0] += *(data->csurf.pa+n);
    p[1] += *(data->csurf.pb+n);
    p[2] += *(data->csurf.pc+n);
    grid2cart(&p[0], &x[0]);
*/
/* dist to grid point */
            da = pa - g[0];
            db = pb - g[1];
/* close enough? */
            td2 = prad*prad - da*da - db*db;
            if (td2 > 0.0)
              {
flag2++;
/* bottom of probe surface */
              height = pc - sqrt(td2);
/* is it the lowest so far? */
              if (height < g[2])
                {
                g[2] = height; 
/* candidate probe centre for this re-entrant surface pt. */
                qa = pa;
                qb = pb;
                qc = pc;
                flag=1;
                }
              }
            }  /* pbc loop bi */
          }    /* pbc loop ai */
        }      /* search loop n*/

/* record - was a 'proper' height found? */
      if (flag)
        {
        *(data->csurf.h+k) = g[2];  /* yes */
        *(data->csurf.pa+k) = qa;
        *(data->csurf.pb+k) = qb;
        *(data->csurf.pc+k) = qc;
        }
      else
        {
#if DEBUG_CALCCON
printf("Warning, no valid probe positions (%d candidates).\n",flag2);
#endif
        *(data->csurf.touch+k) = HOLE;
        *(data->csurf.h+k) = min[ix[2]];
        }
      }      /* if re-entrant */
/* next grid point */
    k++;
    }        /* grid loop j */
  }          /* grid loop i */

#if DEBUG_CALCCON
/* touch grid putput */
/*
k=0;
for (i=0 ; i<grid ; i++)
  {
  for (j=0 ; j<grid ; j++)
    {
    touch = *(data->csurf.touch+k);
    if (touch < 0)
      {
      printf("*   ");
      }
    else
      {
      printf("%3d ",touch);
      }
    k++;
    }
  printf("\n");
  }
*/
#endif

/*************************************/
/* post scan 3 - surface integration */
/*************************************/
area_exp = (gfloat *) g_malloc(num_exp*sizeof(gfloat));
area_pac = (gfloat *) g_malloc(num_exp*sizeof(gfloat));
for (i=0 ; i<num_exp ; i++)
  {
  *(area_exp+i) = 0.0;
  *(area_pac+i) = 0.0;
  }
flag=0;
atot=0.0;

/* project scan plane (ab) surface area */
/* a vector */
a[0] = a[1] = a[2] = 0.0;
a[ix[0]] = 1.0;

vecmat(data->latmat,a);
/* b vector */
b[0] = b[1] = b[2] = 0.0;
b[ix[1]] = 1.0;


vecmat(data->latmat,b);
/* get area via cross prod */
crossprod(g,a,b);
gsa = psa = VEC3MAG(g);

/* grid point ab projected surace area */
gsa /= (grid-1)*(grid-1);

k=0;
for (i=0 ; i<grid ; i++)
  {
  for (j=0 ; j<grid ; j++)
    {
/* skip if true hole */
    if (*(data->csurf.touch+k) == HOLE)
      {
      k++;
      flag++;
      continue;
      }
/* get surface pt. (touch) */
    p[0] = (gfloat) i*step[ix[0]];
    p[1] = (gfloat) j*step[ix[1]];
    p[2] = 0.0;
    grid2cart(p,g);
    g[0] += min[ix[0]];
    g[1] += min[ix[1]];
    g[2] = *(data->csurf.h+k);
    vecmat(data->latmat,g);
/* get associated probe centre */
    p[0] = *(data->csurf.pa+k);
    p[1] = *(data->csurf.pb+k);
    p[2] = *(data->csurf.pc+k);
    grid2cart(p,x);
/* surface normal is vec from touch to probe centre */
    p[0] = x[0] - g[0];
    p[1] = x[1] - g[1];
    p[2] = x[2] - g[2];
/* often a good check */
#if DEBUG_CALCCON
    r2 = sqrt(p[0]*p[0] + p[1]*p[1] + p[2]*p[2]);
    if (fabs(r2-prad) > 0.01)
      printf("Error: bad probe center coordinates (k=%i,dist=%f).\n",k,r2);
#endif

/* normal should be normalized, so OpenGL doesn't dim the surface colour */
    VEC3MUL(p, 1.0/prad);

/* CURRENT - store normal */
    *(data->csurf.n[0]+k) = p[0];
    *(data->csurf.n[1]+k) = p[1];
    *(data->csurf.n[2]+k) = p[2];
/* get angle between this and the verticle normal (ie c) */
    g[0] = 0.0;
    g[1] = 0.0;
    g[2] = 1.0;
    grid2cart(g,x);
/* compute molec. surf. area at this grid point */
    angle = via(&p[0],&x[0],3);
    gamma = fabs(cos(angle));
/* FIXME - crude hack for (near) vertical surface element */
    if (gamma > 0.01)
      area = gsa / gamma;
    else
      area = gsa / 0.01;
/* if contact, assign area to assoc. atom */
    for (n=0 ; n<num_exp ; n++)
      if (*(data->csurf.touch+k) == *(atom_exp+n))
        *(area_exp+n) += area;
/* add to total area */
    atot += area;
/* next grid point */
    k++;
    }
  }

/* output status/summary */
if (data->csurf.rsf)
  {
  if (flag)
    printf("Area calculation warning: surface has %d holes.\n",flag);
  printf("Found %d exposed atoms.\n",num_exp);
/*
  printf(" Projected surface area: %f\n",psa);
*/
  printf("   Net VdW surface area: %f\n",atot);
/* POSE - % of self exposed */
/* compute % accessibility */
  area=0.0;
  for (n=0 ; n<num_exp ; n++)
    {
/* atom index */
    i = *(atom_exp+n);
/* element reference */
    j = (data->atoms+i)->atom_code;
    rad = elements[j].vdw;
    pa = 4.0*PI*rad*rad;
    if (pa > 0.001)
      ta = *(area_exp+n)/pa;
    else
      ta = 0.0;
/* percentage accessibility */
    *(area_pac+n) = 100.0*ta;
/* net contact surface */
    area += *(area_exp+n);
    }

/* further summary */
  printf("   Contact surface area: %f\n",area);
  printf("Re-entrant surface area: %f\n",atot-area);

/* sort the accessibilities */
  swap=1;
  while(swap)
    {
    swap=0;
    for (n=0 ; n<num_exp-1 ; n++)
      {
/* if order is bad - swap */
      if (*(area_pac+n) < *(area_pac+n+1))
        {
        ta = *(area_pac+n);
        tb = *(area_exp+n);
        i = *(atom_exp+n);
        *(area_pac+n) = *(area_pac+n+1);
        *(area_exp+n) = *(area_exp+n+1);
        *(atom_exp+n) = *(atom_exp+n+1);
        *(area_pac+n+1) = ta;
        *(area_exp+n+1) = tb;
        *(atom_exp+n+1) = i;
        swap++;
        }
      }
    }
/* print them out */
  for (n=0 ; n<num_exp ; n++)
    {
    i = *(atom_exp+n);
    printf("[%2s] %5.2f%% of atom accessible (raw area %f)\n",
          (data->atoms+i)->element, *(area_pac+n), *(area_exp+n));
    }
  }

/* setup for display */
data->csurf_on = TRUE;
data->csurf.grid = grid;
data->csurf.ix[0] = ix[0];
data->csurf.ix[1] = ix[1];
data->csurf.ix[2] = ix[2];
data->csurf.min[0] = min[0];
data->csurf.min[1] = min[1];
data->csurf.min[2] = min[2];
data->csurf.step[0] = step[0];
data->csurf.step[1] = step[1];
data->csurf.step[2] = step[2];
/* calculate pixel coords & redraw */
calc_coords(REFRESH, data);
cent_coords(data);
calc_coords(REFRESH, data);
redraw_canvas(SINGLE);

/* status */
if (!data->periodic)
  show_text("Warning: non periodic model, surface may contain edge artifacts.\n");
else
  show_text("2D vdW surface generated.");

/* cleanup & exit */
g_free(atom_exp);
g_free(area_exp);
g_free(area_pac);
return;
}

/***************************/
/* global spin update hook */
/***************************/
gint change_spin(GtkWidget *w, gpointer *spin)
{
gint id;
gfloat r;
struct model_pak *data=NULL;

/* checks & setup */
g_return_val_if_fail(spin != NULL, FALSE);
r = SPIN_FVAL(GTK_SPIN_BUTTON(spin));
data = (struct model_pak *) gtk_object_get_data(GTK_OBJECT(spin), "ptr");
g_return_val_if_fail(data != NULL, FALSE);
id = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(spin), "id"));

/* mode dependent spinner update */
switch(id)
  {
/* contact surface */
  case GRID:
    data->csurf.grid = r;
    break;
  case OFFSET:
    data->csurf.offset = r;
    init_objs(CENT_COORDS, data);
    redraw_canvas(SINGLE);
    break;
/* update for both (FIXME - problems with this?) */
  case PROBE:
    data->csurf.prad = r;
    data->mysurf[0].prad = r;
    break;
  case SURF_ACCURACY:
    data->mysurf[0].shell_size = (gint) r;
    break;
  default:
    printf("unknown mode.\n");
  }

return(FALSE);
}

/********************/
/* simple hide hook */
/********************/
void hide_con(struct model_pak *data)
{
data->csurf_on = FALSE;
redraw_canvas(SINGLE);
}

/**************************/
/* results summary toggle */
/**************************/
void rsf_toggle(struct model_pak *data)
{
data->csurf.rsf ^= 1;
}

/*********************/
/* select scan plane */
/*********************/
void switch_plane(gint plane)
{
switch(plane)
  {
  case PLANE_XY:
    ix[0] = 0;
    ix[1] = 1;
    ix[2] = 2;
    break;
  case PLANE_YZ:
    ix[0] = 1;
    ix[1] = 2;
    ix[2] = 0;
    break;
  case PLANE_XZ:
    ix[0] = 0;
    ix[1] = 2;
    ix[2] = 1;
    break;
  }
}

/***************************/
/* Molecular surface setup */
/***************************/
void surf_dialog()
{
gint id;
GSList *surf_plane;
GtkWidget *frame, *vbox, *button, *spin;
GtkAdjustment *adj;
struct dialog_pak *surf_dialog;
struct model_pak *data;

/* checks */
if (!sysenv.num_models)
  return;
if ((id = request_dialog(sysenv.active, SURF)) < 0)
  return;
surf_dialog = &sysenv.dialog[id];

/* get the model data for all subsequent operations */
data = model_ptr(sysenv.active, RECALL);
g_return_if_fail(data != NULL);

/* default values */
data->csurf.grid = 40;        /* square grid side length (points) */
data->csurf.offset = 2.0;     /* drawing offset (Angs) */
data->csurf.prad = 0.97;      /* probe radius (Angs) */
data->csurf.rsf = FALSE;         /* results summary flag */

/* create dialog */
surf_dialog->win = gtk_dialog_new();
gtk_signal_connect(GTK_OBJECT (surf_dialog->win), "destroy",
                   GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);
gtk_window_set_title(GTK_WINDOW(surf_dialog->win), "VdW surface");

/* radio buttons - scan which surface (xy,yz,xz) */
frame = gtk_frame_new ("Scan plane");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
vbox = gtk_hbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
/* do the first (DEFAULT MODE) radio button */
button = gtk_radio_button_new_with_label (NULL, "xy");
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (switch_plane),
                         (gpointer) PLANE_XY);
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);

/* make a radio group */
surf_plane = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
/* do the next button */
button = gtk_radio_button_new_with_label (surf_plane, "yz");
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_plane),
                           (gpointer) PLANE_YZ);
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);
/* subsequent button... */
button = gtk_radio_button_new_with_label(
         gtk_radio_button_group (GTK_RADIO_BUTTON (button)), "xz");
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (switch_plane),
                           (gpointer) PLANE_XZ);
gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, TRUE, 0);

/* grid side length */
frame = gtk_frame_new ("Grid side length");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
vbox = gtk_hbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
/* create spin box */
adj = (GtkAdjustment *) gtk_adjustment_new(data->csurf.grid, 5, 1000, 5, 10, 0);
spin = gtk_spin_button_new(adj, 0, 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), TRUE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX(vbox), spin, FALSE, TRUE,0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(change_spin), (gpointer) spin);
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) GRID);
gtk_object_set_data(GTK_OBJECT(spin), "ptr", (gpointer) data);

/* surface drawing offset */
frame = gtk_frame_new ("Drawing offset");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->vbox),
                   frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
vbox = gtk_hbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
/* create spin box */
adj = (GtkAdjustment *) gtk_adjustment_new
                        (data->csurf.offset, 0.0, 10.0, 0.5, 1.0, 0);
spin = gtk_spin_button_new(adj, 0.5, 2);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_box_pack_start (GTK_BOX (vbox), spin, FALSE, TRUE,0);
/* update hook */
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(change_spin), (gpointer) spin);
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) OFFSET);
gtk_object_set_data(GTK_OBJECT(spin), "ptr", (gpointer) data);

/* probe type */
frame = gtk_frame_new ("Probe radius");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
vbox = gtk_hbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
adj = (GtkAdjustment *) gtk_adjustment_new
                        (data->csurf.prad, 0.1, 10.0, 0.01, 0.05, 0);
spin = gtk_spin_button_new(adj, 0.01, 2);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX (vbox), spin, FALSE, TRUE,0);
/* update hook */
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(change_spin), (gpointer) spin);
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) PROBE);
gtk_object_set_data(GTK_OBJECT(spin), "ptr", (gpointer) data);

/* re-entrant surface flag */
frame = gtk_frame_new (NULL);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->vbox),
                   frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
vbox = gtk_hbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
button = gtk_check_button_new_with_label ("Print results summary");
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (rsf_toggle), (gpointer) data);
gtk_box_pack_start(GTK_BOX (vbox), button, TRUE, TRUE, 0);

/* make, hide, close - terminating buttons */
button = gtk_button_new_with_label ("Make");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->action_area),
                                    button, FALSE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                          GTK_SIGNAL_FUNC(calc_surf), (gpointer) data);

button = gtk_button_new_with_label ("Hide");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->action_area),
                                    button, FALSE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                          GTK_SIGNAL_FUNC(hide_con), (gpointer) data);

button = gtk_button_new_with_label ("Close");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->action_area),
                                    button, FALSE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                          GTK_SIGNAL_FUNC(close_dialog),
                         (gpointer) id);
/* done */
gtk_widget_show_all(surf_dialog->win);
}

/*******************************************/
/* calc 3D molecular/solvent acc. surfaces */
/*******************************************/
#define DEBUG_MYSURF 1
void mysurf(struct model_pak *data)
{
gint i, j, k, n, s;
gint num_unique, max, strips, nump, flag;
gint *unique_list;
gfloat cx, cy, cz, dx, dy, dz, da, r, theta, phi, r2, rcut;
gfloat x[3], p[3], xj[3];
gfloat div, dtheta, dphi, st, ct;
struct sphere_pak *spheres;

#if DEBUG_MYSURF
printf("scanning: %s\n",data->filename);
#endif

/* scan for all unique atom codes */
num_unique = 0;
unique_list = g_malloc(data->num_atoms * sizeof(gint));
for (i=0 ; i<data->num_atoms ; i++)
  {
  flag=0;
  for (j=0 ; j<num_unique ; j++)
    {
    if (*(unique_list+j) == (data->atoms+i)->atom_code)
      {
      flag++;
      break;
      }
    }
  if (!flag)
    {
    *(unique_list+num_unique) = (data->atoms+i)->atom_code;
    num_unique++;
    }
  }

#if DEBUG_MYSURF
printf("Found %d unique atom codes: ",num_unique);
for (i=0 ; i<num_unique ; i++)
  printf("%d ",*(unique_list+i));
printf("\n");
#endif

/* TODO - triangle tessellation for more uniform dot density */

/* split evenly between theta & phi */
/* FIXME - shouldn't da depend on r (vdw radius) ? */
/* actually, want shell_size to vary, with a const da for all r */
da = 4.0*PI/data->mysurf[0].shell_size;
dtheta = sqrt(da);
strips = (gint) (PI/dtheta);
dtheta = PI/strips;

/* magic number - how to properly calc? */
/* ie sum of DISCRETE vals of sin */
div = 2.0 / dtheta;

#if DEBUG_MYSURF
/* these may need adjustment (bit of a fudge) */
printf("sphere strips = %d\n",strips);
printf("div = %f\n",div);
#endif

/*  calc actual num of monopoles */
max = 0;
for (i=0 ; i<=strips ; i++)
  {
  theta = i*dtheta;
/* number of points for this value of theta */
  nump = (gint) (data->mysurf[0].shell_size * sin(theta) / div);
  if (nump < 1) 
    nump = 1;
  max += nump;
  }

#if DEBUG_MYSURF
printf("Requested shell points: %d, actual points: %d\n",data->mysurf[0].shell_size,max);
#endif

/* digitize spheres (each radius) at required density */
spheres = g_malloc(num_unique*sizeof(struct sphere_pak));

for (n=0 ; n<num_unique ; n++)
  {
  r = elements[*(unique_list+n)].vdw;
/* alloc for sphere points */
  (spheres+n)->x = g_malloc(max*sizeof(gfloat)); 
  (spheres+n)->y = g_malloc(max*sizeof(gfloat)); 
  (spheres+n)->z = g_malloc(max*sizeof(gfloat)); 
/* index for sphere points */
  k = 0;
  for (i=0 ; i<=strips ; i++)
    {
    theta = i*dtheta;
    st = sin(theta);
    ct = cos(theta);
/* number of points for this value of theta */
    nump = (gint) (data->mysurf[0].shell_size * st) / div;
    if (!nump)
      nump = 1;
    dphi = 2.0 * PI / nump;
    for (j=0 ; j<nump ; j++)
      {
      phi = j*dphi;
/* convert spherical coords to cartesian */
      (spheres+n)->x[k] = (gfloat) r*st*cos(phi);
      (spheres+n)->y[k] = (gfloat) r*st*sin(phi);
      (spheres+n)->z[k] = (gfloat) r*ct;
      k++;
      }
    }
  (spheres+n)->num_points = k;
#if DEBUG_MYSURF
/* allocation check - FIXME - why do we only sometimes get a miscalc? */
  if (k != max)
    printf("mysurf() warning: sphere mis-allocation.\n");
  printf("digitized sphere %d: points (%d/%d)\n",n,k,max);
#endif
  }

/* TODO - construct near neighbour lists - elim unecessary comparisons */

/* upper bound on memory */
max *= data->num_atoms;
data->mysurf[SOLSURF].num_points = 0;
data->mysurf[SOLSURF].max_points = max;
data->mysurf[SOLSURF].touch = g_malloc(max * sizeof(gint));
data->mysurf[SOLSURF].px = g_malloc(max * sizeof(gint));
data->mysurf[SOLSURF].py = g_malloc(max * sizeof(gint));
data->mysurf[SOLSURF].x = g_malloc(max * sizeof(gfloat));
data->mysurf[SOLSURF].y = g_malloc(max * sizeof(gfloat));
data->mysurf[SOLSURF].z = g_malloc(max * sizeof(gfloat));
data->mysurf[MOLSURF].num_points = 0;
data->mysurf[MOLSURF].max_points = max;
data->mysurf[MOLSURF].touch = g_malloc(max * sizeof(gint));
data->mysurf[MOLSURF].px = g_malloc(max * sizeof(gint));
data->mysurf[MOLSURF].py = g_malloc(max * sizeof(gint));
data->mysurf[MOLSURF].x = g_malloc(max * sizeof(gfloat));
data->mysurf[MOLSURF].y = g_malloc(max * sizeof(gfloat));
data->mysurf[MOLSURF].z = g_malloc(max * sizeof(gfloat));

/* pass 1 - contact surface */
k=0;
for (i=0 ; i<data->num_atoms ; i++)
  {
/* find the right digitized sphere */
  s=0;
  while (s<num_unique)
    if (*(unique_list+s++) == (data->atoms+i)->atom_code)
      break;
  s--;
/* loop over all points */
  for (n=0 ; n<(spheres+s)->num_points ; n++)
    {
/* current point */
    x[0] = (data->atoms+i)->x;
    x[1] = (data->atoms+i)->y;
    x[2] = (data->atoms+i)->z;
    vecmat(data->latmat, x);
    p[0] = x[0] + (spheres+s)->x[n];
    p[1] = x[1] + (spheres+s)->y[n];
    p[2] = x[2] + (spheres+s)->z[n];
/* is this point contained by any other atom? */
/* TODO - near neighbour lists to reduce this loop */
    flag=0;
    for (j=0 ; j<data->num_atoms ; j++)
      {
      xj[0] = (data->atoms+j)->x;
      xj[1] = (data->atoms+j)->y;
      xj[2] = (data->atoms+j)->z;
      vecmat(data->latmat, xj);
      dx = p[0] - xj[0];
      dy = p[1] - xj[1];
      dz = p[2] - xj[2];
      r2 = dx*dx + dy*dy + dz*dz; 
      rcut = elements[(data->atoms+j)->atom_code].vdw - POSITION_TOLERANCE;
      if (r2 < rcut*rcut)
        {
        flag++;
        break;
        }
      }
/* if not contained by another atom */
    if (!flag)
      {
/* vector from atom to molsurf, normalize */
      cx = p[0] - x[0];
      cy = p[1] - x[1];
      cz = p[2] - x[2];
/* normalize - TODO - maybe precalc & mult by 1/r (mult is faster than div) */
      cx /= elements[(data->atoms+i)->atom_code].vdw;
      cy /= elements[(data->atoms+i)->atom_code].vdw;
      cz /= elements[(data->atoms+i)->atom_code].vdw;
/* scale with probe radius */
      cx *= data->mysurf[0].prad;
      cy *= data->mysurf[0].prad;
      cz *= data->mysurf[0].prad;
/* get probe centre (solsurf) */
      cx += p[0];
      cy += p[1];
      cz += p[2];
/* does it overlap any other atom? */
      flag=0;
      for (j=0 ; j<data->num_atoms ; j++)
        {
        xj[0] = (data->atoms+j)->x;
        xj[1] = (data->atoms+j)->y;
        xj[2] = (data->atoms+j)->z;
        vecmat(data->latmat, xj);

        dx = cx - xj[0];
        dy = cy - xj[1];
        dz = cz - xj[2];
        r2 = dx*dx + dy*dy + dz*dz; 
rcut = elements[(data->atoms+j)->atom_code].vdw + data->mysurf[0].prad - POSITION_TOLERANCE;
        if (r2 < rcut*rcut)
          {
          flag++;
          break;
          }
        }
      if (!flag)
        {
        data->mysurf[SOLSURF].touch[k] = i;
        data->mysurf[SOLSURF].x[k] = cx;
        data->mysurf[SOLSURF].y[k] = cy;
        data->mysurf[SOLSURF].z[k] = cz;
        data->mysurf[SOLSURF].num_points++;
        data->mysurf[MOLSURF].touch[k] = i;
        data->mysurf[MOLSURF].x[k] = p[0];
        data->mysurf[MOLSURF].y[k] = p[1];
        data->mysurf[MOLSURF].z[k] = p[2];
        data->mysurf[MOLSURF].num_points++;
        k++;
        }
      }
    }
  }

/* TODO */
/* cube fill may use solvent acc. probe positions for the re-entrant */
/* part of the calculation (ie restrict w/all probes touching >1 atom) */
#if EXP
grid = grid_setup(test if perdiodic)
for (n : grid)
  {
  x = n/
  y = n/ - x
  z = n/ - x - y
  if (pos contained by vdw)
    create_new_vertex();
  }
/* neighbours indicate bulk/surface boundary */
/* want imediate neighbours & next nearest (eg across body diag) */
create_adjacency_list();
for (num iterations)
  {
/* num vertices = num on one side (n) cubed */
/* if double then we have due to edges (2n-1) cubed vertices */
/* thus (2n-1)^3 - n^3 NEW vertices */
  subdiv vertices(x2);
  for all unique i,j vertices (neighbours/next nearest)
    create_new_vertex(at midpoint);
  update_adjacency_list();
  }
#endif

/* clean up */
for (i=0 ; i<num_unique ; i++)
  {
  g_free((spheres+i)->x);
  g_free((spheres+i)->y);
  g_free((spheres+i)->z);
  }
g_free(spheres);
g_free(unique_list);

/* setup */
init_objs(CENT_COORDS, data);

/* draw */
redraw_canvas(SINGLE);
}

/********************************/
/* surface type display toggles */
/********************************/
void toggle_molsurf(gpointer *button)
{
struct model_pak *data;

data = (struct model_pak *) gtk_object_get_data(GTK_OBJECT(button), "ptr");
data->molsurf_on ^= 1;
/* redraw */
init_objs(CENT_COORDS, data);
redraw_canvas(SINGLE);
}

void toggle_solsurf(gpointer *button)
{
struct model_pak *data;

data = (struct model_pak *) gtk_object_get_data(GTK_OBJECT(button), "ptr");
data->solsurf_on ^= 1;
/* redraw */
init_objs(CENT_COORDS, data);
redraw_canvas(SINGLE);
}

/******************************************************************/
/* dot sphere molecular and solvent accessible surface calculator */
/******************************************************************/
void surf2_dialog(void)
{
gint id;
GtkWidget *frame, *vbox, *hbox, *button, *spin;
GtkAdjustment *adj;
struct dialog_pak *surf_dialog;
struct model_pak *data;

/* checks */
if (!sysenv.num_models)
  return;
if ((id = request_dialog(sysenv.active, SURF)) < 0)
  return;
surf_dialog = &sysenv.dialog[id];

/* get the model data for all subsequent operations */
data = model_ptr(sysenv.active, RECALL);
g_return_if_fail(data != NULL);

/* default parameters */
/* NB: MOLSURF *must* use these as well */
data->mysurf[SOLSURF].shell_size = 1000; /* shell digitization */
data->mysurf[SOLSURF].prad = 0.97;       /* probe radius (Angs) */

/* create dialog */
surf_dialog->win = gtk_dialog_new();
gtk_signal_connect(GTK_OBJECT (surf_dialog->win), "destroy",
                   GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);
gtk_window_set_title(GTK_WINDOW(surf_dialog->win), "VdW surface");

/* number of points to digitize the spheres with */
frame = gtk_frame_new ("Dot sphere size");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
hbox = gtk_hbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), hbox);
/* create spin box */
adj = (GtkAdjustment *) gtk_adjustment_new
      (data->mysurf[0].shell_size, 100, 2000, 100, 100, 0);
spin = gtk_spin_button_new(adj, 0, 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, TRUE,0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(change_spin), (gpointer) spin);
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) SURF_ACCURACY);
gtk_object_set_data(GTK_OBJECT(spin), "ptr", (gpointer) data);

/* probe size */
frame = gtk_frame_new ("Probe radius");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
hbox = gtk_hbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), hbox);
adj = (GtkAdjustment *) gtk_adjustment_new
                        (data->mysurf[0].prad, 0.1, 10.0, 0.01, 0.05, 0);
spin = gtk_spin_button_new(adj, 0.01, 2);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX (hbox), spin, FALSE, TRUE,0);
/* update hook */
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(change_spin), (gpointer) spin);
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) PROBE);
gtk_object_set_data(GTK_OBJECT(spin), "ptr", (gpointer) data);

/* surface display type toggles */
frame = gtk_frame_new("Surface type");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
vbox = gtk_vbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), vbox);
button = gtk_check_button_new_with_label ("Contact (TODO re-entrant)");
gtk_box_pack_end (GTK_BOX (vbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (toggle_molsurf),
                           (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);

button = gtk_check_button_new_with_label (" Solvent accessible ");
gtk_box_pack_end (GTK_BOX (vbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (toggle_solsurf),
                           (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "ptr", (gpointer) data);
/* default - on */
gtk_signal_emit_by_name (GTK_OBJECT (button), "clicked");

/* make, hide, close - terminating buttons */
button = gtk_button_new_with_label ("Calc");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->action_area),
                                    button, FALSE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                          GTK_SIGNAL_FUNC(mysurf), (gpointer) data);

button = gtk_button_new_with_label ("Close");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surf_dialog->win)->action_area),
                                    button, FALSE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                          GTK_SIGNAL_FUNC(close_dialog),
                         (gpointer) id);

/* done */
gtk_widget_show_all(surf_dialog->win);
}

/*************************/
/* construct the surface */
/*************************/
#define MAKE_SURFACE_DEBUG 0
void make_surface(struct surface_pak *surfdat)
{
gint i, new, r1size;
gfloat sbe;
struct model_pak *data, *surf;

/* checks */
data = model_ptr(surfdat->model, RECALL);
g_return_if_fail(data != NULL);
if (data->periodic != 3)
  {
  show_text("base model is of wrong type.");
  return;
  }

#if MAKE_SURFACE_DEBUG
printf("Generating surface for model: %d\n", surfdat->model);
#endif

/* NEW - complete molecules */
/* unfrag must preserve core-shell links */
unfragment(data);
init_objs(REFRESH_COLOUR, data);
init_objs(CENT_COORDS, data);
calc_bonds(data);
calc_mols(data);

/* allocate & init for surface data */
new = sysenv.num_models;
surf = model_ptr(new, ASSIGN);
g_return_if_fail(surf != NULL);
surf->mode = FREE;
surf->id = GULP;
surf->periodic = 2;
surf->fractional = FALSE;

/* transfer appropriate GULP setup data to new model */
copy_gulp_data(data, surf);
copy_elem_data(data, surf);

surf->gulp.run = E_SINGLE;
surf->gulp.method = CONV;

/* transfer surface data to new model */
/* NB: memcpy would produce undesired effects if pointers were */
/* later added to the surface struct */
ARR3SET(surf->surface.miller, surfdat->miller);
surf->surface.shift = surfdat->shift;
surf->surface.region[0] = surfdat->region[0];
surf->surface.region[1] = surfdat->region[1];
/* setup for region display */
if (surf->surface.region[0])
  surf->region_empty[REGION1A] = FALSE;
if (surf->surface.region[1])
  surf->region_empty[REGION2A] = FALSE;

#if MAKE_SURFACE_DEBUG
printf("Miller (%d,%d,%d)\n",surf->surface.miller[0]
                            ,surf->surface.miller[1]
                            ,surf->surface.miller[2]);
printf("Shift %f\n",surf->surface.shift);
printf("Region sizes (%d,%d)\n",surf->surface.region[0]
                               ,surf->surface.region[1]);
#endif

/* TODO - valid shift/region size loop??? */
generate_surface(data, surf);

/* model coord init */
surf->axes_on = TRUE;
surf->cell_on = TRUE;
init_objs(INIT_COORDS, surf);
/* we've made a new model - store the original cartesian coords */
save_cart(surf);
/* must be here, as latmat is not calculated until INIT_COORDS */
/* convert cartesian (x,y,z) to gdis fractional */
latmat_fudge(surf);
/* re-init after coordinate change */
init_objs(INIT_COORDS, surf);
calc_bonds(surf);
calc_mols(surf);
/* calculate the bulk energy needed for GULP surface calcs */
sbe = data->gulp.energy;
if (fabs(sbe) < FRACTION_TOLERANCE)
  {
  show_text("WARNING: suspicious bulk energy value.");
  }
/* calc the number of region 1 atoms */
r1size=0;
for (i=0 ; i<surf->num_atoms ; i++)
  if ((surf->atoms+i)->region == REGION1A)
    r1size++;
sbe /= data->num_orig;     /* E per atom in unit cell */
sbe *= r1size;             /* E scaled to region 1 size */
surf->gulp.sbulkenergy = sbe;

/* employ src file? */
new_tree_item(new, APPEND);
tree_select(new);
redraw_canvas(ALL);
}

/***************/
/* rank shifts */
/***************/
void zsort(gfloat *array, gint size)
{
gint i, swap;
gfloat tmp;

swap=1;
while (swap)
  {
  swap=0;
  for (i=1 ; i<size ; i++)
    {
    if (array[i-1] < array[i])
      {
      tmp = array[i-1];
      array[i-1] = array[i];
      array[i] = tmp;
      swap=1;
      }
    }
  }
}

/****************/
/* general sort */
/****************/
void sort(gfloat *array, gint *order, gint size)
{
gint i, swap;
gfloat tmp;

for (i=0 ; i<size ; i++)
  *(order+i) = i;

swap=1;
while (swap)
  {
  swap=0;
  for (i=1 ; i<size ; i++)
    {
    if (array[i-1] < array[i])
      {
/* swap elements in array */
      tmp = array[i-1];
      array[i-1] = array[i];
      array[i] = tmp;
/* register the change in order */
      tmp = order[i-1];
      order[i-1] = order[i];
      order[i] = tmp;
/* elements were swapped */
      swap++;
      }
    }
  }
}

/***************************************/
/* smallest common multiple for a pair */
/***************************************/
gint scf(gint a, gint b)
{
if (!b)
  return(abs(a));
return(scf(b, a%b));
}

/***************************************/
/* smallest common multiple for vector */
/***************************************/
gint simplify(gint *vec, gint size)
{
gint i, n;

n = scf(*vec, *(vec+1));  
for (i=2 ; i<size ; i++)
  n = scf(n, *(vec+i));  
return(n);
}

/**********************************/
/* rank (hkl) faces based on dhkl */
/**********************************/
/* TODO - this is now a BFDH morphology generator */
#define DEBUG_RANK_FACES 0
gint rank_faces(struct surface_pak *surfdat)
{
gint i, j, n, h, k, l, scm;
gint h_max, k_max, l_max, d_max;
gint *index, *miller_h, *miller_k, *miller_l, *family, *family_printed;
gint new, current_id, num_fam, div;
gint f1[3], f2[3];
gfloat *dhkl, scale;
gfloat vec[3];
struct model_pak *data, *surf;
FILE *fp;

/* FIXME - scan range -/+ */
h_max = k_max = l_max = 2;
d_max =  (2*h_max+1) * (2*k_max+1) * (2*l_max+1);
dhkl = g_malloc(d_max * sizeof(gfloat));
index = g_malloc(d_max * sizeof(gint));
miller_h = g_malloc(d_max * sizeof(gfloat));
miller_k = g_malloc(d_max * sizeof(gfloat));
miller_l = g_malloc(d_max * sizeof(gfloat));

/* checks */
data = model_ptr(surfdat->model, RECALL);
g_return_val_if_fail(data != NULL, 1);
if (data->periodic != 3)
  {
  show_text("base model is of wrong type.");
  return(1);
  }
/* new make reciprocal lattice vectors */
make_rlat(data);

/* NEW - complete molecules */
unfragment(data);
init_objs(REFRESH_COLOUR, data);
init_objs(CENT_COORDS, data);
calc_bonds(data);
calc_mols(data);

/* allocate & init for surface */
new = sysenv.num_models;
surf = model_ptr(new, ASSIGN);
g_return_val_if_fail(surf != NULL, 1);
surf->mode = FREE;
surf->id = GULP;
surf->periodic = 2;
surf->fractional = FALSE;

/* loop over required range */
n=0;
for (h=h_max ; h>=-h_max ; h--)
  {
  for (k=k_max ; k>=-k_max ; k--)
    {
    for (l=l_max ; l>=-l_max ; l--)
      {
/* skip (000) */
      if (!h && !k && !l)
        continue;

      VEC3SET(surf->surface.miller, h, k, l);
      scm = simplify(surf->surface.miller, 3);
/* skip things like (222) */
      if (scm != 1)
        continue;
/* get dhkl */
vec[0] = h;
vec[1] = k;
vec[2] = l;
vecmat(data->rlatmat, vec);
surf->surface.dspacing = 1.0/VEC3MAG(vec);

#if DEBUG_RANK_FACES
printf("new [%d]: (%d %d %d) dspacing = %f\n",n,h,k,l,surf->surface.dspacing);
#endif
/* save */
      *(miller_h+n) = h;
      *(miller_k+n) = k;
      *(miller_l+n) = l;
      *(dhkl+n) = surf->surface.dspacing;
      n++;

if (n >= d_max)
  {
  printf("Error: misalloc.\n");
  n = d_max;
  goto rank_faces_done;
  }

      }
    }
  }
rank_faces_done:;

/* label indicating facets of the same family */
family = g_malloc(n * sizeof(gint));
for (i=0 ; i<n ; i++)
  *(family+i) = -1;

/* label equivalent faces */
current_id = 0;
for (i=0 ; i<n ; i++)
  {
  if (*(family+i) == -1)
    {
    *(family+i) = current_id;
    f1[0] = *(miller_h+i);
    f1[1] = *(miller_k+i);
    f1[2] = *(miller_l+i);
    j=i;
    while(j<n)
      {
      f2[0] = *(miller_h+j);
      f2[1] = *(miller_k+j);
      f2[2] = *(miller_l+j);
      if (facet_equiv(data, f1, f2))
        *(family+j) = current_id;
      j++;
      }
    current_id++;
    }
  }
num_fam = current_id;

#if DEBUG_RANK_FACES
printf("Found %d families\n",num_fam);
#endif

/* family printing flag */
family_printed = g_malloc(num_fam * sizeof(gint));
for (i=0 ; i<num_fam ; i++)
  *(family_printed+i) = FALSE;

for (i=0 ; i<n ; i++)
  {
/* get family id for this hkl */
  current_id = *(family+i);
  if (*(family_printed+current_id))
    continue;
  else
    {
/* determine division factor for family by comparing with reflection normals */
    div = IsSysAbsent_hkl((T_SgInfo *) data->sginfo.raw, 
        *(miller_h+i), *(miller_k+i), *(miller_l+i), NULL);
    div = abs(div);
/* adjust all hkl's for this family */
    if (div)
      {
#if DEBUG_RANK_FACES
printf("Sys absence for [%d] (%d %d %d) : %d\n", i,
       *(miller_h+i),*(miller_k+i),*(miller_l+i),div);
#endif
      for (j=0 ; j<n ; j++)
        {
        if (current_id == *(family+j))
          {
/* layer adjustment due to reflection/glide planes */
          *(miller_h+j) *= div;
          *(miller_k+j) *= div;
          *(miller_l+j) *= div;
          *(dhkl+j) /= (gfloat) div;
          }
        }
      *(family_printed+current_id) = TRUE;
      }
    }
  }

/* rank according to dhkl */
sort(dhkl, index, n);
scale = *(dhkl);

/* CURRENT - create a morphology model */
/* write to a temporary file, then call morph_create(filename) */
fp = fopen("BFDH.gmf","wt");
if (!fp)
  {
  printf("Could not open file.\n");
  exit(0);
  }
fprintf(fp, "TITLE BFDH morphology\n");
/* NEW - cell lengths are 1 so that facet distances are kept */
fprintf(fp, "CELL %f %f %f  %f %f %f\n",
        data->pbc[0], data->pbc[1], data->pbc[2], 
        180.0*data->pbc[3]/PI, 180.0*data->pbc[4]/PI, 180.0*data->pbc[5]/PI);
fprintf(fp, "HKL dhkl\n");

/* display */
for (i=0 ; i<num_fam ; i++)
  *(family_printed+i) = FALSE;
for (i=0 ; i<n ; i++)
  {
/* get the index of a hkl - sorted by dhkl */
  j = *(index+i);
/* get family id for this hkl */
  current_id = *(family+j);
  if (*(family_printed+current_id))
    continue;
  else
    {
/* print all hkl's for this family */
#if DEBUG_RANK_FACES
    for (k=0 ; k<n ; k++)
      if (current_id == *(family+k))
        printf("(%d %d %d) ", *(miller_h+k), *(miller_k+k), *(miller_l+k));

    printf("\n   dspacing = %f, length = %f, ranking = %f\n",
    *(dhkl+i), 100.0/(*(dhkl+i)), scale/(*(dhkl+i)));
#endif

    for (k=0 ; k<n ; k++)
      if (current_id == *(family+k))
        fprintf(fp, "%3d %3d %3d    %10.4f\n", 
                     *(miller_h+k), *(miller_k+k), *(miller_l+k), *(dhkl+i));

    *(family_printed+current_id) = TRUE;
    }
  }
fclose(fp);


/* NEW */
fp = fopen("BFDH.gmf","r");
if (!fp)
  {
  show_text("Error: bad or missing filename.");
  return(1);
  }
if (load_gmf(fp, sysenv.num_models, "BFDH.gmf"))
  {
  show_text("Error: bad morphology file.");
  return(1);
  }
fclose(fp);

/* done, tidy & exit */
g_free(miller_h);
g_free(miller_k);
g_free(miller_l);
g_free(dhkl);
g_free(index);
g_free(family);
g_free(family_printed);

delete_model(new);
return(0);
}

/*************************/
/* search for valid cuts */
/*************************/
/* TODO - for a particular (hkl) or a range? */
/*      - option to create or just print the shifts */
/*      - options to converge regions */
#define MAX_CUTS 100
#define DEBUG_CALC_SHIFTS 1
gint calc_shifts(struct surface_pak *surfdat)
{
gint i, j;
gint num_cuts, num_shifts, old, new, ref, r1size;
gfloat sbe;
gfloat vec[3], zcut[MAX_CUTS], shift[MAX_CUTS];
struct model_pak *data, *surf;

/* checks */
data = model_ptr(surfdat->model, RECALL);
g_return_val_if_fail(data != NULL, 1);
if (data->periodic != 3)
  {
  show_text("base model is of wrong type.");
  return(1);
  }

#if MAKE_SURFACE_DEBUG
printf("Generating surface for model: %d\n", surfdat->model);
#endif

/* NEW - complete molecules */
unfragment(data);
init_objs(REFRESH_COLOUR, data);
init_objs(CENT_COORDS, data);
calc_bonds(data);
calc_mols(data);

/* allocate & init for surface */
new = sysenv.num_models;
surf = model_ptr(new, ASSIGN);
g_return_val_if_fail(surf != NULL, 1);
surf->mode = FREE;
surf->id = GULP;
surf->periodic = 2;
surf->fractional = FALSE;

/* surface bulk energy */
sbe = data->gulp.energy;
sbe /= (gfloat) data->num_orig;
/*
printf("sbe (unscaled) = %f\n",sbe);
*/

/* transfer surface data to new model */
/* NB: memcpy would produce undesired effects if pointers were */
/* later added to the surface struct */
ARR3SET(surf->surface.miller, surfdat->miller);
/* force to 0 for the valid cuts determination */
surf->surface.shift = 0.0;
/* only want one slice */
surf->surface.region[0] = 1;
surf->surface.region[1] = 1;

#if DEBUG_CALC_SHIFTS
printf("*** calc shifts ***\n");
printf("Miller (%d,%d,%d)\n",surf->surface.miller[0]
                            ,surf->surface.miller[1]
                            ,surf->surface.miller[2]);
printf("Shift %f\n",surf->surface.shift);
printf("Making surface...\n");
#endif

generate_surface(data, surf);
if (surf->surface.dspacing == 0.0)
  {
  printf("calc_shifts() error, invalid dspacing.\n");
  return(1);
  }

/* transfer appropriate setup data to new model */
copy_gulp_data(data, surf);
copy_elem_data(data, surf);

surf->gulp.run = E_SINGLE;
surf->gulp.method = CONV;

/* need to do this in order to build mols */
init_objs(INIT_COORDS, surf);
latmat_fudge(surf);
init_objs(INIT_COORDS, surf);
calc_bonds(surf);
calc_mols(surf);

/* get possible shifts - molecule centroids */
num_cuts=0;
for (i=surf->num_mols ; i-- ; )
  {
  if (num_cuts < MAX_CUTS)
    {
    ARR3SET(vec, (surf->mols+i)->centroid);
    vecmat(surf->latmat, vec);
    zcut[num_cuts++] = vec[2];
    }
  else
    printf("Max cuts exceeded.\n");
  }

/* rank */
zsort(zcut, num_cuts);

/*
printf("Found %d cuts.\n",num_cuts);
for (i=0 ; i<num_cuts ; i++)
  printf(" %f",zcut[i]);
printf("\n");
*/

/* eliminate any equivalent cuts -> shift[] */
num_shifts = 0;
ref=0;
for (i=0 ; i<num_cuts ; i++)
  {
  if (fabs(zcut[i] - zcut[ref]) > POSITION_TOLERANCE)
    {
    shift[num_shifts] = zcut[i];
    num_shifts++;
    ref = i;
    }
  }

/* convert from physical z coords to a shift value */
for (i=0 ; i<num_shifts ; i++)
   {
   shift[i] *= -1.0/surf->surface.dspacing; 
   while (shift[i] < 0.0)
     shift[i] += 1.0;
   }

/*
printf("Unique shifts: %d\n",num_shifts);
for (i=0 ; i<num_shifts ; i++)
  printf(" %f",shift[i]);
printf("\n");
*/

/* make cuts a little nicer by taking midpoints */
/* plus the 1st -> 0.0 */
for (i=num_shifts-1 ; i-- ; )
  {
  shift[i+1] += shift[i];
  shift[i+1] /= 2.0;
  }
shift[0] = 0.0;

/* re-rank as -ve shifts get +1.0 */
/*
zsort(shift, num_shifts);
*/
/* remove repeats (due to layer symmetry?) */
/* NB: order is important in this loop */
for (i=0 ; i<num_shifts ; i++)
  {
  if (shift[i] > 0.9998)
    {
    num_shifts = i;
    break;
    }
  }

/*
printf("Processed shifts:\n");
for (i=0 ; i<num_shifts ; i++)
  printf(" %f",shift[i]);
printf("\n");
*/

/* generate & check each one */
/* NB: this'll overwrite our 1st model, saving having to delete it */
old = new;
for(i=0 ; i<num_shifts ; i++)
  {
  surf->surface.shift = shift[i];
  generate_surface(data, surf);

/* compute surface bulk energy */
  r1size=0;
  for (j=0 ; j<surf->num_atoms ; j++)
    if ((surf->atoms+j)->region == REGION1A)
      r1size++;

if (!r1size)
  printf("Warning, empty region 1.\n");

  surf->gulp.sbulkenergy = sbe * (gfloat) r1size;
/*
printf("sbulkenergy = %d\n",surf->gulp.sbulkenergy);
*/

/* init (may be needed for gulp write routines) */
surf->id = GULP;
surf->periodic = 2;
surf->fractional = FALSE;
surf->axes_on = TRUE;
surf->cell_on = TRUE;
init_objs(INIT_COORDS, surf);
/* new model - store cartesian coords */
save_cart(surf);
latmat_fudge(surf);
init_objs(INIT_COORDS, surf);
calc_bonds(surf);
calc_mols(surf);

/* use gulp to calc */
/* TODO - write each cut to it's own (sensible) file so */
/*        that if a crash occurs (eg during subsq. regcon) */
/*        the cuts can be re-loaded and converged */
  if (write_gulp("dummy.gin", surf))
    printf("write %d failed.\n",i);
  if (exec_gulp("dummy.gin", "dummy.got"))
    printf("exec %d failed.\n",i);
  if (read_single_gulp("dummy.got", surf))
    printf("read %d failed.\n",i);

/* create new model if valid cut */
/* TODO - by default create, unless turned off */
  if (fabs(surf->gulp.sdipole) < 0.005)
    {
#if DEBUG_CALC_SHIFTS
printf("Shift %f ok (%f)\n",shift[i],surf->gulp.sdipole);
#endif
    new++;
/* alloc for a new model (if not the last shift) */
    if (i != num_shifts-1)
      surf = model_ptr(new, ASSIGN);
    if (!surf)
      {
      printf("Alloc failed - max models exceeded?\n");
      break;
      }
    ARR3SET(surf->surface.miller, surfdat->miller);
/* TODO - converge regions (if desired) */
    surf->surface.region[0] = 1;
    surf->surface.region[1] = 1;

    copy_gulp_data(data, surf);
    copy_elem_data(data, surf);

    surf->gulp.run = E_SINGLE;
    surf->gulp.method = CONV;
    }
  else
    {
/* if on last shift & it was bad - delete */
    if (i == num_shifts-1)
      delete_model(new);
    }
  }

#if DEBUG_CALC_SHIFTS
printf("Created %d models.\n",new-old);
#endif

/* init all new models for display */
for (i=old ; i<new ; i++)
  {
  surf = model_ptr(i, RECALL);
  if (surf)
    {
    data->axes_on = TRUE;
    data->cell_on = TRUE;
    init_objs(REDO_COORDS, surf);
    calc_bonds(surf);
    calc_mols(surf);
    new_tree_item(i, APPEND);
    }
  }

/* select source model (it should still be highlighted) */
pick_model(data->number);
redraw_canvas(ALL);

return(0);
}

/***************************************/
/* converge region sizes for a surface */
/***************************************/
#define DEBUG_CONV_REGIONS 1
gint conv_regions(struct surface_pak *surfdat)
{
gint i, j;
gint new, r1, r2, r1size, flag;
gfloat sbe, old_energy, de;
struct model_pak *data, *surf;

/* checks */
data = model_ptr(surfdat->model, RECALL);
g_return_val_if_fail(data != NULL, 1);
if (data->periodic != 3)
  {
  show_text("base model is of wrong type.");
  return(1);
  }

#if DEBUG_CONV_REGIONS
printf("Converging region sizes for model: %d\n", surfdat->model);
#endif

/* NEW - complete molecules */
unfragment(data);
init_objs(REFRESH_COLOUR, data);
init_objs(CENT_COORDS, data);
calc_bonds(data);
calc_mols(data);

/* allocate & init for surface */
new = sysenv.num_models;
surf = model_ptr(new, ASSIGN);
g_return_val_if_fail(surf != NULL, 1);
surf->mode = FREE;
surf->id = GULP;
surf->periodic = 2;
surf->fractional = FALSE;

/* surface bulk energy */
sbe = data->gulp.energy;
sbe /= (gfloat) data->num_orig;
/* transfer surface data to new model */
/* NB: memcpy would produce undesired effects if pointers were */
/* later added to the surface struct */
ARR3SET(surf->surface.miller, surfdat->miller);
r1 = surfdat->region[0];
r2 = surfdat->region[1];
surf->surface.shift = surfdat->shift;

/* transfer appropriate setup data to new model */
copy_gulp_data(data, surf);
copy_elem_data(data, surf);
/* run setup */
surf->gulp.run = E_SINGLE;
surf->gulp.method = CONV;

#if DEBUG_CONV_REGIONS
printf("*** conv regions ***\n");
printf("Miller (%d,%d,%d)\n",surf->surface.miller[0]
                            ,surf->surface.miller[1]
                            ,surf->surface.miller[2]);
printf("Shift %f\n",surf->surface.shift);
printf("Starting region sizes (%d,%d)\n",r1,r2);
#endif

/* converge region 2 size */
flag=0;
de = old_energy = 0.0;
do
  {
/* setup */
  surf->surface.shift = surfdat->shift;
  surf->surface.region[0] = r1;
  surf->surface.region[1] = r2;
  generate_surface(data, surf);
/* first cycle dspacing is calculated - check regions */
  if (!flag)
    {
#if DEBUG_CONV_REGIONS
printf("# 1st cycle check\n");
#endif
    if (surf->surface.dspacing < 15.0)
      {
      i = 1 + (gint) (15.0 / surf->surface.dspacing);
      if (i > r1 || i > r2)
        {
#if DEBUG_CONV_REGIONS
printf("# setting region sizes using dspacing.\n");
#endif
        if (i > r1)
          r1 = i;
        if (i > r2)
          r2 = i;
        }
      }
    flag++;
    }

/* need to do this in order to build mols */
  init_objs(INIT_COORDS, surf);
  latmat_fudge(surf);
  init_objs(INIT_COORDS, surf);
  calc_bonds(surf);
  calc_mols(surf);

/* compute surface bulk energy */
  r1size=0;
  for (j=0 ; j<surf->num_atoms ; j++)
    if ((surf->atoms+j)->region == REGION1A)
      r1size++;

if (!r1size)
  printf("Warning: empty region 1.\n");

  surf->gulp.sbulkenergy = sbe * (gfloat) r1size;

/* use gulp to calc */
  if (write_gulp("dummy.gin", surf))
    printf("write failed.\n");
  if (exec_gulp("dummy.gin", "dummy.got"))
    printf("exec failed.\n");
  if (read_single_gulp("dummy.got", surf))
    printf("read failed.\n");

  de = fabs(surf->gulp.esurf - old_energy);

#if DEBUG_CONV_REGIONS
printf("[%d:%d] Esurf = %f (%f)\n",r1,r2,surf->gulp.esurf, de);
#endif

  old_energy = surf->gulp.esurf;
  r2++;
  }
while(de > 0.0001);

/* correct for the last trivial inc & the last ++ operation */
r2 -= 2;

/* converge region 1 size */
flag=0;
de = old_energy = 0.0;
do
  {
/* setup */
  surf->surface.shift = surfdat->shift;
  surf->surface.region[0] = r1;
  surf->surface.region[1] = r2;
  generate_surface(data, surf);

/* need to do this in order to build mols */
  init_objs(INIT_COORDS, surf);
  latmat_fudge(surf);
  init_objs(INIT_COORDS, surf);
  calc_bonds(surf);
  calc_mols(surf);

/* compute surface bulk energy */
  r1size=0;
  for (j=0 ; j<surf->num_atoms ; j++)
    if ((surf->atoms+j)->region == REGION1A)
      r1size++;

if (!r1size)
  printf("Warning: empty region 1.\n");

  surf->gulp.sbulkenergy = sbe * (gfloat) r1size;

/* use gulp to calc */
  if (write_gulp("dummy.gin", surf))
    printf("write failed.\n");
  if (exec_gulp("dummy.gin", "dummy.got"))
    printf("exec failed.\n");
  if (read_single_gulp("dummy.got", surf))
    printf("read failed.\n");

  de = fabs(surf->gulp.esurf - old_energy);

#if DEBUG_CONV_REGIONS
printf("[%d:%d] Esurf = %f (%f)\n",r1,r2,surf->gulp.esurf, de);
#endif

  old_energy = surf->gulp.esurf;
  r1++;
  }
while(de > 0.0001);

/* correct for the last trivial inc & the last ++ operation */
r1 -= 2;
printf("Converged region sizes: (%d,%d) \n",r1,r2);

/* cleanup & exit */
delete_model(new);
return(0);
}

/******************************************/
/* process surface creation dialog events */
/******************************************/
gint gensurf_event(GtkWidget *w, gpointer *button)
{
gint i, type;
gfloat r=0;
static struct surface_pak current;
struct model_pak *data;

/* FIXME - ugly hack to make the correct model current the first time around */
if (!w && !button)
  {
  for (i=0 ; i<sysenv.num_models ; i++)
    {
    data = model_ptr(i, RECALL);
    if (data->periodic == 3)
      {
      current.model = data->number;
      return(FALSE);
      }
    }
  return(FALSE);
  }

/* execute (button) has no assoc. spinner */
if (GTK_IS_SPIN_BUTTON(button))
  {
  type = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(button), "id"));
  r = SPIN_FVAL(GTK_SPIN_BUTTON(button));
  }
else
  {
/* FIXME - a bit ugly */
  if (GTK_IS_MENU_ITEM(button))
    {
    r = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(button), "id"));
    current.model = (gint) r;
    return(FALSE);
    }
  else
    type = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(button), "id"));
  }

/* update */
switch(type)
  {
  case MILLER_H:
    current.miller[0] = (gint) r;
    break;
  case MILLER_K:
    current.miller[1] = (gint) r;
    break;
  case MILLER_L:
    current.miller[2] = (gint) r;
    break;
  case SHIFT:
    current.shift = r;
    break;
  case REGION1:
    current.region[0] = (gint) r;
    break;
  case REGION2:
    current.region[1] = (gint) r;
    break;
  case CALC_SHIFTS:
    calc_shifts(&current);
    break;
  case RANK_FACES:
    rank_faces(&current);
    break;
  case CONV_REGIONS:
    conv_regions(&current);  
    break;
  case GENERATE:
    make_surface(&current);
    break;
  default:
    printf("gensurf_event(): bad event type.\n");
  }
return(FALSE);
}

/******************************************/
/* NEW - surface creation/cut elimination */
/******************************************/
void surface_dialog()
{
gint i, id;
GtkWidget *hbox1, *vbox1, *vbox, *hbox, *table, *label, *spin, *button;
GtkWidget *model_src, *model_menu, *item;
GtkAdjustment *adj;
struct model_pak *data, *tmp;
struct dialog_pak *surfwin;

/* act on active model only */
data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;

/* new dialog */
if ((id = request_dialog(sysenv.active, GENSURF)) < 0)
  return;
surfwin = &sysenv.dialog[id];
surfwin->win = gtk_dialog_new();
gtk_window_set_title(GTK_WINDOW (surfwin->win), "Surfaces");
gtk_signal_connect(GTK_OBJECT(surfwin->win), "destroy",
                   GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* main hbox */
hbox1 = gtk_hbox_new(FALSE,0);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surfwin->win)->vbox), hbox1, 
                                                 TRUE, TRUE, 10);

/* left vbox */
vbox = gtk_vbox_new(FALSE,0);
gtk_box_pack_start(GTK_BOX(hbox1), vbox, TRUE, TRUE, 10);
/* table for nice spacing */
table = gtk_table_new(2, 4, FALSE);
gtk_container_add(GTK_CONTAINER(GTK_BOX(vbox)),table);

label = gtk_label_new("Miller");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,0,1);
label = gtk_label_new("Shift");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);
label = gtk_label_new("Region sizes");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);
label = gtk_label_new("Source structure");
gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,3,4);

/* miller */
hbox = gtk_hbox_new(TRUE, 0);
gtk_table_attach_defaults(GTK_TABLE(table),hbox,1,2,0,1);
/* h spinner */
adj = (GtkAdjustment *) gtk_adjustment_new
                        (data->surface.miller[0], -5, 5, 1, 1, 0);
spin = gtk_spin_button_new(adj, 0, 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, TRUE, 0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(gensurf_event), (gpointer) spin);
/* NEW - assoc data */
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) MILLER_H);
/* fake event - to update the local vars in gensurf_event() */
gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");

/* miller - k */
adj = (GtkAdjustment *) gtk_adjustment_new
                        (data->surface.miller[1], -5, 5, 1, 1, 0);
spin = gtk_spin_button_new(adj, 0, 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, TRUE, 0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(gensurf_event), (gpointer) spin);
/* NEW - assoc data */
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) MILLER_K);
/* fake event - to update the local vars in gensurf_event() */
gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");

/* miller - l */
adj = (GtkAdjustment *) gtk_adjustment_new
                        (data->surface.miller[2], -5, 5, 1, 1, 0);
spin = gtk_spin_button_new(adj, 0, 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, TRUE, 0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(gensurf_event), (gpointer) spin);
/* NEW - assoc data */
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) MILLER_L);
/* fake event - to update the local vars in gensurf_event() */
gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");
/*
gtk_object_set_data(GTK_OBJECT(spin), "ptr", (gpointer) data);
*/

/* shift adjustment */
adj = (GtkAdjustment *) gtk_adjustment_new
                        (data->surface.shift, 0.0, 1.0, 0.1, 0.1, 0);
spin = gtk_spin_button_new(adj, 0.1, 4);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_table_attach_defaults(GTK_TABLE(table),spin,1,2,1,2);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(gensurf_event), (gpointer) spin);
/* NEW - assoc model ptr to spinner */
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) SHIFT);
/* fake event - to update the local vars in gensurf_event() */
gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");


/* region size adjustment */
hbox = gtk_hbox_new(FALSE,0);
gtk_table_attach_defaults(GTK_TABLE(table),hbox,1,2,2,3);

/* region 1a */
adj = (GtkAdjustment *) gtk_adjustment_new
                        (data->surface.region[0], 0, 50, 1, 1, 0);
spin = gtk_spin_button_new(adj, 0, 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, TRUE, 0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(gensurf_event), (gpointer) spin);
/* NEW - assoc data */
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) REGION1);
/* fake event - to update the local vars in gensurf_event() */
gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");

/* region 2a */
adj = (GtkAdjustment *) gtk_adjustment_new
                        (data->surface.region[1], 0, 50, 1, 1, 0);
spin = gtk_spin_button_new(adj, 0, 0);
gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(spin), FALSE);
gtk_spin_button_set_shadow_type(GTK_SPIN_BUTTON(spin), GTK_SHADOW_OUT);
gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, TRUE, 0);
gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
                   GTK_SIGNAL_FUNC(gensurf_event), (gpointer) spin);
/* NEW - assoc data */
gtk_object_set_data(GTK_OBJECT(spin), "id", (gpointer) REGION2);
/* fake event - to update the local vars in gensurf_event() */
gtk_signal_emit_by_name(GTK_OBJECT(adj), "value_changed");

/* source model for surface generation */
hbox = gtk_hbox_new(FALSE,0);
gtk_table_attach_defaults(GTK_TABLE(table),hbox,1,2,3,4);
model_src = gtk_option_menu_new();
gtk_widget_ref(model_src);
gtk_box_pack_start(GTK_BOX (hbox), model_src, FALSE, FALSE, 0);
model_menu = gtk_menu_new();

/* NB: these should be sequential in the gdis.h enum{} */
item=NULL;
for (i=0 ; i<sysenv.num_models ; i++)
  {
  tmp=model_ptr(i, RECALL);
  if (tmp->periodic == 3)
    {
    item = gtk_menu_item_new_with_label(tmp->basename);
    gtk_menu_append(GTK_MENU(model_menu), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate", (GtkSignalFunc) gensurf_event, item);
    gtk_object_set_data(GTK_OBJECT(item), "id", (gpointer) i);
/* needed? */
    gtk_widget_show(item);
    }
  }
gtk_option_menu_set_menu(GTK_OPTION_MENU (model_src), model_menu);
/* make the first item active */
gensurf_event(NULL, NULL);
/*
gtk_option_menu_set_history(GTK_OPTION_MENU(model_src), 0);
*/

/* spread things out a little */
gtk_table_set_row_spacings(GTK_TABLE(table), 2);
gtk_table_set_col_spacings(GTK_TABLE(table), 5);

/* right vbox */
vbox1 = gtk_vbox_new(FALSE,0);
gtk_box_pack_start(GTK_BOX(hbox1), vbox1, TRUE, TRUE, 10);

button = gtk_button_new_with_label (" BFDH morphology ");
gtk_box_pack_start(GTK_BOX(vbox1), button, TRUE, TRUE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (gensurf_event),
                           (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) RANK_FACES);

button = gtk_button_new_with_label (" Find shifts ");
gtk_box_pack_start(GTK_BOX(vbox1), button, TRUE, TRUE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (gensurf_event),
                           (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) CALC_SHIFTS);

button = gtk_button_new_with_label (" Converge regions ");
gtk_box_pack_start(GTK_BOX(vbox1), button, TRUE, TRUE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (gensurf_event),
                           (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) CONV_REGIONS);

/* terminating buttons */
hbox = gtk_hbox_new(FALSE,0);
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(surfwin->win)->action_area),
                                       hbox, FALSE, TRUE, 0);

button = gtk_button_new_with_label (" Generate ");
gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (gensurf_event),
                           (gpointer) button);
gtk_object_set_data(GTK_OBJECT(button), "id", (gpointer) GENERATE);

button = gtk_button_new_with_label ("  Close  ");
gtk_box_pack_end(GTK_BOX(hbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (close_dialog),
                           (gpointer) id);

/* done */
gtk_widget_show_all(surfwin->win);
}

