/*
Copyright (C) 2003 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "gdis.h"
#include "coords.h"
#include "matrix.h"
#include "opengl.h"
#include "render.h"
#include "interface.h"

/* externals */
extern struct sysenv_pak sysenv;
extern GtkWidget *window;
extern GdkPixmap *pixmap;

/************************************************************/
/* process data structures to get current model state label */
/************************************************************/
gchar *get_mode_label(struct model_pak *data)
{
gchar *label;

switch(data->mode)
  {
  case FREE:
    label = g_strdup("normal");
    break;
  case ATOM_ADD:
    label = g_strdup("add atoms");
    break;
  case ATOM_MOVE:
    label = g_strdup("move atoms");
    break;
  case ATOM_DELETE:
    label = g_strdup("delete atoms");
    break;
  case BOND_DELETE:
    label = g_strdup("delete bonds");
    break;
  case DELETE_VECTOR:
    label = g_strdup("delete vectors");
    break;
  case BOND_SINGLE:
    label = g_strdup_printf("add single bonds (%1d/2)", data->state);
    break;
  case BOND_DOUBLE:
    label = g_strdup_printf("add double bonds (%1d/2)", data->state);
    break;
  case BOND_TRIPLE:
    label = g_strdup_printf("add triple bonds (%1d/2)", data->state);
    break;
  case MOL_MOVE:
    label = g_strdup("move mol");
    break;
  case MOL_DELETE:
    label = g_strdup("delete mols");
    break;
  case BOND_INFO:
    label = g_strdup("bonds");
    break;
  case DIST_INFO:
    label = g_strdup_printf("distance selection (%1d/2)", data->state);
    break;
  case ANGLE_INFO:
    label = g_strdup_printf("angle selection (%1d/3)", data->state);
    break;
  case DIHEDRAL_INFO:
    label = g_strdup_printf("dihedral selection (%1d/4)", data->state);
    break;
  case DEFINE_RIBBON:
    label = g_strdup("define ribbon");
    break;
  case DEFINE_VECTOR:
    label = g_strdup_printf("define vector (%1d/2)", data->state);
    break;
  case DEFINE_PLANE:
    label = g_strdup_printf("define plane (%1d/3)", data->state);
    break;
  case ANIMATING:
    label = g_strdup_printf("Frame: %-4d", data->cur_frame);
    break;
  case RECORD:
    label = g_strdup("recording");
    break;

/* NEW */
  case SELECT_FRAGMENT:
    label = g_strdup_printf("select bonded atoms (%1d/2)", data->state);
    break;

  default:
    label = g_strdup("undefined");
    break;
  }

return(label);
}

/**********************************/
/* setup pipe list (bond drawing) */
/**********************************/
GSList *render_get_pipes(gint line, gint wire, gint ghost, struct model_pak *model)
{
gint part;
gdouble radius, v1[4], v2[4], mp[3], v1i[4], v2i[4], mp12i[4], mp1i2[4];
gdouble colour1[4], colour2[4];
GSList *list, *pipe_list=NULL;
struct pipe_pak *pipe;
struct bond_pak *bond;
struct core_pak *core1, *core2;

/* checks */
g_assert(model != NULL);

/* common pipe width */
radius = model->scale*sysenv.render.stick_rad;

/* enumerate bonds to construct the pipe list */
for (list=model->bonds; list ; list=g_slist_next(list))
  {
  bond = (struct bond_pak *) list->data;

  if (bond->status == DELETED)
    continue;
  if (bond->status == HIDDEN)
    continue;

/* the two atoms */
  core1 = bond->atom1;
  core2 = bond->atom2;
  ARR3SET(v1, core1->rx);
  ARR3SET(v2, core2->rx);

  switch (bond->periodic)
    {
    case BOND_SPLIT:
/* periodic image positions */
      ARR3SET(v2i, core2->x);
      v2i[3] = 1.0;
      ARR3ADD(v2i, bond->pic);
      vec4mat(model->display_lattice, v2i);

      ARR3SET(v1i, core1->x);
      v1i[3] = 1.0;
      ARR3SUB(v1i, bond->pic);
      vec4mat(model->display_lattice, v1i);

/* periodic image midpoint positions */
      ARR3SET(mp1i2, v1i);
      ARR3ADD(mp1i2, v2);
      VEC3MUL(mp1i2, 0.5);

      ARR3SET(mp12i, v1);
      ARR3ADD(mp12i, v2i);
      VEC3MUL(mp12i, 0.5);
      break;

    default:
/* merged/normal bond midpoint */
      ARR3SET(mp, v1);
      ARR3ADD(mp, v2);
      VEC3MUL(mp, 0.5);
      break;
    }

/* deleted/ zeol hidden */
  if (core1->status & (DELETED | ZEOL_HIDDEN))
    continue;
  if (core2->status & (DELETED | ZEOL_HIDDEN))
    continue;

/* colour setup */
  ARR4SET(colour1, core1->colour);
  VEC3MUL(colour1, 1.0/65535.0);

  ARR4SET(colour2, core2->colour);
  VEC3MUL(colour2, 1.0/65535.0);

/* test which parts of the bond (pipe) should be drawn */
  part = 3;

/* omit 1st half? */
  if (core1->status & HIDDEN)
    part &= 2;
  if (core1->ghost != ghost)
    part &= 2;
  switch(core1->render_mode)
    {
    case CPK:
      part &= 2;
      break;

    case STICK:
      if (!line)
        part &= 2;
      break;

    case BALL_STICK:
    case LIQUORICE:
/* NB: only check this here, so STICK mode doesn't get wiped out in wire frame style */
      if (core1->render_wire != wire)
        part &= 2;
      if (line)
        part &= 2;
      break;
    }

/* omit 2nd half? */
  if (core2->status & HIDDEN)
    part &= 1;
  if (core2->render_mode == CPK)
    part &= 1;
  if (core2->ghost != ghost)
    part &= 1;

  switch(core2->render_mode)
    {
    case CPK:
      part &= 1;
      break;

    case STICK:
      if (!line)
        part &= 1;
      break;

    case BALL_STICK:
    case LIQUORICE:
/* NB: only check this here, so STICK mode doesn't get wiped out in wire frame style */
      if (core2->render_wire != wire)
        part &= 1;
      if (line)
        part &= 1;
      break;
    }

/* the core1 pipe */
  if (part & 1)
    {
    pipe = g_malloc(sizeof(struct pipe_pak));

    ARR3SET(pipe->v1, v1);

    if (bond->periodic == BOND_SPLIT)
      {
      ARR3SET(pipe->v2, mp12i);
      }
    else
      {
      ARR3SET(pipe->v2, mp);
      }

    pipe->radius = radius;

    ARR4SET(pipe->colour, colour1);

    pipe_list = g_slist_prepend(pipe_list, pipe);
    }

/* the core2 pipe */
  if (part & 2)
    {
    pipe = g_malloc(sizeof(struct pipe_pak));

    ARR3SET(pipe->v1, v2);

    if (bond->periodic == BOND_SPLIT)
      {
      ARR3SET(pipe->v2, mp1i2);
      }
    else
      {
      ARR3SET(pipe->v2, mp);
      }

    pipe->radius = radius;

    ARR4SET(pipe->colour, colour2);

    pipe_list = g_slist_prepend(pipe_list, pipe);
    }
  }
return(pipe_list);
}

/*********************************************/
/* construct all pipe lists for bond drawing */
/*********************************************/
#define PIPE_DEPTH 4
void render_make_pipes(GSList **pipes, struct model_pak *model)
{
gint i;
gdouble radius, v1[4], v2[4], mp[3], v1i[4], v2i[4], mp12i[4], mp1i2[4];
gdouble colour1[4], colour2[4];
GSList *list;
struct pipe_pak *pipe;
struct bond_pak *bond;
struct core_pak *core1, *core2;

/* checks */
g_assert(model != NULL);

/* init the return list(s) */
for (i=PIPE_DEPTH ; i-- ; )
  pipes[i] = NULL;

/* common pipe width */
radius = model->scale*sysenv.render.stick_rad;

/* enumerate bonds to construct the pipe list */
for (list=model->bonds; list ; list=g_slist_next(list))
  {
  bond = (struct bond_pak *) list->data;

  if (bond->status == DELETED)
    continue;
  if (bond->status == HIDDEN)
    continue;

/* the two atoms */
  core1 = bond->atom1;
  core2 = bond->atom2;
  ARR3SET(v1, core1->rx);
  ARR3SET(v2, core2->rx);

  switch (bond->periodic)
    {
    case BOND_SPLIT:
/* periodic image positions */
      ARR3SET(v2i, core2->x);
      v2i[3] = 1.0;
      ARR3ADD(v2i, bond->pic);
      vec4mat(model->display_lattice, v2i);

      ARR3SET(v1i, core1->x);
      v1i[3] = 1.0;
      ARR3SUB(v1i, bond->pic);
      vec4mat(model->display_lattice, v1i);

/* periodic image midpoint positions */
      ARR3SET(mp1i2, v1i);
      ARR3ADD(mp1i2, v2);
      VEC3MUL(mp1i2, 0.5);

      ARR3SET(mp12i, v1);
      ARR3ADD(mp12i, v2i);
      VEC3MUL(mp12i, 0.5);
      break;

    default:
/* merged/normal bond midpoint */
      ARR3SET(mp, v1);
      ARR3ADD(mp, v2);
      VEC3MUL(mp, 0.5);
      break;
    }

/* deleted/ zeol hidden */
  if (core1->status & (DELETED | ZEOL_HIDDEN))
    continue;
  if (core2->status & (DELETED | ZEOL_HIDDEN))
    continue;

/* colour setup */
  switch (bond->type)
    {
    case BOND_HBOND:
      VEC4SET(colour1, 1.0, 1.0, 0.6, 0.0);
      VEC4SET(colour2, 1.0, 1.0, 0.6, 0.0);
      break;

    default:
      ARR4SET(colour1, core1->colour);
      VEC3MUL(colour1, 1.0/65535.0);
      ARR4SET(colour2, core2->colour);
      VEC3MUL(colour2, 1.0/65535.0);
    }

/* setup half-bond (pipe) for core1 */
  if (!(core1->status & HIDDEN) && core1->render_mode != CPK)
    {
/* init pipe */
    pipe = g_malloc(sizeof(struct pipe_pak));
    ARR3SET(pipe->v1, v1);
    if (bond->periodic == BOND_SPLIT)
      {
      ARR3SET(pipe->v2, mp12i);
      }
    else
      {
      ARR3SET(pipe->v2, mp);
      }
    pipe->radius = radius;
    ARR4SET(pipe->colour, colour1);

/* assign to appropriate pipe list */
    if (core1->render_mode == STICK)
      pipes[3] = g_slist_prepend(pipes[3], pipe);
    else
      {
      if (core1->ghost)
        pipes[1] = g_slist_prepend(pipes[1], pipe);
      else
        {
        if (core1->render_wire)
          pipes[2] = g_slist_prepend(pipes[2], pipe);
        else
          pipes[0] = g_slist_prepend(pipes[0], pipe);
        }
      }
    }

/* setup half-bond (pipe) for core1 */
  if (!(core2->status & HIDDEN) && core2->render_mode != CPK)
    {
/* init pipe */
    pipe = g_malloc(sizeof(struct pipe_pak));
    ARR3SET(pipe->v1, v2);
    if (bond->periodic == BOND_SPLIT)
      {
      ARR3SET(pipe->v2, mp1i2);
      }
    else
      {
      ARR3SET(pipe->v2, mp);
      }
    pipe->radius = radius;
    ARR4SET(pipe->colour, colour2);

/* assign to appropriate pipe list */
    if (core2->render_mode == STICK)
      pipes[3] = g_slist_prepend(pipes[3], pipe);
    else
      {
      if (core2->ghost)
        pipes[1] = g_slist_prepend(pipes[1], pipe);
      else
        {
        if (core2->render_wire)
          pipes[2] = g_slist_prepend(pipes[2], pipe);
        else
          pipes[0] = g_slist_prepend(pipes[0], pipe);
        }
      }
    }
  }
}

/*****************************/
/* pipe z-ordering primitive */ 
/*****************************/
gint render_pipe_depth_sort(struct pipe_pak *p1, struct pipe_pak *p2)
{
gdouble z1, z2;

/* use pipe z-midpoints for comparison */
z1 = 0.5 * (p1->v1[2] + p1->v2[2]);
z2 = 0.5 * (p2->v1[2] + p2->v2[2]);

if (z1 > z2)
  return(-1);
return(1);
}

GSList *render_sort_pipes(GSList *pipes)
{
return(g_slist_sort(pipes, (gpointer) render_pipe_depth_sort));
}
