
/* 
   $RCSfile: classdata.c,v $

   This file is part of:
   
       GNU Enterprise Application Server (GEAS)

   Copyright (C) 2001 Free Software Foundation

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   $Id: classdata.c,v 1.65 2001/07/06 01:38:09 ntiffin Exp $
   
*/

#include <glib.h>
#include <string.h>
#include <time.h>
#include <assert.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <dirent.h>
#include <errno.h>

/* #include "gcdparser.h" */
#include "config.h"
#include "classdata.h"

#include "lparser.h"
int yyparse (void);

/* spaces to indent per level */
#define INDENT 2

static void odl_dump_base (FILE * out, odl_base * base);
static void odl_dump_container (FILE * out, odl_container * container);
static void odl_dump_item (FILE * out, odl_item * item);

/* ------------------------------------------------------------------------- *\
 * Print the contents of the odl base structure
\* ------------------------------------------------------------------------- */
void
odl_dump_base (FILE * out, odl_base * base)
{
  g_assert (out != NULL);
  g_assert (base != NULL);
  fprintf (out, "\n******* base name: %s, %p\n", base->name, base);
  fprintf (out, "         fullname: %s\n", base->fullname);
  fprintf (out, "     mangled name: %s\n", base->mangledname);
  fprintf (out, "    odl_item_type: %s\n", odl_itemtype_name (base->type));
  fprintf (out, "           access: %s\n", odl_access_name (base->access));
  fprintf (out, "         triggers: %lx\n", base->triggers);
  fprintf (out, "           parent: ");
  if (base->parent != NULL)
    {
      fprintf (out, "%s, ", (base->parent)->name);
    }
  fprintf (out, "%p\n", base->tree);
}

/* ------------------------------------------------------------------------- *\
 * Print the contents of the odl container structure
\* ------------------------------------------------------------------------- */
void
odl_dump_container (FILE * out, odl_container * container)
{
  GList *contents;
  GList *parents;
  GList *indexes;
  odl_base *base;
  _odl_index *index;
  g_assert (out != NULL);
  g_assert (container != NULL);
  if (container->istype)
    {
      fprintf (out, "           istype: true\n");
    }
  else
    {
      fprintf (out, "           istype: false\n");
    }
  fprintf (out, "         order by: %s%s\n", container->orderby ,
	   (container->desc?" (descending)":"hello") );
  contents = container->contents;
  while (contents)
    {
      base = (odl_base *) contents->data;
      fprintf (out, "         contents: %s, %p\n", base->name, base);
      contents = g_list_next (contents);
    }
  parents = container->parents;
  while (parents)
    {
      char *name = (char *) parents->data;
      fprintf (out, "          parents: %s\n", name);
      parents = g_list_next (parents);
    }
  indexes = container->indexes;
  while (indexes)
    {
      fprintf (out, "          indexes:");
      index = (_odl_index *) indexes->data;
      contents = index->fields;
      while (contents)
        {
          fprintf (out, " %s", (char *) contents->data);
          contents = g_list_next (contents);
          if (contents != NULL)
            {
              fprintf (out, ",");
            }
          fprintf (out, " -");
        }
      if (index->unique)
        {
          fprintf (out, " UNIQUE");
        }
      if (index->primary)
        {
          fprintf (out, " PRIMARY");
        }
      indexes = g_list_next (indexes);
      fprintf (out, "\n");
    }
  fprintf( out, "         filename: %s\n", container->filename);
}

/* ------------------------------------------------------------------------- *\
 * Print the contents of the odl item structure
\* ------------------------------------------------------------------------- */
void
odl_dump_item (FILE * out, odl_item * item)
{
  GList *contents;
  g_assert (out != NULL);
  g_assert (item != NULL);
  fprintf (out, "       field type: %s\n",
           odl_fieldtype_name (item->fieldtype));
  fprintf (out, "        data type: %s\n",
           odl_datatype_name (item->datatype));
  fprintf (out, "  data type class: %s\n", item->datatypeclass);
  fprintf (out, "       properties: %lu\n", item->properties);

  fprintf (out, "           format: %s\n", item->format);
  fprintf (out, "    default value: %s\n", item->defaultval);
  fprintf (out, "     temp default: %s\n", item->tmpdefault);

  fprintf (out, "     source field: %s\n", item->sourcefield);
  fprintf (out, "     source class: %s\n", item->sourceclass);

  contents = item->this_fields;
  fprintf (out, "      this fields: ");
  while (contents)
    {
      fprintf (out, "%s ", (char *) contents->data);
      contents = g_list_next (contents);
    }
  fprintf (out, "\n");

  contents = item->source_fields;
  fprintf (out, "    source fields: ");
  while (contents)
    {
      fprintf (out, "%s ", (char *) contents->data);
      contents = g_list_next (contents);
    }
  fprintf (out, "\n");
  fprintf (out, "           bounds: %ld  ", item->bounds);
  if (item->bounds == -1)
    fprintf (out, "(Reference)\n");
  if (item->bounds > -1)
    fprintf (out, "(List)\n");
  if (item->bounds < -1)
    fprintf (out, "(Undefined)\n");

  contents = item->arguments;
  fprintf (out, "        arguments: ");
  while (contents)
    {
      fprintf (out, "%s ", (char *) contents->data);
      contents = g_list_next (contents);
    }
  fprintf (out, "\n");
  fprintf (out, "      calculation: %s\n", item->calculation);
  contents = item->elements;
  fprintf (out, "    enum elements: ");
  while (contents)
    {
      fprintf (out, "%s ", (char *) contents->data);
      contents = g_list_next (contents);
    }
  fprintf (out, "\n");
}

/* ------------------------------------------------------------------------- *\
 * Print the contents of the odl structure
 * recursive = true, means recurse through all children
\* ------------------------------------------------------------------------- */
void
odl_dump1_tree (FILE * out, odl_base * base, gboolean recursive)
{
  switch (base->type)
    {
    case IT_unknown:
      printf ("**** ERROR Unknown item type IT_unknown\n");
      odl_dump_base (out, base);
      break;
    case IT_module:
    case IT_class:
      /* the only containers */
      odl_dump_base (out, base);
      odl_dump_container (out, (odl_container *) base);
      if (recursive)
        {
          GList *contents;
          contents = ((odl_container *) base)->contents;
          while (contents)
            {
              odl_dump1_tree (out, (odl_base *) (contents->data), recursive);
              contents = g_list_next (contents);
            }
        }
      break;
    case IT_field:
    case IT_enum:
    case IT_type:
      odl_dump_base (out, base);
      odl_dump_item (out, (odl_item *) base);
      break;
    case IT_ignore:
      printf ("**** ERROR Ignore item type IT_ignore\n");
      odl_dump_base (out, base);
      break;
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
int
odl_get_error_count ()
{
  return (yyerrorcount);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
int
odl_get_warn_count ()
{
  return (yywarncount);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_access_name (unsigned long int access)
{
  switch (access)
    {
    case ODL_ACCESS_PUBLIC:
      return ("public");
    case ODL_ACCESS_PRIVATE:
      return ("private");
    case ODL_ACCESS_PROTECTED:
      return ("protected");
    case ODL_ACCESS_SYSTEM:
      return ("system");
    case ODL_ACCESS_UNCHANGED:
      return ("unchanged");
    case ODL_ACCESS_UNKNOWN:
      return ("unknown");
    }
  g_assert_not_reached ();
  return ("(unknown access)");
}

/* ------------------------------------------------------------------------- *\
 * Convert item type int to string value.
\* ------------------------------------------------------------------------- */
const char *
odl_itemtype_name (enum odl_itemtype type)
{
  switch (type)
    {
    case IT_type:              /* added this to satisify compiler warnings neilt */
      return ("type");
    case IT_ignore:            /* added this to satisify compiler warnings neilt */
      return ("ignore");
    case IT_unknown:
      return ("unknown");
    case IT_module:
      return ("module");
    case IT_class:
      return ("class");
    case IT_field:
      return ("field");
    case IT_enum:
      return ("enum");
    }
  g_assert_not_reached ();
  return ("unknown");
}

/* ------------------------------------------------------------------------- *\
 * Convert odl_item_type string to int value.
\* ------------------------------------------------------------------------- */
enum odl_itemtype
odl_itemtype_id (const char *name)
{
  g_assert (name != NULL);

  if (g_strcasecmp (name, "module") == 0)
    return (IT_module);
  if (g_strcasecmp (name, "class") == 0)
    return (IT_class);
  if (g_strcasecmp (name, "field") == 0)
    return (IT_field);
  if (g_strcasecmp (name, "enum") == 0)
    return (IT_enum);
  if (g_strcasecmp (name, "type") == 0)
    return (IT_type);
  if (g_strcasecmp (name, "ignore") == 0)
    return (IT_ignore);
  g_assert_not_reached ();
  return (IT_unknown);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_datatype_name (enum odl_datatype type)
{
  switch (type)
    {
    case DT_unknown:
      return ("unknown");
    case DT_char:
      return ("char");
    case DT_int:
      return ("int");
    case DT_int16:
      return ("int16");
    case DT_int32:
      return ("int32");
    case DT_int64:
      return ("int64");
    case DT_boolean:
      return ("boolean");
    case DT_text:
      return ("text");
    case DT_date:
      return ("date");
    case DT_time:
      return ("time");
    case DT_datetime:
      return ("datetime");
    case DT_bool:
      return ("bool");
    case DT_float:
      return ("float");
    case DT_class:
      return ("class");
    case DT_enum:
      return ("enum");
    case DT_void:
      return ("void");
    case DT_object:
      return ("object");
    case DT_unsignedint:
      return ("unsigned int");
    case DT_type:
      return ("compound datatype");
    }
  g_assert_not_reached ();
  return ("unknown");
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
enum odl_datatype
odl_datatype_id (const char *name)
{
#ifdef DEBUG
  assert (name != NULL);
#endif
  if (g_strcasecmp (name, "char") == 0) return (DT_char);
  if (g_strcasecmp (name, "int") == 0) return (DT_int);
  if (g_strcasecmp (name, "int16") == 0) return (DT_int16);
  if (g_strcasecmp (name, "int32") == 0) return (DT_int32);
  if (g_strcasecmp (name, "int64") == 0) return (DT_int64);
  if (g_strcasecmp (name, "boolean") == 0) return (DT_boolean);
  if (g_strcasecmp (name, "text") == 0) return (DT_text);
  if (g_strcasecmp (name, "class") == 0) return (DT_class);
  if (g_strcasecmp (name, "date") == 0) return (DT_date);
  if (g_strcasecmp (name, "time") == 0) return (DT_time);
  if (g_strcasecmp (name, "datetime") == 0) return (DT_datetime);
  if (g_strcasecmp (name, "bool") == 0) return (DT_bool);
  if (g_strcasecmp (name, "float") == 0) return (DT_float);
  if (g_strcasecmp (name, "void") == 0) return (DT_void);
  if (g_strcasecmp (name, "object") == 0) return (DT_object);
  if (g_strcasecmp (name, "enum") == 0) return (DT_enum);
  if (g_strcasecmp (name, "unsignedint") == 0) return (DT_unsignedint);
  if (g_strcasecmp (name, "type") == 0) return (DT_type);
  return (DT_unknown);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_fieldtype_name (enum odl_fieldtype type)
{
  switch (type)
    {
    case FT_basic:
      return ("basic");
    case FT_lookup:
      return ("lookup");
    case FT_reference:
      return ("reference");
    case FT_list:
      return ("list");
    case FT_method:
      return ("method");
    case FT_calculated:
      return ("calculated");
    case FT_readonly:
      return ("readonly");
    case FT_unknown:
      return ("unknown");
    }
  g_assert_not_reached ();
  return ("unknown");
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
enum odl_fieldtype
odl_fieldtype_id (const char *name)
{
#ifdef DEBUG
  assert (name != NULL);
#endif
  if (g_strcasecmp (name, "lookup") == 0)
    return (FT_lookup);
  if (g_strcasecmp (name, "reference") == 0)
    return (FT_reference);
  if (g_strcasecmp (name, "list") == 0)
    return (FT_list);
  if (g_strcasecmp (name, "method") == 0)
    return (FT_method);
  if (g_strcasecmp (name, "calculated") == 0)
    return (FT_calculated);
  if (g_strcasecmp (name, "readonly") == 0)
    return (FT_readonly);
  if (g_strcasecmp (name, "basic") == 0)
    return (FT_basic);
  g_assert_not_reached ();
  return (FT_unknown);
}

/* ========================================================================= *\
 * List handling code
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
odl_filenamelist_free (odl_filenamelist * list)
{
  g_assert (list != NULL);
  odl_namelist_free ((odl_namelist *) list);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_filenamelist *
odl_filenamelist_add (odl_filenamelist * list, const char *filename)
{
  char *f = NULL;

  g_assert (filename != NULL);
  if (filename)
    {
      f = g_strdup (filename);
    }
  if (f)
    {
      list = (odl_filenamelist *) g_list_append ((GList *) list, f);
    }
  return (list);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
odl_namelist_free (odl_namelist * list)
{
  GList *l;

#if 0
  /* printf ("---- odl_namelist_free() ----\n"); */
#endif

  g_return_if_fail (list != NULL);
  l = (GList *) list;
  while (l)
    {
      if (l->data)
        {
#if 0
          /* printf( "...... %s\n" , l->data ); */
#endif
          g_free (l->data);
        }

      l = g_list_next (l);
    }
  if (list)
    {
      g_list_free ((GList *) list);
    }
#if 0
  /* printf( "---- end of odl_namelist_free() ----\n" ); */
#endif
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_namelist_entry (odl_namelist * list, int index)
{
#ifdef DEBUG
  assert (list != NULL);
#endif
  return (g_list_nth_data ((GList *) list, index));
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
gboolean
odl_namelist_contains (odl_namelist * list, const char *name,
                       gboolean casesensitive)
{
  GList *l;

#ifdef DEBUG
  assert (list != NULL);
#endif
  l = list;
  while (l)
    {
      if (!casesensitive)
        {
          if (g_strcasecmp (name, l->data) == 0)
            {
              return (TRUE);
            }
          else if (strcmp (name, l->data) == 0)
            {
              return (TRUE);
            }
        }
      l = g_list_next (l);
    }
  return (FALSE);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
odl_fieldlist_free (odl_fieldlist * list)
{
#ifdef DEBUG
  assert (list != NULL);
#endif
  if (list)
    {
      g_list_free ((GList *) list);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_field *
odl_fieldlist_entry (odl_fieldlist * list, int index)
{
#ifdef DEBUG
  assert (list != NULL);
#endif
  return (g_list_nth_data ((GList *) list, index));
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
odl_argumentlist_free (odl_argumentlist * list)
{
#ifdef DEBUG
  assert (list != NULL);
#endif
  if (list)
    {
      g_list_free ((GList *) list);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_argument *
odl_argumentlist_entry (odl_argumentlist * list, int index)
{
#ifdef DEBUG
  assert (list != NULL);
#endif
  return (g_list_nth_data ((GList *) list, index));
}

/* ========================================================================= *\
 * Locate items based on name
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 *  if id = null, use latest version other wise use the specified version
 *        not implemented.
\* ------------------------------------------------------------------------- */
odl_class *
odl_find_class (odl_tree * tree, const char *name, const char *id)
{
  struct _odl_base *item;

#ifdef DEBUG
  assert (name != NULL);
  assert (tree != NULL);
#endif
  item = (struct _odl_base *) odl_find_item (tree, name, id);

  if (!item)
    {
      return (NULL);
    }
  if (item->type == IT_class)
    {
      return ((odl_class *) item);
    }
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_module *
odl_find_module (odl_tree * tree, const char *name, const char *id)
{
  struct _odl_base *item;
  g_assert (name != NULL);
  /* g_assert (id != NULL); */  /* id's not working yet TODO */

  item = (struct _odl_base *) odl_find_item (tree, name, id);

  if (!item)
    {
      return (NULL);
    }
  if (item->type == IT_module)
    {
      return ((odl_module *) item);
    }
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_field *
odl_find_field (odl_tree * tree, const char *name, const char *id)
{
  struct _odl_base *item;

#ifdef DEBUG
  assert (name != NULL);
  assert (id != NULL);
#endif
  item = (struct _odl_base *) odl_find_item (tree, name, id);
  if (!item)
    {
      return (NULL);
    }
  if (item->type == IT_field)
    {
      return ((odl_field *) item);
    }
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_base *
odl_find_in_container (odl_base * c, const char *name)
{
  GList *l;

#ifdef DEBUG
  assert (c != NULL);
  assert (name != NULL);
#endif
  /* if it's not a container, it really can't contain things ;) */
  if (c->type != IT_class && c->type != IT_module)
    {
      return (NULL);
    }
  l = ((odl_container *) c)->contents;
  while (l)
    {
      if (g_strcasecmp (((odl_base *) l->data)->name, name) == 0)
        {
          return ((odl_base *) l->data);
        }
      l = g_list_next (l);
    }
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * Locate a named item in the class tree
\* ------------------------------------------------------------------------- */
odl_base *
odl_find_item (odl_tree * tree, const char *name, const char *id)
{
  odl_namelist *nl;
  GList *l;
  odl_base *retval = NULL;
  char *tmp;

  g_assert (name != NULL);
  g_assert (tree != NULL);

  /* cos it's a const input, make a temp. duplicate */
  tmp = g_strdup (name);
  if (!tmp)
    {
      return (NULL);
    }
  nl = odl_split_qualified_name (tmp);
  g_free (tmp);
  if (!nl)
    {
      return (NULL);
    }
  /* make everything look like an 'absolute' name */
  if (nl && nl->data && strcmp (nl->data, "root") != 0)
    {
      nl = g_list_prepend (nl, g_strdup ("root"));
    }
  /* handle special case: 'root'. */
  retval = (odl_base *) odl_tree_get_root (tree);
  if (!retval)
    {
      odl_namelist_free (nl);
      return (NULL);
    }
  if (g_strcasecmp (retval->name, nl->data) != 0)
    {
      /* really shouldn't happen.. */
      odl_namelist_free (nl);
      return (NULL);
    }

  l = nl->next;

  /* for each item, look for it inside 'retval' then make that the */
  /* new 'retval' - once last item is found, we have it */
  while (l)
    {
      retval = (odl_base *) odl_find_in_container (retval, l->data);
      if (!retval)
        {
          /* not found, name doesn't exist */
          odl_namelist_free (nl);
          return (NULL);
        }

      /* found it, start on next one */
      l = l->next;
    }
  odl_namelist_free (nl);
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_base *
odl_container_get_parent (odl_container * container)
{
#ifdef DEBUG
  assert (container != NULL);
#endif
  return (container->base.parent);
}

/* ------------------------------------------------------------------------- *\
 * Locate a named item in the class tree
\* ------------------------------------------------------------------------- */
odl_base *
odl_find_relative (odl_base * start, const char *name)
{
  odl_namelist *nl;
  GList *l;
  odl_base *retval;
  char *tmp;

#ifdef DEBUG
  assert (start != NULL);
  assert (name != NULL);
#endif
  retval = start;
  /* cos it's a const input, make a temp. duplicate */
  tmp = g_strdup (name);
  if (!tmp)
    {
      return (NULL);
    }
  nl = odl_split_qualified_name (tmp);
  g_free (tmp);

  l = nl;

  /* for each item, look for it inside 'retval' then make that the */
  /* new 'retval' - once last item is found, we have it */
  while (l)
    {
      retval = (odl_base *) odl_find_in_container (retval, l->data);
      if (!retval)
        {
          /* not found, name doesn't exist */
          odl_namelist_free (nl);
          return (NULL);
        }

      /* found it, start on next one */
      l = l->next;
    }
  odl_namelist_free (nl);
  return (retval);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
enum odl_itemtype
odl_find_named_item_type (odl_tree * tree, const char *name)
{
  odl_base *i;

#ifdef DEBUG
  assert (tree != NULL);
  assert (name != NULL);
#endif
  i = (odl_base *) odl_find_item (tree, name, NULL);
  if (!i)
    {
      return (IT_unknown);
    }
  return (i->type);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_item_get_full_name (odl_base * item)
{
#ifdef DEBUG
  assert (item != NULL);
#endif
  return (odl_get_base_full_name ((odl_base *) item));
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_class *
odl_field_defined_in (odl_class * startclass, const char *name)
{
  odl_class *c;
  odl_namelist *parents;
  GList *l;

#ifdef DEBUG
  assert (startclass != NULL);
  assert (name != NULL);
#endif
  /* look in initial class */
  if (odl_find_in_container ((odl_base *) startclass, name))
    {
      return (startclass);
    }
  /* search parents */
  parents = odl_class_get_parentnames (startclass);
  l = parents;
  while (l)
    {
      c = odl_find_class (startclass->base.tree, l->data, NULL);
      if (c)
        {
          if (odl_find_in_container ((odl_base *) c, name))
            {
              odl_namelist_free (parents);
              return (c);
            }
        }
      l = g_list_next (l);
    }

  /* not found */
  odl_namelist_free (parents);
  return (NULL);
}

static odl_namelist *
real_odl_tree_list_classes (odl_base * item)
{
  odl_namelist *nl = NULL;
  GList *l;

#ifdef DEBUG
  assert (item != NULL);
#endif
  if (!item)
    {
      return (NULL);
    }
  if (item->type == IT_class && ((odl_class *) item)->istype == FALSE)
    {
      nl = g_list_append (nl, g_strdup (odl_item_get_full_name (item)));
    }
  l = ((odl_container *) item)->contents;
  while (l)
    {
      odl_base *i = (odl_base *) l->data;
      if (i->type == IT_module || i->type == IT_class)
        {
          odl_namelist *t = real_odl_tree_list_classes (i);
          if (t)
            {
              if (nl != NULL)
                {
                  nl = g_list_concat (nl, t);
                }
              else
                {
                  nl = t;
                }
            }
        }
      l = g_list_next (l);
    }
  return (nl);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_namelist *
odl_tree_list_classes (odl_tree * tree)
{
#ifdef DEBUG
  assert (tree != NULL);
#endif
  if (tree == NULL)
    {
      return (NULL);
    }
  return (real_odl_tree_list_classes ((odl_base *) tree->root));
}

const char *
odl_module_get_name (odl_module * module)
{
#ifdef DEBUG
  assert (module != NULL);
#endif
  return (module->base.name);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_enum_get_full_name (odl_enum * e)
{
#ifdef DEBUG
  assert (e != NULL);
#endif
  return (odl_get_base_full_name ((odl_base *) e));
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_module_get_full_name (odl_module * module)
{
#ifdef DEBUG
  assert (module != NULL);
#endif
  return (odl_get_base_full_name ((odl_base *) module));
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
unsigned long int
odl_module_get_access (odl_module * module)
{
#ifdef DEBUG
  assert (module != NULL);
#endif
  return (module->base.access);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_base *
odl_module_get_parent (odl_module * module)
{
#ifdef DEBUG
  assert (module != NULL);
#endif
  return (module->base.parent);
}

/* ========================================================================= *\
 * Read data from: class
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_class_get_name (odl_class * classd)
{
#ifdef DEBUG
  assert (classd != NULL);
#endif
  return (classd->base.name);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_class_get_full_name (odl_class * classd)
{
#ifdef DEBUG
  assert (classd != NULL);
#endif
  return (odl_get_base_full_name ((odl_base *) classd));
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_class_get_mangled_name (odl_class * classd)
{
#ifdef DEBUG
  assert (classd != NULL);
#endif
  if (classd->base.mangledname == NULL)
    {
#ifdef DEBUG
      /* printf( "mangling name %s\n" , classd->base.name ); */
#endif
      classd->base.mangledname =
        odl_mangle_qualified_name (odl_class_get_full_name (classd));
    }
#ifdef DEBUG
  /* printf( "mangled name: %s\n" , classd->base.mangledname ); */
#endif
  return (classd->base.mangledname);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
unsigned long int
odl_class_get_access (odl_class * classd)
{
#ifdef DEBUG
  assert (classd != NULL);
#endif
  return (classd->base.access);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
unsigned long int
odl_class_get_triggers (odl_class * classd)
{
#ifdef DEBUG
  assert (classd != NULL);
#endif
  return (classd->base.triggers);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_base *
odl_class_get_parent (odl_class * classd)
{
#ifdef DEBUG
  assert (classd != NULL);
#endif
  return (classd->base.parent);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_namelist *
odl_class_get_parentnames (odl_class * classd)
{
  odl_namelist *nl = NULL;
  GList *l;

#ifdef DEBUG
  assert (classd != NULL);
#endif
  l = classd->parents;
  while (l)
    {
      nl = g_list_append (nl, g_strdup (l->data));
      l = g_list_next (l);
    }
  return (nl);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_field *
odl_class_get_field (odl_class * classd, const char *fieldname)
{
  odl_class *c;

#ifdef DEBUG
  assert (classd != NULL);
  assert (fieldname != NULL);
#endif
  c = odl_field_defined_in (classd, fieldname);
  if (c)
    {
      return ((odl_field *) odl_find_relative ((odl_base *) c, fieldname));
    }
  return (NULL);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_fieldlist *
odl_class_get_fields (odl_class * classd, enum odl_fieldtype type)
{
  /* list of pointers to odl_field with indicated type */
  /* excludes inherited fields */
  odl_fieldlist *fl = NULL;
  GList *l;

#ifdef DEBUG
  assert (classd != NULL);
#endif
  l = classd->contents;
  while (l)
    {
      odl_base *b = l->data;

      if (b->type == IT_field)
        {
          odl_field *f = (odl_field *) b;

          if (f->fieldtype == type)
            fl = g_list_append (fl, f);
        }
      l = g_list_next (l);
    }
  return (fl);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_fieldlist *
odl_class_get_all_fields (odl_class * classd, enum odl_fieldtype type)
{
  /* list of pointers to odl_field with indicated type */
  /* includes inherited fields */
  odl_fieldlist *fl = NULL;
  odl_namelist *parents;
  GList *l;

#ifdef DEBUG
  assert (classd != NULL);
#endif
  parents = odl_class_get_parentnames (classd);
  fl = odl_class_get_fields (classd, type);
  l = parents;
  while (l)
    {
      odl_fieldlist *tmp;
      odl_class *c = odl_find_class (classd->base.tree, l->data, NULL);
      if (c)
        {
          tmp = odl_class_get_fields (c, type);
          if (tmp)
            {
              if (fl)
                {
                  fl = g_list_concat (fl, tmp);
                }
              else
                {
                  fl = tmp;
                }
            }
        }
      l = g_list_next (l);
    }
  if (parents)
    {
      odl_namelist_free (parents);
    }
  return (fl);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
gboolean
odl_class_is_instanceof (odl_class * classd, const char *classname)
{
  odl_namelist *parents;
  char *n1, *n2;

#ifdef DEBUG
  assert (classd != NULL);
  assert (classname != NULL);
#endif
  /* account for "root::" being in the full class name */
  n1 = (char *) odl_class_get_full_name (classd);
  n2 = (char *) classname;
  if (strncmp (n2, "root::", 6) != 0)
    {
      if (strncmp (n1, "root::", 6) == 0)
        {
          n1 += 6;
        }
    }
  if (g_strcasecmp (n1, n2) == 0)
    {
      return (TRUE);
    }
  parents = odl_class_get_parentnames (classd);
  if (odl_namelist_contains (parents, classname, FALSE))
    {
      odl_namelist_free (parents);
      return (TRUE);
    }
  odl_namelist_free (parents);
  return (FALSE);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
enum odl_fieldtype
odl_field_get_type (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->fieldtype);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
enum odl_datatype
odl_field_get_datatype (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->datatype);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_field_get_datatype_classname (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->datatypeclass);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
gboolean
odl_field_has_property (odl_field * field, unsigned long int property)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->properties & property ? TRUE : FALSE);
}

/* ========================================================================= *\
 * Read data from: field
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_field_get_name (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->base.name);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_field_get_full_name (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (odl_get_base_full_name ((odl_base *) field));
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
unsigned long int
odl_field_get_access (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->base.access);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
unsigned long int
odl_field_get_triggers (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->base.triggers);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_base *
odl_field_get_class (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->base.parent);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_field_get_format (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->format);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_field_get_default (odl_field * field)
{
  int len;
  time_t timep;
  struct tm *tm;

#ifdef DEBUG
  assert (field != NULL);
#endif
  if (field->tmpdefault)
    {
      g_free (field->tmpdefault);
    }
  field->tmpdefault = NULL;
  switch (field->datatype)
    {
    case DT_date:
      if (!field->defaultval)
        {
          field->tmpdefault = g_strdup ("");
          return (field->tmpdefault);
        }
      if (g_strcasecmp (field->defaultval, "today") == 0)
        {
          len = strlen ("0000-00-00 AD");
          field->tmpdefault = g_malloc (len + 2);
          timep = time (NULL);
          tm = localtime (&timep);
          strftime (field->tmpdefault, len, "%G-%m-%d", tm);
          field->tmpdefault[len + 1] = '\0';
          return (field->tmpdefault);
        }
      break;
    case DT_time:
      if (!field->defaultval)
        {
          field->tmpdefault = g_strdup ("");
          return (field->tmpdefault);
        }
      if (g_strcasecmp (field->defaultval, "now") == 0)
        {
          len = strlen ("HH:MM:SS AM");
          field->tmpdefault = g_malloc (len + 2);
          timep = time (NULL);
          tm = localtime (&timep);
          strftime (field->tmpdefault, len, "%H:%M:%S", tm);
          field->tmpdefault[len + 1] = '\0';
          return (field->tmpdefault);
        }
      break;
    case DT_datetime:
      if (!field->defaultval)
        {
          field->tmpdefault = g_strdup ("");
          return (field->tmpdefault);
        }
      if (g_strcasecmp (field->defaultval, "now") == 0)
        {
          len = strlen ("0000-00-00 AD HH:MM:SS AM");
          field->tmpdefault = g_malloc (len + 2);
          timep = time (NULL);
          tm = localtime (&timep);
          strftime (field->tmpdefault, len, "%G-%m-%d %H:%M:%S", tm);
          field->tmpdefault[len + 1] = '\0';
          return (field->tmpdefault);
        }
      break;
    case DT_bool:
    case DT_boolean:
      break;
    default:
      break;
    }

  return (field->defaultval);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_field_get_sourcefield (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->sourcefield);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_field_get_sourceclass (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->sourceclass);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_fieldlist *
odl_field_get_this_fields (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->this_fields);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_fieldlist *
odl_field_get_source_fields (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->source_fields);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_argumentlist *
odl_field_get_arguments (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (field->arguments);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
unsigned long int
odl_method_argument_count (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return (g_list_length (field->arguments));
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_field_get_calculation (odl_field * field)
{
#ifdef DEBUG
  assert (field != NULL);
#endif
  return ("Not done yet  :)");
}


/* ========================================================================= *\
 * read data from: method argument
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_argument_get_name (odl_argument * arg)
{
#ifdef DEBUG
  assert (arg != NULL);
#endif
  return (arg->name);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
enum odl_datatype
odl_argument_get_datatype (odl_argument * arg)
{
#ifdef DEBUG
  assert (arg != NULL);
#endif
  return (arg->datatype);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
const char *
odl_argument_get_classdatatype (odl_argument * arg)
{
#ifdef DEBUG
  assert (arg != NULL);
#endif
  return (arg->classdatatype);
}

/* ========================================================================= *\
 * Utility functions
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_namelist *
odl_split_qualified_name (char *qualifiedname)
{
  GList *names = NULL;
  char *p, *q;

#ifdef DEBUG
  assert (qualifiedname != NULL);
  /* printf( "---- odl_split_qualified_name() is splitting: '%s'\n" , qualifiedname ); */
#endif
  /* char *tmp; */
  /* gboolean done = FALSE; */

  if (!qualifiedname)
    {
      return (NULL);
    }
  if (strlen (qualifiedname) == 0)
    {
      return (NULL);
    }
  /* if the name starts with :: ignore it, all other 0 length */
  /* names are errors */
  if (qualifiedname[0] == ':' && qualifiedname[1] == ':')
    {
      names = g_list_append (NULL, g_strdup ("root"));
#ifdef DEBUG
      /* printf( " <root>\n" ); */
#endif
      p = &qualifiedname[2];
    }
  else
    {
      p = qualifiedname;
      names = NULL;
    }
  while (*p != '\0')
    {
      q = strstr (p, "::");
      if (q)
        {
          *q = '\0';
        }
      else
        {
          names = g_list_append (names, g_strdup (p));
          break;                /* done last name */
        }

      if (strlen (p) == 0)
        {
          /* error: "::::" is illegal */
#ifdef DEBUG
          assert (strlen (p) != 0);
#endif
          odl_namelist_free (names);
          return (NULL);
        }
      else
        {
          names = g_list_append (names, g_strdup (p));
#ifdef DEBUG
          /* printf( " :%s:\n" , p ); */
#endif
          p = q + 2;
        }
      if (q)
        {
          *q = ':';
        }
    }
  return (names);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
char *
odl_mangle_qualified_name (const char *qualifiedname)
{
  char *name, *p;
  void *dest, *src;
  size_t size;

#ifdef DEBUG
  assert (qualifiedname != NULL);
#endif
  if (!qualifiedname)
    {
      return (NULL);
    }
  name = g_strdup (qualifiedname);
#ifdef DEBUG
  /* printf( "---- odl_mangle_qualified_name()\n" , name , strlen(name) ); */
  /* printf( "     > '%s' (%d)\n" , name , strlen(name) ); */
#endif
  if (strncmp (name, "root::", 6) == 0)
    {
      dest = name;
      src = &name[6];
      size = strlen (name) - 5;
      memmove (dest, (const void *) src, size);
    }
  p = strstr (name, "::");

  while (p)
    {
      /* replace :: with __ */
      *p = '_';
      p++;
      *p = '_';
      p++;

      /* next, if any */
#ifdef DEBUG
      /* printf( ": '%s' (%d)\n" , name , strlen(name) ); */
#endif
      p = strstr (p, "::");
    }
  return (name);
}

/* ------------------------------------------------------------------------- *\
 * Given an name and a container (parent) make a fullname
 * with all of the parents.
\* ------------------------------------------------------------------------- */
char *
odl_make_fullname( odl_base * parent, char * name )
{
  GString * name_string = g_string_new(NULL);
  char * result;
  odl_base * b = NULL;
 
  g_assert( parent != NULL);
  g_assert( name != NULL);
  
  name_string = g_string_append( name_string, name );
  b = parent;
  while (b)
    {
      name_string = g_string_prepend(name_string, "::");
      name_string = g_string_prepend(name_string, b->name);
      b = b->parent;
    }
  result = name_string->str;
  g_string_free( name_string, FALSE);
  return result;
}

/* ------------------------------------------------------------------------- *\
 * Given an item make a fullname with all of the parents.
\* ------------------------------------------------------------------------- */
char *
odl_make_fullname_base( odl_base * base )
{
  g_assert( base );
  return odl_make_fullname( base->parent, base->name);
}

/* ------------------------------------------------------------------------- *\
 * Actually process the type into new fields
 * Assumes that the field is a field and a type
\* ------------------------------------------------------------------------- */
static void
odl_process_type_field (odl_container * c, odl_field * f)
{
  odl_class *type_class;
  GList *type_contents;

  g_assert (c != NULL);
  g_assert (f != NULL);
  g_assert (f->base.type == IT_field);
  g_assert (f->datatype == DT_type);
#ifdef DEBUG
  /* printf( "---- odl_process_type_field(), look for TYPE '%s'\n" , f->datatypeclass ); */
#endif

  /* mark existing field as no longer needed */
  f->base.type = IT_ignore;
  type_class = odl_find_class (f->base.tree, f->datatypeclass, NULL);
  if (!type_class)
    {
      printf
        ("**** Error could not find TYPE: %s, defined in container %s\n",
         f->datatypeclass, c->base.name);
      printf ("     In filename: %s\n", ( (odl_container *)((odl_base *)f)->parent)->filename );
      return;
    }
  if (!type_class->istype)
    {
      printf ("**** Error trying to process field %s that is not a TYPE.\n", f->base.name);
      printf ("     In container: %s\n", ( ((odl_base *)f)->parent)->name );
      printf ("     In filename: %s\n", ( (odl_container *)((odl_base *)f)->parent)->filename );
      return;
    }
  /* it's a compound type - make new fields */
#ifdef DEBUG
  /* printf( "     Make field '%s.*'\n" , b->name ); */
#endif
  type_contents = type_class->contents;
  while (type_contents)         /* go through all contents of type */
    {
      odl_field *type_field = type_contents->data;
      odl_field *n;
      switch (type_field->base.type)
        {
        case IT_field:
          if (strcmp (type_field->base.name, "objectid") != 0)
            {
              switch (type_field->fieldtype)
                {
                case FT_basic:
                  n = alloc_odl_item ();
                  g_assert (n != NULL);
                  if (n)
                    {
                      n->base.type = IT_field;
                      n->base.parent = (odl_base *) c;
                      n->base.name =
                        g_strdup_printf ("%s.%s", f->base.name,
                                         type_field->base.name);
                      n->base.access = f->base.access;
                      n->base.tree = c->base.tree;
                      n->fieldtype = FT_basic;
                      n->datatype = type_field->datatype;
                      if (type_field->datatypeclass)
                        {
                          n->datatypeclass =
                            g_strdup (type_field->datatypeclass);
                        }
                      else
                        {
                          n->datatypeclass = NULL;
                        }
                      n->properties = type_field->properties;
                      n->format = g_strdup( type_field->format );
                      n->defaultval = g_strdup( type_field->defaultval );
                      c->contents = g_list_append (c->contents, n);
#ifdef DEBUG
                      /* printf( "     Made '%s'\n" , n->base.name ); */
#endif
                    }
                  else
                    {
                      yyerror ("Out of memory");
                    }
                  break;
                case FT_lookup:
                case FT_reference:
                case FT_list:
                case FT_method:
                case FT_calculated:
                case FT_readonly:
                default:
                  break;
                }
            }
          break;
        default:
#ifdef DEBUG
          /* printf
             ("     Did not recongize item in TYPE: %s\n",
             odl_itemtype_name (type_field->base.type)); */
#endif
          break;
        }
      type_contents = g_list_next (type_contents);
    }
}

/* ------------------------------------------------------------------------- *\
 *
\* ------------------------------------------------------------------------- */
void
odl_resolve_implicit_list (odl_item * i)
{
  char *name;
  odl_item *a;
  odl_field *fl;
  odl_class *cont;
  odl_class *load;
  g_assert (i != NULL);

  fl = (odl_field *) i;
  ((odl_base *) i)->type = IT_ignore;  /* mark old item as not needed */
                                       /* TODO should be able to reuse */
  cont = (odl_class *) fl->base.parent;
  load = odl_find_class (((odl_base *) i)->tree, i->datatypeclass, NULL);
  if (!load)
    {
    printf( "\n**** Error field reference to CLASS %s could not be found. (In CLASS %s)\n",
           i->datatypeclass, cont->base.name);
      printf( "     Make sure that %s is not a TYPE or is defined as a CLASS in a .gcd file.\n", i->datatypeclass);
      yyerrorcount++;
      return;
    }
  name = g_strdup_printf ("_%s_%s", odl_class_get_mangled_name
                          (cont), fl->base.name);
  a = alloc_odl_item ();
  g_assert (a != NULL);
  if (a)
    {
      a->base.parent = (odl_base *) load;
      a->base.name = g_strdup (name);
      a->base.access = i->base.access;
      a->base.type = IT_field;
      a->base.tree = i->base.tree;
      a->fieldtype = FT_basic;
      a->datatype = DT_object;
      a->properties = ODL_PROP_NONE;
      load->contents = g_list_append (load->contents, a);
    }
  else
    {
      yyerror ("Out of memory");
    }
  /* turn 'fl' into a list field */
  a = alloc_odl_item ();
  g_assert (a != NULL);
  if (a)
    {
      a->base.name = g_strdup (fl->base.name);
      a->base.access = i->base.access;
      a->base.type = IT_field;
      a->base.tree = i->base.tree;
      a->fieldtype = FT_list;
      a->datatype = DT_class;
      a->properties = ODL_PROP_NONE;
      a->sourceclass = g_strdup (i->datatypeclass);
      a->source_fields = g_list_append (NULL, name);
      a->this_fields = g_list_append (NULL, g_strdup ("objectid"));
      /* find the 'real' version in the class tree */
      cont = odl_find_class (((odl_base *) i)->tree,
                             odl_class_get_full_name (cont), NULL);
      a->base.parent = (odl_base *) cont;
      cont->contents = g_list_append (cont->contents, a);
      /* verify that the list referres to a class and not a type */
    }
  else
    {
      yyerror ("Out of memory");
    }
  fl->fieldtype = FT_list;
}


/* ------------------------------------------------------------------------- *\
 * Convert the field (i) into a reference object field and build the (a) field
 * which is used to store the objectid of the referenced class.
\* ------------------------------------------------------------------------- */
void
odl_resolve_implicit_reference (odl_item * i, GList * append)
{
  odl_item *a;
  odl_field *fl = (odl_field *) i;
  odl_class *current_container = (odl_class *) fl->base.parent;
  odl_class *target_class = NULL;
  g_assert( i != NULL);
  g_assert( append != NULL);
  
  /* convert the existing field to a reference field */
  i->base.type = IT_field;
  i->base.access = i->base.access;
  i->fieldtype = FT_reference;
  i->source_fields =
    (odl_namelist *) g_list_append (NULL, g_strdup ("objectid"));
  i->this_fields =
    (odl_namelist *) g_list_append (NULL,
                                    g_strdup_printf ("_%s", i->base.name));
  i->sourceclass = g_strdup (i->datatypeclass);
  /* verify that the reference is to a class and not a type */
  target_class = odl_find_class(  i->base.tree, i->datatypeclass, NULL);
  if (target_class == NULL)
    {
    printf( "\n**** Error field reference to CLASS %s could not be found. (In CLASS %s)\n",
           i->datatypeclass, current_container->base.name);
    printf( "     Make sure that %s is not a TYPE or is defined as a CLASS in a .gcd file.\n", i->datatypeclass);
    yyerrorcount++;
    return;
    }
  /* create the new _fieldname objectreference field */
  a = alloc_odl_item ();
  g_assert (a != NULL);
  if (a)
    {
      a->base.parent = (odl_base *) current_container;
      a->base.name = g_strdup_printf ("_%s", i->base.name);
      a->base.access = i->base.access;
      a->base.type = IT_field;
      a->base.tree = i->base.tree;
      a->fieldtype = FT_basic;
      a->datatype = DT_object;
      a->properties = ODL_PROP_NONE;
      append = g_list_append (append, a);
    }
  else
    {
      yyerror ("Out of memory");
    }
}


/* ------------------------------------------------------------------------- *\
 * Scan the data base and replace all fields that are TYPES (not real fields)
 * with the wanted exploded fields from the TYPE definition.
 * TODO move to gcdparser.c
\* ------------------------------------------------------------------------- */
static void
odl_process_compound_types (odl_container * c)
{
  GList *container_contents;
  g_assert (c != NULL);

  container_contents = c->contents;
  while (container_contents)
    {
      odl_base *b = container_contents->data;
#ifdef DEBUG
      /*
         if (b->type == IT_field)
         {
         if (((odl_item *) b)->fieldtype == FT_basic)
         {
         if (((odl_item *) b)->datatype == DT_class)
         {
         printf
         ("**** ERROR unresolved field: %s, in container %s\n",
         b->name, c->base.name);
         }
         }
         }
       */
#endif
      /* process sub items */
      switch (b->type)
        {
        case IT_class:
          odl_process_compound_types ((odl_container *) b);
          break;
        case IT_module:
          odl_process_compound_types ((odl_container *) b);
          break;
        case IT_field:
          {
            odl_field *f;
            f = (odl_field *) b;
#ifdef DEBUG
            /*
               if (f->datatype == DT_class)
               {
               printf( "---- odl_process_compound_types(), look for CLASS: %s\n" , f->datatypeclass );
               printf( "                                        fieldtype: %s\n" , odl_fieldtype_name(f->fieldtype) );
               printf( "                                        container: %s\n" , c->base.name );
               }
             */
#endif
            if (f->datatype == DT_type)
              {
                odl_process_type_field (c, f);
              }
          }
          break;
        case IT_unknown:
        case IT_enum:
        case IT_type:
        default:
          /* these are processed later */
          break;
        case IT_ignore:
          break;
        }
      container_contents = g_list_next (container_contents);
    }
}

static void
odl_process_references_lists (odl_container * c)
{
  GList *container_contents;
  g_assert (c != NULL);

  container_contents = c->contents;
  while (container_contents)
    {
      odl_base *b = container_contents->data;
      switch (b->type)
        {
        case IT_class:
          odl_process_references_lists ((odl_container *) b);
          break;
        case IT_module:
          odl_process_references_lists ((odl_container *) b);
          break;
        case IT_field:
          if ((((odl_item *) b)->datatype == DT_class)
              && (((odl_item *) b)->fieldtype == FT_basic))
            {
              /* if -1 turn it into a reference  */
              /* else 0 or positive a list class */
              /* else < -1 invalid               */
              g_assert (((odl_item *) b)->bounds > -2);
              if (((odl_item *) b)->bounds == (-1))
                {
                  /* do reference field */
                  odl_resolve_implicit_reference ((odl_item *) b,
                                                  ((odl_container *) (b->
                                                                      parent))->
                                                  contents);
                }
              else
                {
                  /* do list field */
                  odl_resolve_implicit_list ((odl_item *) b);
                }
            }
          break;
        case IT_unknown:
        case IT_enum:
        case IT_type:
        default:
          break;
        case IT_ignore:
          break;
        }
      container_contents = g_list_next (container_contents);
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static void
odl_link_all_to_tree (odl_base * b, odl_tree * tree)
{
  GList *l;

#ifdef DEBUG
  assert (tree != NULL);
  assert (b != NULL);
#endif
  b->tree = tree;

  if (b->type == IT_field)
    {
      odl_field *f = (odl_field *) b;

      if (f->fieldtype == FT_lookup)
        {
          odl_class *c;

          f->properties = ODL_PROP_READONLY | f->properties;
          c = odl_find_class (tree, f->sourceclass, NULL);
          if (c)
            {
              odl_field *fd;

              fd = odl_class_get_field (c, f->sourcefield);
              if (fd)
                {
                  f->datatype = fd->datatype;
                  if (fd->datatypeclass)
                    {
                      f->datatypeclass = g_strdup (fd->datatypeclass);
                    }
                  if (fd->format)
                    {
                      f->format = g_strdup (fd->format);
                    }
                  if (fd->defaultval)
                    {
                      f->defaultval = g_strdup (fd->defaultval);
                    }
                }
            }
        }
    }

  if (b->type == IT_class || b->type == IT_module)
    {
      l = ((odl_container *) b)->contents;
      while (l)
        {
          odl_link_all_to_tree ((odl_base *) l->data, tree);
          l = g_list_next (l);
        }
    }
}

odl_tree *yycurrenttree = NULL;

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_tree *
odl_load_files (odl_filenamelist * files, odl_tree * tree)
{
  GList *l;
  odl_tree *newtree;
  int max_errors = 0;

  /* use old tree, or allocate a new one */
  if (tree)
    {
      newtree = tree;
    }
  else
    {
      newtree = alloc_odl_tree ();
    }
#ifdef DEBUG
  assert (newtree != NULL);
  assert (files != NULL);
#endif
  if (!newtree)
    {
      return (NULL);
    }

  yycurrenttree = newtree;

  /* setup */
  yyerrorcount = 0;
  yywarncount = 0;

  /* pass 1 */
  current_access = ODL_ACCESS_DEFAULT;
  current_pass = 1;
  expecting_datatype = 1;
  expecting_fieldprop = 0;
  expect_triggertype = 0;
  yycurrent_container = odl_tree_get_root (newtree);
  g_assert (yycurrent_container != NULL);
  l = (GList *) files;
#ifdef DEBUG
      printf ("\n");
#endif
  while (l)
    {
#ifdef DEBUG
      printf ("++++ Pass 1 File: %s\n", (char *) l->data);
#endif
      switch (yystartfile (l->data))
        {
        case 0:
          /* already processed */
          break;
        case 1:
          /* process now */
          errors_show_file (1);
          yyparse ();
          break;
        case (-1):
          /* error */
          yyerror ("Error opening file %s", (char *) l->data);
          break;
        }

      l = g_list_next (l);
    }

  /* end of pass 1 */
  errors_show_file (0);
  clear_file_history ();

  if (yyerrorcount > max_errors)
    {
      yymessage ("Too many errors. File load failed.");
      return (NULL);
    }

  /* pass 2 */
  current_access = ODL_ACCESS_DEFAULT;
  current_pass = 2;
  expecting_datatype = 1;
  expecting_fieldprop = 0;
  expect_triggertype = 0;
  l = (GList *) files;
#ifdef DEBUG
  printf ("\n");
#endif
  while (l)
    {
#ifdef DEBUG
      printf ("++++ Pass 2 File: %s\n", (char *) l->data);
#endif
      switch (yystartfile (l->data))
        {
        case 0:
          /* already processed */
          break;
        case 1:
          /* process now */
          errors_show_file (1);
          yyparse ();
          break;
        case (-1):
          /* error */
          yyerror ("Error opening file %s", (char *) l->data);
          break;
        }

      l = g_list_next (l);
    }

  /* end of pass 2 */
  errors_show_file (0);
  clear_file_history ();

  if (yyerrorcount > max_errors)
    {
      yymessage ("Too many errors in parsing. File load failed.");
      return (NULL);
    }

  /* done, but these may generate more errors */
  odl_link_all_to_tree ((odl_base *) newtree->root, newtree);
  odl_process_compound_types ((odl_container *) newtree->root);
  odl_process_references_lists ((odl_container *) newtree->root);
  
  if (yyerrorcount > max_errors)
    {
      yymessage ("Too many errors in post processing. File load failed.");
      /* return (NULL); */
    }

  odl_link_all_to_tree ((odl_base *) newtree->root, newtree);   /* TODO sloppy, one final time for new entries */
  return (newtree);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
static void
real_odl_display_tree (FILE * out, odl_container * item, int indent,
                       gboolean show_full_name)
{
  int i;
  GList *l;
  GList *s;
  GList *t;
  GList *lst;
  odl_item *it = NULL;
  char *buf;
  g_assert (item != NULL);
  g_assert (out != NULL);

  if (item->base.type == IT_ignore)
    {
      return;
    }

  /* indent this item */
  for (i = 0; i < indent; i++)
    {
      fprintf (out, " ");
    }
  /* print type and name */
  if (item->base.type == IT_class)
    {
      if (item->istype)
        {
          fprintf (out, "%s %s (%s)", odl_access_name (item->base.access),
                   item->base.name, "type");
        }
      else
        {
          fprintf (out, "%s %s (%s)", odl_access_name (item->base.access),
                   item->base.name, "class");
        }
    }
  else
    {
      fprintf (out, "%s %s (%s)", odl_access_name (item->base.access),
               item->base.name, odl_itemtype_name (item->base.type));
    }

  switch (item->base.type)
    {
    case IT_module:
      buf = (char *) odl_module_get_full_name (item);
      break;
    case IT_class:
      buf = (char *) odl_class_get_full_name (item);
      break;
    case IT_field:
      buf = (char *) odl_field_get_full_name ((odl_field *) item);
      break;
    case IT_enum:
      buf = (char *) odl_enum_get_full_name ((odl_enum *) item);
      break;
    default:
      buf = g_strdup ("<unknown>");
    }
  if (buf && show_full_name)
    {
      if (strncmp (buf, "root::", 6) == 0)
        fprintf (out, " [full name: %s]", (&buf[6]));
      else
        fprintf (out, " [full name: %s]", buf);
    }

  /* print children of a module or class */
  switch (item->base.type)
    {
    case IT_class:
      if (item->filename != NULL)
        {
          fprintf(out, " [%s]", g_basename(((odl_container *)item)->filename));
        }
      l = item->parents;
      if (l)
        {
          fprintf (out, "\n");
          for (i = 0; i < indent; i++) fprintf (out, " ");
          fprintf (out, "Parents:");
          while (l)
            {
              fprintf (out, " %s", (char *) l->data);
              if (l->next)
                {
                  fprintf (out, ",");
                }
              l = g_list_next (l);
            }
        }
      l = item->indexes;
      if (l)
        {
          fprintf (out, "\n");
          /*
             for (i = 0; i < indent; i++)
             {
             fprintf (out, " ");
             }
             fprintf (out, "Indexes:\n");
           */
          while (l)
            {
              _odl_index *idx = l->data;
              GList *l2 = idx->fields;

              if ((idx->primary) || (idx->unique) || (l2))
                {
                  for (i = 0; i < indent; i++)
                    {
                      fprintf (out, " ");
                    }
                }
              if (idx->primary)
                {
                  fprintf (out, "PRIMARY KEY: ");
                }
              else if (idx->unique)
                {
                  fprintf (out, "UNIQUE: ");
                }
              else
                {
                  fprintf (out, "INDEX: ");
                }
              while (l2)
                {
                  fprintf (out, "%s", (char *) l2->data);
                  if (l2->next)
                    fprintf (out, ", ");
                  l2 = g_list_next (l2);
                }
              fprintf (out, "\n");
              l = g_list_next (l);
            }
        }
      if (item->orderby)
        {
          for (i = 0; i < indent; i++)
            {
              fprintf (out, " ");
            }
          fprintf (out, "ORDER BY %s%s", ((odl_container *) item)->orderby ,
		   ((odl_container *) item)->desc ? " (descending)" : "" );
		   
          fprintf (out, "\n");
        }
      l = item->contents;
      while (l)
        {
          real_odl_display_tree (out, l->data, indent + INDENT,
                                 show_full_name);
          l = g_list_next (l);
        }
      fprintf (out, "\n");
      break;
    case IT_module:
      l = item->contents;
      if (item->filename != NULL)
        {
          fprintf(out, " [%s]", g_basename(item->filename));
        }
      fprintf(out, "\n");
      while (l)
        {
          real_odl_display_tree (out, l->data, indent + INDENT,
                                 show_full_name);
          l = g_list_next (l);
        }
      fprintf (out, "\n");
      break;
    case IT_enum:
      /* indent this item */
      fprintf (out, "\n");
      for (i = 0; i < indent; i++)
        {
          fprintf (out, " ");
        }
      fprintf (out, "Values: ");
      lst = ((struct _odl_item *) item)->elements;
      while (lst)
        {
          fprintf (out, " %s", (char *) lst->data);
          lst = g_list_next (lst);
        }
      fprintf (out, "\n");
      break;
    case IT_field:
      it = (odl_item *) item;
      switch (it->fieldtype)
        {
        case FT_basic:
          fprintf (out, " Basic: %s", odl_datatype_name (it->datatype));
          if (it->format != NULL)
            {
              fprintf (out, "<%s>", it->format);
            }
          if (it->properties & ODL_PROP_NOTNULL)
            {
              fprintf (out, " NOT NULL");
            }
          if (it->properties & ODL_PROP_READONLY)
            {
              fprintf (out, " READONLY");
            }
          if (it->properties & ODL_PROP_SERVER)
            {
              fprintf (out, " SERVER");
            }
          if (it->properties & ODL_PROP_HIDDEN)
            {
              fprintf (out, " HIDDEN");
            }
          fprintf (out, "\n");
          break;
        case FT_lookup:
          fprintf (out, " Lookup: load field %s from class %s where\n",
                   it->sourcefield, it->sourceclass);
          s = it->source_fields;
          t = it->this_fields;
          while (s && t)
            {
              for (i = 0; i < indent; i++)
                {
                  fprintf (out, " ");
                }
              fprintf (out, "    %s.%s = this.%s\n", it->sourceclass,
                       (char *) s->data, (char *) t->data);
              s = g_list_next (s);
              t = g_list_next (t);
            }
          break;
        case FT_list:
          fprintf (out, " List: load class %s where\n", it->sourceclass);
          s = it->source_fields;
          t = it->this_fields;
          while (s && t)
            {
              for (i = 0; i < indent; i++)
                {
                  fprintf (out, " ");
                }
              fprintf (out, "    %s.%s = this.%s\n", it->sourceclass,
                       (char *) s->data, (char *) t->data);
              s = g_list_next (s);
              t = g_list_next (t);
            }
          break;
        case FT_reference:
          fprintf (out, " Reference: load class %s where\n", it->sourceclass);
          s = it->source_fields;
          t = it->this_fields;
          while (s && t)
            {
              for (i = 0; i < indent; i++)
                {
                  fprintf (out, " ");
                }
              fprintf (out, "    %s.%s = this.%s\n", it->sourceclass,
                       (char *) s->data, (char *) t->data);
              s = g_list_next (s);
              t = g_list_next (t);
            }
          break;
        case FT_method:
          fprintf (out, " Method: returns: %s\n",
                   odl_datatype_name (it->datatype));
          for (i = 0; i < indent; i++)
            {
              fprintf (out, " ");
            }
          fprintf (out, "       arguments:");
          s = it->arguments;
          if (!s)
            {
              fprintf (out, " <none>");
            }
          while (s)
            {
              fprintf (out, " %s %s",
                       odl_datatype_name (((odl_argument *) s->
                                           data)->datatype),
                       ((odl_argument *) s->data)->name);
              s = g_list_next (s);
              if (s)
                {
                  fprintf (out, ",");
                }
            }
          fprintf (out, "\n");
          break;
        case FT_calculated:
          fprintf (out, " Calculated:");
          fprintf (out, "\n");
          break;
        case FT_readonly:
          fprintf (out, " Readonly:");
          fprintf (out, "\n");
          break;
        case FT_unknown:
          fprintf (out, " Unknown field type");
          fprintf (out, "\n");
          break;
        }
      break;
    case IT_unknown:
      /* indent this item */
      fprintf (out, "Unknown item (%s)\n", it->base.name);
      fprintf (out, "\n");
      break;
    case IT_type:
      fprintf (out, "Type: (%s)\n", it->base.name);
      fprintf (out, "\n");
      break;
    case IT_ignore:
      fprintf (out, "Ignore: (%s)\n", it->base.name);
      fprintf (out, "\n");
      break;                    /* added to satisfy compiler warning neilt */
    }
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
odl_display_tree (FILE * out, odl_tree * tree, gboolean show_full_name)
{
  g_assert (out != NULL);
  fprintf (out, "\nCurrent Class Tree\n\n");
  if (tree)
    {
      real_odl_display_tree (out, tree->root, 0, show_full_name);
    }
  else
    {
      fprintf (out, "No clases found\n");
    }
  fprintf (out, "\n");
}


/* ------------------------------------------------------------------------- *\
 * Dump the current tree for debugging purposes
\* ------------------------------------------------------------------------- */
void
odl_dump_tree (FILE * out, odl_tree * tree, gboolean recursive)
{
  g_assert (out != NULL);
  fprintf (out, "\nCurrent Class Tree\n\n");
  if (tree)
    {
      odl_dump1_tree (out, (odl_base *) tree->root, recursive);
    }
  else
    {
      fprintf (out, "No clases found\n");
    }
  fprintf (out, "\n");
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
void
odl_free_tree (odl_tree * tree)
{
  g_assert (tree != NULL);
  free_odl_tree (tree);
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_tree *
odl_create_empty_tree ()
{
  return (alloc_odl_tree ());
}

/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
odl_filenamelist *
odl_get_files_from_dir (odl_filenamelist * filelist, const char *name)
{
  struct dirent *next;
  DIR *reading;
  odl_filenamelist *fl = NULL;
  char *linkbuf, *classdir;
  GList *subdirs = NULL, *tmp;
  GString *currpath = NULL;
  struct stat s, s2;
  int linksize, err;
  gboolean is_dir = FALSE;

  g_assert( name );
  if (name == NULL)
    {
      perror ("Empty name.");
      exit (-1);
    }
  classdir = g_strdup (name);

  /* find GCD files */
  if (classdir == NULL)
    {
      perror
        ("No class definitions were found. Please ensure the configuration file\ndefines a directory with valid '.gcd' files.");
      exit (-1);
    }
  /* whack out ending slashes to create path */
  while (classdir[strlen (classdir) - 1] == '\\' ||
         classdir[strlen (classdir) - 1] == G_DIR_SEPARATOR)
    {
      classdir[strlen (classdir) - 1] = '\0';
    }

  currpath = g_string_new (classdir);
  currpath = g_string_append (currpath, G_DIR_SEPARATOR_S);
  subdirs = g_list_append (subdirs, g_strdup (classdir));

  while (g_list_length (subdirs) != 0)
    {
      reading = opendir (currpath->str);
      if (!reading)
        {
          perror ((const char *) currpath->str);
          closedir (reading);
          exit (-1);
        }
      else
        {
	  while ((next = readdir (reading)) != NULL)
	    {
	      /* Skip ./ and ../ directories */
	      if (strcmp (next->d_name, ".") == 0)
		{
		  next = readdir (reading);
		  continue;
		}
	      if (strcmp (next->d_name, "..") == 0)
		{
		  next = readdir (reading);
		  continue;
		}
              err = lstat (g_strdup_printf ("%s%s", currpath->str, next->d_name), &s);
              if (err == -1)
                {
                  perror (strerror (errno));
                  exit (-1);
                }
              if (S_ISLNK (s.st_mode))
                {
                  linkbuf = (char *)g_new (gchar *, PATH_MAX + 2);
                  linksize = readlink (g_strdup_printf ("%s/%s", currpath->str, next->d_name), 
				       linkbuf, PATH_MAX + 1);
                  if (linksize == -1)
                    {
                      perror (strerror (errno));
                      exit (-1);
                    }
                  linkbuf[linksize] = '\0';
                  if (g_strcasecmp (classdir, linkbuf)
                      && g_strncasecmp ("./", linkbuf, 2)
                      && g_strncasecmp ("../", linkbuf, 3)
                      && g_strcasecmp (currpath->str, linkbuf))
                    {
                      err = stat (linkbuf, &s2);
                      if (err == -1)
                        {
                          perror (strerror (errno));
                          exit (-1);
                        }
                      if (S_ISDIR (s2.st_mode))
                        {
                          subdirs =
                            g_list_append (subdirs,
                                           g_strdup_printf ("%s/", linkbuf));
                        }
                    }
                  g_free (linkbuf);
                }
              if (S_ISDIR (s.st_mode))
                {
                  is_dir = TRUE;
                  subdirs =
                    g_list_append (subdirs,
                                   g_strdup_printf ("%s%s/", currpath->str,
                                                    next->d_name));
                }
              if (odl_is_extension (next->d_name, "gcd"))
                {
                  char *tmp =
                    g_strdup_printf ("%s%s", currpath->str, next->d_name);
                  if (tmp)
                    {
                      fl = odl_filenamelist_add (fl, tmp);
                      g_free (tmp);
                    }
                }
            }
          closedir (reading);
          subdirs = g_list_remove (subdirs, (g_list_first (subdirs))->data);
          if (subdirs)
            {
              g_string_free (currpath, TRUE);
              currpath = g_string_new ((g_list_first (subdirs))->data);
            }
        }
    }

  g_free (classdir);

  tmp = subdirs;
  while (tmp)
    {
      g_free (tmp->data);
      tmp = g_list_next (tmp);
    }
  g_list_free (subdirs);

  return fl;
}

/** \brief Test if a file ends with a particular extension
 */
/* ------------------------------------------------------------------------- *\
 * 
\* ------------------------------------------------------------------------- */
gboolean 
odl_is_extension(char *filename, char *ext)
{
  int flen = strlen (filename);
  int elen = strlen (ext);

  /* printf( "filename: '%s' (%d)\n" , filename , strlen(filename) ); *
     printf( "ext     : '%s' (%d)\n" , ext      , strlen(ext) ); printf( *
     "compare(%s,%s)\n" , &filename[flen - elen], ext ); */

  if ((elen + 1) > flen)
    return (FALSE);
  if (g_strcasecmp (&filename[flen - elen], ext) == 0 &&
      filename[flen - elen - 1] == '.')
    return (TRUE);

  return (FALSE);
}
