/*
   Database communication monitoring

   Copyright (C) 2001 Free Software Foundation

   This file is part of the GNU Enterprise Application Server (GEAS)

   GEAS is free software; you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2, or (at your option) any later
   version.

   GEAS is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.

   You should have received a copy of the GNU General Public License
   along with GEAS; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

   $Id: datamonitor.c,v 1.11 2001/06/06 20:31:24 reinhard Exp $
*/

/* ------------------------------------------------------------------------- *\
 * FIXME:
 * Remove duplicate code in dm_*
 * Use a common prefix (e.g. geas_dm_) for all external functions
 * Use glib logging functions if possible instead of fprintf
\* ------------------------------------------------------------------------- */

#include "config.h"
#include "datamonitor.h"
#include <stdio.h>
#include <time.h>
#include <glib.h>

/* ========================================================================= *\
 * Private structures
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * Name of the logfile and wich events to log into this file
\* ------------------------------------------------------------------------- */
struct _dm_logfile
{
  gchar *name;
  guint32 events;
  guint32 flags;
};

typedef struct _dm_logfile dm_logfile;

/* ========================================================================= *\
 * Global variables
\* ========================================================================= */

static gboolean _dm_open = FALSE;
static GList *_dm_logfiles = NULL;

/* ========================================================================= *\
 * Private functions
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * Return the name of the event
\* ------------------------------------------------------------------------- */
static const gchar *
_dm_get_event_name (guint32 event)
{
  if (event == DM_EVENT_OBJECT_NEW)             return "new";
  if (event == DM_EVENT_OBJECT_DELETE)          return "delete";
  if (event == DM_EVENT_FIELD_WRITE)            return "write";
  if (event == DM_EVENT_FIELD_READ)             return "read";

  if (event == DM_TRANSACTION_ROLLBACK)         return "rollback";
  if (event == DM_TRANSACTION_COMMIT)           return "commit";
  if (event == DM_TRANSACTION_BEGIN)            return "begin";
  if (event == DM_TRANSACTION_CHECKPOINT)       return "checkpoint";

  if (event == DM_EVENT_SECURITY_LOGON_SUCCESS) return "login_success";
  if (event == DM_EVENT_SECURITY_LOGON_FAIL)    return "login_fail";
  if (event == DM_EVENT_SECURITY)               return "security";

  if (event == DM_EVENT_ERROR)                  return "error";
  if (event == DM_EVENT_WARNING)                return "warning";
  if (event == DM_EVENT_MESSAGE)                return "message";

  if (event == DM_EVENT_DATABASE)               return "database";

  return "unknown event";
}

/* ------------------------------------------------------------------------- *\
 * Write the first part of a line of the logfile
\* ------------------------------------------------------------------------- */
static void
_dm_write_logline_start (FILE * fp, const gchar *username,
                         const gchar *transactionid, guint32 event)
{
  /* 'theclock' was going to be a microsecond accurate timer, but that    */
  /* didn't work well. instead, it is incremented each time this function */
  /* is called, wrapping after 4294967295 */
  /* the result is every log entry has a gruanteeed unique first 38 */
  /* characters, and can always be sorted accurately, even when multiple */
  /* entries are made in the same second (time() is limited to 1 sec     */
  /* accuracy */
  /* char *timestr; */
  char timestr[32];
  /* clock_t ticker = clock(); */
  static unsigned long int ticker = 0;
  time_t thetime = time (NULL);
  struct tm *tm;
  static int last_sec = 0;

  /* print time, including microseconds */
  /* timestr = g_strdup( ctime(&thetime) ); */
  /* YYYY-MM-DD HH:MM:SS xxxxxxxx */
  tm = localtime (&thetime);
  strftime (timestr, 32, "%Y-%m-%d %H:%M:%S", tm);

  /* for each entry that occurs in the same second, increment the
     ticker. when we reach a later second, reset to 0.
     Even if the first call is in second 0 of the minute,
     the ticker is statically initialised, so will be usable
     this ensures every call can have a unique ticker value,
     provided this function is made thread safe - TODO */
  if (last_sec == tm->tm_sec)
    ticker++;
  else
    ticker = 1;
  last_sec = tm->tm_sec;

  /* [ time event user transaction ] extra info */
  while (timestr[strlen (timestr) - 1] == '\n' ||
         timestr[strlen (timestr) - 1] == 13 ||
         timestr[strlen (timestr) - 1] == 10)
    timestr[strlen (timestr) - 1] = '\0';
  /* fprintf(fp, "[%s %010lu e:" , timestr , ticker ); */
  fprintf (fp, "[%s %08lu e:", timestr, ticker);
  /* g_free( timestr ); */

  fprintf (fp, " %s", _dm_get_event_name (event));
  if (username)
    fprintf (fp, " u:%s", username);
  else
    fprintf (fp, " u:<server>");
  if (transactionid)
    fprintf (fp, " t:%s", transactionid);
  fprintf (fp, "] ");
}

/* ========================================================================= *\
 * Public functions
\* ========================================================================= */

/* ------------------------------------------------------------------------- *\
 * Initialize datamonitor - to be called at the start of a program
\* ------------------------------------------------------------------------- */
void
init_datamonitor (void)
{
  _dm_logfiles = NULL;
  _dm_open = TRUE;
}

/* ------------------------------------------------------------------------- *\
 * Shut down datamonitor - to be called at the end of a program
\* ------------------------------------------------------------------------- */
void
close_datamonitor (void)
{
  GList *l;

  _dm_open = FALSE;

  if (_dm_logfiles)
    {
      l = _dm_logfiles;
      while (l)
        {
          if (l->data)
            {
              g_free (((dm_logfile *) l->data)->name);
              g_free (l->data);
            }
          l = g_list_next (l);
        }
      g_list_free (_dm_logfiles);
      _dm_logfiles = NULL;
    }
}

/* ------------------------------------------------------------------------- *\
 * Add a file to the list of active logfiles
\* ------------------------------------------------------------------------- */
void
add_datamonitor_logfile (const gchar *filename, guint32 events, guint32 flags)
{
  dm_logfile *logfile;

  g_return_if_fail (filename);

  logfile = g_new0 (dm_logfile, 1);

  logfile->name = g_strdup (filename);
  logfile->events = events;
  logfile->flags = flags;

  _dm_logfiles = g_list_append (_dm_logfiles, logfile);
}

/* ------------------------------------------------------------------------- *\
 * Log a predefined event
\* ------------------------------------------------------------------------- */
void
dm_event (const gchar *username, const gchar *transactionid, guint32 event,
          ...)
{
  GList *l = _dm_logfiles;
  gboolean write;
  dm_logfile *lf;

  /* write event to log files */
  while (l)
    {
      lf = (dm_logfile *) l->data;
      /* printf( "Checking file %s\n" , lf->name ); */
      write = FALSE;
      /* printf( "[%d %d %d] " , lf->events , event , lf->events & event ); */
      if ((lf->events & event) == event)
        {
          write = TRUE;
          if (transactionid)
            {
              /* transaction, only write if allowed */
              /* printf( "   %d %d " , lf->flags , lf->flags & DM_LOG_ALLOW_TRANSACTIONS ); */
              if ((lf->flags & DM_LOG_ALLOW_TRANSACTIONS) == 0)
                {
                  write = FALSE;
                  /* printf( "transaction not allowed\n" ); */
                }
            }
          else
            {
              /* no transaction: if required then don't write */
              /* printf( "   %d %d " , lf->flags , lf->flags & DM_LOG_REQUIRE_TRANSACTIONS ); */
              if ((lf->flags & DM_LOG_REQUIRE_TRANSACTIONS) ==
                  DM_LOG_REQUIRE_TRANSACTIONS)
                {
                  write = FALSE;
                  /* printf( "transaction required\n" ); */
                }
            }
        }
      else
        {
          /* printf( "event not logged\n" ); */
          write = FALSE;
        }

      if (write)
        {
          FILE *fp = fopen (lf->name, "a");
          if (fp)
            {
              va_list a;

              /* printf( "  event logged\n" ); */
              _dm_write_logline_start (fp, username, transactionid, event);

              /* display event details */
              switch (event)
                {
                case DM_EVENT_OBJECT_NEW:
                  /* classname/object id */
                  va_start (a, event);
                  vfprintf (fp, "%s/%s", a);
                  va_end (a);
                  break;
                case DM_EVENT_OBJECT_DELETE:
                  /* classname/objectid */
                  va_start (a, event);
                  vfprintf (fp, "%s/%s", a);
                  va_end (a);
                  break;
                case DM_EVENT_FIELD_WRITE:
                  /* classname/objectid.field = value */
                  va_start (a, event);
                  vfprintf (fp, "%s/%s.%s = %s", a);
                  va_end (a);
                  break;
                case DM_EVENT_FIELD_READ:
                  /* classname/objectid.fieldname */
                  va_start (a, event);
                  vfprintf (fp, "%s/%s.%s", a);
                  va_end (a);
                  break;
                case DM_TRANSACTION_BEGIN:
                case DM_TRANSACTION_COMMIT:
                case DM_TRANSACTION_ROLLBACK:
                case DM_TRANSACTION_CHECKPOINT:
                  /* no extra info for transaction events */
                  break;
                }
              fprintf (fp, "\n");

              /* done */
              fclose (fp);
            }
        }
      l = g_list_next (l);
    }
  /* printf( "\n" ); */


  /* handle an event */

  /* abort data monitor if not initialised */
  if (!_dm_open)
    return;

  /* code here should be able to track data requests, and optimise */
  /* the data cache and other systems */
}

/* ------------------------------------------------------------------------- *\
 * Log anything except a predefined event
\* ------------------------------------------------------------------------- */
void
dm_logentry (guint32 type, const gchar *username, const gchar *fmt, ...)
{
  GList *l = _dm_logfiles;
  gboolean write;
  dm_logfile *lf;

  /* write event to log files */
  while (l)
    {
      lf = (dm_logfile *) l->data;
      /* printf( "Checking file %s\n" , lf->name ); */
      write = FALSE;
      /* printf( "[%d %d %d] " , lf->events , event , lf->events & event ); */
      if ((lf->events & type) == type)
        {
          write = TRUE;
        }
      else
        {
          /* printf( "event not logged\n" ); */
          write = FALSE;
        }

      if (write)
        {
          FILE *fp = fopen (lf->name, "a");
          if (fp)
            {
              va_list a;

              /* printf( "  event logged\n" ); */
              _dm_write_logline_start (fp, username, NULL, type);

              /* display event details */
              if (fmt)
                {
                  va_start (a, fmt);
                  vfprintf (fp, fmt, a);
                  va_end (a);
                }
              fprintf (fp, "\n");

              /* done */
              fclose (fp);
            }
        }
      l = g_list_next (l);
    }
  /* printf( "\n" ); */
}

/* ------------------------------------------------------------------------- *\
 * Same as dm_logentry, but with va_list parameter
\* ------------------------------------------------------------------------- */
void
dm_vlogentry (guint32 type, const gchar *username, const gchar *fmt, va_list a)
{
  GList *l = _dm_logfiles;
  gboolean write;
  dm_logfile *lf;

  /* write event to log files */
  while (l)
    {
      lf = (dm_logfile *) l->data;
      /* printf( "Checking file %s\n" , lf->name ); */
      write = FALSE;
      /* printf( "[%d %d %d] " , lf->events , event , lf->events & event ); */
      if ((lf->events & type) == type)
        {
          write = TRUE;
        }
      else
        {
          /* printf( "event not logged\n" ); */
          write = FALSE;
        }

      if (write)
        {
          FILE *fp = fopen (lf->name, "a");
          if (fp)
            {
              /* printf( "  event logged\n" ); */
              _dm_write_logline_start (fp, username, NULL, type);

              /* display event details */
              if (fmt)
                {
                  vfprintf (fp, fmt, a);
                }
              fprintf (fp, "\n");

              /* done */
              fclose (fp);
            }
        }
      l = g_list_next (l);
    }
  /* printf( "\n" ); */
}
