
/*
   geas-skeleton.h
   
   Part of GNU Enterprise Application Server (GEAS)
 
   Copyright (C) 2000-2001 Free Software Foundation
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 
   $Id: geas-skeleton.h,v 1.28 2001/06/10 12:56:12 ntiffin Exp $
*/

#ifndef SKELETON_H
#define SKELETON_H

#include <glib.h>
#include <stdio.h>
#include "geas.h"

/** \brief Prepare servants for handling objects. */
int initialise_object_servants (PortableServer_POA root_poa);

#define OBJECTTYPE_CONNECTION  "01"
#define OBJECTTYPE_DATA        "02"
#define OBJECTTYPE_LIST        "03"
#define OBJECTTYPE_ADMIN       "04"
#define OBJECTTYPE_TRANSACTION "05"
#define OBJECTTYPE_QUERY       "06"
#define OBJECTTYPE_CONSTRAINT  "07"

enum object_ref_type
{
  RT_connection, RT_data, RT_list, RT_admin, RT_transaction,
  RT_constraint, RT_query
};

/* returns NULL on error, otherwise a pointer to a string. g_free() the result once done */
char *reference_to_string (PortableServer_ObjectId * oid);

/**
 * \brief Information about an arbitrary GEAS class.
 * ie, ObjectList, DataObject and subclasses (business objects), etc
 */
typedef struct GEAS_object_reference
{
  enum object_ref_type type;
  char *listid;
  char *username;
  char *classname;
  char *objectid;
  char *uniqueid;
  char *sessionid;
  char *parentid;

  /* read from database */
  char *currenttransaction;
  gboolean transactionopen;

  /* created in geas-skeleton.c for use in functions called,
   * in order to call methods on the server */
  GEAS_Connection server;
  struct GEAS_object_reference *serverid;
}
GEAS_object_reference;

/* Business Object Keys */
typedef unsigned char *ObjectKey;
unsigned long int object_key_size (ObjectKey key);
ObjectKey create_new_object_key (void);
void free_object_key (ObjectKey key);
const char *object_key_as_string (ObjectKey key);
void print_object_key (FILE * fp, ObjectKey key);

ObjectKey object_key_from_string (const char *str);

void cleanup_oid (char *oid);

/* Business Object Identifiers */
typedef struct ObjectIdentifier
{
  char *classname;
  char *key;
}
ObjectIdentifier;
ObjectIdentifier *create_object_identifier (const char *classname,
                                            const char *key);
void free_object_identifier (ObjectIdentifier * id);

#ifdef DEBUG
void display_geas_object_reference (GEAS_object_reference * oid);
#else
#define display_geas_object_reference(oid)      /* empty */
#endif

/** \briefConvert a CORBA object ID to a GEAS internal object ID (ASCII) */
GEAS_object_reference *corba_id_to_geas_reference (PortableServer_ObjectId *
                                                   oid,
                                                   CORBA_Environment * ev);

/** \brief Convert a CORBA object reference to a GEAS internal object ID (ASCII) */
GEAS_object_reference *corba_object_to_geas_reference (PortableServer_POA poa,
                                                       CORBA_Object obj,
                                                       CORBA_Environment *
                                                       ev);

/** \brief TDB: handle 'Current' interface */
GEAS_object_reference *current_object_to_geas_reference (CORBA_Environment *
                                                         ev);

/** \brief Make a CORBA object reference to a GEAS object */
CORBA_Object
make_connection_reference (const char *username, const char *session,
                           CORBA_Environment * ev);

CORBA_Object
make_transaction_reference (const char *username, const char *objectid,
                            const char *sessionid, CORBA_Environment * ev);

CORBA_Object
make_admin_reference (const char *username, const char *sessionid,
                      CORBA_Environment * ev);

CORBA_Object
make_list_reference (const char *listid, const char *username,
                     const char *sessionid, CORBA_Environment * ev);

CORBA_Object
make_dataobject_reference (const char *objectclass, const char *objectid,
                           const char *username, const char *sessionid,
                           CORBA_Environment * ev);

CORBA_Object
make_query_reference (const char *uniqueid, const char *username,
                      const char *sessionid, CORBA_Environment * ev);

CORBA_Object
make_constraint_reference (const char *uniqueid, const char *parentid,
                           const char *username, const char *sessionid,
                           CORBA_Environment * ev);

/* free memory */
void free_geas_object_reference (GEAS_object_reference * oid);

void refill_oid_buffer (int max);

GEAS_ConnectionFactory
impl_GEAS_ConnectionFactory__create (PortableServer_POA poa,
                                     CORBA_Environment * ev);
#endif /* SKELETON_H */
