/*
  methods_glibmodule.h - C Method support.

  Copyright (C) 2001 Free Software Foundation

  This file is part of the GNU Enterprise Application Server (GEAS)

  GEAS is free software; you can redistribute it and/or modify it under
  the terms of the GNU General Public License as published by the Free
  Software Foundation; either version 2, or (at your option) any later
  version.

  GEAS is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
  for more details.

  You should have received a copy of the GNU General Public License
  along with GEAS; if not, write to the Free Software Foundation, Inc.,
  59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

  $Id: methods_glibmodule.h,v 1.12 2001/07/25 17:48:24 reinhard Exp $
*/

#ifndef USE_GLIBMODULE_METHODS
#error "USE_GLIBMODULE_METHODS was not defined."
#endif

typedef char *(*METHODFUNC) (GEAS_DataObject obj, GEAS_Arguments * args,
                             GEAS_Connection server, CORBA_Environment * ev);

#define DEFAULT_IDL_FILE   "geas.idl"
#define DEFAULT_METHOD_DIR "../examples/businessobjects/methods"

#define ERR_OK              (0)
#define ERR_NO_GMODULE      (-1)
#define ERR_NO_FUNCTIONS    (-2)
#define ERR_MEMORY          (-3)

#include <glib.h>
#include <gmodule.h>
#include <dirent.h>

static GList *provider_list = NULL;

gboolean
initialise_method_handling (configuration config)
{
  g_return_val_if_fail (config, FALSE);

  if (g_module_supported ())
    return (TRUE);
  return (FALSE);
}

gboolean
load_method_handlers (configuration config)
{
  char *dirname;
  struct dirent;
  GList *l, *l3;
  odl_class *c;
  odl_namelist *classlist;
  char abspath[PATH_MAX];
  odl_fieldlist *fields;

  g_return_val_if_fail (config, FALSE);

  /* get directory where code files are stored */
  dirname =
    (char *) get_global_option_str (config, "methoddir", DEFAULT_METHOD_DIR);
  if (dirname == NULL)
    {
      return (FALSE);
    }
  dirname = g_strdup (dirname);
  while (dirname[strlen (dirname) - 1] == '\\' ||
         dirname[strlen (dirname) - 1] == '/')
    dirname[strlen (dirname) - 1] = '\0';
  /* message( "method dir name: '%s'" , dirname ); */

  /* find modulename_classname.so for methods */
  classlist = odl_tree_list_classes (all_classes);
  l = classlist;
  while (l)
    {
      c = odl_find_class (all_classes, l->data, NULL);
      /* debug_output( DEBUGLEVEL_2 , "    class %s" ,
         odl_class_get_name(c) ); */

      fields = odl_class_get_fields (c, FT_method);
      if (fields)
        {
          char *path;
          char *mangled = odl_mangle_qualified_name (l->data);
          GModule *module = NULL;
          GModule *storemodule = NULL;

          /* this class has some methods, so attempt to load its provider module */
          if (realpath (dirname, abspath) == NULL)
            {
              /* directory not found */
              return (FALSE);
            }
          strcat (abspath, "/.libs");
          path = g_module_build_path (abspath, mangled);
          /* message( "looking for module '%s'" , path ); */
          module = g_module_open (path, 0);
          storemodule = module;
          if (!module)
            {
              errormsg ("Required module '%s' was missing : %s", path,
                        g_module_error ());
              return (FALSE);
            }
          else
            {
              /* validate all the methods */
              /* message( "module loaded, now validating method functions" ); */
              l3 = fields;
              while (l3)
                {
                  METHODFUNC *funcptr;
                  odl_field *f = (odl_field *) l3->data;
                  char *funcname =
                    g_strdup_printf ("%s_%s", mangled,
                                     odl_field_get_name (f));
                  /* printf ("looking for %s.%s (%s)",
                          odl_class_get_full_name (c), odl_field_get_name (f),
                          funcname); */

                  g_module_symbol (module, funcname, (gpointer) & funcptr);
                  /* if (funcptr)
                    printf ("...found\n");
                  else
                    printf ("...not found\n"); */
                  g_free (funcname);

                  /* store the pointer to this function */
                  if (funcptr)
                    {
                      provider_t *p =
                        alloc_provider_data (mangled, NULL,
                                             odl_field_get_name (f),
                                             (void *) storemodule);
                      if (p)
                        {
                          provider_list = g_list_append (provider_list, p);
                          storemodule = NULL;
                          p->function = (void *) funcptr;
                        }
                    }
                  else
                    {
                      /* error - not found */
                      errormsg ("Could not find %s.%s : %s", l->data,
                                odl_field_get_name (f), g_module_error ());
                      return (FALSE);
                    }

                  l3 = g_list_next (l3);
                }
            }
          odl_fieldlist_free (fields);
          g_free (mangled);
        }

      l = g_list_next (l);
    }
  if (l != NULL)
    {
      odl_namelist_free (l);
    }
  return (TRUE);
}

gboolean
shutdown_method_handling (void)
{
  return (TRUE);
}

CORBA_char *
execute_method (GEAS_object_reference *obj,
                const char *methodname, GEAS_Arguments *args,
                CORBA_Environment *ev)
{
  char *classname = obj->classname;
  GList *l = provider_list;
  char *mangled = odl_mangle_qualified_name (obj->classname);
  provider_t *p;
  METHODFUNC func;

  g_return_val_if_fail (obj, NULL);
  g_return_val_if_fail (methodname, NULL);
  g_return_val_if_fail (args, NULL);
  g_return_val_if_fail (ev, NULL);

  /* message( "looking for method %s in class %s" , methodname , classname ); */
  while (l)
    {
      p = (provider_t *) l->data;
      if (g_strcasecmp (mangled, p->classname) == 0)
        {
          if (g_strcasecmp (methodname, p->methodname) == 0)
            {
              char *retval = NULL;
              CORBA_Environment ev2;
              GEAS_DataObject objref;
              GEAS_Connection serverref;

              /* message( "found it" ); */
              CORBA_exception_init (&ev2);
              objref =
                make_dataobject_reference (obj->classname, obj->objectid,
                                           obj->username, obj->sessionid,
                                           &ev2);
              serverref =
                make_connection_reference (obj->username, "server allocated",
                                           &ev2);

              func = p->function;
              retval = func (objref, args, serverref, ev);
              if (ev->_major != CORBA_NO_EXCEPTION)
                {
                  char *buf;
                  buf =
                    g_strdup_printf ("CORBA error: '%s'",
                                     CORBA_exception_id (ev));
                  CORBA_exception_free (ev);
                  CORBA_exception_init (ev);
                  make_MethodError_exception (ev, buf);
                }

              CORBA_Object_release (objref, &ev2);
              CORBA_Object_release (serverref, &ev2);

              g_free (mangled);
              CORBA_exception_free (&ev2);
              if (retval == NULL && ev->_major == CORBA_NO_EXCEPTION)
                {
                  return (CORBA_string_dup (""));
                }
              return (retval);
            }
        }
      l = g_list_next (l);
    }

  g_free (mangled);
  make_ServerError_exception (ev, "Could not find implementation of %s.%s()",
                              classname, methodname);
  return (NULL);
}
