{-
HOpenGL - a binding of OpenGL and GLUT for Haskell.
Copyright (C) 2000  Sven Panne <Sven.Panne@BetaResearch.de>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library (COPYING.LIB); if not, write to the Free
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

This module corresponds to chapter 4 (Window Management) of the GLUT3 API docs.
-}

module GLUT_Window (
   Window(..), -- constructor is used only internally, otherwise ADT
   createSubWindow, destroyWindow, postRedisplay,
   postWindowRedisplay,                                     -- @glut_geq_4_11@
   swapBuffers, getWindow, setWindow, setWindowTitle,
   setIconTitle, positionWindow, reshapeWindow, popWindow,
   pushWindow, iconifyWindow, showWindow, hideWindow,
   fullScreen, Cursor(..), setCursor,                       -- @glut_api_geq3@
   unmarshalCursor,                                         -- @glut_api_geq3@, internal use only
   warpPointer                                              -- @glut_geq_4_9@
) where

import Monad            (liftM)
import CForeign         (CInt, CString, withCString)

import GL_BasicTypes    (WindowPosition(..), WindowSize(..))
import GLUT_Constants   (glut_CURSOR_RIGHT_ARROW, glut_CURSOR_LEFT_ARROW,
                         glut_CURSOR_INFO, glut_CURSOR_DESTROY, glut_CURSOR_HELP,
                         glut_CURSOR_CYCLE, glut_CURSOR_SPRAY, glut_CURSOR_WAIT,
                         glut_CURSOR_TEXT, glut_CURSOR_CROSSHAIR,
                         glut_CURSOR_UP_DOWN, glut_CURSOR_LEFT_RIGHT,
                         glut_CURSOR_TOP_SIDE, glut_CURSOR_BOTTOM_SIDE,
                         glut_CURSOR_LEFT_SIDE, glut_CURSOR_RIGHT_SIDE,
                         glut_CURSOR_TOP_LEFT_CORNER,
                         glut_CURSOR_TOP_RIGHT_CORNER,
                         glut_CURSOR_BOTTOM_RIGHT_CORNER,
                         glut_CURSOR_BOTTOM_LEFT_CORNER, glut_CURSOR_INHERIT,
                         glut_CURSOR_NONE, glut_CURSOR_FULL_CROSSHAIR)

---------------------------------------------------------------------------

newtype Window = Window CInt deriving (Eq,Ord)

createSubWindow :: Window -> WindowPosition -> WindowSize -> IO Window
createSubWindow (Window win) (WindowPosition x y) (WindowSize w h) =
   liftM Window (glutCreateSubWindow win (fromIntegral x) (fromIntegral y) (fromIntegral w) (fromIntegral h))

foreign import "glutCreateSubWindow" unsafe glutCreateSubWindow :: CInt -> CInt -> CInt -> CInt -> CInt -> IO CInt

setWindow :: Window -> IO ()
setWindow (Window win) = glutSetWindow win

foreign import "glutSetWindow" unsafe glutSetWindow :: CInt -> IO ()

getWindow :: IO Window
getWindow = liftM Window glutGetWindow

foreign import "glutGetWindow" unsafe glutGetWindow :: IO CInt

destroyWindow :: Window -> IO ()
destroyWindow (Window win) = glutDestroyWindow win

foreign import "glutDestroyWindow" unsafe glutDestroyWindow :: CInt -> IO ()

foreign import "glutPostRedisplay" unsafe postRedisplay :: IO ()

postWindowRedisplay :: Window -> IO ()   -- @glut_geq_4_11@
postWindowRedisplay (Window win) = glutPostWindowRedisplay win

foreign import "glutPostWindowRedisplay" unsafe glutPostWindowRedisplay :: CInt -> IO ()   -- @glut_geq_4_11@

foreign import "glutSwapBuffers" unsafe swapBuffers :: IO ()

positionWindow :: WindowPosition -> IO ()
positionWindow (WindowPosition x y) = glutPositionWindow (fromIntegral x) (fromIntegral y)

foreign import "glutPositionWindow" unsafe glutPositionWindow :: CInt -> CInt -> IO ()

reshapeWindow :: WindowSize -> IO ()
reshapeWindow (WindowSize w h) = glutReshapeWindow (fromIntegral w) (fromIntegral h)

foreign import "glutReshapeWindow" unsafe glutReshapeWindow :: CInt -> CInt -> IO ()

foreign import "glutFullScreen" unsafe fullScreen :: IO () -- @glut_api_geq3@
foreign import "glutPopWindow" unsafe popWindow :: IO ()
foreign import "glutPushWindow" unsafe pushWindow :: IO ()
foreign import "glutShowWindow" unsafe showWindow :: IO ()
foreign import "glutHideWindow" unsafe hideWindow :: IO ()
foreign import "glutIconifyWindow" unsafe iconifyWindow :: IO ()

setWindowTitle :: String -> IO ()
setWindowTitle title = withCString title glutSetWindowTitle

foreign import "glutSetWindowTitle" unsafe glutSetWindowTitle :: CString -> IO ()

setIconTitle :: String -> IO ()
setIconTitle title = withCString title glutSetIconTitle

foreign import "glutSetIconTitle" unsafe glutSetIconTitle :: CString -> IO ()

---------------------------------------------------------------------------
-- Cursors

-- @glut_api_geq3@
data Cursor =
     CursorRightArrow
   | CursorLeftArrow
   | CursorInfo
   | CursorDestroy
   | CursorHelp
   | CursorCycle
   | CursorSpray
   | CursorWait
   | CursorText
   | CursorCrosshair
   | CursorUpDown
   | CursorLeftRight
   | CursorTopSide
   | CursorBottomSide
   | CursorLeftSide
   | CursorRightSide
   | CursorTopLeftCorner
   | CursorTopRightCorner
   | CursorBottomRightCorner
   | CursorBottomLeftCorner
   | CursorInherit
   | CursorNone
   | CursorFullCrosshair
   deriving (Eq,Ord)

marshalCursor :: Cursor -> CInt
marshalCursor CursorRightArrow        = glut_CURSOR_RIGHT_ARROW
marshalCursor CursorLeftArrow         = glut_CURSOR_LEFT_ARROW
marshalCursor CursorInfo              = glut_CURSOR_INFO
marshalCursor CursorDestroy           = glut_CURSOR_DESTROY
marshalCursor CursorHelp              = glut_CURSOR_HELP
marshalCursor CursorCycle             = glut_CURSOR_CYCLE
marshalCursor CursorSpray             = glut_CURSOR_SPRAY
marshalCursor CursorWait              = glut_CURSOR_WAIT
marshalCursor CursorText              = glut_CURSOR_TEXT
marshalCursor CursorCrosshair         = glut_CURSOR_CROSSHAIR
marshalCursor CursorUpDown            = glut_CURSOR_UP_DOWN
marshalCursor CursorLeftRight         = glut_CURSOR_LEFT_RIGHT
marshalCursor CursorTopSide           = glut_CURSOR_TOP_SIDE
marshalCursor CursorBottomSide        = glut_CURSOR_BOTTOM_SIDE
marshalCursor CursorLeftSide          = glut_CURSOR_LEFT_SIDE
marshalCursor CursorRightSide         = glut_CURSOR_RIGHT_SIDE
marshalCursor CursorTopLeftCorner     = glut_CURSOR_TOP_LEFT_CORNER
marshalCursor CursorTopRightCorner    = glut_CURSOR_TOP_RIGHT_CORNER
marshalCursor CursorBottomRightCorner = glut_CURSOR_BOTTOM_RIGHT_CORNER
marshalCursor CursorBottomLeftCorner  = glut_CURSOR_BOTTOM_LEFT_CORNER
marshalCursor CursorInherit           = glut_CURSOR_INHERIT
marshalCursor CursorNone              = glut_CURSOR_NONE
marshalCursor CursorFullCrosshair     = glut_CURSOR_FULL_CROSSHAIR

unmarshalCursor :: CInt -> Cursor
unmarshalCursor cursor
   | cursor == glut_CURSOR_RIGHT_ARROW         = CursorRightArrow
   | cursor == glut_CURSOR_LEFT_ARROW          = CursorLeftArrow
   | cursor == glut_CURSOR_INFO                = CursorInfo
   | cursor == glut_CURSOR_DESTROY             = CursorDestroy
   | cursor == glut_CURSOR_HELP                = CursorHelp
   | cursor == glut_CURSOR_CYCLE               = CursorCycle
   | cursor == glut_CURSOR_SPRAY               = CursorSpray
   | cursor == glut_CURSOR_WAIT                = CursorWait
   | cursor == glut_CURSOR_TEXT                = CursorText
   | cursor == glut_CURSOR_CROSSHAIR           = CursorCrosshair
   | cursor == glut_CURSOR_UP_DOWN             = CursorUpDown
   | cursor == glut_CURSOR_LEFT_RIGHT          = CursorLeftRight
   | cursor == glut_CURSOR_TOP_SIDE            = CursorTopSide
   | cursor == glut_CURSOR_BOTTOM_SIDE         = CursorBottomSide
   | cursor == glut_CURSOR_LEFT_SIDE           = CursorLeftSide
   | cursor == glut_CURSOR_RIGHT_SIDE          = CursorRightSide
   | cursor == glut_CURSOR_TOP_LEFT_CORNER     = CursorTopLeftCorner
   | cursor == glut_CURSOR_TOP_RIGHT_CORNER    = CursorTopRightCorner
   | cursor == glut_CURSOR_BOTTOM_RIGHT_CORNER = CursorBottomRightCorner
   | cursor == glut_CURSOR_BOTTOM_LEFT_CORNER  = CursorBottomLeftCorner
   | cursor == glut_CURSOR_INHERIT             = CursorInherit
   | cursor == glut_CURSOR_NONE                = CursorNone
   | cursor == glut_CURSOR_FULL_CROSSHAIR      = CursorFullCrosshair
   | otherwise                                 = error "unmarshalCursor"

setCursor :: Cursor -> IO ()   -- @glut_api_geq3@
setCursor = glutSetCursor . marshalCursor

foreign import "glutSetCursor" unsafe glutSetCursor :: CInt -> IO ()   -- @glut_api_geq3@

warpPointer :: WindowPosition -> IO ()   -- @glut_geq_4_9@
warpPointer (WindowPosition x y) = glutWarpPointer (fromIntegral x) (fromIntegral y)

foreign import "glutWarpPointer" unsafe glutWarpPointer :: CInt -> CInt -> IO ()   -- @glut_geq_4_9@
