#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h> /* for basename() */
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <getopt.h>
#include <gtk/gtk.h>

#include "../config.h"
#include "defines.h"

#include "gksu.h"

#include "gui.h"
#include "xauth.h"
#include "su.h"
#include "sudo.h"

struct option long_opts[] = {
    /*
     * { name  has_arg  *flag  val } 
     */
    {"help", no_argument, NULL, 'h'},
    {"login", no_argument, NULL, 'l'},
    {"reset-env", no_argument, NULL, 'r'},
    {"user", required_argument, NULL, 'u'},
    {"use-sudo", no_argument, NULL, 'S'},
    {"print-pass", required_argument, NULL, 'p'},
    {"message", required_argument, NULL, 'm'},
    {0, 0, 0, 0}
};

void
help (gchar *cmdname)
{
  printf (_("GKsu version %s\n\n"
	    "Usage: %s [-u <user>] [-k] [-l] <command>\n\n"
	    "Common Options:\n"
	    "\t--user <user>, -u <user>\n"
	    "\t\tCalls <command> as the specified user\n"
	    "\t--message <message>, -m <message>\n"
	    "\t\tReplaces the standard message shown to ask for\n"
	    "\t\tpassword for the argument passed to the option\n"
	    "\t--print-pass <message>, -p <message>\n"
	    "\t\tAsks gksu to print the password to stdout, just\n"
	    "\t\tlike ssh-askpass. Useful to use in scripts with\n"
	    "\t\tprograms that accept receiving the passowrd on\n"
	    "\t\tstdin.\n"
	    "\t\tThis option requires the message that will be shown\n"
	    "\t\tto the user as argument and excludes all others\n"
	    "\n" 
	    "GKsu Options:\n"
	    "\t--login, -l\n"
	    "\t\tMakes this a login shell\n"
	    "\t--reset-env, -r\n"
	    "\t\tSet environment to the target user's defaults\n"
	    "\t\tinstead of using the same values as the current\n"
	    "\t\tThis means that the Xauthority magic will not work\n"
	    "\t\tso you have to authorize your user in another way\n"
	    "\t\tto access your display.\n"
	    "\n"
	    "GKsudo Options:\n"
	    "\t--use-sudo, -S\n"
	    "\t\tUse sudo instead of su (or use gksudo)\n"
	    "\t--set-home, -H\n"
	    "\t\tSets $HOME for the target users's home. The\n"
	    "\t\tdefault behavior is to keep your user's $HOME\n"
	    "\n"),
	    VERSION, cmdname);
}

/*
  clears variables used to store password 
*/
void
free_pass (gchar *pass)
{
  if (pass != NULL)
    {
      memset (pass, 0, strlen(pass));
      g_free (pass);
      pass = NULL;
    }
}

int 
main (int argc, char **argv)
{
  gchar *myname;

  int c;
  gchar template[] = "/tmp/" PACKAGE "-XXXXXX";
  gchar *dir= NULL;
  gchar *xauth = NULL;
  gchar *xauth_env = NULL;
  gchar *user = NULL;
  gchar *command = NULL; /* gksu */

  /* gksudo: the biggest pointer we expect to have */
  gchar **cmd = g_malloc (sizeof(gchar**)*(argc+8));
  gint cmd_starts = 0;

  gchar *message = NULL;

  gboolean use_sudo = FALSE;
  gboolean print_pass = FALSE;

  gboolean login_shell = FALSE;
  gboolean keep_env = TRUE;

  setlocale (LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);  
  textdomain(PACKAGE);
  gtk_set_locale();

  gtk_init (&argc, &argv);

  myname = basename (argv[0]);

  if (!strcmp(myname, "gksudo"))
    use_sudo = TRUE;

  while ((c = getopt_long(argc, argv, "?hu:lSHp:m:r", long_opts, NULL))
	 != EOF)
    {
      switch (c)
	{
	case 'h':
	  help (argv[0]);
	  exit(0);
	  break;
	case '?':
	  help (argv[0]);
	  exit(0);
	  break;
	case 'u':
	  user = g_strdup (optarg);
	  break;
	case 'l':
	  login_shell = TRUE;
	  break;
	case 'p':
	  print_pass = TRUE;
	  message = g_strdup (optarg);
	  break;
	case 'm':
	  message = g_strdup (optarg);
	  break;
	case 'S':
	  use_sudo = TRUE;
	  break;
	case 'r':
	case 'H':
	  keep_env = FALSE;
	  break;
	default:
	  fprintf (stderr, _("Using %s as username...\n"), argv[optind]);
	}
    }

  /* 
     if --print-pass is passed we do not need to care
     about xauth stuff and etc... let's just ask the
     password and print it
  */
  if (print_pass)
    {
      gchar *password;

      password = ask_password (message);
      printf ("%s\n", password);
      free_pass (password);

      exit (0);
    }

  /* now we can beggin to care about a command, user and
     stuff */
  if (argc == optind)
    {
      gk_dialog (_("Missing command to run."));
      exit (1);
    }

  /* if user is not specified, use root */
  if (user == NULL)
      user = g_strdup ("root");

  /* 
     sudo likes to receive program's args as
     separated args 
  */
  if (use_sudo)
    cmd_starts = create_cmd (user, cmd, argc, optind, argv, keep_env);
  else
    command = g_strdup (argv[optind]);

  /* xauth stuff, borrowed from gnome-sudo */
  dir = mkdtemp (template);
  if (!dir)
    {
      gk_dialog(strerror(errno));
      exit (1);
    }

  /* directory must not be 700 */
  chmod (dir, S_IRUSR|S_IWUSR|S_IXUSR|S_IRGRP|S_IROTH|S_IXOTH);
  
  xauth = g_strdup_printf ("%s/.Xauthority", g_get_home_dir());
  copy (xauth, dir);
  g_free (xauth);

  xauth = g_strdup_printf ("%s/.Xauthority", dir);
  xauth_env = getenv ("XAUTHORITY");
  setenv ("XAUTHORITY", xauth, TRUE);

  /* the magic begins */
  if (use_sudo)
    sudo_do (dir, message, user, cmd, cmd_starts); /* sudo it, baby */
  else
    su_do (dir, message, user, command, login_shell, keep_env);
  
  /* reset the env var as it was before or clean it  */
  if (xauth_env)
    setenv ("XAUTHORITY", xauth_env, TRUE);
  else
    unsetenv ("XAUTHORITY");

  g_free (xauth);

  if (use_sudo)
    clean_cmd (cmd);
  else
    g_free (command);

  g_free (user);

 return 0;
}
