/*  Glimmer - color-settings.c
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "declarations.h"
#include "color-settings.h"
#include "getprefs.h"
#include "main.h"
#include "settings.h"

static void save_settings(GtkWidget *widget, gpointer data);
static void apply_settings(GtkWidget *widget, gpointer data);
static void get_defaults(GtkWidget *widget, gpointer data);
static void get_backups(GtkWidget *widget, gpointer data);
static void put_cache(struct highlight_cache *cache);
static void get_cache(struct highlight_cache *cache);
static void highlight_tables_property_change(void);

struct highlight_cache DefaultCache;
static struct highlight_cache BackupCache;

static GtkWidget *settings_window;

static GtkWidget *color_widgets[MAX_COLORS];
static GtkWidget *italic_widgets[MAX_COLORS];
static GtkWidget *bold_widgets[MAX_COLORS];
static GtkWidget *under_widgets[MAX_COLORS];
static GtkWidget *strike_widgets[MAX_COLORS];

static char *color_names[] = {
   "String:",
   "Character:",
   "Character String:",
   "Keyword:",
   "Comment:",
   "Preprocessor:",
   "Data Type:",
   "Function:",
   "Macro:",
   "Variable:",
   "Symbol:",
   "Operator:",
   "Punctuation:",
   "Decimal:",
   "Hexidecimal:",
   "Octal:"
};

void change_colors_cb(GtkWidget *widget, gpointer data)
{
   GtkWidget *main_box;
   GtkWidget *main_frame;
   GtkWidget *lhbox;
   GtkWidget *rhbox;
   GtkWidget *label;
   GtkWidget *box;
   GtkWidget *util_box;
   GtkWidget *buttons_box;
   GtkWidget *ok_button;
   GtkWidget *apply_button;
   GtkWidget *defaults_button;
   GtkWidget *undo_button;
   GtkWidget *cancel_button;
   GtkWidget *sep;
   GtkWidget *pixmap;
   gint i = 0;

   if(settings_window && GTK_IS_WINDOW(settings_window))
   {
      gdk_window_raise(settings_window->window);
      gtk_widget_grab_focus(settings_window);
      return;
   }

   BackupCache = HighlightCache;
   settings_window = gtk_window_new(GTK_WINDOW_DIALOG);
   gtk_container_set_border_width(GTK_CONTAINER(settings_window), 5);

   main_box = gtk_vbox_new(FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(main_box), 0);
   gtk_container_add(GTK_CONTAINER(settings_window), main_box);
   gtk_widget_show(main_box);

   main_frame = gtk_frame_new(_("Highlighting Colors & Fonts"));
   gtk_box_pack_start(GTK_BOX(main_box), main_frame, TRUE, TRUE, 0);
   gtk_widget_show(main_frame);

   util_box  = gtk_hbox_new(FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(util_box), 10);
   gtk_container_add(GTK_CONTAINER(main_frame), util_box);
   gtk_widget_show(util_box);

   lhbox = gtk_vbox_new(FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(lhbox), 0);
   gtk_box_pack_start(GTK_BOX(util_box), lhbox, TRUE, TRUE, 0);
   gtk_widget_show(lhbox);

   sep = gtk_vseparator_new();
   gtk_box_pack_start(GTK_BOX(util_box), sep, FALSE, TRUE, 10);
   gtk_widget_show(sep);

   rhbox = gtk_vbox_new(FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(rhbox), 0);
   gtk_box_pack_start(GTK_BOX(util_box), rhbox, TRUE, TRUE, 0);
   gtk_widget_show(rhbox);

   box = lhbox;
   for(i = 0; i < MAX_COLORS; i++)
   {
      util_box  = gtk_hbox_new(FALSE, 0);
      gtk_box_pack_start(GTK_BOX(box), util_box, FALSE, TRUE, 0);         
      gtk_widget_show(util_box);
      label = gtk_label_new(color_names[i]);
      gtk_box_pack_start(GTK_BOX(util_box), label, FALSE, FALSE, 0);         
      gtk_widget_show(label);
      pixmap = gnome_stock_pixmap_widget_at_size(settings_window, GNOME_STOCK_PIXMAP_TEXT_STRIKEOUT, 18, 18);
      strike_widgets[i] = gtk_toggle_button_new();
      gtk_container_add(GTK_CONTAINER(strike_widgets[i]), pixmap);
      gtk_box_pack_end(GTK_BOX(util_box), strike_widgets[i], FALSE, FALSE, 0);
      gtk_widget_show_all(strike_widgets[i]);
      pixmap = gnome_stock_pixmap_widget_at_size(settings_window, GNOME_STOCK_PIXMAP_TEXT_UNDERLINE, 18, 18);
      under_widgets[i] = gtk_toggle_button_new();
      gtk_container_add(GTK_CONTAINER(under_widgets[i]), pixmap);
      gtk_box_pack_end(GTK_BOX(util_box), under_widgets[i], FALSE, FALSE, 0);
      gtk_widget_show_all(under_widgets[i]);
      pixmap = gnome_stock_pixmap_widget_at_size(settings_window, GNOME_STOCK_PIXMAP_TEXT_BOLD, 18, 18);
      bold_widgets[i] = gtk_toggle_button_new();
      gtk_container_add(GTK_CONTAINER(bold_widgets[i]), pixmap);
      gtk_box_pack_end(GTK_BOX(util_box), bold_widgets[i], FALSE, FALSE, 0);
      gtk_widget_show_all(bold_widgets[i]);
      pixmap = gnome_stock_pixmap_widget_at_size(settings_window, GNOME_STOCK_PIXMAP_TEXT_ITALIC, 18, 18);
      italic_widgets[i] = gtk_toggle_button_new();
      gtk_container_add(GTK_CONTAINER(italic_widgets[i]), pixmap);
      gtk_box_pack_end(GTK_BOX(util_box), italic_widgets[i], FALSE, FALSE, 0);
      gtk_widget_show_all(italic_widgets[i]);
      color_widgets[i] = gnome_color_picker_new();
      gnome_color_picker_set_dither(GNOME_COLOR_PICKER(color_widgets[i]), FALSE);
      gnome_color_picker_set_use_alpha(GNOME_COLOR_PICKER(color_widgets[i]), FALSE);
      gtk_box_pack_end(GTK_BOX(util_box), color_widgets[i], FALSE, FALSE, 0);
      gtk_widget_show(color_widgets[i]);
      if(i == 7) box = rhbox;
   }

   sep = gtk_hseparator_new();
   gtk_box_pack_start(GTK_BOX(main_box), sep, FALSE, TRUE, 10);
   gtk_widget_show(sep);

   buttons_box = gtk_hbutton_box_new();
   gtk_box_pack_start(GTK_BOX(main_box), buttons_box, FALSE, FALSE, 0);
   gtk_button_box_set_layout(GTK_BUTTON_BOX(buttons_box), gnome_preferences_get_button_layout());
   gtk_button_box_set_spacing(GTK_BUTTON_BOX(buttons_box), GNOME_PAD);
   gtk_widget_show(buttons_box);

   ok_button = gnome_stock_button(GNOME_STOCK_BUTTON_OK);
   gtk_box_pack_start(GTK_BOX(buttons_box), ok_button, FALSE, TRUE, 0);
   gtk_signal_connect(GTK_OBJECT(ok_button), "clicked", GTK_SIGNAL_FUNC(save_settings), 0);
   gtk_signal_connect(GTK_OBJECT(ok_button), "clicked", GTK_SIGNAL_FUNC(kill_widget), GTK_OBJECT(settings_window));
   GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
   gtk_widget_show(ok_button);

   apply_button = gnome_stock_button(GNOME_STOCK_BUTTON_APPLY);
   gtk_box_pack_start(GTK_BOX(buttons_box), apply_button, FALSE, TRUE, 0);
   gtk_signal_connect(GTK_OBJECT(apply_button), "clicked", GTK_SIGNAL_FUNC(apply_settings), 0);
   GTK_WIDGET_SET_FLAGS(apply_button, GTK_CAN_DEFAULT);
   gtk_widget_show(apply_button);

   defaults_button = gnome_stock_or_ordinary_button(_("Defaults"));
   gtk_box_pack_start(GTK_BOX(buttons_box), defaults_button, FALSE, TRUE, 0);   
   gtk_signal_connect(GTK_OBJECT(defaults_button), "clicked", GTK_SIGNAL_FUNC(get_defaults), 0);
   GTK_WIDGET_SET_FLAGS(defaults_button, GTK_CAN_DEFAULT);
   gtk_widget_show(defaults_button);

   undo_button = gnome_stock_or_ordinary_button(_("Undo"));
   gtk_box_pack_start(GTK_BOX(buttons_box), undo_button, FALSE, TRUE, 0);   
   gtk_signal_connect(GTK_OBJECT(undo_button), "clicked", GTK_SIGNAL_FUNC(get_backups), 0);
   GTK_WIDGET_SET_FLAGS(undo_button, GTK_CAN_DEFAULT);
   gtk_widget_show(undo_button);

   cancel_button = gnome_stock_button(GNOME_STOCK_BUTTON_CLOSE);
   gtk_box_pack_start(GTK_BOX(buttons_box), cancel_button, FALSE, TRUE, 0);   
   gtk_signal_connect(GTK_OBJECT(cancel_button), "clicked", GTK_SIGNAL_FUNC(kill_widget), GTK_OBJECT(settings_window));
   GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);
   gtk_widget_grab_default(cancel_button);
   gtk_widget_show(cancel_button);

   gtk_signal_connect(GTK_OBJECT(settings_window), "destroy", GTK_SIGNAL_FUNC(kill_widget_ptr), &settings_window);
   put_cache((struct highlight_cache *) &HighlightCache);
   gtk_widget_show(settings_window);
}

static void save_settings(GtkWidget *widget, gpointer data)
{
   get_cache((struct highlight_cache *) &HighlightCache);
   highlight_tables_property_change();
   apply_color_changes();
   save_colors();
}

static void apply_settings(GtkWidget *widget, gpointer data)
{
   get_cache((struct highlight_cache *) &HighlightCache);
   highlight_tables_property_change();
   apply_color_changes();
}

static void get_defaults(GtkWidget *widget, gpointer data)
{
   put_cache((struct highlight_cache *) &DefaultCache);
}

static void get_backups(GtkWidget *widget, gpointer data)
{
   put_cache((struct highlight_cache *) &BackupCache);
}

static void put_cache(struct highlight_cache *cache)
{
   gint i = 0;
   for(i = 0; i < MAX_COLORS; i++)
   {
      gnome_color_picker_set_i16(GNOME_COLOR_PICKER(color_widgets[i]), cache->colors[i].red, cache->colors[i].green, cache->colors[i].blue, 0);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(italic_widgets[i]), (cache->types[i] == FONT_ITALIC || cache->types[i] == FONT_BOLD_ITALIC));
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(bold_widgets[i]), (cache->types[i] == FONT_BOLD || cache->types[i] == FONT_BOLD_ITALIC));
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(under_widgets[i]), (cache->flags[i] & GTK_EXTEXT_STYLE_UNDER));
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(strike_widgets[i]), (cache->flags[i] & GTK_EXTEXT_STYLE_STRIKE));
   }
}

static void get_cache(struct highlight_cache *cache)
{
   gboolean it;
   gboolean b;
   gboolean u;
   gboolean s;
   gint i = 0;

   for(i = 0; i < MAX_COLORS; i++)
   {
      cache->flags[i] = 0;
      cache->types[i] = FONT_NORMAL;
      gnome_color_picker_get_i16(GNOME_COLOR_PICKER(color_widgets[i]), &cache->colors[i].red, &cache->colors[i].green, &cache->colors[i].blue, 0);
      it = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(italic_widgets[i]));
      b = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(bold_widgets[i]));
      u = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(under_widgets[i]));
      s = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(strike_widgets[i]));
      if(it && b) cache->types[i] = FONT_BOLD_ITALIC;
      else if(it) cache->types[i] = FONT_ITALIC;
      else if(b) cache->types[i] = FONT_BOLD;
      if(u) cache->flags[i] |= GTK_EXTEXT_STYLE_UNDER;
      if(s) cache->flags[i] |= GTK_EXTEXT_STYLE_STRIKE;
      cache->fonts[i] = get_font(cache->types[i]);
   }
}

static void highlight_tables_property_change(void)
{
   GList *tables = NULL;
   GList *entry = NULL;
   GdsFileHighlightTable *table;
   GtkExTextSyntaxEntry *syntax;
   GtkExTextPatternEntry *pattern;
   GtkExTextEmbeddedEntry *embedded;
   gint i = 0;

   for(tables = g_list_first(highlight_tables); tables; tables = tables->next)
   {
      table = (GdsFileHighlightTable *)tables->data;
      if(table->syntax)
      for(entry = g_list_first(table->syntax->entries); entry; entry = entry->next)
      {
         syntax = (GtkExTextSyntaxEntry *)entry->data;
         for(i = 0; i < MAX_COLORS; i++)
         {
            if(syntax->color == &HighlightCache.colors[i])
            {
               syntax->font = HighlightCache.fonts[i];
               syntax->flags = HighlightCache.flags[i];
               syntax->data = GINT_TO_POINTER(HighlightCache.types[i]);
            }
         }
      }
      if(table->pattern)
      for(entry = g_list_first(table->pattern->entries); entry; entry = entry->next)
      {
         pattern = (GtkExTextPatternEntry *)entry->data;
         for(i = 0; i < MAX_COLORS; i++)
         {
            if(pattern->color == &HighlightCache.colors[i])
            {
               pattern->font = HighlightCache.fonts[i];
               pattern->flags = HighlightCache.flags[i];
               pattern->data = GINT_TO_POINTER(HighlightCache.types[i]);
            }
         }
      }
      if(table->embedded)
      for(entry = g_list_first(table->embedded->entries); entry; entry = entry->next)
      {
         embedded = (GtkExTextEmbeddedEntry *)entry->data;
         for(i = 0; i < MAX_COLORS; i++)
         {
            if(embedded->color == &HighlightCache.colors[i])
            {
               embedded->font = HighlightCache.fonts[i];
               embedded->flags = HighlightCache.flags[i];
               embedded->data = GINT_TO_POINTER(HighlightCache.types[i]);
            }
         }
      }
   }
}
