/* napster.c - all napster-related protocol stuff */

#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include "gnapster.h"
#include "upload.h"
#include "browse.h"
#include "queue.h"
#include "regexp.h"
#include "chan.h"
#include "xtext.h"

#include "commands.h"

extern GnapsterMain *gmain;
extern UserInfo user_info;
extern ListChanDlg list_chan_dlg;

extern GList *ignore_list;
extern GList *signore_list;

extern int nick_curr, nick_end;
extern GList *nick_history;

extern int browse_results;

int last_dns = 0;
int flag = 0;

char *conn_table[] = {
   "Unknown", "14.4", "28.8", "33.6", "56k", "64k ISDN", "128k ISDN",
   "Cable", "DSL", "T1", "T3+", NULL
};


int napster_send(int sock, int cmd, char *fmt, ...) {
   Packet *p;
   va_list args;
   char *data = NULL;
   int x;
   
   if (cmd < 0 || sock < 0) 
     return -1;
   
   if (fmt) {
      va_start(args, fmt);
      data = d_strdup_vprintf(fmt, args);
      va_end(args);
   }
   
   p = d_calloc(sizeof(Packet));

   p->len = data ? strlen(data) : 0;
   p->cmd = cmd;
   
   /* if our host order is BIG_ENDIAN, we need to reverse the
    * bytes */
   p->len = BSWAP16(p->len);
   p->cmd = BSWAP16(p->cmd);
   
   x = send(sock, p, 4, 0);
   if (data && x > 0)
     x += n_send(sock, "%s", data);
   
   d_free(data);
   d_free(p);
   
   return x;
}

void napster_send_get(int sock, char *user, char *file, unsigned long int size) {
   n_send(sock, "GET");
   n_send(sock, "%s \"%s\" %lu", user, file, size);
}

void napster_send_send(int sock, char *user, char *file, unsigned long int size) {
   n_send(sock, "SEND");
   n_send(sock, "%s \"%s\" %lu", user, file, size);
}

int opennap_version(STab *stab, int major, int minor) {
   int ma, mi;
   unsigned short on_version;
   
   on_version = ((stab->ci->state & OPENNAP_MASK) >> 16);

   if (!on_version)
     return 0;
   
   ma = on_version >> 8;
   mi = on_version & 0xff;
   
   if (ma > major || (ma == major && mi >= minor))
     return 1;
   
   return 0;
}

int is_opennap_server(STab *stab, char *data) {
   char *ver, *server, *server_ver;
   int ver_minor, ver_major;
   
   ver = next_arg(data, &data);
   server = next_arg(data, &data);
   server_ver = next_arg(data, &data);
   
   NA_ERR_HANDLE(0);

   if (strcmp(ver, "VERSION")) 
     return 0;
   
   if (strcmp(server, "opennap")) 
     return 0;
   
   ver_major = my_atoi(next_arg_full(server_ver, &server_ver, '.'));
   ver_minor = my_atoi(next_arg_full(server_ver, &server_ver, '.'));
   
   NA_RESET();
   
   stab->ci->state |= (((ver_major & 0xff) << 8 | (ver_minor)) << 16);
     
   if (ver_minor < 0) 
     return 0;
   
   if (ver_major > 0 || (ver_major == 0 && ver_minor >= 12))
     return 1;
   
   return 0;
}

int user_ignored(char *user) {
   GList *ptr;
   char *nick;
   
   if (!user) 
     return 0;
   
   for(ptr=ignore_list; ptr; ptr=ptr->next) {
      nick = ptr->data;
      if (!nick)
	continue;
      
      if (!strcmp(nick, user))
	return 1;
   }
   
   return 0;
}

void napster_show_links(STab *stab) {
   create_server_links(stab);
}

/*void napster_show_links(STab *stab, GtkCTreeNode *node, char *print_str) {
   char *buf;
   char print_str_local[80];
   
   if (print_str) 
     strcpy(print_str_local, print_str);
   else
     print_str_local[0] = '\0';
   
   if (!node)
     node = GTK_CTREE_NODE(GTK_CLIST(stab->ct->links_map)->row_list);
   
   while(node) {
      buf = GTK_CELL_PIXTEXT(GTK_CTREE_ROW(node)->row.cell[0])->text;
      
      gnapster_text_insert(stab, CURR, SYSTEM,
			   "%s", print_str_local);
      
      if (GTK_CTREE_ROW(node)->parent)
	gnapster_text_insert(stab, CURR, MESSAGE,
			     "%s", (GTK_CTREE_ROW(node)->sibling) ? "|-- " : "`-- ");
      
      gnapster_text_insert(stab, CURR, MESSAGE, "%s\n", buf);
      
      if (GTK_CTREE_ROW(node)->children) {
	 if (GTK_CTREE_ROW(node)->parent)
	   strcat(print_str_local, GTK_CTREE_ROW(node)->sibling ? "|   " : "    ");
	 else
	   strcat(print_str_local, "    ");
	 
	 napster_show_links(stab, GTK_CTREE_ROW(node)->children, print_str_local);
	 
	 print_str_local[strlen(print_str_local) - strlen("    ")] = 0;
      }
      
      node = GTK_CTREE_ROW(node)->sibling;
   }
}

GtkCTreeNode *napster_find_link(STab *stab, GtkCTreeNode *node, char *s) {
   GtkCTreeNode *work;
   char *buf;
   
   if (!s)
     return NULL;
   
   if (!node)
     node = GTK_CTREE_NODE(GTK_CLIST(stab->ct->links_map)->row_list);
   
   while(node) {
      buf = GTK_CELL_PIXTEXT(GTK_CTREE_ROW(node)->row.cell[0])->text;
      
      if (!strcmp(buf, s))
	return node;
      
      if (GTK_CTREE_ROW(node)->children &&
	  (work = napster_find_link(stab, GTK_CTREE_ROW(node)->children, s)))
	return work;
      
      node = GTK_CTREE_ROW(node)->sibling;
   }
   
   return NULL;
}*/

void napster_handle_data(STab *stab, int c, char *data) {
   ConnInfo *ci;
   char ts[1024];
   
/*   stab = spack->stab;*/
   ci = stab->ci;

   if (debug(ci))
     fprintf(stderr, "%-2i %-4i %s\n", ci->sock, c, data);

   /* YUCCCKKKKKKKKKKK 
    * This should *NOT* be a switch, the only reason it is is because the
    * code has grown to be so massive I fear that I would break something
    * to change it into a hash table! */
   switch(c) {
    case NAPSTER_MESSAGE:
      g_snprintf(ts, sizeof(ts), "%s\n", data);

/*      if (!GTK_XTEXT(stab->mt->text)->text_first) {
	 if (is_opennap_server(stab, data)) {
	    gtk_widget_show(gmain->st->not_label);
	    gtk_widget_show(gmain->st->not_entry);
	    break;
	 }
      }*/

      gnapster_text_insert(stab, MOTD, SYSTEM, ts);
      
      break;
    case NAPSTER_ADMIN_MESSAGE:
	{
	   char *user, *message;
	   int c;
	   
	   user = next_arg(data, &data);
	   message = last_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();

	   if (!strcmp(user, stab->ci->account->user))
	     break;
	   
	   if ((c = chan_lookup(stab, "#admins")) == -1)
	     c = CONSOLE;
	   
	   hook_text_insert(stab, c, MESSAGE, "wallop", "%s\4%s",
			    user, message);
	}
      break;
    case NAPSTER_GLOBAL_MESSAGE:
	{
	   char *user, *message;
	   
	   user = next_arg(data, &data);
	   message = last_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   hook_text_insert(stab, CONSOLE, MESSAGE, "global", "%s\4%s",
			    user, message);
	}
      break;
    case NAPSTER_LOGIN_ACK:
	{
	   notebook_set_sensitive(gmain->srv_notebook, stab->pn + 4, 1);
	   notebook_set_label(gmain->srv_notebook, stab->pn + 4, ci->server);
	   
	   remove_server_menu();
	   append_server_menu();
	   
	   if (!searching(ci) && !browsing(ci))
	     update_status(1, stab, NULL, 0);

	   if (!shared_list_handle(stab, user_info.upload_dir, SUBMIT_LIST))
	     autojoin_handle(stab);
	}
      break;
    case NAPSTER_STATUS_MESSAGE:
	{
	   int songs, users, gigs;
      
	   users = my_atoi(next_arg(data, &data));
	   songs = my_atoi(next_arg(data, &data));
	   gigs = my_atoi(next_arg(data, &data));
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   ci->users = users;
	   ci->songs = songs;
	   ci->gigs = gigs;

	   if (!searching(ci) && !browsing(ci))
	     update_status(1, stab, NULL, 0);
	}
      break;
/*    case NAPSTER_PTP_BROWSE:
      napster_send(ci->sock, NAPSTER_PTP_BROWSE_ACCEPT, data);
      break;
    case NAPSTER_PTP_BROWSE_ACCEPT:
	{
	   char *nick;
	   unsigned long ip;
	   unsigned short port;
	   
	   nick = next_arg(data, &data);
	   convert(next_arg(data, &data), "%lu", &ip);
	   convert(next_arg(data, &data), "%hu", &port);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   browse_connect(stab, nick, ip, port);
	}
      break;
    case NAPSTER_PTP_BROWSE_ERROR:
	{
	   char *nick;
	   
	   nick = next_arg(data, &data);
	   
	   end_browse(stab);
	}
      break;*/
    case CMDS_IGNOREENTRY:
	{
	   char *nick;
	   
	   nick = next_arg(data, &data);
	   
	   gnapster_text_insert(stab, CURR, MESSAGE, "      %s\n",
				nick);
	}
      break;
    case CMDS_REMOVEALLFILES:
      break;
    case NAPSTER_USER_MESSAGE:
	{
	   char *user, *message;
	   int index;
	   
	   user = next_arg(data, &data);
	   message = last_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   if (user_ignored(user)) 
	     break;
	   
	   if (!strncmp(message, "\1SEND ", 6)) {
	      handle_dcc_request(stab, user, message);
	      break;
	   }
	   
/*	   if (!strncmp(message, "\1SEND ", 6)) {
	      Transfer *dcc;
	      char *ip_str;
	      unsigned long int ip = 0;
	      
              dcc = d_new(N_DOWNLOAD);
	      
	      next_arg(message, &message);
	      
	      dcc->user = d_strdup(next_arg(message, &message));
	      ip_str = next_arg(message, &message);
	      dcc->port = my_atoi(next_arg(message, &message));
	      dcc->file = d_strdup(next_arg(message, &message));
	      dcc->connection = 0;
	      dcc->id = d_strdup(next_arg(message, &message));
	      dcc->kbps = my_atoi(next_arg(message, &message));
	      dcc->status = 1;
	      
	      if (dcc->kbps < 0) {
		 j_free(TRANSFER, dcc);
		 break;
	      }
	      
	      sscanf(ip_str, "%lu", &ip);
	      dcc->ip = BSWAP32(ip);
	      dcc->trunc_file = d_strdup(dcc->file);
	      
	      network_add_download(dcc);
	      napster_connect_user(dcc);
	      
	      break;
	   }*/
	   
	   if (user_info.conf[AUTOQUERY])
	     if (user_lookup(stab, user) == -1)
	       real_query(stab, 0, user);

	   index = user_lookup(stab, user);
	   if (index > 1)
	     hook_text_insert(stab, index, MESSAGE, "public_msg", "%s\4%s",
			      user, message);
	   else
	     hook_text_insert(stab, CURR, MESSAGE, "privmsg", "%s\4%s",
			      user, message);
	   
	   nick_history = add_nick_history(nick_history, user);
	}
      break;
    case NAPSTER_SERVER_STATS:
	{
	   unsigned long clients, files, t, uptime, users, numusers, memory;
	   unsigned short servers, gigs, channels;
	   
	   convert(next_arg(data, &data), "%lu", &clients);
	   convert(next_arg(data, &data), "%hu", &servers);
	   convert(next_arg(data, &data), "%lu", &users);
	   convert(next_arg(data, &data), "%lu", &files);
	   convert(next_arg(data, &data), "%hu", &gigs);
	   convert(next_arg(data, &data), "%hu", &channels);
	   convert(next_arg(data, &data), "%lu", &t);
	   convert(next_arg(data, &data), "%lu", &uptime);
	   convert(next_arg(data, &data), "%lu", &memory);
	   convert(next_arg(data, &data), "%lu", &numusers);
	   
	   NA_ERR_HANDLE_BREAK();

	   gnapster_text_insert(stab, CURR, SYSTEM, "clients = %lu, servers = %hu, users = %lu, files = %lu, gigs = %hu, channels = %hu, time = %lu, uptime = %lu, memory = %lu, numusers = %hu\n",
				clients, servers, users, files, gigs,
				channels, t, uptime, memory, numusers);
	}
      break;
    case NAPSTER_BAN_LIST_ENTRY:
    case NAPSTER_CBAN_LIST_ENTRY:
	{
	   char *ip, *nick, *reason;
	   time_t ban_time;
	   
	   if (!flag)
	     gnapster_text_insert(stab, CURR, SYSTEM, "%-20s %-15s %s\n",
				  "Ban", "Who", "Reason");
	   
	   flag = 1;

	   if (ci->ban_exp && !wild_match(ci->ban_exp, data))
	     break;
	   
	   ip = next_arg(data, &data);
	   nick = next_arg(data, &data);
	   reason = next_arg(data, &data);
	   convert(next_arg(data, &data), "%lu", &ban_time);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   gnapster_text_insert(stab, CURR, SYSTEM, "%-20s %-15s %s\n",
				ip, nick, reason);
	}
      break;
    case NAPSTER_BAN_LIST:
      if (!flag)
	hook_text_insert(stab, CURR, SYSTEM, "general_message", "%s",
			 "no global bans found");
      
      ci->ban_exp = d_free(ci->ban_exp);
      flag = 0;
      break;
/*    case NAPSTER_CBAN_LIST_ENTRY:
	{
	   char *ban, *who, *reason;
	   time_t ban_time;
	   
	   if (!flag)
	     gnapster_text_Insert(stab, CURR, SYSTEM, "\2%-20s %-15s %s\n",
				  "Ban", "Who", "Reason");
	   flag = 1;
	   
	   ban = next_arg(data, &data);
	   who = next_arg(data, &data);
	   reason = next_arg(data, &data);
	   convert(next_arg(data, &data, "%lu", &ban_time));
	   if (ban_time < 0)
	     break;
	   
	   gnapster_text_insert(stab, CURR, SYSTEM, "%-20s %-15s %s\n",
				ban, who, reason);
	}
      break;*/
    case NAPSTER_CBAN_LIST:
      if (!flag)
	hook_text_insert(stab, CURR, SYSTEM, "general_message", "%s",
			 "no channel bans found");
      
      flag = 0;
      break;
    case NAPSTER_CHAN_JOIN:
	{
	   GtkCList *clist;
	   
	   append_console_tab(stab, data, 0);
	   clist = chan_lookup_clist(stab, data);
	   if (!clist) 
	     break;
	   
	   gtk_notebook_set_page(GTK_NOTEBOOK(stab->ct->notebook),
				 chan_lookup(stab, data) - 1);
	   gtk_clist_freeze(clist);
	}
      break;
    case NAPSTER_PART_CHAN:
	{
	   char *chan;
	   
	   chan = next_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   remove_console_tab(stab, chan, 0);
	}
      break;
    case NAPSTER_CHAN_LIST_ENTRY:
	{
	   char *chan, *users, *topic;
	   Channel *nchan;
	   
	   chan = next_arg(data, &data);
	   users = next_arg(data, &data);
	   topic = last_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   if (!dialog_open(list_chan_dlg.window))
	     break;
	   
	   nchan = d_new(N_CHAN_INFO);

	   nchan->chan = d_strdup(chan);
	   nchan->users = my_atoi(users);
	   nchan->topic = d_strdup(topic);
	   
	   gnapster_clist_append(list_chan_dlg.clist, NULL, nchan,
				 nchan->chan, users, nchan->topic, NULL);
	   
	   gtk_clist_sort(GTK_CLIST(list_chan_dlg.clist));
	}
      break;
    case NAPSTER_CHAN_LIST:
      if (!dialog_open(list_chan_dlg.window))
	break;
      
      gtk_clist_thaw(GTK_CLIST(list_chan_dlg.clist));
      gtk_label_set_text(GTK_LABEL(list_chan_dlg.label), "");
      break;
    case NAPSTER_CHAN_JOINUSER:
    case NAPSTER_CHAN_PARTUSER:
	{
	   char *chan, *user, *sharing;
	   int link_type, x;
	   ChannelInfo *ch;
	   
	   chan = next_arg(data, &data);
	   user = next_arg(data, &data);
	   sharing = next_arg(data, &data);
	   link_type = my_atoi(next_arg(data, &data));
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   if (!user_info.conf[HIDE_JOINPART])
	     hook_text_insert(stab, chan_lookup(stab, chan),
			      (c == NAPSTER_CHAN_JOINUSER) ? SYSTEM_P : SYSTEM_N,
			      (c == NAPSTER_CHAN_JOINUSER) ? "channel_join" : "channel_part",
			      "%s\4%s\4%s\4%s", user, chan,
			      napster_conn_as_str(link_type), sharing);
	   
	   ch = chan_lookup_data(stab, chan);
	   if (!ch)
	     break;
	   
	   x = g_list_length(GTK_CLIST(ch->clist)->row_list);
	   
	   if (c == NAPSTER_CHAN_JOINUSER) {
	      User *nusr;
	      
	      nusr = d_new(N_USER);

	      nusr->user = d_strdup(user);
	      nusr->shares = my_atoi(sharing);
	      nusr->conn = link_type;
	      
	      gnapster_clist_append(GTK_WIDGET(ch->clist), NULL, nusr,
				    nusr->user, sharing, 
				    napster_conn_as_str(link_type), NULL);
	      
	      gtk_clist_sort(GTK_CLIST(ch->clist));
	   } else if (c == NAPSTER_CHAN_PARTUSER) {
	      int i;
	      char *buf;
	      
	      for(i=0; i<x; i++) {
		 gtk_clist_get_text(GTK_CLIST(ch->clist), i, 0, &buf);
		 
		 if (!j_strcmp(buf, user)) {
		    gtk_clist_remove(GTK_CLIST(ch->clist), i);
		    gtk_clist_sort(GTK_CLIST(ch->clist));
		    break;
		 }
	      }
	   }

	   set_total_users(ch);
	}
      break;
    case NAPSTER_CHAN_USERLIST:
	{
	   char *chan, *user, *share;
	   int conn;
	   User *nusr;
	   GtkCList *clist;
	   
	   chan = next_arg(data, &data);
	   user = next_arg(data, &data);
	   share = next_arg(data, &data);
	   conn = my_atoi(next_arg(data, &data));
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   clist = chan_lookup_clist(stab, chan);
	   
	   if (!clist) 
	     break;

	   nusr = d_new(N_USER);

	   nusr->user = d_strdup(user);
	   nusr->shares = my_atoi(share);
	   nusr->conn = conn;

	   gnapster_clist_append(GTK_WIDGET(clist), NULL, nusr, nusr->user,
				 share, napster_conn_as_str(conn), NULL);
	}
      break;
    case NAPSTER_CHAN_USERLIST_END:
	{
	   char *chan;
	   ChannelInfo *ch;
	   
	   chan = next_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   ch = chan_lookup_data(stab, chan);
	   if (!ch)
	     break;
	   
	   gtk_clist_sort(GTK_CLIST(ch->clist));
	   gtk_clist_thaw(GTK_CLIST(ch->clist));
	   
	   set_total_users(ch);
	}
      break;
    case NAPSTER_CHAN_TOPIC:
	{
	   char *chan, *topic;
	   ChannelInfo *ch;

	   chan = next_arg(data, &data);
	   topic = last_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   ch = chan_lookup_data(stab, chan);
	   if (!ch)
	     break;
	   
	   gtk_entry_set_text(GTK_ENTRY(ch->topic), topic);
	   
	   hook_text_insert(stab, chan_lookup(stab, chan), SYSTEM, "channel_topic",
			    "%s\4%s", chan, topic);
	}
      break;
    case NAPSTER_CHAN_MESSAGE_IN:
	{
	   char *chan, *user, *message;
	   
	   chan = next_arg(data, &data);
	   user = next_arg(data, &data);
	   message = last_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   if (user_ignored(user)) 
	     break;
	   
	   if (strcmp(user, stab->ci->account->user)) {
	      char *al1, *al2;

	      al1 = d_strdup(all_lower(message));
	      al2 = d_strdup(all_lower(stab->ci->account->user));
	      if (!al1 || !al2)
		break;
	      	      
	      if (strstr(al1, al2))
		hook_text_insert(stab, chan_lookup(stab, chan), MESSAGE, "public_msg_highlight",
				 "%s\4%s", user, message);
	      else
		hook_text_insert(stab, chan_lookup(stab, chan), MESSAGE, "public_msg",
				 "%s\4%s", user, message);
	      
	      d_free(al1);
	      d_free(al2);
	   }
	}
      break;
    case NAPSTER_LINKS:
	{
	   if (!data || !(*data)) {
	      GList *iter_ptr;
	      char *dptr;
	      
	      create_server_links(stab);
	      
	      ITER_LIST(stab->ct->links) {
		 LIST_DATA(dptr);
		 d_free(dptr);
	      }
	      
	      g_list_free(stab->ct->links);
	      stab->ct->links = NULL;
	      
	      return;
	   }
	   
	   stab->ct->links = g_list_append(stab->ct->links, d_strdup(data));
	   
/*	   char *host1, *host2, *text[2], *hops;
	   int port1, port2;
	   GtkCTreeNode *parent;
	   
	   if (!data || !(*data)) {
	      if (stab->ct->links_map) {
		 napster_show_links(stab);
	      } else
		hook_text_insert(stab, CURR, SYSTEM, "general_message", "%s",
				 "no links found");
	      
	      stab->ct->links_map = NULL;
	      
	      break;
	   }
	   
	   host1 = next_arg(data, &data);
	   port1 = my_atoi(next_arg(data, &data));
	   host2 = next_arg(data, &data);
	   port2 = my_atoi(next_arg(data, &data));
	   hops = next_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();

	   if (!stab->ct->links_map)
	     stab->ct->links_map = gtk_ctree_new(2, 0);
	   
	   parent = napster_find_link(stab, NULL, host1);
	   if (!parent) {
	      text[0] = host1;
	      text[1] = "-1";
	      text[2] = NULL;
	      
	      parent = gtk_ctree_insert_node(GTK_CTREE(stab->ct->links_map),
					     NULL, NULL, text, 0, NULL, NULL, NULL,
					     NULL, 0, 0);
	   }
	   
	   text[0] = host2;
	   text[1] = hops;
	   text[2] = NULL;
	   
	   gtk_ctree_insert_node(GTK_CTREE(stab->ct->links_map),
				 parent, NULL, text, 0, NULL, NULL, NULL, 
				 NULL, 0, 0);*/
	}
      break;
    case NAPSTER_WHOIS_REPLY:
	{
	   char *nick, *account, *channels, *status, *client, *host, *email;
	   int online_time, sharing, downloads, uploads, connection;
	   int t_uploads, t_downloads, l_port, d_port;
	   
	   nick = next_arg(data, &data);
	   account = next_arg(data, &data);
	   convert(next_arg(data, &data), "%i", &online_time);
	   channels = next_arg(data, &data);
	   status = next_arg(data, &data);
	   convert(next_arg(data, &data), "%i", &sharing);
	   convert(next_arg(data, &data), "%i", &downloads);
	   convert(next_arg(data, &data), "%i", &uploads);
	   convert(next_arg(data, &data), "%i", &connection);
	   client = next_arg(data, &data);
	   convert(next_arg(data, &data), "%i", &t_uploads);
	   convert(next_arg(data, &data), "%i", &t_downloads);
	   host = next_arg(data, &data);
	   convert(next_arg(data, &data), "%i", &l_port);
	   convert(next_arg(data, &data), "%i", &d_port);
	   email = next_arg(data, &data);
	   
/*	   printf("*** nick = %s, account = %s, online_time = %i, "
		  "channels = %s, status = %s, sharing = %i, "
		  "downloads = %i, uploads = %i, connection = %i, "
		  "client = %s, t_uploads = %i, t_downloads = %i, "
		  "host = %s, l_port = %i, d_port = %i, email = %s\n",
		  nick, account, online_time, channels, status, sharing,
		  downloads, uploads, connection, client, t_uploads, t_downloads,
		  host, l_port, d_port, email);*/
	   
	   NA_RESET();
	   
	   if (channels && *channels == ' ')
	     channels++;
	   
	   if (client && !email)
	     hook_text_insert(stab, CURR, MESSAGE, "whois", "%s\4%s\4%i m %i s\4%s\4%s\4%i\4%i\4%i\4%s\4%s",
			      nick, account, online_time / 60, 
			      online_time % 60, channels, status, sharing, 
			      downloads, uploads,
			      napster_conn_as_str(connection), client);
	   else if (client && email)
	     hook_text_insert(stab, CURR, MESSAGE, "whois_admin", "%s\4%s\4%i m %i s\4%s\4%s\4%i\4%i\4%i\4%s\4%s\4%i\4%i\4%s\4%i\4%i\4%s",
			      nick, account, online_time / 60, 
			      online_time % 60, channels, status, sharing, 
			      downloads, uploads,
			      napster_conn_as_str(connection), client,
			      t_uploads, t_downloads, host, l_port, d_port,
			      email);
	}
      break;
    case NAPSTER_WHOIS_OFFLINE:
	{
	   char *user, *account;

	   user = next_arg(data, &data);
	   account = next_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();

	   hook_text_insert(stab, CURR, SYSTEM, "whois_failure", "%s",
			    user);
	}
      break;
    case NAPSTER_UPLOAD_REQUEST:
	{
	   char *user, *file;
	   int connection;
	   
	   user = next_arg(data, &data);
	   file = next_arg(data, &data);
	   connection = 0;
/*	   convert(next_arg(data, &data), "%i", &connection);*/
	   
	   NA_ERR_HANDLE_BREAK();

	   if (user_ignored(user))
	     break;

	   if (!upload_accepted(stab, user, file))
	     break;
	   
	   pending_upload(stab, user, file);
	   
	   napster_send(ci->sock, NAPSTER_ACCEPT_UPLOAD_REQUEST,
			"%s \"%s\"", user, file);
	}
      break;
    case NAPSTER_SERVER_MESSAGE:
	{
	   GList *ptr;
	   char *str;
	   int match = 0, c;
	   
	   for(ptr=signore_list; ptr; ptr=ptr->next) {
	      str = ptr->data;
	      if (str) {
		 if (wild_match(str, data))
		   match = 1;
	      }
	      if (match)
		break;
	   }
	   
	   if (!strncasecmp(data, "duplicate file", strlen("duplicate file")))
	     match = 1;
	   
	   c = wild_match("You * killed *", data) ? CONSOLE : CURR;
	   
	   if (!match)
	     hook_text_insert(stab, c, SYSTEM, "server_msg", "%s", data);
	   
	   match = 0;
	   if (wild_match("User % is not * online*", data))
	     match = 1;
	   
	   if (match)
	     end_search(stab, NULL);
	}
      break;
    case NAPSTER_RESUME_RESULT:
	{
	   Search *resume;
	   
	   resume = gnapster_parse_resume(data);
	   
	   gnapster_clist_append(gmain->st->clist, gmain->st->vbox,
				 resume, resume->trunc_file, 
				 napster_conn_as_str(resume->connection),
				 resume->user, "", "", NULL);
	}
      break;
    case NAPSTER_RESUME_END:
      end_search(stab, "No resume entries found");

      break;
    case NAPSTER_SEARCH_RESULT:
	{
	   Search *search;
	   
	   if (cancels(ci))
	     break;
	   
	   search = gnapster_parse_search(stab, data);
	   if (!search) 
	     break;
	   
	   insert_search_result(gmain->st, search, 1);
	}
      break;
    case NAPSTER_END_SEARCH:
      if (!cancels(ci))
	end_search(stab, "No results matched your search criteria");
      else
	cancels_inc(ci, -1);
      
      break;
    case NAPSTER_OPENNAP_BROWSE_RESULT:
	{
	   Search *browse;
	   char *user, *path, id[41];

	   user = next_arg(data, &data);
	   path = next_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   for(;;) {
	      browse = gnapster_parse_browse(stab, data, &data, 1);
	      if (!browse) 
		break;
	      
	      memset(&id, '0', sizeof(id));
	      id[40] = 0;
	      
	      if (!strcmp(browse->id, "0")) {
		 d_free(browse->id);
		 browse->id = d_strdup(id);
	      }
	      
	      d_strexp(&browse->file, "%s\\%s",
		       path, browse->trunc_file);
	      
	      browse->user = d_strdup(user);
	      
	      browse_results++;
	      
	      browse_tab_add(browse);
	   }
	}
      break;
    case NAPSTER_OPENNAP_BROWSE:
	{
	   char *results;
	   
	   update_status(1, stab, NULL, 0);
	   
	   gtk_entry_set_text(GTK_ENTRY(gmain->bt->entry), "");
	   gtk_entry_set_editable(GTK_ENTRY(gmain->bt->entry), 1);
	   
	   gtk_ctree_sort_recursive(GTK_CTREE(gmain->bt->ctree), NULL);
	   gtk_clist_thaw(GTK_CLIST(gmain->bt->ctree));
	   
	   if (g_list_length(GTK_CLIST(gmain->bt->ctree)->row_list) == 0)
	     gnapster_clist_append(gmain->bt->ctree, gmain->bt->vbox, NULL,
				   "No mp3s found.", "", "", "", "", NULL);

	   d_msprintf(&results, "%i files", browse_results);
	   gtk_clist_set_text(GTK_CLIST(gmain->bt->ctree), 0, 1, results);
	   d_free(results);

	   ci->state &= ~BROWSING_MASK;
	   browse_results = 0;
	}
      break;
    case NAPSTER_BROWSE_RESULT:
	{
	   Search *browse;
	   
	   browse = gnapster_parse_browse(stab, data, &data, 0);
	   if (!browse) 
	     break;

	   browse_results++;
	   
	   browse_tab_add(browse);
	}
      break;
    case NAPSTER_END_BROWSE:
      end_browse(stab);
      break;
    case NAPSTER_BROWSE_ERROR:
      hook_text_insert(stab, CONSOLE, SYSTEM, "general_message", "%s",
		       "user is not online");
      
      update_status(1, stab, NULL, 0);
      
      gtk_clist_thaw(GTK_CLIST(gmain->bt->ctree));
      
      ci->state &= ~BROWSING_MASK;
      
      break;
    case NAPSTER_DOWNLOAD_REPLY:
	{
	   Transfer *download, *real;
	   int row = -1;
	   
	   download = gnapster_parse_download(stab, data, 0);
	   if (!download) 
	     break;
	   
	   real = real_download(stab, download->user, download->file, &row);
	   if (!real || real->status != 0) {
	      j_free(TRANSFER, download);
	      break;
	   }
	   
	   real->ttype = DOWNLOAD;
	      
	   real->status++;
	   
      	   if (download->port == 0) {
	      napster_send(ci->sock, NAPSTER_ALT_GET, "%s \"%s\"",
			   download->user, download->file);
	      
	      gtk_clist_set_text(GTK_CLIST(gmain->dt->clist), row, 4,
				 "Awaiting connection...");	      
	   } else {
	      d_free(real->file);
	      d_free(real->user);
	      
	      push_transfer_info(real, download->user, download->file, NULL);

	      real->ip = download->ip;
	      real->port = download->port;
	      
	      gtk_clist_set_text(GTK_CLIST(gmain->dt->clist),
				 row, 4, "Connecting...");
	      
	      napster_connect_user(real);
	   }

	   j_free(TRANSFER, download);
	}
      break;
    case NAPSTER_ALT_DOWNLOAD_REPLY:
	{
	   Transfer *upload;
	   
	   upload = gnapster_parse_download(stab, data, 1);
	   if (!upload) 
	     break;
	   
	   /* Ok, we need to connect to them now */
	   upload->ttype = UPLOAD;
	   
	   if (upload->port == 0) {
	      hook_text_insert(stab, CONSOLE, SYSTEM, "reject_upload", "%s\4%s",
			       upload->user, upload->trunc_file);
	      j_free(TRANSFER, upload);
	   } else
	     napster_connect_user(upload);
	}
      break;
    case NAPSTER_DOWNLOAD_ERROR:
	{
	   char *user, *file;
	   Transfer *real;
	   
	   user = next_arg(data, &data);
	   file = next_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();

	   hook_text_insert(stab, CONSOLE, SYSTEM, "whois_failure", "%s",
			    user);
	   real = real_download(stab, user, file, NULL);
	   download_close(real, TRANSFER_ERROR);
	}
      break;
    case NAPSTER_SENDLIMIT_REACHED:
	{
	   char *user, *file, *trunc_file;
	   unsigned long int size;
	   int limit, row = -1;
	   Transfer *real;
	   
	   user = next_arg(data, &data);
	   file = next_arg(data, &data);
	   convert(next_arg(data, &data), "%lu", &size);
	   limit = my_atoi(next_arg(data, &data));
	   
	   NA_ERR_HANDLE_BREAK();

	   trunc_file = strrchr(file, '\\');
	   if (!trunc_file) 
	     break;
	   
	   trunc_file++;
	   
	   real = real_download(stab, user, file, &row);
	   if (!real)
	     break;
	   
	   gtk_clist_set_text(GTK_CLIST(gmain->dt->clist), row, 4,
			      _("Remotely queued"));
	   
	   /* Try again every 20 seconds */
	   if (!user_info.conf[AUTOCANCEL]) {
	      TIMEOUT_REMOVE(real->start_timeout);
	      
	      real->timeout_tag = TIMEOUT_ADD(25000, remote_queue_check,
					      real);
	   }
	   
	   real->status = 1;
	   
	   if (user_info.conf[AUTOCANCEL])
	     download_close(real, TRANSFER_ERROR);
	}
      break;
    case NAPSTER_PING:
	{
	   char *user;
	   
	   user = next_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   if (user_ignored(user))
	     break;
	   
	   hook_text_insert(stab, CONSOLE, SYSTEM, "remote_ping", "%s",
			    user);
	   napster_send(ci->sock, NAPSTER_PONG, user);
	}
      break;
    case NAPSTER_ACTION:
	{
	   char *chan, *user, *message;
	   
	   chan = next_arg(data, &data);
	   user = next_arg(data, &data);
	   message = next_arg(data, &data);
	   
	   NA_ERR_HANDLE_BREAK();
	   
	   if (user_ignored(user) ||
	       !strcmp(user, stab->ci->account->user))
	     break;

	   hook_text_insert(stab, chan_lookup(stab, chan), SYSTEM, "public_action",
			    "%s\4%s", user, message);
	}
      break;
    case NAPSTER_REQUEST_PORTCHANGE:
	{
	   int port;
	   
	   port = my_atoi(next_arg(data, &data));
	   
	   NA_ERR_HANDLE_BREAK();

	   hook_text_insert(stab, CONSOLE, SYSTEM, "general_message", "%s %i",
			    "server is attempting to change your port to",
			    port);

	   handle_rebind(port);
	}
      break;
    case NAPSTER_NEWUSER_SUCCESS:
      napster_send(ci->sock, NAPSTER_NEW_USER_LOGIN, "%s %s %i \"%s\" %i %s",
		   stab->ci->account->user, stab->ci->account->pass,
		   gmain->port, NAP_VERSION, user_info.connection, 
		   "anonymous@napster.com");
      break;
    case NAPSTER_NEWUSER_REG:
      login(stab, ci->sock);
/*      j_error_dialog(_("The username you have chosen has already been taken on napster.  Please choose a new one."));
      disconnect(stab);*/
      break;
    case NAPSTER_NEWUSER_INVALID_NICK:
	{
	   char *s;
	   
	   d_msprintf(&s, "%s - Username considered invalid by server.",
		      stab->ci->server);
	   j_error_dialog(s);
	   d_free(s);
	   
	   disconnect(stab);
	}
      break;
    case NAPSTER_ERROR:
      if (!strcmp(data, "Invalid Password")) {
	 char *s;
	 d_msprintf(&s, "%s - Invalid Password.  This can be caused by attempting to connect with a user\nnot yet registered to the service.  Try checking New Account before attempting\nto do this and if the same problem occurs, then select a different user.",
		    stab->ci->server);
	 j_error_dialog(s);
      } else if (!strncmp(data, "Both you and ", 13))
	break;
      else
	hook_text_insert(stab, CURR, SYSTEM, "general_error", "%s", data);
      
      fprintf(stderr, "error -> %s\n", data);
      disconnect(stab);
      break;
    default:
/*      fprintf(stderr, "unknown[%i] -> %s\n", c, data);*/
      break;
   }
}

void napster_get_best_host(STab *stab, char *ip, unsigned short port) {   
   dispatch_connect(inet_addr(ip), htons(port),
		    &(stab->ci->sock), &(stab->ci->sock_input),
		    GDK_INPUT_WRITE, conn_main, stab);
}

void napster_connect(STab *stab, char *ip, int port) {
   ConnInfo *ci;
   
   ci = stab->ci;
   
   if (!ip || !port)
     return;
   
   dispatch_connect(inet_addr(ip), htons(port),
		    &(stab->ci->sock), &(ci->sock_input),
		    GDK_INPUT_WRITE | GDK_INPUT_EXCEPTION,
		    conn_final, stab);
}

void napster_connect_user(Transfer *t) {
   t->size = 0;
   
   dispatch_connect(t->ip, htons(t->port), &(t->sock), &(t->sock_input),
		    GDK_INPUT_WRITE | GDK_INPUT_EXCEPTION,
		    conn_outgoing, t);
   
/*   printf("pinging %lu...\n", t->ip);
   start_ping(t->ip);*/
}

void napster_unbind() {
   INPUT_REMOVE(gmain->bind_input);
   gmain->bind_input = -1;
   
   CLOSE(gmain->bind_sock);
}

void napster_bind(unsigned short port) {
   int sock, len = 1;
   struct sockaddr_in server;
   
   gmain->port = port;
   
   if (!port || gmain->bind_input > -1)
     return;
   
   sock = socket(AF_INET, SOCK_STREAM, IPPROTO_IP);
   if (sock < 0) {
      j_error("socket", NULL);
      return;
   }
   
   memset(&server, 0, sizeof(server));
   server.sin_family = AF_INET;
   server.sin_addr.s_addr = htonl(INADDR_ANY);
   server.sin_port = htons(port);

   setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &len, sizeof(len));
   
   if (bind(sock, (struct sockaddr *)&server, sizeof(server)) < 0) {
      j_error("bind", NULL);
      perror("bind");
/*      j_error_dialog(_("An error occured while attempting to bind port 6699.  It is possible that another napster client is running."));*/
      CLOSE(sock);
      
      return;
   }
   
   listen(sock, 10 + user_info.conf[MAX_UPLOADS]);
   
   gmain->bind_sock = sock;
   gmain->bind_input = INPUT_ADD(sock, GDK_INPUT_READ,
				 handle_connection, NULL);
}

void handle_rebind(int nport) {
   if (user_info.conf[FIREWALL])
     nport = 0;
   
   if (gmain->port == nport)
     return;
   
   napster_unbind();
   napster_bind(nport);
}

int napster_conn_as_int(char *s) {
   int x = 0;
   
   while(conn_table[x]) {
      if (!strcmp(conn_table[x], s)) return x;
      x++;
   }
   
   return -1;
}

char *napster_conn_as_str(int i) {
   if (i < 0 || i > 10) return "Unknown";
   return conn_table[i];
}

int napster_user_uploads(STab *stab, char *user) {
   GList *ptr;
   GtkCList *clist;
   GtkCListRow *clist_row;
   Transfer *data;
   int uploads = 0;
   
   clist = GTK_CLIST(gmain->ut->clist);
   
   ptr = clist->row_list;
   if (!ptr)
     return -1;
   
   for(; ptr; ptr=ptr->next) {
      clist_row = ptr->data;
      
      data = clist_row->data;
      
      if (!data)
	break;
      
      if (!strcmp(data->user, user))
	uploads++;
   }
   
   return uploads;
}
