#include <stdio.h>
#include <stdlib.h>

#include <string.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <sys/stat.h>
#include <fcntl.h>

#include <unistd.h>
#include <ctype.h>

#include <time.h>

#include "gnapster.h"
#include "network.h"
#include "browse.h"
#include "ui.h"
#include "upload.h"
#include "callbacks.h"
#include "queue.h"
#include "servers.h"
#include "resume.h"
#include "chan.h"
#include "text.h"
#include "utils.h"
#include "connect.h"
#include "download.h"

#include "debug.h"

#ifndef MSG_WAITALL
#define MSG_WAITALL 0x100
#endif /* MSG_WAITALL */

/* 1 second */
#define CLIST_UPDATE_SPEED 1000

#define CONNECT 0
#define ACCEPT 1

extern GnapsterMain *gmain;
extern UserInfo user_info;

int n_send(int sock, char *fmt, ...) {
   char *data;
   va_list args;
   int x = -1;
   
   data = NULL;
   
   if (sock < 0)
     return x;
   
   if (fmt) {
      va_start(args, fmt);
      data = d_strdup_vprintf(fmt, args);
      va_end(args);
   }
   
   if (data)
     x = send(sock, data, strlen(data), 0);
   
   d_free(data);
   
   return x;
}

void set_blocking(int s, int block) {
   if (block)
     return;
   
   fcntl(s, F_SETFL, (block) ? 0 : O_NONBLOCK);
}

int sock_err(int source) {
   int err, len, x;
   
   len = sizeof(err);
   
   x = getsockopt(source, SOL_SOCKET, SO_ERROR, &err, &len);

   return (x < 0 || err);
}

/* TRANSFERS */

void handle_dcc_request(STab *stab, char *user, char *msg) {
   Transfer *dcc;
   unsigned long ip;
   unsigned short port;
   char *file;
   
   return;
   
   next_arg(msg, &msg);
   next_arg(msg, &msg);
   convert(next_arg(msg, &msg), "%lu", &ip);
   convert(next_arg(msg, &msg), "%hu", &port);
   file = next_arg(msg, &msg);
   
   if (na_err)
     return;
   
   NA_RESET();
   
   dcc = d_new(TRANSFER);
   
   dcc->user = d_strdup(user);
   dcc->file = d_strdup(file);
   dcc->trunc_file = trunc_file(dcc->file);
   dcc->st = stab;
   
   dcc->ip = BSWAP32(ip);
   dcc->port = port;
   
   show_download(dcc);
   
   
}


void download_close(Transfer *download, TransferInfo info) {
   int incomplete = 0, row;
   float avgkps;
   STab *stab;
   
   d_assert(download != NULL);
   
   stab = download->st;
   
   INPUT_REMOVE(download->sock_input);
   
   TIMEOUT_REMOVE(download->update_tag);
   TIMEOUT_REMOVE(download->timeout_tag);
   
   TIMEOUT_REMOVE(download->accept_timeout);
   TIMEOUT_REMOVE(download->start_timeout);

   if (download->fdesc)
     fclose(download->fdesc);
   
   if (info & TRANSFER_FILE_REMOVE && download->fpath && download->status == 2) {
      unlink(download->fpath);
      remove_resume_entry(download);
   }
   
   CLOSE(download->sock);
   
   avgkps = mod_xfer_stats(download);
   
   if (download->status == 2) {
      FORSTABS(napster_send(stab->ci->sock, NAPSTER_DOWNLOAD_COMPLETE, NULL));
   }
   
   if (info & (TRANSFER_CLOSED | TRANSFER_ERROR))
     hook_text_insert(stab, CONSOLE, SYSTEM_N, "cancel_download", "%s\4%s\4%s",
		      download->user, download->trunc_file,
		      simplify_size(download->total - download->size));
   else if (info & TRANSFER_TIMEOUT)
     hook_text_insert(stab, CONSOLE, SYSTEM_N, "timeout_download", "%s\4%s\4%s",
		      download->user, download->trunc_file,
		      simplify_size(download->total = download->size));
   
   if (download->size == 0) {
      remove_resume_entry(download);
      unlink(download->fpath);
   } else if (download->status == 2) {
      if (info & TRANSFER_FINISHED) {	 
	 hook_text_insert(stab, CONSOLE, SYSTEM_P, "download_complete", "%s\4%s\4%.02f",
			  download->user, download->trunc_file, avgkps);
	 
	 handle_exec_dl(download);
	 
	 remove_resume_entry(download);
	 
	 if (download->resume_path) {
	    char *path;
	    d_msprintf(&path, "%s/%s", user_info.download_dir,
		       download->trunc_file);
	    conv_file(path);
	    rename(download->fpath, path);
	 }
      }
      
      incomplete = (info & (TRANSFER_CLOSED | TRANSFER_TIMEOUT));

      if (incomplete && download->fpath && !download->resume_path) {
	 char *path, *file;
	 char *new_path;

	 path = d_strdup(download->fpath);
	 file = strrchr(path, '/');
	 if (file) {
	    int x = -1;
	    struct stat st;

	    *file++ = 0;
	    
	    d_msprintf(&new_path, "%s/incomplete", user_info.download_dir);
	    
	    if (stat(new_path, &st) < 0)
	      x = mkdir(new_path, 0755);
	    
	    if (S_ISDIR(st.st_mode) || x >= 0) {
	       d_free(new_path);
	       
	       d_msprintf(&new_path, "%s/incomplete/%s",
			  user_info.download_dir, file);
	       
	       rename(download->fpath, new_path);
	       
	       modify_resume_entry(new_path, download);
	    }
	    
	    d_free(new_path);
	 }
	 d_free(path);
      }
   }
   
   row = gtk_clist_find_row_from_data(GTK_CLIST(gmain->dt->clist), download);
   if (row >= 0)
     gtk_clist_remove(GTK_CLIST(gmain->dt->clist), row);
   else
     j_free(TRANSFER, download);

   safe_exit_check();
   queue_check(stab);
}

void upload_close(Transfer *upload, TransferInfo info) {
   int row;
   float avgkps;
   STab *stab;
   
   d_assert(upload != NULL);
   
   stab = upload->st;
   
   INPUT_REMOVE(upload->sock_input);
   
   TIMEOUT_REMOVE(upload->update_tag);
   TIMEOUT_REMOVE(upload->timeout_tag);
   
   TIMEOUT_REMOVE(upload->accept_timeout);
   TIMEOUT_REMOVE(upload->start_timeout);

   FORSTABS(napster_send(stab->ci->sock, NAPSTER_UPLOAD_COMPLETE, NULL));

   avgkps = mod_xfer_stats(upload);
   
   if (info & TRANSFER_CLOSED)
     hook_text_insert(stab, CONSOLE, SYSTEM_N, "cancel_upload", "%s\4%s",
		      upload->user, upload->trunc_file);
   else if (info & TRANSFER_FINISHED)
     hook_text_insert(stab, CONSOLE, SYSTEM_P, "upload_complete", "%s\4%s\4%.02f",
		      upload->user, upload->trunc_file, avgkps);
   else if (info & TRANSFER_TIMEOUT)
     hook_text_insert(stab, CONSOLE, SYSTEM_N, "timeout_upload", "%s\4%s",
			                         upload->user, upload->trunc_file);
   
   CLOSE(upload->sock);
   
   if (upload->fdesc)
     fclose(upload->fdesc);
   
   row = gtk_clist_find_row_from_data(GTK_CLIST(gmain->ut->clist), upload);
   if (row >= 0)
     gtk_clist_remove(GTK_CLIST(gmain->ut->clist), row);
   else
     j_free(TRANSFER, upload);
   
   safe_exit_check();
}

void transfer_close(Transfer *t, TransferInfo info) {
   d_assert(t != NULL);
   
   if (t->ttype == DOWNLOAD)
     download_close(t, info);
   else
     upload_close(t, info);
}

int start_timeout(void *data) {
   Transfer *t;

   d_assert_return(data != NULL, 0);
   
   t = data;
   
   hook_text_insert(t->st, CURR, SYSTEM_N, "general_error", "%s",
		    "download start timed out");
   
   if (t->status >= 2)
     fprintf(stderr, "t->status >= 2?!?\n");
   
   download_close(t, TRANSFER_ERROR);
   
   return 0;
}

int accept_timeout(void *data) {
   Connection *c;
   
   d_assert_return(data != NULL, 0);
   
   c = data;
   
   close_accept(c);
   
   return 0;
}

void show_download(Transfer *download) {
   char *kbps, *user;
   int row;

   d_assert(download != NULL);
   
   /* can this file be resumed? if so, setup resume_path */
   check_resume(download);

   /* this is for the updating of the transfer once it's started */
   download->calls = 5;
   
   d_msprintf(&user, "[%i] %s", download->st->pn + 1, download->user);
   d_msprintf(&kbps, "%i", download->kbps);
   
   row = gnapster_clist_append(gmain->dt->clist, gmain->dt->vbox,
			       download, download->trunc_file,
			       napster_conn_as_str(download->connection),
			       user, kbps, "Requesting...", NULL);
   
   set_clist_server(gmain->dt->clist, download->st, row);
   
   d_free(user);
   d_free(kbps);
   
   download->ttype = DOWNLOAD;
   
   /* if we dont start in 5 minutes, cancel it */
   download->start_timeout = TIMEOUT_ADD(60000 * 5, start_timeout,
					 download);
}

FILE *open_path(Transfer *t) {
   FILE *fdesc;

   d_assert_return(t != NULL, NULL);
   
   fdesc = NULL;
   
   if (!t->resume_path && t->ttype == DOWNLOAD)
     fdesc = open_file(t, t->trunc_file);
   
   if (t->ttype == UPLOAD) {
      /* uploading */
      if (!upload_accepted(t->st, t->user, t->file))
	return NULL;
      
      if (!verify_file_shared(t->file)) {
	 n_send(t->sock, "FILE NOT SHARED");
	 return NULL;
      }
      
      fdesc = fopen(UNIXPATH(t->file), "r");
   } else if (t->resume_path) {
      /* downloading + resuming */
      
      fdesc = fopen(t->resume_path, "a");
      t->fpath = d_strdup(t->resume_path);
   }
   
   return fdesc;
}

float get_avg_kps(Transfer *t) {
   time_t ttime;
   float kps;
   
   ttime = time(NULL);
   if (!ttime || ttime <= t->start)
     return 0.0;
   
   ttime -= t->start;
   
   kps = ((float)t->size / 1024.0) / (float)ttime;
   
   return kps;
}

float mod_xfer_stats(Transfer *t) {
   float kps, avgkps, *avg_t_kps;
   unsigned long *xfers;
   STab *stab;
   
   stab = t->st;
   
   kps = get_avg_kps(t);
   
   if (t->ttype == UPLOAD) {
      xfers = &stab->ci->upload_xfers;
      avg_t_kps = &stab->ci->upload_kps;
   } else {
      xfers = &stab->ci->download_xfers;
      avg_t_kps = &stab->ci->download_kps;
   }
   
   (*xfers)++;
   avgkps = ((*avg_t_kps * (*xfers - 1)) + kps) / *xfers;
   *avg_t_kps = avgkps;
   
   return kps;
}

void inc_xfer_size(Transfer *t, size_t n) {
   float kbytes;
   float mbytes;
   
   kbytes = (float)n / 1024.0;
   mbytes = kbytes / 1024.0;
   
   if (t->ttype == UPLOAD) {
      t->st->ci->upload_size += kbytes;
      gmain->out += mbytes;
   } else {
      t->st->ci->download_size += kbytes;
      gmain->in += mbytes;
   }
   
   t->size += n;
   t->temp_size += n;
}

int get_temp_size(int type) {
   int rw;
   GtkCList *clist;
   Transfer *t;
   int tts = 0;
   
   clist = (type == DOWNLOAD) ? 
     GTK_CLIST(gmain->dt->clist) : GTK_CLIST(gmain->ut->clist);
   
   rw = g_list_length(clist->row_list);
   while(rw-- > 0) {
      t = gtk_clist_get_row_data(clist, rw);
      if (!t)
	continue;
      
      tts += t->temp_size;
   }
   
   return tts;
}

void transfers_disable(int type) {
   int rw;
   GtkCList *clist;
   Transfer *t;
   
   clist = (type == DOWNLOAD) ? 
     GTK_CLIST(gmain->dt->clist) : GTK_CLIST(gmain->ut->clist);
   
   rw = g_list_length(clist->row_list);
   while(rw-- > 0) {
      t = gtk_clist_get_row_data(clist, rw);
      if (!t) 
	continue;
      
      INPUT_REMOVE(t->sock_input);
   }
}

void transfers_enable(int type) {
   int rw;
   GtkCList *clist;
   Transfer *t;
   
   clist = (type == DOWNLOAD) ? 
     GTK_CLIST(gmain->dt->clist) : GTK_CLIST(gmain->ut->clist);
   
   rw = g_list_length(clist->row_list);
   while(rw-- > 0) {
      t = gtk_clist_get_row_data(clist, rw);
      if (!t) 
	continue;
      
      transfer_callbacks(t);
   }
}

int h_bandwidth_cap(Transfer *t) {
   int s, tts, cap_size;
   
   cap_size = (t->ttype == DOWNLOAD) ? 
     user_info.conf[BW_DOWN] : user_info.conf[BW_UP];
   
   /* unlimited? */
   if (!cap_size)
     return RW_BUFFER;
   
   cap_size *= 1024; /* convert to bytes */
   
   /* get total tempsize for all transfers of a given type */
   tts = get_temp_size(t->ttype);
   
   s = cap_size - tts;
   if (s > RW_BUFFER)
     s = RW_BUFFER;
   
   /* no more bandwidth left, disable */
   if (s <= 0)
     transfers_disable(t->ttype);
   
   return s;
}

void h_download(void *data, int source, GdkInputCondition cond) {
   Transfer *download;
   char buf[RW_BUFFER];
   int n, s;
   
   d_assert(data != NULL);
   
   download = data;
   
   s = h_bandwidth_cap(download);
   if (s <= 0)
     return;
   
   n = recv(source, buf, s, 0);
   
   if (n <= 0 || download->size == download->total) {
      if (download->size == download->total ||
	  (download->total - download->size) <= RW_BUFFER)
	download_close(download, TRANSFER_FINISHED);
      else
	download_close(download, TRANSFER_CLOSED);
      
      return;
   }
   
   if (!download->fdesc ||
       fwrite(buf, n, sizeof(char), download->fdesc) <= 0) {
      hook_text_insert(download->st, CONSOLE, SYSTEM_N, "general_error", "%s",
		       "error writing to disk");
      
      download_close(download, TRANSFER_CANCEL);
      
      return;
   }
   
   inc_xfer_size(download, n);
}

void h_upload(void *data, int source, GdkInputCondition cond) {
   Transfer *upload;
   char buf[RW_BUFFER];
   int n, s, sent;
   
   d_assert(data != NULL);
   
   upload = data;
   
   s = h_bandwidth_cap(upload);
   if (s <= 0)
     return;
   
   if (upload->size >= upload->total) {
      upload_close(upload, TRANSFER_FINISHED);
      return;
   }
   
   lseek(fileno(upload->fdesc), upload->size, SEEK_SET);

   n = read(fileno(upload->fdesc), buf, sizeof(buf));
   if (n <= 0) {
      upload_close(upload, TRANSFER_CLOSED);
      return;
   }

   sent = send(upload->sock, buf, n, 0);
   if (sent <= 0) {
      upload_close(upload, TRANSFER_CLOSED);
      return;
   }

   inc_xfer_size(upload, sent);
}

int update_transfer(void *data) {
   Transfer *t;
   GtkCList *clist;
   char *str;
   
   d_assert_return(data != NULL, 0);
   
   t = data;
   
   if (!t->total)
     return 1;
   
   t->calls++;
   
   /* calculate kps */
/*   t->kps = get_avg_kps(t);*/
   t->kps = 
     ((float)t->temp_size / 1024.0);
   
   /* timeout calls */
   if (t->calls >= 3 * (CLIST_UPDATE_SPEED / 1000)) {
      if (t->temp_size == 0)
	t->timeout_calls++;
      else
	t->timeout_calls = 0;
      
      t->calls = 0;

      /* if we have 0kps for 5 minutes, timeout */
      if (t->timeout_calls == ((5 * 60) / 3)) {
	 transfer_close(t, TRANSFER_TIMEOUT);
	 return 0;
      }
   }

   /* reset */
   t->temp_size = 0;
   transfers_enable(t->ttype);
   
   /* only update every CLIST_UPDATE_SPEED interval */
   if (t->calls < (CLIST_UPDATE_SPEED / 1000))
     return 1;

   /* update the user interface */
   d_msprintf(&str, "%.02fM [%i%%] @ %.01fk/s",
	      ((float)t->size / 1024) / 1024,
	      (int)((float)t->size / (float)t->total * 100),
	      (float)t->kps);
   
   clist = (t->ttype == DOWNLOAD) ? 
     GTK_CLIST(gmain->dt->clist) : GTK_CLIST(gmain->ut->clist);
   
   gtk_clist_set_text(clist, gtk_clist_find_row_from_data(clist, t),
		      (t->ttype == DOWNLOAD) ? 4 : 3, str);
   
   d_free(str);
   
   return 1;
}

/* Peer-To-Peer Protocol:
 * 
 *  *NOTE* the client that accept's the connection must immediately send
 *  a '1' to the other peer, dont ask me why, you just do :)
 * 
 *  * accept():
 * 
 *    * Downloading
 * 
 *      <REMOTE> SENDuser "file" total_size
 *       Their client sends this to show you that the user 'user' is going
 *       to send you the file 'file', which it's total size is 'total_size'.
 * 
 *      <LOCAL> local_size
 *       You send back the size of the file that you currently have.  Most
 *       often this will be 0, as you do not have the file yet, but if you do,
 *       and you are going to resume it, send the current size of the file
 *       that you have.
 * 
 *    * Uploading
 * 
 *      <REMOTE> GETuser "file" remote_size
 *       The remote client sends this to ask that you send the file to them
 *       on this TCP connection.  remote_size is the size that their file is,
 *       usually 0, but if it is > 0, you must start sending the file at
 *       that offset
 * 
 *      <LOCAL> total_size
 *       You then respond with the total size of the file on your machine
 * 
 *  * connect():
 * 
 *    * Downloading
 *      
 *      <LOCAL> GETmynick "file" local_size
 *       user should be your nickname here, so their client knows who
 *       is downloading from them.  local_size is the size as you have it
 *       completed on your drive, 0 if it's a new file
 * 
 *      <REMOTE> total_size
 *       The remote client will tell you the total size
 * 
 *    * Uploading
 * 
 *      <LOCAL> SENDmynick "file" total_size
 *       Instruct their client that you will be sending the file with
 *       the given size to them
 * 
 *      <REMOTE> remote_size
 *       They will tell you at what offset to resume at
 */

void finalize_connect_download(void *data, int source, GdkInputCondition cond) {
   Transfer *t;
   
   d_assert(data != NULL);
   
   t = data;
   
   if (!connect_complete(t)) {
      download_close(t, TRANSFER_ERROR);
      
      return;
   }
   
   INPUT_REMOVE(t->sock_input);
   
   setup_download_finish(t);
}

void setup_download_finish(Transfer *t) {   
   TIMEOUT_REMOVE(t->start_timeout);
   
   FORSTABS(napster_send(stab->ci->sock, NAPSTER_DOWNLOAD_START, NULL));
   
   if (!t->resume_path)
     append_resume_entry(t);
   
   transfer_callbacks(t);
}

int connect_complete(Transfer *t) {
   int n;
   char *ptr, *nptr, buf[RW_BUFFER], new_buf[RW_BUFFER];
   unsigned long int size;
   
   n = recv(t->sock, buf, sizeof(buf) - 1, MSG_PEEK);
   if (n <= 0)
     return 0;
   
   buf[n] = 0;
   
   if (!isdigit(buf[0])) /* we got an error */
     return 0;
   
   for(ptr=buf, nptr=new_buf; *ptr && isdigit(*ptr); ptr++)
     *nptr++ = *ptr;
   
   *nptr++ = 0;
   
   /* skip the appropriate number of bytes now */
   recv(t->sock, buf, strlen(new_buf), 0);
   
   convert(new_buf, "%lu", &size);
   
   /* ok, this data is either the remote offset, or the total size */
   if (t->ttype == DOWNLOAD)
     t->total = size;
   else
     t->size = size;
   
   return 1;
}

int setup_transfer(Transfer *t) {
   struct stat st;
   
   t->fdesc = open_path(t);
   if (!t->fdesc) {
      if (t->ttype == DOWNLOAD)
	j_error_dialog("Failed to open file for write.  Please verify that your download directory exists");
      
      return 0;
   }
   
   DOSPATH(t->file);
   
   if (t->ttype == UPLOAD) {
      fstat(fileno(t->fdesc), &st);
      
      t->total = st.st_size;
      
      /* let the other end know the size */
      if (t->conn_type == ACCEPT)
	n_send(t->sock, "%lu", t->total);
      else
	napster_send_send(t->sock, t->st->ci->account->user, t->file,
			  t->total);
   }
   
   if (t->ttype == DOWNLOAD && t->resume_path) {
      fstat(fileno(t->fdesc), &st);

      t->size = st.st_size;
      
      hook_text_insert(t->st, CURR, SYSTEM_P, "download_resume", "%s\4%s\4%s",
		       t->user, t->trunc_file, 
		       simplify_size(t->size));
   }
   
   if (t->ttype == DOWNLOAD) {
      if (t->conn_type == ACCEPT)
	n_send(t->sock, "%li", t->size);
      else
	napster_send_get(t->sock, t->st->ci->account->user, t->file,
			 t->size);
   }

   /* we need to get the rest of the data now */
   if (t->ttype == DOWNLOAD && t->conn_type == CONNECT) {
      t->sock_input = INPUT_ADD(t->sock, GDK_INPUT_READ,
				finalize_connect_download, t);
   }

   return 1;
}

void transfer_callbacks(Transfer *t) {
   GdkInputCondition cond;
   void *func;
   
   d_assert(t != NULL);
   
   if (t->sock_input >= 0)
     return;
   
   /* time this download started */
   if (!t->start)
     t->start = time(NULL);
   
   cond = (t->ttype == DOWNLOAD) ?
     GDK_INPUT_READ : GDK_INPUT_WRITE;
   
   func = (t->ttype == DOWNLOAD) ?
     h_download : h_upload;
   
   t->sock_input = INPUT_ADD(t->sock, cond, func, t);
   
   if (t->update_tag < 0)
     t->update_tag = TIMEOUT_ADD(1000, update_transfer, t);
}

/* The start_transfer functions will properly setup and then start
 * a transfer */
void start_download(Transfer *download) {
   d_assert(download != NULL);
   
   download->status++;
   
   /* status 2 means it is ready for start_download */
   if (download->status != 2) {
      download_close(download, TRANSFER_ERROR);
      return;
   }
   
   if (!setup_transfer(download)) {
      download_close(download, TRANSFER_ERROR);
      
      return;
   }
   
   if (download->conn_type != CONNECT)
     setup_download_finish(download);
/*
   napster_send(stab->ci->sock, NAPSTER_DOWNLOAD_START, NULL);   

   if (!download->resume_path)
     append_resume_entry(download);

   transfer_callbacks(stab, download);*/
}

void start_upload(Transfer *upload) {
   char *user;
   int row;
   
   d_assert(upload != NULL);
   
   remove_pending_upload(upload);
   
   TIMEOUT_REMOVE(upload->accept_timeout);
   
   if (!setup_transfer(upload)) {
      upload_close(upload, TRANSFER_ERROR);
      
      return;
   }

   /* report to all servers */
   FORSTABS(napster_send(stab->ci->sock, NAPSTER_UPLOAD_START, NULL));
/*   napster_send(upload->st->ci->sock, NAPSTER_UPLOAD_START, NULL);*/
   
   hook_text_insert(upload->st, CONSOLE, SYSTEM_P, "accept_upload", "%s\4%s",
		    upload->user, upload->trunc_file);
   
   d_msprintf(&user, "[%i] %s", upload->st->pn + 1, upload->user);
   
   row = gnapster_clist_append(gmain->ut->clist, gmain->ut->vbox,
			       upload, upload->trunc_file,
			       napster_conn_as_str(upload->connection),
			       user, "0.00M [%0] @ 0.0k/s", NULL);
   
   d_free(user);
   
   set_clist_server(gmain->ut->clist, upload->st, row);
   
   upload->calls = 5;
   
   upload->ttype = UPLOAD;
   
   transfer_callbacks(upload);
}

void start_transfer(Transfer *t) {
   d_assert(t != NULL);
   
   if (t->ttype == DOWNLOAD)
     start_download(t);
   else
     start_upload(t);
}


/* LIST SUBMITTION */


void finish_submit(STab *stab, Connection *c) {
   shared_free(stab);
   
   if (c) {
      close_accept(c);
      return;
   }
   
   if (connected(stab->ci))
     autojoin_handle(stab);
   
   INPUT_REMOVE(stab->ci->upload_input);
}

GList *find_pending_dir(GList **list) {
   GList *ptr, *child, *cptr, *tmp;

   for(ptr=(*list); ptr; ) {
      child = ptr->data;
      if (!child)
	break;
      
      for(cptr=child->next; cptr; cptr=cptr->next) {
	 if (cptr->data)
	   return child;
      }
      
      /* the above child had nothing valuable in it, lets
       * destroy it */
      
      tmp = ptr->next;
      
      *list = g_list_remove_link(*list, ptr);
      
      /* get rid of the dir name */
      d_free(child->data);
      
      /* destroy the g_list */
      g_list_free(ptr->data);
      
      g_list_free_1(ptr);
      
      ptr = tmp;
   }
   
   return NULL;
}

char *construct_packet(STab *stab, Connection *c) {
   ShareData *shr;
   GList *ptr;
   char *p;
   
   if (!stab->ut->shared)
     return NULL;

   for(ptr=stab->ut->shared; ptr; ptr=ptr->next) {
      shr = ptr->data;
      if (!shr)
	continue;
      
      d_msprintf(&p, "\"%s\" %s %lu %hu %hu %lu", shr->filename,
		 shr->checksum, shr->filesize, shr->bitrate, shr->frequency,
		 shr->time);
      
      stab->ut->shared = g_list_remove(stab->ut->shared, shr);

      j_free(SHARE, shr);
      
      return p;
   }
   
   return NULL;
}

void handle_submit(void *data, int source, GdkInputCondition cond) {
   STab *stab;
   char *packet;

   d_assert(data != NULL);
   
   stab = data;

   /* if c is NULL, we assume we will submit to the server */
   packet = construct_packet(stab, NULL);
   
   if (!packet) {
      /* no packet was found, that means we've sent it all */
      finish_submit(stab, NULL);
      return;
   }

   napster_send(stab->ci->sock, CMDS_ADDFILE, "%s", packet);
   
/*   d_free(type);*/
   d_free(packet);
}


/* INCOMING CONNECTIONS */


void push_transfer_info(Transfer *t, char *nick, char *file, unsigned long int *size) {
   d_assert(t != NULL);
   
   t->user = d_strdup(nick);
   t->file = d_strdup(file);
   
   t->trunc_file = trunc_file(t->file);
   
   if (size) {
      if (t->ttype == UPLOAD) /* resume offset */
	t->size = *size;
      else /* total size */
	t->total = *size;
   }
}

int accept_complete(Transfer *t) {
   STab *stab;
   Transfer *real;
   char *dup, *ptr, *nick, *file;
   unsigned long int size;
   
   d_assert_return(t != NULL, 0);
   
   dup = d_strdup(t->header);
   ptr = dup;
   
   nick = next_arg(ptr, &ptr);
   file = next_arg(ptr, &ptr);
   convert(next_arg(ptr, &ptr), "%lu", &size);
   
   NA_ERR_HANDLE(0);
   
   if (t->ttype == DOWNLOAD) {
      /* connect in to the download attached to the clist */
      real = real_download_guess(&stab, nick, file, NULL);
      if (!real) /* they're probably trying to exploit you ;) */
	return 0;
      
      real->sock = t->sock;
      real->ttype = t->ttype;
      real->conn_type = t->conn_type;
      
      TIMEOUT_REMOVE(t->accept_timeout);
      
      /* just destroy the old transfer, we really only needed it
       * as a scratch pad */
      j_free(TRANSFER, t);
      
      t = real;
   }
   
   if (t->ttype == UPLOAD) {
      /* ok, we need to know the STab that this is coming from... */
      real_upload_guess(&stab, nick, file);
      if (!stab)
	return 0;
   }
   
   t->size = 0;
   
   d_free(t->user);
   d_free(t->file);
   
   t->st = stab;
   push_transfer_info(t, nick, file, &size);
   
   d_free(dup);
   
   start_transfer(t);
   
   return 1;
}

void close_accept(Connection *c) {
   d_assert(c != NULL);
   
   INPUT_REMOVE(c->tag);
   TIMEOUT_REMOVE(c->timeout);

   CLOSE(c->sock);
   
   j_free(CONNECTION, c);
}

void close_transfer(Transfer *t) {
   d_assert(t != NULL);
   
   CLOSE(t->sock);
   INPUT_REMOVE(t->sock_input);
   
   TIMEOUT_REMOVE(t->accept_timeout);
   j_free(TRANSFER, t);
}

void setup_accept_browse(Connection *c, char *buf) {
/*   int push = 0;
   
   if (!strncmp(buf, "GETLIST", 7)) {
      push = 7;
      
      if (((STab *)ttbl->data)->ut->shared) {
	 close_accept(c);
	 return;
      }

      n_send(c->sock, "%s\n", get_account(NULL));
      
      shared_list_handle(ttbl->data, c, user_info.download_dir, SUBMIT_LIST);
   } else if (!strncmp(buf, "SENDLIST", 8)) {
      push = 8;
      
      c->tag = INPUT_ADD(c->sock, GDK_INPUT_READ,
			     browse_read_nick, c);
   }
   
   if (push)
     recv(c->sock, buf, push, 0);
   else
     close_accept(c);*/
}

void setup_accept_transfer(Connection *c, char *buf) {
   Transfer *t;
   int push;
   
   t = d_new(TRANSFER);
   transfer_init(t);
   
   t->conn_type = ACCEPT;
   
   push = 0;

   if (!strncmp(buf, "GET", 3)) {
      push = 3;
      t->ttype = UPLOAD;
   } else if (!strncmp(buf, "SEND", 4)) {
      push = 4;
      t->ttype = DOWNLOAD;
   }
   
   if (!push) {
      close_accept(c);
      return;
   }
   
   recv(c->sock, buf, push, 0);
   
   t->sock = c->sock;
   
   j_free(CONNECTION, c);
   
   t->sock_input = INPUT_ADD(t->sock, GDK_INPUT_READ, 
			     transfer_header, t);
}

void handle_accept_header(void *data, int source, GdkInputCondition cond) {
   Connection *c;
   char buf[10];
   int n;
   
   d_assert(data != NULL);
   
   c = data;
   
   n = recv(source, buf, sizeof(buf) - 1, MSG_PEEK);
   if (n <= 0) {
      close_accept(c);
      return;
   }
   
   buf[n] = 0;
   
   INPUT_REMOVE(c->tag);
   TIMEOUT_REMOVE(c->timeout);
   
   if (!strncmp(buf, "GETLIST", 7) || !strncmp(buf, "SENDLIST", 8)) {
      close_accept(c);
/*      setup_accept_browse(c, buf);*/
   } else if (!strncmp(buf, "GET", 3) || !strncmp(buf, "SEND", 4)) {
      setup_accept_transfer(c, buf);
   }
}

void transfer_header(void *data, int source, GdkInputCondition cond) {
   Transfer *t;
   char buf[1024];
   int n;
   
   d_assert(data != NULL);
   
   t = data;

   n = recv(source, buf, sizeof(buf) - 1, 0);
   
   if (n <= 0) {
      close_transfer(t);
      return;
   }
   
   buf[n] = 0;
   
   t->header = d_strdup(buf);
   
   INPUT_REMOVE(t->sock_input);
   
   accept_complete(t);
}

void handle_connection(void *data, int source, GdkInputCondition cond) {
   Connection *c;
   struct sockaddr_in from;
   int len;
   
   c = d_new(CONNECTION);
   
   len = sizeof(from);
   
   c->sock = 
     accept(gmain->bind_sock, (struct sockaddr *)&from, &len);
   
   if (c->sock < 0) {
      j_error("accept", NULL);
      j_free(CONNECTION, c);
      
      return;
   }
   
   set_blocking(c->sock, 0);

   c->timeout = 
     TIMEOUT_ADD(30000, accept_timeout, c);
   
   n_send(c->sock, "1");
   
   c->tag = INPUT_ADD(c->sock, GDK_INPUT_READ,
		      handle_accept_header, c);
}

void conn_outgoing_header(void *data, int source, GdkInputCondition cond) {
   Transfer *t;
   int n;
   char c;
   
   d_assert(data != NULL);
   
   t = data;
   
   INPUT_REMOVE(t->sock_input);
   
   n = recv(source, &c, 1, 0);
   
   if (n != 1 || c != '1') {
      download_close(t, TRANSFER_ERROR);
      
      return;
   }

   /* this handles errors as well */
   start_transfer(t);
}

void conn_outgoing(void *data, int source, GdkInputCondition cond) {
   Transfer *t;
   
   d_assert(data != NULL);
   
   t = data;
   
   INPUT_REMOVE(t->sock_input);
   
   if (sock_err(source)) {
      napster_send(t->st->ci->sock, NAPSTER_PORT_ERROR, "%s",
		   t->user);
      
      hook_text_insert(t->st, CONSOLE, SYSTEM_N, "general_error", "%s %s",
		       "error connecting to user", t->user);

      transfer_close(t, TRANSFER_ERROR);
      
      return;
   }

   set_blocking(source, 1);
   
   t->conn_type = CONNECT;
   
   /* this will further setup the transfer.... */
   t->sock_input = INPUT_ADD(source, GDK_INPUT_READ,
			     conn_outgoing_header, t);
}


/* MAIN READ FUNCTION */


void network_read_cb(void *data, int source, GdkInputCondition cond) {
   int n;
   STab *stab;
   ConnInfo *ci;
   NBRead *nb;
   Packet *p;
   
   d_assert(data != NULL);
   
   stab = data;
   ci = stab->ci;
   
   nb = nb_active(ci->sock + 1);
   
   if (!nb->flag)
     nb->flag++;
   
   if (nb->flag == 1) {
      /* read the 4 byte header */
      n = nb_read(nb, source, 4, 0);
      if (n <= 0) {
/*	 perror("recv1");*/
	 disconnect(stab);
	 return;
      }

      if (nb->len < 4)
	return;

      nb_terminate(nb);
      
      p = d_malloc(sizeof(Packet));
      
      memcpy(p, nb->data, 4);
      p->len = BSWAP16(p->len);
      p->cmd = BSWAP16(p->cmd);
      
      /* one packet will always be held in memory, the last one seen */
      d_free(ci->cp);
      ci->cp = p;
      
      nb->flag++;
      if (p->len == 0)
	nb->flag++;
   } else if (nb->flag == 2) {
      n = nb_read(nb, source, ci->cp->len, 0);
      if (n <= 0) {
/*	 perror("recv2");*/
	 disconnect(stab);
	 return;
      }
      
      if (nb->len >= ci->cp->len)
	nb->flag++;
   }
   
   if (nb->flag == 3) {
      char buf[nb->len + 1];
      
      nb_terminate(nb);
      
      if (!connected(ci)) {
	 ci->state &= ~CONNECTING_MASK;
	 ci->state |= CONNECTED_MASK;
	 
	 /* this will fail if it has already been done, so we're safe */
	 napster_bind(gmain->port);
      }
      
      /* it is possible that napster_handle_data may want to
       * destroy all NB hooks with disconnect... so we cant assume anything
       * after this point */
      g_snprintf(buf, sizeof(buf), "%s", nb->data);
      
      nb_finish(nb);
      
      napster_handle_data(stab, ci->cp->cmd, buf);
   }
}


/* MAIN NETWORK CONNECTION */


void read_best_host(void *data, int source, GdkInputCondition cond) {
   int n;
   STab *stab;
   char buf[RW_BUFFER], *ptr, *ip;
   unsigned short port;
   
   stab = data;

   n = recv(source, buf, sizeof(buf) - 1, 0);
   if (n <= 0) {
      disconnect(stab);
      return;
   }
   
   buf[n] = 0;
   ptr = buf;
   
   ip = next_arg_full(ptr, &ptr, ':');
   convert(next_arg(ptr, &ptr), "%hu", &port);
   
   if (na_err) {
      disconnect(stab);
      return;
   }
   
   NA_RESET();
   
   INPUT_REMOVE(stab->ci->sock_input);
   CLOSE(source);

   /* napster's stupid way of telling you the servers are full */
   if (!strcmp(ip, "127.0.0.1")) {
      hook_text_insert(stab, CONSOLE, SYSTEM, "general_error", "%s",
		       "Napster query server reported all servers were busy");
      
      disconnect(stab);
      
      return;
   }
   
   d_free(stab->ci->ip);
   stab->ci->ip = d_strdup(ip);
   stab->ci->port = port;
   
   update_status(1, stab, _("Best host found, connecting..."), -1);
   napster_connect(stab, stab->ci->ip, stab->ci->port);
}

void conn_main(void *data, int source, GdkInputCondition cond) {
   STab *stab;
   
   d_assert(data != NULL);
   
   stab = data;
   
   INPUT_REMOVE(stab->ci->sock_input);
   
   if (sock_err(source)) {
      hook_text_insert(stab, CONSOLE, SYSTEM, "general_error", "%s",
		       "failure connecting to napster server");
      
      disconnect(stab);
      
      return;
   }
   
   set_blocking(source, 1);
   
   stab->ci->sock_input = INPUT_ADD(source, GDK_INPUT_READ,
				    read_best_host, stab);
}

void login(STab *stab, int source) {
   char *str;
   
   /* your guess is as good as mine */
/*   napster_send(source, NAPSTER_LOGIN_SEQUENCE, "1");*/
   
/*   if (stab->ci->account->new_user) {
      napster_send(source, NAPSTER_NEW_USER, stab->ci->account->user);
      user_info.new_user = 0;
   } else {*/
      napster_send(source, NAPSTER_LOGIN, "%s %s %i \"%s\" %i",
		   stab->ci->account->user, stab->ci->account->pass, 
		   gmain->port, NAP_VERSION, user_info.connection);
      
      d_msprintf(&str, "Connected (%s:%i)...awaiting login reply...",
		 stab->ci->ip, stab->ci->port);
      
      update_status(1, stab, str, -1);
      
      d_free(str);
/*   }*/
}

void conn_final(void *data, int source, GdkInputCondition cond) {
   STab *stab;
   
   d_assert(data != NULL);
   
   stab = data;

   INPUT_REMOVE(stab->ci->sock_input);
   
   if (sock_err(source)) {
      hook_text_insert(stab, CONSOLE, SYSTEM, "general_error", "%s",
		       "failure connecting to napster server");
      
      disconnect(stab);
      
      return;
   }
   
   set_blocking(source, 1);

   /* send the new user numeric just in case...if it comes back that
    * this user is already known, attempt a login */
   napster_send(source, NAPSTER_NEW_USER, "%s", stab->ci->account->user);
/*   login_check(stab, source);*/
/*   login(stab, source);*/

   /* start reading packets */
   stab->ci->sock_input = INPUT_ADD(source, GDK_INPUT_READ,
				    network_read_cb, stab);
}
