/* parse_ext.c - modular sharing system */

#include "gnapster.h"

#include "upload.h"
#include "parse_ext.h"
#include "parse_mp3.h"
#include "parse_vorbis.h"
#include "parse_avi.h"
#include "parse_mpg.h"

GList *parse_exts;

void register_exts() {
   mp3_register();
/* #ifdef HAVE_VORBIS */
   vorbis_register();
/* #endif *//* HAVE_VORBIS */
   avi_register();
   mpg_register();
}

void register_ext_funcs(char *name, char *mimetype, void *parse, void *ext) {
   ParseExt *pext;
   
   pext = d_malloc(sizeof(ParseExt));
   
   pext->name = d_strdup(name);
   pext->mimetype = d_strdup(mimetype);
   
   pext->parse = parse;
   pext->ext = ext;
   
   parse_exts = g_list_append(parse_exts, pext);
}

char *get_mime_packet(char *packet) {
   char *type, *pkt, *file, *md5;
   unsigned long size;
   
   file = next_arg(packet, &packet);
   md5 = next_arg(packet, &packet);
   convert(next_arg(packet, &packet), "%lu", &size);
   
   NA_ERR_HANDLE(NULL);
   
   type = get_mime_type(file);
   if (!type)
     return NULL;
   
   d_msprintf(&pkt, "\"%s\" %lu %s %s\n",
	      file, size, md5, type);
   
   return pkt;
}

char *get_mime_type(char *file) {
   GList *iter_ptr;
   ParseExt *pext;
   char *ext;
   
   ext = get_ext(file);

   ITER_LIST(parse_exts) {
      LIST_DATA(pext);
      
      if (!(pext->ext(ext)))
	continue;

      return pext->mimetype;
   }
   
   return NULL;
}

char *get_ext(char *file) {
   char *ext;
   
   if ((ext = strrchr(file, '.')))
     ext++;
   
   return ext;
}

void write_share(int writefd, ShareData *shr) {
   unsigned short x;
   char *pbuf;

   d_msprintf(&pbuf, "%li \"%s\" %s-%lu %lu %hu %hu %li", 
	      shr->mtime, shr->filename, shr->checksum, shr->filesize,
	      shr->filesize, shr->bitrate, shr->frequency, shr->time);

   x = strlen(pbuf);
   
   write(writefd, &x, 2);
   write(writefd, pbuf, x);

   d_free(pbuf);
}

void process_file(STab *stab, int writefd, char *path) {
   int fd;
   GList *ptr;
   ParseExt *dptr;
   ShareData *shr;
   char *ext;
   
   /* if local mtime <= written mtime, just send back the data we've
    * already processed */
   if ((shr = find_share_mtime(stab, path))) {
      write_share(writefd, shr);
      
      share_remove(stab, shr);
      j_free(SHARE, shr);
      
      return;
   }

   ext = get_ext(path);
   if (!ext)
     return;

   for(ptr=parse_exts; ptr; ptr=ptr->next) {
      dptr = ptr->data;
      
      if (!(dptr->ext(ext)))
	continue;

      fd = open(path, O_RDONLY);      
      if (fd < 0)
	break;
      
      shr = dptr->parse(path, fd);
      
      close(fd);
      
      if (!shr)
	return;

      write_share(writefd, shr);
      
      j_free(SHARE, shr);
      
      return;
   }

   /* ERROR */
}

void recursive_parse_start(STab *stab, int writefd, char *root) {
   char *ul, *dup, *ptr;
   
   dup = d_strdup(root);
   ptr = dup;
   
   while((ul = next_arg_full(ptr, &ptr, '\4')))
     recursive_parse(stab, writefd, ul);
   
   d_free(dup);
}

void recursive_parse(STab *stab, int writefd, char *dirpath) {
   DIR *dir;
   struct stat st;
   struct dirent *dptr;
   char *newpath = NULL;
   
   dir = opendir(dirpath);
   if (!dir)
     return;

   for(;;) {
      /* do the cleanup here as not to dirty the code below */
      d_free(newpath);
      newpath = NULL;
      
      dptr = readdir(dir);
      if (!dptr)
	break;
      
      if (!strcmp(dptr->d_name, ".") || !strcmp(dptr->d_name, ".."))
	continue;
      
      /* build the new file/directory path */
      d_msprintf(&newpath, "%s/%s", dirpath, dptr->d_name);
      
      if (stat(newpath, &st) < 0)
	continue;
      
      if (S_ISLNK(st.st_mode)) {
	 char link[1024 + 1];
	 struct stat lst;
	 
	 if (readlink(newpath, link, 1024) < 0)
	   continue;
	 
	 if (stat(link, &lst) < 0)
	   continue;
      }
      
      if (S_ISDIR(st.st_mode))
	recursive_parse(stab, writefd, newpath);
      else
	process_file(stab, writefd, newpath);
   }
   
   closedir(dir);
}
