/* parse_mp3.c - Processing of MP3 header data */

/* FIXME: VBR support is not fully incorporated! */

#include "gnapster.h"

#include "parse_mp3.h"
#include "parse_ext.h"

unsigned short mp3_bitrates[2][3][15] = {
   {
      { 0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256 },
      { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160 },
      { 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160 }
   },
   {
      { 0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448 },
      { 0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384 },
      { 0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320 }
   },
};

unsigned short mp3_frequencies[2][2][3] = {
   {
      { 11025, 12000, 8000 },
      { 0, 0, 0},
   },
   {
      { 22050, 24000, 16000 },
      { 44100, 48000, 32000 }
   }
};

#define VBR_FRAME_FLAG 0x0001

void mp3_register() {
   REGISTER_EXT("mp3", NULL, parse_mp3, mp3_ext);
}

int mp3_ext(char *ext) {
   /* 1 on match, 0 on failure */
   return (strcasecmp(ext, "mp3") == 0);
}

int read_byte(int fd, guint32 *hdr) {
   unsigned char c;
   int n;
   
   n = read(fd, &c, 1);
   if (n <= 0)
     return n;

   *hdr = (*hdr << 8) | c;
   
   return n;
}

/* this function submitted by: mblomenkamp@web.de */
int vbr_frame_count(unsigned char *hdr) {
   int h_id, h_mode, h_flags;
   
   h_id = (hdr[1] >> 3) & 0x01;
   h_mode = (hdr[3] >> 6) & 0x03;
   
   if (h_id)
     hdr += (h_mode == 3) ? 17 + 4 : 32 + 4;
   else
     hdr += (h_mode == 3) ? 9 + 4 : 17 + 4;
   
   if (strncmp(hdr, "Xing", 4))
     return 0;
   
   hdr += 4;
   
   h_flags = PACK(hdr);
   hdr += 4;
   
   return (h_flags & VBR_FRAME_FLAG) ? PACK(hdr) : 0;
}

int find_mp3_header(int fd, guint32 *hdr) {
   int offs, n;
   
   offs = 0;
   
   while((n = read_byte(fd, hdr))) {
      if ((((*hdr >> 16) & 0xffe0) == 0xffe0) &&
	  (((*hdr >> 16) & 0x00ff) != 0x00ff))
	return (offs - 3);
      
      /* sanity check */
      if (offs > 300000)
	break;
      
      offs += n;
   }
   
   return -1;
}

void get_mp3_header(guint32 hdr, MP3Header *h) {
/*      get_bits(&hdr, 11);
      hd->index = get_bits(&hdr, 1);
      hd->id = get_bits(&hdr, 1);
      hd->layer = get_bits(&hdr, 2);
      hd->protection_bit = get_bits(&hdr, 1);
      hd->bitrate_index = get_bits(&hdr, 4);
      hd->frequency = get_bits(&hdr, 2);
      hd->padding_bit = get_bits(&hdr, 1);
      hd->private_bit = get_bits(&hdr, 1);
      hd->mode = get_bits(&hdr, 2);
      hd->mode_ext = get_bits(&hdr, 2);
      if (!hd->mode)
          hd->mode_ext = 0;
      hd->copyright = get_bits(&hdr, 1);
      hd->original = get_bits(&hdr, 1);
      hd->emphasis = get_bits(&hdr, 2);*/
   
   memcpy(h, &hdr, sizeof(MP3Header));
   
   if (!h->mode)
     h->mode_ext = 0;
}

ShareData *process_mp3_header(int fd, guint32 hdr) {
   ShareData *shr;
   MP3Header *h;
   struct stat st;
   
   h = d_malloc(sizeof(MP3Header));
   
   get_mp3_header(hdr, h);
   
   if (h->id > 1 || h->layer > 2 || h->bitrate_index > 14)
     return d_free(h);
   
   shr = d_new(SHARE);
   
   fstat(fd, &st);
   
   shr->filesize = st.st_size;
   shr->mtime = st.st_mtime;
   
   shr->bitrate = mp3_bitrates[h->id][3 - h->layer][h->bitrate_index];
   shr->frequency = mp3_frequencies[h->index][h->id][h->frequency];
   
   if (shr->frequency > 0) {
      int framesize, total;
      
      framesize = (((h->id ? 144000 : 72000) * shr->bitrate) / shr->frequency);
      total = (shr->filesize / (framesize + 1)) - 1;
      shr->time = (time_t) (total * (h->id ? 1152 : 576) / shr->frequency);
   } else
     shr->time = 0;
   
   d_free(h);
   
   return shr;
}

char *get_checksum(int fd, unsigned long mapsize) {
   unsigned char digest[16];
   md5_state_t state;
   char buffer[2048 + 1];
   struct stat st;
   unsigned long size = 300000;
   int di = 0;
   int rc;
   char *m;
   
   if (fstat(fd, &st) < 0)
     return d_strdup("");
   
   *buffer = 0;
   md5_init(&state);
   
   if (!mapsize) {
      if (st.st_size < size)
	size = st.st_size;
   } else if (st.st_size < mapsize)
     size = st.st_size;
   else 
     size = mapsize;
   
   if ((m = mmap((void *)0, size, PROT_READ, MAP_PRIVATE, fd, 0)) != MAP_FAILED) {
      md5_append(&state, (unsigned char *)m, size);
      md5_finish(digest, &state);
      munmap(m, size);
      memset(buffer, 0, 200);
      
      for(di=0, rc=0; di<16; ++di, rc+=2)
	sprintf(buffer + rc, "%02x", digest[di]);
      
      sprintf(buffer + rc, "-%li", st.st_size);
   }
   
   return d_strdup(buffer);
}

ShareData *parse_mp3(char *file, int fd) {
   ShareData *shr;
   unsigned long offs;
   guint32 hdr;
   
   hdr = 0;
   
   offs = find_mp3_header(fd, &hdr);
   if (offs < 0)
     return NULL;
   
   shr = process_mp3_header(fd, hdr);
   if (!shr)
     return NULL;

   shr->filename = d_strdup(file);
   shr->checksum = d_strdup("00000000000000000000000000000000");
/*   shr->checksum = get_checksum(fd, 300000);*/

   return shr;
}
