#include "gnapster.h"

#define HISTORY_MAX 10

extern UserInfo user_info;
extern GnapsterMain *gmain;

void submit_results(STab *stab) {
   GList *m, *results;
   char *search;
   int pos;
   
   m = g_list_find_custom(gmain->sresults, stab->lsearch, search_find);
   
   /* create it */
   if (!m)
     m = search_new(stab->lsearch);
   
   /* push the new results back in */
   pos = g_list_position(gmain->sresults, m);
   
   results = g_list_nth_data(gmain->sdata, pos);
   if (results)
     LIST_REMOVE(gmain->sdata, results);
   
   g_list_foreach(stab->sresults, append_sresult, &results);
   LIST_INSERT(gmain->sdata, results, pos);

   search = GET_DATA(gmain->st->history, "lsearch");
   if (!strcmp(search, stab->lsearch))
     search_sort_results(gmain->st->clist);
   
   g_list_free(stab->sresults);
   stab->sresults = NULL;
}

void search_delete(int idex) {
   char *s;
   GList *r;
   GtkWidget *w;
   
   printf("deleting %i\n", idex);
   
   s = g_list_nth_data(gmain->sresults, idex);
   LIST_REMOVE(gmain->sresults, s);
   d_free(s);
   
   r = g_list_nth_data(gmain->sdata, idex);
   LIST_REMOVE(gmain->sdata, r);
   g_list_foreach(r, searchtab_free_result, NULL);
   
   w = g_list_nth_data(GTK_MENU_SHELL(gmain->st->hist_menu)->children, idex);
   gtk_widget_destroy(w);
}

GList *search_new(char *search) {
   GList *ret;
   char *dup;
   GtkWidget *mitem;
   
   /* do not add twice */
   if ((ret = g_list_find_custom(gmain->sresults, search, search_find))) {
      /* previously found, add the cache'd results */
      searchtab_load_history(NULL, search);
      
      return ret;
   }
   
   if (LIST_LENGTH(gmain->sresults) >= HISTORY_MAX)
     search_delete(0);
   
   dup = d_strdup(search);
   
   LIST_APPEND(gmain->sresults, dup);
   ret = g_list_last(gmain->sresults);
   
   /* insert into the history menu */
   mitem = gtk_menu_item_new_with_label(search);
   gtk_signal_connect(GTK_OBJECT(mitem), "activate",
		      GTK_SIGNAL_FUNC(searchtab_load_history), dup);
   gtk_menu_append(GTK_MENU(gmain->st->hist_menu), mitem);
   gtk_menu_item_activate(GTK_MENU_ITEM(mitem));
   gtk_widget_show(mitem);

   if (!gtk_option_menu_get_menu(GTK_OPTION_MENU(gmain->st->history)))
     gtk_option_menu_set_menu(GTK_OPTION_MENU(gmain->st->history),
			      gmain->st->hist_menu);
   
   gtk_option_menu_set_history(GTK_OPTION_MENU(gmain->st->history),
			       g_list_length(gmain->sresults) - 1);
   
   return ret;
}
   
void append_sresult(void *data, void *ud) {
   GList **r;
   Search *s;
   char *lsearch;
   
   d_assert(data != NULL);
   d_assert(ud != NULL);

   s = data;
   r = ud;
   
   /* don't add a duplicate search (ip and filename match) */
   if (g_list_find_custom(*r, data, search_match))
     return;
   
   /* add the data to the clist */
   lsearch = GET_DATA(gmain->st->history, "lsearch");
   if (!strcmp(lsearch, s->st->lsearch))
     insert_search_result(gmain->st, search_copy(s), 0);
   
   LIST_APPEND(*r, data);
}

int search_find(const void *a, const void *b) {
   char *s1, *s2;
   
   d_assert_return(a != NULL, -1);
   d_assert_return(b != NULL, -1);
   
   s1 = (char *)a;
   s2 = (char *)b;
   
   if (!strcmp(s1, s2))
     return 0;
   
   return -1;
}

/* match on a unique search result */
int search_match(const void *a, const void *b) {
   Search *s1, *s2;
   
   d_assert_return(a != NULL, -1);
   d_assert_return(b != NULL, -1);
   
   s1 = (Search *)a;
   s2 = (Search *)b;
   
   if (s1->ip == s2->ip &&
       !strcmp(s1->file, s2->file))
     return 0;
   
   return -1;
}

Search *search_copy(Search *s) {
   Search *ret;
   
   ret = d_new(N_SEARCH);
   
   memcpy(ret, s, sizeof(Search));
   
   /* duplicate strings */
   ret->file = d_strdup(s->file);
   ret->trunc_file = trunc_file(ret->file);
   ret->id = d_strdup(s->file);
   ret->user = d_strdup(s->user);
   ret->resume_path = d_strdup(s->resume_path);
   
   return ret;
}

void insert_search_result(SearchTab *st, Search *s, int searching) {
   GdkPixmap *pixmap = NULL;
   GdkBitmap *bitmap = NULL;
   Transfer *t;
   char *user, *size, *tme, *kbps;
   int row;

   /* append to the search history */
   if (searching) {
      LIST_APPEND(s->st->sresults, s);
      return;
   }
   
   d_msprintf(&user, "[%i] %s", s->st->pn + 1, s->user);
   d_msprintf(&size, "%.02fM", ((float)s->size / 1024) / 1024);
   d_msprintf(&tme, "%i:%02i", s->num / 60, s->num % 60);
   d_msprintf(&kbps, "%i", s->kbps);
   
   /* this function is hacked to NOT free N_SEARCH's! */
   row = gnapster_clist_append(st->clist, st->vbox, s,
			       s->trunc_file,
			       napster_conn_as_str(s->connection),
			       user, size, tme, kbps, NULL);
   
   /* set the server shade */
   set_clist_server(st->clist, s->st, row);
   
   d_free(user);
   d_free(size);
   d_free(tme);
   d_free(kbps);

   get_conn_pixmap(s->connection, &pixmap, &bitmap);
   
   gtk_clist_set_pixtext(GTK_CLIST(st->clist), row, 0,
			 s->trunc_file, 3, pixmap, bitmap);
   
   /* find out if this result is capable of being resumed... */
   t = d_new(TRANSFER);
   transfer_init(t);
   
   t->size = s->size;
   t->id = d_strdup(s->id);
   
   check_resume(t);
   
   if (t->resume_path)
     set_green_style(st->clist, row, 3);
   
   j_free(TRANSFER, t);  
}

void end_search(STab *stab, char *msg) {
   char *str = NULL;

   /* push these results to the main search tab for history */
   stab->lsearch = LIST_POP(stab->ls);
   submit_results(stab);
   d_free(stab->lsearch);
   
   if (msg)
     d_msprintf(&str, "%s", msg);

   update_status(1, stab, NULL, 0);

/*   gtk_entry_set_text(GTK_ENTRY(gmain->st->gtk_entry), "");
   gtk_entry_set_text(GTK_ENTRY(gmain->st->gtk_not_entry), "");*/

/*   gtk_entry_set_editable(GTK_ENTRY(gmain->st->gtk_entry), 1);*/

   /* sorting methods for the search clist */
   search_sort_results(gmain->st->clist);
/*   gtk_clist_thaw(GTK_CLIST(gmain->st->clist));*/
   
   if (msg) {
      if (!CLIST_LENGTH(gmain->st->clist))
	gnapster_clist_append(gmain->st->clist, gmain->st->vbox, NULL,
			      str, "", "", "", "", "", NULL);

      d_free(str);
   }
   
   stab->ci->state &= ~SEARCHING_MASK;
}

int dispatch_search(STab *stab, char *s) {
   if (!connected(stab->ci) || searching(stab->ci))
     return 0;
   
   gtk_entry_set_text(GTK_ENTRY(gmain->st->gtk_entry), "");
   gtk_entry_set_text(GTK_ENTRY(gmain->st->gtk_not_entry), "");
/*   stab->search++;*/
   
   update_status(1, stab, _("Searching..."), 1);
   
   stab->ci->state |= SEARCHING_MASK;
   
   napster_send(stab->ci->sock, NAPSTER_SEARCH, s);
   
/*   gtk_clist_clear(GTK_CLIST(gmain->st->clist));*/
/*   gtk_clist_freeze(GTK_CLIST(gmain->st->clist));
   gtk_entry_set_editable(GTK_ENTRY(gmain->st->gtk_entry), 0);*/
   
   return 1;
}

void search_activate_cb() {
   char *cmp, *ptr;      
   char *not_string, *ostr, *string, *ext;
   char extra_optns[1024], temp[1024];
   char *bps_table[] = { "96", "112", "128", "160", "192", NULL };
   
   ostr = string = gtk_entry_get_text(GTK_ENTRY(gmain->st->gtk_entry));
   if (!string || !(*string))
     return;

   ostr = d_strdup(ostr);
   string = d_strdup(string);
   
   not_string = gtk_entry_get_text(GTK_ENTRY(gmain->st->gtk_not_entry));
   if (!not_string)
     return;
   
   not_string = d_strdup(not_string);
   
   ext = gtk_entry_get_text(GTK_ENTRY(gmain->st->ext_entry));
   if (!ext)
     return;
   
   if (*(ext) && strcasecmp(ext, "mp3"))
     d_strexp(&string, "%s .%s", string, ext);

   memset(extra_optns, 0, sizeof(extra_optns));
   
   if (user_info.conf[LINE_SPEED_SWITCH]) {
      cmp = d_strdup(listint_to_str(user_info.conf[LINE_SPEED_CMP], 0));

      for(ptr=cmp; *ptr; ptr++)
	*ptr = toupper(*ptr);
      
      g_snprintf(temp, sizeof(temp), " LINESPEED \"%s\" %i", cmp, user_info.conf[LINE_SPEED]);
      strcat(extra_optns, temp);
      d_free(cmp);
   }
   
   if (user_info.conf[BITRATE_SWITCH]) {
      char *bit;
      
      cmp = d_strdup(listint_to_str(user_info.conf[BITRATE_CMP], 0));
      
      for(ptr=cmp; *ptr; ptr++)
	*ptr = toupper(*ptr);

      bit = bps_table[user_info.conf[BITRATE]];
      
      g_snprintf(temp, sizeof(temp), " BITRATE \"%s\" \"%s\"", cmp, bit);
      
      strcat(extra_optns, temp);
      
      d_free(cmp);
   }
   
/*   if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(gmain->st->local_only)))
     strcat(extra_optns, " LOCAL_ONLY");*/
   
   if (*not_string) {
      char buf[1024];

      g_snprintf(buf, sizeof(buf), " FILENAME EXCLUDES \"%s\"", not_string);
      strcat(extra_optns, buf);
   }

   for(ptr=string; *ptr; ptr++)
     *ptr = tolower(*ptr);

   d_strexp(&string, "FILENAME CONTAINS \"%s\" MAX_RESULTS 150%s",
	    string, extra_optns);
   d_free(not_string);
   
   search_new(ostr);
   
   FORSTABS(if (dispatch_search(stab, string)) {
               LIST_PREPEND(stab->ls, d_strdup(ostr));
	    });
   
   d_free(ostr);
   
/*   if (user_info.conf[BROADCAST]) {
      FORSTABS(dispatch_search(stab, string));
   } else
     dispatch_search(st, string);*/

   d_free(string);
}

