#include <unistd.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <dirent.h>

#include <ctype.h>

#include "gnapster.h"
#include "upload.h"
#include "commands.h"
#include "chan.h"

#include "themes.h"

extern GList *hooks;
extern PropDlg prop_dlg;
extern UserInfo user_info;

void themes_select_row(GtkWidget *w, int row, int col, GdkEvent *ev) {
   Theme *theme;
   
   theme = gnapster_get_selection_data(w, NULL);
   if (!theme || !theme->file)
     return;
   
   gtk_entry_set_text(GTK_ENTRY(prop_dlg.curr_theme), theme->file);
}

int get_theme_data(char *file, Theme *theme) {
   FILE *f;
   char *cmd, *cmd_buf, *args, *dptr, *name, *author;
   GList *ptr, *felem;
   FileEntry *fent;
   
   if (!file || !theme)
     return -1;
   
   f = fopen(file, "r");
   if (!f)
     return -1;
   
   felem = read_file(f);
   
   fclose(f);
   
   name = author = NULL;
   
   for(ptr=felem; ptr; ptr=ptr->next) {
      fent = ptr->data;
      if (!fent)
	continue;
      
      dptr = fent->lptr;
      
      cmd = next_arg(dptr, &dptr);
      cmd_buf = next_arg(dptr, &dptr);
      args = next_arg(dptr, &dptr);
      
      NA_ERR_HANDLE_CONTINUE();

      if (!strcmp(cmd, "parse")) {
	 if (!strcmp(cmd_buf, "theme_name"))
	   name = d_strdup(args);
	 else if (!strcmp(cmd_buf, "theme_author"))
	   author = d_strdup(args);
      }
      
      if (name && author)
	break;
   }
   
   read_file_free(felem);
   
   if (!name || !author) {
      /* these are safe */
      d_free(name);
      d_free(author);
      
      return -1;
   }

   theme->file = d_strdup(file);
   
   theme->name = name;
   theme->author = author;
   
   return 1;
}

void scan_theme_dir(char *n) {
   DIR *dir;
   Theme *theme;
   struct stat st;
   char *dot, p[4096 + 1];
   struct dirent *dptr;

   dir = opendir(n);
   if (!dir)
     return;
   
   for(;;) {
      dptr = readdir(dir);
      if (!dptr)
	break;
      
      sprintf(p, "%s/%s", n, dptr->d_name);
      stat(p, &st);
      
      if (!S_ISREG(st.st_mode))
	continue;
      
      dot = strrchr(p, '.');
      if (!dot)
	continue;
      
      if (strcmp(dot, ".theme"))
	continue;
      
      theme = d_new(THEME);

      if (get_theme_data(p, theme) < 0) {
	 d_free(theme);
	 continue;
      }
      
      gnapster_clist_append(prop_dlg.themes, NULL, theme, theme->name,
			    theme->author);
   }
   
   closedir(dir);
}

void populate_themes_clist() {
   char *path;

   gtk_clist_freeze(GTK_CLIST(prop_dlg.themes));
   gtk_clist_clear(GTK_CLIST(prop_dlg.themes));

   /* scans both global and local paths */
   path = global_path("themes", NULL);
   scan_theme_dir(path);
   d_free(path);
   
   path = local_path("themes", NULL);
   scan_theme_dir(path);
   d_free(path);
   
   gtk_clist_thaw(GTK_CLIST(prop_dlg.themes));
}

GList *add_hook_ref(GList *ptr, char *cmd_ref, char *args) {
   ThemeHook *hook;
   
   hook = d_new(HOOK);
   hook->name = d_strdup(cmd_ref);
   hook->data = d_strdup(args);
   
   return g_list_append(ptr, hook);
}

char *true_color(char *data) {
   static char ctag[5] = { 0 };
   
   if (!data)
     return NULL;
   
   if (strlen(data) < 4 || !isdigit(data[3]))
     return NULL;
   
   if (!strncmp(data + 3, "0}", 2))
     g_snprintf(ctag, sizeof(ctag) - 1, "\017");
   else
     g_snprintf(ctag, sizeof(ctag) - 1, "\003%s", data + 3);
   
   if (ctag[2] == '}')
     ctag[2] = 0;
   
   ctag[3] = 0;
   
   return ctag;
}

char *lookup_hook(char *var, GList *assigns) {
   GList *ptr;
   ThemeHook *hook;
   
   for(ptr=assigns; ptr; ptr=ptr->next) {
      hook = ptr->data;
      if (!hook)
	continue;
      
      if (!strcmp(hook->name, var))
	return hook->data;
   }
   
   return NULL;
}

ThemeHook *lookup_hook_by_name(char *name, GList *hook) {
   GList *ptr;
   ThemeHook *h;
   
   for(ptr=hook; ptr; ptr=ptr->next) {
      h = ptr->data;
      if (!h)
	continue;
      
      if (!strcmp(h->name, name))
	return h;
   }
   
   return NULL;
}

void destroy_theme_hook(GList *list) {
   GList *ptr;
   
   for(ptr=list; ptr; ptr=ptr->next) {
      if (!ptr->data)
	continue;
      
      j_free(HOOK, ptr->data);
   }
   
   g_list_free(list);
}

char *hook_replace(char *h, char *text) {
   GList *rep = NULL;
   ThemeHook *hook;
   char *output, *n, *num;
   int x = 1;
   size_t size = 0;
   
   if (!h)
     return NULL;
   
   hook = lookup_hook_by_name(h, hooks);

   if (!hook)
     return NULL;

   while((n = next_arg_full(text, &text, '\4'))) {
      d_msprintf(&num, "%i", x++);
      rep = add_hook_ref(rep, num, n);
      d_free(num);
   }
   
   NA_RESET();
   
   replace(hook, rep, 0, &size);
   output = replace(hook, rep, size, NULL);
   
   destroy_theme_hook(rep);
   
   return output;
}

char *replace(ThemeHook *hook, GList *assigns, size_t s, size_t *size) {
   char *dptr, *ashook, out[64];
   char *ptr, *rtrn;
   int flag = 0, x = 0;
   
   if (!hook)
     return NULL;
   
   dptr = hook->data;
   if (!dptr)
     return NULL;
   
   ptr = dptr;
   
   rtrn = (s > 0 && !size) ? 
     d_malloc(s + 1) : NULL;
   
   if (rtrn)
     memset(rtrn, 0, s + 1);
   
   while(*ptr) {
      if (!strncmp(ptr, "${", 2))
	flag++;
      else if (*ptr == '}' && flag > 0) {
	 out[x] = 0;
	 ashook = lookup_hook(&out[2], assigns);
	 if (ashook) {
	    if (!strncmp(ashook, "${c", 3) && isdigit(ashook[3]))
	      ashook = true_color(ashook);
	    if (size)
	      *size += strlen(ashook);
	    else
	      strcat(rtrn, ashook);
	 } else {
	    out[x] = '}';
	    out[x+1] = 0;
	    if (size)
	      *size += strlen(out);
	    else
	      strcat(rtrn, out);
	 }
	 x = 0;
	 flag--;
      }

      if (flag)
	out[x++] = *ptr;
      else if (*ptr != '}') {
	 if (size)
	   *size += 1;
	 else
	   strncat(rtrn, ptr, 1);
      }
            
      ptr++;
   }
 
   return rtrn;
}

void replace_assigns(GList *hooks, GList *assigns) {
   GList *ptr;
   ThemeHook *hook;
   char *out;
   size_t size = 0;
   
   for(ptr=hooks; ptr; ptr=ptr->next) {
      hook = ptr->data;
      if (!hook)
	continue;

      replace(hook, assigns, 0, &size);
      out = replace(hook, assigns, size, NULL);
      
      d_free(hook->data);
      hook->data = out;

      size = 0;
   }
}

char *escape(char *args) {
   char *ptr, *mem;
   int no_cr;
   
   if (!args)
     return NULL;
   
   mem = d_strdup(args);
   no_cr = 0;
   
   ptr = mem;
   
   while(*ptr) {
      if (*ptr == '\\') {
	 ptr++;
	 switch(*ptr) {
	  case 'N':
	    no_cr = 1;
	    strcpy(ptr - 1, ptr + 1);
	    break;
	  case 'n':
	    *(ptr - 1) = '\r';
	    *ptr = '\n';
	    break;
	  case 't':
	    *(ptr - 1) = '\t';
	    strcpy(ptr, ptr + 1);
	    ptr--;
	    break;
	  case '\'':
	    *(ptr - 1) = '\"';
	    strcpy(ptr, ptr + 1);
	    ptr--;
	    break;
	  case '\\':
	    strcpy(ptr, ptr + 1);
	    ptr--;
	    break;
	  default:
	    break;
	 }
      }
      ptr++;
   }

   d_msprintf(&ptr, "%s%s", mem, no_cr ? "" : "\n");
   d_free(mem);
   mem = ptr;
   
   return mem;
}

void load_theme() {
   FILE *f;
   GList *ptr, *felem;
   FileEntry *fent;
   GList *assigns;
   char *theme;
   char *dptr, *cmd, *cmd_ref, *args, *esc;
   
   theme = d_config_get_string("/gnapster/Display/theme");
   if (!theme) {
      theme = global_path("themes", "default.theme");
      j_config_set_string("/gnapster/Display/theme", theme);
      j_config_sync();
   }
   
   f = fopen(theme, "r");

   if (!f)
     j_error_dialog("Fatal error loading the given theme!  If this is your first time running make sure you properly make install'd!");

   d_free(theme);

   if (!f)
     return;
   
   felem = read_file(f);
   
   fclose(f);
   
   assigns = hooks = NULL;
   
   for(ptr=felem; ptr; ptr=ptr->next) {
      fent = ptr->data;
      if (!fent)
	continue;
      
      dptr = fent->lptr;
      
      if (is_comment(dptr))
	continue;
      
      cmd = next_arg(dptr, &dptr);
      cmd_ref = next_arg(dptr, &dptr);
      args = next_arg(dptr, &dptr);
      
      NA_ERR_HANDLE_CONTINUE();
      
      esc = escape(args);

      if (!strcmp(cmd, "assign"))
	assigns = add_hook_ref(assigns, cmd_ref, esc);
      else if (!strcmp(cmd, "set") || !strcmp(cmd, "parse"))
	hooks = add_hook_ref(hooks, cmd_ref, esc);
      
      d_free(esc);
   }
   
   read_file_free(felem);
   
   replace_assigns(hooks, assigns);
   
   destroy_theme_hook(assigns);
   
   /* successful loading */
   d_free(user_info.theme);
   user_info.theme = d_config_get_string("/gnapster/Display/theme");
   
   hook_text_insert(get_current_stab(), CURR, SYSTEM, "load_theme", "%s\4%s",
		    lookup_hook("theme_name", hooks),
		    lookup_hook("theme_author", hooks));
}
