/* ui.c - user interface creation and modification */

#include "xtext.h"

#include <netinet/in.h>
#include <ctype.h>

#include "gnapster.h"
#include "ui.h"
#include "menus.h"
#include "callbacks.h"
#include "servers.h"
#include "resume.h"
#include "chan.h"
#include "text.h"
#include "utils.h"
#include "themes.h"
#include "menu.h"

#include "debug.h"

#include <gdk/gdkx.h>
#include <gtk/gtkinvisible.h>

extern GnapsterMain *gmain;

extern PropDlg prop_dlg;
extern BuildDlg build_dlg;
extern BrowseDlg browse_dlg;
extern AddDlg add_dlg;
extern EditDlg edit_dlg;
extern ListChanDlg list_chan_dlg;
extern ResumeDlg resume_dlg;
extern RefreshDlg refresh_dlg;

extern UserInfo user_info;

extern GdkColor colors[];
extern GdkFont *xtext_font;

extern GList *dialogs;

#ifndef HAVE_GNOME
GtkItemFactory *search_popup_menu, *upload_popup_menu, *download_popup_menu,
               *browse_popup_menu, *queue_popup_menu, *chat_popup_menu;
GtkItemFactory *ifactory;
#endif /* HAVE_GNOME */

void entry_set_text(GtkWidget *w, char *fmt, ...) {
   va_list args;
   char *s;
   
   va_start(args, fmt);
   s = d_strdup_vprintf(fmt, args);
   va_end(args);
   
   if (!s)
     return;
   
   gtk_entry_set_text(GTK_ENTRY(w), s);
   
   d_free(s);
}

void configure_event(GtkWidget *w, GdkEventConfigure *e, GtkWidget *text) {
   if (GTK_XTEXT(text)->transparent)
     gtk_widget_queue_draw(text);
}

void notebook_set_sensitive(GtkWidget *notebook, int pn, int sens) {
   GtkWidget *label, *page;
   
   page = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), pn);
   
   label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook), page);
   
   gtk_widget_set_sensitive(label, sens);
}

void notebook_set_label(GtkWidget *notebook, int pn, char *text) {
   GtkWidget *label, *page;
   
   page = gtk_notebook_get_nth_page(GTK_NOTEBOOK(notebook), pn);
   
   label = gtk_notebook_get_tab_label(GTK_NOTEBOOK(notebook), page);
   
   gtk_label_set_text(GTK_LABEL(label), text);
}

void search_sort_results(GtkWidget *w) {
   GtkCList *clist = GTK_CLIST(w);
   int sc;
   
   /* sort by filename -> user -> [ default ] */
   sc = clist->sort_column;
   search_sort(clist, 0);
   search_sort(clist, 2);
   search_sort(clist, sc);
}

/* i should really do this by setting data to the widgets */
STab *console_tab_notebook(GtkWidget *widget) {
   GList *ptr;
   STab *stab;
   
   for(ptr=ttbl; ptr; ptr=ptr->next) {
      stab = ptr->data;
      if (!stab)
	continue;
      
      if (stab->ct->notebook == widget)
	return stab;
   }   
   
   return NULL;
}

int search_tab_clist(GtkWidget *widget) {
   GList *ptr;
   STab *stab;
   
   for(ptr=ttbl; ptr; ptr=ptr->next) {
      stab = ptr->data;
      if (!stab)
	continue;
      
      if (stab->st->clist == widget)
	return 1;
   }
   
   return 0;
}

int is_queue_clist(GtkWidget *widget) {
   GList *ptr;
   STab *stab;
   
   for(ptr=ttbl; ptr; ptr=ptr->next) {
      stab = ptr->data;
      if (!stab)
	continue;
      
      if (gmain->dt->queue_clist == widget)
	return 1;
   }
   
   return 0;
}

/*
int is_browse_ctree(GtkWidget *widget) {
   FORSTABS(if (stab->bt->ctree == widget) return 1);
   
   return 0;
}
*/

void set_clist_server(GtkWidget *w, STab *stab, int row) {
   GtkStyle *style;
   int shades[][3] = {
      { 0xfff, 0,     0 },	/* red */
      { 0,     0xfff, 0 },	/* green */
      { 0,     0,     0xfff },	/* blue */
      { 0,     0xfff, 0xfff },	/* cyan */
      { 0xfff, 0,     0xfff },	/* purple */
      { 0xfff, 0xfff, 0 },	/* yellow */
   };
   int i;
   
   /* do not shade the first server */
/*   if (stab->pn == 0)
     return;*/
   
   style = gtk_style_copy(gtk_widget_get_style(w));
   
   if (!style)
     return;
   
   i = (stab->pn) % (sizeof(shades) / 3);

   COLOR_MOD(style->bg[GTK_STATE_NORMAL].red, shades[i][0]);
   COLOR_MOD(style->bg[GTK_STATE_NORMAL].green, shades[i][1]);
   COLOR_MOD(style->bg[GTK_STATE_NORMAL].blue, shades[i][2]);

   gtk_clist_set_background(GTK_CLIST(w), row, &style->bg[GTK_STATE_NORMAL]);
   
   free(style);
}

void set_green_style(GtkWidget *w, int row, int col) {
   GtkCList *clist;
   GtkStyle *style;
   
   if (!w)
     return;
   
   clist = GTK_CLIST(w);
   
   style = gtk_clist_get_cell_style(clist, row, col);
   if (!style)
     style = gtk_style_copy(gtk_widget_get_style(w));

   if (!style)
     return;

   style->fg[GTK_STATE_NORMAL].red = 0;
   style->fg[GTK_STATE_NORMAL].green = 0xd100;
   style->fg[GTK_STATE_NORMAL].blue = 0;
   
   gtk_clist_set_cell_style(clist, row, col, style);
/*   
   free(style);*/
}

int dialog_open(GtkWidget *window) {
   GList *ptr;
   GtkWidget *widget;
   
   if (!window) /* obviously, it's not open :) */
     return 0;
   
   for(ptr=dialogs; ptr; ptr=ptr->next) {
      widget = ptr->data;
      if (!widget)
	continue;
      
      if (window == widget)
	return 1;
   }
   
   return 0;
}

GtkWidget *j_label_new(GtkWidget *box, char *s) {
   GtkWidget *label;
   
   label = gtk_label_new(s);
   j_box_pack(box, label, 0, 0, 0);
   gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
   
   return label;
}

GtkWidget *j_line_new(GtkWidget *w, int i1, int i2) {
   GtkWidget *temp;
   
   temp = gtk_hbox_new(FALSE, 5);
   gtk_box_pack_start(GTK_BOX(w), temp, i1, i2, 0);
   
   return temp;
}

GtkWidget *j_combo_new(int type, ...) {
   GtkWidget *widget;
   char *s;
   GList *list = NULL;
   va_list args;
   
   widget = gtk_combo_new();
   
   va_start(args, type);
   while(1) {
      s = va_arg(args, char *);
      if (!s)
	break;
      list = g_list_append(list, s);
   }
   va_end(args);
   
   gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(widget)->entry), type);
   gtk_combo_set_popdown_strings(GTK_COMBO(widget), list);
   
   return widget;
}

#ifndef HAVE_GNOME

static char *translatelambda(char *s, void *data) {
   return dgettext(PACKAGE, s); 
}

#define CREATE_POPUP(data,name) \
   name = gtk_item_factory_new(GTK_TYPE_MENU, "<Menu>", NULL); \
   gtk_item_factory_set_translate_func(name, (GtkTranslateFunc)translatelambda, NULL, NULL); \
   gtk_item_factory_create_items(name, sizeof(data)/sizeof(data[0]), data, NULL);

#endif /* !HAVE_GNOME */

#if 0
GtkWidget *create_gnapster_tab(char *label, char *stock) {
   GtkWidget *vbox;
   GtkWidget *stock_widget;
   GtkWidget *label_widget;
   
   vbox = gtk_hbox_new(FALSE, 0);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 0);
   
   stock_widget = gnome_stock_pixmap_widget_at_size(gmain->window, stock,
						    15, 15);
   label_widget = gtk_label_new(label);
   
   j_box_pack(vbox, stock_widget, 0, 0, 0);
   j_box_pack(vbox, label_widget, 0, 0, 0);
   
   gtk_widget_show(stock_widget);
   gtk_widget_show(label_widget);
   
   return vbox;
}
#endif /* 0 */

void unhook_configure_event(GtkWidget *w, void *data) {
   gtk_signal_disconnect_by_data(GTK_OBJECT(gmain->window), w);;
}

void hook_configure_event(GtkWidget *w, GtkWidget *text) {
   gtk_signal_connect(GTK_OBJECT(w), "configure_event",
		      GTK_SIGNAL_FUNC(configure_event), text);
   gtk_signal_connect(GTK_OBJECT(text), "destroy",
		      GTK_SIGNAL_FUNC(unhook_configure_event), text);
}

void j_dialog_close_cb(GtkWidget *w, void *data) {
   j_dialog_close(GTK_WIDGET(data));
}

void build_links(GtkWidget *ctree, GList *links) {
   char *hosts, *dup, *h1, *h2;
   GtkCTreeNode *node;
   GList *iter_ptr;
   
   ITER_LIST(links) {
      LIST_DATA(hosts);
      
      dup = d_strdup(hosts);
      hosts = dup;
      
      h1 = next_arg(hosts, &hosts);
      next_arg(hosts, &hosts);
      h2 = next_arg(hosts, &hosts);
      
      node = tree_find_node(ctree, h1);
      if (!node)
	node = tree_insert_node(ctree, tree_first_node(ctree), 0, NULL, h1, NULL);
	 
      tree_insert_node(ctree, node, 0, NULL, h2, NULL);
      
      d_free(dup);
   }
}

void create_server_links(STab *stab) {
   GtkWidget *window, *vbox, *hbox, *sw, *ctree;

   window = j_dialog_new(_("Server Links"),
			 "Button_Ok", NULL);
   
   gtk_widget_set_usize(window, 300, 350);
   
   vbox = j_dialog_vbox(window);
   
   hbox = j_line_new(vbox, 1, 1);
   
   sw = gtk_scrolled_window_new(NULL, NULL);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
				  GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
   j_box_pack(hbox, sw, 1, 1, 0);
   
   ctree = gtk_ctree_new(1, 0);
   build_links(ctree, stab->ct->links);
   gtk_container_add(GTK_CONTAINER(sw), ctree);

   j_dialog_button_connect(window, 0, j_dialog_close_cb, window);
   
   gtk_signal_connect(GTK_OBJECT(window), "destroy",
		      GTK_SIGNAL_FUNC(j_dialog_close_cb), window);
   
   j_dialog_show(window, vbox);
}

void xfer_close(GtkWidget *w, void *data) {
   j_dialog_close(GTK_WIDGET(data));
}

/* if stab is NULL, use all servers */
GtkWidget *create_xfer_stats(STab *stab) {
   GtkWidget *window, *vbox, *hbox, *box, *frame;
   char *buf;
   unsigned long d, u, td, tu;
   float d_kps, u_kps, td_kps, tu_kps;
   
   get_xfer_stats(stab, &d, &u, &td, &tu, &d_kps, &u_kps, &td_kps, &tu_kps);
   
   window = j_dialog_new(_("Transfer Statistics"),
			 "Button_Ok", NULL);
   
   gtk_widget_set_usize(window, 300, -1);
   
   vbox = j_dialog_vbox(window);
   
   hbox = j_line_new(vbox, 0, 0);
   
   d_msprintf(&buf, "Statistics for %s", stab ? stab->ci->server :
	      "all servers");
   j_label_new(hbox, buf);
   
   frame = gtk_frame_new("Downstream");
   j_box_pack(vbox, frame, 0, 0, 2);
   
   box = gtk_vbox_new(0, 5);
   gtk_container_set_border_width(GTK_CONTAINER(box), 5);
   gtk_container_add(GTK_CONTAINER(frame), box);
   
   d_msprintf(&buf, "Total Downloads: %lu", td);
   j_label_new(box, buf);
   
   d_strexp(&buf, "Total Incoming: %luK", d);
   j_label_new(box, buf);
   
   if (!stab) {
      d_strexp(&buf, "Total Client Incoming: %.01fM", gmain->in);
      j_label_new(box, buf);
   }
   
   d_strexp(&buf, "Average Transfer Speed: %.02fK/s (%.02fK/s total)", d_kps,
	    td_kps);
   j_label_new(box, buf);
   
   frame = gtk_frame_new("Upstream");
   j_box_pack(vbox, frame, 0, 0, 2);
   
   box = gtk_vbox_new(0, 5);
   gtk_container_set_border_width(GTK_CONTAINER(box), 5);
   gtk_container_add(GTK_CONTAINER(frame), box);

   d_msprintf(&buf, "Total Uploads: %lu", tu);
   j_label_new(box, buf);
   
   d_strexp(&buf, "Total Outgoing: %luK", u);
   j_label_new(box, buf);
   
   if (!stab) {
      d_strexp(&buf, "Total Client Outgoing: %.01fM", gmain->out);
      j_label_new(box, buf);
   }
   
   d_strexp(&buf, "Average Transfer Speed: %.02fK/s (%.02fK/s total)", u_kps,
	    tu_kps);
   j_label_new(box, buf);
   
   d_free(buf);
   
   j_dialog_button_connect(window, 0, xfer_close, window);
   
   gtk_signal_connect(GTK_OBJECT(window), "destroy",
		      GTK_SIGNAL_FUNC(xfer_close), window);
   
   j_dialog_show(window, vbox);
   
   return NULL;
}

void create_gnapster_main() {
   GtkWidget *prop_notify;
   GtkItemFactory *item_factory = NULL;
   GtkWidget *vbox;
   
#ifdef HAVE_GNOME
   gmain->window = gnome_app_new("gnapster", "Gnapster");
#else
   GtkAccelGroup *accel_group;

   gint nmenu_items = sizeof (gnapster_menu) / sizeof (gnapster_menu[0]);

   gmain->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
   gmain->vbox = gtk_vbox_new(FALSE, 5);
#endif /* HAVE_GNOME */
   
   gtk_widget_set_usize(GTK_WIDGET(gmain->window), 
			DEFAULT_WIDTH, DEFAULT_HEIGHT);
   gtk_window_set_wmclass(GTK_WINDOW(gmain->window),
			  "gnapster", "Gnapster");
   
#ifdef HAVE_GNOME
   gnome_app_create_menus(GNOME_APP(gmain->window), gnapster_menu);
#else
   accel_group = gtk_accel_group_new();
   item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", 
				       accel_group);
   ifactory = item_factory;
   gtk_item_factory_create_items(item_factory, nmenu_items, gnapster_menu,
				 NULL);
   gtk_window_add_accel_group(GTK_WINDOW(gmain->window), 
			      item_factory->accel_group);
   gmain->menubar = gtk_item_factory_get_widget(item_factory, "<main>");
   j_box_pack(gmain->vbox, gmain->menubar, 0, 0, 0);
   
   CREATE_POPUP(search_popup_menu_data,search_popup_menu);
   CREATE_POPUP(browse_popup_menu_data,browse_popup_menu);
   CREATE_POPUP(upload_popup_menu_data,upload_popup_menu);
   CREATE_POPUP(queue_popup_menu_data,queue_popup_menu);
   CREATE_POPUP(chat_popup_menu_data,chat_popup_menu);
   CREATE_POPUP(download_popup_menu_data,download_popup_menu);   
#endif /* HAVE_GNOME */
   
   gtk_window_set_title(GTK_WINDOW(gmain->window), "Gnapster");
   gtk_window_set_policy(GTK_WINDOW(gmain->window), 1, 1, 1);
   
   gtk_signal_connect(GTK_OBJECT(gmain->window), "delete-event",
		      GTK_SIGNAL_FUNC(gnapster_exit_cb), NULL);
   
   gmain->srv_notebook = gtk_notebook_new();
   gtk_notebook_set_show_tabs(GTK_NOTEBOOK(gmain->srv_notebook), 
			      user_info.conf[SHOW_STABS]);
   gtk_signal_connect(GTK_OBJECT(gmain->srv_notebook), "switch_page",
		      GTK_SIGNAL_FUNC(switch_page_cb), NULL);
   
#ifdef HAVE_GNOME
   gnome_app_set_contents(GNOME_APP(gmain->window), 
			  gmain->srv_notebook);
   
   gmain->appbar = gnome_appbar_new(TRUE, TRUE, GNOME_PREFERENCES_NEVER);
   gmain->progress = gnome_appbar_get_progress(GNOME_APPBAR(gmain->appbar));
   gtk_progress_set_activity_mode(GTK_PROGRESS(gmain->progress), TRUE);
   gtk_progress_set_value(GTK_PROGRESS(gmain->progress), 0);
   
   gnome_app_set_statusbar(GNOME_APP(gmain->window),
			   gmain->appbar);
   
   gtk_container_set_border_width(GTK_CONTAINER(GNOME_APP(gmain->window)->contents), 5);
#else
   gmain->appbar = gtk_statusbar_new();
   
   gmain->progress = GTK_PROGRESS(gtk_progress_bar_new ());
   gtk_progress_set_activity_mode(GTK_PROGRESS(gmain->progress), TRUE);
   gtk_progress_set_value(GTK_PROGRESS(gmain->progress), 0);
   gtk_box_pack_end(GTK_BOX(gmain->appbar), GTK_WIDGET(gmain->progress), 0, 0, 0);
   
   vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
   
   j_box_pack(gmain->vbox, vbox, 1, 1, 0);
   
   j_box_pack(vbox, gmain->srv_notebook, 1, 1, 0);

   gtk_box_pack_end(GTK_BOX(gmain->vbox), gmain->appbar, 0, 0, 0);
   
   gtk_container_add(GTK_CONTAINER(gmain->window), gmain->vbox);

/*   gtk_container_set_border_width(GTK_CONTAINER(gmain->vbox), 5);*/
#endif /* HAVE_GNOME */
   
   /* from xchat ;P */
   prop_notify = gtk_invisible_new();
   gtk_widget_show(prop_notify);
   
   gdk_window_set_user_data(GDK_ROOT_PARENT(), prop_notify);
   
   XSelectInput(GDK_DISPLAY(), GDK_ROOT_WINDOW(), PropertyChangeMask);
   
   gtk_signal_connect(GTK_OBJECT(prop_notify), "property-notify-event",
		      GTK_SIGNAL_FUNC(property_notify), NULL);

   set_tog_widgets(item_factory);
}

/* menu interface */
void create_stab() {
   append_server_tab();
}

void remove_stab() {
   STab *stab;
   
   stab = get_current_stab();
   d_assert(stab != NULL);
   
   remove_server_menu();

   remove_server_tab(stab);

   append_server_menu();
}

STab *append_server_tab() {
   GtkWidget *vbox, *notebook;
   STab *stab;
   
/*   if (ttbl)
     gtk_notebook_set_show_tabs(GTK_NOTEBOOK(gmain->srv_notebook), 1);*/
   
   vbox = gtk_vbox_new(FALSE, 0);

   /* create the notebook to hold Search, Browse, etc etc */
   notebook = gtk_notebook_new();
   gtk_signal_connect(GTK_OBJECT(notebook), "switch_page",
		      GTK_SIGNAL_FUNC(switch_page_cb), NULL);
   j_box_pack(vbox, notebook, 1, 1, 0);
      
   /* append the high level server tab */
   gtk_notebook_append_page(GTK_NOTEBOOK(gmain->srv_notebook), 
			    vbox, gtk_label_new("no server"));
   
   stab = d_new(STAB);
   
   stab->notebook = notebook;
   stab->vbox = vbox;
   
   stab->pn = g_list_length(ttbl);
   
   stab->ci = d_new(CONN_INFO);
   conn_info_init(stab->ci);
   
   /* build the UI held within */
   stab->st = d_new(SEARCH_TAB);
   stab->bt = d_new(BROWSE_TAB);
   stab->dt = d_new(DOWNLOAD_TAB);
   stab->ut = d_new(UPLOAD_TAB);
/*   stab->st = create_search_tab(stab, notebook);
   stab->bt = create_browse_tab(stab, notebook);
   stab->dt = create_download_tab(stab, notebook);
   stab->ut = create_upload_tab(stab, notebook);*/
   stab->ct = create_console_tab(stab, notebook);
/*   stab->mt = create_motd_tab(stab, notebook);*/
   
   ttbl = g_list_append(ttbl, stab);
   
   gtk_widget_show_all(vbox);
/*   gtk_widget_hide(stab->st->not_label);
   gtk_widget_hide(stab->st->not_entry);*/

   /* rehash the servers menu */
   remove_server_menu();
   append_server_menu();
   
   /* focus the new page */
   gtk_notebook_set_page(GTK_NOTEBOOK(gmain->srv_notebook), stab->pn + 4);
   
   return stab;
}

int reorganize_stab() {
   int i = 0;
   
   FORSTABS((stab->pn = i++));
   
   return i;
}

void remove_server_tab(STab *stab) {
   if (g_list_length(ttbl) <= 1)
     return;
   
   disconnect(stab);
   remove_server_menu();
   TIMEOUT_REMOVE(stab->ci->reconnect_timeout);
   
   gtk_notebook_remove_page(GTK_NOTEBOOK(gmain->srv_notebook), stab->pn + 4);

   j_free(CONN_INFO, stab->ci);
   
   ttbl = g_list_remove(ttbl, stab);
   reorganize_stab();
/*   if (reorganize_stab() <= 1)
     gtk_notebook_set_show_tabs(GTK_NOTEBOOK(gmain->srv_notebook), 0);*/
   
/*   j_free(STAB, stab);*/
}

void accounts_dlg_close(GtkWidget *button, void *data) {
   j_dialog_close(GTK_WIDGET(data));
}

void fill_accounts(GtkWidget *w, int combo) {
   FILE *f;
   GList *felem, *ptr, *acnts = NULL;
   FileEntry *fent;
   Account *acnt;
   char *dptr, *user, *pass;
   int primary;
   
   f = open_local_path("accounts", NULL, "r");
   if (!f)
     return;
   
   felem = read_file(f);
   
   fclose(f);
   
   for(ptr=felem; ptr; ptr=ptr->next) {
      fent = ptr->data;
      if (!fent)
	continue;
      
      dptr = fent->lptr;
      
      user = next_arg_full(dptr, &dptr, ':');
      pass = next_arg(dptr, &dptr);
      convert(next_arg(dptr, &dptr), "%i", &primary);
      
      NA_ERR_HANDLE_BREAK();
      
      if (combo) {
	 acnts = g_list_append(acnts, d_strdup(user));
	 continue;
      }
      
      acnt = d_new(ACCOUNT);
      
      acnt->user = d_strdup(user);
      acnt->pass = d_strdup(pass);
      acnt->primary = primary;
      
      gnapster_clist_append(w, NULL, acnt, user, NULL);
   }
   
   read_file_free(felem);
   
   if (combo && acnts) {
      gtk_combo_set_popdown_strings(GTK_COMBO(browse_dlg.account), acnts);
      gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(browse_dlg.account)->entry),
			 ((char *)acnts->data));
      
      char_list_free(acnts);
   }
}

void create_accounts_cb(GtkWidget *w, void *data) {
   GtkWidget *window, *hbox, *vbox, *clist, *vboxr, *user, *pass, *label;
   GtkWidget *add, *change, *rem, *primary;
   GtkWidget *combo;
   
   combo = (data) ? data : NULL;
   
   window = j_dialog_new(_("Account Manager"), 
			 "Button_Ok", "Button_Cancel", NULL);
      
   gtk_widget_set_usize(window, 450, 300);

   SET_WIDGET(window, combo);
   
   vbox = j_dialog_vbox(window);
   
   hbox = j_line_new(vbox, 1, 1);
   
   clist = gtk_clist_new(1);
   SET_WIDGET(window, clist);
   gtk_signal_connect(GTK_OBJECT(clist), "select-row",
		      GTK_SIGNAL_FUNC(account_show), window);
   fill_accounts(clist, 0);
   j_box_pack(hbox, clist, 1, 1, 0);
   
   vboxr = gtk_vbox_new(0, 5);
   j_box_pack(hbox, vboxr, 0, 0, 0);
   
   hbox = j_line_new(vboxr, 0, 0);
   
   primary = gtk_check_button_new_with_label("Primary Account (Official Napster)");
   SET_WIDGET(window, primary);
   gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(primary), 
			       g_list_length(GTK_CLIST(clist)->row_list) ? 0 : 1);
   j_box_pack(hbox, primary, 0, 0, 0);
   
   hbox = j_line_new(vboxr, 0, 0);
   
   label = gtk_label_new("Username: ");
   j_box_pack(hbox, label, 0, 0, 0);
   
   user = gtk_entry_new();
   SET_WIDGET(window, user);
   j_box_pack(hbox, user, 1, 1, 0);
   
   hbox = j_line_new(vboxr, 0, 0);
   
   label = gtk_label_new("Password: ");
   j_box_pack(hbox, label, 0, 0, 0);
   
   pass = gtk_entry_new();
   gtk_entry_set_visibility(GTK_ENTRY(pass), 0);
   SET_WIDGET(window, pass);
   gtk_signal_connect(GTK_OBJECT(pass), "activate", 
		      GTK_SIGNAL_FUNC(account_add), window);
   j_box_pack(hbox, pass, 1, 1, 0);
   
   hbox = j_line_new(vboxr, 0, 0);
   
   add = gtk_button_new_with_label("Add");
   gtk_signal_connect(GTK_OBJECT(add), "clicked",
		      GTK_SIGNAL_FUNC(account_add), window);
   j_box_pack(hbox, add, 1, 1, 0);
   
   change = gtk_button_new_with_label("Change");
   gtk_signal_connect(GTK_OBJECT(change), "clicked",
		      GTK_SIGNAL_FUNC(account_change), window);
   j_box_pack(hbox, change, 1, 1, 0);
   
   rem = gtk_button_new_with_label("Remove");
   gtk_signal_connect(GTK_OBJECT(rem), "clicked",
		      GTK_SIGNAL_FUNC(account_rem), window);
   j_box_pack(hbox, rem, 1, 1, 0);
   
   j_dialog_button_connect(window, 0, accounts_dlg_close, window);
   j_dialog_button_connect(window, 1, accounts_dlg_close, window);
   
   gtk_signal_connect(GTK_OBJECT(window), "destroy",
		      GTK_SIGNAL_FUNC(accounts_dlg_close), window);
   
   j_dialog_show(window, vbox);
}

GtkWidget *prop_entry_new(char *key, int *value) {
   char *s, *tk;
   GtkWidget *pent;
   
   s = NULL;
   
   if (key) {
      d_msprintf(&tk, "/gnapster/%s", key);

      s = d_config_get_string(tk);

      d_free(tk);

      /* we still need to create the widget, even though it wasnt
       * found in the config */
      if (!s)
	s = d_strdup("");
   }
   
   pent = gtk_entry_new();
   
   if (!value)
     gtk_entry_set_text(GTK_ENTRY(pent), s);
   else
     entry_set_text(pent, "%i", *value);
   
   gtk_signal_connect(GTK_OBJECT(pent), "changed",
		      GTK_SIGNAL_FUNC(preferences_changed_cb), NULL);
   
   d_free(s);
   
   j_box_pack(prop_dlg.hbox, pent, 0, 0, 0);
   
   return pent;
}

void create_prop_dlg(void *changed_cb, void *button_cb, void *destroy_cb, void *clicked_cb, int *new_user_val) {
   char *themes_title[3] = { "Theme", "Author", NULL };
   char *download_dir, *upload_dir;
   int conn;
   GtkWidget *vbox;
   
   download_dir = d_config_get_string("/gnapster/User/dl_dir");
   upload_dir = d_config_get_string("/gnapster/User/ul_dir");
   conn = j_config_get_int("/gnapster/User/connection");
   
   prop_dlg.calling = 0;
   
   prop_dlg.window = j_property_box_new();

   /* USER INFORMATION */
   
   prop_dlg.vbox = gtk_vbox_new(FALSE, 5);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   gtk_container_border_width(GTK_CONTAINER(prop_dlg.vbox), 5);
   
   prop_dlg.accounts = gtk_button_new_with_label("Accounts...");
   gtk_signal_connect(GTK_OBJECT(prop_dlg.accounts), "clicked",
		      GTK_SIGNAL_FUNC(create_accounts_cb), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.accounts, 0, 0, 0);
/*   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   j_label_new(prop_dlg.hbox, "Username: ");
   
   prop_dlg.user_entry = prop_entry_new("User/user", NULL);
   
   prop_dlg.new_user = gtk_check_button_new_with_label(_("New Account"));
   if (new_user_val) 
     gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.new_user),
				  *new_user_val);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.new_user), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), new_user_val);
   gtk_box_pack_start(GTK_BOX(prop_dlg.hbox), prop_dlg.new_user,
		      FALSE, FALSE, 0);

   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   j_label_new(prop_dlg.hbox, _("Password: "));
   
   prop_dlg.pass_entry = prop_entry_new("User/pass", NULL);
   gtk_entry_set_visibility(GTK_ENTRY(prop_dlg.pass_entry), 0);*/

   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   j_label_new(prop_dlg.hbox, _("Data port: "));
   
   prop_dlg.nap_port = prop_entry_new(NULL, (int *)(&(gmain->port)));
   gtk_entry_set_max_length(GTK_ENTRY(prop_dlg.nap_port), 5);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);

   j_label_new(prop_dlg.hbox, _("Connection: "));
   
   prop_dlg.connection = j_combo_new(0, "Unknown", "14.4", "28.8", "33.6",
				     "56k", "64k ISDN", "128k ISDN", "Cable",
				     "DSL", "T1", "T3+", NULL);
   gtk_widget_set_usize(prop_dlg.connection, 100, -1);
   gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(prop_dlg.connection)->entry),
		      napster_conn_as_str(conn));
   gtk_signal_connect(GTK_OBJECT(GTK_COMBO(prop_dlg.connection)->entry), "changed",
		      GTK_SIGNAL_FUNC(changed_cb), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.connection, 0, 0, 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
  
   j_label_new(prop_dlg.hbox, _("Download Directory: "));
   
   prop_dlg.download_dir = j_file_entry_new(NULL, _("Browse Download Directory"));
   if (download_dir) 
     gtk_entry_set_text(GTK_ENTRY(j_file_entry_gtk_entry(prop_dlg.download_dir)), download_dir);
   gtk_signal_connect(GTK_OBJECT(j_file_entry_gtk_entry(prop_dlg.download_dir)),
		      "changed", GTK_SIGNAL_FUNC(changed_cb), NULL);
   gtk_box_pack_start(GTK_BOX(prop_dlg.hbox), prop_dlg.download_dir,
		      TRUE, TRUE, 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);

   prop_dlg.exec_dl_optn = gtk_check_button_new_with_label(_("Execute after download: "));
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.exec_dl_optn),
				user_info.conf[EXEC_DL_OPTN]);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.exec_dl_optn), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &(user_info.conf[EXEC_DL_OPTN]));
   j_box_pack(prop_dlg.hbox, prop_dlg.exec_dl_optn, 0, 0, 0);

   prop_dlg.exec_dl = prop_entry_new("User/exec_dl=xmms -e", NULL);

/*   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);

   j_label_new(prop_dlg.hbox, _("Upload Directory: "));

   prop_dlg.upload_dir = j_file_entry_new(NULL, _("Browse Upload Directory"));
   if (upload_dir) 
     gtk_entry_set_text(GTK_ENTRY(j_file_entry_gtk_entry(prop_dlg.upload_dir)), 
			upload_dir);
   gtk_signal_connect(GTK_OBJECT(j_file_entry_gtk_entry(prop_dlg.upload_dir)),
		      "changed", GTK_SIGNAL_FUNC(changed_cb), NULL);
   gtk_box_pack_start(GTK_BOX(prop_dlg.hbox), prop_dlg.upload_dir,
		      TRUE, TRUE, 0);

   prop_dlg.build_list = gtk_button_new_with_label(_("Build mp3 list"));
   gtk_signal_connect(GTK_OBJECT(prop_dlg.build_list), "clicked",
		      GTK_SIGNAL_FUNC(build_list_cb), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.build_list, 0, 0, 0);

   prop_dlg.update_list = gtk_button_new_with_label(_("Update list"));
   gtk_signal_connect(GTK_OBJECT(prop_dlg.update_list), "clicked",
		      GTK_SIGNAL_FUNC(update_list_cb), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.update_list, 0, 0, 0);*/
   
   j_property_box_append_page(prop_dlg.vbox,
			      gtk_label_new(_("User Information")));
   
   /* SHARING */
   
   prop_dlg.vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_border_width(GTK_CONTAINER(prop_dlg.vbox), 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   j_label_new(prop_dlg.hbox, "Share path: ");
   
   prop_dlg.upload_dir = j_file_entry_new(NULL, _("Choose Share Directory"));
   gtk_signal_connect(GTK_OBJECT(j_file_entry_gtk_entry(prop_dlg.upload_dir)),
		      "activate", GTK_SIGNAL_FUNC(share_addpath), &prop_dlg);
   gtk_signal_connect(GTK_OBJECT(j_file_entry_gtk_entry(prop_dlg.upload_dir)),
		      "changed", GTK_SIGNAL_FUNC(changed_cb), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.upload_dir, 1, 1, 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 1, 1);
   
   prop_dlg.share_paths = gtk_clist_new(1);
   fill_share_paths(prop_dlg.share_paths);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.share_paths), "select-row",
		      GTK_SIGNAL_FUNC(share_showpath), &prop_dlg);
   j_box_pack(prop_dlg.hbox, prop_dlg.share_paths, 1, 1, 0);
   
   vbox = gtk_vbox_new(FALSE, 5);
   j_box_pack(prop_dlg.hbox, vbox, 0, 0, 0);
   
   prop_dlg.add_share = gtk_button_new_with_label(_("Add Path"));
   gtk_signal_connect(GTK_OBJECT(prop_dlg.add_share), "clicked",
		      GTK_SIGNAL_FUNC(share_addpath), &prop_dlg);
   j_box_pack(vbox, prop_dlg.add_share, 0, 0, 0);
   
   prop_dlg.rem_share = gtk_button_new_with_label(_("Remove Path"));
   gtk_signal_connect(GTK_OBJECT(prop_dlg.rem_share), "clicked",
		      GTK_SIGNAL_FUNC(share_rempath), &prop_dlg);
   j_box_pack(vbox, prop_dlg.rem_share, 0, 0, 0);
   
   prop_dlg.build_list = gtk_button_new_with_label(_("Build list"));
   gtk_signal_connect(GTK_OBJECT(prop_dlg.build_list), "clicked",
		      GTK_SIGNAL_FUNC(build_list_cb), NULL);
   j_box_pack(vbox, prop_dlg.build_list, 0, 0, 0);
   
   prop_dlg.update_list = gtk_button_new_with_label(_("Update list"));
   gtk_signal_connect(GTK_OBJECT(prop_dlg.update_list), "clicked",
		      GTK_SIGNAL_FUNC(update_list_cb), NULL);
   j_box_pack(vbox, prop_dlg.update_list, 0, 0, 0);
   
   j_property_box_append_page(prop_dlg.vbox,
			      gtk_label_new(_("Sharing")));
   
   /* BANDWIDTH */
   
   prop_dlg.vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_border_width(GTK_CONTAINER(prop_dlg.vbox), 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   j_label_new(prop_dlg.hbox, _("A setting of 0 is assumed to mean unlimited"));
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);

   gtk_widget_set_usize(j_label_new(prop_dlg.hbox, _("Downstream: ")), 150, -1);
      
   prop_dlg.bw_down = gtk_spin_button_new((GtkAdjustment *)gtk_adjustment_new(0.0, 0.0, 1024.0, 1.0, 16.0, 0.0),
					  0, 0);
   gtk_widget_set_usize(prop_dlg.bw_down, 55, -2);
   config_combo_widget(prop_dlg.bw_down, "bw_down");   
   j_box_pack(prop_dlg.hbox, prop_dlg.bw_down, 0, 0, 0);
   
   j_label_new(prop_dlg.hbox, _("kilobytes per second"));
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);

   gtk_widget_set_usize(j_label_new(prop_dlg.hbox, _("Upstream: ")), 150, -1);
   
   prop_dlg.bw_up = gtk_spin_button_new((GtkAdjustment *)gtk_adjustment_new(0.0, 0.0, 1024.0, 1.0, 16.0, 0.0),
					0, 0);
   gtk_widget_set_usize(prop_dlg.bw_up, 55, -2);
   config_combo_widget(prop_dlg.bw_up, "bw_up");
   j_box_pack(prop_dlg.hbox, prop_dlg.bw_up, 0, 0, 0);
   
   j_property_box_append_page(prop_dlg.vbox,
			      gtk_label_new(_("Bandwidth")));
   
   /* CONNECTION INFORMATION */
   
   prop_dlg.vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_border_width(GTK_CONTAINER(prop_dlg.vbox), 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);

   prop_dlg.firewall = gtk_check_button_new_with_label(_("Firewalled without the ability to accept connections"));
   gtk_signal_connect(GTK_OBJECT(prop_dlg.firewall), "clicked",
		      GTK_SIGNAL_FUNC(changed_cb), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.firewall, 0, 0, 0);
   
   j_label_new(prop_dlg.hbox, _("(required for SOCKS proxy)"));
   
   vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
   
   prop_dlg.frame = gtk_frame_new("SOCKS (EXPERIMENTAL)");
   gtk_container_add(GTK_CONTAINER(prop_dlg.frame), vbox);
   j_box_pack(prop_dlg.vbox, prop_dlg.frame, 1, 1, 0);
   
   prop_dlg.hbox = j_line_new(vbox, 0, 0);
   
   prop_dlg.socks_firewall = gtk_check_button_new_with_label(_("SOCKS Proxy: "));
   gtk_signal_connect(GTK_OBJECT(prop_dlg.socks_firewall), "clicked",
		      GTK_SIGNAL_FUNC(changed_cb), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.socks_firewall, 0, 0, 0);
   
   prop_dlg.hbox = j_line_new(vbox, 0, 0);
   
   prop_dlg.label = gtk_label_new(_("SOCKS Server IP: "));
   j_box_pack(prop_dlg.hbox, prop_dlg.label, 0, 0, 0);
   
   prop_dlg.socks_server = prop_entry_new("Connection/socks_server", NULL);
   
   prop_dlg.label = gtk_label_new(_(" Port: "));
   j_box_pack(prop_dlg.hbox, prop_dlg.label, 0, 0, 0);
   
   prop_dlg.socks_port = prop_entry_new("Connection/socks_port", NULL);
   
   prop_dlg.hbox = j_line_new(vbox, 0, 0);
   
   prop_dlg.socks4 = gtk_radio_button_new_with_label(NULL, _("SOCKS4"));
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.socks4),
				user_info.conf[SOCKS4]);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.socks4), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &(user_info.conf[SOCKS4]));
   j_box_pack(prop_dlg.hbox, prop_dlg.socks4, 0, 0, 0);

   prop_dlg.hbox = j_line_new(vbox, 0, 0);
   
   prop_dlg.socks5 = gtk_radio_button_new_with_label(gtk_radio_button_group(GTK_RADIO_BUTTON(prop_dlg.socks4)),
						     _("SOCKS5"));
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.socks5),
				user_info.conf[SOCKS5]);
   
   gtk_signal_connect(GTK_OBJECT(prop_dlg.socks5), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &(user_info.conf[SOCKS5]));
   j_box_pack(prop_dlg.hbox, prop_dlg.socks5, 0, 0, 0);
   
   prop_dlg.hbox = j_line_new(vbox, 0, 0);

   j_label_new(prop_dlg.hbox, _("SOCKS5 Username: "));
   
   prop_dlg.socks5_username = prop_entry_new("Connection/socks5_username", NULL);
   
   prop_dlg.hbox = j_line_new(vbox, 0, 0);
   
   prop_dlg.label = gtk_label_new("SOCKS5 Password: ");
   j_box_pack(prop_dlg.hbox, prop_dlg.label, 0, 0, 0);

   prop_dlg.socks5_password = prop_entry_new("Connection/socks5_password", NULL);
      
   config_toggle_widget(prop_dlg.firewall, "firewall",
			prop_dlg.socks_firewall, NULL);

   config_toggle_widget(prop_dlg.socks_firewall, "socks_firewall",
			prop_dlg.socks_server, prop_dlg.socks_port,
			prop_dlg.socks4, prop_dlg.socks5,
			prop_dlg.socks5_username, prop_dlg.socks5_password,
			NULL);

   j_property_box_append_page(prop_dlg.vbox,
			      gtk_label_new(_("Connection")));
   
   /* OPTIONS */

   prop_dlg.vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_border_width(GTK_CONTAINER(prop_dlg.vbox), 5);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.autoquery = gtk_check_button_new_with_label(_("Auto-query a user upon incoming message"));
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.autoquery),
				user_info.conf[AUTOQUERY]);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.autoquery), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &(user_info.conf[AUTOQUERY]));
   j_box_pack(prop_dlg.hbox, prop_dlg.autoquery, 0, 0, 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.name_fix = gtk_check_button_new_with_label(_("Convert spaces to underscores?"));
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.name_fix),
				user_info.conf[NAME_FIX]);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.name_fix), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &(user_info.conf[NAME_FIX]));
   gtk_box_pack_start(GTK_BOX(prop_dlg.hbox), prop_dlg.name_fix,
		      FALSE, FALSE, 0);

   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.hide_users = gtk_check_button_new_with_label(_("Auto-hide userlist on channel join"));
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.hide_users),
				user_info.conf[HIDE_USERS]);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.hide_users), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &(user_info.conf[HIDE_USERS]));
   j_box_pack(prop_dlg.hbox, prop_dlg.hide_users, 0, 0, 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.hide_joinpart = gtk_check_button_new_with_label(_("Hide join/part messages"));
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.hide_joinpart),
				user_info.conf[HIDE_JOINPART]);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.hide_joinpart), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &(user_info.conf[HIDE_JOINPART]));
   j_box_pack(prop_dlg.hbox, prop_dlg.hide_joinpart, 0, 0, 0);
   
/*   prop_dlg.theme_colors = gtk_check_button_new_with_label(_("Use themes state colors for text widget\n(may cause problems with some themes)"));
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.theme_colors), 
				user_info.conf[THEME_COLORS]);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.theme_colors), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &(user_info.conf[THEME_COLORS]));
   gtk_box_pack_start(GTK_BOX(prop_dlg.hbox), prop_dlg.theme_colors,
		      FALSE, FALSE, 0);*/
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.reject_ul = gtk_check_button_new_with_label(_("Reject uploads while downloads are active"));
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.reject_ul),
				user_info.conf[REJECT_UL]);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.reject_ul), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &(user_info.conf[REJECT_UL]));
   j_box_pack(prop_dlg.hbox, prop_dlg.reject_ul, 0, 0, 0);
   
   j_property_box_append_page(prop_dlg.vbox,
			      gtk_label_new(_("Options")));
   
   /* LOGGING */
   
   prop_dlg.vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_border_width(GTK_CONTAINER(prop_dlg.vbox), 5);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.label = gtk_label_new("Note: Gnapster logs to ~/.gnapster/logs/");
   j_box_pack(prop_dlg.hbox, prop_dlg.label, 0, 0, 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.logging = gtk_check_button_new_with_label(_("Log messages to disk"));
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.logging), 
				user_info.conf[LOGGING]);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.logging), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &(user_info.conf[LOGGING]));
   j_box_pack(prop_dlg.hbox, prop_dlg.logging, 0, 0, 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.timestamp = gtk_check_button_new_with_label(_("Timestamp events"));
   gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prop_dlg.timestamp),
				user_info.conf[TIMESTAMP_EV]);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.timestamp), "clicked",
		      GTK_SIGNAL_FUNC(clicked_cb), &(user_info.conf[TIMESTAMP_EV]));
   j_box_pack(prop_dlg.hbox, prop_dlg.timestamp, 0, 0, 0);
   
   j_property_box_append_page(prop_dlg.vbox,
			      gtk_label_new(_("Logging")));
   
   /* XTEXT PREFS */
   
   prop_dlg.vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_border_width(GTK_CONTAINER(prop_dlg.vbox), 5);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.label = gtk_label_new("Font: ");
   j_box_pack(prop_dlg.hbox, prop_dlg.label, 0, 0, 0);
   
   prop_dlg.font_entry = gtk_entry_new();
   if (user_info.default_font)
     gtk_entry_set_text(GTK_ENTRY(prop_dlg.font_entry), user_info.default_font);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.font_entry), "changed",
		      GTK_SIGNAL_FUNC(changed_cb), NULL);

   d_free(user_info.default_font);
   user_info.default_font = d_config_get_string("/gnapster/xtext/font");
   
   j_box_pack(prop_dlg.hbox, prop_dlg.font_entry, 1, 1, 0);
   
   prop_dlg.font_browse = gtk_button_new_with_label("Browse...");
   gtk_signal_connect(GTK_OBJECT(prop_dlg.font_browse), "clicked",
		      GTK_SIGNAL_FUNC(browse_font_cb), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.font_browse, 0, 0, 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.label = gtk_label_new("Note: changing font requires a client restart");
   j_box_pack(prop_dlg.hbox, prop_dlg.label, 0, 0, 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.trans = gtk_check_button_new_with_label("Transparency");
   j_box_pack(prop_dlg.hbox, prop_dlg.trans, 0, 0, 0);
   
   prop_dlg.update_tint = gtk_button_new_with_label("Update XText");
   gtk_signal_connect(GTK_OBJECT(prop_dlg.update_tint), "clicked",
		      GTK_SIGNAL_FUNC(update_xtext), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.update_tint, 0, 0, 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.label = gtk_label_new("Red: ");
   j_box_pack(prop_dlg.hbox, prop_dlg.label, 0, 0, 0);
   
   prop_dlg.tint_red = 
     gtk_spin_button_new((GtkAdjustment *)gtk_adjustment_new(255.0, 0.0, 255.0, 1.0, 5.0, 0.0),
			 0, 0);
   config_combo_widget(prop_dlg.tint_red, "tint_red");
   j_box_pack(prop_dlg.hbox, prop_dlg.tint_red, 0, 0, 0);
   
   prop_dlg.label = gtk_label_new("Green: ");
   j_box_pack(prop_dlg.hbox, prop_dlg.label, 0, 0, 0);
   
   prop_dlg.tint_green =
     gtk_spin_button_new((GtkAdjustment *)gtk_adjustment_new(255.0, 0.0, 255.0, 1.0, 5.0, 0.0),
			 0, 0);
   config_combo_widget(prop_dlg.tint_green, "tint_green");
   j_box_pack(prop_dlg.hbox, prop_dlg.tint_green, 0, 0, 0);
   
   prop_dlg.label = gtk_label_new("Blue: ");
   j_box_pack(prop_dlg.hbox, prop_dlg.label, 0, 0, 0);
   
   prop_dlg.tint_blue = 
     gtk_spin_button_new((GtkAdjustment *)gtk_adjustment_new(255.0, 0.0, 255.0, 1.0, 5.0, 0.0),
			 0, 0);
   config_combo_widget(prop_dlg.tint_blue, "tint_blue");
   j_box_pack(prop_dlg.hbox, prop_dlg.tint_blue, 0, 0, 0);
   
   config_toggle_widget(prop_dlg.trans, "tint",
			prop_dlg.tint_red, prop_dlg.tint_green,
			prop_dlg.tint_blue, NULL);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
      
   prop_dlg.label = gtk_label_new("Current theme: ");
   j_box_pack(prop_dlg.hbox, prop_dlg.label, 0, 0, 0);
   
   prop_dlg.curr_theme = gtk_entry_new();
   gtk_entry_set_editable(GTK_ENTRY(prop_dlg.curr_theme), 0);
   gtk_entry_set_text(GTK_ENTRY(prop_dlg.curr_theme), user_info.theme);
   gtk_signal_connect(GTK_OBJECT(prop_dlg.curr_theme), "changed",
		      GTK_SIGNAL_FUNC(changed_cb), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.curr_theme, 1, 1, 0);

   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 1, 1);
   
   prop_dlg.scroller = gtk_scrolled_window_new(NULL, NULL);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(prop_dlg.scroller),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
   j_box_pack(prop_dlg.hbox, prop_dlg.scroller, 1, 1, 0);
   
   prop_dlg.themes = gtk_clist_new_with_titles(2, themes_title);
   gtk_clist_set_column_width(GTK_CLIST(prop_dlg.themes), 0, 150);
   populate_themes_clist();
   gtk_signal_connect(GTK_OBJECT(prop_dlg.themes), "select-row",
		      GTK_SIGNAL_FUNC(themes_select_row), NULL);
   gtk_container_add(GTK_CONTAINER(prop_dlg.scroller), prop_dlg.themes);
   
   j_property_box_append_page(prop_dlg.vbox,
			      gtk_label_new(_("Display")));
   /* AUTO-JOIN */
   
   prop_dlg.vbox = gtk_vbox_new(FALSE, 5);
   gtk_container_border_width(GTK_CONTAINER(prop_dlg.vbox), 5);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 0, 0);
   
   prop_dlg.aj_entry = gtk_entry_new();
   gtk_signal_connect(GTK_OBJECT(prop_dlg.aj_entry), "activate",
		      GTK_SIGNAL_FUNC(autojoin_add), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.aj_entry, 1, 1, 0);
   
   prop_dlg.aj_add = gtk_button_new_with_label("Add");
   gtk_signal_connect(GTK_OBJECT(prop_dlg.aj_add), "clicked",
		      GTK_SIGNAL_FUNC(autojoin_add), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.aj_add, 0, 0, 0);
   
   prop_dlg.aj_remove = gtk_button_new_with_label("Remove");
   gtk_signal_connect(GTK_OBJECT(prop_dlg.aj_remove), "clicked",
		      GTK_SIGNAL_FUNC(autojoin_remove), NULL);
   j_box_pack(prop_dlg.hbox, prop_dlg.aj_remove, 0, 0, 0);
   
   prop_dlg.hbox = j_line_new(prop_dlg.vbox, 1, 1);
   
   prop_dlg.aj_clist = gtk_clist_new(1);
   populate_autojoin_clist();
   j_box_pack(prop_dlg.hbox, prop_dlg.aj_clist, 1, 1, 0);
   
   j_property_box_append_page(prop_dlg.vbox,
			      gtk_label_new(_("Auto-Join")));

#ifndef HAVE_GNOME
   j_dialog_show(prop_dlg.window, NULL);
#else
   /* simulate a j_dialog_new */
   dialogs = g_list_prepend(dialogs, GTK_WIDGET(prop_dlg.window));
   
   gtk_signal_connect(GTK_OBJECT(prop_dlg.window), "apply",
		      GTK_SIGNAL_FUNC(preferences_button_cb), "apply");
   gtk_signal_connect(GTK_OBJECT(prop_dlg.window), "destroy",
		      GTK_SIGNAL_FUNC(preferences_destroy_cb), "destroy");   
#endif /* !HAVE_GNOME */
}

void create_build_dlg() {
   build_dlg.window = j_dialog_new(_("Building MP3 list..."),
				   "Button_Cancel",
				   NULL);
   
   gtk_widget_set_usize(build_dlg.window, 250, -1);
   
   build_dlg.vbox = j_dialog_vbox(build_dlg.window);
  
   build_dlg.hbox = j_line_new(build_dlg.vbox, 0, 0);
   
   build_dlg.label = gtk_label_new("Building list...");
   j_box_pack(build_dlg.hbox, build_dlg.label, 0, 0, 0);
   
   build_dlg.hbox = j_line_new(build_dlg.vbox, 0, 0);
   
   build_dlg.label = gtk_label_new("");
   j_box_pack(build_dlg.hbox, build_dlg.label, 0, 0, 0);
   
   j_dialog_button_connect(build_dlg.window, 0,
			   build_dlg_cancel_cb, NULL);
   
   j_dialog_show(build_dlg.window, build_dlg.vbox);
}

void create_browse_dlg() {
   char *titles[] = { "Server title", "Address", "Users", "Files", "Gigs", NULL };
   
   
   browse_dlg.window = j_dialog_new(_("Browse servers"),
				    "Connect",
				    "Button_Cancel",
				    NULL);

   gtk_widget_set_usize(browse_dlg.window, 650, 450);

   browse_dlg.vbox = j_dialog_vbox(browse_dlg.window);

   browse_dlg.hbox = j_line_new(browse_dlg.vbox, 0, 0);
   
   browse_dlg.add = gtk_button_new_with_label("Add server");
   gtk_signal_connect(GTK_OBJECT(browse_dlg.add), "clicked",
		      GTK_SIGNAL_FUNC(browse_dlg_add), NULL);
   j_box_pack(browse_dlg.hbox, browse_dlg.add, 0, 0, 0);
   
   browse_dlg.edit = gtk_button_new_with_label("Edit server");
   gtk_signal_connect(GTK_OBJECT(browse_dlg.edit), "clicked",
		      GTK_SIGNAL_FUNC(browse_dlg_edit), NULL);
   j_box_pack(browse_dlg.hbox, browse_dlg.edit, 0, 0, 0);
   
   browse_dlg.delete = gtk_button_new_with_label("Delete server");
   gtk_signal_connect(GTK_OBJECT(browse_dlg.delete), "clicked",
		      GTK_SIGNAL_FUNC(browse_dlg_delete), NULL);
   j_box_pack(browse_dlg.hbox, browse_dlg.delete, 0, 0, 0);
   
   browse_dlg.refresh = gtk_button_new_with_label("Refresh list");
   gtk_signal_connect(GTK_OBJECT(browse_dlg.refresh), "clicked",
		      GTK_SIGNAL_FUNC(browse_dlg_refresh), NULL);
   j_box_pack(browse_dlg.hbox, browse_dlg.refresh, 0, 0, 0);
   
   browse_dlg.trefresh = gtk_check_button_new_with_label(_("Remove server list before refresh"));
   config_toggle_widget(browse_dlg.trefresh, "list_refresh", NULL);
   j_box_pack(browse_dlg.hbox, browse_dlg.trefresh, 0, 0, 0);

/*   browse_dlg.hbox = j_line_new(browse_dlg.vbox, 0, 0);
   
   browse_dlg.label = gtk_label_new("Show Network: ");
   j_box_pack(browse_dlg.hbox, browse_dlg.label, 0, 0, 0);
   
   browse_dlg.show_network = gtk_combo_new();
   j_box_pack(browse_dlg.hbox, browse_dlg.show_network, 1, 1, 0);*/
   
   browse_dlg.hbox = j_line_new(browse_dlg.vbox, 1, 1);

   browse_dlg.scroller = gtk_scrolled_window_new(NULL, NULL);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(browse_dlg.scroller),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
   j_box_pack(browse_dlg.hbox, browse_dlg.scroller, 1, 1, 0);
   
   browse_dlg.ctree = gtk_ctree_new_with_titles(5, 0, titles);
   gtk_clist_set_sort_type(GTK_CLIST(browse_dlg.ctree), GTK_SORT_ASCENDING);
   gtk_clist_set_compare_func(GTK_CLIST(browse_dlg.ctree), int_sort);
   gtk_clist_set_sort_column(GTK_CLIST(browse_dlg.ctree), 4);
   gtk_clist_set_column_width(GTK_CLIST(browse_dlg.ctree), 0, 275);
   gtk_clist_set_column_width(GTK_CLIST(browse_dlg.ctree), 1, 95);
   gtk_clist_set_column_width(GTK_CLIST(browse_dlg.ctree), 2, 50);
   gtk_clist_set_column_width(GTK_CLIST(browse_dlg.ctree), 3, 70);
   gtk_signal_connect(GTK_OBJECT(browse_dlg.ctree), "button_press_event",
		      GTK_SIGNAL_FUNC(browse_dlg_button_cb), NULL);
   gtk_signal_connect(GTK_OBJECT(browse_dlg.ctree), "click-column",
		      GTK_SIGNAL_FUNC(column_sort), "ssiii");
   browse_dlg.show_network_id = -1;
   
/*   browse_dlg.ctree_mem = gtk_ctree_new(2, 0);   
   populate_browse_dlg_clist();
   network_changed(NULL, NULL);*/
   populate_browse_dlg_clist();
   
   gtk_container_add(GTK_CONTAINER(browse_dlg.scroller), browse_dlg.ctree);
   
   browse_dlg.hbox = j_line_new(browse_dlg.vbox, 0, 0);
   
   browse_dlg.label = gtk_label_new("Account: ");
   j_box_pack(browse_dlg.hbox, browse_dlg.label, 0, 0, 0);
   
   browse_dlg.account = gtk_combo_new();
   gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(browse_dlg.account)->entry), 0);
   fill_accounts(browse_dlg.account, 1);
   j_box_pack(browse_dlg.hbox, browse_dlg.account, 0, 0, 0);
   
   browse_dlg.accounts = gtk_button_new_with_label("Accounts...");
   gtk_signal_connect(GTK_OBJECT(browse_dlg.accounts), "clicked",
		      GTK_SIGNAL_FUNC(create_accounts_cb), browse_dlg.account);
   j_box_pack(browse_dlg.hbox, browse_dlg.accounts, 0, 0, 0);

/*   gtk_signal_connect(GTK_OBJECT(GTK_COMBO(browse_dlg.show_network)->entry), "changed",
		      GTK_SIGNAL_FUNC(network_changed), NULL);*/
   
   j_dialog_button_connect(browse_dlg.window, 0,
			   browse_dlg_connect_cb, NULL);
   
   j_dialog_button_connect(browse_dlg.window, 1,
			   browse_dlg_cancel_cb, NULL);
   
   gtk_signal_connect(GTK_OBJECT(browse_dlg.window), "delete_event",
		      browse_dlg_cancel_cb, NULL);

   j_dialog_show(browse_dlg.window, browse_dlg.vbox);
}

void create_refresh_dlg() {
   refresh_dlg.window = j_dialog_new(_("Refresh server list"),
				     "Button_Cancel",
				     NULL);
   
   refresh_dlg.vbox = j_dialog_vbox(refresh_dlg.window);
   
   refresh_dlg.hbox = j_line_new(refresh_dlg.vbox, 0, 0);
   
   refresh_dlg.label = gtk_label_new("Refreshing server list from\nhttp://www.napigator.com/servers.php?version=107...");
   j_box_pack(refresh_dlg.hbox, refresh_dlg.label, 0, 0, 0);
   
   j_dialog_button_connect(refresh_dlg.window, 0,
			   refresh_dlg_cb, NULL);
   
   j_dialog_show(refresh_dlg.window, refresh_dlg.vbox);
}

void create_add_dlg() {
   add_dlg.window = j_dialog_new(_("Add server"),
				 "Button_Ok",
				 "Button_Cancel",
				 NULL);

   add_dlg.vbox = j_dialog_vbox(add_dlg.window);
   
   add_dlg.hbox = j_line_new(add_dlg.vbox, 0, 0);

   add_dlg.label = gtk_label_new("IP: ");
   j_box_pack(add_dlg.hbox, add_dlg.label, 0, 0, 0);
   
   add_dlg.server = gtk_entry_new();
   j_box_pack(add_dlg.hbox, add_dlg.server, 1, 1, 0);   

   add_dlg.hbox = j_line_new(add_dlg.vbox, 0, 0);

   add_dlg.label = gtk_label_new("Port: ");
   j_box_pack(add_dlg.hbox, add_dlg.label, 0, 0, 0);
   
   add_dlg.port = gtk_entry_new();
   gtk_widget_set_usize(add_dlg.port, 40, -1);
   j_box_pack(add_dlg.hbox, add_dlg.port, 1, 1, 0);
   
   add_dlg.hbox = j_line_new(add_dlg.vbox, 0, 0);

   add_dlg.label = gtk_label_new("Description: ");
   j_box_pack(add_dlg.hbox, add_dlg.label, 0, 0, 0);
   
   add_dlg.desc = gtk_entry_new();
   j_box_pack(add_dlg.hbox, add_dlg.desc, 1, 1, 0);
   
   add_dlg.hbox = j_line_new(add_dlg.vbox, 0, 0);
   
   add_dlg.label = gtk_label_new("Network: ");
   j_box_pack(add_dlg.hbox, add_dlg.label, 0, 0, 0);
   
   add_dlg.network = gtk_entry_new();
   j_box_pack(add_dlg.hbox, add_dlg.network, 1, 1, 0);
   
   j_dialog_button_connect(add_dlg.window, 0,
			   add_dlg_cb, "ok");
   j_dialog_button_connect(add_dlg.window, 1,
			   add_dlg_cb, "cancel");

   j_dialog_show(add_dlg.window, add_dlg.vbox);
}

void create_edit_dlg(Server *nsrv, char *desc) {
   char *port;
   
   if (!nsrv || !desc) 
     return;
   
   edit_dlg.window = j_dialog_new(_("Edit server"),
				  "Button_Ok",
				  "Button_Cancel",
				  NULL);

   edit_dlg.vbox = j_dialog_vbox(edit_dlg.window);
   
   edit_dlg.hbox = j_line_new(edit_dlg.vbox, 0, 0);

   edit_dlg.label = gtk_label_new("IP: ");
   j_box_pack(edit_dlg.hbox, edit_dlg.label, 0, 0, 0);
   
   edit_dlg.server = gtk_entry_new();
   gtk_entry_set_text(GTK_ENTRY(edit_dlg.server), nsrv->ip);
   j_box_pack(edit_dlg.hbox, edit_dlg.server, 1, 1, 0);   

   edit_dlg.hbox = j_line_new(edit_dlg.vbox, 0, 0);

   edit_dlg.label = gtk_label_new("Port: ");
   j_box_pack(edit_dlg.hbox, edit_dlg.label, 0, 0, 0);
   
   edit_dlg.port = gtk_entry_new();
   d_msprintf(&port, "%i", nsrv->port);
   gtk_entry_set_text(GTK_ENTRY(edit_dlg.port), port);
   d_free(port);
   gtk_widget_set_usize(edit_dlg.port, 40, -1);
   j_box_pack(edit_dlg.hbox, edit_dlg.port, 1, 1, 0);
   
   edit_dlg.hbox = j_line_new(edit_dlg.vbox, 0, 0);

   edit_dlg.label = gtk_label_new("Description: ");
   j_box_pack(edit_dlg.hbox, edit_dlg.label, 0, 0, 0);
   
   edit_dlg.desc = gtk_entry_new();
   gtk_entry_set_text(GTK_ENTRY(edit_dlg.desc), desc);
   j_box_pack(edit_dlg.hbox, edit_dlg.desc, 1, 1, 0);

   edit_dlg.hbox = j_line_new(edit_dlg.vbox, 0, 0);
   
   edit_dlg.label = gtk_label_new("Network: ");
   j_box_pack(edit_dlg.hbox, edit_dlg.label, 0, 0, 0);
   
   edit_dlg.network = gtk_entry_new();
   gtk_entry_set_text(GTK_ENTRY(edit_dlg.network), nsrv->network);
   j_box_pack(edit_dlg.hbox, edit_dlg.network, 1, 1, 0);
   
   j_dialog_button_connect(edit_dlg.window, 0,
			   edit_dlg_cb, nsrv);
   j_dialog_button_connect(edit_dlg.window, 1,
			   edit_dlg_cb, NULL);
   
   j_dialog_show(edit_dlg.window, edit_dlg.vbox);
}

void create_list_chan_dlg() {
   char *titles[] = { "Channel", "Users", "Topic", NULL };
   
   list_chan_dlg.window = j_dialog_new(_("Channel Listing"),
				       "Join",
				       "Button_Ok",
				       NULL);
   
   gtk_widget_set_usize(list_chan_dlg.window, 400 + 150, 300 + 80);
   
   list_chan_dlg.vbox = j_dialog_vbox(list_chan_dlg.window);
   
   list_chan_dlg.hbox = j_line_new(list_chan_dlg.vbox, 0, 0);
   
   list_chan_dlg.label = gtk_label_new(_("Requesting channel listing..."));
   j_box_pack(list_chan_dlg.hbox, list_chan_dlg.label, 0, 0, 0);
   
   list_chan_dlg.hbox = j_line_new(list_chan_dlg.vbox, 1, 1);

   list_chan_dlg.scroller = gtk_scrolled_window_new(NULL, NULL);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(list_chan_dlg.scroller),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
   j_box_pack(list_chan_dlg.hbox, list_chan_dlg.scroller, 1, 1, 0);

   list_chan_dlg.clist = gtk_clist_new_with_titles(3, titles);
   gtk_clist_set_column_width(GTK_CLIST(list_chan_dlg.clist), 0, 100);
   gtk_clist_set_column_width(GTK_CLIST(list_chan_dlg.clist), 2, 50);
   gtk_clist_set_column_width(GTK_CLIST(list_chan_dlg.clist), 3, 150);
   gtk_signal_connect(GTK_OBJECT(list_chan_dlg.clist), "button_press_event",
		      GTK_SIGNAL_FUNC(list_chan_dlg_clist_cb), NULL);
   gtk_signal_connect(GTK_OBJECT(list_chan_dlg.clist), "click-column",
		      GTK_SIGNAL_FUNC(column_sort), "sis");
   gtk_container_add(GTK_CONTAINER(list_chan_dlg.scroller), list_chan_dlg.clist);
   
   j_dialog_button_connect(list_chan_dlg.window, 0,
			   list_chan_dlg_cb, "join");
   j_dialog_button_connect(list_chan_dlg.window, 1,
			   list_chan_dlg_cb, "ok");
   
   j_dialog_show(list_chan_dlg.window, list_chan_dlg.vbox);
}

void create_resume_dlg() {
   char *titles[] = { "Filename", "Remote size", NULL };
   
   resume_dlg.window = j_dialog_new(_("Resume Download"),
				    "Resume",
				    "Remove",
				    "Button_Cancel",
				    NULL);
   
   gtk_widget_set_usize(resume_dlg.window, 400, 300);
   
   resume_dlg.vbox = j_dialog_vbox(resume_dlg.window);

   resume_dlg.hbox = j_line_new(resume_dlg.vbox, 0, 0);
   
   resume_dlg.label = gtk_label_new(_(""));
   j_box_pack(resume_dlg.hbox, resume_dlg.label, 0, 0, 0);
   
   resume_dlg.hbox = j_line_new(resume_dlg.vbox, 1, 1);

   resume_dlg.scroller = gtk_scrolled_window_new(NULL, NULL);
   gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(resume_dlg.scroller),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
   j_box_pack(resume_dlg.hbox, resume_dlg.scroller, 1, 1, 0);

   resume_dlg.clist = gtk_clist_new_with_titles(2, titles);
   gtk_clist_set_column_width(GTK_CLIST(resume_dlg.clist), 0, 275);
   gtk_clist_set_column_width(GTK_CLIST(resume_dlg.clist), 2, 50);
   gtk_signal_connect(GTK_OBJECT(resume_dlg.clist), "button_press_event",
		      GTK_SIGNAL_FUNC(resume_dlg_clist_cb), NULL);
   gtk_container_add(GTK_CONTAINER(resume_dlg.scroller), resume_dlg.clist);

   gtk_signal_connect(GTK_OBJECT(resume_dlg.window), "delete_event",
		      resume_dlg_cb, "ok");

   j_dialog_button_connect(resume_dlg.window, 0,
			   resume_dlg_cb, "resume");
   j_dialog_button_connect(resume_dlg.window, 1,
			   resume_dlg_cb, "remove");
   j_dialog_button_connect(resume_dlg.window, 2,
			   resume_dlg_cb, "ok");
   
   j_dialog_show(resume_dlg.window, resume_dlg.vbox);
}
