/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */
using namespace std;

////// Ideas for fixing this mess: 
// - Don't have invisible columns; pkgtree.cc should just not 
//   insert them.

#include "drawtree.h"
#include <algorithm>
#include <gdk/gdkkeysyms.h>
#include <math.h>
#include "gnome-apt.h"


// code from CTree
#define PM_SIZE 8
#define PM_INSET 3
#define EXPANDER_WIDTH (PM_SIZE+PM_INSET*2)

//////// Callback clutter

gint DrawTree::configure_event(GtkWidget* w, GdkEventConfigure* event, gpointer data)
{
  DrawTree* dt = static_cast<DrawTree*>(data);

  g_return_val_if_fail(data, FALSE);

  return dt->configure(event);
}


gint DrawTree::expose_event(GtkWidget* w, GdkEventExpose* event, gpointer data)
{
  DrawTree* dt = static_cast<DrawTree*>(data);

  g_return_val_if_fail(data, FALSE);

  return dt->expose(event);
}


gint DrawTree::key_event(GtkWidget* w, GdkEventKey* event, gpointer data)
{
  DrawTree* dt = static_cast<DrawTree*>(data);

  g_return_val_if_fail(data, FALSE);

  // Hack this out for now; the main app window manually passes
  //  us all key events.
  return FALSE; 

  return dt->key(event);
}


gint DrawTree::button_event(GtkWidget* w, GdkEventButton* event, gpointer data)
{
  DrawTree* dt = static_cast<DrawTree*>(data);

  g_return_val_if_fail(data, FALSE);

  return dt->button(event);
}


gint DrawTree::motion_event(GtkWidget* w, GdkEventMotion* event, gpointer data)
{
  DrawTree* dt = static_cast<DrawTree*>(data);

  g_return_val_if_fail(data, FALSE);

  return dt->motion(event);
}

gint DrawTree::leave_event(GtkWidget* w, GdkEventCrossing* event, gpointer data)
{
  DrawTree* dt = static_cast<DrawTree*>(data);

  g_return_val_if_fail(data, FALSE);

  return dt->leave(event);
}

void DrawTree::realize_cb(GtkWidget* w, gpointer data)
{
  DrawTree* dt = static_cast<DrawTree*>(data);

  g_return_if_fail(data);

  dt->realize();
}


//////////// End static callbacks


////////// Begin method callbacks

void 
DrawTree::realize()
{
  g_return_if_fail(da_->window != 0);
  g_return_if_fail(da_->style != 0);

  // Immediately set background, this just improves cosmetics -
  // otherwise you can see memory garbage for an instant while we
  // complete the pixmap draw

  gdk_window_set_background (da_->window, 
                             &da_->style->base[GTK_STATE_NORMAL]);

}

gint 
DrawTree::configure(GdkEventConfigure* event)
{
  if (da_->allocation.width == pixwidth_ && 
      da_->allocation.height == pixheight_) return FALSE; // nothing to do

  if (pixmap_ != 0) {
    gdk_pixmap_unref(pixmap_);
  }
 
  pixwidth_ = da_->allocation.width;
  pixheight_ = da_->allocation.height;

  pixmap_ = gdk_pixmap_new(da_->window,
                           pixwidth_,
                           pixheight_,
                           -1);
		   
  recalc_widths(); 
  recalc_rows();

  setup_adjustments(); // page size will have changed

  queue_display_update();

  return TRUE;
}

gint 
DrawTree::expose(GdkEventExpose* event)
{
  gdk_draw_pixmap(da_->window,
                  da_->style->fg_gc[GTK_WIDGET_STATE (da_)],
                  pixmap_,
                  event->area.x, event->area.y,
                  event->area.x, event->area.y,
                  event->area.width, event->area.height);

  return FALSE;
}


gint 
DrawTree::button(GdkEventButton* event)
{
  gint r;

  switch (event->type) {
  case GDK_BUTTON_PRESS:
    g_return_val_if_fail (resize_ == 0, FALSE);
    switch (event->button) {
    case 1:
      {
        //    scroll_by(-1);
        r = row_from_coords(static_cast<gint>(event->x), 
                            static_cast<gint>(event->y));
        
        if (r >= 0) 
          {
            gint vrn = r-row_;
            g_assert(static_cast<size_t>(vrn) < visible_nodes_.size());
            
            change_selection(visible_nodes_[vrn]);
            
            gint col = column_from_x(static_cast<gint>(event->x));

            if (col >= 0) 
              {
                if (columns_[col]->style == BoolCol) 
                  {
                    if (visible_nodes_[vrn]->display_bool(col)) 
                      {
                        visible_nodes_[vrn]->set_bool(col,
                                                      !visible_nodes_[vrn]->get_bool(col));
                      }
                  }
                else if (columns_[col]->style == TreeCol)
                  {
                    // see if we are on the expander - yes these magic 
                    //  numbers should not be cut-and-pasted all over this
                    //  file. bleh.
                    gint start = columns_[col]->x + depths_[vrn]*indent_;
                    gint end = start + EXPANDER_WIDTH;
                    
                    if (static_cast<gint>(event->x) >= start && 
                        static_cast<gint>(event->x) <= end) 
                      {
                        return openclose_node(visible_nodes_[vrn]);
                      }
                  }
              }
          }
        else
          {
            change_selection(0);

            // Perhaps we clicked a column
            // First assemble vector of visible column indices
            vector<Column*>::iterator ci = columns_.begin();
            gushort col = 0;
            vector<gushort> visible;
            while (ci != columns_.end()) 
              {
                if ((*ci)->visible) 
                  {
                    visible.push_back(col);
                  }
                ++ci;
                ++col;
              }

            vector<gushort>::const_iterator vi = visible.begin();
            vector<gushort>::const_iterator prev_vi = visible.end();
            while (vi != visible.end())
              {
                Column* c = columns_[*vi];
                col = *vi;         // variable recycling, evil
                

                GdkRectangle crect;
                crect.x = c->x;
                crect.y = margin_;
                crect.width = c->width + c->extra;
                crect.height = titles_height_;
                
                if (col != 0 && 
                    abs(static_cast<gint>(event->x) - crect.x) <= 4)
                  {
                    g_return_val_if_fail(resize_ == 0, true);
                    
                    gint prev = -1;
                    
                    // Find a visible column to the left
                    if (prev_vi != visible.end())
                      prev = *prev_vi;
                    
                    // If none, we can't resize this
                    // Also check that the next column isn't a 
                    //  better match
                    if (prev >= 0 &&
                        !((event->x > (gint)crect.x) && 
                          (crect.width < 3) && 
                          (vi+1 != visible.end())))
                      {
                        resize_ = new Resize(prev, col);
                        
                        resize_->xor_gc_ = gdk_gc_new(da_->window);
                        gdk_gc_copy(resize_->xor_gc_, da_->style->white_gc);
                        
                        gdk_gc_set_function(resize_->xor_gc_, GDK_XOR);
                        
                        GdkEventMask emask = 
                          static_cast<GdkEventMask>(GDK_POINTER_MOTION_MASK | GDK_BUTTON_RELEASE_MASK);
                        
                        gdk_pointer_grab (da_->window,
                                          FALSE,
                                          emask,
                                          NULL,
                                          NULL,
                                          event->time);
                        
                        GdkCursor* cursor = 
                          gdk_cursor_new (GDK_SB_H_DOUBLE_ARROW);
                          
                        gdk_window_set_cursor(da_->window,
                                              cursor);
                          
                        gdk_cursor_destroy(cursor);
                          
                        draw_resize_line();
                          
#ifdef GNOME_ENABLE_DEBUG
                        ga_debug("Starting resize between col %u and %u\n",
                                resize_->left_, resize_->right_);
#endif
                        return true;
                      }  // If there's a previous
                  } // if it's on the inter-column line
                prev_vi = vi;
                ++vi;
              } // while iterating over visible columns
          } // else if we're resizing

      } // end of switch
      return true;
      break;
    case 2:
      break;
    default:
      break;
    }
    break;
    
  case GDK_2BUTTON_PRESS:
    g_return_val_if_fail (resize_ == 0, FALSE);
    switch (event->button) {
      
    case 1:
      r = row_from_coords(static_cast<gint>(event->x), 
                          static_cast<gint>(event->y));

      if (r >= 0) {
        gint vrn = r-row_;

        return openclose_node(visible_nodes_[vrn]);
      }
      break;

    default:
      break;
    }
    break;
    
  case GDK_BUTTON_RELEASE:
    {
      if (resize_ == 0) return false;
      if (event->button != 1) return false;
    
      // We are resizing and we just stopped
      draw_resize_line(); // erase
      gdk_pointer_ungrab(event->time);
      gdk_window_set_cursor(da_->window,
                            NULL);
                      
      delete resize_;
      resize_ = 0;
      queue_display_update();
    }
    break;

  default:
    break;
  }
    
  return false;
}

gint 
DrawTree::consider_key_event(GdkEventKey* event)
{
  return key(event);
}

gint 
DrawTree::key(GdkEventKey* event)
{
  switch (event->keyval) {
  case GDK_Up:
    move_selection(-1);
    return true;
    break;
  case GDK_Down:
    move_selection(1);
    return true;
    break;
  case GDK_Return:
    if (selected_node_ != 0)
      openclose_node(selected_node_);
    return true;
    break;
  case GDK_Page_Down:
    if (visible_nodes_.size() > 1)
      {
        scroll_by(int(visible_nodes_.size()) - 2);
        change_selection(0);
      }
    break;
  case GDK_Page_Up:
    if (visible_nodes_.size() > 1)
      {
        scroll_by(-(int(visible_nodes_.size()) - 2));
        change_selection(0);
      }
    break;
  default:
    break;
  }

  if (selected_node_) 
    {
      guint col = 0;
      while (col < columns_.size()) 
        {
          if (columns_[col]->style == BoolCol) 
            {
              if (event->keyval == columns_[col]->shortcut1 ||
                  event->keyval == columns_[col]->shortcut2)
                {
                  if (selected_node_->display_bool(col)) 
                    {
                      selected_node_->set_bool(col,
                                               !selected_node_->get_bool(col));
                      return true;
                    }
                }
            }
          ++col;
        }
    }

  return false;
}

gint 
DrawTree::motion(GdkEventMotion* event)
{
#if 0
  // This should be in enter_notify, I'm just lazy for now
  if (!GTK_WIDGET_HAS_FOCUS(da_)) 
    gtk_widget_grab_focus(da_);
#endif

  gint x, y;
  GdkModifierType state;
  
  if (event->is_hint)
    gdk_window_get_pointer (event->window, &x, &y, &state);
  else {
    x = static_cast<gint>(event->x);
    y = static_cast<gint>(event->y);
    state = static_cast<GdkModifierType>(event->state);
  }

  if (resize_ == 0)
    {
      gint candidate = row_from_coords(x,y);
      
      if (candidate == prelight_) return false; // already done.
      
      gint oldprelight = prelight_;
      
      if ( candidate >= 0 && 
           (static_cast<gint>(candidate - row_) != row_index(selected_node_))) {
        
        prelight_ = candidate;
        
        update_row(candidate-row_, GTK_STATE_PRELIGHT);
      }
      else {
        prelight_ = -1;
      }
      
      if ((prelight_ != oldprelight) && 
          (oldprelight >= 0) && 
          (oldprelight >= static_cast<gint>(row_))) {
        update_row(oldprelight - row_, 
                   (static_cast<gint>(oldprelight - row_) == row_index(selected_node_)) 
                   ? GTK_STATE_SELECTED : GTK_STATE_NORMAL);
      }
      
    }
  else
    {
      g_return_val_if_fail(resize_->left_ < resize_->right_, FALSE);
      g_return_val_if_fail(resize_->right_ < columns_.size(), FALSE);

      Column* left = columns_[resize_->left_];
      Column* right = columns_[resize_->right_];

      draw_resize_line(); // erase

      gint newwidth = x - left->x;
      
      if (newwidth < 0)
        {
          newwidth = 0;
        }

      left->width = newwidth;

      recalc_widths();
      draw_resize_line();
      return true;
    }

  return false;
}

gint 
DrawTree::leave(GdkEventCrossing* event)
{
  if ((prelight_ >= 0) && 
      (prelight_ >= static_cast<gint>(row_)) &&
      (static_cast<gint>(prelight_ - row_) != row_index(selected_node_))) {
    update_row(prelight_ - row_, GTK_STATE_NORMAL);
    prelight_ = -1;
  }
  return false;
}

void
DrawTree::draw_expander(GdkRectangle* cell_rect, bool expanded)
{
  GdkPoint points[3];

  gint y = cell_rect->y + (cell_rect->height - PM_SIZE) / 2;
            //           - (cell_rect->height + 1) % 2);
  gint x = cell_rect->x + PM_INSET;

  gdk_draw_rectangle (pixmap_,
                      da_->style->base_gc[GTK_STATE_NORMAL], TRUE,
                      x, y, PM_SIZE, PM_SIZE);
  gdk_draw_rectangle (pixmap_,
                      da_->style->fg_gc[GTK_STATE_NORMAL], FALSE,
                      x, y, PM_SIZE, PM_SIZE);

  gdk_draw_line (pixmap_, da_->style->fg_gc[GTK_STATE_NORMAL], 
                 x + 2, 
                 y + PM_SIZE / 2, 
                 x + PM_SIZE - 2, 
                 y + PM_SIZE / 2);

  if (!expanded)
	gdk_draw_line (pixmap_, da_->style->fg_gc[GTK_STATE_NORMAL],
                   x + PM_SIZE / 2, y + 2,
                   x + PM_SIZE / 2, y + PM_SIZE - 2);

}

void
DrawTree::draw_resize_line()
{
  g_return_if_fail(resize_ != 0);

  Column* left = columns_[resize_->left_];

  // We draw at the real width, not the "padded with extra" 
  //  width, because the padded width creates a weird 
  //  user interaction experience (line not aligned with pointer)
  gdk_draw_line(da_->window,
                resize_->xor_gc_,
                left->x + left->width,
                0,
                left->x + left->width,
                da_->allocation.height);

  
#ifdef GNOME_ENABLE_DEBUG
  ga_debug("Resize line widths: ");

  vector<Column*>::const_iterator q = columns_.begin();
  while (q != columns_.end()) 
    {

      ga_debug("%d ", (*q)->width);

      ++q;
    }

  ga_debug("\n");
#endif
}

// FIXME it would be best to eliminate the 'state' argument,
//  and just check prelight_ and selected_node_

void 
DrawTree::update_row(guint vrn, GtkStateType state, bool copy_to_window)
{
  // This can happen if we get an event between cache swaps
  if (vrn >= visible_nodes_.size()) return; 

  if (update_queued_) return; // we are going to redraw everything anyway

  GdkRectangle rect;
  rect.x = margin_;
  rect.y = row_offset(row_ + vrn);
  rect.width = total_width_ - margin_*2;
  rect.height = row_height_;

  GdkGC* bg_gc = 0;
  if (state == GTK_STATE_NORMAL)
    bg_gc = da_->style->base_gc[state];
  else
    bg_gc = da_->style->bg_gc[state];

  gdk_draw_rectangle (pixmap_,
                      bg_gc,
                      TRUE,
                      rect.x, 
                      rect.y, 
                      rect.width, 
                      rect.height);

  /* Draw each label on the row */
  vector<Column*>::iterator ci = columns_.begin();
  gushort remaining = columns_.size();
  gushort col = 0;
  while (ci != columns_.end()) {
    --remaining;
    
    gint inset = 0;
    if ((*ci)->style == TreeCol)
      {
        inset = depths_[vrn] * indent_;
      }

    if ((*ci)->visible) 
      {
        
        GdkRectangle crect;
        crect.x = (*ci)->x + inset;
        crect.y = row_offset(vrn + row_) + margin_;
        crect.width = (*ci)->width + (*ci)->extra;
        crect.height = row_height_ - margin_*2;
        
        if ((*ci)->style == BoolCol) 
          {
            
            if (visible_nodes_[vrn]->display_bool(col)) {
              
              GtkStateType checkstate = state;
              if (checkstate == GTK_STATE_SELECTED) {
                // you can't select the buttons (you should
                //  be able to prelight them though, this 
                //  is kind of broken.)
                checkstate = GTK_STATE_NORMAL;
              }
              
              static const gint checkbutton_size = 10;
              
              // we really want "whatever this theme uses for 
              //  check items" but hey
              GtkStyle* default_style = gtk_widget_get_default_style();
              
              crect.x += (MAX(crect.width - checkbutton_size, 0))/2;
              crect.y += (MAX(crect.height - checkbutton_size, 0))/2; 
              crect.width = MIN(crect.width, checkbutton_size);
              crect.height = MIN(crect.height, checkbutton_size);
              
#if 0
              gtk_paint_flat_box(default_style, 
                                 pixmap_, 
                                 checkstate, 
                                 GTK_SHADOW_ETCHED_OUT, 
                                 0, 
                                 da_, 
                                 "radiobutton",
                                 crect.x, 
                                 crect.y,
                                 crect.width, 
                                 crect.height);
#endif
              
              GtkShadowType shadow_type;
              
              if (visible_nodes_[vrn]->get_bool(col))
                shadow_type = GTK_SHADOW_IN;
              else
                shadow_type = GTK_SHADOW_OUT;
              
              gtk_paint_option (default_style, 
                                pixmap_,
                                checkstate,
                                shadow_type,
                                0,
                                da_, 
                                "radiobutton",
                                crect.x + 1, 
                                crect.y + 1,
                                crect.width-1, 
                                crect.height-1);
            }
          }
        else {        
          if ((*ci)->style == TreeCol)
            {
              // leave space for expander, even if we don't draw it.
              static const gint expander_width = PM_SIZE + PM_INSET*2;

              if (visible_nodes_[vrn]->expandable())
                {
                  draw_expander(&crect, visible_nodes_[vrn]->expanded());
                }

              crect.x += expander_width;
              crect.width -= expander_width;
            }

          visible_nodes_[vrn]->draw_column(col, remaining,
                                           state,
                                           pixmap_,
                                           &crect);
        }
      }
    
    ++ci;
    ++col;
  }

  // If there's a full update queued, we don't need to 
  //  honor copy_to_window here. Avoids weird flicker.
  // (now redundant since there's a check at the beginning)
  if (copy_to_window && update_queued_ == false) 
    {  
      gdk_draw_pixmap(da_->window,
                      da_->style->fg_gc[GTK_WIDGET_STATE (da_)],
                      pixmap_,
                      rect.x,rect.y,rect.x,rect.y,
                      rect.width, 
                      rect.height);
      
    }  
}

// fixme, make it a function
#define toprow_y() (margin_*2 + titles_height_)

gint 
DrawTree::row_from_coords(gint x, gint y)
{
  gint onerow = row_height_ + row_gap_;
  
  if (toprow_y() > y) return -1; 

  guint n_above = (y - toprow_y()) / onerow;

  if (visible_nodes_.size() <= n_above) return -1; // we're off the bottom
  
  guint candidate = row_ + n_above;

  // compute row rectangle
  GdkRectangle rect;
  rect.x = margin_;
  rect.y = row_offset(candidate);
  rect.width = total_width_ - margin_*2;
  rect.height = row_height_;

  if ( (y > rect.y) &&
       (y < rect.y + rect.height) &&
       (x > rect.x) &&
       (x < rect.x + rect.width) ) {
    return candidate;
  }
  else return -1;
}

gint 
DrawTree::column_from_x(gint x)
{
  vector<Column*>::iterator j = columns_.begin();

  gint last_col = -1;

  while (j != columns_.end()) {
    if ( ! (*j)->visible) 
      ++j;
    else if ((*j)->x > x) 
      break;
    else 
      {
        ++last_col;
        ++j;
      }
  }
  
  return last_col;
}

// -1 for not visible
gint 
DrawTree::row_index(Node* n)
{
  // optimize some common cases
  if (n == 0) return -1;
  if (n->hidden()) return -1;

  // This is ridiculously inefficient O(n) crap, 
  //  but works for now. never more than 50 or so items
  //  in the vector.
  
  size_t end = visible_nodes_.size();
  size_t i = 0;
  while (i < end) {
    if (visible_nodes_[i] == n) return i;
    ++i;
  }
  return -1;
}

void 
DrawTree::change_selection(Node* newnode)
{
  if (newnode == selected_node_) return;

  if (selected_node_ != 0) {
    gint oldrow = row_index(selected_node_);
    if (oldrow >= 0) {
      update_row(oldrow, 
                 (oldrow == prelight_) ? GTK_STATE_PRELIGHT : GTK_STATE_NORMAL);
    }
    selected_node_->unselect();
    selected_node_ = 0;
  }

  selected_node_ = newnode;

  if (selected_node_ != 0) {
    gint newrow = row_index(selected_node_);
    if (newrow >= 0) {
      update_row(newrow,GTK_STATE_SELECTED);      
    }
    selected_node_->select();
  }
}

void 
DrawTree::change_selection(gint x, gint y)
{
  gint r = row_from_coords(x, y);
  
  if (r >= 0) 
    {
      gint vrn = r-row_;
      g_assert(static_cast<size_t>(vrn) < visible_nodes_.size());
      
      change_selection(visible_nodes_[vrn]);
    }
}

void 
DrawTree::focus_node(Node* node)
{
  // This is all kinds of hacky magic variable crap,
  //  and gratuitously slow to boot
  focus_node_ = node;
  recalc_rows(); // this sets focus_index_
  focus_node_ = 0;
  gint scroll = gint(focus_index_) - gint(row_);
  focus_index_ = 0;

  scroll_by(scroll);
}

bool
DrawTree::openclose_node(Node* node)
{
  g_return_val_if_fail(node != 0, false);

  bool changed = false;

  if (node->expandable()) {
    if (node->expanded()) {
      node->collapse();
      changed = true;
    }
    else {
      node->expand();
      changed = true;
    }
  }

  if (changed)
    {
      queue_recalc_rows();
      queue_display_update();
    }

  return changed;
}

// Man, this DrawTree is becoming a mess. 
//  Ugh.
void
DrawTree::move_selection(gint steps)
{
  g_return_if_fail(steps != 0);

  if (selected_node_ == 0)
    {
      if (visible_nodes_.empty()) 
        return;
      else 
        {
          // select something so the next move does
          //  a more sensible action
          guint middle = visible_nodes_.size()/2;
          change_selection(visible_nodes_[middle]);
          return;
        }      
      g_assert_not_reached();
    }
  else 
    {
      gint onscreen = row_index(selected_node_);

      if (onscreen < 0) 
        {
          // get it onscreen so it works next
          // time.
          focus_node(selected_node_);
          return;
        }
      else
        {
          if (steps < 0)
            {
              // check for going off screen
              if (onscreen < (- steps))
                {
                  if (row_ == 0) 
                    return;
                  else
                    {
                      scroll_by(onscreen + steps);
                      // force, don't wait for queue
                      recalc_rows(); 
                      if (!visible_nodes_.empty())
                        change_selection(visible_nodes_[0]);
                      
                      return;
                    }
                }
            }
          else // steps > 0 
            {
              gint from_bottom = visible_nodes_.size() - onscreen - 1;
              // check for going off screen
              if (from_bottom <= steps)
                {
                  scroll_by(steps);
                  recalc_rows();
                  if (!visible_nodes_.empty())
                    {
                      vector<Node*>::iterator i = visible_nodes_.end();
                      --i;
                      change_selection(*i);
                    }
                  return;
                }
            }

          // if we didn't return, we're moving to an on-screen
          //  location.

          gint newindex = onscreen + steps;
          
          g_return_if_fail(newindex >= 0);
          g_return_if_fail(((guint)newindex) < visible_nodes_.size());

          change_selection(visible_nodes_[newindex]);
          return;
        }
    }
}

DrawTree::Node* 
DrawTree::selected_node()
{
  return selected_node_;
}

DrawTree::DrawTree()
  : da_(0), pixmap_(0),
    update_queued_(false),
    row_gap_(1),
    col_gap_(1),
    row_height_(20),
    indent_(20),
    margin_(1),
    titles_height_(22),
    nodes_(0),
    focus_node_(0),
    focus_index_(0),
    total_width_(0),
    recalc_rows_queued_(false),
    row_(0),
    nrows_(0),
    selected_node_(0),
    prelight_(-1),
    gc_(0),
    font_(0),
    hadjustment_(0),
    vadjustment_(0),
    pixwidth_(0),
    pixheight_(0),
    resize_(0),
    update_idle_id_(0),
    recalc_idle_id_(0)
{
  da_ = gtk_drawing_area_new(); 

  GTK_WIDGET_SET_FLAGS (da_, GTK_CAN_FOCUS);
  
  gtk_widget_set_events ( da_, GDK_EXPOSURE_MASK
                          | GDK_BUTTON_PRESS_MASK 
                          | GDK_BUTTON_RELEASE_MASK
                          | GDK_KEY_PRESS_MASK
                          | GDK_POINTER_MOTION_MASK
                          | GDK_POINTER_MOTION_HINT_MASK
                          | GDK_LEAVE_NOTIFY_MASK);


  // other construction happens in set_nodelist(),
  //  so we don't get any signals with NULL nodes_
}

void 
DrawTree::set_nodelist(NodeList* nl) { 
  g_return_if_fail(nodes_ == 0);
  nodes_ = nl;

  gtk_signal_connect(GTK_OBJECT(da_),
                     "realize",
                     GTK_SIGNAL_FUNC(realize_cb),
                     this);
  gtk_signal_connect(GTK_OBJECT(da_),
                     "configure_event",
                     GTK_SIGNAL_FUNC(configure_event),
                     this);
  gtk_signal_connect(GTK_OBJECT(da_),
                     "expose_event",
                     GTK_SIGNAL_FUNC(expose_event),
                     this);
  gtk_signal_connect(GTK_OBJECT(da_),
                     "button_press_event",
                     GTK_SIGNAL_FUNC(button_event),
                     this);

  gtk_signal_connect(GTK_OBJECT(da_),
                     "button_release_event",
                     GTK_SIGNAL_FUNC(button_event),
                     this);

  // after so we can handle the key event before the window
  gtk_signal_connect_after(GTK_OBJECT(da_),
                     "key_press_event",
                     GTK_SIGNAL_FUNC(key_event),
                     this);
  gtk_signal_connect(GTK_OBJECT(da_),
                     "motion_notify_event",
                     GTK_SIGNAL_FUNC(motion_event),
                     this);
  gtk_signal_connect(GTK_OBJECT(da_),
                     "leave_notify_event",
                     GTK_SIGNAL_FUNC(leave_event),
                     this);
}


DrawTree::~DrawTree()
{
  if (pixmap_) gdk_pixmap_unref(pixmap_);

  vector<Column*>::iterator i = columns_.begin();
  while (i != columns_.end()) {
    delete *i;
    ++i;
  }

  if (recalc_idle_id_ != 0)
    gtk_idle_remove(recalc_idle_id_);

  if (update_idle_id_ != 0)
    gtk_idle_remove(update_idle_id_);

  // assume drawing area is in a container.

  // don't touch any nodes; NodeList may well be destroyed
}

gint 
DrawTree::update_idle(gpointer data)
{
  DrawTree* tree = static_cast<DrawTree*>(data);
  g_return_val_if_fail(tree != 0, FALSE);

  tree->do_display_update();

  tree->update_idle_id_ = 0;

  return FALSE; // remove idle
}

void 
DrawTree::queue_display_update()
{
  if (!update_queued_) { 
    update_idle_id_ = gtk_idle_add(update_idle, this); 
    update_queued_ = true;
  }
}


gint 
DrawTree::recalc_rows_idle(gpointer data)
{
  DrawTree* tree = static_cast<DrawTree*>(data);
  g_return_val_if_fail(tree != 0, FALSE);

  // If recalc_rows got called in between adding the idle
  //  and calling the idle, just return. There's a race
  //  condition with display update that can cause this.
  if (!tree->recalc_rows_queued_) return FALSE; 

  tree->recalc_rows();

  tree->recalc_idle_id_ = 0;

  return FALSE; // remove idle
}

void 
DrawTree::queue_recalc_rows()
{
  if (!recalc_rows_queued_) { 
    recalc_idle_id_ = gtk_idle_add(recalc_rows_idle, this); 
    recalc_rows_queued_ = true;
  }
}

void 
DrawTree::do_display_update()
{
  update_queued_ = false;

  if (recalc_rows_queued_) {
    // This saves us from a race condition where we get here without
    // recalcing the rows.
    recalc_rows();
  }

  gc_ = da_->style->black_gc;
  font_ = da_->style->font;

  GdkRectangle rect;
  rect.x = 0;
  rect.y = 0;
  rect.width = da_->allocation.width;
  rect.height = da_->allocation.height;

  // draw background
  gdk_draw_rectangle (pixmap_,
                      da_->style->base_gc[GTK_STATE_NORMAL],
                      TRUE,
                      rect.x, rect.y, rect.width, rect.height);

  // draw column titles

  vector<Column*>::iterator ci = columns_.begin();
  while (ci != columns_.end()) {    
    if ((*ci)->visible) {

      GdkRectangle crect;
      crect.x = (*ci)->x;
      crect.y = margin_;
      crect.width = (*ci)->width + (*ci)->extra;
      crect.height = titles_height_;
      
      gtk_paint_box(da_->style, pixmap_, 
                    static_cast<GtkStateType>(GTK_WIDGET_STATE(da_)),
                    GTK_SHADOW_OUT, 
                    0, 
                    da_, 
                    "apt_titles",
                    crect.x, 
                    crect.y,
                    crect.width, 
                    crect.height);
      
      // leave an edge around the string
      crect.x += 2;
      crect.y += 2;
      crect.width -= 4;
      crect.height -= 4;

      gtk_paint_string(da_->style,
                       pixmap_,
                       static_cast<GtkStateType>(GTK_WIDGET_STATE(da_)),
                       &crect,
                       da_,
                       "apt_titles",
                       crect.x,
                       static_cast<gint>(crect.y + (crect.height/1.6)),
                       (*ci)->name.c_str());

    }
    
    ++ci;
  }

  // Draw rows 
  
  g_assert(visible_nodes_.size() == depths_.size());

  size_t ni = 0;
  while (ni < visible_nodes_.size()) {
    GtkStateType nstate = GTK_STATE_NORMAL;
    if (selected_node_ == visible_nodes_[ni]) {
      nstate = GTK_STATE_SELECTED;
    }
    else if (prelight_ == static_cast<gint>(ni+row_)) {
      nstate = GTK_STATE_PRELIGHT;
    }
    update_row(ni,
               nstate,
               false);
    ++ni;
  }

  // copy to window
  gdk_draw_pixmap(da_->window,
                  da_->style->fg_gc[GTK_WIDGET_STATE (da_)],
                  pixmap_,
                  rect.x,rect.y,rect.x,rect.y,
                  rect.width, 
                  rect.height);
}

void 
DrawTree::set_visible(gushort col, bool visibility) 
{ 
  g_return_if_fail(col < columns_.size());
  columns_[col]->visible = visibility;
  recalc_widths();
  queue_display_update();
}

bool
DrawTree::is_visible(gushort col) const
{
  g_return_val_if_fail(col < columns_.size(), false);
  return columns_[col]->visible;
}

void
DrawTree::set_columns(const vector<ColumnStyle> & cols, 
                      const vector<string> & column_names,
                      const vector<guint> & shortcuts1,
                      const vector<guint> & shortcuts2)
{
  g_return_if_fail(cols.size() == column_names.size());

  // shortcuts2 is now a relic, we ignore it.

  vector<Column*>::iterator j = columns_.begin();
  while (j != columns_.end()) {
    delete *j;
    ++j;
  }
  
  columns_.clear();

  columns_.reserve(cols.size());

  vector<ColumnStyle>::const_iterator i = cols.begin();
  vector<string>::const_iterator      k = column_names.begin();
  vector<guint>::const_iterator s1 = shortcuts1.begin();
  while (i != cols.end()) {
    Column* c = new Column;
    
    c->style = *i;
    
    c->visible = true;

    // for all but bool
    c->shortcut1 = '\0';
    c->shortcut2 = '\0'; 

    c->width = 5;

    // defaults 
    switch (c->style) {
    case BoolCol:
      c->shortcut1 = *s1;
      ++s1;
      break;
    case DrawCol:
      break;
    case TreeCol:
      break;
    default:
      g_warning("Bad column style, ouch.");
      break;
    }
    
    c->name = *k;

    columns_.push_back(c);

    ++k;
    ++i;
  }

  recalc_widths();

  queue_display_update();
}
  
void 
DrawTree::get_column_widths(vector<gint> & widths)
{
  g_return_if_fail(widths.empty());

#ifdef GNOME_ENABLE_DEBUG
  ga_debug("DrawTree columns: ");
#endif

  vector<Column*>::iterator i = columns_.begin();
  while (i != columns_.end()) 
    {
#ifdef GNOME_ENABLE_DEBUG
      ga_debug("%s - %d ", (*i)->name.c_str(), (*i)->width);
#endif
      widths.push_back((*i)->width);
      ++i;
    }

#ifdef GNOME_ENABLE_DEBUG
  ga_debug("\n");
#endif
}

void 
DrawTree::set_column_widths(const vector<gint> & widths)
{
  g_return_if_fail(widths.size() == columns_.size());

  vector<Column*>::iterator i = columns_.begin();
  vector<gint>::const_iterator j = widths.begin();
  while (i != columns_.end()) 
    {
      (*i)->width = *j;

#ifdef GNOME_ENABLE_DEBUG
      if ((*i)->width < 0) g_warning("negative width");
#endif

      ++i;
      ++j;
    }

  recalc_widths();

  queue_display_update();
}

void
DrawTree::set_column(gushort col, ColumnStyle style)
{
  g_return_if_fail(col < columns_.size());
  columns_[col]->style = style;
  
  recalc_widths();

  queue_display_update();
}

// This function is so fucked up. 
// Need to fix the whole columns thing.
void 
DrawTree::recalc_widths()
{
  int needed = da_->allocation.width;
  int total = margin_*4; // The margin is the margin around row box and inside row box
  int nvisible = 0;

  vector<Column*>::iterator j = columns_.begin();
  while (j != columns_.end()) {
    if ((*j)->visible) {
      ++nvisible;
      total += (*j)->width;
      (*j)->extra = 0; // reset.
    }
    ++j;
  }

  total += col_gap_ * nvisible;

  total_width_ = total;

  if (total < needed) {
    const int extra = (needed - total);

    Column* rightmost = 0;
    vector<Column*>::iterator i = columns_.begin();
    while (i != columns_.end()) 
      {
        if ((*i)->visible) 
          {
            // This is just too weird when resizing columns
#if 0
            double ratio = double((*i)->width)/total;
            (*i)->extra = int(extra*ratio+0.5); // .5 to round
#else
            rightmost = *i;
#endif            
          }
        ++i;
      }
    if (rightmost != 0) rightmost->extra = extra;
    
    total_width_ = needed;
  }
  else if (needed > total) {
    // do some scrollbar stuff, FIXME
    g_warning("Hscrollbar");
    vector<Column*>::iterator i = columns_.begin();
    while (i != columns_.end()) {
      if ((*i)->style != BoolCol) {
        (*i)->extra = 0;
        ++i;
      }
    }
  }
  
  int x = margin_*2;

  vector<Column*>::iterator i = columns_.begin();
  while (i != columns_.end()) {
    if ((*i)->visible) {
      (*i)->x = x;
      x += (*i)->width + (*i)->extra + col_gap_;
    }
    else ((*i)->x = 0); // so we don't get weirdo garbage
    ++i;
  }
}

guint 
DrawTree::max_rows_onscreen()
{

  guint each_row = row_height_ + row_gap_;
  guint space = da_->allocation.height - toprow_y();

  if (space < 0) space = 0; // negative space means no space
  
  guint max = space/each_row;
  max += 1; // partial row on the bottom

  return max;
}

float 
DrawTree::exact_rows_onscreen()
{
  float each_row = row_height_ + row_gap_;
  float space = da_->allocation.height - toprow_y();

  if (space < 0) space = 0; // negative space means no space
  
  return space/each_row;
}

void
DrawTree::recalc_rows()
{
  recalc_rows_queued_ = false;

  visible_nodes_.clear();
  depths_.clear();

  guint max = max_rows_onscreen();

  // Unfortunately we are now going to iterate over the whole freaking
  //  tree, but it saves us ever doing it again.

  bool selection_still_exists = false;
  vector<NodeList::iterator> begins;
  vector<NodeList::iterator> ends;
  gushort depth = 0;

  nrows_ = 0;

  begins.push_back(nodes_->begin());
  ends.push_back(nodes_->end());

 row_recursion_loop:         // ah, one pines for Scheme
  g_assert(begins.size() == ends.size());

  // could abort as soon as we have enough on screen, but we don't for now
  while (!begins.empty()) {

    NodeList::iterator i = begins.back();
    NodeList::iterator e = ends.back();

    while (i != e) {
      
      Node* n = *i;
      ++nrows_;

#ifdef GNOME_ENABLE_DEBUG
      if (n == 0) g_warning("Null node got in the tree!");
#endif

      if (n == selected_node_) selection_still_exists = true;
      if (n == focus_node_) focus_index_ = nrows_ - 1;

      if ( !n->hidden() && 
           (nrows_ > row_) && 
           (visible_nodes_.size() < max) ) {
        // this one is visible, add it
        visible_nodes_.push_back(n);
        depths_.push_back(depth);
      }

      if (n->expanded()) {
        // Huge hack-o-rama since depends list may have changed
        n->refresh_expansion();

        // we need to "recurse"
        ++depth;
        
        // update the saved iterator for our position
        begins.pop_back();
        ++i; // start next time at next node.
        begins.push_back(i);
        
        // push the child tree on the stack, 
        //  and go to the top of the loop.
        begins.push_back(n->begin());
        ends.push_back(n->end());
        goto row_recursion_loop;
      }
      ++i;
    }
    
    // nothing was expanded, so we can go back down the stack.
    --depth;
    begins.pop_back();
    ends.pop_back();
  }

  // you can go at most 1 past the end
  if (row_ >= nrows_) row_ = nrows_;

  if (selection_still_exists == false) selected_node_ = 0;

  // If the selected node is hidden, unselect it.
  if (selected_node_ && selected_node_->hidden()) change_selection(0);

  setup_adjustments();

  g_assert(depths_.size() == visible_nodes_.size());
  g_assert(visible_nodes_.size() <= max);

#ifdef GNOME_ENABLE_DEBUG
  ga_debug(" ** recalc_rows: top: %u, visible: %u  total: %u\n", 
          row_, visible_nodes_.size(), nrows_);
#endif
}

void 
DrawTree::scroll_by(gint rows)
{
  if (rows == 0) return;

  int introw = row_;

  introw += rows;

  if (introw < 0) introw = 0;

  if (static_cast<gint>(row_) == introw) return; // no change

  row_ = introw;

  prelight_ = -1;

  queue_recalc_rows();
  queue_display_update();
}


void 
DrawTree::set_hadjustment(GtkAdjustment* a)
{
  hadjustment_ = a;

  // FIXME need to save id and disconnect
  gtk_signal_connect(GTK_OBJECT(hadjustment_),
                     "value_changed",
                     GTK_SIGNAL_FUNC(hadjustment_changed_signal),
                     this);

  ga_debug("installed hadjustment\n");

  if (GTK_WIDGET_REALIZED(da_)) {
    setup_adjustments();
  }
}

void 
DrawTree::set_vadjustment(GtkAdjustment* a)
{
  vadjustment_ = a;

  // FIXME need to save id and disconnect
  gtk_signal_connect(GTK_OBJECT(vadjustment_),
                     "value_changed",
                     GTK_SIGNAL_FUNC(vadjustment_changed_signal),
                     this);

#ifdef GNOME_ENABLE_DEBUG
  ga_debug("installed vadjustment\n");
#endif

  if (GTK_WIDGET_REALIZED(da_)) {
    setup_adjustments();
  }
}

void 
DrawTree::setup_adjustments()
{
  if (hadjustment_) {

  }

  if (vadjustment_) {
#ifdef GNOME_ENABLE_DEBUG
    ga_debug("setting up vadjustment\n");
#endif

    // It's important to emit signals only if there's an actual
    // change, otherwise we get weird scrollbar flicker.
    bool value_changed = false;
    bool other_changed = false;

    if (vadjustment_->value != row_)
      {
#ifdef GNOME_ENABLE_DEBUG
        ga_debug(" - setting value\n");
#endif
        vadjustment_->value          = row_;
        value_changed = true;
      }

    float onscreen = exact_rows_onscreen();
    // can't scroll below this
    float newupper = nrows_; //nrows_ - onscreen;
    //if (newupper < 0.0) newupper = 0.0;

    if (vadjustment_->upper != newupper || vadjustment_->page_size != onscreen)
      {
#ifdef GNOME_ENABLE_DEBUG
        ga_debug(" - setting other\n");
#endif
        vadjustment_->upper          = newupper;
        vadjustment_->page_increment = 
          MAX(visible_nodes_.size()-1,1);
        vadjustment_->page_size      = onscreen;
        other_changed = true;
      }

    vadjustment_->step_increment = 1;
    vadjustment_->lower          = 0;

    //    other_changed = value_changed = true;

    // Note that these trigger our own callbacks,
    //  but we properly handle the scroll == 0 case

    if (other_changed)
      gtk_adjustment_changed      (vadjustment_);
    if (value_changed) 
      gtk_adjustment_value_changed(vadjustment_);
  }
}

void  
DrawTree::hadjustment_changed_signal(GtkAdjustment* a, gpointer data)
{
  DrawTree* dt = static_cast<DrawTree*>(data);
  
  g_return_if_fail(data);

  dt->hadjustment_changed();
}

void 
DrawTree::vadjustment_changed_signal(GtkAdjustment* a, gpointer data)
{
  DrawTree* dt = static_cast<DrawTree*>(data);
  
  g_return_if_fail(data);

  dt->vadjustment_changed();
}

void 
DrawTree::hadjustment_changed()
{

}

void 
DrawTree::vadjustment_changed()
{
  gint nvalue = CLAMP(static_cast<gint>(vadjustment_->value), 
                      0, ((gint)nrows_));
  gint introw = static_cast<gint>(row_);

#ifdef GNOME_ENABLE_DEBUG
  ga_debug("vadjustment value changed: %f upper: %f pagesize: %f\n",
           vadjustment_->value, vadjustment_->upper, vadjustment_->page_size);
#endif

  scroll_by(nvalue - introw);
}

DrawTree::Node::~Node() 
{
  iterator i = children_.begin();
  while (i != children_.end()) {
    delete *i;
    ++i;
  }
}

guint 
DrawTree::Node::rows()
{
  guint r = 1;  // one for ourselves

  if (expanded()) {
    vector<Node*>::iterator i = children_.begin();
    while (i != children_.end()) {
      r += (*i)->rows();
      ++i;
    }    
  }

  return r;
}
