/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"

typedef struct
{
	gboolean dead_end;
	GdkPixmap *pixmap;
	GdkBitmap *bitmap;
	GdkPixmap *small_pixmap;
	GdkBitmap *small_bitmap;
} MimeCacheEntry;


static gchar *pixmap_files [NUM_PIXMAPS] = {
	"file_type_regular.xpm",
	"file_type_regular.xpm",
	"file_type_dir.xpm",
	"file_type_fifo.xpm",
	"file_type_socket.xpm",
	"file_type_char_device.xpm",
	"file_type_block_device.xpm",
	"file_type_sym_regular.xpm",
	
	"file_type_regular_big.xpm",
	"file_type_regular_big.xpm",
	"file_type_dir_big.xpm",
	"file_type_fifo_big.xpm",
	"file_type_socket_big.xpm",
	"file_type_char_device_big.xpm",
	"file_type_block_device_big.xpm",
	"file_type_sym_regular.xpm",

	"file_type_exe_big.xpm",
	"cvs_fish_small.xpm",

	"gnome_cmd_arrow_up.xpm",
	"gnome_cmd_arrow_down.xpm",
	"gnome_cmd_arrow_blank.xpm",

	"server.xpm",
	"logo.xpm",
	"exec.xpm",
	"mkdir.xpm",
	"lock.xpm"
};


static GdkPixmap *pixmaps[NUM_PIXMAPS];
static GdkBitmap *masks[NUM_PIXMAPS];

GHashTable *mime_cache = NULL;

GtkWidget *image_loading_widget = NULL;

//#define DEBUG_LOADING


void IMAGE_init (GtkWidget *widget)
{
    int i;

	if (!GTK_IS_WIDGET (widget))
		return;

	image_loading_widget = widget;

	mime_cache = g_hash_table_new (g_str_hash, g_str_equal);
	
	for ( i=0 ; i<NUM_PIXMAPS ; i++ )
	{
		char *path = g_strdup_printf ("%s/%s", PIXMAPS_DIR, pixmap_files[i]);

#ifdef DEBUG_LOADING		
		g_printerr ("imageloader: loading pixmap: %s\n", path);
#endif
		
		pixmaps[i] = gdk_pixmap_create_from_xpm (
			widget->window, &masks[i], NULL, path);

		if (pixmaps[i] == NULL) {
			warn_print (_("Couldn't load installed pixmap, trying to load from source-dir\n"));
			g_free (path);
			path = g_strdup_printf ("%s/%s", "../pixmaps", pixmap_files[i]);
			warn_print (_("Trying to load %s instead\n"), path);
			
			pixmaps[i] = gdk_pixmap_create_from_xpm (
				widget->window, &masks[i], NULL, path);

			if (pixmaps[i] == NULL)
				warn_print (_("Cant find the pixmap anywhere. Make sure you have installed the program or is executing gnome-commander from the gnome-commander-x.y.z/src directory\n"));
		}
		
		g_free (path);
	}
}


GdkPixmap *IMAGE_get_pixmap (Pixmap pixmap_id)
{
	return pixmaps[pixmap_id];
}


GdkBitmap *IMAGE_get_mask (Pixmap pixmap_id)
{
	return masks[pixmap_id];
}


static char *
get_icon_name (const gchar *mime_type) {
	gint i;
	gchar *icon_name, *tmp;
	const gchar *default_icon_name = gnome_vfs_mime_get_icon (mime_type);

	if (default_icon_name)
		return g_strdup (default_icon_name);

	/* Since there is a lot of icons missing in the mime-db we'll apply
	 * some nautilus magic here to get the right pixmap
	 */
	tmp = g_strdup (mime_type);

	/* replace '/' with '-' */
	for ( i=0 ; i<strlen(tmp) ; i++ ) {
		if (tmp[i] == '/')
			tmp[i] = '-';
	}

	/* add 'gnome-' */
	icon_name = g_strdup_printf ("gnome-%s", tmp);
	g_free (tmp);

	return icon_name;
}


gboolean IMAGE_get_mime_pixmap_and_mask (const gchar *mime_type,
										 GdkPixmap **pixmap,
										 GdkBitmap **bitmap,
										 gboolean big)
{
	MimeCacheEntry *entry;

	if (!mime_type)
		return FALSE;
	
	entry = (MimeCacheEntry*)g_hash_table_lookup (mime_cache, mime_type);
	if (!entry) {
		/* We're looking up this mime-type for the first time
		 */
		gchar *icon_name;
		GdkPixmap *pm = NULL;
		GdkBitmap *bm = NULL;
		GdkPixmap *spm = NULL;
		GdkBitmap *sbm = NULL;

		icon_name = get_icon_name (mime_type);

		if (icon_name) {
			gchar *icon_path;
			GdkPixbuf *pixbuf;
			GdkPixbuf *small_pixbuf = NULL;

			icon_path = g_strdup_printf ("%s/%s.png",
										 gnome_cmd_data_get_mime_icon_dir(),
										 icon_name);

			pixbuf = gdk_pixbuf_new_from_file (icon_path);
			
			if (pixbuf) {
				small_pixbuf = gdk_pixbuf_scale_simple (
					pixbuf,
					gnome_cmd_data_get_mime_icon_size(),
					gnome_cmd_data_get_mime_icon_size(),
					gnome_cmd_data_get_mime_scale_quality());
				
				gdk_pixbuf_render_pixmap_and_mask (
					pixbuf, &pm, &bm, 128);
				gdk_pixbuf_unref (pixbuf);
			}

			if (small_pixbuf) {
				gdk_pixbuf_render_pixmap_and_mask (
					small_pixbuf, &spm, &sbm, 128);
				gdk_pixbuf_unref (small_pixbuf);
			}

			g_free (icon_path);
		}

		g_free (icon_name);

		entry = g_new (MimeCacheEntry, 1);
		entry->dead_end = (pm == NULL || bm == NULL);
		entry->pixmap = pm;
		entry->bitmap = bm;
		entry->small_pixmap = spm;
		entry->small_bitmap = sbm;

		g_hash_table_insert (mime_cache, (gpointer)mime_type, entry);
	}

	if (big) {
		*pixmap = entry->pixmap;
		*bitmap = entry->bitmap;
	}
	else {
		*pixmap = entry->small_pixmap;
		*bitmap = entry->small_bitmap;
	}
	return !entry->dead_end;
}


static gboolean remove_entry (const gchar *key, MimeCacheEntry *entry, gpointer user_data)
{
	g_return_val_if_fail (entry, TRUE);

	if (!entry->dead_end) {
		gdk_pixmap_unref (entry->pixmap);
		gdk_pixmap_unref (entry->small_pixmap);
		gdk_bitmap_unref (entry->bitmap);
		gdk_bitmap_unref (entry->small_bitmap);
	}
	
	g_free (entry);
	return TRUE;
}

void IMAGE_clear_mime_cache (void)
{
	g_return_if_fail (mime_cache != NULL);

	g_hash_table_foreach_remove (mime_cache, (GHRFunc)remove_entry, NULL);
}

