/* vi:set ts=8 sts=0 sw=8:
 * $Id: misc.c,v 1.5 2001/01/26 08:34:44 kahn Exp kahn $
 *
 * Copyright (C) 2000 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <gnome.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <unistd.h>
#include <signal.h>
#include "prefs.h"
#include "msgbar.h"
#include "results.h"
#include "misc.h"
#include "debug.h"
#include "pathmax.h"


static GString *file_get_selected(win_t *w);


/*** global function definitions ***/
void
cmd_err(const char *msg, int errnum)
{
	GtkWidget *dlg;
	char *txt, *emsg = NULL;

	if (errnum)
		emsg = g_strerror(errnum);

	txt = g_strdup_printf("%s%c%s", msg, errnum ? '\n' : '\0', emsg);
	dlg = gnome_message_box_new(txt, GNOME_MESSAGE_BOX_ERROR,
				    GNOME_STOCK_BUTTON_OK, NULL);
	g_free(txt);
	gtk_window_set_position(GTK_WINDOW(dlg), GTK_WIN_POS_MOUSE);
	(void)gnome_dialog_run_and_close(GNOME_DIALOG(dlg));
} /* cmd_err */


int
cmd_exec(const char *newmsg, win_t *w, char *origcmd,
	 gboolean repl_files, gboolean report_err)
{
	char *finalcmd, *oldmsg, *tmp, *shell;
	pid_t pid;
	float val;
	int pstat;
	GnomeAppBar *abar;
	GtkProgress *pbar;
	int ret = 0;

	if (!origcmd || strlen(origcmd) < 1) {
		cmd_err(_("No command prefs specified"), 0);
		return -1;
	}

	if (prefs_bool_get(Prefs_cmd_prompting)) {
		GtkWidget *dlg;
		char *txt;
		int ret;

		txt = g_strdup_printf("Execute this command?\n\"%s\"", origcmd);
		dlg = gnome_message_box_new(txt,
					    GNOME_MESSAGE_BOX_QUESTION,
					    GNOME_STOCK_BUTTON_YES,
					    GNOME_STOCK_BUTTON_NO,
					    NULL);
		g_free(txt);
		gtk_window_set_position(GTK_WINDOW(dlg), GTK_WIN_POS_MOUSE);
		ret = gnome_dialog_run_and_close(GNOME_DIALOG(dlg));
		if (ret == 1)
			return -1;
	}

	if (repl_files) {
		GString *files = file_get_selected(w);
		APPDBG_FILE(("cmd_exec: files are '%s'\n", files->str));
		if (files->len < 1) {
			cmd_err(_("No files selected"), 0);
			return -1;
		}

		finalcmd = strrepl(origcmd, "$files", files->str);
		APPDBG_FILE(("cmd_exec: after $files repl '%s'\n", finalcmd));
		if (!finalcmd) {
			g_string_free(files, TRUE);
			cmd_err(_("$files not specified in command prefs"), 0);
			return -1;
		}

		g_string_free(files, TRUE);
	} else {
		finalcmd = g_strdup(origcmd);
	}

	APPDBG_FILE(("cmd_exec: finalcmd is '%s'\n", finalcmd));

	switch (pid = fork()) {
	case -1:
		ret = errno;
		cmd_err(_("fork() failed"), ret);
		break;
	case 0:
		shell = prefs_string_get(Prefs_shell);
		if (!shell || strlen(shell) < 1)
			shell = "/bin/sh";
		if (execl(shell, "sh", "-c", finalcmd, NULL) == -1) {
			ret = errno;
			cmd_err(_("execl() failed"), ret);
		}
		_exit(127);
		break;
	default:
		oldmsg = NULL;	/* quash silly compiler warning */
		pbar = results_progbar_widget(w);
		if (!pbar || !GTK_WIDGET_VISIBLE(w->reswgts[0]))
			pbar = w->progbar;
		abar = results_appbar_widget(w);
		if (!abar || !GTK_WIDGET_VISIBLE(w->reswgts[0]))
			abar = GNOME_APPBAR(w->msgbar);
		if (abar) {
			gtk_label_get(GTK_LABEL(abar->status), &tmp);
			oldmsg = g_strdup(tmp);
			gnome_appbar_set_status(abar, newmsg);
		}

		for (;;) {
			if (gtk_events_pending())
				gtk_main_iteration();
			ret = waitpid(pid, &pstat, WNOHANG);
			if (ret == -1 || w->close_window || w->stop_search) {
				(void)kill(pid, SIGTERM);
				break;
			} else if (ret > 0) {
				g_assert(ret == pid);
				ret = 0;
				break;
			} else {
				if (pbar) {
					val = gtk_progress_get_value(pbar) +0.2;
					if (val > 100)
						val = 0;
					gtk_progress_set_value(pbar, val);
				}
				msgbar_printf(w, _("Executing: '%s'"),
					      finalcmd);
			}
		}

		if (w->close_window) {
			ret = -1;
			break;
		}

		if (abar) {
			gtk_progress_set_value(pbar, 0);
			gnome_appbar_set_status(abar, oldmsg);
			g_free(oldmsg);
		}

		if (!report_err)
			break;

		if (WIFSIGNALED(pstat)) {
			char *msg = g_strdup_printf(
					_("Error executing command\n"
					  "Abnormal termination (signal = %d)"),
					WTERMSIG(pstat));
			ret = WTERMSIG(pstat);
			cmd_err(msg, 0);
			g_free(msg);
			APPDBG_FILE(("cmd_exec: error in '%s'\n", finalcmd));
		} else if (WIFEXITED(pstat) && WEXITSTATUS(pstat)) {
			cmd_err(_("Error executing command"),
				WEXITSTATUS(pstat));
			ret = WEXITSTATUS(pstat);
			APPDBG_FILE(("cmd_exec: error in '%s'\n", finalcmd));
		}
	} /* switch */

	g_free(finalcmd);
	return ret;
} /* cmd_exec */


GtkWidget **
my_widgets_setup(win_t *w, GtkWidget **wgts, int num, ef_t ef_wgts[],
		 char *name, int top_id, int max_id,
		 gboolean destroy_and_rebuild)
{
	GladeXML *xml;
	ef_t *efp;

	if (wgts) {
		g_assert(GTK_IS_WINDOW(wgts[top_id]));
		if (destroy_and_rebuild) {
			gtk_widget_destroy(wgts[top_id]);
		} else {
			gtk_widget_show(wgts[top_id]);
			gdk_window_raise(wgts[top_id]->window);
			return wgts;
		}
	}

	xml = my_glade_xml_get(name);
	g_assert(xml);

	wgts = g_new0(GtkWidget *, num);

#ifdef ENABLE_DEBUG
	APPDBG_MISC(("my_widgets_setup: num = %d (max = %d)\n", num, max_id));
	num = 0;
#endif
	for (efp = ef_wgts; efp->glname; efp++) {
		win_widgets_setup(xml, w, wgts, efp);
#ifdef ENABLE_DEBUG
		g_assert(efp->wgt_id == num);
		num++;
#endif
	}

	if (top_id != max_id) {
		gtk_window_set_transient_for(GTK_WINDOW(wgts[top_id]),
					     GTK_WINDOW(win_main_toplev(w)));
	}
	gtk_object_destroy(GTK_OBJECT(xml));
	return wgts;
} /* my_widgets_setup */


void
my_gtk_events_flush(void)
{
	while (gtk_events_pending())
		gtk_main_iteration();
} /* my_gtk_events_flush */


/*
 * Gets the widget corresponding to the specified xml.
 */
GtkWidget *
my_glade_widget_get(GladeXML *xml, char *name, int flags)
{
	GtkWidget *wgt;

	if ((wgt = glade_xml_get_widget(xml, name)) == NULL) {
		g_warning("Widget missing: %s\n", name);
		return NULL;
	}

	if (flags & GLADE_MODAL)
		gtk_window_set_modal(GTK_WINDOW(wgt), TRUE);
	if (flags & GLADE_POS_MOUSE)
		gtk_window_set_position(GTK_WINDOW(wgt), GTK_WIN_POS_MOUSE);
	if (flags & GLADE_SHOW)
		gtk_widget_show(wgt);
	if (flags & GLADE_HIDE)
		gtk_widget_hide(wgt);

	return wgt;
} /* my_glade_widget_get */


/*
 * Loads the Glade XML object from the .glade file.
 */
GladeXML *
my_glade_xml_get(char *root)
{
	GladeXML *xml;
	char *glade_file;

	if (g_file_exists("./gnome-find.glade"))
		return glade_xml_new("./gnome-find.glade", root);

	glade_file = g_concat_dir_and_file(DATADIR, "gnome-find.glade");
	if (!g_file_exists(glade_file))
		g_error("%s not found!\n", glade_file);
	xml = glade_xml_new(glade_file, root);
	g_free(glade_file);
	return xml;
} /* my_glade_xml_get */


void
my_prefs_window_set_size(GtkWidget *toplev, int prefs_id, int w_id, int h_id)
{
	if (prefs_bool_get(prefs_id) &&
	    prefs_int_get(w_id) > 0 && prefs_int_get(h_id) > 0) {

		gtk_window_set_default_size(GTK_WINDOW(toplev),
					    prefs_int_get(w_id),
					    prefs_int_get(h_id));
	}
} /* my_prefs_window_set_size */


/*
 * PUBLIC: strrepl
 *
 * given "orig" string, look for substring "pat", and if found, replace it with
 * string "repl".  returns a newly allocated buffer if "pat" was found, NULL if
 * not.
 */
char *
strrepl(char *orig, char *pat, char *repl)
{
	int p1, p2, p3;
	char *needle;
	char *buf;

	if (!orig || !pat || ((needle = strstr(orig, pat)) == NULL))
		return NULL;

	p1 = (needle - orig);
	p2 = (repl) ? strlen(repl) : 0;
	p3 = strlen(orig) - (p1 + strlen(pat));

	buf = g_new(char, p1 + p2 + p3 + 1);
	strncpy(buf, orig, p1);
	buf[p1] = '\0';
	if (repl)
		strcat(buf, repl);
	strcat(buf, needle + strlen(pat));

	return buf;
} /* strrepl */


/*
 * Given "path", if the "~" appears in it, expand it as the home directory.
 * Returns a newly allocated buffer, or NULL on error.
 */
char *
my_dir_expand_home(char *path)
{
	char *buf, *first_slash, *copy;
	int complete_len;
	struct passwd *pwd;

	if (path && path[0] == '~') {
		if ((first_slash = strchr(path, '/')))
			complete_len = first_slash - path - 1;
		else
			complete_len = strlen(path) + 1;

		if (complete_len && (strlen(path) > 1)) {
			/* ~user */
			copy = g_new(char, complete_len + 1);
			strncpy(copy, path + 1, complete_len);
			copy[complete_len] = '\0';
			pwd = getpwnam(copy);
			g_free(copy);  
			buf = (pwd) ? g_strdup(pwd->pw_dir) : g_strdup(path);
		} else {
			/* ~ */
			char *home = g_get_home_dir();
			g_assert(home);
			buf = strrepl(path, "~", home);
		}
	} else {
		buf = g_strdup(path);
	}

	return buf;
} /* my_dir_expand_home */


void
my_clist_down(GtkWidget **wgts, int clist_id)
{
	GtkCList *clist;
	int firstrow, lastrow, i;

	if (!wgts)
		return;

	g_assert(GTK_IS_CLIST(wgts[clist_id]));
	clist = GTK_CLIST(wgts[clist_id]);

	if (!clist->selection || !clist->selection_end)
		return;

	lastrow = GPOINTER_TO_INT(clist->selection_end->data);
	if (lastrow == clist->rows - 1)
		return;

	firstrow = GPOINTER_TO_INT(clist->selection->data);

	for (i = lastrow; i >= firstrow; i--)
		gtk_clist_row_move(clist, i, i + 1);
} /* my_clist_down */


void
my_clist_up(GtkWidget **wgts, int clist_id)
{
	GtkCList *clist;
	int firstrow, lastrow, i;

	if (!wgts)
		return;

	g_assert(GTK_IS_CLIST(wgts[clist_id]));
	clist = GTK_CLIST(wgts[clist_id]);

	if (!clist->selection || !clist->selection_end)
		return;

	firstrow = GPOINTER_TO_INT(clist->selection->data);
	if (firstrow == 0)
		return;

	lastrow = GPOINTER_TO_INT(clist->selection_end->data);

	for (i = firstrow; i <= lastrow; i++)
		gtk_clist_row_move(clist, i, i - 1);
} /* my_clist_up */


void
my_clist_del(GtkWidget **wgts, int clist_id)
{
	GtkCList *clist;

	if (!wgts)
		return;

	g_assert(GTK_IS_CLIST(wgts[clist_id]));
	clist = GTK_CLIST(wgts[clist_id]);

	while (clist->selection) {
		int rownum = GPOINTER_TO_INT(clist->selection->data);
		gtk_clist_remove(clist, rownum);
	}
} /* my_clist_del */


GSList *
my_argvl_build(char *txt, int *argc, GSList *argvl)
{
	char *buf;

	buf = strtok(txt, " ");
	do {
		argvl = g_slist_append(argvl, g_strdup(buf));
		(*argc)++;
	} while ((buf = strtok(NULL, " ")));

	return argvl;
} /* my_argvl_build */


char *
my_tempname(void)
{
	char *tmpdir;

	tmpdir = prefs_string_get(Prefs_tmpdir);
	if (!tmpdir)
		tmpdir = "/tmp";

	return g_strdup_printf("%s/gnome-find-%d-%d-%d",
			       tmpdir, (int)getpid(), (int)time(NULL), rand());
} /* my_tempname() */


/*
 * PUBLIC: file_full_pathname_make
 *
 * given a filename, tries to construct the full pathname to the file.  if
 * successful, returns a newly allocated buffer with the full pathname.
 */
char *
file_full_pathname_make(char *fname)
{
	char *full, *cwd;

	if (!fname)
		return NULL;

	if (fname[0] == '/')
		return g_strdup(fname);

	if ((cwd = getcwd(NULL, PATH_MAX)) == NULL)
		return g_strdup(fname);

	if (cwd[strlen(cwd) - 1] == '/')
		cwd[strlen(cwd) - 1] = '\0';

	if ((strlen(fname) > 2) && fname[0] == '.' && fname[1] == '/')
		full = g_concat_dir_and_file(cwd, fname + 2);
	else
		full = g_concat_dir_and_file(cwd, fname);

	/* use free() here, not g_free(), since getcwd() uses malloc() */
	free(cwd);

	return full;
} /* file_full_pathname_make */


/*
 * returns a newly allocated string, listing all files selected, separated by a
 * white space.
 */
static GString *
file_get_selected(win_t *w)
{
	GtkCList *clist;
	GList *l;
	char *fname;
	GString *files;

	files = g_string_new(NULL);
	clist = GTK_CLIST(results_clist_widget(w));
	for (l = clist->selection; l; l = l->next) {
		GtkCTreeNode *cnode = (GtkCTreeNode *)l->data;
		GtkCListRow *clrow = (GtkCListRow *)cnode->list.data;
		fname = clrow->cell->u.text;
		files = g_string_append(files, fname);
		if (l->next)
			files = g_string_append_c(files, ' ');
	}

	return files;
} /* file_get_selected */


/* the end */
