/****************************
  Reception for ICQ v6-7 protocol (Oscar)
  Olivier Crete (c) 2001
  GnomeICU
*****************************/

#include "common.h"
#include "gnomeicu.h"
#include "gtkfunc.h"
#include "personal_info.h"
#include "response.h"
#include "rus_conv.h"
#include "sendcontact.h"
#include "showlist.h"
#include "util.h"
#include "v7recv.h"
#include "v7send.h"
#include "v7snac13.h"

void     v7_connected (GTcpSocket *socket, GInetAddr *ia,
                   GTcpSocketConnectAsyncStatus status, V7Connection *conn);
gboolean v7_handler (GIOChannel *iochannel, GIOCondition condition,
                     V7Connection *conn);
gboolean v7_error_handler (GIOChannel *iochannel, GIOCondition condition,
                     V7Connection *conn);

/*  Handlers for the different type of packets */
void   v7_snac_channel_handler      (V7Packet *packet, V7Connection *conn);
void   v7_newconn_channel_handler   (V7Packet *packet, V7Connection *conn);

void   v7_make_requests             (V7Connection *conn);
void   v7_ack_ack_rate              (V7Connection *conn);
void   v7_receive_msg               (Snac *snac, V7Connection *conn);
void   v7_text_message              (UIN_T uin, gchar *here);
void   v7_special_message           (UIN_T uin, gchar *here);
void   v7_user_added_you            (UIN_T uin);
void   v7_advanced_message          (UIN_T uin, gchar *here, int len,
                                     gchar *begofmes);
void   v7_advanced_message_ack      (UIN_T uin, guchar *data, gchar *begofmes);
void   v7_incoming_user             (Snac *snac);
void   v7_outgoing_user             (Snac *snac);
void   v7_acked_adv_msg             (Snac *snac, V7Connection *conn);

void   v7_rec_contactlist           (UIN_T uin, gchar *msg, int msglen,
                                     time_t msg_time);

void   v7_rate_changed              (Snac *snac, V7Connection *conn);
void   v7_saved_data                (Snac *snac);
void   v7_offline_message           (DWORD reqid, gchar *data);
void   v7_saved_data_extension      (DWORD reqid, gchar *data, gchar *end);
gchar *v7_search_reply              (DWORD reqid, BYTE result, gchar *here);
void   v7_end_of_search_reply       (DWORD reqid, BYTE result, gchar *here,
                                     gchar *buffer_end);

void   v7_parse_main_home_info      (DWORD reqid, Contact_Member *DestContact, 
                                     BYTE result, gchar *here);
void   v7_parse_more_info           (DWORD reqid, Contact_Member *DestContact, 
                                     BYTE result, gchar *here);
void   v7_parse_about_info          (DWORD reqid, Contact_Member *DestContact, 
                                     BYTE result, gchar *here);
void   v7_parse_work_info           (DWORD reqid, Contact_Member *DestContact, 
                                     BYTE result, gchar *here);
void   v7_parse_emails_info         (DWORD reqid, Contact_Member *DestContact, 
                                     BYTE result, gchar *here);


void v7_connect(gchar *address, guint port, gchar *cookie, guint cookielen)
{
  V7Connection *conn;
  
#ifdef TRACE_FUNCTION
  g_print( "v7_connect\n" );
#endif

  mainconnection = conn = g_malloc0(sizeof(V7Connection));

  conn->cookie = g_memdup(cookie, cookielen);
  conn->cookielen = cookielen;

  conn->gsocketasyncid = gnet_tcp_socket_connect_async( address, port,
                             (GTcpSocketConnectAsyncFunc) v7_connected, conn);
  conn->last_reqid = 1;
  conn->status = NOT_CONNECTED;
}



void v7_connected (GTcpSocket *socket, GInetAddr *ia,
                   GTcpSocketConnectAsyncStatus status, V7Connection *conn)

{

#ifdef TRACE_FUNCTION
  g_print( "v7_connected\n" );
#endif
  
  if (status != GTCP_SOCKET_CONNECT_ASYNC_STATUS_OK) {

    g_free(conn);
    mainconnection = NULL;
    Current_Status = STATUS_OFFLINE;
    ready_set();
    enable_online_events = FALSE;
    animate_off();

    return;
  }
  
  
  conn->gsocket = socket;
  
  conn->status = CONNECTING;
  
  conn->iochannel = gnet_tcp_socket_get_iochannel (socket);
  
  conn->in_watch = g_io_add_watch ( conn->iochannel, G_IO_IN|G_IO_PRI,
                                    (GIOFunc) v7_handler, conn);
  conn->err_watch = g_io_add_watch ( conn->iochannel,
                                     G_IO_ERR|G_IO_HUP|G_IO_NVAL,
                                     (GIOFunc) v7_error_handler, conn);
  
}

gboolean v7_error_handler (GIOChannel *iochannel, GIOCondition condition,
                     V7Connection *conn)
{
#ifdef TRACE_FUNCTION
  g_print( "v7_error_handler\n" );
#endif

  icq_set_status_offline( NULL, NULL);
  return FALSE; /* not sure what to return -- menesis */
}

gboolean v7_handler (GIOChannel *iochannel, GIOCondition condition,
                     V7Connection *conn)
{
  V7Packet *packet;
  
#ifdef TRACE_FUNCTION
  g_print( "v7_handler\n" );
#endif
  
  
  /* If we dont have the whole packet stop here */
  if ((packet = read_flap(conn)) == NULL)
    return TRUE;
  
  switch (packet->channel) {
    
  case CHANNEL_NEWCONN:
    v7_newconn_channel_handler (packet, conn) ;
    break;
  case CHANNEL_SNAC:
    v7_snac_channel_handler( packet, conn);
    break;
  case CHANNEL_ERROR:
    
    break;
  case CHANNEL_CLOSE:
    gnome_error_dialog(_("Server forced disconnect."));
    v7_quit();
    break;
  case CHANNEL_KEEPALIVE:
    
    break;
    
  default:
    g_warning("New unknown channel: %d", packet->channel);
  }

  g_free(packet->data);
  g_free(packet);
  
  return TRUE;
}



void v7_snac_channel_handler(V7Packet *packet, V7Connection *conn) 
{
  Snac *snac;

#ifdef TRACE_FUNCTION
  g_print( "v7_snac_channel_handler\n" );
#endif

  snac = read_snac(packet);

  switch (snac->family) {
  case FAMILY_01_GENERIC:
    switch (snac->type) {
    case F01_SERVER_READY:
      v7_iam_icq(conn);
      break;
    case F01_SERVER_ACK_RATE:
      v7_ack_ack_rate(conn);
      v7_make_requests(conn);
      break;
    case F01_SERVER_SENDING_TOO_FAST:
      v7_rate_changed(snac, conn);
      break;
    case F01_SERVER_PERSONAL_INFO:
      break;
    case F01_SERVER_MOTD:
      v7_request_rate (conn);
      break;
    case F01_SERVER_ACK_I_AM_ICQ:
      /* We dont understand the data, ignore it */
      break;
    default:
    }
    break;
  case FAMILY_02_LOCATION:
    switch (snac->type) {
    case F02_SERVER_RIGHTS_LOCATION: 
      /* We dont understand the data, ignore it */
      break;
    default:
    }
    break;
  case FAMILY_03_CONTACTLIST:
    switch (snac->type) {
    case F03_SERVER_CL_INFO:
      /* We dont understand the data, ignore it */
      break;
    case F03_SERVER_INCOMING_USER:
      v7_incoming_user(snac);
      break;
    case F03_SERVER_OUTGOING_USER:
      v7_outgoing_user(snac);
      break;
    default:
    }
    break;
  case FAMILY_04_MESSAGING:
    switch (snac->type) {
    case F04_SERVER_RIGHTS_MSG:
      /* We dont understand the data, ignore it */
      break;
    case F04_SERVER_RECV_MESSAGE:
      v7_receive_msg(snac, conn);
      break;
    case F04_BOTH_ACK_ADV_MSG:
      v7_acked_adv_msg(snac, conn);
      break;
    default:
    }
    break;
  case FAMILY_09_LISTS:
    switch (snac->type) {
    case F09_SERVER_LIST_RIGHTS:
      /* We dont understand the data, ignore it */
      break;
    default:
    }
    break;
  case FAMILY_0B_UNKNOWN_1:
    switch (snac->type) {
    default:
    }
    break;
  case FAMILY_13_SAVED_LIST:
    switch (snac->type) {
    case F13_SERVER_REQ_RIGHTS_ANS: /* 0x03 */
      /* reply to F13_SERVER_UNKNOWN_1 */
      break;
    case F13_SERVER_SAVED_LIST: /* 0x06 */
      v7_read_contacts_list (snac, conn);
      break;
    case F13_SERVER_ACK: /* 0x0E */
      v7_snac13_check_srv_ack (snac);
      break;
    case F13_SERVER_LIST_NO_CHANGE: /* 0x0F */
      /* we get this because the value sent didn't change */
      /* send 13,07 */
      snac_send (conn, NULL, 0, FAMILY_13_SAVED_LIST,
		 F13_CLIENT_RDY_TO_USE_LIST, NULL, ANY_ID);
      v7_set_user_info (conn);
      v7_add_icbm (conn);
      v7_setstatus (preset_status);
      v7_client_ready (conn);
      break;
    case F13_SERVER_AUTH_REQUEST: /* 0x19 */
      v7_recv_auth_request (snac);
      break;
    default:
    }
    break;
  case FAMILY_15_SAVED_DATA:
    switch (snac->type) {
    case F15_SERVER_SAVED_DATA:
      v7_saved_data(snac);
    default:
    }
    break;
  case FAMILY_17_NEW_USER:
    switch (snac->type) {
    default:
   }
    break;
  default:
    break;
    
  }

  g_free (snac->data);
  g_free(snac);
  
}



void v7_newconn_channel_handler (V7Packet *packet, V7Connection *conn) 
{
  TLVstack *cookietlv;

#ifdef TRACE_FUNCTION
  g_print( "v7_newconn_channel_handler\n" );
#endif

  if ( packet->len == 4  &&
       packet->data[0] == packet->data[1] == packet->data[2] == 0 &&
       packet->data[3] == 1) {
    
    
    cookietlv  = new_tlvstack ("\0\0\0\x01", 4);
    
    add_tlv(cookietlv, 6, conn->cookie, conn->cookielen);
      
    if (! flap_send(conn, CHANNEL_NEWCONN, cookietlv->beg, cookietlv->len))
      conn->status = BROKEN;

    free_tlvstack(cookietlv);

    
  } else
    g_warning ("Unknown packet on login on main server\n");
}



void v7_ack_ack_rate    (V7Connection *conn)
{
  BYTE v7_ack_ack_rate_array[10] = "\x00\x01\x00\x02\x00\x03\x00\x04\x00\x05";

#ifdef TRACE_FUNCTION
  g_print( "v7_ack_ack_rate\n" );
#endif

  snac_send(conn, v7_ack_ack_rate_array, 10, FAMILY_01_GENERIC,
            F01_CLIENT_ACK_ACK_RATE, NULL, ANY_ID);

  /*v7_sendcontactlist();*/
  /*
  if (Current_Status == STATUS_INVISIBLE)
    v7_sendvisiblelist();
  else
    v7_sendinvisiblelist();

  if (preset_status != STATUS_OFFLINE)
    v7_setstatus(preset_status);
  else
    v7_setstatus(STATUS_ONLINE);
  */
}

void v7_receive_msg (Snac *snac, V7Connection *conn)
{
  gchar *here = snac->data + 8;
  WORD msgformat;
  BYTE uinlen;
  gchar *uinstr;
  UIN_T uin;
  int len = snac->datalen;
  TLV *blah;
  int tlvcount,i;

#ifdef TRACE_FUNCTION
  g_print( "v7_receive_msg\n" );
#endif

  msgformat = CharsBE_2_Word(here);

  here += 2;
  len -= 2;
  
  uinlen = here[0];
  here += 1;
  len -=1;

  uinstr = g_malloc0(uinlen+1);
  g_memmove(uinstr, here, uinlen);
  uin = strtol(uinstr, NULL, 10);
  g_free(uinstr);

  here += uinlen;
  len -= uinlen;

  here += 2;
  len -= 2;
  
  tlvcount = CharsBE_2_Word(here);
  here += 2;
  len -=2;

  for(i=0; i<tlvcount; i++) {
    blah = new_tlv(here); 
    here += blah->len + 4; 
    len -=  (blah->len + 4);
    delete_tlv(blah); 
  }

  switch(msgformat) {
  case 0x01:
    v7_text_message(uin, here);
    break;
  case 0x02:
    v7_advanced_message(uin, here, len, snac->data);
    break;
  case 0x04:
    v7_special_message(uin, here);
    break;
  default:
    g_warning("unknown message format: %x\n", msgformat);
  }
  
}

void v7_text_message(UIN_T uin, gchar *here)
{
  
  WORD msglen;
  gchar *msg;
  TLV *msgtlv;
  time_t msg_time;

#ifdef TRACE_FUNCTION
  g_print( "v7_text_message\n" );
#endif
  
  msgtlv = new_tlv(here); 
  
  /* I dont know why 3, but it does work */
  msglen = CharsBE_2_Word(msgtlv->value+ 6 + msgtlv->value[3]); 
  msglen -= 4;                /* to skip the 4 zeros */
  
  if (msglen + 13 > msgtlv->len) { 
    g_warning("MSGLEN TOO LONG: %d > %d\n", msglen+12+msgtlv->value[3],
              msgtlv->len);
    msglen = msgtlv->len -12 -msgtlv->value[3];
  }
  
  msg = g_malloc0(msglen +1);
  
  g_memmove(msg, msgtlv->value + 12 + msgtlv->value[3], msglen);
  
  delete_tlv(msgtlv); 
  
  time(&msg_time);
  
  msg_received(uin, msg, msg_time);
  
  g_free(msg);

}

void v7_special_message(UIN_T uin, gchar *here)
{

  WORD msglen;
  gchar *msg, *desc, *nick, *first, *last, *email, *tmp=NULL;
  TLV *msgtlv;
  BYTE msgtype, msgflags;
  time_t msg_time;

#ifdef TRACE_FUNCTION
  g_print( "v7_special_message\n" );
#endif
  
  msgtlv = new_tlv(here); 
  
  msgtype = msgtlv->value[4];
  msgflags = msgtlv->value[5];

  msglen = Chars_2_Word(msgtlv->value+6);

  msg = g_malloc0(msglen +1);

  g_memmove(msg, msgtlv->value+8, msglen);

  delete_tlv(msgtlv); 

  time(&msg_time);
  
  switch(msgtype) {
  case 0x01:
    msg_received(uin, msg, msg_time);
    break;
  case 0x04:
    desc = strchr( msg, '\xFE' ) + 1;
    *(desc-1) = '\0';
    url_received (uin, msg, desc, msg_time);
    break;
  case 0x13:
    v7_rec_contactlist(uin, msg, msglen, msg_time);
    break;
  case 0x0C:
    user_added_you(uin, msg_time);
    break;
  case 0x06:
    nick = msg;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL ) 
      return; /* Bad Packet */
    *tmp = '\0';
    rus_conv(RUS_WIN_KOI,nick);
    tmp++;

    first = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    last = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;
    rus_conv(RUS_WIN_KOI,last);

    email = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;
    rus_conv(RUS_WIN_KOI,email);

    desc = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp ++;
    rus_conv(RUS_WIN_KOI,desc);

    authorisation_request(uin, nick, first, last, email, desc, msg_time);
    break;
  default:
    g_warning("Special message type not handled, type: %x, text: %s\n", msgtype, msg);
  }

  g_free(msg);
}

void v7_advanced_message(UIN_T uin, gchar *data, int len, gchar *begofmes)
{

  TLV *maintlv, *insidetlv;
  WORD type1, msglen;
  BYTE msgtype;
  gchar *msg, *here, *desc, *nick, *first, *last, *email, *tmp=NULL, *dertfed;
  time_t msg_time;
  int i;

#ifdef TRACE_FUNCTION
  g_print( "v7_advanced_message\n" );
#endif


  maintlv = new_tlv(data);

  if (maintlv->type != 5) {
    g_warning("Advanced msg has a tlv of a bizzare type: %d\n", maintlv->type);
    print_data(maintlv->value, maintlv->len);
    delete_tlv(maintlv);
    return;
  } 

  type1 = CharsBE_2_Word (maintlv->value);

  if (type1 == 1) {
    g_warning("We have an abort request, we dont handle that right now!\n");
    delete_tlv(maintlv);
    return;
  } 

  here = maintlv->value+26;
  
  for (i=0; i<type1+2; i++) {
    insidetlv = new_tlv(here);
    switch (insidetlv->type) {
    case 0x000A:
    case 0x0005:
    case 0x0003:
    case 0x000F:
      break;
    default:
      g_warning("We have a tlv that we dont know of type: %x\n",
                insidetlv->type);
      delete_tlv(insidetlv);
      delete_tlv(maintlv);
      print_data(data, len);
      return;
    }
    here += insidetlv->len+4;
    delete_tlv(insidetlv);
  }
  
  insidetlv = new_tlv(here);
  
  msgtype = insidetlv->value[45];
  
  msglen = Chars_2_Word(insidetlv->value+51);

  msg = g_malloc0(msglen +1);
  
  g_memmove(msg, insidetlv->value + 53, msglen);
  
  time(&msg_time);

  switch(msgtype) {
  case 0x01: /* text message */
    dertfed = dertf(msg);
    msg_received(uin, dertfed, msg_time);
    g_free(dertfed);
    break;
  case 0x04: /* URL */
    desc = strchr( msg, '\xFE' ) + 1;
    *(desc-1) = '\0';
    url_received (uin, msg, desc, msg_time);
    break;
  case 0x0C: /* User Added you */
    user_added_you(uin, msg_time);
    break;
  case 0x06:  /* Auth request */
    nick = msg;
    tmp = strchr( msg, '\xFE' );
    if ( tmp == NULL ) {
      g_free(msg);
      return; /* Bad Packet */
    }
    *tmp = '\0';
    rus_conv(RUS_WIN_KOI,nick);
    tmp++;

    first = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL ) {
      g_free(msg);
      return; /* Bad Packet */
    }
    *tmp = '\0';
    tmp++;

    last = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL ) {
      g_free(msg);
      return; /* Bad Packet */
    }
    *tmp = '\0';
    tmp++;
    rus_conv(RUS_WIN_KOI,last);

    email = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL ) {
      g_free(msg);
      return; /* Bad Packet */
    }
    *tmp = '\0';
    tmp++;
    rus_conv(RUS_WIN_KOI,email);

    desc = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL ) {
      g_free(msg);
      return; /* Bad Packet */
    }
    *tmp = '\0';
    tmp ++;
    rus_conv(RUS_WIN_KOI,desc);

    authorisation_request(uin, nick, first, last, email, desc, msg_time);
    break;
  case 0xE8:  /* Request for away message */
  case 0xE9:
  case 0xEA:
  case 0xEB:
  case 0xEC:
    break;
  case 0x1A:
  case 0x13: /* Contacts */
    g_free(msg);

    msglen = Chars_2_Word(insidetlv->value+105);
    
    msg = g_malloc0(msglen +1);
    
    g_memmove(msg, insidetlv->value + 109, msglen);
    
    v7_rec_contactlist(uin, msg, msglen, msg_time);

    break;
  default:
    g_warning("Advanced cessage type not handled, type: %x, text: %s\n",
              msgtype, msg);
    delete_tlv(insidetlv);
    delete_tlv(maintlv);
    g_free(msg);
    print_data(data, len);
    return;
  }

  g_free(msg);

  v7_advanced_message_ack(uin, insidetlv->value, begofmes); 
  
  delete_tlv(insidetlv);
  delete_tlv(maintlv);
  
}

void v7_advanced_message_ack (UIN_T uin, guchar *data, gchar *begofmes) 
{

  TLVstack *tlvs;

#ifdef TRACE_FUNCTION
  g_print( "v7_advanced_message_ack\n" );
#endif

  tlvs = new_tlvstack(begofmes,11+begofmes[10]);

  add_nontlv_w_be(tlvs, 0x0003);

  /* I dont know what all of that means, but it works */
  
  add_nontlv(tlvs, data, 26);
  add_nontlv(tlvs, "\0", 1);
  add_nontlv(tlvs, data+27, 20);
  add_nontlv_dw_be(tlvs,0); /* Always send 0 as accept status */

  /* Add away message if away */
  if ( Current_Status == STATUS_AWAY ||
       Current_Status == STATUS_NA ||
       Current_Status == STATUS_OCCUPIED ||
       Current_Status == STATUS_DND) {
    add_nontlv_w_le(tlvs, strlen(Away_Message)+1);
    add_nontlv(tlvs, Away_Message, strlen(Away_Message)+1);
  }
  else
    add_nontlv_w_le(tlvs, 0);  

  if (data[45] < 0xE0) {  /* add this if not autoawaymessage reply */
    add_nontlv_dw_be(tlvs, 0);
    add_nontlv_dw_be(tlvs, 0xFFFFFF00);
  }
  
  snac_send(mainconnection, tlvs->beg, tlvs->len, FAMILY_04_MESSAGING,
            F04_BOTH_ACK_ADV_MSG, NULL, ANY_ID);

  free_tlvstack(tlvs);
}

void v7_incoming_user (Snac *snac)
{

  UIN_T uin=0; 
  GSList *contact;
  gchar *here;
  int tlvcount, i;
  TLV *tlv;
  gboolean status_has_changed = FALSE;

#ifdef TRACE_FUNCTION
  g_print( "v7_incoming_user\n" );
#endif
  
  uin = strtol(snac->data+1, NULL, 10);

  contact = Find_User( uin );
  if( contact == NULL )
    return;
  
  here = snac->data + snac->data[0] + 3;

  tlvcount = CharsBE_2_Word(here);

  here += 2;
  
  for(i=0; i<tlvcount; i++) {
    tlv = new_tlv(here); 
    here += tlv->len + 4; 

    switch (tlv->type) {
    case 0x0C:
      kontakt->has_direct_connect = Chars_2_DW(tlv->value) ? TRUE : FALSE;
      kontakt->port = CharsBE_2_DW(tlv->value+4);
      kontakt->version = CharsBE_2_Word(tlv->value+9);
      break;
    case 0x0A:
      kontakt->current_ip = IP_2_DW( tlv->value );
      break;
    case 0x06:
      if (kontakt->status != CharsBE_2_Word(tlv->value+2))
      	status_has_changed = TRUE;
      kontakt->status = CharsBE_2_Word(tlv->value+2);
      break;
    }

    delete_tlv(tlv); 
  }

  if (kontakt->status == STATUS_OFFLINE) {
    status_has_changed = TRUE;
    kontakt->status = STATUS_ONLINE;
  }

  kontakt->last_time = time( NULL );

  if (status_has_changed)
    User_Online(contact);
}

void v7_outgoing_user (Snac *snac)
{
  
  UIN_T uin=0;
  GSList *contact;
  
#ifdef TRACE_FUNCTION
  g_print( "v7_outgoing_user\n" );
#endif
  
  uin = strtol(snac->data+1, NULL, 10);
  
  contact = Find_User( uin );
  if( contact == NULL )
    return;


  User_Offline(contact);
}


void v7_acked_adv_msg (Snac *snac, V7Connection *conn)
{

  BYTE type;
  gchar *awaymessage;
  UIN_T sender_uin;
  WORD sender_status;

#ifdef TRACE_FUNCTION
  g_print( "v7_acked_adv_msg\n" );
#endif
  
  type = snac->data[11+snac->data[10]+2+26+1+18];

  switch (type) {
  case 0xE8:
    sender_status = STATUS_AWAY;
    break;
  case 0xE9:
    sender_status = STATUS_OCCUPIED;
    break;
  case 0xEA:
    sender_status = STATUS_NA;
    break;
  case 0xEB:
    sender_status = STATUS_DND;
    break;
  case 0xEC:
    sender_status = STATUS_FREE_CHAT;
    break;
  default:
    return;
  }

  sender_uin = atoi(snac->data+11);

  awaymessage = g_strndup(snac->data+11+snac->data[10]+2+26+1+20+1+3+2,
                          Chars_2_Word(snac->data+11+snac->data[10]+2+26+1+20+1+3));
  
  recv_awaymsg( sender_uin, sender_status, awaymessage);

}

void v7_saved_data(Snac *snac)
{
  TLV *tlv;
  WORD type, reqid;

#ifdef TRACE_FUNCTION
  g_print( "v7_saved_data\n" );
#endif

  tlv = new_tlv(snac->data);
  
  type = CharsBE_2_Word(tlv->value+6);
  
  reqid = Chars_2_Word(tlv->value+8);

  switch(type) {
  case 0x4100:
    v7_offline_message(snac->reqid, tlv->value+10);
    break;
  case 0x4200: 
    v7_send_often(snac->reqid);
    break; 
  case 0xDA07:
    v7_saved_data_extension(snac->reqid, tlv->value+10, tlv->value+tlv->len);
  }
  
  delete_tlv(tlv);

}

void v7_saved_data_extension (DWORD reqid, gchar *data, gchar *end)
{
  WORD subtype;
  BYTE result; 
  GSList *requestdat=NULL, *contact=Contacts;
  gchar *nick = NULL;

#ifdef TRACE_FUNCTION
  g_print( "v7_saved_data_extension\n" );
#endif
  
  subtype = CharsBE_2_Word(data);

  result = data[2];
  
  data += 3;

  switch (subtype) {
  case 0xC800:
	nick = data + 2;
  case 0xDC00:
  case 0xEB00:
  case 0x0E01:
  case 0xD200:
  case 0xE600:
  case 0xF000:
  case 0xFA00:
    
    requestdat = v7_get_request_by_reqid(reqid);

    if (requestdat == NULL)
      return;
    
    while( contact != NULL ) {
      if( kontakt->uin == ((Request*)(requestdat->data))->uin )
        break;
      contact = contact->next;
    }
    
    if (contact == NULL)
      return;

    /* Anti spam action here */
    if (((Request*)requestdat->data)->type == NEW_USER_VERIFICATION) {
      if (result != 0x0A) {
        remove_contact (((Request*)(requestdat->data))->uin, TRUE);
        requests = g_slist_remove(requests, requestdat->data);
        return;
      }
      else {
        if (kontakt->confirmed == FALSE) {
          kontakt->confirmed = TRUE;
          if (nick != NULL)
          	strncpy (kontakt->nick, nick, 19);
          Show_Quick_Status_lower (UPDATE_ONLINE | UPDATE_OFFLINE | UPDATE_NOTLIST, NULL);
        }
      }
    }
    else if (((Request*)requestdat->data)->type != FULL_INFO)
      return;
        
    if (result != 0x0A)
      return;
    
    if( kontakt->info == NULL )
      kontakt->info = g_malloc0( sizeof(USER_INFO_STRUCT));
    
    break;
  default:
  }

  switch (subtype) {
  case 0x9001:
  case 0xA401:
    v7_search_reply(reqid, result, data);
    break;
  case 0x9A01:
  case 0xAE01:
    v7_end_of_search_reply(reqid, result, data, end);
    break;
  case 0xC800:
    v7_parse_main_home_info(reqid, kontakt, result, data);
    break;
  case 0xDC00:
    v7_parse_more_info(reqid, kontakt, result, data);
    break;
  case 0xD200:
    v7_parse_work_info(reqid, kontakt, result, data);
    break;
  case 0xE600:
    v7_parse_about_info(reqid, kontakt, result, data);
    break;
  case 0xEB00:
    v7_parse_emails_info(reqid, kontakt, result, data);
  case 0x0E01:
    /* unknown */
  case 0xF000:
    /* personal-interests-info */
  case 0xFA00:
    /* past-background-info */

    /* This one seems to come last.. so we delete the requestdat here */
    if (requestdat)
      requests = g_slist_remove(requests, requestdat->data);
    break;
  }

}

void v7_offline_message(DWORD reqid, gchar *data)
{

  time_t msg_time;
  struct tm build_time;
  UIN_T uin;
  BYTE msgtype, msg_flags;
  WORD msglen;
  gchar *msg = data+14;
  gchar *desc, *nick, *first, *last, *email, *tmp=NULL;

#ifdef TRACE_FUNCTION
  g_print( "v7_offline_message\n" );
#endif
  
  uin = Chars_2_DW(data);

  build_time.tm_year = Chars_2_Word(data+4) - 1900;
  build_time.tm_mon = data[6]-1;  /* jan = 1 */
  build_time.tm_mday = data[7];
  build_time.tm_hour = data[8];
  build_time.tm_min = data[9];
  build_time.tm_sec = 0;
  msg_time = mktime(&build_time);

  msgtype  = data[10];
  msg_flags = data[11];
  msglen = Chars_2_Word(data+12);

  switch(msgtype) {
  case 0x01:
    msg_received(uin, msg, msg_time);
    break;
  case 0x04:
    desc = strchr( msg, '\xFE' ) + 1;
    *(desc-1) = '\0';
    url_received (uin, msg, desc, msg_time);
    break;
  case 0x0C:
    user_added_you(uin, msg_time);
    break;
  case 0x06:
    nick = msg;
    tmp = strchr( msg, '\xFE' );
    if ( tmp == NULL ) 
      return; /* Bad Packet */
    *tmp = '\0';
    rus_conv(RUS_WIN_KOI,nick);
    tmp++;

    first = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    last = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;
    rus_conv(RUS_WIN_KOI,last);

    email = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;
    rus_conv(RUS_WIN_KOI,email);

    desc = tmp;
    tmp = strchr( tmp, '\xFE' );
    if ( tmp == NULL )
      return; /* Bad Packet */
    *tmp = '\0';
    tmp ++;
    rus_conv(RUS_WIN_KOI,desc);

    authorisation_request(uin, nick, first, last, email, desc, msg_time);
    break;
  case 0x13:
    v7_rec_contactlist(uin, msg, msglen, msg_time);
    break;
  default:
    g_warning("Offline message type not handled, type: %x, text: %s\n", msgtype, msg);
  }

}

/* 
   Returns the end of what has been read (a pointer to what's next)
   for the last packet in row type
*/


gchar *v7_search_reply(DWORD reqid, BYTE result, gchar *here)
{

  WORD len, templen;
  gchar *hereorig = here;
  UIN_T uin=0;
  gchar *nick=NULL, *first=NULL, *last=NULL, *email=NULL;
  gchar auth=0, status=0, sex=0, age=0;
  
#ifdef TRACE_FUNCTION
  g_print( "v7_search_reply\n" );
#endif 

  len = Chars_2_Word (here);
  here += 2;

  uin = Chars_2_DW(here);
  here += 4;

  templen = Chars_2_Word(here);
  here += 2;
  if (strlen(here) > templen || hereorig+len <= here) goto full;
  nick = here;
  here += strlen(nick)+1;
  
  templen = Chars_2_Word(here);
  here += 2;
  if (strlen(here) > templen || hereorig+len <= here) goto full;
  first = here;
  here += strlen(first)+1;;
  
  templen = Chars_2_Word(here);
  here += 2;
  if (strlen(here) > templen || hereorig+len <= here) goto full;
  last = here;
  here += strlen(last)+1;;
  
  templen = Chars_2_Word(here);
  here += 2;
  if (strlen(here) > templen || hereorig+len <= here) goto full;
  email = here;
  here += strlen(email)+1;
  
  if (hereorig+len >= here+5) {
    auth   = here[0];
    status = here[1];
    sex    = here[3];
    age    = here[4];
  }
  else {
    auth = sex = age = 0;
    status = 2;
  }

 full:
  
  Display_Search_Reply(reqid, uin, nick, first, last, email, auth, status,
                       sex, age);

  return hereorig+len;
}

void v7_end_of_search_reply(DWORD reqid, BYTE result, gchar *here,
                            gchar *buffer_end)
{

#ifdef TRACE_FUNCTION
  g_print( "v7_end_of_search_reply\n" );
#endif 
  if (result != 0x32) {
    here = v7_search_reply(reqid, result, here)+1;

    if (here < buffer_end && CharsBE_2_DW(here))
      gnome_ok_dialog (_("Too many matches, not all have been retrieved"));
    else
      gnome_ok_dialog (_("Search complete."));
  }
  else
    gnome_ok_dialog (_("No matches found."));
}


/* Retrieves:
 * - Nickname
 * - First/Last Name
 * - Email
 * - City, State
 * - Phone, Fax
 * - Street Address
 * - Cellular phone
 * - Zip code, Country
 * - Time zone
 */
void v7_parse_main_home_info (DWORD reqid, Contact_Member *DestContact, 
                              BYTE result, gchar *here)
{
  GSList *requestdat;

  int len_str;
     
#ifdef TRACE_FUNCTION
  g_print( "v7_parse_main_home_info\n" );
#endif 

  requestdat = v7_get_request_by_reqid(reqid);
  
  g_free(DestContact->info->nick);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->nick = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->nick);
  here += len_str;
  if (((Request*)requestdat->data)->type == NEW_USER_VERIFICATION)
    strncpy(DestContact->nick, DestContact->info->nick, 20);

  g_free(DestContact->info->first);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->first = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->first);
  here += len_str;

  g_free(DestContact->info->last);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->last = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->last);
  here += len_str;

  g_free(DestContact->info->email);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->email = g_strdup( here );
  here += len_str;

  g_free(DestContact->info->city);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->city = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->city);
  here += len_str;

  g_free(DestContact->info->state);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->state = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->state);
  here += len_str;

  g_free(DestContact->info->phone);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->phone = g_strdup( here );
  here += len_str;

  g_free(DestContact->info->fax);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->fax = g_strdup( here );
  here += len_str;

  g_free(DestContact->info->street);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->street = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->street);
  here += len_str;

  g_free(DestContact->info->cellular);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->cellular = g_strdup( here );
  here += len_str;

  g_free(DestContact->info->zip);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->zip = g_strdup( here );
  here += len_str;

  DestContact->info->country = Chars_2_Word( here );
  here += 2;
  
  DestContact->info->timezone = *here;

  update_personal_info(DestContact->uin);
}

/* Retrieves:
 * - Age
 * - Sex
 * - Homepage
 * - Birth year, month, day
 * - Languages (3)
 */
void v7_parse_more_info (DWORD reqid, Contact_Member *DestContact, 
                         BYTE result, gchar *here)
{
  int len_str;

#ifdef TRACE_FUNCTION
  g_print( "v7_parse_more_info\n" );
#endif 
  
  if( Chars_2_Word( here ) != (WORD) -1 )
    DestContact->info->age = Chars_2_Word( here );
  else
    DestContact->info->age = 0;
  here += 2;

  /* 0x01: FEMALE, 0x02: MALE */
  DestContact->info->sex = *here;
  if ( *here > 2 || *here == 0)
    DestContact->info->sex = NOT_SPECIFIED;
  here++;

  g_free(DestContact->info->homepage);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->homepage = g_strdup( here );
  here += len_str;

  if( ( *(here+1) > 0 ) && ( *(here+1) < 12 ) )
    {
      DestContact->info->birth_year = Chars_2_Word(here);
      DestContact->info->birth_month = *(here+2);
      DestContact->info->birth_day = *(here+3);
    }
  here += 3;
  DestContact->info->language1 = here[0];
  DestContact->info->language2 = here[1];
  DestContact->info->language3 = here[2];
  
  update_personal_info(DestContact->uin);
}

/* Retrieves:
 * - About
 */
void v7_parse_about_info (DWORD reqid, Contact_Member *DestContact, 
                         BYTE result, gchar *here)
{
  int len_str;

#ifdef TRACE_FUNCTION
  g_print( "v7_parse_about_info\n" );
#endif 
  
  g_free(DestContact->info->about);
  len_str = Chars_2_Word( here );
  here += 2;  DestContact->info->about = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->about);

  
  update_personal_info(DestContact->uin);
}

/* Retrieves:
 * - Work city, state
 * - Work phone, fax
 * - Work address, zip code
 * - Company name
 * - Department
 * - Job position
 * - Work homepage
 */
void v7_parse_work_info (DWORD reqid, Contact_Member *DestContact, 
                         BYTE result, gchar *here)
{
  int len_str;

#ifdef TRACE_FUNCTION
  g_print( "v7_parse_work_info\n" );
#endif 
  
  g_free(DestContact->info->work_city);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->work_city = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->work_city);
  here += len_str;

  g_free(DestContact->info->work_state);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->work_state = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->work_state);
  here += len_str;

  g_free(DestContact->info->work_phone);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->work_phone = g_strdup( here );
  here += len_str;

  g_free(DestContact->info->work_fax);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->work_fax = g_strdup( here );
  here += len_str;

  g_free(DestContact->info->work_address);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->work_address = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->work_address);
  here += len_str;
  
  g_free(DestContact->info->work_zip);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->work_zip = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->work_zip);
  here += len_str;

  DestContact->info->work_country = Chars_2_Word( here );
  here += 2; /* Skip unknown data */

  g_free(DestContact->info->company_name);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->company_name = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->company_name);
  here += len_str;

  g_free(DestContact->info->department);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->department = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->department);
  here += len_str;

  g_free(DestContact->info->job_pos);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->job_pos = g_strdup( here );
  rus_conv(RUS_WIN_KOI, DestContact->info->job_pos);
  here += len_str;

  here += 2; /* Skip unknown here */

  g_free(DestContact->info->work_homepage);
  len_str = Chars_2_Word( here );
  here += 2;
  DestContact->info->work_homepage = g_strdup( here );
  here += len_str;
  
  update_personal_info(DestContact->uin);
}

    
void v7_parse_emails_info(DWORD reqid, Contact_Member *DestContact, 
                          BYTE result, gchar *here)
{
  int len_str, count, i;
  gchar *email;

#ifdef TRACE_FUNCTION
  g_print( "v7_parse_emails_info\n" );
#endif 
  
  count = here[0];
  here ++;

  for (i=0; i<count; i++) {
    here++; /* Skipping hide, if it was hidden, we wouldnt see it */
    len_str = Chars_2_Word( here );
    here += 2;
    email = g_malloc0(len_str+1);
    strncpy( email+1, here, len_str );
    here += len_str;
    
    rus_conv(RUS_WIN_KOI, email+1);
    DestContact->info->emails = g_list_append(DestContact->info->emails,
                                              email);
  }

  update_personal_info(DestContact->uin);
}


void v7_rec_contactlist(UIN_T uin, gchar *msg, int msglen, time_t msg_time)
{
  gchar *countstr, *tmp;
  GSList *contacts=NULL;
  ContactPair *cpair;
  int i, count;

#ifdef TRACE_FUNCTION
  g_print( "v7_rec_contactlist\n" );
#endif 

    countstr = msg;
    tmp = strchr( msg, '\xFE' );
    if ( tmp == NULL ) 
      return; /* Bad Packet */
    *tmp = '\0';
    tmp++;

    count = atoi(countstr);

    if (count == 0)
      return;

    for (i=0; i < count; i++) {
      cpair = g_malloc(sizeof(ContactPair));
      
      cpair->textuin = tmp;
      tmp = strchr( tmp, '\xFE' );
      if ( tmp == NULL )
        return; /* Bad Packet */
      *tmp = '\0';
      cpair->textuin = g_strdup(cpair->textuin);
      tmp++;
      
      cpair->nick = tmp;
      tmp = strchr( tmp, '\xFE' );
      if ( tmp == NULL && i != count-1) 
        return; /* Bad Packet */
      if (tmp)
        *tmp = '\0';
      cpair->nick = g_strdup(cpair->nick);
      tmp++;

      contacts = g_slist_append(contacts, cpair);
    }
    
    contact_list_received (uin, contacts, msg_time);
}

void  v7_rate_changed (Snac *snac, V7Connection *conn)
{

  WORD code, class;
  DWORD window_size, clear, alert, limit, disconnect, currentavg, maxavg;
  gchar *here;
  
#ifdef TRACE_FUNCTION
  g_print( "v7_rate_changed\n" );
#endif 

  here = snac->data;

  code = CharsBE_2_DW(here);
  here += 4;
  
  /*
    1 = Changed
    2 = Warning
    3 = Limit
    4 = Limit cleared
  */
  class = CharsBE_2_DW(here);
  here += 4;
  
  window_size = CharsBE_2_Word(here);
  here += 2;
  
  clear = CharsBE_2_Word(here);
  here += 2;
  
  alert = CharsBE_2_Word(here);
  here += 2;
  
  limit = CharsBE_2_Word(here);
  here += 2;
  
  disconnect = CharsBE_2_Word(here);
  here += 2;
  
  currentavg = CharsBE_2_Word(here);
  here += 2;
  
  maxavg = CharsBE_2_Word(here);
  here += 2;
  
  /*
  g_print("Code: %d Class: %d\n", code, class);
  g_print("WinSize: %d Clear: %d Alert: %d Limit: %d Discon: %d Avg: %d MaxAvg:%d\n", window_size, clear, alert, limit, disconnect, currentavg, maxavg);
  */
  if (class == 3)
    gnome_warning_dialog(_("You are sending messages too fast, your last message has been dropped, please wait 10 seconds and resend."));

}
