# GNU Enterprise Common - Application Services - Debugging support
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: $

import string
import sys
import time
from traceback import *
import inspect
import os
import __builtin__

_fh = sys.__stderr__
_conttest = 0
_DEBUG_LEVELS = []
_DEBUGGER = None

__starttime = time.time ()

class _stderrcatcher:
    def write(self, str):
        global _fh, _conttest
        for ch in str:
            if (_conttest == 0):
                _fh.write("DB000: ")
                _conttest = 1
            if ( ch != '\n'):
                _fh.write(ch)
            else:
                _fh.write('\n')
                _conttest = 0

    def writelines(self, list):
        for line in list:
            self.write(str)

def setDebugger(debugger):
    global _DEBUGGER
    _DEBUGGER = debugger

def setDebug (level, file=None):
  global _DEBUG_LEVELS, printMesg

  levels = []
  for entry in level.split(','):
    values=entry.split('-')
    if len(values) > 1:
      levels+=range(int(values[0]),int(values[1])+1)
    else:
      levels+=[int(entry)]
      
  _DEBUG_LEVELS=levels
      
  if _DEBUG_LEVELS != []:
    printMesg = _printMesg
  if (file):
    fh = open( file, 'w' )
    catchStderr( fh )
  else:
    catchStderr( sys.__stderr__ )
  __builtin__.__dict__['gDebug'] = printMesg


def catchStderr(fh):
    global _fh
    _fh = fh
    sys.stderr = _stderrcatcher()


def handleException(exc_info):
    #
    # Not used at present
    #
    type, exception, traceback = exc_info
    if (isinstance(exception, GETrace) ):
        printMesg( exception.level, exception.message)
    elif (not isinstance(exception, SystemExit)):
        strings = format_exception(type, exception, traceback)
        text = string.join(strings, '')
        printMesg(0, text)

#
# We use _noPrintMesg and _printMesg assignment to printMesg
# to speed things up when debug mode is not being used.  It's
# assigned in setDebug()
#
def _noPrintMesg(level, message, dropToDebugger=0):
  pass

def _printMesg (level, message, dropToDebugger = 0):
    if level in _DEBUG_LEVELS :
      global _fh, _DEBUGGER
      if type(message)==type(u''):
          message=message.encode('utf-8')
      caller = extract_stack()[-2]
      try:
        if caller[0][-3:] == '.py':
          file = "[%s:%s] " % (string.split(caller[0][:-3],'/')[-1], caller[1])
        else:
          file = "[%s:%s] " % (string.split(caller[0],'/')[-1], caller[1])
      except:
        file = ""

      __dumpMessage (level, file, message, dropToDebugger)


# -----------------------------------------------------------------------------
# Dump a message to the debug-output
# -----------------------------------------------------------------------------

def __dumpMessage (level, filename, message, dropToDebugger = False):
  """
  This function writes a message to the debug-output adding the time elapsed
  since __starttime.

  @param level: the debug-level the message will be logged in
  @param filename: the filename the message originated from
  @param message: the message to be logged
  @param dropToDebugger: if set to True, the message will be dropped to the
      currently set debugger.
  """
  
  s = time.time () - __starttime
  (m, s) = divmod (s, 60)
  (h, m) = divmod (m, 60)
  stamp  = "%d:%02d:%06.3f" % (h, m, s)

  lines = "%s" % message
  for line in lines.splitlines ():
    _fh.write ("DB%03d: %s %s%s%s" % (level, stamp, filename, line, os.linesep))

  if dropToDebugger and _DEBUGGER:
    _DEBUGGER.set_trace()



printMesg = _noPrintMesg

__builtin__.__dict__['gDebug'] = printMesg


# -----------------------------------------------------------------------------
# Add a function-signature to the debug output
# -----------------------------------------------------------------------------

def gEnter (level = 1):
  """
  This function adds another line to the debug-output, describing the caller's
  function with all it's arguments.

  @param level: the debug-level the message will be logged in
  """

  if not level in _DEBUG_LEVELS:
    return

  # Get the caller's frame
  frame = sys._getframe (1)

  try:
    (args, vargs, vkw, flocals) = inspect.getargvalues (frame)

    # If the function has a 'self' argument we add the class referenced by self
    # to the name of the function
    funcName = frame.f_code.co_name
    if 'self' in args:
      funcName = "%s.%s" % (flocals ['self'].__class__, funcName)

    params = []

    # First add all 'normal' arguments
    for item in args:
      value = item == 'self' and hex (id (flocals ['self'])) or flocals [item]
      params.append (repr (value))

    # Next, add all variable arguments (*arg)
    if vargs:
      params.extend ([repr (i) for i in flocals [vargs]])

    # and finally add all keyword arguments (**kwarg)
    if vkw is not None:
      params.extend (["%s = %s" % (repr (k), repr (v)) \
                      for (k, v) in flocals [vkw].items ()])

    message  = "Entering function %s (%s)" % (funcName,
                                              string.join (params, ", "))

    path = frame.f_code.co_filename
    if path [-3:] == '.py':
      path = string.split (path [:-3], '/') [-1]
    else:
      path = string.split (path, '/') [-1]

    filename = "[%s:%s] " % (path, frame.f_code.co_firstlineno)

    __dumpMessage (level, filename, message)

  finally:
    # Make sure to release the reference to the frame object. This keeps
    # garbage collection doing a fine job :)
    del frame


# -----------------------------------------------------------------------------
# Add a line to debug-output describing the end of a function call
# -----------------------------------------------------------------------------

def gLeave (level = 1, *result):
  """
  This function adds a line to the debug-output describing the end of a
  function call, optionally containing the function's result.

  @param level: debug-level to send the message to
  @param result: the function's result (if any)
  """

  # We cannot use something like 'retVal = len (result) and result [0] or None'
  # here, cause this won't work for 'empty' results like [] or {}.
  retVal = None
  if len (result):
    retVal = result [0]

  if not level in _DEBUG_LEVELS:
    return retVal

  # Get the caller's frame
  frame = sys._getframe (1)

  try:
    (args, vargs, vkw, flocals) = inspect.getargvalues (frame)

    # If the function has a 'self' argument we add the class referenced by self
    # to the name of the function
    fName = frame.f_code.co_name
    hId   = ''
    if 'self' in args:
      fName = "%s.%s" % (flocals ['self'].__class__, fName)
      hId   = repr (hex (id (flocals ['self'])))

    resStr  = len (result) and ' == %s' % repr (retVal) or ''
    message = "Leaving function %s (%s)%s" % (fName, hId, resStr)

    path = frame.f_code.co_filename
    if path [-3:] == '.py':
      path = string.split (path [:-3], '/') [-1]
    else:
      path = string.split (path, '/') [-1]

    filename = "[%s:%s] " % (path, frame.f_code.co_firstlineno)

    __dumpMessage (level, filename, message)

  finally:
    # Make sure to release the reference to the frame object. This keeps
    # garbage collection doing a fine job :)
    del frame
    return retVal




__builtin__.__dict__ ['gEnter'] = gEnter
__builtin__.__dict__ ['gLeave'] = gLeave


class GETrace(Exception):
    #
    #Exception class representing a debug message
    #not yet used for anything and probably won't be :)
    #
    def __init__(self, level=0, message="Trace Message"):
        Exception.__init__(self)
        self.level = level
        self.message = message
