#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# FILE:
# GNObjects.py
#
# DESCRIPTION:
# Set of classes that implement GNUe Navigator's xml-objects
#
# NOTES:
#

import sys, string, os, os.path
from gnue.common.definitions import GObjects, GRootObj
from gnue.common.apps import GDebug, GConfig
from gnue import paths
import GNParser
try:
  from gnue.reports.base import GREngine
  from gnue.reports.base import GRExceptions
  from gnue.reports.base import GRReportInfo
  from gnue.reports.client import GRRunUI

  REPORTS_SUPPORT = True

except ImportError:
  REPORTS_SUPPORT = False

# =============================================================================
# Base class for all Navigator objects
# =============================================================================

class GNObject (GObjects.GObj):
  pass


# =============================================================================
# This class implements the root objects of a process tree
# =============================================================================

class GNProcesses (GRootObj.GRootObj, GNObject):

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, parent = None):

    GRootObj.GRootObj.__init__ (self, 'processes', GNParser.getXMLelements,
                                GNParser)
    GNObject.__init__ (self, parent, type = "GNProcesses")
    self._clientHandlers = {'form'  : self._runForm,
                            'report': self._runReport,
                            'app'   : self._runApp}

  # ---------------------------------------------------------------------------
  # update the dictionary of client handlers
  # ---------------------------------------------------------------------------

  def setClientHandlers (self, handlers):
    """
    This function updates the dictionary of client handlers

    @param handlers: dictionary with key as handler type and a function pointer
        as value.
    """

    self._clientHandlers.update (handlers)


  # ---------------------------------------------------------------------------
  # Resolve the location of a step
  # ---------------------------------------------------------------------------

  def __resolveLocation (self, step, cfgDir):
    """
    This function resolves the location of a step using a base directory as
    defined by the configuration file. If the location of a step has no
    directory part it get's extended by the directory given by the config.

    @param step: GNStep instance
    @param cfgDir: section to take the default directory from.

    @return: location to use for the step
    """

    if os.path.basename (step.location) == step.location:
      try:
        stepdir = gConfigNav (cfgDir)

      except KeyError:
        stepdir = ""

      result = os.path.join (paths.data, stepdir, step.location)

    else:
      result = step.location

    return result


  # ---------------------------------------------------------------------------
  # Basic client handler for forms
  # ---------------------------------------------------------------------------

  def _runForm (self, step, parameters = {}):
    """
    """

    formCommand = gConfigNav ('RunFormCommand')
    formfile    = self.__resolveLocation (step, 'FormDir')

    if sys.platform != 'win32':
      gDebug (1, 'Running command " %s %s"' % (formCommand, formfile))
      os.spawnlpe (os.P_NOWAIT, formCommand, formCommand, formfile, os.environ)

    else:
      command = string.split ("%s" % (formCommand))
      command.append ('"%s"' % formfile)

      gDebug (1, 'Running command " %s "' % (command))
      os.spawnve (os.P_NOWAIT, command [0], command, os.environ)


  # ---------------------------------------------------------------------------
  # Basic client handler for apps
  # ---------------------------------------------------------------------------

  def _runApp (self, step, parameters = {}):
    """
    """

    gDebug (1, 'Running app "%s"' % (step.location))

    command = string.split (step.location)

    if sys.platform != 'win32':
      os.spawnvpe (os.P_NOWAIT, command [0], command, os.environ)
    else:
      # TODO: Not tested
      os.spawnve (os.P_NOWAIT, command [0], command, os.environ)


  # ---------------------------------------------------------------------------
  # Basic client handler for reports
  # ---------------------------------------------------------------------------

  def _runReport (self, step, parameters = {}):
    """
    """

    global REPORTS_SUPPORT

    if not REPORTS_SUPPORT:
      print "WARNING: Reports support not installed!"
      return

    reportfile = self.__resolveLocation (step, 'ReportDir')
    reportInfo = GRReportInfo.GRReportInfo (reportfile)

    # default parameter should be loaded from GREngine itself
    params = {
      'destination_type'   : 'file',
      'destination'        : '-',
      'destination_options': '',
      'filter'             : 'raw',
      'list_filters'       : 0,
      'filter_options'     : '',
      'sortoption'         : '',
      'comment'            : None,
      'exclude_xml'        : None}

    # If a step defines one of the standard parameters take it over and remove
    # it from the step's private parameter dictionary
    for key in params.keys ():
      if step._params.has_key (key):
        params [key] = step._params [key]
        del step._params [key]

    topObject = self.findParentOfType (None)

    runUI = GRRunUI.GRRunUI (reportInfo,
                             topObject._configManager,
                             topObject._connections,
                             topObject._ui_type,
                             step._params,
                             params ['destination'],
                             params ['destination_type'],
                             params ['filter'],
                             params ['sortoption'])
    runUI.run ()

    params ['destination']      = runUI.getDestination ()
    params ['destination_type'] = runUI.getDestinationType ()
    params ['filter']           = runUI.getFilter ()
    params ['sortoption']       = runUI.getSortOption ()
    userParameters              = runUI.getParameters ()

    try:
      engine = GREngine.GREngine (topObject._connections)
      engine.processReport (reportfile,
                           destination = params["destination"],
                           destinationType = params["destination_type"],
                           destinationOptions = params["destination_options"],
                           filter = params['filter'],
                           filterOptions = params['filter_options'],
                           parameters = userParameters,
                           sortoption=params["sortoption"],
                           includeStructuralComments=params["comment"],
                           omitGNUeXML=params["exclude_xml"])

    except GRExceptions.ReportMarkupException, msg:
      self.handleStartupError (msg)
    except GRExceptions.ReportUnreadable, msg:
      self.handleStartupError (msg)




  def handleStartupError(self,errortext):
    print
    print '-' * 60
    print _("Error: %s") % errortext
    print '-' * 60

    sys.exit()


class GNProcess(GNObject):
  def __init__(self, parent):
    GNObject.__init__(self, parent, type="GNProcess")


class GNStep(GNObject):
  def __init__(self, parent):
    GNObject.__init__(self, parent, type="GNStep")
    self._params = {}
    
    self._inits = [self.primaryInit]
    
  def primaryInit(self):
    for child in self._children:
      if child._type=="GNParameter":
        self._params[child.name]=child.value
        
  def _buildObject(self):
    self._controller = self.findParentOfType('GNProcesses')

  def run(self):
    try:
      GDebug.printMesg(1,'Running %s %s' % (self.type, self.location))
      self._controller._clientHandlers[self.type](self, self._params)
    except KeyError:
      print 'Error! Unknown step type "%s"' % self.type


class GNDescription(GNObject):
  def __init__(self, parent):
    GNObject.__init__(self, parent, type="GNDescription")

class GNParameter(GNObject):
  def __init__(self, parent):
    GNObject.__init__(self, parent, type="GNParameter")


class GNImportItem(GObjects.GObj):
  def __init__(self, parent=None, type="GNImport-Item"):
    GObj.__init__(self, parent, type=type)
    self._loadedxmlattrs = {} # Set by parser

  def _buildObject(self):
    if hasattr(self,'_xmltag'):
      self._type = 'GN%s' % self._xmltag
    if not hasattr(self,'_importclass'):
      self._importclass = GNParser\
         .getXMLelements()[string.lower(self._type[9:])][0]
    return GObj._buildObject(self)

