/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*----------------------------------------------------------------------
 
  gpiv - Graphic program for Particle Image Velocimetry, based on gtk/gnome
          libraries.

   Copyright (C) 2002 Gerber van der Graaf

   This file is part of gpiv.

   Gpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

----------------------------------------------------------------------*/

/*
 * Graphical User Interface for gpiv
 * $Log: gpiv_gtk.h,v $
 * Revision 1.9  2005/01/19 15:53:41  gerber
 * Initiation of Data Acquisition (DAC); trigerring of lasers and camera
 * by using RTAI and Realtime Linux, recording images from IEEE1394
 * (Firewire) IIDC compliant camera's
 *
 * Revision 1.8  2004/10/15 19:24:05  gerber
 * GPIV_ and Gpiv prefix to defines and structure names of libgpiv
 *
 * Revision 1.7  2004/06/14 21:19:23  gerber
 * Image depth up to 16 bits.
 * Improvement "single int" and "drag int" in Eval tab.
 * Viewer's pop-up menu.
 * Adaption for gpiv_matrix_* and gpiv_vector_*.
 * Resizing console.
 * See Changelog for further info.
 *
 * Revision 1.6  2003/09/04 13:31:55  gerber
 * init of printing (unfinished)
 *
 * Revision 1.5  2003/09/01 11:17:14  gerber
 * improved monitoring of interrogation process
 *
 * Revision 1.4  2003/08/22 15:24:52  gerber
 * interactive spatial scaling
 *
 * Revision 1.3  2003/07/25 15:40:23  gerber
 * removed/disabled setting of correlation in Eval tab, Correlation type in Image info tab
 *
 * Revision 1.2  2003/06/27 13:47:26  gerber
 * display ruler, line/point evaluation
 *
 * Revision 1.1.1.1  2003/06/17 17:10:52  gerber
 * Imported gpiv
 *
 */

#ifndef GPIV_H
#define GPIV_H

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h> /* for the "pause" function: */
#include <time.h>
#include <string.h>
#include <assert.h>
#include <math.h> /* for isnan */
#include <gnome.h>

#ifdef HAVE_GNOME_PRINT
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-job.h>
#include <libgnomeprint/gnome-font.h>
#include <libgnomeprintui/gnome-print-dialog.h>
#include <libgnomeprintui/gnome-print-job-preview.h>
#endif /* HAVE_GNOME_PRINT */

#include <gpiv.h>

/*  Revision Control System (RCS) version */
#define RCSID "$Id: gpiv_gtk.h,v 1.9 2005/01/19 15:53:41 gerber Exp $" 
#define ERR_IVAL = 99           /* Some arbitrary integer error value */
#define ADJ_MIN -50.0	        /* minimum value for adjustment */
#define ADJ_MAX 50.0	        /* maximum value for adjustment */
#define ADJ_STEP 0.001          /* Step increment for adjustment */
#define ADJ_PAGE 0.1            /* Page increment for adjustment */
#define ADJ_ISTEP 1             /* Step increment for adjustment for integers */
#define ADJ_IPAGE 10            /* Page increment for adjustment for integers*/

#define MAX_LIST 5
/*
 * SVGA: 1280 x 1024
 * Sony PCR-100 video camera: 1152 x 864
 */
#ifndef IMAGE_WIDTH_MAX
#define IMAGE_WIDTH_MAX     GPIV_MAX_IMG_SIZE
#endif /* IMAGE_WIDTH_MAX */

#ifndef IMAGE_HEIGHT_MAX
#define IMAGE_HEIGHT_MAX    GPIV_MAX_IMG_SIZE
#endif  /* IMAGE_HEIGHT_MAX */

#ifndef IMAGE_DEPTH_MAX
#define IMAGE_DEPTH_MAX     GPIV_MAX_IMG_DEPTH
#endif  /* IMAGE_DEPTH_MAX */


char IMAGE_CORRELATION_LABEL[GPIV_MAX_CHARS];
char IMAGE_WIDTH_LABEL[GPIV_MAX_CHARS];
char IMAGE_HEIGHT_LABEL[GPIV_MAX_CHARS];
char IMAGE_DEPTH_LABEL[GPIV_MAX_CHARS];

/*
 * Extra marge for display
 */
/* #define VIEW_HMARGE 38 */
/* #define VIEW_VMARGE 62 */
#define VIEW_HMARGE 19
#define VIEW_VMARGE 31
/* #define VIEW_HMARGE 0 */
/* #define VIEW_VMARGE 0 */
#define RULER_WIDTH 17  /* width of the rulers in the display window */

#define MAX_DATA (IMAGE_WIDTH_MAX / GPIV_MIN_INTERR_SIZE)    /* Maximum number of data/estimators/Interr. Areas per row */
#define MAX_BUFS GPIV_NIMG_MAX
/*20*/		  /* maximum number of buffers (=displays) */

/* #define MAX_SWEEP 6 */

/* #define CANVAS_AA */               /* Use Anti Aliased canvas for viewer */
/* BUGFIX: Color presenation for AA */
#define THICKNESS 1		/* line thickness to be drawn in te canvases */
#define ARROW_LENGTH 0.4	/* length of (piv) vector arrow head */
#define ARROW_EDGE 0.6 		/* edge of (piv) vector arrow head */
#define ARROW_WIDTH 0.2 	/* width of (piv) vector arrow head */
#define ARROW_ADD 2.0		/* additional quantity for arrow head */
#define ARROW_FACT 0.2		/* magnification factor of vector_scale for arrow head */

#define BYTEVAL 255
#define BITSHIFT_RED 24         /* bitshift for color red value (used in display_all_scalar_intregs) */
#define BITSHIFT_GREEN 16       /* bitshift for color green value */
#define BITSHIFT_BLUE 8         /* bitshift for color blue value */


#include "console_interface.h"
#include "display_interface.h"


enum TablabelPage {
#ifdef DISABLE_DAC
    PAGE_IMGH,
#else
    PAGE_DAC,
    PAGE_IMGH,
#endif
    PAGE_PIVEVAL,
    PAGE_PIVVALID,
    PAGE_PIVPOST
};


enum  ShowBackground {
    SHOW_BG_DARKBLUE,
    SHOW_BG_BLACK,
    SHOW_BG_IMG1,
    SHOW_BG_IMG2
};


enum  ShowScalar {
    SHOW_SC_NONE,
    SHOW_SC_VORTICITY,
    SHOW_SC_SSTRAIN,
    SHOW_SC_NSTRAIN
};


enum ZoomIndex {
    ZOOM_0,
    ZOOM_1,
    ZOOM_2,
    ZOOM_3,
    ZOOM_4,
    ZOOM_5,
    ZOOM_6,
    ZOOM_7
};


static float zfactor[] = {
    0.25, 
    0.5, 
    0.83, 
    1.0, 
    1.3, 
    1.6, 
    2.0,
    4.0
};


enum  VectorScale {
    VECTOR_SCALE_0 = 1,
    VECTOR_SCALE_1 = 2,
    VECTOR_SCALE_2 = 4,
    VECTOR_SCALE_3 = 8,
    VECTOR_SCALE_4 = 16,
    VECTOR_SCALE_5 = 32,
    VECTOR_SCALE_6 = 64,
    VECTOR_SCALE_7 = 128,
    VECTOR_SCALE_8 = 256
} /* v_scale */;


enum VectorColor {
    SHOW_PEAKNR,
    SHOW_SNR,
    SHOW_MAGNITUDE_GRAY,
    SHOW_MAGNITUDE
} v_color;


enum WidgetSet {
    DAC,
    DAC_TRIG,
    DAC_TIMING,
    DAC_CAM,
    IMG,
    IMGPROC,
    EVAL,
    INTREGS,
    VALID,
    POST
};


enum MouseSelect {
    NO_MS,
    AOI_MS,
    SINGLE_AREA_MS,
    SINGLE_POINT_MS,
    DRAG_MS,
    V_LINE_MS,
    H_LINE_MS,
    ENABLE_POINT_MS,
    DISABLE_POINT_MS,
    ENABLE_AREA_MS,
    DISABLE_AREA_MS,
    SPANLENGTH_MS,
    V_SPANLENGTH_MS,
    H_SPANLENGTH_MS
} m_select;


enum ClistPut {
    PREPEND,
    INSERT,
    APPEND
};


/*
 * Variables of gpiv that will be stored for future sessions, but cannot be
 * defined as keys during startup of gpiv:
 */
typedef struct __GpivVar Var;
struct __GpivVar {
    enum TablabelPage tab_pos;  /* page of the notebook to be shown */
    gint number_fnames_last;
    gchar *fn_last[MAX_LIST];   /* last image name that has been loaded */
    gchar *fname_last;          /* last image name that has been loaded */
    gchar *fname_last_print;    /* last filename that has been printed to */
    gboolean fname_date;        /* Extends the name with current date */
    gboolean fname_time;        /* Extends the name with current time */
    gfloat img_span_px;         /* spanned length that takes N mm in the img */
    gfloat img_length_mm;       /* length in mm to be measured in img */
    gfloat piv_disproc_zoom;    /* magnification factor for displaying correlation in PIV evaluation tabulator */
    gint piv_disproc_vlength;   /* vector length of PIV estimator in PIV evaluation tabulator */
    gfloat dl_min;              /* minimum displacement of a piv data-set */
    gfloat dl_max;              /* maximum displacement of a piv data-set */
} gpiv_var;


/*
 * Parameters of gpiv:
 */
typedef struct __GpivPar Par;
struct __GpivPar {
/*
 * BUGFIX TODO
 * Use an enum  ShowBackground integer for display_img1, display_img2
 * Use an enum  ShowScalar integer for display_none, display_vor, sstrain, 
 * display_nstrain
 */
    gboolean display_img1;      /* display first image in display */
    gboolean display_img2;      /* display second image in display */
    gboolean display_intregs;   /* display interrogation regions in display */
    gboolean display_piv;       /* display piv vectors in display */
    gboolean display_none;      /* no displaying of scalar data */
    gboolean display_vor;       /* display vorticity in display */
    gboolean display_sstrain;   /* display shear strain in display */
    gboolean display_nstrain;   /* display normal strain in display */
    gboolean display_process;	/* display interrogation areas and covariance function during image analyzing process  */
    gboolean show_tooltips;	/* flag to show tooltips or hints (in pop-up window) */
    
    gboolean process_dac;       /* used by gpiv toolbar to run dac */
    gboolean process_piv;       /* used by gpiv toolbar to run piv */
    gboolean process_gradient;  /* used by gpiv toolbar to run gradient */
    gboolean process_resstats;  /* used by gpiv toolbar to run resstats */
    gboolean process_errvec;    /* used by gpiv toolbar to run errvec */
    gboolean process_peaklock;  /* used by gpiv toolbar to run peaklock */
    gboolean process_average;   /* used by gpiv toolbar to run average */
    gboolean process_scale;     /* used by gpiv toolbar to run scale */
    gboolean process_subtract;  /* used by gpiv toolbar to run subtract */
    gboolean process_vorstra;   /* used by gpiv toolbar to run vorstra */
    
    enum  ShowBackground background;    /* display background color */
    gboolean stretch;           /* stretch automatic window after sooming in */
    enum ZoomIndex zoom_index;  /* index for displayed zooming */
    enum  VectorScale vector_scale;     /* scale of vectors to be displayed */
    enum VectorColor vector_color;      /* color representation of vectors */
    gboolean stretch_window;    /* stretch display window for image area */
    gboolean view_tabulator;    /* display tabulator of process parameters in main window */
    gboolean view_gpivbuttons;  /* display gpiv buttons in main window */
    gint nbins;                 /* number of bins to display histograms */

    gboolean trigger_cam;       /* enable/disable camera and laser triggering */

    gboolean x_corr;            /* images for cross correlation */
    gint img_width;             /* number of columns in image */
    gint img_height;            /* number of rows in image */
    gint img_depth;             /* image depth (in bits) */

    gboolean hdf;               /* store data in hdf 5 format, with .gpi extension */
    gboolean hdf_img;           /* include image data in .gpi file (hdf 5 format) */

/*
 * all logics whether have been copied/read
 */
    gboolean display_img1_logic;      /* display first image in display */
    gboolean display_img2_logic;      /* display second image in display */
    gboolean display_intregs_logic;   /* display interrogation regions in display */
    gboolean display_piv_logic;       /* display piv vectors in display */
    gboolean display_none_logic;      /* no displaying of scalar data */
    gboolean display_vor_logic;       /* display vorticity in display */
    gboolean display_sstrain_logic;   /* display shear strain in display */
    gboolean display_nstrain_logic;   /* display normal strain in display */
    gboolean display_process_logic;	/* display interrogation areas and covariance function during image analyzing process  */
    gboolean show_tooltips_logic;	/* flag to show tooltips or hints (in pop-up window) */
    
    gboolean process_dac_logic;       /* used by gpiv toolbar to run dac */
    gboolean process_piv_logic;       /* used by gpiv toolbar to run piv */
    gboolean process_gradient_logic;  /* used by gpiv toolbar to run gradient */
    gboolean process_resstats_logic;  /* used by gpiv toolbar to run resstats */
    gboolean process_errvec_logic;    /* used by gpiv toolbar to run errvec */
    gboolean process_peaklock_logic;  /* used by gpiv toolbar to run peaklock */
    gboolean process_average_logic;   /* used by gpiv toolbar to run average */
    gboolean process_scale_logic;     /* used by gpiv toolbar to run scale */
    gboolean process_subtract_logic;  /* used by gpiv toolbar to run subtract */
    gboolean process_vorstra_logic;   /* used by gpiv toolbar to run vorstra */
    
    gboolean background_logic;    /* display background color */
    gboolean stretch_logic;           /* stretch automatic window after sooming in */
    gboolean zoom_index_logic;  /* index for displayed zooming */
    gboolean vector_scale_logic;     /* scale of vectors to be displayed */
    gboolean vector_color_logic;      /* color representation of vectors */
    gboolean stretch_window_logic;    /* stretch display window for image area */
    gboolean view_tabulator_logic;    /* display tabulator of process parameters in main window */
    gboolean view_gpivbuttons_logic;  /* display gpiv buttons in main window */
    gboolean nbins_logic;                 /* number of bins to display histograms */

    gboolean trigger_cam_logic;       /* enable/disable camera and laser triggering */

    gboolean x_corr_logic;            /* images for cross correlation */
    gboolean img_width_logic;             /* number of columns in image */
    gboolean img_height_logic;            /* number of rows in image */
    gboolean img_depth_logic;             /* image depth (in bits) */

    gboolean hdf_logic;               /* store data in hdf 5 format, with .gpi extension */
    gboolean hdf_img_logic;           /* include image data in .gpi file (hdf 5 format) */


} default_par, gpiv_par;


/*
 * Parameters and variables for each process:
 * ImagePar should be unique for each image
 */
#ifndef DISABLE_DAC
GpivDacPar dac_par, dac_par_default;
GpivCamVar cam_var;
#endif /* DISABLE_DAC */
GpivImagePar image_par;
GpivEvalPar piv_eval_par, piv_eval_par_default;
GpivValidPar piv_valid_par, piv_valid_par_default;
GpivPostPar piv_post_par, piv_post_par_default;


/*
 * Other global variables
 */
GnomeProgram *Gpiv_app;
GnomeCanvasItem *gci_aoi, *gci_line;

char fname[GPIV_MAX_CHARS];
gboolean print_par, verbose, exec_process, cancel_process;

Display *display[MAX_BUFS], *display_act;
gint nbufs;

gint camera_act;
gchar *msg_default, msg_display[40];
char c_line[GPIV_MAX_LINES_C][GPIV_MAX_CHARS];
int var_scale, nc_lines, display_intregs;
int m_select_index_y, m_select_index_x;

/* flag to suppres creating intregs during load_buffer; after displaying 
 * results into correct effect */
gboolean view_toggle_intregs_flag;



#endif /* GPIV_H */
