/* $Id: gps.cc,v 1.23 2001/02/23 13:48:05 bergo Exp $ */

/*

    GPS - Graphical Process Statistics
    Copyright (C) 1999-2000 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "transient.h"

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <pwd.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/resource.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "details.h"
#include "diefast.h"
#include "direct.h"
#include "filter.h"
#include "gps.h"
#include "history.h"
#include "msgbox.h"
#include "polling.h"
#include "nativepollers.h"
#include "netwatch.h"
#include "netpolling.h"
#include "renice.h"
#include "tabledata.h"
#include "treeview.h"
#include "tstring.h"
#include "userbox.h"

#include "pixmaps/xray-tux.xpm"
#include "pixmaps/wheel.xpm"

#include "pixmaps/spinnera.xpm"
#include "pixmaps/spinnerb.xpm"
#include "pixmaps/spinnerc.xpm"
#include "pixmaps/halt.xpm"

#include "pixmaps/hgreen.xpm"
#include "pixmaps/hred.xpm"
#include "pixmaps/hgmem.xpm"
#include "pixmaps/hgswap.xpm"

#include "importglobals.h"

GtkWidget *psl,*show_long_names_item,*local_poll_item,
          *item2,*refb,*kb[2],*sc[4],*wheel,*nethalt,*mainmenu,
          *closebutton;

/* progress wheel */
int wheel_state=0; // 0-4
GdkPixmap *wheel_canvas=NULL;

/* multi-thread, multi-problems */
int freeze_level=0;

gint rows_on_list=0;
tabledata *mainlist=NULL;

gint to_tag=-1;

gint waysort[PROCESS_LIST_COLUMNS]={0,0,0,0,0,0,0,0,0,0,0}; /* 0=asc 1=des */
gint last_sort=0;

/* visible fields stuff */
gint visible_fields[PROCESS_LIST_COLUMNS]={1,1,1,1,1,1,1,1,1,1,1};
gint startup_width[PROCESS_LIST_COLUMNS]=
                              {50,90,50,55,35,35,45,45,30,30,150};

gchar col_desc[PROCESS_LIST_COLUMNS][48]=
      {"PID (Process ID)",
       "Name (or command line)",
       "Owner's username",
       "Hostname",
       "Process state",
       "CPU usage",
       "Size (memory+swap footprint)",
       "RSS (Resident Segment Size = memory footprint)",
       "Nicety (static priority)",
       "Pri (dynamic priority)",
       "Start time"};

static char smonth[12][4]={"Jan","Feb","Mar","Apr","May","Jun",
			   "Jul","Aug","Sep","Oct","Nov","Dec"};

int delete_only_once=0,got_a_poller=0,wanttop=0;
			   
int main (int argc, char *argv[])
{
  int i,j;
  gint dashc=FALSE,dashl=FALSE;
  int nostartingnet=0;
  char *dmode1,*dmode2,*dmode3;

  // direct mode
  if (argc>=2) {
    if (argv[1][0]=='k') {
      dmode1=(char *)g_malloc(1024);
      dmode2=(char *)g_malloc(1024);
      dmode3=(char *)g_malloc(1024);
      strcpy(dmode1,argv[1]);
      dmode2[0]=0;
      strcpy(dmode3,"HUP");
     
      for(i=2;i<argc;i++) {
	if (argv[i][0]=='-') {
	  strcpy(dmode3,&argv[i][1]);
	  continue;
	}
	strcat(dmode2,argv[i]);
	strcat(dmode2,"&");
      }

      init_local_polling();
      i=direct_mode_kill(dmode1,dmode2,dmode3);
      close_local_polling();
      g_free(dmode1);
      g_free(dmode2);
      g_free(dmode3);
      return i;
    }
  }

  init_default_colors();

  for(i=1;i<argc;i++) {
    if (strcmp(argv[i],"-gb")==0) {
      jelly_bar=TRUE;
      continue;
    }
    if (strcmp(argv[i],"-gl")==0) {
      init_low_depth_colors();
      continue;
    }
    if (strcmp(argv[i],"-c")==0) {
      dashc=TRUE;
      continue;
    }
    if (strcmp(argv[i],"-l")==0) {
      dashl=TRUE;
      continue;
    }
    if (strcmp(argv[i],"-pt")==0) {
      wanttop=1;
      continue;
    }
    if (strcmp(argv[i],"-dn")==0) {
      nostartingnet=1;
      continue;
    }
    if ((i+1)<argc)
      if (strcmp(argv[i],"-sc")==0) {
	j=atoi(argv[i+1]);
	i++;
	if (j>9) j=0;
	last_sort=j;
	continue;
      }
    if (strcmp(argv[i],"-sd")==0) {
      for(j=0;j<PROCESS_LIST_COLUMNS;j++)
	waysort[j]=1;
      continue;
    }
    if (strcmp(argv[i],"-sa")==0) {
      for(j=0;j<PROCESS_LIST_COLUMNS;j++)
	waysort[j]=0;
      continue;
    }

    if ((strcmp(argv[i],"--version")==0)||(strcmp(argv[i],"-v")==0)) {
      printf("gPS version %s  (C) 1999-2000 Felipe Bergo <bergo@seul.org>\n",
	     GPS_RELEASE);
      printf("Distributed under the terms of the"\
	     " GNU General Public License\n\n");
      return(0);
    }

    fprintf(stderr,"** gps: unrecognized command line argument %s,"\
	           " exiting.\n\n",argv[i]);
    return(2);
  }
    
  gtk_init (&argc, &argv);

  if (dashc)
    continuum=TRUE;

  if (dashl)
    show_long_names=TRUE;

  read_rc();

#ifdef ENABLE_NETWORK
  if (!nostartingnet)
    netwatch_init();
#endif

  init_local_polling();
  init_main_window ();
  nice(8);
  gotgui=1;
  refresh();
  gdk_rgb_init(); /* for history gauges */
  gtk_main ();

  gotgui=0;

  netwatch_destroy();
  close_local_polling();

  return 0;
}

void init_local_polling() {
  gethostname(this_host,128); // local name
  
#ifdef HAVELINUXPROCFS
  if (!wanttop) {
    list_poller=new LinuxProcFsProcessListPoller();
    details_poller=new LinuxProcFsProcessDetailsPoller();
    info_provider=new LinuxProcFsSystemInfoProvider();
    got_a_poller=1;
    delete_only_once=0;
  }
#endif

#ifdef HAVEFREEBSD
  if (!wanttop) {
    list_poller=new FreeBSDProcessListPoller();
    details_poller=new FreeBSDProcessDetailsPoller();
    info_provider=new FreeBSDSystemInfoProvider();
    got_a_poller=1;
    delete_only_once=0;
  }
#endif

#ifdef HAVETOP
  if (got_a_poller==0) { /* see pipedtop.h / .cc */
    list_poller=new PipedTopListPoller();
    details_poller=new PipedTopDetailsPoller();
    info_provider=new PipedTopInfoProvider();
    got_a_poller=1;
    delete_only_once=0;
  }
#endif
}

void close_local_polling() {
  if (!delete_only_once) {
    list_poller->terminate();
    details_poller->terminate();
    info_provider->terminate();
    delete(list_poller);
    delete(details_poller);
    delete(info_provider);
  } else {
    delete(list_poller);
  }
}

void init_main_window() {
  GtkWidget *mainw,*v1,*h1,*pl,*b1,*b2,*b3,*b4,*sw,*h4,*h5;
  GtkWidget *mb,*mi[40],*mj[20],*m,*n,*o,*net,*sbhb,*al[4];
  GtkWidget *bott,*both[3],*botf[3];
  GtkAccelGroup *mmag;

  GtkStyle *style;
  GdkBitmap *mask;
  GdkPixmap *xray,*phalt;

  int i;
  char *buffer,*bu2;

  buffer=(char *)g_malloc(512);
  bu2=(char *)g_malloc(512);
  gethostname(bu2,512);
  sprintf(buffer,"gPS: Process List (%s)",bu2);

  /* MAIN WINDOW */
  mainw=gtk_window_new(GTK_WINDOW_TOPLEVEL);
  MainWindow=GTK_WINDOW(mainw);
  gtk_window_set_default_size(GTK_WINDOW(mainw),650,400);
  gtk_window_set_position (GTK_WINDOW(mainw), GTK_WIN_POS_CENTER);
  gtk_window_set_title (GTK_WINDOW (mainw), buffer);
  gtk_widget_show (mainw);

  g_free(bu2);
  g_free(buffer);
  
  /* MENU */
  v1=gtk_vbox_new(FALSE,0);
  gtk_container_add(GTK_CONTAINER(mainw),v1);  

  mmag=gtk_accel_group_new();

  /* View */
  m=gtk_menu_new();
  mi[0]=gtk_check_menu_item_new_with_label("Show long process names");
  mi[1]=gtk_check_menu_item_new_with_label("Continuous Refresh");
  mi[2]=gtk_menu_item_new_with_label("Settings...");

  mi[3]=gtk_menu_item_new();

  mi[4]=gtk_menu_item_new_with_label("Tree view...");
  mi[5]=gtk_menu_item_new_with_label("CPU and memory usage...");
  mi[6]=gtk_menu_item_new_with_label("CPU and memory usage per user...");

  mi[7]=gtk_menu_item_new();
  mi[8]=gtk_menu_item_new_with_label("About gPS...");
  mi[9]=gtk_menu_item_new_with_label("Close");

  show_long_names_item=mi[0];
  item2=mi[1];
  gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(mi[0]),TRUE);
  gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(mi[1]),TRUE);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(show_long_names_item),
				 show_long_names);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item2),
				 continuum);
  for(i=0;i<10;i++)
    gtk_menu_append(GTK_MENU(m),mi[i]);
  for(i=0;i<10;i++) {
    gtk_widget_show(mi[i]);
  }

  gtk_accel_group_add(mmag,GDK_L,GDK_CONTROL_MASK,GTK_ACCEL_VISIBLE,
		      GTK_OBJECT(mi[0]),"activate");
  gtk_accel_group_add(mmag,GDK_U,GDK_CONTROL_MASK,GTK_ACCEL_VISIBLE,
		      GTK_OBJECT(mi[5]),"activate");
  gtk_accel_group_add(mmag,GDK_Q,GDK_CONTROL_MASK,GTK_ACCEL_VISIBLE,
		      GTK_OBJECT(mi[9]),"activate");
  for(i=0;i<10;i++)
    gtk_widget_lock_accelerators(mi[i]);


#ifndef ENABLE_SYSINFO
	gtk_widget_set_sensitive(mi[5],FALSE);
#endif

  gtk_widget_set_sensitive(mi[3],FALSE);
  gtk_widget_set_sensitive(mi[7],FALSE);

  gtk_signal_connect (GTK_OBJECT (mi[0]), "activate",
		      GTK_SIGNAL_FUNC (toggle_long), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[1]), "activate",
		      GTK_SIGNAL_FUNC (toggle_cont), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[2]), "activate",
		      GTK_SIGNAL_FUNC (settings), NULL);

  gtk_signal_connect (GTK_OBJECT (mi[4]), "activate",
		      GTK_SIGNAL_FUNC (pop_treeview), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[5]), "activate",
		      GTK_SIGNAL_FUNC (pop_history), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[6]), "activate",
		      GTK_SIGNAL_FUNC (pop_user_box), NULL);

  gtk_signal_connect (GTK_OBJECT (mi[8]), "activate",
		      GTK_SIGNAL_FUNC (about_gps), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[9]), "activate",
		      GTK_SIGNAL_FUNC (destroy), NULL);

  /* Filter */
  n=gtk_menu_new();
  mi[5]=gtk_menu_item_new_with_label("Set filter...");
  mi[6]=gtk_menu_item_new_with_label("Clear filter");

  for(i=5;i<7;i++)
    gtk_menu_append(GTK_MENU(n),mi[i]);
  for(i=5;i<7;i++) {
    gtk_widget_lock_accelerators(mi[i]);
    gtk_widget_show(mi[i]);
  }

  gtk_signal_connect (GTK_OBJECT (mi[5]), "activate",
		      GTK_SIGNAL_FUNC (set_filter_dialog), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[6]), "activate",
		      GTK_SIGNAL_FUNC (clear_filter), NULL);

  /* Action */
  o=gtk_menu_new();
  mi[7]=gtk_menu_item_new_with_label("Send Hang Up Signal (SIGHUP)");
  mi[8]=gtk_menu_item_new_with_label("Send Kill Signal (SIGKILL)");
  mi[9]=gtk_menu_item_new_with_label("Send POSIX Signal");
  mi[10]=gtk_menu_item_new();
  mi[11]=gtk_menu_item_new_with_label("Renice...");
  mi[12]=gtk_menu_item_new();
  mi[13]=gtk_menu_item_new_with_label("Details...");

  sc[0]=mi[7];
  sc[1]=mi[8];
  sc[2]=mi[11];

  /* POSIX critters */
  mj[0]=gtk_menu_new();

  buffer=(char *)g_malloc(128);
  
  for(i=0;i<POSIXSIG;i++) {
    sprintf(buffer,"%s (%d)",posixnames[i],posixvalues[i]);
    mj[i+1]=gtk_menu_item_new_with_label(buffer);
    gtk_menu_append(GTK_MENU(mj[0]),mj[i+1]);

    switch(posixvalues[i]) {
    case SIGSTOP:
      gtk_accel_group_add(mmag,GDK_S,GDK_CONTROL_MASK,GTK_ACCEL_VISIBLE,
			  GTK_OBJECT(mj[i+1]),"activate");
      break;
    case SIGCONT:
      gtk_accel_group_add(mmag,GDK_C,GDK_CONTROL_MASK,GTK_ACCEL_VISIBLE,
			  GTK_OBJECT(mj[i+1]),"activate");
      break;
    }

    gtk_widget_lock_accelerators(mj[i+1]);
    gtk_signal_connect(GTK_OBJECT(mj[i+1]),"activate",
		       (GtkSignalFunc)PosixKill,&posixvalues[i]);
    gtk_widget_show(mj[i+1]);
  }

  g_free(buffer);

  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi[9]),mj[0]);
  gtk_menu_item_configure(GTK_MENU_ITEM(mi[9]),FALSE,TRUE);

  gtk_accel_group_add(mmag,GDK_N,GDK_CONTROL_MASK,GTK_ACCEL_VISIBLE,
		      GTK_OBJECT(mi[11]),"activate");

  for(i=7;i<14;i++)
    gtk_menu_append(GTK_MENU(o),mi[i]);
  for(i=7;i<14;i++) {
    gtk_widget_lock_accelerators(mi[i]);
    gtk_widget_show(mi[i]);
  }

  gtk_widget_set_sensitive(mi[10],FALSE);
  gtk_widget_set_sensitive(mi[12],FALSE);

  gtk_signal_connect (GTK_OBJECT (mi[7]), "activate",
		      GTK_SIGNAL_FUNC (kill_with_sighup), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[8]), "activate",
		      GTK_SIGNAL_FUNC (kill_with_sigkill), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[11]), "activate",
		      GTK_SIGNAL_FUNC (renice_dialog), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[13]), "activate",
		      GTK_SIGNAL_FUNC (show_details), NULL);

  /* Network */
  net=gtk_menu_new();
  mi[20]=gtk_menu_item_new_with_label("Network watch...");
  mi[21]=gtk_menu_item_new_with_label("Restart connections");
  local_poll_item=mi[22]=
    gtk_check_menu_item_new_with_label("Local polling only");

  gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(mi[22]),TRUE);
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(local_poll_item),
				 poll_local_only);

  for(i=20;i<23;i++)
    gtk_menu_append(GTK_MENU(net),mi[i]);
  for(i=20;i<23;i++) {
    gtk_widget_lock_accelerators(mi[i]);
    gtk_widget_show(mi[i]);
  }

  gtk_signal_connect (GTK_OBJECT (mi[20]), "activate",
		      GTK_SIGNAL_FUNC (netwatch_config), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[21]), "activate",
		      GTK_SIGNAL_FUNC (netwatch_restart), NULL);
  gtk_signal_connect (GTK_OBJECT (mi[22]), "activate",
		      GTK_SIGNAL_FUNC (toggle_local), NULL);

  /* All */
  mainmenu=mb=gtk_menu_bar_new();
  gtk_box_pack_start(GTK_BOX(v1),mb,FALSE,TRUE,0);
  gtk_widget_show(mb);

  mi[0]=gtk_menu_item_new();
  al[0]=gtk_label_new("View");
  gtk_label_set_pattern(GTK_LABEL(al[0]),"_");

  mi[1]=gtk_menu_item_new();
  al[1]=gtk_label_new("Filter");
  gtk_label_set_pattern(GTK_LABEL(al[1]),"_");

  mi[2]=gtk_menu_item_new();
  al[2]=gtk_label_new("Action");
  gtk_label_set_pattern(GTK_LABEL(al[2]),"_");

  mi[3]=gtk_menu_item_new();
  al[3]=gtk_label_new("Network");
  gtk_label_set_pattern(GTK_LABEL(al[3]),"_");

  for(i=0;i<4;i++) {
    gtk_container_add(GTK_CONTAINER(mi[i]),al[i]);
    gtk_widget_show(al[i]);
  }

  gtk_accel_group_add(mmag,GDK_V,GDK_MOD1_MASK,GTK_ACCEL_LOCKED,GTK_OBJECT(mi[0]),"activate_item");
  gtk_accel_group_add(mmag,GDK_F,GDK_MOD1_MASK,GTK_ACCEL_LOCKED,GTK_OBJECT(mi[1]),"activate_item");
  gtk_accel_group_add(mmag,GDK_A,GDK_MOD1_MASK,GTK_ACCEL_LOCKED,GTK_OBJECT(mi[2]),"activate_item");
  gtk_accel_group_add(mmag,GDK_N,GDK_MOD1_MASK,GTK_ACCEL_LOCKED,GTK_OBJECT(mi[3]),"activate_item");

  gtk_accel_group_lock(mmag);
  gtk_window_add_accel_group(GTK_WINDOW(mainw),mmag);


  sc[3]=mi[2];
  gtk_widget_show(mi[0]);
  gtk_widget_show(mi[1]);
  gtk_widget_show(mi[2]);
  gtk_widget_show(mi[3]);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi[0]),m);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi[1]),n);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi[2]),o);
  gtk_menu_item_set_submenu(GTK_MENU_ITEM(mi[3]),net);
  gtk_menu_bar_append(GTK_MENU_BAR(mb),mi[0]);
  gtk_menu_bar_append(GTK_MENU_BAR(mb),mi[1]);
  gtk_menu_bar_append(GTK_MENU_BAR(mb),mi[2]);

#ifdef ENABLE_NETWORK
  gtk_menu_bar_append(GTK_MENU_BAR(mb),mi[3]);
#endif

  /* LIST */

  h5=gtk_hbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(v1),h5,TRUE,TRUE,4);
  sw=gtk_scrolled_window_new(NULL,NULL);
  gtk_box_pack_start(GTK_BOX(h5),sw,TRUE,TRUE,4);
  gtk_widget_show(sw);
  gtk_container_set_border_width(GTK_CONTAINER(sw),0);
  
  pl=gtk_clist_new(PROCESS_LIST_COLUMNS);
  gtk_clist_set_shadow_type(GTK_CLIST(pl),GTK_SHADOW_IN);
  gtk_clist_set_selection_mode(GTK_CLIST(pl),GTK_SELECTION_SINGLE);
  for(i=0;i<PROCESS_LIST_COLUMNS;i++)
    gtk_clist_set_column_title(GTK_CLIST(pl),i,(gchar *)(col_title[i]));
  gtk_clist_column_titles_passive(GTK_CLIST(pl));
  gtk_clist_column_titles_show(GTK_CLIST(pl));

  for(i=5;i<=9;i++)
    gtk_clist_set_column_justification(GTK_CLIST(pl),i,GTK_JUSTIFY_RIGHT);

  for(i=0;i<PROCESS_LIST_COLUMNS;i++)
    gtk_clist_set_column_width(GTK_CLIST(pl),i,startup_width[i]);

  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
				 GTK_POLICY_AUTOMATIC,
				 GTK_POLICY_ALWAYS);
  gtk_container_add(GTK_CONTAINER(sw),pl);

  psl=pl;
  assert_field_visibility();
  gtk_widget_show(pl);

  refb=b1=gtk_button_new_with_label("Refresh");
  kb[0]=b2=gtk_button_new_with_label("Hang Up (SIGHUP)");
  kb[1]=b4=gtk_button_new_with_label("Kill (SIGKILL)");
  closebutton=b3=gtk_button_new_with_label("Close");

  h1=gtk_hbox_new(TRUE,4);
  gtk_box_pack_start(GTK_BOX(v1),h1,FALSE,TRUE,0);

  gtk_box_pack_start(GTK_BOX(h1),b1,FALSE,TRUE,6);
  gtk_box_pack_start(GTK_BOX(h1),b2,FALSE,TRUE,6);
  gtk_box_pack_start(GTK_BOX(h1),b4,FALSE,TRUE,6);
  gtk_box_pack_start(GTK_BOX(h1),b3,FALSE,TRUE,6);

  /* status bar */
  sbhb=gtk_hbox_new(FALSE,0);
  bott=gtk_table_new(1,3,TRUE);

  botf[0]=gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(v1),botf[0],FALSE,TRUE,0);
  gtk_box_pack_start(GTK_BOX(v1),sbhb,FALSE,TRUE,0);
  gtk_box_pack_start(GTK_BOX(sbhb),bott,TRUE,TRUE,0);

  /* Filter: */
  both[0]=gtk_hbox_new(FALSE,16);
  gtk_table_attach_defaults(GTK_TABLE(bott),both[0],2,3,0,1);
  status_label[0]=gtk_label_new("Filter: none");
  gtk_box_pack_end(GTK_BOX(both[0]),status_label[0],FALSE,TRUE,16);

  /* # of processes and progress wheel */
  both[1]=gtk_hbox_new(FALSE,2);
  gtk_table_attach_defaults(GTK_TABLE(bott),both[1],0,2,0,1);

  style=gtk_widget_get_style(mainw);
  {
    GtkWidget *hpix;
    nethalt=gtk_button_new();
    
    phalt=gdk_pixmap_create_from_xpm_d (mainw->window, &mask,
					&style->bg[GTK_STATE_NORMAL],
					(gchar **) halt_xpm);
    hpix=gtk_pixmap_new(phalt,mask);
    gtk_container_add(GTK_CONTAINER(nethalt),
		      hpix);
    gtk_widget_show(nethalt);
    gtk_widget_show(hpix);
    gdk_pixmap_unref(phalt);
    gtk_box_pack_start(GTK_BOX(both[1]),nethalt,FALSE,FALSE,0);
    gtk_widget_set_sensitive(nethalt,FALSE);
    gtk_signal_connect(GTK_OBJECT(nethalt),"clicked",
		       GTK_SIGNAL_FUNC(nethalt_click),NULL);
  }

  wheel=gtk_drawing_area_new();
  gtk_widget_set_events (wheel, GDK_EXPOSURE_MASK);
  gtk_drawing_area_size(GTK_DRAWING_AREA(wheel),64,20);
  gtk_box_pack_start(GTK_BOX(both[1]),wheel,FALSE,TRUE,2);

  status_label[1]=gtk_label_new(" ");
  gtk_box_pack_start(GTK_BOX(both[1]),status_label[1],FALSE,TRUE,2);

  gtk_widget_show(h5);
  gtk_widget_show(b1);
  gtk_widget_show(b2);
  gtk_widget_show(b4);
  gtk_widget_show(b3);
  gtk_widget_show(h1);

  gtk_widget_show(status_label[0]);
  gtk_widget_show(both[0]);
  gtk_widget_show(status_label[1]);
  gtk_widget_show(both[1]);
  gtk_widget_show(wheel);

  gtk_widget_show(botf[0]);
  gtk_widget_show(bott);
  gtk_widget_show(sbhb);

  gtk_widget_show(v1);

  gtk_widget_set_sensitive(kb[0],FALSE);
  gtk_widget_set_sensitive(kb[1],FALSE);

  gtk_widget_set_sensitive(sc[0],FALSE);
  gtk_widget_set_sensitive(sc[1],FALSE);
  gtk_widget_set_sensitive(sc[2],FALSE);
  gtk_widget_set_sensitive(sc[3],FALSE);

  gtk_container_set_border_width(GTK_CONTAINER(h1),6);

  /* set icon */
  xray = gdk_pixmap_create_from_xpm_d (mainw->window, &mask,
				       &style->bg[GTK_STATE_NORMAL],
				       (gchar **) xray_tux_xpm);
  gdk_window_set_icon (mainw->window, NULL, xray, mask);
  gdk_window_set_icon_name(mainw->window,"Processes");

  /* signal plumbing */
  gtk_signal_connect (GTK_OBJECT (mainw), "delete_event",
		      GTK_SIGNAL_FUNC (kill_main), NULL);
  gtk_signal_connect (GTK_OBJECT (mainw), "destroy",
		      GTK_SIGNAL_FUNC (destroy), NULL);
  gtk_signal_connect (GTK_OBJECT (b1), "clicked",
		      GTK_SIGNAL_FUNC (refresh_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (b2), "clicked",
		      GTK_SIGNAL_FUNC (kill_with_sighup), NULL);
  gtk_signal_connect (GTK_OBJECT (b4), "clicked",
		      GTK_SIGNAL_FUNC (kill_with_sigkill), NULL);
  gtk_signal_connect (GTK_OBJECT (b3), "clicked",
		      GTK_SIGNAL_FUNC (destroy), NULL);
  gtk_signal_connect (GTK_OBJECT(pl),"select_row",
		      GTK_SIGNAL_FUNC(selection_callback),NULL);
  gtk_signal_connect (GTK_OBJECT(pl),"unselect_row",
		      GTK_SIGNAL_FUNC(deselection_callback),NULL);
  gtk_signal_connect (GTK_OBJECT(pl),"click_column",
		      GTK_SIGNAL_FUNC(sort_callback),NULL);
  gtk_signal_connect (GTK_OBJECT(pl),"key_press_event",
		      GTK_SIGNAL_FUNC(userkey),NULL);
  gtk_signal_connect (GTK_OBJECT (wheel), "expose_event",
		      (GtkSignalFunc) wheel_expose_event, NULL);
  gtk_signal_connect (GTK_OBJECT (wheel), "configure_event",
		      (GtkSignalFunc) wheel_configure_event, NULL);

  if (continuum) {
    to_tag=gtk_timeout_add(list_interval,refresh_to,NULL);
    gtk_widget_set_sensitive(refb,FALSE);
  }
}

gint kill_main (GtkWidget * widget, GdkEvent * event, gpointer data)
{
  if (netcritical)
    return TRUE;
  return FALSE;
}

void destroy (GtkWidget * widget, gpointer data)
{
  if (to_tag!=-1) { /* kill refresh timer */
    gtk_timeout_remove(to_tag);
    to_tag=-1;
  }
  while(!enter_mutex()) usleep(140); // hope this doesn't lock
  gtk_main_quit();
}

void refresh() {
  gint row2sel=-1,rowcount,do_append;
  int i,nproc,sproc;
  tabledata *aux;
  char buffer[256];
  ProcessListItem *pli;
  GList *pt,*biglist=NULL;
  NetworkListPoller *nlp;

  static gchar *pcoll[PROCESS_LIST_COLUMNS];

  if (netcritical)
    return;

  if (mainlist!=NULL)
    delete mainlist;
  mainlist=NULL;

  /* build lists */

  /* local */
  list_poller->poll();

  biglist=g_list_copy(list_poller->process_list);

  /* remote */
  if (!poll_local_only) {
    for(pt=listwatch;pt!=NULL;pt=g_list_next(pt)) {
      nlp=(NetworkListPoller *)pt->data;
      nlp->poll();
      if (nlp->process_list!=NULL)
	biglist=g_list_concat(biglist,g_list_copy(nlp->process_list));
    }
  }

  gtk_clist_freeze(GTK_CLIST(psl));

  rowcount=0;
  nproc=0; sproc=0; /* number of processes / sieved processes */
  for(pt=biglist;pt!=NULL;pt=g_list_next(pt)) {
    pli=(ProcessListItem *)(pt->data);
    pcoll[0]=pli->pid;
    if (show_long_names)
      pcoll[1]=pli->longname;
    else
      pcoll[1]=pli->name;
    pcoll[2]=pli->owner;
    pcoll[3]=pli->machine;
    pcoll[4]=pli->state;
    pcoll[5]=pli->cpu;
    pcoll[6]=pli->size;
    pcoll[7]=pli->rss;
    pcoll[8]=pli->nice;
    pcoll[9]=pli->priority;
    pcoll[10]=pli->start;
    
    i=atoi(pcoll[0]);

    nproc++;

    do_append=TRUE;
    if (have_filter)
      if (power_cmp(fi_work.field,pcoll[fi_work.field],
		    fi_work.value)!=fi_work.cmp) {
	do_append=FALSE;
	sproc++;
      }

    if (do_append) {
      aux=new tabledata();
      for(int paprika=0;paprika<PROCESS_LIST_COLUMNS;paprika++)
	aux->setv(paprika,pcoll[paprika]);
      if (mainlist==NULL)
	mainlist=aux;
      else
	mainlist->add(aux);
      if ((i==pid_sel)&&(strcmp(host_sel,pcoll[3])==0)) {
	row2sel=rowcount;
      }
      ++rowcount;
    }
  }
  if (row2sel<0) {
    pid_sel=-1;
    host_sel[0]=0;
  }
  pre_sort();
  update_list_widget();
  gtk_clist_thaw(GTK_CLIST(psl));

  sprintf(buffer,"%d processes, %d shown",nproc,nproc-sproc);
  gtk_label_set_text(GTK_LABEL(status_label[1]),buffer);
  g_list_free(biglist);
}

void toggle_long (GtkWidget * widget, gpointer data) {
  if (show_long_names)
    show_long_names=FALSE;
  else
    show_long_names=TRUE;
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(show_long_names_item),
				 show_long_names);
  refresh();
}

void toggle_local (GtkWidget * widget, gpointer data) {
  if (poll_local_only)
    poll_local_only=FALSE;
  else
    poll_local_only=TRUE;
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(local_poll_item),
				 poll_local_only);
  refresh();
}

void toggle_cont (GtkWidget * widget, gpointer data) {
  if (continuum)
    continuum=FALSE;
  else
    continuum=TRUE;
  gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item2),
				 continuum);
  if (continuum) {
    gtk_widget_set_sensitive(refb,FALSE);
    to_tag=gtk_timeout_add(list_interval,refresh_to,NULL);
  } else {
    gtk_timeout_remove(to_tag);
    to_tag=-1;
    gtk_widget_set_sensitive(refb,TRUE);
  }
}

void refresh_cb (GtkWidget * widget, gpointer data) {
  refresh();
}

gint refresh_to (gpointer data) {

  if (!enter_mutex())
    return FALSE;
  refresh();
  exit_mutex();
  return TRUE;
}

void selection_callback(GtkWidget *widget,gint row,gint column,
			GdkEventButton *event,gpointer data) {
  gchar *pp;
  int np;
  //  GTK_CLIST(psl)->focus_row=-1;
  gtk_clist_get_text(GTK_CLIST(psl),row,0,&pp);
  np=atoi(pp);
  pid_sel=np;

  gtk_clist_get_text(GTK_CLIST(psl),row,3,&pp);
  strcpy(host_sel,pp);

  gtk_widget_set_sensitive(kb[0],TRUE);
  gtk_widget_set_sensitive(kb[1],TRUE);
  gtk_widget_set_sensitive(sc[0],TRUE);
  gtk_widget_set_sensitive(sc[1],TRUE);
  gtk_widget_set_sensitive(sc[2],TRUE);
  gtk_widget_set_sensitive(sc[3],TRUE);

  if (event!=NULL)
    if (event->type==GDK_2BUTTON_PRESS)
      open_details(pid_sel,host_sel);
}

void deselection_callback(GtkWidget *widget,gint row,gint column,
			GdkEventButton *event,gpointer data) {
  gchar *pp;
  pid_sel=-1;
  host_sel[0]=0;
  gtk_widget_set_sensitive(kb[0],FALSE);
  gtk_widget_set_sensitive(kb[1],FALSE);
  gtk_widget_set_sensitive(sc[0],FALSE);
  gtk_widget_set_sensitive(sc[1],FALSE);
  gtk_widget_set_sensitive(sc[2],FALSE);
  gtk_widget_set_sensitive(sc[3],FALSE);
  //  GTK_CLIST(psl)->focus_row=-1;
}

void sort_callback(GtkCList *clist,gint column,gpointer data) {
  if (last_sort==column)
    waysort[column]=(++waysort[column])%2;
  last_sort=column;
  pre_sort();
  update_list_widget();
}

gint hyper_cmp(const gpointer a,const gpointer b) {
  tabledata *p1,*p2;
  p1=(tabledata *)a;
  p2=(tabledata *)b;
  return(power_cmp(last_sort,p1->getv(last_sort),p2->getv(last_sort)));
}

/*

 ABOUT DIALOG 

 */

void about_gps(GtkWidget *,gpointer) {
  static GtkWidget *dlg;
  GtkStyle *style;
  GdkPixmap *xray;
  GdkBitmap *mask;
  GtkWidget *xrayp;
  char about[1024];
  int i;

  GtkWidget *v,*h,*h2,*p,*text,*b;

  dlg=gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_transient_for(GTK_WINDOW(dlg),MainWindow);
  gtk_window_set_position(GTK_WINDOW(dlg),GTK_WIN_POS_CENTER);
  gtk_widget_realize(dlg);
  gtk_window_set_policy(GTK_WINDOW(dlg),TRUE,TRUE,TRUE); /* 0.3.6 */
  gtk_window_set_title (GTK_WINDOW (dlg), "About gPS");

  style=gtk_widget_get_style(dlg);
  xray=gdk_pixmap_create_from_xpm_d(dlg->window,&mask,
				    &style->bg[GTK_STATE_NORMAL],
				    (gchar **)xray_tux_xpm);
  xrayp=gtk_pixmap_new(xray,mask);

  v=gtk_vbox_new(FALSE,2);
  gtk_container_add(GTK_CONTAINER(dlg),v);

  h=gtk_hbox_new(FALSE,2);
  gtk_box_pack_start(GTK_BOX(v),h,FALSE,TRUE,2);
  gtk_box_pack_start(GTK_BOX(h),xrayp,FALSE,TRUE,6);

  strcpy(about,"gPS - Graphical Process Statistics,  version ");
  strcat(about,GPS_RELEASE);
  strcat(about,"\nCopyright (C) 1999-2000 Felipe Paulo G. Bergo\nemail: bergo@seul.org\nWeb: http://gps.seul.org\nFTP: ftp://ftp.seul.org/pub/gps\n\n");

  /* compiled pollers */
  strcat(about,"Polling methods compiled:\n");

#ifdef HAVELINUXPROCFS
  strcat(about,"Linux 2.[0-4] native /proc fs poller\n");
#endif

#ifdef HAVEFREEBSD
  strcat(about,"FreeBSD i386 native kernel poller\n");
#endif

#ifdef HAVETOP
  strcat(about,"Unix generic piped top poller\n");
#endif

  strcat(about,"\n");

  strcat(about,"gPS comes with ABSOLUTELY NO WARRANTY;\n");
  strcat(about,"This is free software, and you are welcome to redistribute\n");
  strcat(about,"it under certain conditions. Read the file COPYING,\n");
  strcat(about,"located in ");
  strcat(about,WDOC);
  strcat(about,"/gps-");
  strcat(about,GPS_RELEASE);
  strcat(about," for details.\n");
  text=gtk_label_new(about);
  gtk_box_pack_start(GTK_BOX(h),text,FALSE,TRUE,2);
  gtk_label_set_justify(GTK_LABEL(text),GTK_JUSTIFY_LEFT);

  h2=gtk_hbox_new(TRUE,2);
  gtk_box_pack_start(GTK_BOX(v),h2,FALSE,TRUE,2);

  for(i=0;i<3;i++) {
    p=gtk_label_new(" ");
    gtk_box_pack_start(GTK_BOX(h2),p,FALSE,FALSE,2);
    gtk_widget_show(p);
  }

  b=gtk_button_new_with_label("Dismiss");
  gtk_box_pack_start(GTK_BOX(h2),b,FALSE,TRUE,4);

  gtk_container_set_border_width(GTK_CONTAINER(dlg),6);

  gtk_widget_show(b);
  gtk_widget_show(text);
  gtk_widget_show(xrayp);
  gtk_widget_show(h2);
  gtk_widget_show(h);
  gtk_widget_show(v);
  gtk_widget_show(dlg);
  gtk_signal_connect(GTK_OBJECT(b),"clicked",
		     GTK_SIGNAL_FUNC(dismiss_about),(gpointer)(&dlg));
  gtk_grab_add(dlg);
}

void dismiss_about(GtkWidget *wid,gpointer data) {
  GtkWidget **dlg;
  dlg=(GtkWidget **)(data);
  gtk_grab_remove(*dlg);
  gtk_widget_destroy(*dlg);
}

/* process signaling */

void kill_with_sighup (GtkWidget * widget, gpointer data) {
  deal_with_kill(SIGHUP);
}

void kill_with_sigkill (GtkWidget * widget, gpointer data) {
  deal_with_kill(SIGKILL);
}

void PosixKill(GtkWidget *widget,gpointer data) {
  int r,sig;
  char *emsg;

  if (netcritical)
    return;

  sig=*((int *)data);

  /* can only kill local processes */
  if (pid_sel==-1)
    return;
  if (strcmp(host_sel,this_host)!=0) {
    message_box(MainWindow,
		ERR_NON_LOCAL,
		"Error",
		MSGBOX_OK,
		MSGBOX_ICON_ERROR);
    return;
  }
  r=kill(pid_sel,sig);
  if (r==0) {
    if (!continuum)
      refresh();
    return;
  }
  switch(errno) {
  case EINVAL:
    emsg=ERR_KILL_INS2;
    break;
  case ESRCH:
    emsg=ERR_KILL_NOS2;
    break;
  case EPERM:
    emsg=ERR_KILL_PER2;
    break;
  default:
    emsg=ERR_BAD_MAN_PAGES;
  }
  message_box(MainWindow,
	      emsg,
	      "Cannot send signal",
	      MSGBOX_OK,MSGBOX_ICON_ERROR);
}

void deal_with_kill(gint sig) {
  int r;
  char *emsg;
  if (pid_sel==-1)
    return;
  if (netcritical)
    return;

  if (strcmp(host_sel,this_host)!=0) {
    message_box(MainWindow,
		ERR_NON_LOCAL,
		"Error",
		MSGBOX_OK,
		MSGBOX_ICON_ERROR);
    return;
  }
  r=kill(pid_sel,sig);
  if (r==0) {
    if (!continuum)
      refresh();
    return;
  }
  switch(errno) {
  case EINVAL:
    emsg=ERR_KILL_INSG;
    break;
  case ESRCH:
    emsg=ERR_KILL_NOSP;
    break;
  case EPERM:
    emsg=ERR_KILL_PERM;
    break;
  default:
    emsg=ERR_BAD_MAN_PAGES;
  }
  message_box(MainWindow,
	      emsg,
	      "Unable to kill",
	      MSGBOX_OK,
	      MSGBOX_ICON_ERROR);
}

gint power_cmp(gint column,char *k,char *m) {
  int i,j;
  double c,d;
  long l1,l2;
  time_t t1,t2;
  int values[128];
  char b1[128],b2[128];

  if (k==NULL)
    return -1;
  if (m==NULL)
    return 1;
  switch((ColumnId)column) {
  case CI_NAME: // text cases
  case CI_OWNER:
  case CI_STATE:
  case CI_HOST:
    i=strcmp(k,m);
    if (i>0) i=1;
    if (i<0) i=-1;
    return(i);
  case CI_PID: // simple numbers
  case CI_NICE:
  case CI_PRI:
    i=atoi(k);
    j=atoi(m);
    if (i<j) return -1;
    if (i>j) return 1;
    return 0;
  case CI_CPU: // fractionary
    c=atof(k);
    d=atof(m);
    if (c<d) return -1;
    if (c>d) return 1;
    return 0;
  case CI_SIZE: // memory
  case CI_RSS:
    for(i=0;i<128;i++)
      values[i]=0;
    values['K']=1;
    values['M']=2;
    values['G']=3;
    values['T']=4;
    values['P']=5;
    if (k[strlen(k)-1]!=m[strlen(m)-1]) {
      if (values[k[strlen(k)-1]]>values[m[strlen(m)-1]])
	return 1;
      else
	return -1;
    }
    strcpy(b1,k);
    strcpy(b2,m);
    b1[strlen(b1)-1]=0;
    b2[strlen(b2)-1]=0;
    l1=atol(b1);
    l2=atol(b2);
    if (l1>l2)
      return 1;
    if (l1<l2)
      return -1;
    return 0;
  case CI_START:
    t1=gps_to_time(k);
    t2=gps_to_time(m);
    if (t1<t2)
      return -1;
    if (t1==t2)
      return 0;
    return 1;
  default:
    return 0;
  }
}

void update_list_widget() {
  gint i,j,k,rts;
  static gchar *dp[PROCESS_LIST_COLUMNS];
  tabledata *p;

  gtk_clist_freeze(GTK_CLIST(psl));

  for(i=0;i<PROCESS_LIST_COLUMNS;i++)
    dp[i]=col_title[i];

  deselection_callback(NULL,0,0,NULL,NULL);
  gtk_clist_unselect_all(GTK_CLIST(psl));

  if (mainlist==NULL) {
    gtk_clist_clear(GTK_CLIST(psl));
    gtk_clist_thaw(GTK_CLIST(psl));
    rows_on_list=0;
    return;
  }

  i=mainlist->count();

  if (i<rows_on_list)
    for(j=rows_on_list-i;j>0;j--)
      gtk_clist_remove(GTK_CLIST(psl),0);

  if (i>rows_on_list)
    for(j=i-rows_on_list;j>0;j--)
      gtk_clist_append(GTK_CLIST(psl),dp);

  rows_on_list=i;
  p=mainlist;
  rts=-1;
  for(j=0;j<rows_on_list;j++,p=p->getNext()) {
    if (p->selected())
      rts=j;
    for(i=0;i<PROCESS_LIST_COLUMNS;i++)
      gtk_clist_set_text(GTK_CLIST(psl),j,i,p->getv(i));
  }

  if (rts>=0)
    gtk_clist_select_row(GTK_CLIST(psl),rts,-1);

  gtk_clist_thaw(GTK_CLIST(psl));
}

char * power_of_two_suffix(gint expon) {
  static char sarray[16]={'K',0,'M',0,'G',0,'T',0,'P',0,'<','!','>',0};
  gint i;

  i=expon;
  if (i==0)
    return(&sarray[1]);
  i=((i%7)-1)<<1;
  return(&sarray[i]);
}

void show_details(GtkWidget *a,gpointer b) {
  open_details(pid_sel,host_sel);
}

int which_number(char *s) {
  gint sl,i;
  sl=strlen(s);
  for(i=0;i<sl;i++)
    if ((s[i]<'0')||(s[i]>'9'))
      return -1;
  return(atoi(s));
}

gboolean userkey(GtkWidget *widget, GdkEventKey *gek,gpointer data) {
  if (gek==NULL)
    return FALSE;
  if (gek->keyval==GDK_Return)
    if (pid_sel!=-1)
      open_details(pid_sel,host_sel);
  return FALSE;
}

void pre_sort() {
  GList *gl,*v;
  tabledata *p,*q;
  gint wts;

  gl=NULL;

  for(p=mainlist;p!=NULL;p=p->getNext())
    gl=g_list_prepend(gl,(gpointer)p);

  if (gl==NULL)
    return;

  /* dirty trick to keep equal rows straight */
  if (last_sort>0) {
    wts=last_sort;
    last_sort=0;
    gl=g_list_sort(gl,(GCompareFunc)hyper_cmp);
    last_sort=wts;
  }
  gl=g_list_sort(gl,(GCompareFunc)hyper_cmp);
 
  if (waysort[last_sort])
    gl=g_list_reverse(gl);
  
  q=(tabledata *)gl->data;
  p=q;
  for(v=g_list_next(gl);v!=NULL;v=g_list_next(v)) {
    p->setNext((tabledata *)v->data);

    if (pid_sel!=-1)
      if ((atoi(p->getv(0))==pid_sel)&&
	  (strcmp(p->getv(3),host_sel)==0))
	p->select_item();

    p=p->getNext();
  }
  p->setNext(NULL);
  mainlist=q;
  g_list_free(gl); /* 0.3.7 */
}

/* RC file */

char *get_rc_path() {
  static char buffer[512];
  sprintf(buffer,"%s/%s",getenv("HOME"),GPS_RC);
  return(buffer);
}

void read_rc() {
  FILE *f;
  GList *pt;
  WatchedHost *wh;
  char buffer[512];
  char *p;
  int i,n;
  TString *t;

  list_interval   =1000;
  history_interval=250;

  f=fopen(get_rc_path(),"r");
  if (f==NULL) return;

  t=new TString(1024);
  while(1) {
    if (fgets(buffer,512,f)==NULL) {
      fclose(f);
      delete t;
      return;
    }
    t->set(buffer);
    p=t->token(" \n\t");
    if (p==NULL) continue;
    if (strcmp(p,"HOSTENTRIES")==0) 
      break;
  }

  if (hostlist_fixed!=NULL) {
    for(pt=hostlist_fixed;pt!=NULL;pt=g_list_next(pt))
      delete((WatchedHost *)(pt->data));
    g_list_free(hostlist_fixed);
  }
  hostlist_fixed=NULL;

  n=atoi(t->token(" \n\t"));

  for(i=0;i<n;i++) {
    wh=new WatchedHost(f);
    hostlist_fixed=g_list_append(hostlist_fixed,(gpointer)wh);
  }

  /* visibility */
  fseek(f,0,SEEK_SET);

  while(1) {
    if (fgets(buffer,512,f)==NULL) {
      fclose(f);
      delete t;
      return;
    }
    t->set(buffer);
    p=t->token(T_SNT);
    if (p==NULL) continue;
    if (strcmp(p,"VISIBLEFIELDS")==0) 
      break;
  }

  for(i=0;i<PROCESS_LIST_COLUMNS;i++)
    visible_fields[i]=atoi(t->token(",\n"));

  /* timeouts */
  while(1) {
    if (fgets(buffer,512,f)==NULL) {
      fclose(f);
      delete t;
      return;
    }    
    t->set(buffer);
    p=t->token(T_SNT);
    if (p==NULL) continue;
    if (strcmp(p,"INTERVALS")==0)
      break;
  }
  
  list_interval=atoi(t->token(T_SNT));
  history_interval=atoi(t->token(T_SNT));

  fclose(f);
  delete t;
}

void write_rc() {
  FILE *f;
  GList *pt;
  int i;

  f=fopen(get_rc_path(),"w");
  if (f==NULL) {
    fprintf(stderr,"** gPS: unable to write to %s\n\n",get_rc_path());
    return;
  }

  fprintf(f,"# gPS RC file (last written by gPS %s) #\n",GPS_RELEASE);

  /* hosts */
  fprintf(f,"HOSTENTRIES %d\n",g_list_length(hostlist_fixed));
  for(pt=hostlist_fixed;pt!=NULL;pt=g_list_next(pt))
    ((WatchedHost *)(pt->data))->write(f);

  /* field visibility */
  fprintf(f,"VISIBLEFIELDS ");
  for(i=0;i<PROCESS_LIST_COLUMNS;i++) {
    fprintf(f,"%d",visible_fields[i]);
    if (i<(PROCESS_LIST_COLUMNS-1))
      fprintf(f,",");
  }
  fprintf(f,"\n");
  
  /* intervals */
  fprintf(f,"INTERVALS %d %d\n",list_interval,history_interval);
  fclose(f);
}

// progress wheel control

int lwx=0,lwy=0,lww=16,lwh=16;

gboolean wheel_expose_event(GtkWidget *widget,GdkEventExpose *ee,
			    gpointer data) {

  if (ee!=NULL) {
    lwx=ee->area.x;
    lwy=ee->area.y;
    lww=ee->area.width;
    lwh=ee->area.height;
    if (lww<64) lww=64;
    if (lwh<20) lwh=20;
  }
  gdk_draw_pixmap(widget->window,
		  widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		  wheel_canvas,
		  lwx,lwy,lwx,lwy,lww,lwh);
  return FALSE;
}

gboolean wheel_configure_event(GtkWidget *widget,GdkEventConfigure *ce,
			    gpointer data) {
  GdkPixmap *whole_wheel;
  GdkBitmap *mask;
  GtkStyle *style;
  int x,y;

  if (wheel_canvas!=NULL)
    gdk_pixmap_unref(wheel_canvas);

  style = gtk_widget_get_style(widget);
  wheel_canvas=gdk_pixmap_new(widget->window,x=widget->allocation.width,
			      y=widget->allocation.height,-1);	

  gdk_draw_rectangle(wheel_canvas,style->bg_gc[GTK_STATE_NORMAL],
		     TRUE,0,0,x,y);

  // border
  whole_wheel = gdk_pixmap_create_from_xpm_d (widget->window, &mask,
					      &style->bg[GTK_STATE_NORMAL],
					      (gchar **) spinnerc_xpm);
  gdk_draw_pixmap(wheel_canvas,style->white_gc,whole_wheel,
		  0,0,0,(y-20)/2,64,20);
  gdk_pixmap_unref(whole_wheel);

  if (wheel_state)
    whole_wheel = gdk_pixmap_create_from_xpm_d (widget->window, &mask,
						&style->bg[GTK_STATE_NORMAL],
						(gchar **) spinnera_xpm);
  else
    whole_wheel = gdk_pixmap_create_from_xpm_d (widget->window, &mask,
						&style->bg[GTK_STATE_NORMAL],
						(gchar **) spinnerb_xpm);

  gdk_draw_pixmap(wheel_canvas,style->white_gc,whole_wheel,
		  wheel_state%22,0,1,2+(y-20)/2,62,16);

  gdk_pixmap_unref(whole_wheel);
}

void walk_wheel() {
  if (!netcritical) {
    gtk_widget_set_sensitive(nethalt,TRUE);
    gtk_widget_set_sensitive(mainmenu,FALSE);
    gtk_widget_set_sensitive(closebutton,FALSE);
    gtk_grab_add(GTK_WIDGET(MainWindow));
  }
  netcritical=1;
  wheel_state+=2;
  if (wheel_state>21) wheel_state=1;
  wheel_configure_event(wheel,NULL,NULL);
  wheel_expose_event(wheel,NULL,NULL);
}

void stop_wheel() {
  gtk_widget_set_sensitive(nethalt,FALSE);
  gtk_grab_remove(GTK_WIDGET(MainWindow));
  gtk_widget_set_sensitive(mainmenu,TRUE);
  gtk_widget_set_sensitive(closebutton,TRUE);
  netcritical=0;
  wheel_state=0;
  gtk_widget_queue_resize(wheel);
}

void freeze_timeouts() {
  //printf("freeze %d\n",freeze_level);

  ++freeze_level;
  if ((freeze_level==1)&&(continuum)) {
    gtk_timeout_remove(to_tag);
    to_tag=-1;
  }
  if ((to_history>=0)&&(freeze_level==1)) {
    gtk_timeout_remove(to_history);
    to_history=-2;
  }
}

void unfreeze_timeouts() {
  if (freeze_level==0) return;
  --freeze_level;
  
  if ((continuum)&&(freeze_level==0)) {
    if (to_tag==-1)
      to_tag=gtk_timeout_add(list_interval,refresh_to,NULL);
  }
  if ((to_history==-2)&&(freeze_level==0)) {
    to_history=gtk_timeout_add(history_interval,refresh_history,NULL);
  }

}

/* settings dialog */

GtkWidget *s_dlg,*vf_cb[PROCESS_LIST_COLUMNS],*intv1,*intv2;

void settings(GtkWidget *w,gpointer data) {
  GtkWidget *fr1,*fr2,*tv,*tb,*e1,*e2,*l1,*l2,
            *v1,*hs,*okb,*cab,*h1,*f1,
            *hb[PROCESS_LIST_COLUMNS];
  char z[64];
  int i;

  // dialog
  s_dlg=gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_transient_for(GTK_WINDOW(s_dlg),MainWindow);
  gtk_window_set_position(GTK_WINDOW(s_dlg),GTK_WIN_POS_CENTER);

  // main vbox
  tv=gtk_vbox_new(FALSE,2);
  gtk_container_add(GTK_CONTAINER(s_dlg),tv);


  // visibility frame
  fr1=gtk_frame_new("Visible Fields");
  gtk_frame_set_shadow_type(GTK_FRAME(fr1),GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(tv),fr1,FALSE,TRUE,2);

  v1=gtk_vbox_new(FALSE,2);
  gtk_container_add(GTK_CONTAINER(fr1),v1);
  gtk_window_set_title(GTK_WINDOW(s_dlg),"Settings...");
  gtk_container_set_border_width(GTK_CONTAINER(s_dlg),6);
  gtk_window_set_policy(GTK_WINDOW(s_dlg),TRUE,TRUE,TRUE);
  gtk_widget_set_uposition(s_dlg,200,50);
  gtk_widget_realize(s_dlg);

  for(i=0;i<PROCESS_LIST_COLUMNS;i++) {
    hb[i]=gtk_hbox_new(FALSE,2);
    vf_cb[i]=gtk_check_button_new_with_label(col_desc[i]);
    gtk_box_pack_start(GTK_BOX(hb[i]),vf_cb[i],FALSE,FALSE,2);
    gtk_box_pack_start(GTK_BOX(v1),hb[i],FALSE,TRUE,1);
    gtk_widget_show(vf_cb[i]);
    gtk_widget_show(hb[i]);
  }

  // intervals frame
  fr2=gtk_frame_new("Refresh Intervals");
  gtk_frame_set_shadow_type(GTK_FRAME(fr2),GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(tv),fr2,FALSE,TRUE,2);  

  tb=gtk_table_new(2,2,FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(tb),4);
  gtk_container_add(GTK_CONTAINER(fr2),tb);
  gtk_table_set_col_spacings(GTK_TABLE(tb),4);
  gtk_table_set_row_spacings(GTK_TABLE(tb),3);

  l1=gtk_label_new("List Refresh Interval (msec)");
  l2=gtk_label_new("CPU/Memory Refresh Interval (msec)");

  e1=gtk_entry_new();
  e2=gtk_entry_new();

  gtk_table_attach_defaults(GTK_TABLE(tb),l1,0,1,0,1);
  gtk_table_attach_defaults(GTK_TABLE(tb),l2,0,1,1,2);
  gtk_table_attach_defaults(GTK_TABLE(tb),e1,1,2,0,1);
  gtk_table_attach_defaults(GTK_TABLE(tb),e2,1,2,1,2);

  // buttons on bottom
  hs=gtk_hseparator_new();
  gtk_box_pack_start(GTK_BOX(tv),hs,FALSE,TRUE,3);

  h1=gtk_hbox_new(TRUE,8);

  f1=gtk_label_new(" ");
  gtk_box_pack_start(GTK_BOX(h1),f1,FALSE,TRUE,0);
  gtk_widget_show(f1);
  f1=gtk_label_new(" ");
  gtk_box_pack_start(GTK_BOX(h1),f1,FALSE,TRUE,0);
  gtk_widget_show(f1);

  okb=gtk_button_new_with_label(" Ok ");
  cab=gtk_button_new_with_label(" Cancel ");

  gtk_box_pack_start(GTK_BOX(h1),okb,FALSE,TRUE,0);
  gtk_box_pack_start(GTK_BOX(h1),cab,FALSE,TRUE,0);
  gtk_box_pack_start(GTK_BOX(tv),h1,FALSE,TRUE,2);

  gtk_widget_show(hs);
  gtk_widget_show(cab);
  gtk_widget_show(okb);
  gtk_widget_show(h1);
  gtk_widget_show(v1);

  gtk_widget_show(l1);
  gtk_widget_show(l2);
  gtk_widget_show(e1);
  gtk_widget_show(e2);
  gtk_widget_show(tb);

  gtk_widget_show(fr1);
  gtk_widget_show(fr2);
  gtk_widget_show(tv);
  gtk_widget_show(s_dlg);

  gtk_signal_connect(GTK_OBJECT(okb),"clicked",
		     GTK_SIGNAL_FUNC(sett_ok),NULL);
  gtk_signal_connect(GTK_OBJECT(cab),"clicked",
		     GTK_SIGNAL_FUNC(sett_cancel),NULL);

  for(i=0;i<PROCESS_LIST_COLUMNS;i++)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(vf_cb[i]),
				 ((visible_fields[i])?TRUE:FALSE));

  sprintf(z,"%d",list_interval);
  gtk_entry_set_text(GTK_ENTRY(e1),z);
  sprintf(z,"%d",history_interval);
  gtk_entry_set_text(GTK_ENTRY(e2),z);
  intv1=e1;
  intv2=e2;

  gtk_grab_add(s_dlg);
}

void sett_ok(GtkWidget *widget,gpointer data) {
  int i,v1,v2;

  for(i=0;i<PROCESS_LIST_COLUMNS;i++)
    if (GTK_TOGGLE_BUTTON(vf_cb[i])->active)
      visible_fields[i]=1;
    else
      visible_fields[i]=0;

  v1=atoi(gtk_entry_get_text(GTK_ENTRY(intv1)));
  v2=atoi(gtk_entry_get_text(GTK_ENTRY(intv2)));

  if (v1<200) v1=200;
  if (v2<100) v2=100;

  list_interval=v1;
  history_interval=v2;

  gtk_grab_remove(s_dlg);
  gtk_widget_destroy(s_dlg);

  freeze_timeouts();
  assert_field_visibility();
  unfreeze_timeouts();
  write_rc();
}

void sett_cancel(GtkWidget *widget,gpointer data) {
  gtk_grab_remove(s_dlg);
  gtk_widget_destroy(s_dlg);
}

void assert_field_visibility() {
  int i;
  gtk_clist_freeze(GTK_CLIST(psl));
  for(i=0;i<PROCESS_LIST_COLUMNS;i++)
    gtk_clist_set_column_visibility(GTK_CLIST(psl),i,
				    visible_fields[i]);
  for(i=0;i<PROCESS_LIST_COLUMNS;i++)
    gtk_clist_set_column_width(GTK_CLIST(psl),i,startup_width[i]);
  gtk_clist_thaw(GTK_CLIST(psl));
}

void time_to_gps(time_t v,char *dest) {
  char a[64],b[64];
  strcpy(b,ctime(&v));
  memset(a,0,64);
  memcpy(a,b+11,8);
  strcat(a," ");
  memcpy(a+9,b,11);
  memcpy(a+20,b+20,4);
  strcpy(dest,a);
}

time_t gps_to_time(char *src) {
  char a[64];
  struct tm thetime;
  int i;
  time_t v;

  if (src[2]!=':')
    return 0;

  strcpy(a,src);
  a[2]=0;
  thetime.tm_hour=atoi(a);
  strcpy(a,&src[3]);
  a[2]=0;
  thetime.tm_min=atoi(a);
  strcpy(a,&src[6]);
  a[2]=0;
  thetime.tm_sec=atoi(a);

  strcpy(a,&src[13]);
  a[3]=0;

  for(i=0,thetime.tm_mon=0;i<12;i++)
    if (!strcmp(a,smonth[i])) {
      thetime.tm_mon=i;
      break;
    }

  strcpy(a,&src[17]);
  a[2]=0;
  thetime.tm_mday=atoi(a);
  strcpy(a,&src[20]);
  thetime.tm_year=atoi(a)-1900;
  v=mktime(&thetime);
  return(v);
}

void nethalt_click(GtkWidget *widget,gpointer data) {
  net_halt_request=1;
}

void init_default_colors() {
  c_history_cpu_a  = 0x00ff00;
  c_history_cpu_b  = 0x004000;
  c_history_cpu_c  = 0x006000;
  c_history_grid   = 0x008000;
  c_history_swap_a = 0xff0000;
  c_history_swap_b = 0x800000;
  c_history_swap_c = 0x600000;
  c_blue_a         = 0x6060c0;
  c_blue_b         = 0x404090;
  c_blue_c         = 0x8080ff;
  c_yellow_a       = 0xffff00;
  c_yellow_b       = 0x808000;
  c_yellow_c       = 0x606000;
  c_ubox_cpu_a     = 0x7070e0;
  c_ubox_cpu_b     = 0x202080;
  c_ubox_cpu_c     = 0x8080f0;
  c_ubox_cpu_d     = 0x000060;
  c_ubox_mem_a     = 0xffaf4f;
  c_ubox_mem_b     = 0xff7700;
  c_ubox_mem_c     = 0xffe060;
  c_ubox_mem_d     = 0x806015;
  c_ubox_uline     = 0xb0b0b0;
  c_ubox_link      = 0xffffff;
  c_ubox_border    = 0xe0e0e0;
  c_ubox_label     = 0xffffff;

  pix_mem=hgreen_xpm;
  pix_swp=hred_xpm;
}

void init_low_depth_colors() {
  c_history_cpu_a  = 0xc0c0c0;
  c_history_cpu_b  = 0x808080;
  c_history_cpu_c  = 0x606060;
  c_history_grid   = 0x404040;
  c_history_swap_a = 0x808080;
  c_history_swap_b = 0x404040;
  c_history_swap_c = 0x404040;
  c_blue_a         = 0xc0c0c0;
  c_blue_b         = 0x909090;
  c_blue_c         = 0xffffff;
  c_yellow_a       = 0xffffff;
  c_yellow_b       = 0x808080;
  c_yellow_c       = 0x606060;
  c_ubox_cpu_a     = 0xffffff;
  c_ubox_cpu_b     = 0x808080;
  c_ubox_cpu_c     = 0xffffff;
  c_ubox_cpu_d     = 0x000000;
  c_ubox_mem_a     = 0xc0c0c0;
  c_ubox_mem_b     = 0x404040;
  c_ubox_mem_c     = 0xffffff;
  c_ubox_mem_d     = 0x000000;
  c_ubox_uline     = 0xc0c0c0;
  c_ubox_link      = 0xffffff;
  c_ubox_border    = 0xffffff;
  c_ubox_label     = 0xffffff;

  pix_mem=hgmem_xpm;
  pix_swp=hgswap_xpm;
}
