#!/usr/bin/env python

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import pygtk, gtk, os, re, sys, popen2, pango, time, gobject
import gquilt_utils, gquilt_gtk, gquilt_icons
import gquilt_tool, gquilt_quilt

ifce = gquilt_tool.interface("dummy")

class gquilt_patch_list(gquilt_gtk.text_list):
    def __init__(self):
        gquilt_gtk.text_list.__init__(self)
        self.update_contents()
    def _decorated_line_entry(self, patch, status):
        if status == gquilt_tool.APPLIED:
            return (patch, pango.STYLE_NORMAL, "black", gquilt_icons.applied_ok)
        elif status == gquilt_tool.TOP_PATCH:
            return (patch, pango.STYLE_NORMAL, "black", gquilt_icons.top_ok)
        elif status == gquilt_tool.APPLIED_NEEDS_REFRESH:
            return (patch, pango.STYLE_NORMAL, "black", gquilt_icons.applied_not_ok)
        elif status == gquilt_tool.TOP_PATCH_NEEDS_REFRESH:
            return (patch, pango.STYLE_NORMAL, "black", gquilt_icons.top_not_ok)
        elif status == gquilt_tool.NOT_APPLIED:
            return (patch, pango.STYLE_ITALIC, "dark grey", None)
        else:
            return (patch, pango.STYLE_ITALIC, "magenta", None)
    def update_contents(self):
        res, (sn, series), err = ifce.get_series()
        self.store.clear()
        for patch, status in series:
            self.store.append(self._decorated_line_entry(patch, status))
        col0 = self.view.get_column(0)
        if sn == "":
            col0.set_title("Patch Series")
        else:
            col0.set_title("Patch Series: " + sn)
    def _unique_and_applied(self, arg=None):
        sel = self.view.get_selection()
        sel_sz = sel.count_selected_rows()
        if sel_sz == 1:
            model, iter = self.view.get_selection().get_selected()
            if iter is None:
                return False
            else:
                return model.get_value(iter, 3) != None
        else:
            return False
    def append_applied_menu_item(self, label, tooltip, stock_id, action, arg=None):
        self.append_conditional_menu_item(label, tooltip, stock_id, action, self._unique_and_applied, arg)
    def _unique_and_unapplied(self, arg=None):
        sel = self.view.get_selection()
        sel_sz = sel.count_selected_rows()
        if sel_sz == 1:
            model, iter = self.view.get_selection().get_selected()
            if iter is None:
                return False
            else:
                return model.get_value(iter, 3) is None
        else:
            return False
    def append_unapplied_menu_item(self, label, tooltip, stock_id, action, arg=None):
        self.append_conditional_menu_item(label, tooltip, stock_id, action, self._unique_and_unapplied, arg)

class gquilt_playground_tree(gquilt_gtk.dir_file_tree):
    def __init__(self, title="Playground Files", tooltips=None, console=None, update_views=None):
        gquilt_gtk.dir_file_tree.__init__(self, title, tooltips=tooltips)
        self.console = console
        self._update_views = update_views
    def _wait_for_child_timeout(self, pid):
        rpid, status = os.waitpid(pid, os.WNOHANG)
        return rpid != pid
    def _clean_up_after_child(self, pid):
        gobject.timeout_add(2000, self._wait_for_child_timeout, pid)
    def peruse_selected_files(self):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        pid = gquilt_utils.peruse_files(files)
        self._clean_up_after_child(pid)
    def edit_selected_files(self):
        files = self.get_selected_files()
        if len(files) == 0:
            return
        filelist = []
        dirs = ""
        for f in files:
            if os.path.isdir(f):
                dirs += "\n" + f
            else:
                filelist.append(f)
        if dirs != "":
            emsg = "The directories:\n" + dirs + "\n\nhave been removed from selection\n"
            if not gquilt_gtk.info_ok_dialog(emsg).ask():
                return
        if len(filelist) > 0:
            res, so, se = ifce.get_patch_files(None, False)
            if res != gquilt_tool.OK:
                gquilt_gtk.info_dialog("\n".join([so, se])).inform()
                return
            added_files = []
            for f in filelist:
                if not f in so:
                    added_files.append(f)
            if len(added_files) > 0:
                res, so, se = ifce.do_add_files_to_patch(self.console, added_files, None)
                if res == gquilt_tool.OK:
                    if self._update_views is not None:
                        self._update_views()
                    pid = gquilt_utils.edit_files(filelist)
                    self._clean_up_after_child(pid)
                else:
                    gquilt_gtk.info_dialog("\n".join([so, se])).inform()
    def _handle_double_click(self):
        self.edit_selected_files()

class gquilt_patch_files_tree(gquilt_gtk.path_file_tree):
    def __init__(self, title='Files (in top patch)', patchname=None):
        gquilt_gtk.path_file_tree.__init__(self, title)
        self.patchname = patchname
        self.reread_files()
    def _wait_for_child_timeout(self, pid):
        rpid, status = os.waitpid(pid, os.WNOHANG)
        return rpid != pid
    def _clean_up_after_child(self, pid):
        gobject.timeout_add(2000, self._wait_for_child_timeout, pid)
    def _read_files(self):
        res, lines, se = ifce.get_patch_files(self.patchname)
        if res != gquilt_tool.OK:
            gquilt_gtk.info_dialog("\n".join([lines, se])).inform()
            return
        for line, status in lines:
            if status == gquilt_tool.ADDED:
                fg = "#006600"
            elif status == gquilt_tool.DELETED:
                fg = "#AA0000"
            else:
                fg = "black"
            found, iter = self.find_or_insert_file(line, foreground=fg)
            if not found:
                self.view.expand_to_path(self.store.get_path(iter))
            else:
                self.store.set_value(iter, 3, fg)
    def edit_selected_files(self, widget=None):
        files = self.get_selected_files()
        if len(files) == 0:
            res, files, se = ifce.get_patch_files(None, False)
            if res != gquilt_tool.OK:
                gquilt_gtk.info_dialog(se).inform()
                return
        pid = gquilt_utils.edit_files(files)
        self._clean_up_after_child(pid)
    def _handle_double_click(self):
        self.edit_selected_files()
    def reread_files(self):
        self.store.clear()
        self._read_files()
        self.view.expand_all()
    def update_files(self):
        res, lines, se = ifce.get_patch_files(self.patchname, False)
        if res != gquilt_tool.OK:
            gquilt_gtk.info_dialog("\n".join([lines, se])).inform()
            return
        for f in self.store.get_file_paths():
            try:
                i = lines.index(f)
            except:
                self.delete_file(f)
        self._read_files()

class gquilt_patch_file_dialog(gquilt_gtk.update_dialog):
    def __init__(self, patchname):
        gquilt_gtk.update_dialog.__init__(self, "Patch Files: " + patchname)
        self.file_tree = gquilt_patch_files_tree('Files for "' + patchname + '" patch', patchname)
        self.file_tree.set_size_request(240, 320)
        self.vbox.add(self.file_tree)
    def do_update(self):
        self.file_tree.update_files()

class gquilt_diff_dialog(gquilt_gtk.update_save_dialog):
    def __init__(self, patchname, files=[]):
        if patchname is None:
            self.patch = ifce.top_patch()
        else:
            self.patch = patchname
        title = 'gquilt: "' + self.patch + '" diff' + " ".join(files)
        gquilt_gtk.update_save_dialog.__init__(self, title)
        self.text = gquilt_gtk.scrolled_diff_text(title)
        x, y = self.text.view.buffer_to_window_coords(gtk.TEXT_WINDOW_TEXT, 480, 240)
        self.text.set_size_request(x, y)
        self.vbox.add(self.text)
        self.files = files
        self.do_update()
    def do_update(self):
        res, op, err = ifce.get_diff(self.files, self.patch)
        self.text.set_contents(op)
        if res != 0:
            gquilt_gtk.info_dialog(err).inform()
    def do_write_to_file(self, filename):
        return self.text.save_to_file(filename)

class gquilt_combined_diff_dialog(gquilt_gtk.update_save_dialog):
    def __init__(self, start_patch=None, end_patch=None):
        self.start_patch = start_patch
        self.end_patch = end_patch
        title = 'gquilt: combined diff'
        if start_patch is None:
            if end_patch is None:
                title += ' (all applied patches)'
        else:
            title += ' from patch "' + end_patch + '"'
        if end_patch is not None:
            title += ' to patch "' + end_patch + '"'
        gquilt_gtk.update_save_dialog.__init__(self, title)
        self.text = gquilt_gtk.scrolled_diff_text(title)
        x, y = self.text.view.buffer_to_window_coords(gtk.TEXT_WINDOW_TEXT, 480, 240)
        self.text.set_size_request(x, y)
        self.vbox.add(self.text)
        self.do_update()
    def do_update(self):
        res, op, err = ifce.get_combined_diff(self.start_patch, self.end_patch)
        self.text.set_contents(op)
        if res != 0:
            gquilt_gtk.info_dialog(err).inform()
    def do_write_to_file(self, filename):
        return self.text.save_to_file(filename)

class gquilt_descr_dialog(gtk.Dialog):
    def __init__(self, patchname, console=None):
        gtk.Dialog.__init__(self, "Description: " + patchname, None,
            gtk.DIALOG_DESTROY_WITH_PARENT,
            ("_Update", 1, "_Save", 2, gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
        self.connect("response", self._process_response)
        self.connect("close", self._process_close)
        #self.set_response_sensitive(2, False)
        self.view = gtk.TextView()
        self.text = self.view.get_buffer()
        x, y = self.view.buffer_to_window_coords(gtk.TEXT_WINDOW_TEXT, 480, 240)
        self.view.set_size_request(x, y)
        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.add(self.view)
        sw.show_all()
        self.vbox.add(sw)
        self.view.set_cursor_visible(True)
        self.view.set_editable(True)
        self.patchname = patchname
        self.console = console
        self._do_update()
    def _do_update(self):
        if self.text.get_modified() and not gquilt_gtk.info_ok_dialog("Contents modified.  Really update?").ask():
            return
        res, so, se = ifce.get_patch_description(self.patchname)
        if res != 0:
            gquilt_gtk.info_dialog(se).inform()
            return
        self.text.set_text(so)
        self.text.set_modified(False)
    def _do_save(self):
        text = self.text.get_text(self.text.get_start_iter(), self.text.get_end_iter())
        res, so, se = ifce.do_set_patch_description(self.console, self.patchname, text)
        if res != 0:
            gquilt_gtk.info_dialog(se).inform()
            return
        self.text.set_modified(False)
    def _process_response(self, dialog, response):
        if response == 1:
            self._do_update()
        elif response == 2:
            self._do_save()
        else:
            self.destroy()
    def _process_close(self, dialog):
        if self.text.get_modified() and not gquilt_gtk.info_ok_dialog("Contents modified.  Really close?").ask():
            return
        self.destroy()
    def display(self):
        self.show_all()

class gquilt(gtk.Window):
    def _create_menu_item(self, label, stock_id, action, tooltip=None, arg=None):
        mi = gtk.ImageMenuItem(label)
        if stock_id is not None:
            im = gtk.Image()
            im.set_from_stock(stock_id, gtk.ICON_SIZE_MENU)
            mi.set_image(im)
        mi.show()
        if arg == None:
            mi.connect("activate", action)
        else:
            mi.connect("activate", action, arg)
        if tooltip != None:
            self.tooltips.set_tip(mi, tooltip)
        return mi
    def _create_menu_bar(self):
        mb = gtk.MenuBar()
        pgm = gtk.Menu()
        pgm.append(self._create_menu_item(gtk.STOCK_OPEN, gtk.STOCK_OPEN, self._open_playground, "Select/create a new quilt playground"))
        pgm.append(gtk.SeparatorMenuItem())
        pgm.append(self._create_menu_item(gtk.STOCK_QUIT, gtk.STOCK_QUIT, self._quit, "Quit this program"))
        pgmi = gtk.MenuItem("_Playground")
        pgmi.set_submenu(pgm)
        mb.append(pgmi)
        actm = gtk.Menu()
        actm.append(self._create_menu_item("Pop  All", gquilt_icons.pop, self._pop_all_patches, "Remove all applied patches"))
        actm.append(self._create_menu_item("Push  All", gquilt_icons.push, self._push_all_patches, "Apply all patches in series"))
        actm.append(self._create_menu_item("Combined Diff", gquilt_icons.diff, self._display_combined_diff, "Get a combined diff for all applied patches"))
        actmi = gtk.MenuItem("_Actions")
        actmi.set_submenu(actm)
        mb.append(actmi)
        vctm = gtk.Menu()
        vctm.append(self._create_menu_item("Update  All", None, self._update_views, "Update all views"))
        vctm.append(self._create_menu_item("Update  Playground", None, self._update_playground_view, "Update playground files view"))
        vctm.append(self._create_menu_item("Update  Files", None, self._update_files_view, "Update patch files view"))
        vctm.append(self._create_menu_item("Update  Patches", None, self._update_patches_view, "Update patch list view"))
        vctmi = gtk.MenuItem("_Views")
        vctmi.set_submenu(vctm)
        mb.append(vctmi)
        mb.show_all()
        return mb
    def _tb_button(self, label, tooltip, stock_id, callback):
        im = gtk.Image()
        im.set_from_stock(stock_id, gtk.ICON_SIZE_LARGE_TOOLBAR)
        tbb = gtk.ToolButton(im, label)
        tbb.set_expand(False)
        tbb.connect("clicked", callback)
        if tooltip is not None:
            tbb.set_tooltip(self.tooltips, tooltip)
        return tbb
    def _create_tool_bar(self):
        tb = gtk.Toolbar()
        tb.set_orientation(gtk.ORIENTATION_HORIZONTAL)
        tb.set_style(gtk.TOOLBAR_BOTH)
        tb.insert(self._tb_button("Refresh", "Refresh the top patch", gtk.STOCK_REFRESH, self._refresh_top_patch), -1)
        tb.insert(self._tb_button("Push", "Push the next patch in the series", gquilt_icons.push, self._push_next_patch), -1)
        tb.insert(self._tb_button("Pop", "Pop the top patch", gquilt_icons.pop, self._pop_top_patch), -1)
        tb.insert(self._tb_button("New", "Start a new patch", gtk.STOCK_ADD, self._new_patch), -1)
        tb.insert(self._tb_button("Import", "Import an external patch", gquilt_icons.import_patch, self._import_patch), -1)
        tb.insert(self._tb_button("Fold", "Merge an external patch into the top patch", gquilt_icons.fold, self._fold_patch), -1)
        self.cmd_line = gtk.Entry()
        self.cmd_line.connect("activate", self._exec_typed_cmd)
        tbcl = gtk.ToolItem()
        tbcl.set_tooltip(self.tooltips, "Type in a \"quilt\" command for execution")
        tbcl.set_expand(True)
        tbcl.add(self.cmd_line)
        tb.insert(tbcl, -1)
        return tb
    def _create_console(self):
        self.console = gquilt_gtk.console()
        self.console.set_size_request(720, 160)
        return self.console
    def _create_playground_file_tree(self):
        self.playground_files = gquilt_playground_tree(tooltips=self.tooltips, console=self.console, update_views=self._update_views)
        self.playground_files.set_size_request(240, 320)
        self.playground_files.view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.playground_files.append_selection_menu_item("_Add", "Add a the selected files to the top patch", gtk.STOCK_ADD, self._add_playground_selection_to_patch)
        self.playground_files.append_selection_menu_item("_Edit", "Edit the selected files after adding them to the top patch", gtk.STOCK_EDIT, self._edit_playground_selection_in_top_patch)
        self.playground_files.append_selection_menu_item("_Peruse", "Peruse the selected files", gtk.STOCK_FILE, self._peruse_playground_selection)
        self.playground_files.append_menu_item("_New", "Add a new file to the top patch", gtk.STOCK_NEW, self._add_new_file_to_patch)
        return self.playground_files
    def _create_patch_file_tree(self):
        self.patch_files = gquilt_patch_files_tree()
        self.patch_files.set_tooltips(self.tooltips)
        self.patch_files.set_size_request(240, 320)
        self.patch_files.view.get_selection().set_mode(gtk.SELECTION_MULTIPLE)
        self.patch_files.append_menu_item("_Diff", "Display diff for selected files", gquilt_icons.diff, self._display_diff)
        self.patch_files.append_conditional_menu_item("_Meld", "Display diff for selected files using \"meld\"", gquilt_icons.meld, self._display_diff_with_meld, self._unique_selection_and_meld)
        self.patch_files.append_menu_item("_New", "Add a new file to the top patch", gtk.STOCK_NEW, self._add_new_file_to_patch)
        self.patch_files.append_menu_item("_Edit", "Edit selected files in the top patch", gtk.STOCK_EDIT, self._edit_selected_files)
        self.patch_files.append_selection_menu_item("_Remove", "Remove the selected files from the top patch", gtk.STOCK_REMOVE, self._remove_files_from_patch)
        self.patch_files.append_selection_menu_item("_Delete", "Delete the selected files from the playground", gtk.STOCK_DELETE, self._delete_selected_files)
        return self.patch_files
    def _create_patch_list(self):
        self.patches = gquilt_patch_list()
        self.patches.set_tooltips(self.tooltips)
        self.patches.set_size_request(240, 320)
        self.patches.append_unique_selection_menu_item("D_escription", "View/edit patch description", gtk.STOCK_EDIT, self._edit_patch_description)
        self.patches.append_unique_selection_menu_item("_Files", "Show the files modified by the selected patch", gtk.STOCK_FILE, self._display_patch_files)
        self.patches.append_unique_selection_menu_item("D_iff", "Displa the diff for the selected patch", gquilt_icons.diff, self._display_patch_diff)
        self.patches.append_unique_selection_menu_item("Re_name", "Rename the selected patch", None, self._rename_patch)
        self.patches.append_unique_selection_menu_item("_Delete", "Delete the selected patch from the series", gtk.STOCK_DELETE, self._delete_patch)
        self.patches.append_unique_selection_menu_item("_Refresh", "Refresh the selected patch", gtk.STOCK_REFRESH, self._refresh_selected_patch)
        self.patches.append_applied_menu_item("_Pop to", "Pop to the selected patch", gquilt_icons.pop, self._pop_to_selected_patch)
        self.patches.append_unapplied_menu_item("P_ush to", "Push to the selected patch", gquilt_icons.push, self._push_to_selected_patch)
        return self.patches
    def __init__(self):
        gtk.Window.__init__(self, gtk.WINDOW_TOPLEVEL)
        try:
            self.set_icon_from_file(os.environ['GQUILT_ICON'])
        except:
            pass
        self.last_import = None
        self.connect("destroy", self._quit)
        self.set_title("gquilt: " + os.getcwd())
        self.tooltips = gtk.Tooltips()
        vbox = gtk.VBox()
        self.add(vbox)
        vbox.pack_start(self._create_menu_bar(), False)
        vbox.pack_start(self._create_tool_bar(), False)
        vpane = gtk.VPaned()
        vbox.add(vpane)
        hpane_outer = gtk.HPaned()
        hpane_inner = gtk.HPaned()
        self._create_console()
        hpane_outer.add1(self._create_playground_file_tree())
        hpane_outer.add2(hpane_inner)
        hpane_inner.add1(self._create_patch_file_tree())
        hpane_inner.add2(self._create_patch_list())
        vpane.add1(hpane_outer)
        vpane.add2(self.console)
        self.show_all()
        self.show()
        self._set_playground(os.getcwd())
        self.set_focus(self.cmd_line)
    def _quit(self, widget):
        gtk.main_quit()
    def _add_playground_to_title(self):
        self.set_title("gquilt: " + os.getcwd())
    def _show_busy(self):
        self.window.set_cursor(gtk.gdk.Cursor(gtk.gdk.WATCH))
        while gtk.events_pending():
            gtk.main_iteration()
    def _unshow_busy(self):
        self.window.set_cursor(None)
    def _update_patches_view(self, widget=None):
        self._show_busy()
        self.patches.update_contents()
        self._unshow_busy()
    def _update_files_view(self, widget=None):
        self._show_busy()
        self.patch_files.update_files()
        self._unshow_busy()
    def _update_playground_view(self, widget=None):
        self._show_busy()
        self.playground_files.update()
        self._unshow_busy()
    def _update_views(self, widget=None):
        self._show_busy()
        self.patches.update_contents()
        self.patch_files.update_files()
        self.playground_files.update()
        self._unshow_busy()
    def _add_playground_selection_to_patch(self, mi):
        files = self.playground_files.get_selected_files()
        if len(files) == 0:
            return
        filelist = []
        dirs = ""
        for f in files:
            if os.path.isdir(f):
                dirs += "\n" + f
            else:
                filelist.append(f)
        if dirs != "":
            emsg = "The directories:\n" + dirs + "\n\nhave been removed from addition\n"
            if not gquilt_gtk.info_ok_dialog(emsg).ask():
                return
        if len(filelist) > 0:
            self._show_busy()
            res, so, se = ifce.do_add_files_to_patch(self.console, filelist, None)
            self._unshow_busy()
            if res == gquilt_tool.OK:
                self._update_views()
            else:
                gquilt_gtk.info_dialog("\n".join([so, se])).inform()
    def _edit_playground_selection_in_top_patch(self, mi):
        self.playground_files.edit_selected_files()
        return
    def _peruse_playground_selection(self, mi):
        self.playground_files.peruse_selected_files()
        return
    def _add_new_file_to_patch(self, mi):
        ok, name = gquilt_gtk.read_new_file_name()
        if not ok:
            return
        if os.path.isdir(name):
            gquilt_gtk.info_dialog("Cannot add directories to patch").inform()
            return
        cwd = os.getcwd()
        lcwd = len(cwd)
        if len(name) <= lcwd or cwd != name[0:lcwd] or name[lcwd] != "/":
            gquilt_gtk.info_dialog("File is outside playground").inform()
            return
        lname = name[lcwd + 1:]
        self._show_busy()
        res, so, se = ifce.do_add_files_to_patch(self.console, [lname], None)
        self._unshow_busy()
        if res == gquilt_tool.OK:
            if not os.path.exists(lname):
                self.console.exec_cmd("touch " + lname)
            self._update_views()
        else:
            gquilt_gtk.info_dialog("\n".join([so, se])).inform()
    def _remove_files_from_patch(self, mi, pft=None):
        if pft is None:
            files = self.patch_files.get_selected_files()
            patch = None
        else:
            files = pft.get_selected_files()
            patch = pft.patchname
        if len(files) == 0:
            return
        elif len(files) == 1:
            emsg = files[0] + "\n\nConfirm remove selected file from patch?\n"
        else:
            emsg = "\n".join(files) + "\n\nConfirm remove selected files from patch?\n"
        if not gquilt_gtk.info_ok_dialog(emsg).ask():
            return
        self._show_busy()
        res, so, se = ifce.do_remove_files_from_patch(self.console, files, patch)
        self._unshow_busy()
        if res == gquilt_tool.OK:
            if pft is not None:
                pft.update_files()
            self._update_views()
        else:
            gquilt_gtk.info_dialog("\n".join([so, se])).inform()
    def _edit_selected_files(self, mi):
        self.patch_files.edit_selected_files()
    def _delete_selected_files(self, mi):
        files = self.patch_files.get_selected_files()
        if len(files) == 0:
            return
        elif len(files) == 1:
            emsg = files[0] + "\n\nConfirm delete selected file?\n"
        else:
            emsg = "\n".join(files) + "\n\nConfirm delete selected files?\n"
        if not gquilt_gtk.info_ok_dialog(emsg).ask():
            return
        for file in files:
            try:
                os.remove(file)
            except OSError, (errno, error_message):
                emsg = file + ": " + error_message + "\nContinue?"
                if gquilt_gtk.info_ok_dialog(emsg).ask():
                    continue
        self._refresh_patch()
        self._update_views()
    def _display_diff(self, mi, pft=None):
        if pft == None:
            pft = self.patch_files
        files = pft.get_selected_files()
        dialog = gquilt_diff_dialog(pft.patchname, files)
        dialog.display()
    def _display_combined_diff(self, widget):
        dialog = gquilt_combined_diff_dialog()
        dialog.display()
    def _unique_selection_and_meld(self, pft):
        if gquilt_utils.which("meld") is None:
            return False
        if pft == None:
            pft = self.patch_files
        return pft.view.get_selection().count_selected_rows() == 1
    def _display_diff_with_meld(self, mi, pft=None):
        if pft is not None:
            file = pft.get_selected_files()[0]
            patch = pft.patchname
        else:
            file = self.patch_files.get_selected_files()[0]
            patch = None
        ifce.display_file_diff_in_viewer("meld", file, patch)
    def _edit_patch_description(self, mi):
        patch = self.patches.get_selected_text()
        if patch != None:
            gquilt_descr_dialog(patch, self.console).display()
    def _display_patch_diff(self, mi):
        patch = self.patches.get_selected_text()
        if patch != None:
            gquilt_diff_dialog(patch).display()
    def _display_patch_files(self, mi):
        patch = self.patches.get_selected_text()
        if patch != None:
            dialog = gquilt_patch_file_dialog(patch)
            dialog.file_tree.set_tooltips(self.tooltips)
            dialog.file_tree.append_menu_item("_Diff", "Display diff for selected files", gquilt_icons.diff, self._display_diff, dialog.file_tree)
            dialog.file_tree.append_conditional_menu_item("_Meld", "Display diff for selected files using \"meld\"", gquilt_icons.meld, self._display_diff_with_meld, self._unique_selection_and_meld, dialog.file_tree, dialog.file_tree)
            dialog.file_tree.append_unique_selection_menu_item("_Remove", "Remove the selected files from the patch", gtk.STOCK_REMOVE, self._remove_files_from_patch, dialog.file_tree)
            dialog.display()
    def _delete_patch(self, mi):
        patch = self.patches.get_selected_text()
        if patch != None and gquilt_gtk.info_ok_dialog('Confirm delete "' + patch + '" patch?').ask():
            self._show_busy()
            res, so, se = ifce.do_delete_patch(self.console, patch)
            self._unshow_busy()
            if res != gquilt_tool.OK:
                gquilt_gtk.info_dialog("\n".join([so, se])).inform()
        self._update_views()
    def _refresh_selected_patch(self, mi):
        patch = self.patches.get_selected_text()
        if patch != None:
            self._refresh_patch(patch)
        self._update_views()
    def _rename_patch(self, mi):
        patch = self.patches.get_selected_text()
        if patch != None:
            dialog = gquilt_gtk.text_entry_dialog("Rename patch:")
            dialog.set_text(patch)
            res, newname = dialog.read_text()
            if res:
                self._show_busy()
                res, so, se = ifce.do_rename_patch(self.console, patch, newname)
                self._unshow_busy()
                if res != gquilt_tool.OK:
                    gquilt_gtk.info_dialog("\n".join([so, se])).inform()
            self._update_views()
    def _pop_to_selected_patch(self, mi):
        patch = self.patches.get_selected_text()
        if patch != None:
            self._pop_to_patch(patch)
    def _push_to_selected_patch(self, mi):
        patch = self.patches.get_selected_text()
        if patch != None:
            self._push_to_patch(patch)
    def _set_playground(self, newpg):
        os.chdir(newpg)
        self._show_busy()
        self.playground_files.set_root(newpg)
        self.patches.update_contents()
        self.patch_files.reread_files()
        self._add_playground_to_title()
        self.console.append_dat(": playground: " + newpg + "\n")
        self._unshow_busy()
    def _open_playground(self, widget):
        ok, newpg = gquilt_gtk.read_directory_name("Select playground ..")
        if ok:
            self._set_playground(newpg)
    def _import_patch(self, widget):
        ok, name = gquilt_gtk.read_file_name("Select patch to import ..", self.last_import)
        if ok:
            self.last_import = name
            self._show_busy()
            res, so, se = ifce.do_import_patch(self.console, self.last_import)
            self._unshow_busy()
            if res != gquilt_tool.OK:
                gquilt_gtk.info_dialog("\n".join([so, se])).inform()
            self._update_views()
    def _fold_patch(self, widget):
        ok, name = gquilt_gtk.read_file_name("Select patch to merge ..", self.last_import)
        if ok:
            self.last_import = name
            self._show_busy()
            res, so, se = ifce.do_merge_patch(self.console, self.last_import)
            self._unshow_busy()
            if res != gquilt_tool.OK:
                gquilt_gtk.info_dialog("\n".join([so, se])).inform()
            self._update_views()
    def _new_patch(self, widget):
        res, name = gquilt_gtk.text_entry_dialog("Enter new patch name:").read_text()
        if res:
            self._show_busy()
            res, so, se = ifce.do_create_new_patch(self.console, name)
            self._unshow_busy()
            if res != gquilt_tool.OK:
                gquilt_gtk.info_dialog("\n".join([so, se])).inform()
            self._update_views()
    def _refresh_patch(self, patch=None):
        self._show_busy()
        res, op, err = ifce.do_refresh_patch(self.console, patch)
        self._unshow_busy()
        if res == gquilt_tool.ERROR_FORCE:
            if gquilt_gtk.info_force_dialog("\n".join([op, err])).ask():
                res, op, err = ifce.do_refresh_patch(self.console, patch, force=True)
        if res != gquilt_tool.OK:
            gquilt_gtk.info_dialog("\n".join([op, err])).inform()
    def _refresh_top_patch(self, widget=None):
        self._refresh_patch()
        self._update_views()
    def _pop_patch(self, patch=None):
        while True:
            self._show_busy()
            res, op, err = ifce.do_pop_to_patch(self.console, patch)
            self._unshow_busy()
            if res != gquilt_tool.OK:
                # if there's a problem make sure the views are up to date
                # before asking questions
                self._update_files_view()
                self._update_patches_view()
                if res == gquilt_tool.ERROR_REFRESH:
                    if gquilt_gtk.info_refresh_dialog("\n".join([op, err])).ask():
                        self._refresh_patch()
                        continue
                    else:
                        return False
                elif res == gquilt_tool.ERROR_FORCE_OR_REFRESH:
                    ans = gquilt_gtk.info_force_or_refresh_dialog("\n".join([op, err])).ask()
                    if ans == gquilt_gtk.REFRESH_AND_RETRY:
                        self._refresh_patch()
                        continue
                    elif ans == gquilt_gtk.FORCE:
                        self._show_busy()
                        res, op, err = ifce.do_pop_patch(self.console, force=True)
                        self._unshow_busy()
                    else:
                        return False
                if res != gquilt_tool.OK:
                    # if there's still a problem let the user know
                    gquilt_gtk.info_dialog("\n".join([op, err])).inform()
                    return False
            return True
    def _pop_to_patch(self, patch):
        while ifce.top_patch() != patch:
            if not self._pop_patch(patch):
                break
        self._update_views()
    def _pop_top_patch(self, widget=None):
        self._pop_patch()
        self._update_views()
    def _pop_all_patches(self, widget):
        self._pop_to_patch("")
    def _push_patch(self, patch=None):
        while True:
            self._show_busy()
            res, op, err = ifce.do_push_to_patch(self.console, patch)
            self._unshow_busy()
            if res != gquilt_tool.OK:
                # if there's a problem make sure the views are up to date
                # before asking questions
                self._update_files_view()
                self._update_patches_view()
                if res == gquilt_tool.ERROR_REFRESH:
                    if gquilt_gtk.info_refresh_dialog("\n".join([op, err])).ask():
                        self._refresh_patch()
                        continue
                    else:
                        return False
                elif res == gquilt_tool.ERROR_FORCE:
                    if gquilt_gtk.info_force_dialog("\n".join([op, err])).ask():
                        self._show_busy()
                        res, op, err = ifce.do_push_patch(self.console, force=True)
                        self._unshow_busy()
                    else:
                        return False
                if res != gquilt_tool.OK:
                    # if there's still a problem let the user know
                    gquilt_gtk.info_dialog("\n".join([op, err])).inform()
                    return False
            return True
    def _push_to_patch(self, patch):
        while ifce.top_patch() != patch:
            if not self._push_patch(patch):
                break
        self._update_views()
    def _push_next_patch(self, widget=None):
        self._push_patch()
        self._update_views()
    def _push_all_patches(self, widget):
        self._push_to_patch(ifce.last_patch_in_series())
    def _exec_typed_cmd(self, entry):
        self._show_busy()
        res, so, se =ifce.do_exec_tool_cmd(self.console, entry.get_text())
        self._unshow_busy()
        if res != gquilt_tool.OK:
            gquilt_gtk.info_dialog("\n".join([so, se])).inform()
        self._update_views()
    def _quit(self, widget):
        gtk.main_quit()

if __name__ == "__main__":
    ifce = gquilt_quilt.interface()
    app = gquilt()
    gtk.main()
