# -*- python -*-

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# This file describes the interface that must be implemented for each
# underlying tool in order for gquilt to provide an interface to for the tool

import os, os.path, shutil, gquilt_pfuns, gquilt_const

class interface:
    def __init__(self, name):
        self.name = name
    def new_playground(self, dir=None):
        assert 0, 'new_playground() must be provided in child class!!!'
    def is_playground(self, dir=None):
        assert 0, 'is_playground() must be provided in child class!!!'
    def is_patch_applied(self, patch):
        assert 0, 'is_patch_applied() must be provided in child class!!!'
    def patch_file_name(self, patch):
        assert 0, 'patch_file_name() must be provided in child class!!!'
    def top_patch(self):
        assert 0, 'top_patch() must be provided in child class!!!'
    def last_patch_in_series(self):
        assert 0, 'last_patch_in_series() must be provided in child class!!!'
    def count_ok_meld(selfself, count):
        # override in back end if there are restrictions on its use of meld
        return True
    def display_files_diff_in_viewer(self, viewer, file, patch=None):
        assert 0, 'display_file_diff_in_viewer() must be provided in child class!!!'
    def get_patch_status(self, patch):
        assert 0, 'get_patch_status() must be provided in child class!!!'
    def get_patch_description(self, patch):
        # (almost) back end independent implementation needs to be overridden
        # by a back end if desired
        pfn = self.patch_file_name(patch)
        if os.path.exists(pfn):
            res, lines = gquilt_pfuns.get_patch_descr_lines(pfn)
            if res:
                return (gquilt_const.OK, "\n".join(lines) + "\n", "")
            else:
                return (gquilt_const.ERROR, "", "Error reading patch description\n")
        else:
            return (gquilt_const.OK, "", "")
    def get_series(self):
        assert 0, 'get_series() must be provided in child class!!!'
    def get_diff(self, filelist=[], patch=None):
        assert 0, 'get_diff() must be provided in child class!!!'
    def get_combined_diff(self, start_patch=None, end_patch=None):
        assert 0, 'get_combined_diff() must be provided in child class!!!'
    def get_patch_files(self, patch=None, withstatus=True):
        assert 0, 'get_patch_files() must be provided in child class!!!'
    def do_set_patch_description(self, console, patch, description):
        # (almost) back end independent implementation needs to be overridden
        # by a back end if desired
        pfn = self.patch_file_name(patch)
        res = gquilt_pfuns.set_patch_descr_lines(pfn, description.splitlines())
        if res:
            res = gquilt_const.OK
            se = ""
        else:
            res = gquilt_const.ERROR
            se = "Error reading patch description\n"
        if console is not None:
            console.log_cmd('set description for "' + patch + '"', "\n", se)
        return (res, "", se)
    def do_rename_patch(self, console, patch, newname):
        assert 0, 'do_rename_patch() must be provided in child class!!!'
    def do_pop_patch(self, console, force=False):
        assert 0, 'do_pop_patch() must be provided in child class!!!'
    def do_push_patch(self, console, force=False):
        assert 0, 'do_push_patch() must be provided in child class!!!'
    def do_pop_to_patch(self, console, patch=None):
        assert 0, 'do_pop_to_patch() must be provided in child class!!!'
    def do_push_to_patch(self, console, patch=None):
        assert 0, 'do_push_to_patch() must be provided in child class!!!'
    def do_refresh_patch(self, console, patch=None, force=False):
        assert 0, 'do_refresh_patch() must be provided in child class!!!'
    def do_create_new_patch(self, console, name):
        assert 0, 'do_create_new_patch() must be provided in child class!!!'
    def do_import_patch(self, console, filename, patchname=None, force=False):
        assert 0, 'do_import_patch() must be provided in child class!!!'
    def do_merge_patch(self, console, patch):
        assert 0, 'do_merge_patch() must be provided in child class!!!'
    def do_merge_patch_file(self, console, filename):
        assert 0, 'do_merge_patch() must be provided in child class!!!'
    def do_delete_patch(self, console, patch):
        assert 0, 'do_delete_patch() must be provided in child class!!!'
    def do_add_files_to_patch(self, console, filelist, patch=None):
        assert 0, 'do_add_files_to_patch() must be provided in child class!!!'
    def do_copy_file(self, console, file, dest, force=False):
        # back end independent implementation needs to be overridden by a
        # back end if the back end includes the concept of copy
        if os.path.isdir(dest):
            dest = os.path.join(dest, os.path.basename(file))
        if not force and os.path.exists(dest):
            return (gquilt_const.ERROR_FORCE, "", "File \"%s\" already exists. Select \"force\" to overwrite." % dest)
        res, patch_files, so = self.get_patch_files(None, False)
        if not dest in patch_files:
            res, so, se = self.do_add_files_to_patch(console, [dest], None)
            if res is not gquilt_const.OK:
                return (res, so, se)
        try:
            shutil.copy(file, dest)
            return (gquilt_const.OK, "", "")
        except (IOError, os.error, shutil.Error), why:
            se = "Copy %s to %s failed. %s." % (file, dest, str(why))
            console.append_stderr(se + "\n")
            return (gquilt_const.ERROR, "", se)
    def do_move_file(self, console, file, dest, force=False):
        # back end independent implementation needs to be overridden by a
        # back end if the back end includes the concept of move
        if os.path.isdir(dest):
            dest = os.path.join(dest, os.path.basename(file))
        if not force and os.path.exists(dest):
            return (gquilt_const.ERROR_FORCE, "", "File \"%s\" already exists. Select \"force\" to overwrite." % dest)
        res, patch_files, so = self.get_patch_files(None, False)
        if file not in patch_files:
            res, so, se = self.do_add_files_to_patch(console, [file], None)
            if res is not gquilt_const.OK:
                return (res, so, se)
        if dest not in patch_files:
            res, so, se = self.do_add_files_to_patch(console, [dest], None)
            if res is not gquilt_const.OK:
                return (res, so, se)
        try:
            os.rename(file, dest)
            return (gquilt_const.OK, "", "")
        except (IOError, os.error, shutil.Error), why:
            se = "Copy %s to %s failed. %s." % (file, dest, str(why))
            console.append_stderr(se + "\n")
            return (gquilt_const.ERROR, "", se)
    def do_remove_files_from_patch(self, console, filelist, patch=None):
        assert 0, 'do_remove_files_from_patch() must be provided in child class!!!'
    def do_exec_tool_cmd(self, console, cmd):
        assert 0, 'do_exec_tool_cmd() must be provided in child class!!!'
