//*****************************************************************************
//                                CmdLinePcr.cpp                              *
//                               ----------------                             *
//  Started     : 22/02/2005                                                  *
//  Last Update : 12/10/2009                                                  *
//  Copyright   : (C) 2005 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "CmdLinePcr.hpp"

//*****************************************************************************
// Constructor.

CmdLinePcr::CmdLinePcr( void ) : wxCmdLineParser( )
{
  bClear( );

  m_iArgC   = 0;
  m_ppsArgV = NULL;
}

//*****************************************************************************
// Destructor.

CmdLinePcr::~CmdLinePcr( )
{
}

//*****************************************************************************
// Process command line option -a : analysis page specifier.
//
// Argument List :
//   piArg - A pointer to the current command line argument
//           (incremented as part of processing)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bProcOption_a( int * piArg )
{
  wxString  osArg;

  ( *piArg )++;

  if( *piArg >= m_iArgC )
  {
    std::cout << "Missing analysis type specifier.\n";
    return( FALSE );
  }

  osArg = m_ppsArgV[ *piArg ];
  osArg.MakeUpper( );

  if(      osArg == wxT("OP") ) ;
  else if( osArg == wxT("DC") ) ;
  else if( osArg == wxT("AC") ) ;
  else if( osArg == wxT("TR") ) ;
  else if( osArg == wxT("FO") ) ;
//else if( osArg == wxT("DI") ) ;
//else if( osArg == wxT("NO") ) ;
//else if( osArg == wxT("PZ") ) ;
//else if( osArg == wxT("SE") ) ;
//else if( osArg == wxT("TF") ) ;
  else
  {
    std::cout << "Analysis type invalid or not supported : "
              << wxString( m_ppsArgV[ *piArg ] ).mb_str( ) << '\n';
    return( FALSE );
  }

  m_osAnaType = osArg;

  return( TRUE );
}

//*****************************************************************************
// Process command line option -c : Rebuild/clean the configuration file.
//
// Note : This function must be called twice to actually rebuild/clean the
//        configuration file.
//
// Argument List :
//   piArg - A pointer to the current command line argument
//           (incremented as part of processing)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bProcOption_c( int * piArg )
{
  // Signify that the configuration file needs to be rebuilt/cleaned
  if( ! m_bCleanCfgFile )
  {
    m_bCleanCfgFile = TRUE;
    return( TRUE );
  }


  return( TRUE );
}

//*****************************************************************************
// Process command line option -g : Guile procedure used when importing
//                                  schematic file.
//
// Argument List :
//   piArg - A pointer to the current command line argument
//           (incremented as part of processing)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bProcOption_g( int * piArg )
{
  PrcGNetList  oGNetList;
  wxString     osArg;

  // Check that there are more arguments and that the next is not an option
  osArg = m_ppsArgV[ *piArg ];
  if( *piArg<(m_iArgC-1) && !osArg.StartsWith( wxT("-") ) )
  {
    // Test the Guile procedure name
    if( oGNetList.bSetGuileProc( osArg ) )
      m_osGuileProc = osArg;
    else
    { // The argument isn't a Guile procedure so is it a file name? If it is,
      //   assume it's a schematic file and use the default Guile procedure.
      wxRegEx  oRegEx( wxT("[./]") );
      if( ! oRegEx.Matches( osArg ) )
      {
        std::cout << "Invalid Guile procedure name : " << osArg.mb_str( ) << '\n';
        return( FALSE );
      }
    }

    // Increment argument pointer
    ( *piArg )++;
  }
  else // Use the default value
    m_osGuileProc = wxT("spice-sdb");

  return( TRUE );
}

//*****************************************************************************
// Process command line option -h : display usage and exit.
//
// Argument List :
//   piArg - A pointer to the current command line argument
//           (incremented as part of processing)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bProcOption_h( int * piArg )
{
  Usage( );

  return( FALSE );
}

//*****************************************************************************
// Process command line option -r : specify a configuration file.
//
// Argument List :
//   piArg - A pointer to the current command line argument
//           (incremented as part of processing)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bProcOption_r( int * piArg )
{
  wxFileName  ofn1;

  ( *piArg )++;

  // Check that there are enough arguments
  if( *piArg >= m_iArgC )
  {
    std::cout << "Missing configuration file name.\n";
    return( FALSE );
  }

  // Check the validity of the file name
  ofn1 = wxString( m_ppsArgV[ *piArg ], *wxConvCurrent );
  if( ! ofn1.IsOk( ) )
  {
    std::cout << "Invalid file name : " << m_ppsArgV[ *piArg ] << '\n';
    return( FALSE );
  }
  if( ! ofn1.FileExists( ) )
  {
    std::cout << "File doesn't exist : " << m_ppsArgV[ *piArg ] << '\n';
    return( FALSE );
  }
  if( ofn1.IsRelative( ) ) ofn1.MakeAbsolute( );

  m_osConfigFile = ofn1.GetFullPath( );

  return( TRUE );
}

//*****************************************************************************
// Process command line option -s : simulator engine specifier.
//
// Argument List :
//   piArg - A pointer to the current command line argument
//           (incremented as part of processing)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bProcOption_s( int * piArg )
{
  wxString  osArg;

  ( *piArg )++;

  if( *piArg >= m_iArgC )
  {
    std::cout << "Missing simulator engine specifier.\n";
    return( FALSE );
  }

  osArg = m_ppsArgV[ *piArg ];
  osArg.MakeUpper( );

  if(      wxString( wxT("GNUCAP") ) .StartsWith( osArg ) ) osArg = CLP_GNUCAP;
  else if( wxString( wxT("NGSPICE") ).StartsWith( osArg ) ) osArg = CLP_NGSPICE;
  else
  {
    std::cout << "Invalid simulator engine specifier : "
              << wxString( m_ppsArgV[ *piArg ] ).mb_str( ) << '\n';
    return( FALSE );
  }

  m_osSimrType = osArg;

  return( TRUE );
}

//*****************************************************************************
// Process command line option -v : display app. version and exit.
//
// Argument List :
//   piArg - A pointer to the current command line argument
//           (incremented as part of processing)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bProcOption_v( int * piArg )
{
  wxString  os1;

  os1 << APP_NAME << wxT(", ") << APP_VERSION << wxT('\n');
  std::cout << os1.mb_str( );

  return( FALSE );
}

//*****************************************************************************
// Process command line option -w : waveform viewer specifier.
//
// Argument List :
//   piArg - A pointer to the current command line argument
//           (incremented as part of processing)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bProcOption_w( int * piArg )
{
  wxString  osArg;

  ( *piArg )++;

  if( *piArg >= m_iArgC )
  {
    std::cout << "Missing waveform viewer specifier.\n";
    return( FALSE );
  }

  osArg = m_ppsArgV[ *piArg ];
  osArg.MakeUpper( );

  if(      wxString( wxT("GWAVE") ).StartsWith( osArg ) ) osArg = CLP_GWAVE;
  else if( wxString( wxT("GAW") )  .StartsWith( osArg ) ) osArg = CLP_GAW;
  else
  {
    std::cout << "Invalid waveform viewer specifier : "
              << wxString( m_ppsArgV[ *piArg ] ).mb_str( ) << '\n';
    return( FALSE );
  }

  m_osViewer = osArg;

  return( TRUE );
}

//*****************************************************************************
// Process schematic file name command line argument/s.
//
// Argument List :
//   piArg - A pointer to the current command line argument
//           (incremented as part of processing)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bProcSchems( int * piArg )
{
  wxFileName  ofn1;
  wxString    os1;

  // Check that there are enough arguments
  if( *piArg >= m_iArgC )
  {
    std::cout << "Missing schematic file name/s.\n";
    return( FALSE );
  }

  // Check the validity of the file name/s
  while( *piArg < m_iArgC )
  {
    ofn1 = wxString( m_ppsArgV[ *piArg ], *wxConvCurrent );
    if( ! ofn1.IsOk( ) )
    {
      os1 = m_ppsArgV[ *piArg ];
      std::cout << "Invalid schematic file name : " << os1.mb_str( ) << '\n';
      return( FALSE );
    }
    if( ! ofn1.FileExists( ) )
    {
      os1 = m_ppsArgV[ *piArg ];
      std::cout << "Schematic file doesn't exist : " << os1.mb_str( ) << '\n';
      return( FALSE );
    }
    if( ofn1.IsRelative( ) ) ofn1.MakeAbsolute( );

    m_oasSchemFiles.Add( ofn1.GetFullPath( ) );

    ( *piArg )++;
  }

  return( TRUE );
}

//*****************************************************************************
// Process netlist file name command line argument.
//
// Argument List :
//   piArg - A pointer to the current command line argument
//           (incremented as part of processing)
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bProcNetList( int * piArg )
{
  wxFileName  ofn1;
  wxString    os1;

  // Check that there are enough arguments
  if( *piArg >= m_iArgC )
  {
    std::cout << "Missing netlist file name.\n";
    return( FALSE );
  }

  // Check the validity of the file name
  ofn1 = wxString( m_ppsArgV[ *piArg ], *wxConvCurrent );
  if( ! ofn1.IsOk( ) )
  {
    os1 = m_ppsArgV[ *piArg ];
    std::cout << "Invalid netlist file name : " << os1.mb_str( ) << '\n';
    return( FALSE );
  }
  if( ! ofn1.FileExists( ) )
  {
    os1 = m_ppsArgV[ *piArg ];
    std::cout << "Netlist file doesn't exist : " << os1.mb_str( ) << '\n';
    return( FALSE );
  }
  if( ofn1.IsRelative( ) ) ofn1.MakeAbsolute( );

  ( *piArg )++;

  m_osNetLstFile = ofn1.GetFullPath( );

  return( TRUE );
}

//*****************************************************************************
// Print usage message on console.

void  CmdLinePcr::Usage( void )
{
  wxString  os1, os2;

  os2 = wxString( m_ppsArgV[ 0 ] ).AfterLast( wxT('/') );

  os1 << wxT("\n")
      << wxT("Analyse a electronic circuit using a GUI to a numerical simulation engine\n")
      << wxT("\n")
      << wxT("USAGE   : ") << os2 << wxT(" [-OPTION [ARG]] [FILE/S]\n")
      << wxT("\n")
      << wxT("OPTIONS : -h        : Print usage (this message)\n")
      << wxT("          -v        : Print version information\n")
      << wxT("          -r RCFILE : Specify a configuration file\n")
      << wxT("                      RCFILE = ~/.gspiceui.conf (default)\n")
      << wxT("          -c        : Rebuild/clean the configuration file\n")
      << wxT("          -s SIMENG : Specify the simulation engine to be used\n")
      << wxT("                      SIMENG = gnucap (default) or ngspice\n")
      << wxT("          -a ANA    : Specify the analysis page to be displayed\n")
      << wxT("                      ANA    = op, dc (default), ac, tr, fo, di, no, pz, se or tf\n")
      << wxT("          -g [PROC] : Guile procedure to import a schematic file using gNetList\n")
      << wxT("                      PROC   = spice-sdb (default), pcb, protelii, verilog, etc.\n")
      << wxT("          -w VIEWER : Specify the waveform viewer to be used\n")
      << wxT("                      VIEWER = gwave (default) or gaw\n")
      << wxT("\n")
      << wxT("ARGS    : FILE/S    : Import schematic file/s or load a circuit description file\n")
      << wxT("\n")
      << wxT("NOTES   : Option -s must come before -a if both are specified\n")
      << wxT("\n");

  std::cout << os1.mb_str( );
}

//*****************************************************************************
// Print version information on console.

void  CmdLinePcr::Version( void )
{
  wxString  os1;

  os1 << wxT("\n            ") << APP_NAME
      << wxT("\n  ") << APP_VERSION << wxT("\n\n");

  std::cout << os1.mb_str( );
}

//*****************************************************************************
// Set the configuration file name and update its contents based on the command
// line arguments.

void  CmdLinePcr::SetConfig( void )
{
  wxConfig * poCfg;
  wxString   os1;
  int        i1;

  // Create the global configuration object
  if( ! m_osConfigFile.IsEmpty( ) )
       os1 = rosGetConfigFile( );
  else os1 << wxT('.') << wxString( m_ppsArgV[ 0 ] ).AfterLast( wxT('/') )
           << wxT(".conf");
  poCfg = new wxConfig( wxEmptyString, wxEmptyString, os1 );
  wxConfig::Set( poCfg );
  poCfg->SetRecordDefaults( TRUE );

  // Set the simulation engine type
  if( ! m_osSimrType.IsEmpty( ) )
  {
    poCfg->SetPath( wxT("/Simulator") );
    poCfg->Write( wxT("Engine"), m_osSimrType );
  }

  // Set the analysis type
  if( ! m_osAnaType.IsEmpty( ) )
  {
    poCfg->SetPath( wxT("/Simulator") );
    poCfg->Write( wxT("Analysis"), m_osAnaType );
  }

  // Set the Guile procedure to be used by GNetList
  if( ! m_osGuileProc.IsEmpty( ) )
  {
    poCfg->SetPath( wxT("/gNetList") );
    poCfg->Write( wxT("GuileProc"), m_osGuileProc );
  }

  // Set the waveform viewer type
  if( ! m_osViewer.IsEmpty( ) )
  {
    poCfg->SetPath( wxT("/Viewer") );
    poCfg->Write( wxT("Name"), m_osViewer );
  }

  // Set the netlist file name
  if( ! m_osNetLstFile.IsEmpty( ) )
  {
    poCfg->SetPath( wxT("/Files") );
    poCfg->Write( wxT("NetList"), m_osNetLstFile );
  }

  // Set the schematic file name/s
  if( ! m_oasSchemFiles.IsEmpty( ) )
  {
    poCfg->SetPath( wxT("/Files") );
    for( i1=0, os1.Empty( ); i1<(int)m_oasSchemFiles.GetCount( ); i1++ )
      os1 << m_oasSchemFiles.Item( i1 ) << wxT(' ');
    poCfg->Write( wxT("Schematics"), os1 );
  }

  // Write any changes to file
  poCfg->Flush( );
}

//*****************************************************************************
// Rebuild/clean the configuration file.

void  CmdLinePcr::CleanCfgFile( void )
{
  wxConfig * poCfg;
  wxString   os1;
  long       li1;
  uint       ui1;
  bool       b1;

  // Get a pointer to the configuration object
  poCfg = (wxConfig *) wxConfig::Get( );

  // Record the number of groups
  ui1 = poCfg->GetNumberOfGroups( );

  // Look for any superceded groups
  for( b1=poCfg->GetFirstGroup(os1,li1); b1; b1=poCfg->GetNextGroup(os1,li1) )
  {
    if(      os1 == wxT("Help")        ) continue;
    else if( os1 == wxT("Main")        ) continue;
    else if( os1 == wxT("Simulator")   ) continue;
    else if( os1 == wxT("gNetList")    ) continue;
    else if( os1 == wxT("Files")       ) continue;
    else if( os1 == wxT("Directories") ) continue;
    else if( os1 == wxT("Viewer")      ) continue;
    else if( os1 == wxT("GNU-Cap")     ) continue;

    poCfg->DeleteGroup( os1 );
  }

  // Write any changes to file
  poCfg->Flush( );

  // Display message to the user
  ui1 -= poCfg->GetNumberOfGroups( );
  os1.Empty( );
  os1 << wxT("Rebuild/clean configuration file : ");
  if( ui1 > 0 ) os1 << ui1 << wxT(" unused group/s found & deleted\n");
  else          os1 << wxT("nothing to be done\n");
  std::cout << os1.mb_str( );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bClear( void )
{
  m_osSimrType   .Empty( );
  m_osAnaType    .Empty( );
  m_osGuileProc  .Empty( );
  m_osViewer     .Empty( );
  m_osConfigFile .Empty( );
  m_osNetLstFile .Empty( );
  m_oasSchemFiles.Empty( );
  m_bCleanCfgFile = FALSE;

  return( TRUE );
}

//*****************************************************************************
// Set the command line to be processed.
//
// Argument List :
//   iArgV   - The argment count
//   ppsArgV - The string array of argument fields
//
// Return Values:
//   TRUE  - Success
//   FALSE - Failure

bool  CmdLinePcr::bSetCmdLine( int iArgC, wxChar ** ppsArgV )
{
  // Check validity of arguments
  if( iArgC <= 0 )      return( FALSE );
  if( ppsArgV == NULL ) return( FALSE );

  m_iArgC   = iArgC;
  m_ppsArgV = ppsArgV;

  return( TRUE );
}

//*****************************************************************************
// Process any command line arguments past to the application at start-up.
//
// Return Values :
//   TRUE  - Success (continue  application execution)
//   FALSE - Failure (terminate application execution)

bool  CmdLinePcr::bProcArgs( void )
{
  wxString  osArg, os1;
  int       iArg;

  // Check validity of arguments
  if( m_iArgC <= 0 )                 return( FALSE );
  if( *m_ppsArgV == NULL )           return( FALSE );

  // Process options
  for( iArg=1, osArg=m_ppsArgV[1]; iArg<m_iArgC; ++iArg, osArg=m_ppsArgV[iArg] )
  {
    if( osArg.GetChar( 0 )=='-' && osArg.Length( )>2 )
    { // Concatenated options
      std::cout << "Options must be specfied separately : " << osArg.mb_str( ) << '\n';
      return( FALSE );
    }
    else if( osArg == wxT("-v") )
    { // Print version information
      if( ! bProcOption_v( &iArg ) ) return( FALSE );
    }
    else if( osArg == wxT("-h") )
    { // Print usage information
      if( ! bProcOption_h( &iArg ) ) return( FALSE );
    }
    else if( osArg == wxT("-r") )
    { // Specify a configuration file
      if( ! bProcOption_r( &iArg ) ) return( FALSE );
    }
    else if( osArg == wxT("-c") )
    { // Rebuild/clean the configuration file
      if( ! bProcOption_c( &iArg ) ) return( FALSE );
    }
    else if( osArg == wxT("-s") )
    { // Process simulation engine specifier
      if( ! bProcOption_s( &iArg ) ) return( FALSE );
    }
    else if( osArg == wxT("-a") )
    { // Process analysis type specfier
      if( ! bProcOption_a( &iArg ) ) return( FALSE );
    }
    else if( osArg == wxT("-g") )
    { // Specify Guile procedure for importing a schematic file
      if( ! bProcOption_g( &iArg ) ) return( FALSE );
    }
    else if( osArg == wxT("-w") )
    { // Process waveform viewer specifier
      if( ! bProcOption_w( &iArg ) ) return( FALSE );
    }
    else if( osArg.GetChar( 0 ) == wxT('-') )
    { // Invalid option
      std::cout << "Invalid option : " << osArg.mb_str( ) << '\n';
      return( FALSE );
    }
    else break;
  }

  // Process file names
  if( ! m_osGuileProc.IsEmpty( ) )
  {
    if( ! bProcSchems( &iArg ) )     return( FALSE );
    iArg++;
  }
  else if( iArg < m_iArgC )
  {
    if( ! bProcNetList( &iArg ) )    return( FALSE );
    iArg++;
  }

  // Check that all arguments have been processed
  if( iArg < m_iArgC )
  {
    os1 << wxT("Too many arguments :");
    for( osArg=m_ppsArgV[ iArg ]; iArg<m_iArgC; osArg=m_ppsArgV[ ++iArg ] )
      os1 << wxT(' ') << osArg;
    std::cout << os1.mb_str( ) << '\n';
    return( FALSE );
  }

  // Update the configuration file
  SetConfig( );

  // Rebuild/clean the configuration file if requested
  if( m_bCleanCfgFile )
  {
    CleanCfgFile( );
    return( FALSE );
  }

  // All's well so display the system banner on the console
  Version( );

  return( TRUE );
}

//*****************************************************************************
