//*****************************************************************************
//                                PrcBase.cpp                                 *
//                               -------------                                *
//  Started     : 29/01/2004                                                  *
//  Last Update : 14/07/2009                                                  *
//  Copyright   : (C) 2004 by MSWaters                                        *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "base/PrcBase.hpp"

//*****************************************************************************
// Constructor.

PrcBase::PrcBase( int iFlags ) : wxProcess( iFlags )
{
  m_iPid   = -1;            // Clear the process ID number
  m_ofnLog = DEF_LOG_FILE;  // Set the default log file path and name
  m_osErrMsg.Empty( );      // Clear the error message
}

//*****************************************************************************
// Destructor.

PrcBase::~PrcBase( )
{
  // Delete the process log file
  bDelLogFile( );
}

//*****************************************************************************
// Find a binary using the user's PATH environment variable given the binary
// name.
//
// Argument List :
//   ofnBin - The binary name without the path
//
// Return Values :
//   TRUE  - Success (path prepended to oFnBinary)
//   FALSE - Failure

bool  PrcBase::bFindBinary( wxFileName & ofnBinary )
{
  wxPathList  opl1;
  wxFileName  ofn1;
  wxString    os1, os2;

  m_osErrMsg.Empty( );

  // Initial checks
  if( ! ofnBinary.IsOk( ) ) return( FALSE );

  // Searth the env. var. PATH for the first occurrence of the binary
  opl1.AddEnvList( wxT("PATH") );
  ofn1 = opl1.FindAbsoluteValidPath( ofnBinary.GetFullName( ) );

  // Check whether the binary was successfully found
  if( !ofn1.IsOk( ) || !ofn1.FileExists( ) )
  {
    os1.Empty( );
    os1 << wxT("Can't find the binary :  ") << ofnBinary.GetFullName( )
        << wxT("\n\nYour PATH is :\n") << os1;
    wxGetEnv( wxT("PATH"), &os2 );
    os1 << os2;
    SetErrMsg( os1 );
    return( FALSE );
  }

  m_ofnBinary = ofn1;

  return( TRUE );
}

//*****************************************************************************
// Get the console output after a process has been initiated and record it in
// the log file.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcBase::bLogOutput( void )
{
  wxString  os1, os2;
  size_t    szt1;
  wxChar    oc1;

  // Check that the log file name is valid
  if( ! m_ofnLog.IsOk( ) )          return( FALSE );

  // Open the file
  wxTextFile  oFileLog( m_ofnLog.GetFullPath( ) );
  if( oFileLog.Exists( ) )
       { if( ! oFileLog.Open( )   ) return( FALSE ); }
  else { if( ! oFileLog.Create( ) ) return( FALSE ); }

  // Clear the file if it contains lines
  for( szt1=oFileLog.GetLineCount( ); szt1>0; szt1-- )
    oFileLog.RemoveLine( 0 );

  // Read the console input
  while( bIsExec( ) || GetInputStream( )->IsOk() || GetErrorStream( )->IsOk() )
  {
    // Get a line of data from stdout
    while( IsInputAvailable( ) )
    {
      oc1 = GetInputStream( )->GetC( );
      if( oc1==wxT('\t') || (oc1>31 && oc1!=127) ) os1 << oc1;
      if( oc1 == wxT('\n') ) { oFileLog.AddLine( os1 ); os1 = wxT(""); }
    }

    // Get a line of data from stderr
    while( IsErrorAvailable( ) )
    {
      oc1 = GetErrorStream( )->GetC( );
      if( oc1==wxT('\t') || (oc1>31 && oc1!=127) ) os2 << oc1;
      if( oc1 == wxT('\n') ) { oFileLog.AddLine( os2 ); os2 = wxT(""); }
    }

    wxYield( ); // Yield CPU to other processes
  }

  if( ! os1.IsEmpty( ) ) oFileLog.AddLine( os1 );
  if( ! os2.IsEmpty( ) ) oFileLog.AddLine( os2 );

  oFileLog.Write( ); // Save the changes to disk
  oFileLog.Close( ); // Close the results file

  return( TRUE );
}

//*****************************************************************************
// Does the binary exist?
//
// Return Values :
//   TRUE  - The binary does    exist
//   FALSE - The binary doesn't exist

bool  PrcBase::bBinExists( void )
{
  wxString  os1, os2;

  m_osErrMsg.Empty( );

  if( !m_ofnBinary.FileExists( ) || !m_ofnBinary.IsOk( ) )
  {
    os1.Empty( );
    os1 << wxT("Can't find the binary :  ") << m_ofnBinary.GetFullName( )
        << wxT("\n\nYour PATH is :\n") << os1;
    wxGetEnv( wxT("PATH"), &os2 );
    os1 << os2;
    SetErrMsg( os1 );
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Set the file name of the binary to be executed.
//
// Argument List :
//   rosFileName - A string containing the binary file name
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcBase::bSetBinary( const wxString & rosFileName )
{
  if( rosFileName.IsEmpty( ) )       return( FALSE );

  m_ofnBinary = rosFileName;

  if( ! bFindBinary( m_ofnBinary ) ) return( FALSE );

  return( TRUE );
}

//*****************************************************************************
// Set the argument list to be appended to the binary name.
//
// Argument List :
//   rosArgLst - A string containing the argument list
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcBase::bSetArgLst( const wxString & rosArgLst )
{
  if( rosArgLst.IsEmpty( ) ) return( FALSE );

  m_osArgLst = rosArgLst;

  return( TRUE );
}

//*****************************************************************************
// Set the process log file name.
// (If set all process output can be captured to this file.)
//
// Argument List :
//   rosFileName - A string containing the full path and file name
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcBase::bSetLogFile( const wxString & rosFileName )
{
  wxFileName  ofn1;

  ofn1 = rosFileName;

  if( ! ofn1.IsOk( ) ) return( FALSE );

  if( ofn1.GetPath( ).IsEmpty( ) ) ofn1.SetPath( wxT(".") );

  m_ofnLog = ofn1;

  return( TRUE );
}

//*****************************************************************************
// Execute a process synchronously.
// (Ie. wait for the process to terminate before returning).
//
// ???

//bool  PrcBase::bExecSync( const wxString & rosCmd )
//{
//  long  liRtn;

//  wxEnableTopLevelWindows( FALSE );

//  liRtn = wxExecute( rosCmd, wxEXEC_SYNC );

//  wxEnableTopLevelWindows( TRUE );

//  if( liRtn != 0 ) return( FALSE );

//  return( TRUE );

// This is how I'd like to do it but it doesn't work 20/11/2003 ???
//  wxEnableTopLevelWindows( FALSE );
//  int iRtn = (int) wxExecute( osCmd, wxEXEC_SYNC );
//  wxEnableTopLevelWindows( TRUE );

//  wxProcess  oProcess( wxPROCESS_DEFAULT );
//  iPid = (int) wxExecute( osCmd, wxEXEC_SYNC );

//  if( iPid == 0 )
//  { // Error gnetlist could not be executed
//    cerr << "The command:\n   " << osCmd << "\ncould not be executed.";
//    return( FALSE );
//  }

//  for( ui1=0; ui1<300; ui1++ )
//  { // Wait up to 30 seconds for the process to complete
//cerr << iPid << '\n';
//    wxUsleep( 100 ); // Sleep for 100msec
//    if( ! oProcess.Exists( iPid ) ) break;
//  }
//  if( oProcess.Exists( iPid ) )
//  { // Error gnetlist had to be terminated prematurely
//    oProcess.Kill( iPid );
//    cerr << "The command:\n   " << osCmd << "\ntook more than 30 sec. to "
//         << "execute and so was terminated prematurely.";
//    return( FALSE );
//  }
//}

//*****************************************************************************
// Execute a process asynchronously.
// (Ie. Return as soon as the process has been launched).
//
// Argument List :
//   rosArgList - The argument list to be passed to the binary
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcBase::bExec( void )
{
  wxString  os1, os2;

  // Only execute simulation if it isn't already running
  if( bIsExec( ) )      return( FALSE );

  // Clear error attributes
  m_osErrMsg.Empty( );

  // Check that the binary exists
  if( ! bBinExists( ) ) return( FALSE );

  // Construct the command line to be executed
  os1 = rofnGetBinary( ).GetFullPath( );
  if( ! rosGetArgLst( ).IsEmpty( ) )
    os1 << wxT(' ') << rosGetArgLst( );

  // Attempt to execute the simulation
  m_iPid = (int) wxExecute( os1, wxEXEC_ASYNC, this );
  if( m_iPid <= 0 )
  {
    os2.Empty( );
    os2 << wxT("Couldn't start process : ") << os1;
    SetErrMsg( os2 );
    return( FALSE );
  }

  return( TRUE );
}

//*****************************************************************************
// Stop the simulation currently in progress.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  PrcBase::bKill( void )
{
  if( ! bIsExec( ) )       return( TRUE );
  if( ! Exists( m_iPid ) ) return( TRUE );

  if( Kill( m_iPid, wxSIGTERM ) != wxKILL_OK ) return( FALSE );

  m_iPid = -1;

  return( TRUE );
}

//*****************************************************************************
// Remove the log file.
//
// Return Values :
//   Success - TRUE
//   Failure - FALSE

bool  PrcBase::bDelLogFile( void )
{
  if( m_ofnLog.GetFullPath( ).IsEmpty( ) ) return( TRUE );

  return( ::wxRemoveFile( m_ofnLog.GetFullPath( ) ) );
}

//*****************************************************************************
// Call-back function called when the simulator process terminates.
// (WARNING: Be careful of this function definition, if it is incorrect the
//           application can segment fault.)
//
// Argument List :
//   iPid    - The PID of the process which just terminated
//   iStatus - The exit code of the process

void  PrcBase::OnTerminate( int iPid, int iStatus )
{
  m_iPid = -1;
}

//*****************************************************************************
// Collect the output from the simulator and print to a text control.
//
// Argument List :
//   roTxtCtl - The text control to contain the simulator output

void  PrcBase::Print( TextCtrl & roTxtCtl )
{
  roTxtCtl.bClear( );    // Clear the text control

  PrintCmd( roTxtCtl );  // Print the process command

  PrintRsp( roTxtCtl );  // Print the process command response
}

//*****************************************************************************
// Print the command envoking the process to a text control.
//
// Argument List :
//   roTxtCtl - The text control to contain process input

void  PrcBase::PrintCmd( TextCtrl & roTxtCtl )
{
  wxString  os1;

  // Print the process command
  os1.Empty( );
  os1 << wxT("                    ")
      << wxT("*************** PROCESS COMMAND ***************");
  roTxtCtl.bAppendLine( os1 );
  os1.Empty( );
  roTxtCtl.bAppendLine( os1 );
  os1 << rofnGetBinary( ).GetFullPath( );
  if( ! rosGetArgLst( ).IsEmpty( ) ) os1 << wxT(' ') << rosGetArgLst( );
  roTxtCtl.bAppendLine( os1 );
  os1.Empty( );
  roTxtCtl.bAppendLine( os1 );
}

//*****************************************************************************
// Collect the output from the process and print it to a text control.
//
// Argument List :
//   roTxtCtl - The text control to contain the simulator output

void  PrcBase::PrintRsp( TextCtrl & roTxtCtl )
{
  wxString  os1;

  // Print the process command response
  if( ! m_ofnLog.GetFullPath( ).IsEmpty( ) )
  {
    os1.Empty( );
    os1 << wxT("                    ")
        << wxT("*************** PROCESS RESPONSE **************");
    roTxtCtl.bAppendLine( os1 );
    os1.Empty( );
    roTxtCtl.bAppendLine( os1 );
    if( m_ofnLog.IsOk( ) && m_ofnLog.FileExists( ) )
      roTxtCtl.bAppendFile( m_ofnLog.GetFullPath( ) );
    else
    {
      os1 << wxT("Couldn't load the file containing the process ouput : ")
          << m_ofnLog.GetFullPath( );
      roTxtCtl.bAppendLine( os1 );
    }
  }
}

//*****************************************************************************
