//*****************************************************************************
//                              CmdNgSpiceOPT.cpp                             *
//                             -------------------                            *
// Started     : 23/08/2006                                                   *
// Last Update : 07/10/2009                                                   *
// Copyright   : (C) 2006 by M.S.Waters                                       *
// Email       : M.Waters@bom.gov.au                                          *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "ngspice/commands/CmdNgSpiceOPT.hpp"

//*****************************************************************************
// Constructor.

CmdNgSpiceOPT::CmdNgSpiceOPT( void )
{
  bClear( );
}

//*****************************************************************************
// Destructor.

CmdNgSpiceOPT::~CmdNgSpiceOPT( )
{
}

//*****************************************************************************
// Check that the object attributes are valid.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceOPT::bValidate( void )
{
  double  df1;
  long    li1;

  CmdBase::bValidate( );

  if( ! ConvertType::bStrToDFlt( m_osABSTOL, &df1 ) )
    SetErrMsg( wxT("Invalid value for ABSTOL.") );

  if( ! ConvertType::bStrToDFlt( m_osCHGTOL, &df1 ) )
    SetErrMsg( wxT("Invalid value for CHGTOL.") );

  if( ! ConvertType::bStrToDFlt( m_osDEFAD,  &df1 ) )
    SetErrMsg( wxT("Invalid value for DEFAD.")  );

  if( ! ConvertType::bStrToDFlt( m_osDEFAS,  &df1 ) )
    SetErrMsg( wxT("Invalid value for DEFAS.")  );

  if( ! ConvertType::bStrToDFlt( m_osDEFL,   &df1 ) )
    SetErrMsg( wxT("Invalid value for DEFL.")   );

  if( ! ConvertType::bStrToDFlt( m_osDEFW,   &df1 ) )
    SetErrMsg( wxT("Invalid value for DEFW.")   );

  if( ! ConvertType::bStrToDFlt( m_osGMIN,   &df1 ) )
    SetErrMsg( wxT("Invalid value for GMIN.")   );

  if( ! ConvertType::bStrToDFlt( m_osPIVREL, &df1 ) )
    SetErrMsg( wxT("Invalid value for PIVREL.") );

  if( ! ConvertType::bStrToDFlt( m_osPIVTOL, &df1 ) )
    SetErrMsg( wxT("Invalid value for PIVTOL.") );

  if( ! ConvertType::bStrToDFlt( m_osRELTOL, &df1 ) )
    SetErrMsg( wxT("Invalid value for RELTOL.") );

  if( ! ConvertType::bStrToDFlt( m_osTEMP,   &df1 ) )
    SetErrMsg( wxT("Invalid value for TEMP.")   );

  if( ! ConvertType::bStrToDFlt( m_osTNOM,   &df1 ) )
    SetErrMsg( wxT("Invalid value for TNOM.")   );

  if( ! ConvertType::bStrToDFlt( m_osTRTOL,  &df1 ) )
    SetErrMsg( wxT("Invalid value for TRTOL.")  );

  if( ! ConvertType::bStrToDFlt( m_osVNTOL,  &df1 ) )
    SetErrMsg( wxT("Invalid value for VNTOL.")  );

  if( ! ConvertType::bStrToLong( m_osITL1,   &li1 ) )
    SetErrMsg( wxT("Invalid value for ITL1.") );

  if( ! ConvertType::bStrToLong( m_osITL2,   &li1 ) )
    SetErrMsg( wxT("Invalid value for ITL2.") );

  if( ! ConvertType::bStrToLong( m_osITL4,   &li1 ) )
    SetErrMsg( wxT("Invalid value for ITL4.") );

  if( m_osMETHOD.IsEmpty( ) )
    SetErrMsg( wxT("Invalid value for METHOD.") );

  return( bIsValid( ) );
}

//*****************************************************************************
// Clear the object attributes.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceOPT::bClear( void )
{
  CmdBase::bClear( );

  m_eSimEng  = eSIMR_NGSPICE;
  m_eCmdType = eCMD_OPT;

  ConvertType::bDFltToStr( NGS_ABSTOL, m_osABSTOL );
  ConvertType::bDFltToStr( NGS_CHGTOL, m_osCHGTOL );
  ConvertType::bDFltToStr( NGS_DEFAD , m_osDEFAD  );
  ConvertType::bDFltToStr( NGS_DEFAS , m_osDEFAS  );
  ConvertType::bDFltToStr( NGS_DEFL  , m_osDEFL   );
  ConvertType::bDFltToStr( NGS_DEFW  , m_osDEFW   );
  ConvertType::bDFltToStr( NGS_GMIN  , m_osGMIN   );
  ConvertType::bDFltToStr( NGS_ITL1  , m_osITL1   );
  ConvertType::bDFltToStr( NGS_ITL2  , m_osITL2   );
  ConvertType::bDFltToStr( NGS_ITL4  , m_osITL4   );
  ConvertType::bDFltToStr( NGS_PIVREL, m_osPIVREL );
  ConvertType::bDFltToStr( NGS_PIVTOL, m_osPIVTOL );
  ConvertType::bDFltToStr( NGS_RELTOL, m_osRELTOL );
  ConvertType::bDFltToStr( NGS_TEMP  , m_osTEMP   );
  ConvertType::bDFltToStr( NGS_TNOM  , m_osTNOM   );
  ConvertType::bDFltToStr( NGS_TRTOL , m_osTRTOL  );
  ConvertType::bDFltToStr( NGS_VNTOL , m_osVNTOL  );

  m_bBADMOS3 = NGS_BADMOS3;

  m_osMETHOD = NGS_METHOD;

  m_uiWidth  = NGS_WIDTH;
  m_uiNumDgt = NGS_NUMDGT;
  m_osUnits  = NGS_UNITS;

  return( TRUE );
}

//*****************************************************************************
// Parse the command string.
//
// Eg.s : .OPTIONS NOPAGE
//        .OPTIONS NOPAGE ABSTOL=1.10p BADMOS3 CHGTOL=11.00f DEFAD=10.00f
//                 DEFAS=10.00f DEFL=110.00u DEFW=110.00u GMIN=1.10p ITL1=110
//                 ITL2=60 ITL4=20 METHOD=GEAR PIVREL=2.00m PIVTOL=110.00f
//                 RELTOL=2.00m TEMP=30.00 TNOM=30.00 TRTOL=8.00 VNTOL=2.00u
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceOPT::bParse( void )
{
  wxStringTokenizer  ostk1;
  wxString           os1, os2;
  size_t             sz1;

  // Clear the object attributes
  os1 = (wxString &) *this;
  bClear( );
  assign( os1 );

  // Tokenize the command string
  ostk1.SetString( *this );
  if( ostk1.CountTokens( ) < 2 ) return( bValidate( ) );

  // Check command type
  os1 = ostk1.GetNextToken( ).Left( 4 ).Upper( );
  if( os1 != wxT(".OPT") )       return( bValidate( ) );

  // Extract each parameter value
  while( ostk1.HasMoreTokens( ) )
  {
    // Extract the field name and the associated value
    os1 = ostk1.GetNextToken( );
    os2 = wxT("");
    if( (sz1=os1.find( wxT("=") )) != wxString::npos )
    {
      os2 = os1.Right( os1.Length( )-sz1-1 );
      os1 = os1.Left( sz1 );
    }

    // Set the attribute value
    if(      os1 == wxT("ABSTOL")  ) m_osABSTOL = os2;
    else if( os1 == wxT("BADMOS3") ) m_bBADMOS3 = TRUE;
    else if( os1 == wxT("CHGTOL")  ) m_osCHGTOL = os2;
    else if( os1 == wxT("DEFAD")   ) m_osDEFAD  = os2;
    else if( os1 == wxT("DEFAS")   ) m_osDEFAS  = os2;
    else if( os1 == wxT("DEFL")    ) m_osDEFL   = os2;
    else if( os1 == wxT("DEFW")    ) m_osDEFW   = os2;
    else if( os1 == wxT("GMIN")    ) m_osGMIN   = os2;
    else if( os1 == wxT("ITL1")    ) m_osITL1   = os2;
    else if( os1 == wxT("ITL2")    ) m_osITL2   = os2;
    else if( os1 == wxT("ITL4")    ) m_osITL4   = os2;
    else if( os1 == wxT("METHOD")  ) m_osMETHOD = os2;
    else if( os1 == wxT("NOPAGE")  ) ;
    else if( os1 == wxT("PIVREL")  ) m_osPIVREL = os2;
    else if( os1 == wxT("PIVTOL")  ) m_osPIVTOL = os2;
    else if( os1 == wxT("RELTOL")  ) m_osRELTOL = os2;
    else if( os1 == wxT("TEMP")    ) m_osTEMP   = os2;
    else if( os1 == wxT("TNOM")    ) m_osTNOM   = os2;
    else if( os1 == wxT("TRTOL")   ) m_osTRTOL  = os2;
    else if( os1 == wxT("VNTOL")   ) m_osVNTOL  = os2;
    else                         return( bValidate( ) );
  }

  return( bValidate( ) );
}

//*****************************************************************************
// Format the command string.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceOPT::bFormat( void )
{
  wxString  os1, os2;
  double    df1;
  long      li1;

  os1 = wxT(".OPTIONS NOPAGE");

  os1 << wxT(" NUMDGT=") << m_uiNumDgt;

  os1 << wxT(" UNITS=")  << m_osUnits;

  os1 << wxT(" WIDTH=")  << m_uiWidth;

  ConvertType::bStrToDFlt( m_osABSTOL, &df1 );
  if( df1 != NGS_ABSTOL )           os1 << wxT(" ABSTOL=") << m_osABSTOL;

  if( m_bBADMOS3 != NGS_BADMOS3 )   os1 << wxT(" BADMOS3");

  ConvertType::bStrToDFlt( m_osCHGTOL, &df1 );
  if( df1 != NGS_CHGTOL )           os1 << wxT(" CHGTOL=") << m_osCHGTOL;

  ConvertType::bStrToDFlt( m_osDEFAD , &df1 );
  if( df1 != NGS_DEFAD  )           os1 << wxT(" DEFAD=")  << m_osDEFAD;

  ConvertType::bStrToDFlt( m_osDEFAS , &df1 );
  if( df1 != NGS_DEFAS  )           os1 << wxT(" DEFAS=")  << m_osDEFAS;

  ConvertType::bStrToDFlt( m_osDEFL  , &df1 );
  if( df1 != NGS_DEFL   )           os1 << wxT(" DEFL=")   << m_osDEFL;

  ConvertType::bStrToDFlt( m_osDEFW  , &df1 );
  if( df1 != NGS_DEFW   )           os1 << wxT(" DEFW=")   << m_osDEFW;

  ConvertType::bStrToDFlt( m_osGMIN  , &df1 );
  if( df1 != NGS_GMIN   )           os1 << wxT(" GMIN=")   << m_osGMIN;

  ConvertType::bStrToLong( m_osITL1  , &li1 );
  if( li1 != NGS_ITL1   )           os1 << wxT(" ITL1=")   << m_osITL1;

  ConvertType::bStrToLong( m_osITL2  , &li1 );
  if( li1 != NGS_ITL2   )           os1 << wxT(" ITL2=")   << m_osITL2;

  ConvertType::bStrToLong( m_osITL4  , &li1 );
  if( li1 != NGS_ITL4   )           os1 << wxT(" ITL4=")   << m_osITL4;

  os2 = m_osMETHOD.Upper( );
  if( os2.CmpNoCase( NGS_METHOD ) ) os1 << wxT(" METHOD=") << os2;

  ConvertType::bStrToDFlt( m_osPIVREL, &df1 );
  if( df1 != NGS_PIVREL )           os1 << wxT(" PIVREL=") << m_osPIVREL;

  ConvertType::bStrToDFlt( m_osPIVTOL, &df1 );
  if( df1 != NGS_PIVTOL )           os1 << wxT(" PIVTOL=") << m_osPIVTOL;

  ConvertType::bStrToDFlt( m_osRELTOL, &df1 );
  if( df1 != NGS_RELTOL )           os1 << wxT(" RELTOL=") << m_osRELTOL;

  ConvertType::bStrToDFlt( m_osTEMP  , &df1 );
  if( df1 != NGS_TEMP   )           os1 << wxT(" TEMP=")   << m_osTEMP;

  ConvertType::bStrToDFlt( m_osTNOM  , &df1 );
  if( df1 != NGS_TNOM   )           os1 << wxT(" TNOM=")   << m_osTNOM;

  ConvertType::bStrToDFlt( m_osTRTOL , &df1 );
  if( df1 != NGS_TRTOL  )           os1 << wxT(" TRTOL=")  << m_osTRTOL;

  ConvertType::bStrToDFlt( m_osVNTOL , &df1 );
  if( df1 != NGS_VNTOL  )           os1 << wxT(" VNTOL=")  << m_osVNTOL;

  assign( os1 );

  return( bValidate( ) );
}

//*****************************************************************************
// Calculate the required results page width needed for the number of derived
// parameters.
//
// Note : WIDTH = 56 for 2 cols of numbers ie. one X plus one Y. Add 16 char.
//        positions for each extra column of data required.
//
// 56 + 48 = 104
//
// Argument List :
//   uiParaCnt - The number of derived parameters in the results
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdNgSpiceOPT::bSetWidth( uint uiParaCnt )
{
  if( uiParaCnt < 1 ) return( FALSE );

  m_uiWidth = 56 + 16 * ( uiParaCnt - 1 );

  return( TRUE );
}

//*****************************************************************************
// Copy the contents of a CmdGnuCapAC object.
//
// Argument List :
//   roCmdOPT - A reference to a CmdGnuCapOPT object
//
// Return Values :
//   A reference to this object

CmdNgSpiceOPT & CmdNgSpiceOPT::operator = ( const CmdGnuCapOPT & roCmdOPT )
{
  (CmdBase &) *this = (CmdBase &) roCmdOPT;

  m_osABSTOL = roCmdOPT.m_osABSTOL;
  m_osCHGTOL = roCmdOPT.m_osCHGTOL;
  m_osDEFAD  = roCmdOPT.m_osDEFAD;
  m_osDEFAS  = roCmdOPT.m_osDEFAS;
  m_osDEFL   = roCmdOPT.m_osDEFL;
  m_osDEFW   = roCmdOPT.m_osDEFW;
  m_osGMIN   = roCmdOPT.m_osGMIN;
  m_osITL1   = roCmdOPT.m_osITL1;
  m_osITL2   = roCmdOPT.m_osITL2;
  m_osITL4   = roCmdOPT.m_osITL4;
  m_osMETHOD = roCmdOPT.m_osMETHOD;
  m_osRELTOL = roCmdOPT.m_osRELTOL;
  m_osTNOM   = roCmdOPT.m_osTNOM;
  m_osTRTOL  = roCmdOPT.m_osTRTOL;
  m_osVNTOL  = roCmdOPT.m_osVNTOL;

  m_osTEMP   = roCmdOPT.m_osTEMPAMB;

  bFormat( );

  return( *this );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  CmdNgSpiceOPT::Print( const wxString & rosPrefix )
{
  CmdBase::Print( rosPrefix + wxT("CmdBase::") );

  cout << rosPrefix .mb_str( ) << "m_osABSTOL : "
       << m_osABSTOL.mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_bBADMOS3 : "
       << ( m_bBADMOS3 ? "TRUE" : "FALSE" ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osCHGTOL : "
       << m_osCHGTOL.mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osDEFAD  : "
       << m_osDEFAD .mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osDEFAS  : "
       << m_osDEFAS .mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osDEFL   : "
       << m_osDEFL  .mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osDEFW   : "
       << m_osDEFW  .mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osGMIN   : "
       << m_osGMIN  .mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osITL1   : "
       << m_osITL1  .mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osITL2   : "
       << m_osITL2  .mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osITL4   : "
       << m_osITL4  .mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osMETHOD : "
       << m_osMETHOD.mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osPIVREL : "
       << m_osPIVREL.mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osPIVTOL : "
       << m_osPIVTOL.mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osRELTOL : "
       << m_osRELTOL.mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osTEMP   : "
       << m_osTEMP  .mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osTNOM   : "
       << m_osTNOM  .mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osTRTOL  : "
       << m_osTRTOL .mb_str( ) << '\n';
  cout << rosPrefix .mb_str( ) << "m_osVNTOL  : "
       << m_osVNTOL .mb_str( ) << '\n';
}

//*****************************************************************************
//                                                                            *
//                                 Test Utility                               *
//                                                                            *
//*****************************************************************************

#ifdef TEST_UTIL

// System include files


// Application includes


// Function prototypes

void  Usage( char * psAppName );

//*****************************************************************************

int  main( int argc, char * argv[ ] )
{
  wxString  osCmd;
  wxString  os1;

  // Validate the argument count passed to the application
  if( argc > 2 )           { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }

  // Process the command line arguments
  os1 = wxConvLibc.cMB2WC( argv[ 1 ] );
  if( argc > 1 )
  {
    if( os1 == wxT("-h") ) { Usage( argv[ 0 ] ); exit( EXIT_SUCCESS ); }
    else                   { Usage( argv[ 0 ] ); exit( EXIT_FAILURE ); }
  }

  // Display the utility banner
  cout << "\n  Class CmdNgSpiceOPT Test Utility"
       << "\n     Version 1.02 (04/06/2009)\n";

  // Create a NG-SPICE OPTIONS command object
  CmdNgSpiceOPT  oCmd_OPT;

  // Use the following command example to check the formatter and the parser :
  osCmd << wxT(".OPTIONS NOPAGE ABSTOL=1.10p BADMOS3 CHGTOL=11.00f ")
        << wxT("DEFAD=10.00f DEFAS=10.00f DEFL=110.00u DEFW=110.00u ")
        << wxT("GMIN=1.10p ITL1=110 ITL2=60 ITL4=20 METHOD=GEAR ")
        << wxT("PIVREL=2.00m PIVTOL=110.00f RELTOL=2.00m TEMP=30.00 ")
        << wxT("TNOM=30.00 TRTOL=8.00 VNTOL=2.00u");

  // Set things up for a formatter test
  oCmd_OPT.bClear( );
  oCmd_OPT.m_osABSTOL = wxT("1.10p");
  oCmd_OPT.m_bBADMOS3 = TRUE;
  oCmd_OPT.m_osCHGTOL = wxT("11.00f");
  oCmd_OPT.m_osDEFAD  = wxT("10.00f");
  oCmd_OPT.m_osDEFAS  = wxT("10.00f");
  oCmd_OPT.m_osDEFL   = wxT("110.00u");
  oCmd_OPT.m_osDEFW   = wxT("110.00u");
  oCmd_OPT.m_osGMIN   = wxT("1.10p");
  oCmd_OPT.m_osITL1   = wxT("110");
  oCmd_OPT.m_osITL2   = wxT("60");
  oCmd_OPT.m_osITL4   = wxT("20");
  oCmd_OPT.m_osMETHOD = wxT("GEAR");
  oCmd_OPT.m_osPIVREL = wxT("2.00m");
  oCmd_OPT.m_osPIVTOL = wxT("110.00f");
  oCmd_OPT.m_osRELTOL = wxT("2.00m");
  oCmd_OPT.m_osTEMP   = wxT("30.00");
  oCmd_OPT.m_osTNOM   = wxT("30.00");
  oCmd_OPT.m_osTRTOL  = wxT("8.00");
  oCmd_OPT.m_osVNTOL  = wxT("2.00u");
  cout << "\nRun Formatter     : " << ( oCmd_OPT.bFormat( ) ? "OK" : "FAULT" );
  cout << "\nTest Cmd Format   : " << ( oCmd_OPT == osCmd   ? "OK" : "FAULT" );
  cout << "\nExample Command   : " << osCmd   .mb_str( );
  cout << "\noCmd_OPT Contents : " << oCmd_OPT.mb_str( ) << '\n';

  // Set things up for a parser test
  oCmd_OPT.bClear( );
  oCmd_OPT.bSetString( osCmd );
  cout << "\nRun Parser        : " << ( oCmd_OPT.bParse( ) ? "OK" : "FAULT" );
  oCmd_OPT.bFormat( );
  cout << "\nTest Cmd Format   : " << ( oCmd_OPT == osCmd  ? "OK" : "FAULT" );
  cout << "\nExample Command   : " << osCmd   .mb_str( );
  cout << "\noCmd_OPT Contents : " << oCmd_OPT.mb_str( ) << '\n';

  cout << '\n';

  exit( EXIT_SUCCESS );
}

//*****************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)\n";
}

#endif // TEST_UTIL

//*****************************************************************************
