/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "typedefs.h"
#include "bookmarks.h"
#include "file-utils.h"
#include "preferences.h"


Bookmarks *
bookmarks_new (gchar *rc_filename)
{
	Bookmarks *bookmarks;

	g_return_val_if_fail (rc_filename != NULL, NULL);

	bookmarks = g_new (Bookmarks, 1);
	bookmarks->list = NULL;
	bookmarks->names = g_hash_table_new (g_str_hash, g_str_equal);
	bookmarks->tips = g_hash_table_new (g_str_hash, g_str_equal);
	bookmarks->rc_filename = g_strdup (rc_filename);

	return bookmarks;
}


static gboolean
remove_all (gpointer key, gpointer value, gpointer user_data)
{
	g_free (key);
	g_free (value);

	return TRUE;
}


static void
bookmarks_free_data (Bookmarks *bookmarks)
{
	if (bookmarks->list != NULL) {
		path_list_free (bookmarks->list);
		bookmarks->list = NULL;
	}

	g_hash_table_foreach_remove (bookmarks->names, remove_all, NULL);
	g_hash_table_foreach_remove (bookmarks->tips, remove_all, NULL);
}


void
bookmarks_free (Bookmarks *bookmarks)
{
	g_return_if_fail (bookmarks != NULL);

	bookmarks_free_data (bookmarks);
	g_hash_table_destroy (bookmarks->names);
	g_hash_table_destroy (bookmarks->tips);

	if (bookmarks->rc_filename != NULL)
		g_free (bookmarks->rc_filename);

	g_free (bookmarks);
}


static gchar *
get_menu_item_name (gchar *path)
{
	gchar *name;
	gchar *tmp_path;

	tmp_path = g_strdup (pref_util_remove_prefix (path));

	/* if it is a catalog then remove the extension */
	if (pref_util_location_is_catalog (path)
	    || pref_util_location_is_search (path)) 
		tmp_path[strlen (tmp_path) - strlen (CATALOG_EXT)] = 0;

	if (strcmp (tmp_path, "/") == 0) 
		name = g_strdup ("/");
	else
		name = g_strdup (file_name_from_path (tmp_path));

	g_free (tmp_path);

	return name;
}


static gchar *
get_menu_item_tip (gchar *path)
{	
	gint offset = 0;
	gchar *tmp_path;
	gchar *tip;

	tmp_path = g_strdup (path);

	if (pref_util_location_is_catalog (tmp_path)
	    || pref_util_location_is_search (tmp_path)) {
		gchar *rc_dir_prefix;

		/* if it is a catalog then remove the extension */
		tmp_path[strlen (tmp_path) - strlen (CATALOG_EXT)] = 0;

		rc_dir_prefix = g_strconcat (g_get_home_dir (),
					     "/",
					     RC_CATALOG_DIR,
					     "/",
					     NULL);

		offset = strlen (rc_dir_prefix);

		g_free (rc_dir_prefix);
	}

	tip = g_strdup (pref_util_remove_prefix (tmp_path) + offset);
	g_free (tmp_path);

	return tip;
}


static void 
my_insert (GHashTable *hash_table,
	   gpointer key,
	   gpointer value)
{
	if (g_hash_table_lookup (hash_table, key) != NULL) {
		g_free (value);
		return;
	}

	g_hash_table_insert (hash_table, g_strdup (key), value);
}


static void
my_remove (GHashTable *hash_table,
	   gconstpointer lookup_key)
{
	gpointer orig_key, value;

	if (g_hash_table_lookup_extended (hash_table, lookup_key,
					  &orig_key, &value)) {
		g_hash_table_remove (hash_table, lookup_key);
		g_free (orig_key);
		g_free (value);
	}
}


void
bookmarks_add (Bookmarks *bookmarks,
	       gchar *path)
{
	g_return_if_fail (bookmarks != NULL);
	g_return_if_fail (path != NULL);	

	bookmarks->list = g_list_prepend (bookmarks->list, g_strdup (path));

	my_insert (bookmarks->names, 
		   path, 
		   get_menu_item_name (path));

	my_insert (bookmarks->tips, 
		   path, 
		   get_menu_item_tip (path));
}


void
bookmarks_add_with_prefix (Bookmarks *bookmarks,
			   gchar *path,
			   gchar *prefix)
{
	gchar *full_path;

	g_return_if_fail (bookmarks != NULL);
	g_return_if_fail (path != NULL);	

	full_path = g_strconcat (prefix, path, NULL);
	bookmarks_add (bookmarks, full_path);
	g_free (full_path);
}


void
bookmarks_remove (Bookmarks *bookmarks,
		  gchar *path)
{
	GList *link;

	g_return_if_fail (bookmarks != NULL);
	g_return_if_fail (path != NULL);	

	link = bookmarks->list;
	while (link) {
		if (strcmp ((gchar*)link->data, path) == 0)
			break;
		link = link->next;
	}
	if (link == NULL)
		return;

	bookmarks->list = g_list_remove_link (bookmarks->list, link);

	my_remove (bookmarks->names, path);
	my_remove (bookmarks->tips, path);

	g_free (link->data);
	g_list_free (link);
}


#define MAX_LINE_LENGTH 4096
void
bookmarks_load_from_disk (Bookmarks *bookmarks)
{
	gchar  line [MAX_LINE_LENGTH];
	gchar *path;
	FILE  *f;

	g_return_if_fail (bookmarks != NULL);

	bookmarks_free_data (bookmarks);

	path = g_strconcat (g_get_home_dir (),
			    "/",
			    bookmarks->rc_filename,
			    NULL);

	f = fopen (path, "r");
	g_free (path);

	if (!f)	return;

	while (fgets (line, sizeof (line), f)) {
		gchar *path;

		if (line[0] != '"')
			continue;

		line[strlen (line) - 2] = 0;
		path = line + 1;

		bookmarks->list = g_list_prepend (bookmarks->list, 
						  g_strdup (path));
		my_insert (bookmarks->names, 
			   path, 
			   get_menu_item_name (path));

		my_insert (bookmarks->tips, 
			   path, 
			   get_menu_item_tip (path));
	}
	fclose (f);

	bookmarks->list = g_list_reverse (bookmarks->list);
}
#undef MAX_LINE_LENGTH	


#define MAX_LINES 100
void
bookmarks_write_to_disk (Bookmarks *bookmarks)
{
	FILE *f;
	gchar *path;
	GList *scan;
	gint lines;

	g_return_if_fail (bookmarks != NULL);

	path = g_strconcat (g_get_home_dir (),
			    "/",
			    bookmarks->rc_filename,
			    NULL);

	f = fopen (path, "w+");
	g_free (path);
	
	if (!f)	{
		g_print ("ERROR opening bookmark file\n");
		return;
	}
	
	/* write the file list. */
	lines = 0;
	scan = bookmarks->list; 
	while ((lines < MAX_LINES) && scan) {
		if (! fprintf (f, "\"%s\"\n", (gchar*) scan->data)) {
			g_print ("ERROR saving to bookmark file\n");
			fclose (f);
			
			return;
		}
		lines++;
		scan = scan->next;
	}

	fclose (f);
}
#undef lines


const gchar *
bookmarks_get_menu_name (Bookmarks *bookmarks,
			 const gchar *path)
{
	return g_hash_table_lookup (bookmarks->names, path);
}


const gchar *
bookmarks_get_menu_tip (Bookmarks *bookmarks,
			const gchar *path)
{
	return g_hash_table_lookup (bookmarks->tips, path);
}
