/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  GThumb
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <libgnome/libgnome.h>
#include <libgnomevfs/gnome-vfs-types.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-uri.h>

#include <stdio.h>
#include <sys/stat.h>   
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <string.h>

#include "typedefs.h"
#include "thumb-cache.h"
#include "file-utils.h"


static gboolean
has_png_extension (const gchar *filename)
{
	gint l;

	if (filename == NULL)
		return FALSE;

	l = strlen (filename);
	if (l <= 4)
		return FALSE;

	return (strncmp (filename + l - 4, ".png", 4) == 0);
}


static gchar *
get_aa_name (gchar *filename)
{
	gchar *dot_pos;
	gchar *old_name;

	old_name = filename;
	dot_pos = strrchr (filename, '.');
	if (dot_pos) {
		*dot_pos = '\0';
		filename = g_strdup_printf ("%s.aa.%s", old_name, dot_pos + 1);
	} else 
		filename = g_strconcat (old_name, ".aa", NULL);
	g_free (old_name);

	return filename;
}


static gchar*
get_escaped_dir_name (gchar *dirname)
{
	gchar *prefix;
	gint   prefix_l;
	gchar *e_dirname;
	gchar *e_p;
	gchar *p;
	gint  n_slash;
	
	if (dirname == NULL) return NULL;

	prefix = g_strconcat (g_get_home_dir (),
			      "/.nautilus/thumbnails/",
			      "file:%2F%2F",
			      NULL);
	prefix_l = strlen (prefix);

	n_slash = 0;
	p = dirname;
	while (*p != 0) 
		if (*p++ == '/') n_slash++;

	e_dirname = g_malloc (prefix_l + (p - dirname) + (2 * n_slash) + 1);
	strncpy (e_dirname, prefix, prefix_l);

	p = dirname;
	e_p = e_dirname + prefix_l;
	while (*p != 0) {
		if (*p == '/') {
			*e_p++ = '%';
			*e_p++ = '2';
			*e_p++ = 'F';
		} else 
			*e_p++ = *p;
		p++;
	}
	*e_p = 0;

	g_free (prefix);

	return e_dirname;
}


gchar *
cache_get_nautilus_thumbnail_file (const gchar *source) 
{
	gint try;

	if (! source) return NULL;

	for (try = 0; try < 4; try++) {
		gchar *path;
		gchar *directory = NULL;
		gchar *filename = NULL;
		gchar *filedir = NULL;

		switch (try) {
		case 0: /* not aa image in .nautilus/thumbnails dir. */
		case 1: /* aa image in .nautilus/thumbnails dir. */
			filename = g_strdup (file_name_from_path (source));
			if (try == 1)
				filename = get_aa_name (filename);
			filedir = remove_level_from_path (source);
			directory = get_escaped_dir_name (filedir);
			g_free (filedir);
			break;

		case 2: /* not aa image in .thumbnails dir. */
		case 3: /* aa image in .thumbnails dir. */
			filename = g_strdup (file_name_from_path (source));
			if (try == 3)
				filename = get_aa_name (filename);
			filedir = remove_level_from_path (source);
			directory = g_strconcat (filedir,
						 "/", 
						 CACHE_DIR, 
						 NULL);
			g_free (filedir);
			break;
		}

		path = g_strconcat (directory,
				    "/",
				    filename,
				    has_png_extension (filename)? NULL: ".png",
				    NULL);

		g_free (directory);
		g_free (filename);

		if (path_is_file (path))  /* found. */
			return path;

		g_free (path);
	}

	return NULL;
}


gchar *
cache_get_gthumb_cache_name (const gchar *source) 
{
	gchar *path;
	gchar *directory;
	const gchar *filename;

	if (!source) return NULL;

	directory = remove_level_from_path (source);
	filename = file_name_from_path (source);

	path = g_strconcat (g_get_home_dir(), 
			    "/", 
			    RC_THUMBS_DIR, 
			    directory,
			    "/",
			    filename, 
			    CACHE_THUMB_EXT, 
			    NULL);

	g_free (directory);

	return path;
}


gchar *
cache_get_nautilus_cache_name (const gchar *source) 
{
	char *dir;
	char *escaped_dir;
	char *nautilus_name;

	if (! source) return NULL;

	/* get the Nautilus cache file name. */

	dir = remove_level_from_path (source);
	escaped_dir = get_escaped_dir_name (dir);
	nautilus_name = g_strconcat (escaped_dir,
				     "/",
				     file_name_from_path (source),
				     has_png_extension (source) ? NULL : ".png",
				     NULL);

	g_free (dir);
	g_free (escaped_dir);

	return nautilus_name;
}


void
cache_copy (const gchar *src,
	    const gchar *dest)
{
	gchar *cache_src;
	gchar *cache_dest;

	cache_src = cache_get_gthumb_cache_name (src);
	if (path_is_file (cache_src)) {
		cache_dest = cache_get_gthumb_cache_name (dest);
		file_copy (cache_src, cache_dest);
		g_free (cache_dest);
	}
	g_free (cache_src);

	cache_src = cache_get_nautilus_cache_name (src);
	if (path_is_file (cache_src)) {
		cache_dest = cache_get_nautilus_cache_name (dest);
		file_copy (cache_src, cache_dest);
		g_free (cache_dest);
	}
	g_free (cache_src);
}


void
cache_move (const gchar *src,
	    const gchar *dest)
{
	gchar *cache_src;
	gchar *cache_dest;

	cache_src = cache_get_gthumb_cache_name (src);
	if (path_is_file (cache_src)) {
		cache_dest = cache_get_gthumb_cache_name (dest);
		file_move (cache_src, cache_dest);
		g_free (cache_dest);
	}
	g_free (cache_src);

	cache_src = cache_get_nautilus_cache_name (src);
	if (path_is_file (cache_src)) {
		cache_dest = cache_get_nautilus_cache_name (dest);
		file_move (cache_src, cache_dest);
		g_free (cache_dest);
	}
	g_free (cache_src);
}


void
cache_delete (const gchar *filename)
{
	gchar *cache_name;

	cache_name = cache_get_gthumb_cache_name (filename);
	unlink (cache_name);
	g_free (cache_name);

	cache_name = cache_get_nautilus_cache_name (filename);
	unlink (cache_name);
	g_free (cache_name);
}


void
cache_remove_old_previews (const gchar *dir,
			   gboolean recursive,
			   gboolean clear_all)
{
	visit_rc_directory (RC_THUMBS_DIR,
			    CACHE_THUMB_EXT,
			    dir,
			    recursive,
			    clear_all);
}


/* ----- cache_remove_old_previews_async implememtation. ------ */


typedef struct {
	gboolean recursive;
	gboolean clear_all;
} CacheRemoveData;


static void nautilus_cache_remove_old_previews_async (gboolean recursive,
						      gboolean clear_all);


static void
cache_remove_done (const GList *dir_list,
		   gpointer data)
{
	CacheRemoveData *crd = data;

	if (crd->clear_all) {
		const GList *scan;
		for (scan = dir_list; scan; scan = scan->next) {
			gchar *dir = scan->data;
			rmdir (dir);
		}
	}

	nautilus_cache_remove_old_previews_async (crd->recursive,
						  crd->clear_all);
	g_free (crd);
}


static void 
check_cache_file (gchar *real_file, 
		  gchar *rc_file, 
		  gpointer data)
{
	CacheRemoveData *crd = data;

	if (crd->clear_all || ! path_is_file (real_file)) 
		if ((unlink (rc_file) < 0))
			g_warning ("Cannot delete %s\n", rc_file);
}


void
cache_remove_old_previews_async (const gchar *dir,
				 gboolean recursive,
				 gboolean clear_all)
{
	CacheRemoveData *crd;

	crd = g_new (CacheRemoveData, 1);
	crd->recursive = recursive;
	crd->clear_all = clear_all;

	visit_rc_directory_async (RC_THUMBS_DIR,
				  CACHE_THUMB_EXT,
				  dir,
				  recursive,
				  check_cache_file,
				  cache_remove_done,
				  crd);
}


/* ----- nautilus_cache_remove_old_previews_async implememtation. ------ */


typedef struct {
	gboolean recursive;
	gboolean clear_all;
	GList *dirs;
	GList *visited_dirs;
	gchar *nautilus_thumb_dir;
	int nautilus_thumb_dir_l;
} NautilusCacheRemoveData;


void
nautilus_cache_data_free (NautilusCacheRemoveData *ncrd)
{
	if (ncrd == NULL)
		return;

	if (ncrd->dirs != NULL) {
		g_list_foreach (ncrd->dirs, (GFunc) g_free, NULL);
		g_list_free (ncrd->dirs);
	}

	if (ncrd->visited_dirs != NULL) {
		g_list_foreach (ncrd->visited_dirs, (GFunc) g_free, NULL);
		g_list_free (ncrd->visited_dirs);
	}

	if (ncrd->nautilus_thumb_dir)
		g_free (ncrd->nautilus_thumb_dir);

	g_free (ncrd);
}


static void visit_dir_async (const gchar *dir,
			     NautilusCacheRemoveData *ncrd);


#define ESC_FILE_PREFIX    "file:%2F%2F"
#define ESC_FILE_PREFIX_L  11
#define ESC_SLASH          "%2F"
#define ESC_SLASH_L        3


static gchar *
get_real_name_from_nautilus_cache (NautilusCacheRemoveData *ncrd,
				   char *cache_name_full_path)
{
	int cache_name_l;
	int real_name_l;
	int scan_l;
	int n;
	char *cache_name;
	char *real_name;
	char *scan;

	if (strlen (cache_name_full_path) < ncrd->nautilus_thumb_dir_l + 1)
		return NULL;

	cache_name = cache_name_full_path + ncrd->nautilus_thumb_dir_l + 1;

	cache_name_l = strlen (cache_name);
	if (cache_name_l < ESC_FILE_PREFIX_L)
		return NULL;

	if (strncmp (cache_name, ESC_FILE_PREFIX, ESC_FILE_PREFIX_L) != 0)
		return NULL;

	real_name = g_new (char, strlen (cache_name) + 1);
	n = 0;

	scan_l = cache_name_l - ESC_FILE_PREFIX_L;
	for (scan = cache_name + ESC_FILE_PREFIX_L; *scan != 0; ) 
		if ((scan_l >= ESC_SLASH_L) 
		    && strncmp (scan, ESC_SLASH, ESC_SLASH_L) == 0) {
			real_name[n++] = '/';
			scan   += ESC_SLASH_L;
			scan_l -= ESC_SLASH_L;
		} else {
			real_name[n++] = *scan;
			scan++;
			scan_l--;
		}
	real_name[n] = 0;

	real_name_l = strlen (real_name);
	real_name[real_name_l - 4] = 0; /* 4 = strlen (".png") */

	if (! file_is_image (real_name, TRUE))
		real_name[real_name_l - 4] = '.';

	return real_name;
}


static void
path_list_done_cb (PathListData *pld, 
		   gpointer data)
{
	NautilusCacheRemoveData *ncrd = data;
	GList *scan;
	gchar *rc_file, *real_file;
	gchar *sub_dir;

	if (pld->result != GNOME_VFS_ERROR_EOF) {
		gchar *path;

		path = gnome_vfs_uri_to_string (pld->uri,
						GNOME_VFS_URI_HIDE_NONE);
		g_warning ("Error reading cache directory %s.", path);
		g_free (path);

		nautilus_cache_data_free (ncrd);
		return;
	}

	for (scan = pld->files; scan; scan = scan->next) {
		rc_file = (gchar*) scan->data;
		real_file = get_real_name_from_nautilus_cache (ncrd, rc_file);

		if (real_file == NULL)
			continue;

		if (ncrd->clear_all || ! path_is_file (real_file)) 
			if ((unlink (rc_file) < 0)) 
				g_warning ("Cannot delete %s\n", rc_file);

		g_free (real_file);
	}

	if (! ncrd->recursive) {
		path_list_data_free (pld);
		nautilus_cache_data_free (ncrd);
		return;
	}

	ncrd->dirs = g_list_concat (pld->dirs, ncrd->dirs);
	pld->dirs = NULL;
	path_list_data_free (pld);

	if (ncrd->dirs == NULL) {
		if (ncrd->clear_all) {
			const GList *scan = ncrd->visited_dirs;

			for (; scan; scan = scan->next) {
				gchar *dir = scan->data;
				rmdir (dir);
			}
		}
		nautilus_cache_data_free (ncrd);
		return;
	}

	sub_dir = (gchar*) ncrd->dirs->data;
	ncrd->dirs = g_list_remove_link (ncrd->dirs, ncrd->dirs);

	ncrd->visited_dirs = g_list_prepend (ncrd->visited_dirs,
					     g_strdup (sub_dir));
	visit_dir_async (sub_dir, ncrd);

	g_free (sub_dir);
}


static void
visit_dir_async (const gchar *dir,
		 NautilusCacheRemoveData *data)
{
	PathListHandle *handle;

	handle = path_list_async_new (dir, path_list_done_cb, data);
	g_free (handle);
}


static void 
nautilus_cache_remove_old_previews_async (gboolean recursive,
					  gboolean clear_all)
{
	NautilusCacheRemoveData *ncrd;

	ncrd = g_new (NautilusCacheRemoveData, 1);

	ncrd->recursive = recursive;
	ncrd->clear_all = clear_all;
	ncrd->dirs = NULL;
	ncrd->visited_dirs = NULL;

	ncrd->nautilus_thumb_dir = g_strconcat (g_get_home_dir (),
						"/.nautilus/thumbnails",
						NULL);
	ncrd->nautilus_thumb_dir_l = strlen (ncrd->nautilus_thumb_dir);

	visit_dir_async (ncrd->nautilus_thumb_dir, ncrd);
}
