/* widgets.c */
/* gtkfind code for making and displaying the interface, callbacks... */
/*
  gtkfind - a graphical "find" program
  Copyright (C) 1999  Matthew Grossman <mattg@oz.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include"widgets.h"


/* the path to xterm */
#define CONFIG_XTERM "/usr/X11R6/bin/xterm"

/* the path to man */
#define CONFIG_MAN "/usr/bin/man"


/* local variables */

static gfloat current_day = 0.0;
static gfloat current_month = 0.0;
static gfloat current_year = 0.0;
static gfloat current_hour = 0.0;
static gfloat current_minute = 0.0;
static gfloat current_second = 0.0;


void
make_widgets()
     /* make all the widgets and set them up properly */
{
  GtkWidget *hsep = NULL;
  GtkWidget *vbox = NULL;

  toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_container_border_width(GTK_CONTAINER(toplevel), 10);
  gtk_window_set_policy(GTK_WINDOW(toplevel), FALSE, TRUE, FALSE);
  gtk_signal_connect(GTK_OBJECT(toplevel), "delete_event",
		     GTK_SIGNAL_FUNC(quit), NULL);
  gtk_signal_connect(GTK_OBJECT(toplevel), "destroy", GTK_SIGNAL_FUNC(quit),
		     NULL);

  
  vbox = create_vbox(toplevel);

  make_directory_widgets(vbox);

  hsep = create_hseparator(vbox);

  make_notebook_widgets(vbox);
  
  hsep = create_hseparator(vbox);

  make_shell_widgets(vbox);
  
  hsep = create_hseparator(vbox);

  make_button_widgets(vbox);

  gtk_widget_show(toplevel);
  return;
}

void
make_button_widgets(GtkWidget *parent)
     /* make the "quit" and "find" buttons */
{
  GtkWidget *hbox = NULL;
  GtkWidget *button = NULL;
  GtkWidget *label = NULL;

  hbox = create_hbox(parent);

  /* the types aren't quite right, so we can't use create_button for some
     of these */
  
  button = gtk_button_new();
  gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(quit),
		     NULL);
  gtk_container_add(GTK_CONTAINER(hbox), button);
  gtk_widget_show(button);
  label = create_label(button, "Quit");

  button = create_button(hbox, help_callback, NULL);
  label = create_label(button, "Help");
  
  button = create_button(hbox, clear_find_callback, NULL);
  label = create_label(button, "Clear");

  button = create_button(hbox, save_search_callback, NULL);
  label = create_label(button, "Save");

  button = gtk_button_new();
  gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(find),
		     pattern);
  gtk_container_add(GTK_CONTAINER(hbox), button);
  gtk_widget_show(button);
  label = create_label(button, "Find");

  /* we don't show the stop button now, so we can't use create_button */
  stop_button = gtk_button_new();
  gtk_signal_connect(GTK_OBJECT(stop_button), "clicked",
		     GTK_SIGNAL_FUNC(stop_find_callback),
		     NULL);
  gtk_container_add(GTK_CONTAINER(hbox), stop_button);
  
  label = gtk_label_new("Stop");
  gtk_container_add(GTK_CONTAINER(stop_button), label);
  gtk_widget_show(label);

  return;
}


void
make_shell_widgets(GtkWidget *parent)
     /* make the widgets associated with running shell commands */
{
  GtkWidget *toggle = NULL;
  GtkWidget *hbox = NULL;
  GtkWidget *label = NULL;
  GtkWidget *radio = NULL;
  GSList *gslist = NULL;

  /* output format selection */

  hbox = create_hbox(parent);

  radio = gtk_radio_button_new_with_label(NULL, "Print only filename");
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(radio), TRUE);
  gtk_signal_connect(GTK_OBJECT(radio), "toggled",
		     GTK_SIGNAL_FUNC(output_format_toggle_callback), NULL);
  gtk_container_add(GTK_CONTAINER(hbox), radio);
  gtk_widget_show(radio);

  gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(radio));
  
  radio = gtk_radio_button_new_with_label(gslist, "Print extra data");
  gtk_container_add(GTK_CONTAINER(hbox), radio);
  gtk_widget_show(radio);

  /* shell command stuff */

  toggle = create_toggle_button(parent, shell_command_toggle_callback, NULL,
				FALSE, "Run a shell command?");

  hbox = create_hbox(parent);
  label = create_label(hbox, "Shell command:");
  shell_command = create_entry(hbox);

  hbox = create_hbox(parent);

  toggle = create_toggle_button(hbox, print_to_stdout_toggle_callback, NULL,
				FALSE, "Print to stdout");
  toggle = create_toggle_button(hbox, print_to_window_toggle_callback, NULL,
				TRUE, "Print to window");
  set_flag(PRINT_TO_WINDOW_P, 1);

  toggle = create_toggle_button(parent, print_filename_anyway_toggle_callback,
				NULL, FALSE, "Always print filename");

  return;
}

void
make_directory_widgets(GtkWidget *parent)
     /* make the widgets involved with choosing the directory to search */
{
  GtkWidget *label = NULL;
  GtkWidget *toggle = NULL;
  GtkWidget *hbox = NULL;
  GtkWidget *button = NULL;

  hbox = create_hbox(parent);

  label = create_label(hbox, "Search directory:");

  button = create_button(hbox, find_directory_callback, NULL);
  create_label(button, "Choose Directory");

  directory = create_entry(parent);

  toggle = create_toggle_button(parent, search_subdirs_toggle_callback, NULL,
				TRUE, "Search subdirectories");
  set_flag(SEARCH_SUBDIRS_P, 1);
  
  return;
}


void
make_notebook_widgets(GtkWidget *parent)
     /* make the notebook1 and its subwidgets, which we use to select
	the files to match by */
{
  GtkWidget *notebook = NULL;

  get_current_mdy(&current_day, &current_month, &current_year);

  
  notebook = gtk_notebook_new();
  gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook), GTK_POS_TOP);
  gtk_container_add(GTK_CONTAINER(parent), notebook);
  gtk_widget_show(notebook);

  make_match_filename_widgets(notebook);

  make_match_atime_widgets(notebook);
  make_match_ctime_widgets(notebook);
  make_match_mtime_widgets(notebook);

  make_match_mode_widgets(notebook);
  make_match_type_widgets(notebook);
  make_match_login_widgets(notebook);

  make_match_contents_widgets(notebook);
  
  return;
}

void
make_match_contents_widgets(GtkWidget *notebook)
     /* make a widget that lets you match a file by a text string inside it */
{
  GtkWidget *frame = NULL;
  GtkWidget *label = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *hbox = NULL;
  GSList *gslist = NULL;
  GtkWidget *radio = NULL;

  /* Match Filename */
  
  frame = gtk_frame_new("Match Contents");
  gtk_widget_show(frame);

  vbox = create_vbox(frame);

  label = create_label(vbox, "Match file containing:");
  
  /* we want to match simple substrings by default, but have the option
     to match wildcards.  If you want regexps, use grep */

  hbox = create_hbox(vbox);
  
  radio = gtk_radio_button_new_with_label(NULL, "simple substring");
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(radio), TRUE);
  gtk_signal_connect(GTK_OBJECT(radio), "toggled",
		     GTK_SIGNAL_FUNC(content_search_toggle_callback), NULL);
  gtk_object_set_data(GTK_OBJECT(radio), "reset_yourself",
		     GTK_SIGNAL_FUNC(reset_toggle_button_down));
  gtk_container_add(GTK_CONTAINER(hbox), radio);
  gtk_widget_show(radio);
  
  gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(radio));

  radio = gtk_radio_button_new_with_label(gslist, "wildcard pattern");
  gtk_object_set_data(GTK_OBJECT(radio), "reset_yourself",
		     GTK_SIGNAL_FUNC(reset_toggle_button_up));
  gtk_container_add(GTK_CONTAINER(hbox), radio);
  gtk_widget_show(radio);

  content_pattern = create_entry(vbox);
  gtk_object_set_data(GTK_OBJECT(content_pattern), "reset_yourself",
		      reset_entry);

  label = gtk_label_new("Contents");
  gtk_widget_show(label);
  
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  return;

}

void
make_match_filename_widgets(GtkWidget *notebook)
     /* make the first frame of notebook */
{
  GtkWidget *frame = NULL;
  GtkWidget *label = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *hbox = NULL;
  GtkWidget *radio = NULL;
  GSList *gslist = NULL;

  
  /* Match Filename */
  
  frame = gtk_frame_new("Match Filename");
  gtk_widget_show(frame);

  vbox = create_vbox(frame);

  /* regular expression matching isn't implemented */
  /*
  toggle = gtk_toggle_button_new_with_label("Use regular expressions?");
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(toggle), FALSE);
  gtk_container_add(GTK_CONTAINER(vbox), toggle);
  gtk_signal_connect(GTK_OBJECT(toggle), "toggled",
		     GTK_SIGNAL_FUNC(regexp_toggle_callback), NULL);
  gtk_widget_show(toggle);
  */

  label = create_label(vbox, "Match files whose names contain:");
  
  hbox = create_hbox(vbox);
    
  radio = gtk_radio_button_new_with_label(NULL, "simple substring");
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(radio), TRUE);
  gtk_signal_connect(GTK_OBJECT(radio), "toggled",
		     GTK_SIGNAL_FUNC(filename_match_toggle_callback), NULL);
  gtk_object_set_data(GTK_OBJECT(radio), "reset_yourself",
		     GTK_SIGNAL_FUNC(reset_toggle_button_down));
  gtk_container_add(GTK_CONTAINER(hbox), radio);
  gtk_widget_show(radio);
  
  gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(radio));

  radio = gtk_radio_button_new_with_label(gslist, "wildcard pattern");
  gtk_object_set_data(GTK_OBJECT(radio), "reset_yourself",
		     GTK_SIGNAL_FUNC(reset_toggle_button_up));
  gtk_container_add(GTK_CONTAINER(hbox), radio);
  gtk_widget_show(radio);

  pattern = create_entry(vbox);
  gtk_object_set_data(GTK_OBJECT(pattern), "reset_yourself", reset_entry);

  label = gtk_label_new("Filename");
  gtk_widget_show(label);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  return;
}

void
make_match_atime_widgets(GtkWidget *notebook)
{

  GtkWidget *frame = NULL;
  GtkWidget *label = NULL;
  GtkWidget *whatis = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *toggle = NULL;
  GtkWidget *hbox = NULL;

    /* Match Atime */
  
  frame = gtk_frame_new("Match atime");
  gtk_widget_show(frame);

  vbox = create_vbox(frame);

  /* earlier than, later than, or equal too...you can select more than one */

  whatis = create_label(vbox, "atime is the last time the file was read or executed");

  hbox = create_hbox(vbox);

  toggle = create_toggle_button(hbox, atime_et_toggle_callback, NULL, FALSE,
				"Earlier than");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);
  
  toggle = create_toggle_button(hbox, atime_eq_toggle_callback, NULL, FALSE,
				"Equal to:");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);
  
  toggle = create_toggle_button(hbox, atime_lt_toggle_callback, NULL, FALSE,
				"Later than:");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  make_all_spinners(vbox,
		    &atime_day_spin, &atime_month_spin, &atime_year_spin,
		    &atime_hour_spin, &atime_minute_spin, &atime_second_spin);
  
  label = gtk_label_new("atime");
  gtk_widget_show(label);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  return;
}

void
make_match_ctime_widgets(GtkWidget *notebook)
{
  GtkWidget *frame = NULL;
  GtkWidget *label = NULL;
  GtkWidget *whatis = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *toggle = NULL;
  GtkWidget *hbox = NULL;

   /* Match Ctime */
  
  frame = gtk_frame_new("Match ctime");
  gtk_widget_show(frame);

  vbox = create_vbox(frame);

  whatis = create_label(vbox, "ctime is the last time the file's inode was changed");
  
  /* earlier than, later than, or equal too...you can select more than one */

  hbox = create_hbox(vbox);

  toggle = create_toggle_button(hbox, ctime_et_toggle_callback, NULL, FALSE,
				"Earlier than:");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);
  
  toggle = create_toggle_button(hbox, ctime_eq_toggle_callback, NULL, FALSE,
				"Equal to:");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);
  
  toggle = create_toggle_button(hbox, ctime_lt_toggle_callback, NULL, FALSE,
				"Later than");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  make_all_spinners(vbox,
		    &ctime_day_spin, &ctime_month_spin, &ctime_year_spin,
		    &ctime_hour_spin, &ctime_minute_spin, &ctime_second_spin);
  
  label = gtk_label_new("ctime");
  gtk_widget_show(label);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);


  return;
}

void
make_match_mtime_widgets(GtkWidget *notebook)
{
  GtkWidget *frame = NULL;
  GtkWidget *label = NULL;
  GtkWidget *whatis = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *toggle = NULL;
  GtkWidget *hbox = NULL;


 /* Match Mtime */
  
  frame = gtk_frame_new("Match mtime");
  gtk_widget_show(frame);

  vbox = create_vbox(frame);

  whatis = create_label(vbox, "mtime is the last time the file was written");
  
  /* earlier than, later than, or equal too...you can select more than one */

  hbox = create_hbox(vbox);
  toggle = create_toggle_button(hbox, mtime_et_toggle_callback, NULL, FALSE,
				"Earlier than:");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(hbox, mtime_eq_toggle_callback, NULL, FALSE,
				"Equal to:");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(hbox, mtime_lt_toggle_callback, NULL, FALSE,
				"Later than:");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  make_all_spinners(vbox,
		    &mtime_day_spin, &mtime_month_spin, &mtime_year_spin,
		    &mtime_hour_spin, &mtime_minute_spin, &mtime_second_spin);
  
  label = gtk_label_new("mtime");
  gtk_widget_show(label);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  
  return;
} 

void
make_match_mode_widgets(GtkWidget *notebook)
{
  GtkWidget *frame = NULL;
  GtkWidget *label = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *toggle = NULL;
  GtkWidget *hbox = NULL;


  /* Match Mode */

  frame = gtk_frame_new("Match Mode");
  gtk_widget_show(frame);

  hbox = create_hbox(frame);

  /* read */

  vbox = create_vbox(hbox);

  toggle = create_toggle_button(vbox, owner_read_toggle_callback, NULL, FALSE,
				"owner read");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(vbox, group_read_toggle_callback, NULL, FALSE,
				"group read");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(vbox, world_read_toggle_callback, NULL, FALSE,
				"world read");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  /* write */

  vbox = create_vbox(hbox);

  toggle = create_toggle_button(vbox, owner_write_toggle_callback, NULL,
				FALSE, "owner write");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(vbox, group_write_toggle_callback, NULL,
				FALSE, "group write");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(vbox, world_write_toggle_callback, NULL,
				FALSE, "world write");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  /* exec */

  vbox = create_vbox(hbox);

  toggle = create_toggle_button(vbox, owner_exec_toggle_callback, NULL,
				FALSE, "owner execute");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(vbox, group_exec_toggle_callback, NULL,
				FALSE, "group execute");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(vbox, world_exec_toggle_callback, NULL,
				FALSE, "world execute");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  label = gtk_label_new("Mode");
  gtk_widget_show(label);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  return;
}

void
make_match_type_widgets(GtkWidget *notebook)
{
  GtkWidget *frame = NULL;
  GtkWidget *label = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *toggle = NULL;
  GtkWidget *hbox = NULL;


  /* Match Type */


  frame = gtk_frame_new("Match Type");
  gtk_widget_show(frame);

  hbox = create_hbox(frame);

  /* extra bits */

  vbox = create_vbox(hbox);

  
  toggle = create_toggle_button(vbox, setuid_toggle_callback, NULL,
				FALSE, "setuid");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

    
  toggle = create_toggle_button(vbox, setgid_toggle_callback, NULL,
				FALSE, "setgid");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  
  toggle = create_toggle_button(vbox, sticky_toggle_callback, NULL,
				FALSE, "sticky");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  /* directory or regular file */

  vbox = create_vbox(hbox);

  toggle = create_toggle_button(vbox, directory_toggle_callback, NULL,
				FALSE, "directory");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(vbox, regular_toggle_callback, NULL,
				FALSE, "regular");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  /* special files */

  vbox = create_vbox(hbox);

  toggle = create_toggle_button(vbox, raw_device_toggle_callback, NULL,
				FALSE, "raw device");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(vbox, block_device_toggle_callback, NULL,
				FALSE, "block device");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(vbox, symlink_toggle_callback, NULL,
				FALSE, "symlink");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(vbox, socket_toggle_callback, NULL,
				FALSE, "socket");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  toggle = create_toggle_button(vbox, fifo_toggle_callback, NULL,
				FALSE, "fifo");
  gtk_object_set_data(GTK_OBJECT(toggle), "reset_yourself",
		      reset_toggle_button_up);

  label = gtk_label_new("Type");
  gtk_widget_show(label);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  return;
}

void
make_match_login_widgets(GtkWidget *notebook)
{
  GtkWidget *frame = NULL;
  GtkWidget *label = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *radio = NULL;
  GtkWidget *hbox = NULL;
  GSList *gslist = NULL;

  /* Match Login and/or Group */

  
  frame = gtk_frame_new("Match Login/Group");
  gtk_widget_show(frame);

  vbox = create_vbox(frame);
  
  /* login */

  hbox = create_hbox(vbox);

  radio = gtk_radio_button_new_with_label(NULL, "Match login");
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(radio), TRUE);
  gtk_signal_connect(GTK_OBJECT(radio), "toggled",
		     GTK_SIGNAL_FUNC(uid_not_login_toggle_callback), NULL);
  gtk_container_add(GTK_CONTAINER(hbox), radio);
  gtk_object_set_data(GTK_OBJECT(radio), "reset_yourself",
		      reset_toggle_button_down);
  gtk_widget_show(radio);

  gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(radio));

  radio = gtk_radio_button_new_with_label(gslist, "Match uid");
  gtk_container_add(GTK_CONTAINER(hbox), radio);
  gtk_object_set_data(GTK_OBJECT(radio), "reset_yourself",
		      reset_toggle_button_up);
  gtk_widget_show(radio);

  hbox = create_hbox(vbox);

  label = create_label(hbox, "login/uid");

  login_entry = create_entry(hbox);
  gtk_object_set_data(GTK_OBJECT(login_entry), "reset_yourself", reset_entry);
  
  /* group */

  hbox = create_hbox(vbox);

  radio = gtk_radio_button_new_with_label(NULL, "Match group");
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(radio), TRUE);
  gtk_signal_connect(GTK_OBJECT(radio), "toggled",
		     GTK_SIGNAL_FUNC(gid_not_group_toggle_callback), NULL);
  gtk_container_add(GTK_CONTAINER(hbox), radio);
  gtk_object_set_data(GTK_OBJECT(radio), "reset_yourself",
		      reset_toggle_button_down);
  gtk_widget_show(radio);

  gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(radio));

  radio = gtk_radio_button_new_with_label(gslist, "Match gid");
  gtk_container_add(GTK_CONTAINER(hbox), radio);
  gtk_object_set_data(GTK_OBJECT(radio), "reset_yourself",
		      reset_toggle_button_up);
  gtk_widget_show(radio);

  hbox = create_hbox(vbox);
  label = create_label(hbox, "group/gid:");

  group_entry = create_entry(hbox);
  gtk_object_set_data(GTK_OBJECT(group_entry), "reset_yourself", reset_entry);
  
  label = gtk_label_new("Owner");
  gtk_widget_show(label);
  gtk_notebook_append_page(GTK_NOTEBOOK(notebook), frame, label);

  return;
}


/* callbacks */


void
find_directory_callback()
     /* bring up a find file window to choose the directory to search */
{

  create_filesel("Find Directory", filesel_ok_callback,
		 filesel_cancel_callback);
  return;
}

void
filesel_ok_callback(GtkWidget *w, GtkFileSelection *filesel)
     /* we selected ok in the find directory widget */
{
  char *s = NULL;

  s = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  if(strlen(s) > 0) {
    gtk_entry_set_text(GTK_ENTRY(directory), s);
    gtk_widget_destroy(GTK_WIDGET(filesel));
  }

  return;
}

void
filesel_cancel_callback(GtkWidget *w, GtkFileSelection *filesel)
     /* we selected cancel in the find directory widget */
{
  gtk_widget_destroy(GTK_WIDGET(filesel));

  return;
}

void
print_filename_anyway_toggle_callback()
{
  get_flag(PRINT_FILENAME_ANYWAY_P) ? (set_flag(PRINT_FILENAME_ANYWAY_P, 0)) :
    (set_flag(PRINT_FILENAME_ANYWAY_P, 1));
  return;
}

void
search_subdirs_toggle_callback()
{
  get_flag(SEARCH_SUBDIRS_P) ? (set_flag(SEARCH_SUBDIRS_P, 0)) :
    (set_flag(SEARCH_SUBDIRS_P, 1));
  return;
}


void
regexp_toggle_callback()
{
  get_flag(REGEXP_P) ? (set_flag(REGEXP_P, 0)) : (set_flag(REGEXP_P, 1));
  return;
}

void
shell_command_toggle_callback()
{
  get_flag(SHELL_COMMAND_P) ? (set_flag(SHELL_COMMAND_P, 0)) :
    (set_flag(SHELL_COMMAND_P, 1));
  return;
}

void
print_to_stdout_toggle_callback()
{
  get_flag(PRINT_TO_STDOUT_P) ? (set_flag(PRINT_TO_STDOUT_P, 0)) :
    (set_flag(PRINT_TO_STDOUT_P, 1));
  return;
}

void
print_to_window_toggle_callback()
{
  get_flag(PRINT_TO_WINDOW_P) ? (set_flag(PRINT_TO_WINDOW_P, 0)) :
    (set_flag(PRINT_TO_WINDOW_P, 1));
  return;
}

void
owner_read_toggle_callback()
{
  get_flag(OWNER_READ_P) ? (set_flag(OWNER_READ_P, 0)) :
    (set_flag(OWNER_READ_P, 1));

  return;
}

void
group_read_toggle_callback()
{
  get_flag(GROUP_READ_P) ? (set_flag(GROUP_READ_P, 0)) :
    (set_flag(GROUP_READ_P, 1));

  return;
}

void
world_read_toggle_callback()
{
  get_flag(WORLD_READ_P) ? (set_flag(WORLD_READ_P, 0)) :
    (set_flag(WORLD_READ_P, 1));

  return;
}


void
owner_write_toggle_callback()
{
  get_flag(OWNER_WRITE_P) ? (set_flag(OWNER_WRITE_P, 0)) :
    (set_flag(OWNER_WRITE_P, 1));

  return;
}

void
group_write_toggle_callback()
{
  get_flag(GROUP_WRITE_P) ? (set_flag(GROUP_WRITE_P, 0)) :
    (set_flag(GROUP_WRITE_P, 1));

  return;
}

void
world_write_toggle_callback()
{
  get_flag(WORLD_WRITE_P) ? (set_flag(WORLD_WRITE_P, 0)) :
    (set_flag(WORLD_WRITE_P, 1));

  return;
}



void
owner_exec_toggle_callback()
{
  get_flag(OWNER_EXEC_P) ? (set_flag(OWNER_EXEC_P, 0)) :
    (set_flag(OWNER_EXEC_P, 1));

  return;
}

void
group_exec_toggle_callback()
{
  get_flag(GROUP_EXEC_P) ? (set_flag(GROUP_EXEC_P, 0)) :
    (set_flag(GROUP_EXEC_P, 1));

  return;
}

void
world_exec_toggle_callback()
{
  get_flag(WORLD_EXEC_P) ? (set_flag(WORLD_EXEC_P, 0)) :
    (set_flag(WORLD_EXEC_P, 1));

  return;
}

void
setuid_toggle_callback()
{
  get_flag(SETUID_P) ? (set_flag(SETUID_P, 0)) :
    (set_flag(SETUID_P, 1));

  return;
}


void
setgid_toggle_callback()
{
  get_flag(SETGID_P) ? (set_flag(SETGID_P, 0)) : (set_flag(SETGID_P, 1));

  return;
}

void
sticky_toggle_callback()
{
  get_flag(STICKY_P) ? (set_flag(STICKY_P, 0)) : (set_flag(STICKY_P, 1));

  return;
}

void
directory_toggle_callback()
{
  get_flag(DIRECTORY_P) ? (set_flag(DIRECTORY_P, 0)) :
    (set_flag(DIRECTORY_P, 1));

  return;
}


void
regular_toggle_callback()
{
  get_flag(REGULAR_P) ? (set_flag(REGULAR_P, 0)) :
    (set_flag(REGULAR_P, 1));

  return;
}


void
raw_device_toggle_callback()
{
  get_flag(RAW_DEVICE_P) ? (set_flag(RAW_DEVICE_P, 0)) :
    (set_flag(RAW_DEVICE_P, 1));

  return;
}


void
block_device_toggle_callback()
{
  get_flag(BLOCK_DEVICE_P) ? (set_flag(BLOCK_DEVICE_P, 0)) :
    (set_flag(BLOCK_DEVICE_P, 1));

  return;
}


void
symlink_toggle_callback()
{
  get_flag(SYMLINK_P) ? (set_flag(SYMLINK_P, 0)) : (set_flag(SYMLINK_P, 1));

  return;
}


void
socket_toggle_callback()
{
  get_flag(SOCKET_P) ? (set_flag(SOCKET_P, 0)) : (set_flag(SOCKET_P, 1));

  return;
}


void
fifo_toggle_callback()
{
  get_flag(FIFO_P) ? (set_flag(FIFO_P, 0)) : (set_flag(FIFO_P, 1));

  return;
}

void
uid_not_login_toggle_callback()
{
  get_flag(UID_NOT_LOGIN_P) ? (set_flag(UID_NOT_LOGIN_P, 0)) :
    (set_flag(UID_NOT_LOGIN_P, 1));

  return;
}

void
gid_not_group_toggle_callback()
{
  get_flag(GID_NOT_GROUP_P) ? (set_flag(GID_NOT_GROUP_P, 0)) :
    (set_flag(GID_NOT_GROUP_P, 1));

  return;
}


void
atime_et_toggle_callback()
{
  get_flag(ATIME_ET_P) ? (set_flag(ATIME_ET_P, 0)) :
    (set_flag(ATIME_ET_P, 1));

  return;
}


void
atime_eq_toggle_callback()
{
  get_flag(ATIME_EQ_P) ? (set_flag(ATIME_EQ_P, 0)) :
    (set_flag(ATIME_EQ_P, 1));

  return;
}


void
atime_lt_toggle_callback()
{
  get_flag(ATIME_LT_P) ? (set_flag(ATIME_LT_P, 0)) :
    (set_flag(ATIME_LT_P, 1));

  return;
}


void
ctime_et_toggle_callback()
{
  get_flag(CTIME_ET_P) ? (set_flag(CTIME_ET_P, 0)) :
    (set_flag(CTIME_ET_P, 1));

  return;
}


void
ctime_eq_toggle_callback()
{
  get_flag(CTIME_EQ_P) ? (set_flag(CTIME_EQ_P, 0)) :
    (set_flag(CTIME_EQ_P, 1));

  return;
}


void
ctime_lt_toggle_callback()
{
  get_flag(CTIME_LT_P) ? (set_flag(CTIME_LT_P, 0)) : (set_flag(CTIME_LT_P, 1));

  return;
}


void
mtime_et_toggle_callback()
{
  get_flag(MTIME_ET_P) ? (set_flag(MTIME_ET_P, 0)) : (set_flag(MTIME_ET_P, 1));

  return;
}


void
mtime_eq_toggle_callback()
{
  get_flag(MTIME_EQ_P) ? (set_flag(MTIME_EQ_P, 0)) : (set_flag(MTIME_EQ_P, 1));

  return;
}


void
mtime_lt_toggle_callback()
{
  get_flag(MTIME_LT_P) ? (set_flag(MTIME_LT_P, 0)) : (set_flag(MTIME_LT_P, 1));

  return;
}

GtkWidget *
make_spin_button(float *default_value, float min_value, float max_value)
     /* create and return a pointer to a GtkWidget spin button */
{
  GtkWidget *rv = NULL;
  GtkAdjustment *adj = NULL;

  adj = (GtkAdjustment *)gtk_adjustment_new(*default_value, min_value,
					    max_value, 1.0, 2.0, 0.0);
  rv = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_wrap(GTK_SPIN_BUTTON(rv), TRUE);
  gtk_object_set_data(GTK_OBJECT(rv), "default_value", default_value);
  gtk_object_set_data(GTK_OBJECT(rv), "reset_yourself",
		      reset_time_spin_button);

  return(rv);
}
  

void
make_all_spinners(GtkWidget *parent, GtkWidget **day,
		    GtkWidget **month, GtkWidget **year, GtkWidget **hour,
		    GtkWidget **minute, GtkWidget **second)
{
  GtkWidget *hbox = NULL;
  GtkWidget *vbox = NULL;
  GtkWidget *label = NULL;

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(parent), hbox);
  gtk_widget_show(hbox);

  /* the day widget */
  
  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(hbox), vbox);
  gtk_widget_show(vbox);

  label = gtk_label_new("Day");
  gtk_container_add(GTK_CONTAINER(vbox), label);
  gtk_widget_show(label);

  *day = make_spin_button(&current_day, 1.0, 31.0);
  gtk_container_add(GTK_CONTAINER(vbox), *day);
  gtk_widget_show(*day);
  
  /* the month widget */

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(hbox), vbox);
  gtk_widget_show(vbox);

  label = gtk_label_new("Month");
  gtk_container_add(GTK_CONTAINER(vbox), label);
  gtk_widget_show(label);

  *month = make_spin_button(&current_month, 1.0, 12.0);
  gtk_container_add(GTK_CONTAINER(vbox), *month);
  gtk_signal_connect_after(GTK_OBJECT(*month), "changed",
			   GTK_SIGNAL_FUNC(month_changed_callback), *day);
			   

  gtk_widget_show(*month);

  
  /* the year widget */

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(hbox), vbox);
  gtk_widget_show(vbox);

  label = gtk_label_new("Year");
  gtk_container_add(GTK_CONTAINER(vbox), label);
  gtk_widget_show(label);

  *year = make_spin_button(&current_year, 0.0, 9999.0);
  gtk_widget_set_usize(*year, 55, 0); /* make it 4 digits wide */
  gtk_container_add(GTK_CONTAINER(vbox), *year);
  gtk_signal_connect_after(GTK_OBJECT(*year), "changed",
			   GTK_SIGNAL_FUNC(year_changed_callback), *day);
  gtk_widget_show(*year);


  /* the hour widget */

    
  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(hbox), vbox);
  gtk_widget_show(vbox);

  label = gtk_label_new("Hour");
  gtk_container_add(GTK_CONTAINER(vbox), label);
  gtk_widget_show(label);

  *hour = make_spin_button(&current_hour, 0.0, 23.0);
  gtk_container_add(GTK_CONTAINER(vbox), *hour);
  gtk_widget_show(*hour);

  
  /* the minute widget */

  
  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(hbox), vbox);
  gtk_widget_show(vbox);

  label = gtk_label_new("Minute");
  gtk_container_add(GTK_CONTAINER(vbox), label);
  gtk_widget_show(label);

  *minute = make_spin_button(&current_minute, 0.0, 59.0);
  gtk_container_add(GTK_CONTAINER(vbox), *minute);
  gtk_widget_show(*minute);

  
  /* the second widget */

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(hbox), vbox);
  gtk_widget_show(vbox);

  label = gtk_label_new("Second");
  gtk_container_add(GTK_CONTAINER(vbox), label);
  gtk_widget_show(label);

  *second = make_spin_button(&current_second, 0.0, 59.0);
  gtk_container_add(GTK_CONTAINER(vbox), *second);
  gtk_widget_show(*second);

  return;
  
}

void
get_current_mdy(gfloat *current_day, gfloat *current_month,
		gfloat *current_year)
{
  struct tm *tm;
  time_t now = 0;

  now = time(NULL);
  tm = localtime(&now);

  *current_month = (gfloat)tm->tm_mon + 1;
  *current_day = (gfloat)tm->tm_mday;
  *current_year = (gfloat)tm->tm_year + 1900;

  return;
}

void
clear_find_callback()
     /* the "clear" button will reset the search pattern (ie the notebook
	stuff), and not change anything else */
{

  gtk_container_foreach(GTK_CONTAINER(toplevel), reset_widget, NULL);
  reset_flags();

  return;
}

void
reset_widget(GtkWidget *widget, gpointer callback_data)
     /* run on each widget in the program */
{
  caddr_t (*reset_function)();
  
  if(GTK_IS_CONTAINER(widget))
    gtk_container_foreach(GTK_CONTAINER(widget), reset_widget, NULL);
  
  reset_function = gtk_object_get_data(GTK_OBJECT(widget), "reset_yourself");
  if(reset_function)
    (*reset_function)(widget);

  return;
}
			 
void
reset_entry(GtkWidget *widget)
{
  gtk_entry_set_text(GTK_ENTRY(widget), "");

  return;
}

void
reset_toggle_button_up(GtkWidget *widget)
{
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widget), FALSE);

  return;
}

void
reset_toggle_button_down(GtkWidget *widget)
{
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widget), TRUE);

  return;
}

void
reset_time_spin_button(GtkWidget *widget)
     /* here we will save the default value of each spin button as we
	make it, then use it to reset the widget */
{
  gfloat *value = NULL;

  value = gtk_object_get_data(GTK_OBJECT(widget), "default_value");
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(widget), *value);


  return;
}

void
output_format_toggle_callback()
{
  get_flag(LONG_OUTPUT_P) ? (set_flag(LONG_OUTPUT_P, 0)) :
    (set_flag(LONG_OUTPUT_P, 1));

  return;
}

void
stop_find_callback()
{
  stop_flag = 1;
  return;
}

void
content_search_toggle_callback()
{
  get_flag(WILDCARD_CONTENTS_SEARCH_P) ?
    (set_flag(WILDCARD_CONTENTS_SEARCH_P, 0)) :
    (set_flag(WILDCARD_CONTENTS_SEARCH_P, 1));

  return;
}

void
filename_match_toggle_callback()
{
  if(get_flag(WILDCARD_FILENAME_MATCH_P))
    set_flag(WILDCARD_FILENAME_MATCH_P, 0);
  else
    set_flag(WILDCARD_FILENAME_MATCH_P, 1);

  return;
}

void
year_changed_callback(GtkWidget *widget, gpointer callback_data)
     /* we only need to do something about february */
{
  GtkWidget *month_spin = NULL;
  int month = 0;
  int day = 0;
  GtkAdjustment* nadj = NULL;
  int max_day = 0;

  if(callback_data == atime_day_spin)
    month_spin = atime_month_spin;
  else if(callback_data == mtime_day_spin)
    month_spin = mtime_month_spin;
  else if(callback_data == ctime_day_spin)
    month_spin = ctime_month_spin;
  else {
    print_error("year_changed_callback: Can't get month_spin!\n");
    goto ERROR;
  }

  month = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(month_spin));
  if(month == 2) {
    if(leap_year_p(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget))))
      max_day = 29;
    else
      max_day = 28;

    day = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(callback_data));
    if(day > max_day) {
      day = max_day;
      gtk_spin_button_set_value(GTK_SPIN_BUTTON(callback_data), (gfloat)day);
    }
    nadj = (GtkAdjustment *)gtk_adjustment_new((float)day, 1, max_day, 1.0,
					       2.0, 0.0);
    gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(callback_data), nadj);
  }
  
 DONE:
  return;
 ERROR:
  goto DONE;
}
    
void
month_changed_callback(GtkWidget *widget, gpointer callback_data)
{
  int ovalue = 0;
  int nvalue = 0;
  int month = 0;
  int max_date = 0;
  GtkAdjustment *nadj = NULL;
  GtkWidget *year_spin = NULL;
    
  month = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
  ovalue = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(callback_data));

  if(month == 2) {
    if(callback_data == atime_day_spin)
      year_spin = atime_year_spin;
    else if(callback_data == mtime_day_spin)
      year_spin = mtime_year_spin;
    else if(callback_data == ctime_day_spin)
      year_spin = ctime_year_spin;
    else { /* can't happen? */
      print_error("month_changed_callback: Can't identify callback_data!");
      goto ERROR;
    }
    if(leap_year_p(gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(year_spin))))
      max_date = 29;
    else
      max_date = 28;
  }
  else if(month == 9 || month == 4 || month == 11) {
    max_date = 30;
  }
  else {
    max_date = 31;
  }

  if(ovalue > max_date) {
    nvalue = max_date;
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(callback_data), (gfloat)nvalue);
  }
  else
    nvalue = ovalue;
  
  nadj = (GtkAdjustment *)gtk_adjustment_new((float)nvalue, 1, max_date, 1.0,
					     2.0, 0.0);
  gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(callback_data), nadj);

 DONE:
  return;
 ERROR:
  goto DONE;
}

void
help_callback()
{
  int result = 0;
  char *cmd = NULL;

  cmd = (char *)alloca((strlen(CONFIG_XTERM) + strlen(CONFIG_MAN) + 32) *
		       sizeof(char));
  if(!cmd) {
    print_error("help_callback: Can't alloca cmd!");
    goto DONE;
  }

  sprintf(cmd, "exec %s -e %s gtkfind &", CONFIG_XTERM, CONFIG_MAN);

  if(system(NULL) == 0) {
    print_error("help_callback: /bin/sh is not available!");
    goto DONE;
  }
  
  result = system(cmd);
  if(result == -1) {
    print_error("help_callback: %s", strerror(errno));
  }

 DONE:
  return;
}
    
void
save_search_callback()
{
  GtkWidget *filesel = NULL;

  filesel = create_filesel("Save Search", save_search_ok, save_search_cancel);

  return;
}

void
save_search_ok(GtkWidget *w, GtkFileSelection *filesel)
{
  char *s = NULL;

  s = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  if(strlen(s) > 0) {
    save_search_command(s);
    gtk_widget_destroy(GTK_WIDGET(filesel));
  }

  return;
}

void
save_search_cancel(GtkWidget *w, GtkFileSelection *filesel)
{
  gtk_widget_destroy(GTK_WIDGET(filesel));

  return;
}
  
