#ifndef CLIPBOARD_H
#define CLIPBOARD_H

/* Clipboard handling functionality */

typedef enum
{
   CopyMode,
   CutMode
} ObtainContentMode;

/* The two function prototypes below return the current selection of their
 * associated widget and perform whatever action is necessary to do a cut
 * in case of the second function.
 * Both functions may return NULL if no selection is currently available */
typedef const char* (*clipboard_obtaincontent_t) (ObtainContentMode mode, void *data);
/* execute paste request. Passed parameter const char * specifies the
 * selection to be pasted. This function must return true on success */
typedef int         (*clipboard_paste_t)(void *data,const char *);

/* called when a paste has been performed on an object previously obtained
 * through the clipboard_copy_t or clipboard_cut_t callbacks.
 * if success is FALSE, the selection has been lost because some other application/part of gtoaster
 * became user of the clipboard. 
 * Eventually, this callback handler can free the data pointer in this case.
 * If success happens to be TRUE, however, data may not be freed as multiple
 * paste events may happen */
typedef void	    (*clipboard_operationcomplete_cb_t)(void *data, int success, const char *clipboard_data);

/* This structure consists of three callbacks called whenever 
 * the user selects copy/cut/paste in the menu or anywhere else.
 * A widget has to realize it is the current one and register itself
 * with the clipboard handler as soon as this is the case by passing
 * it a pointer to the structure below */
typedef struct
{
   clipboard_obtaincontent_t  	    obtain;
   clipboard_paste_t 	      	    paste;
   clipboard_operationcomplete_cb_t completion;
   void             *data;
} clipboard_handlerinfo;

void clipboard_setcurrenthandler(const clipboard_handlerinfo *currenthandler);
/* Remove current clipboard handler if matching passed handler */
int  clipboard_removeifcurrent(const clipboard_handlerinfo *currenthandler);

/* The functions below can be used to trigger respectively a copy/paste/cut 
 * action using the current clipboard handler */
void clipboard_perform_copy();
void clipboard_perform_paste();
void clipboard_perform_cut();

/* Initialize clipboard. Called by main.c */
void clipboard_init();

#endif
