/* This file defines the setup for the filetypes supported by 
 * stdfiletrack.c as well as the filetype support database itself.
 * on init,it creates a new page in the preferences notebook and
 * displays a vbox containing a list of those definitions.
 * (I would,of course,prefer a clist with all the convenience this would 
 * provide,but a gtk clist does not appear to be capable of holding more
 * than a few columns of text and pixmaps)
 * on exit,the filetype definitions should be stored to the config file
 * within a separate section) */


/* uncomment for debugging */
/* #define DEBUG */

#include <gtk/gtk.h>
#include <stdlib.h>
#include <stdio.h>

#include "int.h"

#include "main.h"
#include "preferences.h"
#include "filetypes.h"
#include "tracks.h"
#include "helpings.h"
#include "configfile.h"
#include "forms.h"

#include "defaultentries.h"

/* the following are nothing but example definitions for the filetype list
 * this is quite comfortable and shows the user how to use the list for
 * his own entries */
char *filetypes_filterfunctions[]=
{
	DEFAULTENTRIES_LITTLEENDIAN,
	DEFAULTENTRIES_BIGENDIAN,
	NULL
}
;

char *filetypes_filterexamples[]=
{
   DEFAULTENTRIES_EXEC_MPG123,
     DEFAULTENTRIES_EXEC_SOXWAV,
     NULL
}
;

char *filetypes_detectsizeexamples[]=
{   
   DEFAULTENTRIES_EXEC_MP3INFO,
     NULL /* no external tracksize detectors currently available */
}
;

char *filetypes_sizealgexamples[]=
{
   DEFAULTENTRIES_SIZEALG_MP3,
     DEFAULTENTRIES_SIZEALG_WAV,
     "getpos(<y>,<x>)",
     NULL /* "" maps into size of the original file */
}
;


char *filetypes_suffixexamples[]=
{
	".wav",".mp3",".raw",".cdda",NULL
}
;
/* by convention with stdfiletrack.c an empty string means that no conversion
 * whatsoever is done to the initial values,data.
 * e.g. "" in detectsize means that the size of the input file is taken
 * as tracksize. "" as filtername means that no filter should be applied to
 * that file. */
	
GList *filetypes_filetypeentries=NULL;

GtkWidget *filetypes_typearea;

/* some size definition for the filetype list */
#define suffix_size 64
#define tracktype_size 64
#define precache_size 20
#define filter_size 96
#define filtername_size 100
#define detectsize_size 90
#define sizealg_size 90
#define selectbutton_size 20

#define entryheight 20

filetypes_filetypeentry *filetypes_entrycreate(char *suffix,
					       char *tracktype,
					       int  precache,
					       char *filter,
					       char *filtername,
					       char *detectsize,
					       char *sizealg,
					       char *encoder,
					       int  enc_lendian,
					       int  enc_default,
					       char *settitle
					       )
{
   filetypes_filetypeentry *entry;
   
   entry=(filetypes_filetypeentry*)malloc(sizeof(filetypes_filetypeentry));
   /* initialize the clear text fields only,all visuals are created
    * when the entry is registered to the databse */	
   strcpy(entry->suffix,suffix);
   strcpy(entry->tracktype,tracktype);
   entry->precache=precache;
   strcpy(entry->filter,filter);
   strcpy(entry->filtername,filtername);
   strcpy(entry->detectsize,detectsize);
   strcpy(entry->sizealg,sizealg);
   
   strcpy(entry->encoder,encoder);
   entry->enc_lendian=enc_lendian;
   entry->enc_default=enc_default;
   strcpy(entry->settitle,settitle);
      
   return entry;
}
;

typedef struct 
{
	GtkWidget *fs;
	gpointer data;
} filetypes_fileselectinfo;

void filetypes_fileselected(GtkWidget *w,gpointer data)
{
	filetypes_fileselectinfo *fsinfo;
	
	fsinfo=(filetypes_fileselectinfo*)data;
	
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(fsinfo->data)->entry),
			   gtk_file_selection_get_filename(GTK_FILE_SELECTION(fsinfo->fs)));
	
	gtk_widget_destroy(fsinfo->fs);
	free(fsinfo);
}
;

void filetypes_fileselectcancel(GtkWidget *w,gpointer data)
{
	filetypes_fileselectinfo *fsinfo;
	
	fsinfo=(filetypes_fileselectinfo*)data;
	
	gtk_widget_destroy(fsinfo->fs);
	free(fsinfo);
}
;
		
void filetypes_fileselect(GtkWidget *w,
			  gpointer data)
{
	filetypes_fileselectinfo *fsinfo;
	
	fsinfo=(filetypes_fileselectinfo*)malloc(sizeof(filetypes_fileselectinfo));

	fsinfo->fs=gtk_file_selection_new(_("Select File"));
	fsinfo->data=data;
	gtk_file_selection_set_filename(GTK_FILE_SELECTION(fsinfo->fs),
					gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(fsinfo->data)->entry)));
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fsinfo->fs)->ok_button),"clicked",
			   GTK_SIGNAL_FUNC(filetypes_fileselected),fsinfo);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fsinfo->fs)->cancel_button),"clicked",
			   GTK_SIGNAL_FUNC(filetypes_fileselectcancel),fsinfo);
	gtk_widget_show(fsinfo->fs);
}
;


void filetypes_apply(GtkWidget *w,gpointer data)
{
	filetypes_filetypeentry *entry;
	
	entry=(filetypes_filetypeentry*)data;

	/* take care with that default track! */
	if (strcmp(entry->suffix,"default"))
	  {		  
		  strcpy(entry->suffix,
			 gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(entry->w_suffix)->entry)));
	  }
	;
        strcpy(entry->tracktype,
	       gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(entry->w_tracktype)->entry)));          
        entry->precache=GTK_TOGGLE_BUTTON(entry->w_precache)->active;	   
	strcpy(entry->filter,
	       gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(entry->w_filter)->entry)));
	strcpy(entry->filtername,
	       gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(entry->w_filtername)->entry)));
	strcpy(entry->detectsize,
	       gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(entry->w_detectsize)->entry)));
	strcpy(entry->sizealg,
	       gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(entry->w_sizealg)->entry)));
}
;

void filetypes_undo(GtkWidget *w,gpointer data)
{
	filetypes_filetypeentry *entry;
	
	entry=(filetypes_filetypeentry*)data;

	/* take care with that default track! */
	if (strcmp(entry->suffix,"default"))
	  {		  
		  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_suffix)->entry),
				     entry->suffix);
	  }
	;
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_tracktype)->entry),
			   entry->tracktype);
   	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(entry->w_precache),
				     entry->precache);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_filter)->entry),
			   entry->filter);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_filtername)->entry),
			   entry->filtername);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_detectsize)->entry),
			   entry->detectsize);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_sizealg)->entry),
			   entry->sizealg);
}
;


/* the only handler installed here is filetypes_applyencsettings.
 * It is installed as long as the encoder settings dialog is visible 
 * on the screen and uninstalled once it has disappeared.
 * The data argument is the filteypes_filetypeentry currently edited */
GList *filetypes_encsettingschanged=NULL;

void filetypes_applyencsettings(gpointer data)
{
   
   filetypes_filetypeentry *e=(filetypes_filetypeentry*)data;
   /* only call this once */
   updatehandlers_unregister(&filetypes_encsettingschanged,
			     filetypes_applyencsettings,
			     data);
   /* ensure the encoder selected as default actually 
    * becomes the new default */
   if (e->enc_default)
     {	
	GList *current;
	for (current=filetypes_filetypeentries;current!=NULL;current=current->next)
	  if (current->data!=e)
	    ((filetypes_filetypeentry*)(current->data))->enc_default=0;	
     };
};

void filetypes_encodingoptions(GtkWidget *w,gpointer data)
{
   char buf[255];
   filetypes_filetypeentry *entry=(filetypes_filetypeentry*)data;
   sprintf(buf,_("Default for Tracktype %s"),
	   entry->tracktype);
   forms_create(_("Encoder settings"),
		7,
		&filetypes_encsettingschanged,
		FORMS_ENTRY,
		_("Encoder call"),
		entry->encoder,
		FORMS_NEWLINE,
		FORMS_CHECKBOX,
		_("Encoder needs little Endian"),
		&entry->enc_lendian,
		FORMS_NEWLINE,
		FORMS_CHECKBOX,
		buf,
		&entry->enc_default,
		FORMS_NEWLINE,
		FORMS_ENTRY,
		_("Set title info with"),
		entry->settitle
		);		
   updatehandlers_register(&filetypes_encsettingschanged,
			   filetypes_applyencsettings,
			   data);
};
		
void filetypes_remove(GtkWidget *w,gpointer data)
{
	filetypes_unregister((filetypes_filetypeentry*)data);
}
;

void filetypes_register(filetypes_filetypeentry *entry)
{
	GtkWidget *fileselect;
	GtkWidget *removeentry;
        GtkWidget *encoder;
	
	entry->sampler=gtk_hbox_new(0,1);

	/* do not allow changing the nature of a default track type */
	if (strcmp(entry->suffix,"default"))
	  {		  	
		  entry->w_suffix=gtk_combo_new();
		  gtk_combo_set_popdown_strings(GTK_COMBO(entry->w_suffix),
						helpings_buildglist(filetypes_suffixexamples));
		  gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_suffix)->entry),
				     entry->suffix);
	     	  gtk_entry_set_position(GTK_ENTRY(GTK_COMBO(entry->w_suffix)->entry),
					0);
	  }
	else
	  {
		  entry->w_suffix=gtk_label_new(_("default"));
	  }
	;		  
	gtk_widget_show(entry->w_suffix);
	gtk_widget_set_usize(entry->w_suffix,suffix_size,entryheight);
	gtk_box_pack_start(GTK_BOX(entry->sampler),entry->w_suffix,0,0,0);
		
	entry->w_tracktype=gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(entry->w_tracktype),
				      helpings_buildglist(tracks_tracktypedefinitions));
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_tracktype)->entry),
			   entry->tracktype);
	gtk_entry_set_position(GTK_ENTRY(GTK_COMBO(entry->w_tracktype)->entry),
			       0);  
	gtk_widget_show(entry->w_tracktype);
	gtk_widget_set_usize(entry->w_tracktype,tracktype_size,entryheight);
	gtk_box_pack_start(GTK_BOX(entry->sampler),entry->w_tracktype,0,0,0);

   	entry->w_precache=gtk_check_button_new();
   	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(entry->w_precache),
				     entry->precache);
   	gtk_widget_show(entry->w_precache);
   	gtk_widget_set_usize(entry->w_precache,precache_size,entryheight);
   	gtk_box_pack_start(GTK_BOX(entry->sampler),entry->w_precache,0,0,0);
   	main_settip(entry->w_precache,_("precache file before writing"));
   
	entry->w_filter=gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(entry->w_filter),
				      helpings_buildglist(filetypes_filterfunctions));
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_filter)->entry),
			   entry->filter);
  	gtk_entry_set_position(GTK_ENTRY(GTK_COMBO(entry->w_filter)->entry),
			       0);
	gtk_widget_show(entry->w_filter);
	gtk_widget_set_usize(entry->w_filter,filter_size,entryheight);
	gtk_box_pack_start(GTK_BOX(entry->sampler),entry->w_filter,0,0,0);

	entry->w_filtername=gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(entry->w_filtername),
				      helpings_buildglist(filetypes_filterexamples));
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_filtername)->entry),
			   entry->filtername);
   	gtk_entry_set_position(GTK_ENTRY(GTK_COMBO(entry->w_filtername)->entry),
			       0);  
	gtk_widget_show(entry->w_filtername);
	gtk_widget_set_usize(entry->w_filtername,filtername_size,entryheight);
	gtk_box_pack_start(GTK_BOX(entry->sampler),entry->w_filtername,0,0,0);
	
	fileselect=gtk_button_new_with_label(_("S"));
	gtk_signal_connect(GTK_OBJECT(fileselect),"clicked",
			   GTK_SIGNAL_FUNC(filetypes_fileselect),(gpointer)entry->w_filtername);
	gtk_widget_show(fileselect);	
	gtk_box_pack_start(GTK_BOX(entry->sampler),fileselect,0,0,0);
   	main_settip(fileselect,_("Select program"));
		
	entry->w_detectsize=gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(entry->w_detectsize),
				      helpings_buildglist(filetypes_detectsizeexamples));
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_detectsize)->entry),
			   entry->detectsize);
   	gtk_entry_set_position(GTK_ENTRY(GTK_COMBO(entry->w_detectsize)->entry),
			       0);  
	gtk_widget_show(entry->w_detectsize);
	gtk_widget_set_usize(entry->w_detectsize,detectsize_size,entryheight);
	gtk_box_pack_start(GTK_BOX(entry->sampler),entry->w_detectsize,0,0,0);

	fileselect=gtk_button_new_with_label(_("S"));
	gtk_signal_connect(GTK_OBJECT(fileselect),"clicked",
			   GTK_SIGNAL_FUNC(filetypes_fileselect),(gpointer)entry->w_detectsize);
	gtk_widget_show(fileselect);		
	gtk_box_pack_start(GTK_BOX(entry->sampler),fileselect,0,0,0);
   	main_settip(fileselect,_("Select program"));
	
	entry->w_sizealg=gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(entry->w_sizealg),
				      helpings_buildglist(filetypes_sizealgexamples));
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(entry->w_sizealg)->entry),
			   entry->sizealg);
   	gtk_entry_set_position(GTK_ENTRY(GTK_COMBO(entry->w_sizealg)->entry),
			       0);  
	gtk_widget_show(entry->w_sizealg);
	gtk_widget_set_usize(entry->w_sizealg,sizealg_size,entryheight);
	gtk_box_pack_start(GTK_BOX(entry->sampler),entry->w_sizealg,0,0,0);
   
        encoder=gtk_button_new_with_label(_("Encoding"));
        gtk_signal_connect(GTK_OBJECT(encoder),"clicked",
			   GTK_SIGNAL_FUNC(filetypes_encodingoptions),
			   (gpointer)entry);
	gtk_widget_show(encoder);
	gtk_box_pack_start(GTK_BOX(entry->sampler),encoder,0,0,0);	    
   
	/* do not allow removing the default filetype */
	if (strcmp(entry->suffix,"default"))
	  {		     
		  removeentry=gtk_button_new_with_label(_("Remove"));
		  gtk_signal_connect(GTK_OBJECT(removeentry),"clicked",
				     GTK_SIGNAL_FUNC(filetypes_remove),(gpointer)entry);
	  }    
	else
	  {
		  removeentry=gtk_label_new(_("default"));
	  }
	;
	gtk_widget_show(removeentry);
	gtk_box_pack_start(GTK_BOX(entry->sampler),removeentry,0,0,0);	    
	
	gtk_box_pack_start(GTK_BOX(filetypes_typearea),
		       	   entry->sampler,0,0,0);
	gtk_widget_show(entry->sampler);		
	preferences_setapplyhandler(GTK_SIGNAL_FUNC(filetypes_apply),(gpointer)entry);
	preferences_setundohandler(GTK_SIGNAL_FUNC(filetypes_undo),(gpointer)entry);	
	
	/* if the entry to be registered is the "default" entry,
	 * it gets prepended to the list so that it appears as its first
	 * entry. This is just a layout thing,nothing more... */
	if (strcmp(entry->suffix,"default"))
	    filetypes_filetypeentries=g_list_append(filetypes_filetypeentries,
						    entry);
	else 
	    filetypes_filetypeentries=g_list_prepend(filetypes_filetypeentries,
						     entry);	    
}
;

/* step through Glist and detect matches to the given suffix */
filetypes_filetypeentry *filetypes_getfiletypehandler(char *suffix)
{
	GList *current;
	filetypes_filetypeentry *entry;
	
	entry=NULL;
	current=filetypes_filetypeentries;
	while ((current!=NULL)&&(entry==NULL))
	  {
	     if (helpings_listcomp(((filetypes_filetypeentry*)current->data)->suffix,
				   suffix,
				   " "))
	       entry=(filetypes_filetypeentry*)(current->data);
	     current=current->next;
	  }
	;
	/* use default handler if none available. It must be guaranteed
	 * that this
	 * default filetype is always present. Its settings can be changed
	 * by the user though.
	 * the strcmp prevents this routine from ending up in a recursion
	 * loop, so it can now be used by filetypes_init() to check wether
	 * a default handler could be retrieved from the config file */
	if ((entry==NULL) && (strcmp(suffix,"default")))
	    entry=filetypes_getfiletypehandler("default");
	return entry;		    	
}
;

/* this gets the default encoder for the specified tracktype.
 * The rules are as follows:
 * if no default is specified the first encoder capable of encoding
 * our stream will be taken.
 * if one or more defaults are set, the last default is chosen */
filetypes_filetypeentry *filetypes_getdefaultencoder(char *tracktype)
{
   GList *current;
   filetypes_filetypeentry *entry;
	
   entry=NULL;
   current=filetypes_filetypeentries;
   while (current!=NULL)
     {
	if (!strcasecmp(((filetypes_filetypeentry*)current->data)->tracktype,
			tracktype))
	  if ((entry==NULL)||
	      (((filetypes_filetypeentry*)current->data)->enc_default))
	    entry=(filetypes_filetypeentry*)(current->data);
	current=current->next;
     }
   ;
   return entry;
};
   
int filetypes_getenccount(char *tracktype)
{
   int result=0;
   GList *current;
   
   for (current=filetypes_filetypeentries;
	current!=NULL;
	current=current->next)
     {
	if ((current->data!=NULL)
	    &&(!strcasecmp(((filetypes_filetypeentry*)current->data)->tracktype,
			   tracktype)
	       ))
	  result++;
     };   
   return result;
};

GList *filetypes_getencoders(char *tracktype)
{
   GList *result=NULL;
   GList *current;
   
   for (current=filetypes_filetypeentries;
	current!=NULL;
	current=current->next)
     {
	if ((current->data!=NULL)
	    &&(!strcasecmp(((filetypes_filetypeentry*)current->data)->tracktype,
			   tracktype)
	       ))
	  result=g_list_append(result,current->data);
     };
   return result;
};

/* Maplist functions */

typedef struct  
{     
   char *tracktype;    
   char *suffix;     
} filetypes_encmap;

filetypes_encmap *filetypes_getencmap(GList *maplist,
				      char *tracktype)
{
   GList *current;
   filetypes_encmap *result=NULL;
   if (maplist)
     for (current=maplist;
	  current!=NULL;
	  current=current->next)
       if (!strcasecmp(((filetypes_encmap*)current->data)->tracktype,
		       tracktype))
	 result=(filetypes_encmap*)current->data;
   return result;
};

void filetypes_removemapping(GList **maplist,
			     char *tracktype)
{
   filetypes_encmap *r=filetypes_getencmap(*maplist,tracktype);
   if (r!=NULL)
     *maplist=g_list_remove(*maplist,
			    (gpointer)r);
};
   

char *filetypes_getmapping(GList *maplist,
			   char *tracktype)
{
   filetypes_encmap *r=filetypes_getencmap(maplist,tracktype);
   if (r!=NULL)
     return r->suffix;
   else
     return NULL;
};

void filetypes_addmapping(GList **maplist,
			  char *tracktype,
			  char *suffix)
{
   filetypes_encmap *newmap;
     
   /* ensure there's only one mapping per tracktype */
   if (filetypes_getmapping(*maplist,
			    tracktype)!=NULL)
     filetypes_removemapping(maplist,
			     tracktype);
   newmap=(filetypes_encmap*)malloc(sizeof(filetypes_encmap));
   newmap->tracktype=strdup(tracktype);
   newmap->suffix=strdup(suffix);
   *maplist=g_list_append(*maplist,
			  (gpointer)newmap);
};

void filetypes_clearmapping(GList **maplist)
{
   while (*maplist!=NULL)
     {
	filetypes_encmap *e=(filetypes_encmap*)((*maplist)->data);
	free(e->tracktype);
	free(e->suffix);	
	free((*maplist)->data);
	*maplist=g_list_remove(*maplist,(*maplist)->data);
     };
};
          
/* destroy a filetype registry entry including its visuals */
void filetypes_unregister(filetypes_filetypeentry*entry)
{
   #ifdef DEBUG
     	printf ("filetypes_unregister: removing entry for %s\n",
		entry->suffix);
   #endif
	filetypes_filetypeentries=g_list_remove(filetypes_filetypeentries,
						(gpointer)entry);
 	preferences_removeapplyhandler_by_data((gpointer)entry);
	preferences_removeundohandler_by_data((gpointer)entry); 
	gtk_widget_destroy(entry->sampler);
}
;

/* update a filetype entry externally */
void filetypes_modify(filetypes_filetypeentry *entry,
		      char *suffix,     /* NULL == don't modify */
		      char *tracktype,
		      int  precache,    /* -1   == don't modify */
		      char *filter,
		      char *filtername,
		      char *detectsize,
		      char *sizealg,
		      
		      char *encoder,
		      int  enc_lendian,
		      int  enc_default,
		      char *settitle
		      )
{
   /* unregister filetype entry */
   filetypes_unregister(entry);
   /* modify filetype entry according to our parameters */
   if ((suffix!=NULL)&&
       /* don't make default registry entry non-default */
       (strcasecmp(entry->suffix,"default")))
     strcpy(entry->suffix,suffix);
   if (tracktype!=NULL)
     strcpy(entry->tracktype,tracktype);
   if (precache!=-1)
     entry->precache=precache;
   if (filter!=NULL)
     strcpy(entry->filter,filter);
   if (filtername!=NULL)
     strcpy(entry->filtername,filtername);
   if (detectsize!=NULL)
     strcpy(entry->detectsize,detectsize);
   if (sizealg!=NULL)
     strcpy(entry->sizealg,sizealg);
   if (encoder!=NULL)
     strcpy(entry->encoder,encoder);
   if (enc_lendian!=-1)
     entry->enc_lendian=enc_lendian;
   if (enc_default!=-1)
     entry->enc_default=enc_default;
   if (settitle!=NULL)
     strcpy(entry->settitle,settitle);
   
   /* re-register filetype entry */
   filetypes_register(entry);
};

/* step through Glist and store all the filetype definitions */
void filetypes_storefiletypes(FILE *fd,gpointer data)
{
	GList *current;
	filetypes_filetypeentry *entry;
	
	current=filetypes_filetypeentries;
	while (current!=NULL)
	  {
		  entry=(filetypes_filetypeentry*)current->data;
		  fprintf(fd,"suffix=%s\n",entry->suffix);
		  fprintf(fd,"tracktype=%s\n",entry->tracktype);
	     	  fprintf(fd,"precache=%i\n",entry->precache);
		  fprintf(fd,"filter=%s\n",entry->filter);
		  fprintf(fd,"filtername=%s\n",entry->filtername);
		  fprintf(fd,"detectsize=%s\n",entry->detectsize);
	          fprintf(fd,"encoder=%s\n",entry->encoder);
	          fprintf(fd,"enc_lendian=%i\n",entry->enc_lendian);
	          fprintf(fd,"enc_default=%i\n",entry->enc_default);
	     	  fprintf(fd,"settitle=%s\n",entry->settitle);
		  fprintf(fd,"sizealg=%s\n",entry->sizealg);
		  current=current->next;
	  }
	;
}
;

/* call this handler if filetype registry couldnt be retrieved from disk
 * for some reason */
void filetypes_defaulttypes()
{	

	/* define a few standard types here.
	 * its up to the stdfiletrack routines once again to react on those
	 * settings appropriately */
	filetypes_register(filetypes_entrycreate(".mp3",
						 "audio",
						 0,
						 /* No, I don't like this
						  * either, but mpg123
						  * seems to send it's output
						  * in little/big Endian
						  * depending on the 
						  * platform it is running
						  * on. We can't take
						  * care of this at runtime
						  * so at least we have 
						  * to make sure that
						  * we're starting
						  * with a correct default */
						 ((preferences_bigendian)
						  ?DEFAULTENTRIES_BIGENDIAN
						  :DEFAULTENTRIES_LITTLEENDIAN),
						 DEFAULTENTRIES_EXEC_MPG123,
						 "",
						 DEFAULTENTRIES_SIZEALG_MP3,

						 DEFAULTENTRIES_ENCODER_MP3,
						 DEFAULTENTRIES_ENC_LENDIAN_MP3,
						 DEFAULTENTRIES_ENC_DEFAULT_MP3,
						 DEFAULTENTRIES_SETTITLE_MP3
						 ));

	filetypes_register(filetypes_entrycreate(".wav",
						 "audio",
						 0,
						 ((preferences_bigendian)
						  ?DEFAULTENTRIES_LITTLEENDIAN
						  :DEFAULTENTRIES_BIGENDIAN),
						 DEFAULTENTRIES_EXEC_SOXWAV,
						 "",
						 DEFAULTENTRIES_SIZEALG_WAV,
						 
						 DEFAULTENTRIES_ENCODER_WAV,
						 DEFAULTENTRIES_ENC_LENDIAN_WAV,
						 DEFAULTENTRIES_ENC_DEFAULT_WAV,
						 ""
						 ));
   
	filetypes_register(filetypes_entrycreate(".shn",
						 "audio",
						 0,
						 ((preferences_bigendian)
						  ?DEFAULTENTRIES_LITTLEENDIAN
						  :DEFAULTENTRIES_BIGENDIAN),
						 DEFAULTENTRIES_EXEC_SHORTEN,
						 DEFAULTENTRIES_DETSIZE_SHN,
						 DEFAULTENTRIES_SIZEALG_SHN,
						 
						 "",
						 0,
						 DEFAULTENTRIES_ENC_DEFAULT_SHN,
						 ""
						 ));
   
	filetypes_register(filetypes_entrycreate(".ogg",
						 "audio",
						 0,
						 ((preferences_bigendian)
						  ?DEFAULTENTRIES_BIGENDIAN
						  :DEFAULTENTRIES_LITTLEENDIAN),
						 DEFAULTENTRIES_EXEC_OGG123,
						 DEFAULTENTRIES_DETSIZE_OGG,
						 DEFAULTENTRIES_SIZEALG_OGG,
						 
						 DEFAULTENTRIES_ENCODER_OGG,
						 DEFAULTENTRIES_ENC_LENDIAN_OGG,
						 DEFAULTENTRIES_ENC_DEFAULT_OGG,
						 ""
						 ));   
}
;

void filetypes_loadregistry(char *filename,char *section)
{
	FILE *f;
	char name[256];
   	char value[256]; /* set real vars depending on name 
			  * to make the config file entries extensible */
   	int  nextent;    /* this var is set true to indicate that one registry
			  * entry has been read completely and that we're now
			  * ready to proceed to the next one */
   	int  endoflist=0;
	filetypes_filetypeentry *e;
	
	f=fopen(filename,"r");
	if (f!=NULL)
	  {
		  if (configfile_seeksection(f,section))
		    {
		       do 
			 {
			    e=filetypes_entrycreate("",
						    "",
						    0,
						    "",
						    "",
						    "",
						    "",
						    "",
						    0,
						    0,
						    ""
						    );
			    nextent=0;
			    do 
			      {
				 configfile_getnextentry(f,name,value);
				 if (!strcasecmp(name,"suffix"))
				   strcpy(e->suffix,value);
				 if (!strcasecmp(name,"tracktype"))
				   strcpy(e->tracktype,value);
				 if (!strcasecmp(name,"precache"))
				   sscanf(value,"%d",&e->precache);
				 if (!strcasecmp(name,"filter"))
				   strcpy(e->filter,value);
				 if (!strcasecmp(name,"filtername"))
				   strcpy(e->filtername,value);
				 if (!strcasecmp(name,"detectsize"))
				   strcpy(e->detectsize,value);
				 if (!strcasecmp(name,"encoder"))
				   strcpy(e->encoder,value);
				 if (!strcasecmp(name,"enc_lendian"))
				   sscanf(value,"%d",&e->enc_lendian);
				 if (!strcasecmp(name,"enc_default"))
				   sscanf(value,"%d",&e->enc_default);
				 if (!strcasecmp(name,"settitle"))
				   strcpy(e->settitle,value);
				 if (!strcasecmp(name,"sizealg"))
				   {				      
				      strcpy(e->sizealg,value);
				      nextent=1;
				      /* this entry specifies the end of the list */
				   };
			      }
			    while ((!nextent)&&(strlen(name)>0));
			    if (strlen(name)>0)
			      filetypes_register(e);
			    else
			      {
				 free(e);
				 endoflist=1;
			      };
			 } 
		       while (!endoflist);
		    }
		  else 
		      filetypes_defaulttypes();
	     fclose(f);
	  }
	else
	    filetypes_defaulttypes();
	/* check if default handler was installed correctly */
	if (filetypes_getfiletypehandler("default")==NULL)
	  {		  
		  /* this tracktype is the default returned by the find functions
		   * if the suffix couldnt be determined.
		   * It cant be deleted by the user (delete Button is disconnected) */	
		  filetypes_register(filetypes_entrycreate("default",
							   "data",
							   0,
							   "",
							   "",
							   "",
							   "$filesize",
							   "",
							   0,
							   0,
							   ""
							   ));
	  }
	;
	
}
;			    

void filetypes_add(GtkWidget *w,gpointer data)
{
	filetypes_register(filetypes_entrycreate("","audio",0,"","","","$filesize","",0,0,""));
}
;

void filetypes_init()
{
	GtkWidget *prefs;
	GtkWidget *label;
	GtkWidget *labelling;
	GtkWidget *scroll;
	GtkWidget *addbutton;
	
	/* FIXME: load filetype definitions here */
	
	prefs=gtk_vbox_new(0,0);
	gtk_widget_show(prefs);
	label=gtk_label_new(_("Filetypes"));
	gtk_widget_show(label);
   	preferences_append_page(prefs,label);
	
	labelling=gtk_hbox_new(0,0);
	gtk_box_pack_start(GTK_BOX(prefs),labelling,0,0,0);
	gtk_widget_show(labelling);
	
	label=gtk_label_new(_("Suffix"));
	gtk_widget_set_usize(label,suffix_size,entryheight);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(labelling),
			   label,0,0,0);	

	label=gtk_label_new(_("Tracktype"));
	gtk_widget_set_usize(label,tracktype_size,entryheight);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(labelling),
			   label,0,0,0);	

	label=gtk_label_new(_("PC"));
	gtk_widget_set_usize(label,precache_size,entryheight);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(labelling),
			   label,0,0,0);
   
	label=gtk_label_new(_("Format"));
	gtk_widget_set_usize(label,filter_size,entryheight);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(labelling),
			   label,0,0,0);	

	label=gtk_label_new(_("Filtername"));
	gtk_widget_set_usize(label,filtername_size,entryheight);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(labelling),
			   label,0,0,0);	

	label=gtk_label_new(_(" "));
	gtk_widget_set_usize(label,selectbutton_size,entryheight);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(labelling),
			   label,0,0,0);	
      
	label=gtk_label_new(_("Detect Size"));
	gtk_widget_set_usize(label,detectsize_size,entryheight);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(labelling),
			   label,0,0,0);

	label=gtk_label_new(_(" "));
	gtk_widget_set_usize(label,selectbutton_size,entryheight);
	gtk_widget_show(label);
	gtk_box_pack_start(GTK_BOX(labelling),
			   label,0,0,0);	
   
	addbutton=gtk_button_new_with_label(_("Add Filetype"));
	gtk_signal_connect(GTK_OBJECT(addbutton),"clicked",
			   filetypes_add,NULL);
	gtk_widget_show(addbutton);
	gtk_box_pack_start(GTK_BOX(labelling),
			   addbutton,0,0,0);

	scroll=gtk_scrolled_window_new(NULL,NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(prefs),scroll,1,1,1);
	gtk_widget_show(scroll);
	
	filetypes_typearea=gtk_vbox_new(0,0);
	gtk_widget_show(filetypes_typearea);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll),
					      filetypes_typearea);

	filetypes_loadregistry(varman_replacevars_copy(dynamic_defs,
						       "$HOME/.gtoasterrc"),			       
			       "[filetypes]");
	
	/* register storage handler for filetype registry */
	configfile_registersection("[filetypes]",
				   (configfile_store)filetypes_storefiletypes,
				   NULL);
   
}
;
	
	
	
    
