/* Copyright (C) 1998-1999 Adrian E. Feiguin <adrian@ifir.ifir.edu.ar>
 * Copyright (C) 1999-2000 Matias Mutchinick
 *
 * The GtkSheet widget was written by Adrian Feiguin.
 * The GtkMatrix widget, a modification of GtkSheet was
 * written by Matias Mutchinick
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef __GTK_MATRIX_H__
#define __GTK_MATRIX_H__


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef enum {
	GTK_MATRIX_IS_EDITABLE,
	GTK_MATRIX_IS_VISIBLE
} GtkMatrixAttrType;
	

	
enum
{
	GTK_MATRIX_LEFT_BORDER     = 1 << 0, 
	GTK_MATRIX_RIGHT_BORDER    = 1 << 1, 
	GTK_MATRIX_TOP_BORDER      = 1 << 2, 
	GTK_MATRIX_BOTTOM_BORDER   = 1 << 3, 
}; 



/* matrix flags */
enum                    
{
	GTK_MATRIX_IS_LOCKED           = 1 << 0,
	GTK_MATRIX_IS_FROZEN           = 1 << 1,                                     
	GTK_MATRIX_IN_XDRAG            = 1 << 2,                                        
	GTK_MATRIX_IN_YDRAG            = 1 << 3,                                        
	GTK_MATRIX_IN_SELECTION        = 1 << 4,
	GTK_MATRIX_ROW_FROZEN          = 1 << 5,  /* set rows to be resizeable */
	GTK_MATRIX_COLUMN_FROZEN       = 1 << 6,  /* set cols to be resizeable */
	GTK_MATRIX_AUTORESIZE          = 1 << 7,  /* resize column if text width is 
						    great than column width */
	GTK_MATRIX_ROW_TITLES_VISIBLE  = 1 << 8,
	GTK_MATRIX_COL_TITLES_VISIBLE  = 1 << 9
}; 


#define GTK_MATRIX(obj)                     GTK_CHECK_CAST(obj,\
                                           gtk_matrix_get_type(),GtkMatrix)

#define GTK_MATRIX_CLASS(klass)             GTK_CHECK_CLASS_CAST(klass,\
                                           gtk_matrix_get_type(),GtkMatrixClass)

#define GTK_IS_MATRIX(obj)                  GTK_CHECK_TYPE(obj,gtk_matrix_get_type())

#define GTK_MATRIX_FLAGS(matrix)             (GTK_MATRIX (matrix)->flags)
#define GTK_MATRIX_SET_FLAGS(matrix,flag)    (GTK_MATRIX_FLAGS (matrix) |= (flag))
#define GTK_MATRIX_UNSET_FLAGS(matrix,flag)  (GTK_MATRIX_FLAGS (matrix) &= ~(flag))

#define GTK_MATRIX_IS_LOCKED(matrix)         (GTK_MATRIX_FLAGS (matrix) & GTK_MATRIX_IS_LOCKED)
#define GTK_MATRIX_IS_FROZEN(matrix)         (GTK_MATRIX_FLAGS (matrix) & GTK_MATRIX_IS_FROZEN)
#define GTK_MATRIX_IN_XDRAG(matrix)          (GTK_MATRIX_FLAGS (matrix) & GTK_MATRIX_IN_XDRAG)
#define GTK_MATRIX_IN_YDRAG(matrix)          (GTK_MATRIX_FLAGS (matrix) & GTK_MATRIX_IN_YDRAG)
#define GTK_MATRIX_IN_SELECTION(matrix)      (GTK_MATRIX_FLAGS (matrix) & GTK_MATRIX_IN_SELECTION)
#define GTK_MATRIX_ROW_FROZEN(matrix)        (GTK_MATRIX_FLAGS (matrix) & GTK_MATRIX_ROW_FROZEN)
#define GTK_MATRIX_COLUMN_FROZEN(matrix)     (GTK_MATRIX_FLAGS (matrix) & GTK_MATRIX_COLUMN_FROZEN)
#define GTK_MATRIX_AUTORESIZE(matrix)        (GTK_MATRIX_FLAGS (matrix) & GTK_MATRIX_AUTORESIZE)
#define GTK_MATRIX_ROW_TITLES_VISIBLE(matrix)(GTK_MATRIX_FLAGS (matrix) & GTK_MATRIX_ROW_TITLES_VISIBLE)
#define GTK_MATRIX_COL_TITLES_VISIBLE(matrix)(GTK_MATRIX_FLAGS (matrix) & GTK_MATRIX_COL_TITLES_VISIBLE)


typedef struct _GtkMatrix           GtkMatrix;
typedef struct _GtkMatrixClass      GtkMatrixClass;
typedef struct _GtkMatrixChild      GtkMatrixChild;
typedef struct _GtkMatrixRow        GtkMatrixRow;
typedef struct _GtkMatrixColumn     GtkMatrixColumn;
typedef struct _GtkMatrixCell       GtkMatrixCell;
typedef struct _GtkMatrixRange      GtkMatrixRange;
typedef struct _GtkMatrixButton     GtkMatrixButton;
typedef struct _GtkMatrixCellAttr   GtkMatrixCellAttr;
typedef struct _GtkMatrixAttr       GtkMatrixAttr;


struct _GtkMatrixChild
{
	GtkWidget  *widget;
	GdkWindow  *window;
	gint       x;
	gint       y;
	gboolean   attached_to_cell;
	gint       row;
	gint       col;
	gfloat     x_align;
	gfloat     y_align;
};



struct _GtkMatrixButton
{
	gint   state;
	gchar *label;
};


struct _GtkMatrixCellAttr
{
	gint is_editable: 1;
	gint is_visible : 1;
};



struct _GtkMatrixCell
{
	GdkRectangle      area;
	gint              row;
	gint              col;
	GtkMatrixCellAttr  attributes;
	gchar            *text;
	gpointer          link;
};


struct _GtkMatrixRange
{
	gint row0;
	gint col0; 
	gint rowi;
	gint coli;
};


struct _GtkMatrixRow
{
	gchar            *name;
	gint              height;
	gint              top_ypixel;
	
	GtkMatrixButton    button;
	
	gint              is_sensitive :1;
	gint              is_visible   :1;
};



struct _GtkMatrixColumn
{
	gchar            *name;
	gint              width;
	gint              left_xpixel;
	
	GtkMatrixButton    button;
	
	gint              left_text_column;  /* min left column displaying 
						text on this column */
	gint              right_text_column; /* max right column displaying 
						text on this column */
	
	gint              is_sensitive:1;
	gint              is_visible:1;
};


struct _GtkMatrix{

	GtkContainer container;
	
	guint16 flags;
	
	/* matrix children */
	GList *children;
	
	/* allocation rectangle after the container_border_width
	   and the width of the shadow border */
	GdkRectangle internal_allocation;
	
	GtkMatrixRow *row;
	GtkMatrixColumn *column;
	
	/* max number of diplayed cells */
	gint maxrow;
	gint maxcol;
	
	/* Displayed range */
	
	GtkMatrixRange view; 
	
	/* matrix data: dynamically allocated array of cell pointers */
	GtkMatrixCell ***data;
	
	/* max number of allocated cells */
	gint maxallocrow;
	gint maxalloccol;
	
	/* active cell */
	GtkMatrixCell active_cell;
	GtkWidget *matrix_entry;
	GdkWindow *matrix_entry_window; /* for NO_WINDOW entry widgets(ala GtkLayout) */
	
	GtkType entry_type;
	
	/* timer for automatic scroll during selection */  
	gint32 timer;
	
		
	/* global selection button */
	GtkWidget *button;
	
	
	/* matrix attributes */
	GtkMatrixAttr *attributes;
	gint maxrange;
	
	/*the scrolling window and it's height and width to
	 * make things a little speedier */
	GdkWindow *matrix_window;
	gint matrix_window_width;
	gint matrix_window_height;
	
	/* matrix backing pixmap */  
	GdkWindow *pixmap;    
	
	/* offsets for scrolling */
	gint hoffset;
	gint voffset;
	gfloat old_hadjustment;
	gfloat old_vadjustment;
  
	/* border shadow style */
	GtkShadowType shadow_type;
	
	/* Between the row and col titles */
	GdkRectangle  pivot_area;
	GdkWindow    *pivot_window;
	
	/* Column Titles */
	GdkRectangle  column_title_area;
	GdkWindow    *column_title_window;
	
	/* Row Titles */
	GdkRectangle  row_title_area;
	GdkWindow    *row_title_window;
	
	/*scrollbars*/
	GtkAdjustment *hadjustment;
	GtkAdjustment *vadjustment;
	
	/* xor GC for the verticle drag line */
	GdkGC *xor_gc;
	
	/* gc for drawing unselected cells */
	GdkGC *fg_gc;
	GdkGC *bg_gc;

	
	GdkCursor    *cursor;
	
	/* the current x-pixel location of the xor-drag vline */
	gint x_drag;
	gint col_drag;

	/* the current y-pixel location of the xor-drag hline */
	gint y_drag;
	gint row_drag;
};



struct _GtkMatrixClass
{
	GtkContainerClass parent_class;
	
	void (*set_scroll_adjustments) (GtkMatrix      *matrix,
					GtkAdjustment *hadjustment,
					GtkAdjustment *vadjustment);
	
	gint (*traverse)       	        (GtkMatrix *matrix,
					 gint      row, 
					 gint      column,
					 gint     *new_row, 
					 gint     *new_column);
	
	
	gint (*deactivate)	 	(GtkMatrix *matrix,
					 gint      row, 
					 gint      column);
	
	gint (*activate) 		(GtkMatrix *matrix,
					 gint      row,
					 gint      column);
	
	void (*set_cell) 		(GtkMatrix *matrix,
					 gint      row, 
					 gint      column);
	
	void (*clear_cell) 		(GtkMatrix *matrix,
					 gint      row, 
					 gint      column);
	
	void (*entry_changed) 		(GtkMatrix *matrix,
					 gint      row, 
					 gint      column);
	
	void (*new_column_width)       (GtkMatrix *matrix,
					gint      col,
					gint      width);
	
	void (*new_row_height)       	(GtkMatrix *matrix,
					 gint      row,
					 gint      height);

	void (*row_button_press)       	(GtkMatrix *matrix,
					 gint       row,
					 gint       col);
	
};
  
guint gtk_matrix_get_type (void);

/* create a new matrix */
GtkWidget *
gtk_matrix_new 				(gint   rows,
					 gint   columns);


/* create a new browser matrix. It cells can not be edited */
GtkWidget *
gtk_matrix_new_browser 			(gint    rows,
					 gint    columns);


/* create a new matrix with custom entry */
GtkWidget *
gtk_matrix_new_with_custom_entry 	(gint       rows, 
					 gint       columns, 
					 GtkWidget *entry);
/* change scroll adjustments */
void
gtk_matrix_set_hadjustment		(GtkMatrix *matrix,
					 GtkAdjustment *adjustment); 
void
gtk_matrix_set_vadjustment		(GtkMatrix *matrix,
					 GtkAdjustment *adjustment); 

/* Change entry */
void
gtk_matrix_change_entry			(GtkMatrix *matrix, GtkWidget *widget);


/* Returns matrix's entry widget */
GtkWidget *
gtk_matrix_get_entry			(GtkMatrix *matrix);

/* Returns matrix->state 
 * Added by Steven Rostedt <steven.rostedt@lmco.com> */
gint
gtk_matrix_get_state 			(GtkMatrix *matrix);
void
gtk_matrix_get_visible_range		(GtkMatrix *matrix,
					 GtkMatrixRange *range);


/* freeze all visual updates of the matrix.
 * Then thaw the matrix after you have made a number of changes.
 * The updates will occure in a more efficent way than if
 * you made them on a unfrozen matrix */
void
gtk_matrix_freeze			(GtkMatrix *matrix);
void
gtk_matrix_thaw				(GtkMatrix *matrix);



/* set column title */ 
void
gtk_matrix_set_column_title 		(GtkMatrix * matrix,
					 gint column,
					 gchar * title);
/* set row title */
void
gtk_matrix_set_row_title 		(GtkMatrix * matrix,
					 gint row,
					 gchar * title);

/* set button label */
void
gtk_matrix_row_button_add_label		(GtkMatrix *matrix, 
					gint row, char *label);
void
gtk_matrix_column_button_add_label	(GtkMatrix *matrix, 
					gint column, char *label);
void
gtk_matrix_row_button_justify		(GtkMatrix *matrix, 
					gint row, gint justification);
void
gtk_matrix_column_button_justify		(GtkMatrix *matrix, 
					gint column, gint justification);

/* scroll the viewing area of the matrix to the given column
 * and row; row_align and col_align are between 0-1 representing the
 * location the row should appear on the screnn, 0.0 being top or left,
 * 1.0 being bottom or right; if row or column is negative then there
 * is no change */
void
gtk_matrix_moveto (GtkMatrix * matrix,
		  gint row,
		  gint column,
	          gfloat row_align,
                  gfloat col_align);

/* resize column/row titles window */
void 
gtk_matrix_set_row_titles_width(GtkMatrix *matrix, gint width);
void 
gtk_matrix_set_column_titles_height(GtkMatrix *matrix, gint height);

/* show/hide column/row titles window */
void
gtk_matrix_show_column_titles		(GtkMatrix *matrix);
void
gtk_matrix_show_row_titles		(GtkMatrix *matrix);
void
gtk_matrix_hide_column_titles		(GtkMatrix *matrix);
void
gtk_matrix_hide_row_titles		(GtkMatrix *matrix);

/* set column button sensitivity. If sensitivity is TRUE it can be toggled,  
 *  otherwise it acts as a title */
void 
gtk_matrix_column_set_sensitivity	(GtkMatrix *matrix, 
					gint column, gint sensitive);

/* set sensitivity for all column buttons */
void
gtk_matrix_columns_set_sensitivity	(GtkMatrix *matrix, gint sensitive);


/* set row button sensitivity. If sensitivity is TRUE can be toggled, 
 * otherwise it acts as a title */
void 
gtk_matrix_row_set_sensitivity		(GtkMatrix *matrix, 
					gint row,  gint sensitive);

/* set sensitivity for all row buttons */
void
gtk_matrix_rows_set_sensitivity		(GtkMatrix *matrix, gint sensitive);

/* set column visibility. The default value is TRUE. If FALSE, the 
 * column is hidden */
void
gtk_matrix_column_set_visibility		(GtkMatrix *matrix, 
					gint column, gint visible);

/* set row visibility. The default value is TRUE. If FALSE, the 
 * row is hidden */
void
gtk_matrix_row_set_visibility		(GtkMatrix *matrix, 
					 gint row, gint visible);


/* get scrollbars adjustment */
GtkAdjustment *
gtk_matrix_get_vadjustment 		(GtkMatrix * matrix);

GtkAdjustment *
gtk_matrix_get_hadjustment 		(GtkMatrix * matrix);



/* set active cell where the entry will be displayed 
 * returns FALSE if current cell can't be deactivated or
 * requested cell can't be activated */
gint
gtk_matrix_set_active_cell 		(GtkMatrix *matrix, 
					gint row, gint column);
void
gtk_matrix_get_active_cell 		(GtkMatrix *matrix, 
					gint *row, gint *column);

/* set cell contents and allocate memory if needed */
void 
gtk_matrix_set_cell			(GtkMatrix *matrix, 
					int row, int col,
                   			gchar *text);
void 
gtk_matrix_set_cell_text			(GtkMatrix *matrix, 
					int row, int col,
                   			gchar *text);

/* get cell contents */
gchar *     
gtk_matrix_cell_get_text 		(GtkMatrix *matrix, gint row, gint col);


/* clear cell contents */
void 
gtk_matrix_cell_clear			(GtkMatrix *matrix, gint row, gint col);
/* clear cell contents and remove links */
void 
gtk_matrix_cell_delete			(GtkMatrix *matrix, gint row, gint col);

/* clear range contents. If range==NULL the whole matrix will be cleared */
void 
gtk_matrix_range_clear			(GtkMatrix *matrix, GtkMatrixRange *range);
/* clear range contents and remove links */
void 
gtk_matrix_range_delete			(GtkMatrix *matrix, GtkMatrixRange *range);

/* get cell state: GTK_STATE_NORMAL, GTK_STATE_SELECTED */
int
gtk_matrix_cell_get_state 		(GtkMatrix *matrix, int row, int col);

/* Handles cell links */
void
gtk_matrix_link_cell			(GtkMatrix *matrix, int row, int col,
 					 gpointer link);

gpointer
gtk_matrix_get_link			(GtkMatrix *matrix, int row, int col);
void
gtk_matrix_remove_link			(GtkMatrix *matrix, int row, int col);

/* get row and column correspondig to the given position in the screen */
gint
gtk_matrix_get_pixel_info (GtkMatrix * matrix,
			  gint x,
			  gint y,
			  gint * row,
			  gint * column);

/* get area of a given cell */
gint
gtk_matrix_get_cell_area (GtkMatrix *matrix,
                         gint row,
                         gint column,
                         GdkRectangle *area);

/* set column width */
void
gtk_matrix_set_column_width (GtkMatrix * matrix,
			    gint column,
			    gint width);

/* set row height */
void
gtk_matrix_set_row_height (GtkMatrix * matrix,
			  gint row,
			  gint height);

/* append ncols columns to the end of the matrix */
void
gtk_matrix_add_column			(GtkMatrix *matrix, gint ncols);

/* append nrows row to the end of the matrix */
void
gtk_matrix_add_row			(GtkMatrix *matrix, gint nrows);

/* insert nrows rows before the given row and pull right */
void
gtk_matrix_insert_rows			(GtkMatrix *matrix, gint row, gint nrows);

/* insert ncols columns before the given col and pull down */
 void
gtk_matrix_insert_columns		(GtkMatrix *matrix, gint col, gint ncols);

/* delete nrows rows starting in row */
void
gtk_matrix_delete_rows			(GtkMatrix *matrix, gint row, gint nrows);

/* delete ncols columns starting in col */
void
gtk_matrix_delete_columns		(GtkMatrix *matrix, gint col, gint ncols);



GtkMatrixChild *
gtk_matrix_put 				(GtkMatrix *matrix, 
					 GtkWidget *widget, 
					 gint x, gint y);

void
gtk_matrix_attach			(GtkMatrix *matrix,
					 GtkWidget *widget,
					 gint row, gint col,
					 gfloat x_align, gfloat y_align);

void
gtk_matrix_move_child 			(GtkMatrix *matrix, 
					 GtkWidget *widget, 
					 gint x, gint y);

gint
gtk_matrix_optimal_column_width          (GtkMatrix *matrix, gint col);


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __GTK_MATRIX_H__ */


