/*
 * Parser for standard guessnet configuration file
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#define APPNAME PACKAGE
#else
#warning No config.h found: using fallback values
#define APPNAME __FILE__
#define VERSION "unknown"
#endif

#include "GuessnetParser.h"
#include <wibble/regexp.h>
#include "Environment.h"

#include <string>

using namespace std;

/* Parse the input from `input'
 * To make it simple, use regexps on input lines instead of implementing a real
 * parser.
 */
void GuessnetParser::parse(FILE* input, ScanConsumer& sc)
{
#define MACPATTERN "[0-9A-Fa-f]{2}:[0-9A-Fa-f]{2}:[0-9A-Fa-f]{2}:[0-9A-Fa-f]{2}:[0-9A-Fa-f]{2}:[0-9A-Fa-f]{2}"
#define IPPATTERN "[0-9]+\\.[0-9]+\\.[0-9]+\\.[0-9]+"
	wibble::ERegexp null_line("^[[:blank:]]*(#.*)?$");
	wibble::ERegexp peer_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"peer[[:blank:]]+"
		"(" IPPATTERN ")"
		"([[:blank:]]+(" MACPATTERN "))?"
		"([[:blank:]]+(" IPPATTERN "))?"
		"[[:blank:]]*$", 7);
	wibble::ERegexp cable_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"missing-cable[[:blank:]]*$", 2);
	wibble::ERegexp script_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"(script|command)[[:blank:]]+(.+)$", 4);
	wibble::ERegexp dhcp_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"dhcp[[:blank:]]*$", 2);
	wibble::ERegexp pppoe_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"pppoe[[:blank:]]*$", 2);
	wibble::ERegexp wireless_mac_essid_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"wireless[[:blank:]]+mac[[:blank:]]+([^[:blank:]]+)[[:blank:]]+essid[[:blank:]](.+)$", 4);
	wibble::ERegexp wireless_mac_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"wireless[[:blank:]]+mac[[:blank:]]+([^[:blank:]]+)[[:blank:]]*$", 3);
	wibble::ERegexp wireless_essid_line(
		"^[[:blank:]]*"
		"([^[:blank:]]+)[[:blank:]]+"
		"wireless[[:blank:]]+essid[[:blank:]](.+)$", 3);
	wibble::ERegexp old_input_line(
		"^[[:blank:]]*(" IPPATTERN ")[[:blank:]]+"
		"(" MACPATTERN ")"
		"[[:blank:]]+(" IPPATTERN ")[[:blank:]]+([[:alnum:]_+-]+)"
		"[[:blank:]]*$", 4);

	string line;
	int linenum = 1;
	int found = 0;
	int c;
	while ((c = fgetc(input)) != EOF)
	{
		if (c != '\n')
			line += c;
		else
		{
			if (null_line.match(line))
			{
				//fprintf(stderr, "EMPTY\n");
			}
			else if (old_input_line.match(line))
			{	
				string src = old_input_line[1];
				string mac = old_input_line[2];
				string ip = old_input_line[3];
				string name = old_input_line[4];
				struct ether_addr macAddr;
				parse_mac(&macAddr, mac);

				IPAddress ipAddr(ip);

				sc.handleScan(new PeerScan(name, macAddr, ipAddr, src));
				found++;
			}
			else if (peer_line.match(line))
			{
				//fprintf(stderr, "0, %.*s\n", PFSTR(peer_line[0]));
				string name = peer_line[1];
				string ip = peer_line[2];
				string mac = peer_line[4];
				string src = peer_line[6];

				IPAddress ipAddr(ip);

				struct ether_addr macAddr;
				if (mac.empty())
					bzero(&macAddr, sizeof(struct ether_addr));
				else
					parse_mac(&macAddr, mac);

				if (src.empty())
					sc.handleScan(new PeerScan(name, macAddr, ipAddr));
				else
					sc.handleScan(new PeerScan(name, macAddr, ipAddr, IPAddress(src)));

				found++;
			}
			else if (cable_line.match(line))
			{
				string name = cable_line[1];
				//fprintf(stderr, "TEST: %.*s\n", PFSTR(cmd));
				//debug("Will use script %.*s to test %.*s\n",
				//	PFSTR(cmd), PFSTR(name));

				sc.handleScan(new LinkBeatScan(name));
				found++;
			}
			else if (script_line.match(line))
			{
				string name = script_line[1];
				string cmd = script_line[3];
				//fprintf(stderr, "TEST: %.*s\n", PFSTR(cmd));
				//debug("Will use script %.*s to test %.*s\n",
				//	PFSTR(cmd), PFSTR(name));

				sc.handleScan(new ScriptScan(name, cmd));
				found++;
			}
			else if (dhcp_line.match(line))
			{
				string name = dhcp_line[1];
				//fprintf(stderr, "TEST: %.*s\n", PFSTR(cmd));
				//debug("Will use script %.*s to test %.*s\n",
				//	PFSTR(cmd), PFSTR(name));

				sc.handleScan(new DHCPScan(name));
				found++;
			}
			else if (pppoe_line.match(line))
			{
				string name = script_line[1];
				//fprintf(stderr, "TEST: %.*s\n", PFSTR(cmd));
				//debug("Will use script %.*s to test %.*s\n",
				//	PFSTR(cmd), PFSTR(name));

				sc.handleScan(new ScriptScan(name, string("pppoe -I ") + Environment::get().iface() + " -A >/dev/null 2>&1"));
				found++;
			}
			else if (wireless_mac_essid_line.match(line))
			{
				string name = wireless_mac_essid_line[1];
				sc.handleScan(new ScriptScan(name, string(SCRIPTDIR "/test-wireless ") + Environment::get().iface() + " mac " + wireless_mac_essid_line[2] + " essid \"" + wireless_mac_essid_line[3] + "\"" ));
				found++;
			}
			else if (wireless_mac_line.match(line))
			{
				string name = wireless_mac_line[1];
				sc.handleScan(new ScriptScan(name, string(SCRIPTDIR "/test-wireless ") + Environment::get().iface() + " mac " + wireless_mac_line[2] ));
				found++;
			}
			else if (wireless_essid_line.match(line))
			{
				string name = wireless_essid_line[1];
				sc.handleScan(new ScriptScan(name, string(SCRIPTDIR "/test-wireless ") + Environment::get().iface() + " essid \"" + wireless_essid_line[2] + "\"" ));
				found++;
			}
			else
			{
				warning("Parse error at line %d: line ignored\n", linenum);
			}
			line = string();
			linenum++;
		}
	}
	debug("%d candidates found in input\n", found);
}

// vim:set ts=4 sw=4:
