/*
Gwenview - A simple image viewer for KDE
Copyright (C) 2000-2002 Aurlien Gteau

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

// Qt includes
#include <qlabel.h>
#include <qpainter.h>
#include <qevent.h>
#include <qpixmap.h>
#include <qpopupmenu.h>

// KDE includes
#include <kaction.h>
#include <kconfig.h>
#include <kdebug.h>
#include <klocale.h>
#include <kstdaction.h>

// Our includes
#include "gvpixmap.h"

#include "scrollpixmapview.moc"


const char* CONFIG_ZOOM_STEP="zoom step";


ScrollPixmapView::ScrollPixmapView(QWidget* parent,GVPixmap* pixmap,bool enabled)
: QScrollView(parent), mGVPixmap(pixmap), mPopupMenu(0L), mZoom(1), mZoomStep(0.5), mDragStarted(false)
{
	enableView(enabled);

	mAutoZoom=new KToggleAction(i18n("&Auto zoom"),"autozoom",0,this,SLOT(slotAutoZoom()),this);
	mZoomIn=KStdAction::zoomIn(this,SLOT(slotZoomIn()),this);
	mZoomOut=KStdAction::zoomOut(this,SLOT(slotZoomOut()),this);
	mResetZoom=KStdAction::actualSize(this,SLOT(slotResetZoom()),this);
	mResetZoom->setIcon("actualsize");
}


void ScrollPixmapView::plugActionsToAccel(KAccel* accel) {
	mAutoZoom->plugAccel(accel);
	mZoomIn->plugAccel(accel);
	mZoomOut->plugAccel(accel);
	mResetZoom->plugAccel(accel);
}


void ScrollPixmapView::enableView(bool enabled) {
	disconnect(mGVPixmap,0,this,0);

	if (enabled) {
		connect(mGVPixmap,SIGNAL(loaded()),
			this,SLOT(updateView()) );
	}
}


void ScrollPixmapView::updateView() {
	if (mGVPixmap->isNull()) {
		resizeContents(0,0);
		updateZoomActions();
		viewport()->repaint(true);
		return;
	}

	if (mAutoZoom->isChecked()) {
		setZoom(computeAutoZoomValue());
	} else {
		setZoom(1.0);
		horizontalScrollBar()->setValue(0);
		verticalScrollBar()->setValue(0);
	}
}


void ScrollPixmapView::setZoom(double zoom) {
	mZoom=zoom;
	updateContentSize();
	updateImageOffset();
	viewport()->repaint(false);
	updateZoomActions();
	emit zoomChanged(mZoom);
}


void ScrollPixmapView::setZoomStep(double zoomStep) {
	mZoomStep=zoomStep;
	updateZoomActions();
}


void ScrollPixmapView::installRBPopup(QPopupMenu* menu) {
	mPopupMenu=menu;
}


//-Overloaded methods----------------------------------
void ScrollPixmapView::resizeEvent(QResizeEvent* event) {
	if (mAutoZoom->isChecked()) {
		setZoom(computeAutoZoomValue());
	} else {
		updateContentSize();
		updateImageOffset();
	}
	QScrollView::resizeEvent(event);
}



void ScrollPixmapView::drawContents(QPainter* painter,int clipx,int clipy,int clipw,int cliph) {
	if (mGVPixmap->isNull()) {
		painter->eraseRect(clipx,clipy,clipw,cliph);
		return;
	}


	painter->scale(mZoom,mZoom);
/*

 1 : Viewport
 2 : Image
 3 : Paint area

mXOffset / mZoom
 |<-------->|
 |          |
 +-------------------------------+
 |1                              |
 |    +--------+                 |
 |    |3       |                 |
 |    |     +--|-----+           |
 |    |     | /|  _/\|           |
 |    +--------+\/  2|           |
 |          +--------+           |
 |                               |
 |                               |
 |                               |
 +-------------------------------+
 |    |     |
 |    |<--->|
 |    |leftEmptySize / mZoom
 |    |
 |<-->|
 clipx / mZoom

*/

	int xOffset= int(mXOffset / mZoom);
	int yOffset= int(mYOffset / mZoom);
	clipx=int(clipx/mZoom);
	clipy=int(clipy/mZoom);
	clipw=int(clipw/mZoom);
	cliph=int(cliph/mZoom);

// Erase empty areas
	int pixWidth=mGVPixmap->width();
	int pixHeight=mGVPixmap->height();

	int leftEmptySize=QMAX(xOffset - clipx+1,0);
	int topEmptySize=QMAX(yOffset - clipy+1,0);
	int rightEmptySize=QMAX(clipx + clipw - xOffset - pixWidth+1,0);
	int bottomEmptySize=QMAX(clipy + cliph - yOffset - pixHeight+1,0);
	
	int rightOfImage=clipx+leftEmptySize+pixWidth;
	int bottomOfImage=clipy+topEmptySize+pixHeight;

// Erase top line
	if (leftEmptySize && topEmptySize) {
		painter->eraseRect(clipx,clipy,leftEmptySize,topEmptySize);
	}
	if (topEmptySize) {
		painter->eraseRect(clipx+leftEmptySize,clipy,pixWidth,topEmptySize);
	}
	if (rightEmptySize && topEmptySize) {
		painter->eraseRect(rightOfImage,clipy,rightEmptySize,topEmptySize);
	}

// Erase borders
	if (leftEmptySize) {
		painter->eraseRect(clipx,clipy+topEmptySize,leftEmptySize,pixHeight);
	}
	if (rightEmptySize) {
		painter->eraseRect(rightOfImage,clipy+topEmptySize,rightEmptySize,pixHeight);
	}

// Erase bottom line
	if (leftEmptySize && bottomEmptySize) {
		painter->eraseRect(clipx,bottomOfImage,leftEmptySize,bottomEmptySize);
	}
	if (bottomEmptySize) {
		painter->eraseRect(clipx+leftEmptySize,bottomOfImage,pixWidth,bottomEmptySize);
	}
	if (rightEmptySize && bottomEmptySize) {
		painter->eraseRect(rightOfImage,bottomOfImage,rightEmptySize,bottomEmptySize);
	}

// Now draw the pixmap
	painter->drawPixmap( clipx,clipy,
		mGVPixmap->pixmap(),
		clipx-xOffset-1, clipy-yOffset-1,
		clipw+2, cliph+2 );
}


void ScrollPixmapView::viewportMousePressEvent(QMouseEvent* event) {
	mScrollStartX=event->x();
	mScrollStartY=event->y();
	if (event->button()==RightButton || mAutoZoom->isChecked()) return;
	mDragStarted=true;
	setCursor(QCursor(SizeAllCursor));
}


void ScrollPixmapView::viewportMouseMoveEvent(QMouseEvent* event) {
	if (!mDragStarted) return;

	int deltaX,deltaY;

	deltaX=mScrollStartX - event->x();
	deltaY=mScrollStartY - event->y();

	mScrollStartX=event->x();
	mScrollStartY=event->y();
	scrollBy(deltaX,deltaY);
}


void ScrollPixmapView::viewportMouseReleaseEvent(QMouseEvent* event) {
	if (mDragStarted) {
		setCursor(QCursor(ArrowCursor));
		mDragStarted=false;
	}

	switch (event->button()) {
	case Qt::LeftButton:
		if (event->stateAfter() & Qt::RightButton) {
			emit selectPrevious();
		}
		return;
	case Qt::RightButton:
		if (event->stateAfter() & Qt::LeftButton) {
			emit selectNext();
		} else {
			if (!mGVPixmap->isNull() && mPopupMenu) {
				mPopupMenu->popup(event->globalPos());
			}
		}
		return;
	default:
		break;
	}
}


void ScrollPixmapView::wheelEvent(QWheelEvent* event) {
	if (! (event->state() & Qt::ControlButton) ) {
		QScrollView::wheelEvent(event);
		return;
	}

	if (event->delta()<0) {
		emit selectNext();
	} else {
		emit selectPrevious();
	}
	event->accept();
}


//-Slots----------------------------------------------
void ScrollPixmapView::slotAutoZoom() {
	if (mAutoZoom->isChecked()) {
		mOldZoom=mZoom;
		setHScrollBarMode(AlwaysOff);
		setVScrollBarMode(AlwaysOff);
		setZoom(computeAutoZoomValue());
	} else {
		setHScrollBarMode(Auto);
		setVScrollBarMode(Auto);
		setZoom(mOldZoom);
	}
}


void ScrollPixmapView::slotZoomIn() {
	setZoom(mZoom + mZoomStep);
}


void ScrollPixmapView::slotZoomOut() {
	setZoom(mZoom - mZoomStep);
}


void ScrollPixmapView::slotResetZoom() {
	if (mAutoZoom->isChecked()) {
		mAutoZoom->setChecked(false);
		setHScrollBarMode(Auto);
		setVScrollBarMode(Auto);
	}
	setZoom(1.0);
}


//-Private---------------------------------------------
void ScrollPixmapView::updateContentSize() {
	resizeContents(
		QMAX( int(mGVPixmap->pixmap().width()*mZoom), width()-2*frameWidth() ),
		QMAX( int(mGVPixmap->pixmap().height()*mZoom), height()-2*frameWidth() ) );
}


void ScrollPixmapView::updateZoomActions() {
// Disable everything if there's no image
	if (mGVPixmap->isNull()) {
		mZoomIn->setEnabled(false);
		mZoomOut->setEnabled(false);
		mAutoZoom->setEnabled(false);
		mResetZoom->setEnabled(false);
		return;
	}

	mAutoZoom->setEnabled(true);
	mResetZoom->setEnabled(true);

	if (mAutoZoom->isChecked()) {
		mZoomIn->setEnabled(false);
		mZoomOut->setEnabled(false);
	} else {
		mZoomIn->setEnabled(true);
		mZoomOut->setEnabled(mZoom-mZoomStep>0);
	}
}


void ScrollPixmapView::updateImageOffset() {
	int pixWidth=int( mGVPixmap->pixmap().width() * mZoom);
	int pixHeight=int( mGVPixmap->pixmap().height() * mZoom);

	mXOffset=QMAX( (width()-pixWidth)/2, 0);
	mYOffset=QMAX( (height()-pixHeight)/2, 0);
}


double ScrollPixmapView::computeAutoZoomValue() const {
	int srcWidth=mGVPixmap->width();
	int srcHeight=mGVPixmap->height();
	int dstWidth=width()-frameWidth()*2;
	int dstHeight=height()-frameWidth()*2;
	double scale;

// Need to scale ?
	if (srcWidth<=dstWidth && srcHeight<=dstHeight) {
	// No scale needed
		scale=1.0;
	} else {
	// Yes, compute scale
		double widthScale=double(dstWidth) / double(srcWidth);
		double heightScale=double(dstHeight) / double(srcHeight);
		
		if ( heightScale < widthScale ) {
			scale=heightScale;
		} else {
			scale=widthScale;
		}
	}
	return scale;
}


//-Config----------------------------------------------
void ScrollPixmapView::readConfig(KConfig* config, const QString& group) {
	config->setGroup(group);
	setZoomStep(config->readDoubleNumEntry(CONFIG_ZOOM_STEP,1.0));
}


void ScrollPixmapView::writeConfig(KConfig* config, const QString& group) const {
	config->setGroup(group);
	config->writeEntry(CONFIG_ZOOM_STEP,mZoomStep);
}

