#include "vars.h"
#include "hitopcore.h"
#include "stringproc.h"
#include "error.h"
#include "dtutils.h"

#include <fstream>
#include <algorithm>

class HTML;

void Vars::PrefixEngine(map<string,string>& results,const string& prefix,
  unsigned int len)const{
  if(m_parent!=NULL) m_parent->PrefixEngine(results,prefix,len);
  for(map<string,string>::const_iterator i=m_map.begin(); i!=m_map.end(); ++i){
    if(i->first.substr(0,len)==prefix) results.insert(*i);
  }
}

void Vars::Prefix(map<string,string>& results,string prefix)const{
  if(!g_isXML) transform(prefix.begin(),prefix.end(),prefix.begin(),ToUpper());
  PrefixEngine(results,prefix,prefix.length());
}

bool Vars::IsDefEngine(const string& name)const{
  if(m_map.find(name)!=m_map.end()) return true;
  if(m_parent!=NULL) return m_parent->IsDefEngine(name);
  return false;
}

bool Vars::IsDefName(string name)const{
  if(!g_isXML) transform(name.begin(),name.end(),name.begin(),ToUpper());
  return IsDefEngine(name);
}

bool Vars::GetEngine(const string& name,string& value)const{
  map<string,string>::const_iterator pos=m_map.find(name);
  if(pos!=m_map.end()){
    value=pos->second;
    return true;
  }
  if(m_parent!=NULL) return m_parent->GetEngine(name,value);
  return false;
}

bool Vars::Get(const HTML& curHTML,string name,string& value)const{
  string::iterator pos=name.begin(),sec;
  if(pos==name.end()) return false;
  char first=name[0];
  if((first=='\'')||(first=='"')){
    ++pos;
    sec=find(pos,name.end(),first);
    if(sec==name.end()) Error(curHTML,e_UntermQuotes);
    value.assign(pos,sec);
    pos=sec+1;
  }else{
    sec=find(name.begin(),name.end(),':');
    string name(pos,sec);
    if(!g_isXML) transform(name.begin(),name.end(),name.begin(),ToUpper());
    if(!GetEngine(name,value)) return false;
    pos=sec;
  }
  if(pos==name.end()) return true;
  StringProc::ProcessString(curHTML,value,string(++pos,name.end()));
  return true;
}

static const string g_first("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_");
static const string g_rest("0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz_");

bool Vars::IsValidVarName(const string& name){
  string::const_iterator pos=name.begin();
  if((pos==name.end())||(find(g_first.begin(),g_first.end(),*pos)==g_first.end()))  return false;
  ++pos;
  while(pos!=name.end()){
    if(find(g_rest.begin(),g_rest.end(),*pos)==g_rest.end()) return false;
    ++pos;
  }
  return true;
}

bool Vars::SetEngine(const string& name,const string& value,
  Scope scope){
  switch(scope){
  case Local:
    m_map[name]=value;
    break;
  case Parent:
    if(m_parent==NULL){
      m_map[name]=value;
    }else{
      m_map.erase(name);
      m_parent->SetEngine(name,value,Local);
    }
    break;
  case Global:
    if(m_parent==NULL){
      m_map[name]=value;
    }else{
      m_map.erase(name);
      m_parent->SetEngine(name,value,Global);
    }
    break;
  }
  return true;
}

bool Vars::Set(string name,const string& value,Scope scope){
  if(!g_isXML) transform(name.begin(),name.end(),name.begin(),ToUpper());
  if(!IsValidVarName(name)) return false;
  return SetEngine(name,value,scope);
}

bool Vars::UnsetEngine(const string& name,Scope scope){
  switch(scope){
  case Local:
    m_map.erase(name);
    break;
  case Parent:
    m_map.erase(name);
    if(m_parent!=NULL) m_parent->UnsetEngine(name,Local);
    break;
  case Global:
    m_map.erase(name);
    if(m_parent!=NULL) m_parent->UnsetEngine(name,Global);
    break;
  }
  return true;
}

bool Vars::Unset(string name,Scope scope){
  if(!g_isXML) transform(name.begin(),name.end(),name.begin(),ToUpper());
  if(!IsValidVarName(name)) return false;
  return UnsetEngine(name,scope);
}
