#include <bickley/bkl-item.h>
#include <bickley/bkl-item-audio.h>
#include <bickley/bkl-item-image.h>
#include <bickley/bkl-item-video.h>

#include "hrn.h"
#include "hrn-square-clone.h"
#include "hrn-texture-cache.h"
#include "hrn-tileable.h"
#include "hrn-tile.h"
#include "hrn-tile-frame.h"

enum {
    PROP_0,
    PROP_ITEM,
    PROP_NODE
};

struct _HrnTilePrivate {
    BklItem *item;

    ClutterActor *thumbnail;
    NbtkWidget *primary;
    NbtkWidget *secondary;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_TILE, HrnTilePrivate))
static void tileable_interface_init (HrnTileableInterface *iface);
G_DEFINE_TYPE_WITH_CODE (HrnTile, hrn_tile, NBTK_TYPE_TABLE,
                         G_IMPLEMENT_INTERFACE (HRN_TYPE_TILEABLE,
                                                tileable_interface_init));

static void
hrn_tile_finalize (GObject *object)
{
    G_OBJECT_CLASS (hrn_tile_parent_class)->finalize (object);
}

static void
hrn_tile_dispose (GObject *object)
{
    HrnTile *self = (HrnTile *) object;
    HrnTilePrivate *priv = self->priv;

    if (priv->item) {
        g_object_unref (priv->item);
        priv->item = NULL;
    }

    G_OBJECT_CLASS (hrn_tile_parent_class)->dispose (object);
}

static void
frame_action_cb (HrnTileFrame *frame,
                 HrnTile      *tile)
{
    HrnClusterNode *node;

    node = hrn_tileable_get_node ((HrnTileable *) tile);
    hrn_tileable_activated ((HrnTileable *) tile,
                            HRN_TILEABLE_ACTION_PLAY, node);
}

static void
set_thumbnail (HrnTile *tile)
{
    HrnTilePrivate *priv = tile->priv;
    HrnTextureCache *cache = hrn_texture_cache_get_default ();
    ClutterActor *thumbnail;
    const char *uri = NULL;

    uri = bkl_item_extended_get_thumbnail ((BklItemExtended *) priv->item);

    if (priv->thumbnail) {
        clutter_actor_destroy (priv->thumbnail);
    }

    if (uri) {
        thumbnail = hrn_texture_cache_get_texture (cache, uri);

        if (bkl_item_get_item_type (priv->item) == BKL_ITEM_TYPE_IMAGE) {
            BklItemImage *im = (BklItemImage *) priv->item;
            const char *orient;

            orient = bkl_item_image_get_orientation (im);
            if (orient) {
                if (g_str_equal (orient, "right - top")) {
                    clutter_actor_set_anchor_point_from_gravity
                        (thumbnail, CLUTTER_GRAVITY_CENTER);

                    clutter_actor_set_rotation (thumbnail,
                                                CLUTTER_Z_AXIS, 90,
                                                0, 0, 0);

                    clutter_actor_set_anchor_point_from_gravity
                        (thumbnail, CLUTTER_GRAVITY_SOUTH_WEST);
                }
            }
        }
    } else {
        thumbnail = hrn_texture_cache_get_default_texture
            (cache, bkl_item_get_item_type (priv->item));
    }

    priv->thumbnail = g_object_new (HRN_TYPE_TILE_FRAME,
                                    "texture", thumbnail,
                                    "label", hrn_tile_get_label (tile),
                                    NULL);
    g_signal_connect (priv->thumbnail, "primary-action",
                      G_CALLBACK (frame_action_cb), tile);
    g_signal_connect (priv->thumbnail, "secondary-action",
                      G_CALLBACK (frame_action_cb), tile);
    nbtk_table_add_actor (NBTK_TABLE (tile), priv->thumbnail, 0, 0);
}

static void
update_tile (HrnTile *tile)
{
    HrnTilePrivate *priv = tile->priv;
    char *title = NULL, *secondary = NULL;
    guint width, height, season, episode, year;

    switch (bkl_item_get_item_type (priv->item)) {
    case BKL_ITEM_TYPE_AUDIO:
        title = g_strdup (bkl_item_audio_get_title ((BklItemAudio *) priv->item));
        secondary = g_strdup (bkl_item_audio_get_album ((BklItemAudio *) priv->item));
        if (secondary == NULL) {
            secondary = g_strdup_printf ("%d seconds",
                                         bkl_item_audio_get_duration
                                         ((BklItemAudio *) priv->item));
        }
        break;

    case BKL_ITEM_TYPE_IMAGE:
        title = g_strdup (bkl_item_image_get_title ((BklItemImage *) priv->item));
        width = bkl_item_image_get_width ((BklItemImage *) priv->item);
        height = bkl_item_image_get_height ((BklItemImage *) priv->item);
        secondary = g_strdup_printf ("%ux%u", width, height);
        break;

    case BKL_ITEM_TYPE_VIDEO:
        if (bkl_item_video_get_series_name ((BklItemVideo *) priv->item)) {
            title = g_strdup (bkl_item_video_get_series_name ((BklItemVideo *) priv->item));
        } else {
            title = g_strdup (bkl_item_video_get_title ((BklItemVideo *) priv->item));
        }

        year = bkl_item_video_get_year ((BklItemVideo *) priv->item);
        if (year < 1900) {
            season = bkl_item_video_get_season ((BklItemVideo *) priv->item);
            episode = bkl_item_video_get_episode ((BklItemVideo *) priv->item);

            /* FIXME: When out of string freeze again, add support for
               "Season %u" and "Episode %u" */
            if (season > 0 && episode > 0) {
                secondary = g_strdup_printf ("Season %u, Episode %u", season, episode);
            } else {
                secondary = g_strdup ("");
            }
        } else {
            secondary = g_strdup_printf ("(%u)", year);
        }
        break;

    default:
        break;
    }

    if (title == NULL) {
        char *basename;

        basename = g_path_get_basename (bkl_item_get_uri (priv->item));
        title = g_uri_unescape_string (basename, NULL);
        g_free (basename);
    }

    nbtk_label_set_text (NBTK_LABEL (priv->primary), title);
    nbtk_label_set_text (NBTK_LABEL (priv->secondary),
                         secondary ? secondary :
                         bkl_item_get_mimetype (priv->item));

    set_thumbnail (tile);

    g_free (title);
    g_free (secondary);
}

static void
clear_tile (HrnTile *tile)
{
    HrnTilePrivate *priv = tile->priv;
    GError *error = NULL;
    guint32 data = 0;

    nbtk_label_set_text (NBTK_LABEL (priv->primary), "");
    nbtk_label_set_text (NBTK_LABEL (priv->secondary), "");

    clutter_texture_set_from_rgb_data (CLUTTER_TEXTURE (priv->thumbnail),
                                       (const guchar *) &data, TRUE,
                                       1, 1, 1, 4, 0, &error);
    if (error) {
        g_warning ("Error clearing texture: %s", error->message);
        g_error_free (error);
    }
}

static void
hrn_tile_set_property (GObject      *object,
                       guint         prop_id,
                       const GValue *value,
                       GParamSpec   *pspec)
{
    HrnTile *self = (HrnTile *) object;
    HrnTilePrivate *priv = self->priv;

    switch (prop_id) {
    case PROP_ITEM:
        if (priv->item) {
            g_object_unref (priv->item);
        }

        priv->item = g_value_dup_object (value);
        if (priv->item) {
            update_tile (self);
        } else {
            clear_tile (self);
        }
        break;

    case PROP_NODE:
        break;

    default:
        break;
    }
}

static void
hrn_tile_get_property (GObject    *object,
                       guint       prop_id,
                       GValue     *value,
                       GParamSpec *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
hrn_tile_class_init (HrnTileClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;

    o_class->dispose = hrn_tile_dispose;
    o_class->finalize = hrn_tile_finalize;
    o_class->set_property = hrn_tile_set_property;
    o_class->get_property = hrn_tile_get_property;

    g_type_class_add_private (klass, sizeof (HrnTilePrivate));

    g_object_class_install_property (o_class, PROP_ITEM,
                                     g_param_spec_object ("item", "", "",
                                                          BKL_TYPE_ITEM,
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_NODE,
                                     g_param_spec_pointer ("node", "", "",
                                                           G_PARAM_WRITABLE |
                                                           G_PARAM_STATIC_STRINGS));
}

static guint
tileable_get_count (HrnTileable *tileable)
{
    /* There's always only 1 item for HrnTile */
    return 1;
}

static ClutterActor *
tileable_get_thumbnail (HrnTileable *tileable)
{
    HrnTile *tile = HRN_TILE (tileable);
    HrnTilePrivate *priv = tile->priv;

    return priv->thumbnail;
}

static void
tileable_set_position (HrnTileable *tileable,
                       guint        x,
                       guint        y)
{
    /* HrnTile doesn't really care where it's positioned */
}

static HrnClusterNode *
tileable_get_node (HrnTileable *tileable)
{
    HrnTile *tile = (HrnTile *) tileable;
    HrnTileClass *t_class = HRN_TILE_GET_CLASS (tile);

    return t_class->get_node (tile);
}

static void
tileable_interface_init (HrnTileableInterface *iface)
{
    iface->get_count = tileable_get_count;
    iface->get_thumbnail = tileable_get_thumbnail;
    iface->set_position = tileable_set_position;
    iface->get_node = tileable_get_node;
}

static gboolean
label_clicked_cb (ClutterActor       *actor,
                  ClutterButtonEvent *event,
                  HrnTile            *tile)
{
    const char *text;

    if (event->button != 1) {
        return FALSE;
    }

    text = nbtk_label_get_text ((NbtkLabel *) actor);
    hrn_tileable_activated ((HrnTileable *) tile,
                            HRN_TILEABLE_ACTION_SEARCH, (char *) text);
    return TRUE;
}

static gboolean
label_enter_cb (ClutterActor         *actor,
                ClutterCrossingEvent *event,
                HrnTile              *tile)
{
    nbtk_widget_set_style_pseudo_class ((NbtkWidget *) actor, "hover");
    return TRUE;
}

static gboolean
label_leave_cb (ClutterActor         *actor,
                ClutterCrossingEvent *event,
                HrnTile              *tile)
{
    nbtk_widget_set_style_pseudo_class ((NbtkWidget *) actor, NULL);
    return TRUE;
}

static void
hrn_tile_init (HrnTile *self)
{
    HrnTilePrivate *priv = GET_PRIVATE (self);

    self->priv = priv;

    clutter_actor_set_size (CLUTTER_ACTOR (self), ITEM_WIDTH, ITEM_HEIGHT);

    priv->primary = nbtk_label_new ("");
    nbtk_widget_set_style_class_name (priv->primary, "HrnTileablePrimary");
    clutter_actor_set_reactive ((ClutterActor *) priv->primary, TRUE);
    g_signal_connect (priv->primary, "button-release-event",
                      G_CALLBACK (label_clicked_cb), self);
    g_signal_connect (priv->primary, "enter-event",
                      G_CALLBACK (label_enter_cb), self);
    g_signal_connect (priv->primary, "leave-event",
                      G_CALLBACK (label_leave_cb), self);
    nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                          (ClutterActor *) priv->primary, 1, 0,
                                          "x-expand", FALSE,
                                          "x-fill", FALSE,
                                          "y-expand", FALSE,
                                          "y-fill", FALSE,
                                          "x-align", 0.0,
                                          NULL);

    priv->secondary = nbtk_label_new ("");
    nbtk_widget_set_style_class_name (priv->secondary, "HrnTileableSecondary");
    clutter_actor_set_reactive ((ClutterActor *) priv->secondary, TRUE);
    g_signal_connect (priv->secondary, "button-release-event",
                      G_CALLBACK (label_clicked_cb), self);
    g_signal_connect (priv->secondary, "enter-event",
                      G_CALLBACK (label_enter_cb), self);
    g_signal_connect (priv->secondary, "leave-event",
                      G_CALLBACK (label_leave_cb), self);
    nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                          (ClutterActor *) priv->secondary,
                                          2, 0,
                                          "x-expand", FALSE,
                                          "x-fill", FALSE,
                                          "y-expand", FALSE,
                                          "y-fill", FALSE,
                                          "x-align", 0.0,
                                          NULL);
}

const char *
hrn_tile_get_label (HrnTile *tile)
{
    HrnTileClass *klass = HRN_TILE_GET_CLASS (tile);

    return klass->get_label (tile);
}
