// ExMgr.h

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#ifndef EXMGR_H
#define EXMGR_H

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <mlcd.h>

/* VxWorks coding conventions. */
#define OK (0)
#define ERROR (-1)
#define NO_WAIT (0)
#define WAIT_FOREVER (-1)
typedef int STATUS;

/* Miscellaneous macros. */
#define cMaxWriteIrps 8			/* TODO: Adjust? */
#define IN_PROGRESS (OK+1)
#define SETTHIS(x) (this->x=x)
#define DEBUG_STRING(s) (0)
#define dmSetUint32(key,value) /* OK */
#define nettestRegister(args...)

/* Logging. */
enum LogType {
	LOG_TYPE_SYSLOG,
	LOG_TYPE_WARNING,
	LOG_TYPE_ERROR,
	LOG_TYPE_ERROR_FATAL,
	LOG_TYPE_ENTRY,
	LOG_TYPE_EXIT,
	LOG_TYPE_INFO
};
enum { cEXBP,cEXTP };
enum {
	cCauseBadState,
	cCauseNoMem,
	cCauseBadParm,
	cCauseFuncFailed,
	cCausePeriphError,
	cCauseNoHandles
};
void logCommon(char *file,int line,LogType type,int dummy,char *format=0,...);
#define LOG_SYSLOG(a,args...) \
	logCommon(__FILE__,__LINE__,LOG_TYPE_SYSLOG,args)
#define LOG_WARN(a,args...) \
	logCommon(__FILE__,__LINE__,LOG_TYPE_WARNING,args)
#define LOG_ERROR(a,b,args...) \
	logCommon(__FILE__,__LINE__,LOG_TYPE_ERROR,args)
#define LOG_ERROR_FATAL(a,b,args...) \
	logCommon(__FILE__,__LINE__,LOG_TYPE_ERROR_FATAL,args)
#define LOG_ASSERT(condition,a,b,args...) \
	{ if (!(condition)) LOG_ERROR_FATAL(a,b,args); }
#ifndef JD_DEBUGLITE
#define LOGD_ERROR(args...)
#define LOGD_ERROR_FATAL(args...)
#define LOGD_ASSERT(args...)
#define LOG_ENTRY(args...)
#define LOG_EXIT(args...)
#define LOG_INFO(args...)
#else
#define LOGD_ERROR LOG_ERROR
#define LOGD_ERROR_FATAL LOG_ERROR_FATAL
#define LOGD_ASSERT LOG_ASSERT
#define LOG_ENTRY(a,args...) \
	logCommon(__FILE__,__LINE__,LOG_TYPE_ENTRY,args)
#define LOG_EXIT(a,args...) \
	logCommon(__FILE__,__LINE__,LOG_TYPE_EXIT,args)
#define LOG_INFO(a,args...) \
	logCommon(__FILE__,__LINE__,LOG_TYPE_INFO,args)
#endif
#define lmcSetFilter(s1,s2,s3)
#define lmcStartConsole()
#define LOG_MODULE_REGISTER(a,b,c,d)

/* Status code aliases. */
#define S_hlio_BP_CHAN_CLOSING             MLCD_STATUS_CHAN_CLOSING
#define S_hlio_BP_CHAN_CLOSE_HAD_NO_EFFECT MLCD_STATUS_CHAN_CLOSE_HAD_NO_EFFECT
#define S_hlio_BP_SERVICE_LOOKUP_FAILED    MLCD_STATUS_SERVICE_LOOKUP_FAILED
#define S_hlio_BP_LOOKUP_NOT_SUPPORTED     MLCD_STATUS_LOOKUP_NOT_SUPPORTED
#define S_hlio_BP_SOCKET_TABLE_FULL        MLCD_STATUS_SOCKET_TABLE_FULL
#define S_hlio_BP_CONNECTION_REFUSED       MLCD_STATUS_CONNECTION_REFUSED
#define S_hlio_BP_INSUFFICIENT_RESOURCES   MLCD_STATUS_INSUFFICIENT_RESOURCES
#define S_hlio_BP_UNKNOWN_ERROR            MLCD_STATUS_UNKNOWN_ERROR
#define S_hlio_BP_UNSUPPORTED_SOCKET       MLCD_STATUS_UNSUPPORTED_SOCKET
#define S_hlio_BP_CLOSED_BY_PERIPHERAL     MLCD_STATUS_CLOSED_BY_PERIPHERAL
#define S_hlio_BP_PERIPHERAL_ERROR         MLCD_STATUS_PERIPHERAL_ERROR
#define EX_CLOSE_REASON_TRANSPORT_BASE     0x3000

/* Queue implemented as a linked list. */
class QueueEntry {
	friend class Queue;
    protected:
	QueueEntry *prev;
	QueueEntry *next;
    public:
	QueueEntry(): prev(0),next(0) { } 
#ifdef JD_DEBUGLITE
	void dump(void) {
		printf("QueueEntry(0x%8.8X): prev=0x%8.8X, next=0x%8.8X\n",
			(int)this,(int)prev,(int)next);
	}
#endif
	int isEnqueued(void) { return (prev || next); }
    protected:
	QueueEntry *base_getPrev(void) { return prev; }
	QueueEntry *base_getNext(void) { return next; }
};
 
class Queue {
    protected:
	QueueEntry *head,*tail;
	int _depth;
    public:
	Queue(char *name=0): head(0),tail(0),_depth(0) { }
	int depth(void) { return _depth; }
    protected:
	void base_add(QueueEntry *entry,QueueEntry *before=0);
	QueueEntry *base_peek(QueueEntry *entry=0);
	QueueEntry *base_pop(QueueEntry *entry=0);
};

/* Class forward declarations. */
class ExBufferPool;
class ExMsg;
class ExMgr;
typedef ExMgr ExService;
typedef int SCD;
typedef ExMgr ExPhysicalPort;
class ExSessionLookup;
class ExTransport;
typedef class ExTransportChannel *TCD;

/* Base class for most of our classes. */
class ExMsgHandler {
    public:
	ExMsgHandler(void) { }
	virtual ~ExMsgHandler(void) { }
	virtual void handleMsg(ExMsg *pMsg)=0;
};

enum ExMsgType {
	eEXMSG_ACTIVATE_WAIT,
	eEXMSG_ACTIVATE_RESPONSE,
	eEXMSG_LOOKUP_RESPONSE,
	eEXMSG_REMOTE_SOCK_RESPONSE,
	eEXMSG_OPEN_CHAN_RESPONSE,
	eEXMSG_CLOSE_CHAN_RESPONSE,
	eEXMSG_DEACTIVATE_RESPONSE
};
enum { eEXCLASS_TRANSPORT };

class ExMsg: public QueueEntry {
    protected:
	ExMgr *pMgr;
	ExMsgType type;
	int param[4];
	void *vparam;
	ExMsgHandler *pMsgHandler;
    public:
	ExMsg(ExMgr *pMgr) { SETTHIS(pMgr); }
	ExMsgType getType(void) { return type; }
	void setType(ExMsgType type) { SETTHIS(type); }
	int getParam(int index) { return param[index]; }
	void setParams(int a=0,int b=0,int c=0,int d=0) {
		param[0]=a;
		param[1]=b;
		param[2]=c;
		param[3]=d;
	}
	void *getVoidParam(void) { return vparam; }
	void setParams(void *v) { vparam=v; }
	ExMsgHandler *getMsgHandler(void) { return pMsgHandler; }
	void send(ExMsgHandler *pMsgHandler);
};

class ExMsgQueue: public Queue {
    public:
	void add(ExMsg *pMsg) { base_add(pMsg); }
	ExMsg *peek(ExMsg *pMsg=0) { return (ExMsg *)base_peek(pMsg); }
	ExMsg *pop(ExMsg *pMsg=0) { return (ExMsg *)base_pop(pMsg); }
};

class PolledTimer {
    protected:
	struct timeval start;
	struct timeval end;

    public:
	static int getTime(struct timeval *pTimeval) {
		return gettimeofday(pTimeval,0);
	}
	static void delay(struct timeval *pTimeout) {
		struct timeval timeoutCopy;
		if (pTimeout) {
			timeoutCopy.tv_sec=pTimeout->tv_sec;
			timeoutCopy.tv_usec=pTimeout->tv_usec;
			pTimeout=&timeoutCopy;
		}
		select(0,0,0,0,pTimeout);
	}
	static void delay(struct timeval *pTimeout,struct timeval *pActual) {
		struct timeval start;
		getTime(&start);
		delay(pTimeout);
		getTime(pActual);
		if (pActual->tv_usec<start.tv_usec) {
			pActual->tv_sec--;
			pActual->tv_usec+=1000000;
		}
		pActual->tv_usec-=start.tv_usec;
		pActual->tv_sec-=start.tv_sec;
	}
	static int compareTimes(struct timeval *a,struct timeval *b);

	void setTimeout(struct timeval *pTimeout);
	int isTimedOut(void);
#if 0
	void getTimeoutTime(struct timeval *pTime) {
		pTime->tv_sec=end.tv_sec;
		pTime->tv_usec=end.tv_usec;
	}
#endif
	void getRemainingTime(struct timeval *pTime);
};

class ExWatchdogTimer: public QueueEntry {
    protected:
	ExMsgHandler *pMsgHandler;
	ExMsg *pMsg;

	struct timeval delay;
	PolledTimer timer;

	int cancelled;
	int restart;
	int started;

    public:
	ExWatchdogTimer(ExMsgHandler *pMsgHandler=0,ExMsg *pMsg=0,
	    char *mystring=0);
	~ExWatchdogTimer();

#ifdef JD_DEBUGLITE
	void dump(void);
#endif

	void setMsg(ExMsg *pMsg);
	ExMsg *removeMsg(void) {
		ExMsg *oldMsg=pMsg;

		pMsg=0;
		return oldMsg;
	}

	void getDelay(struct timeval *pTime) {
		pTime->tv_sec=delay.tv_sec;
		pTime->tv_usec=delay.tv_usec;
	}
	void setDelay(int seconds,int usec=0) {
		if (!seconds && !usec) usec=1;
		delay.tv_sec=seconds;
		delay.tv_usec=usec;
		if (isPeriodic() && isEnqueued()) {
			cancel();
			start();
		}
	}
	void start(int seconds,int usec=0) {
		setDelay(seconds,usec);
		return start();
	}
	void start(void);
	int isStarted(void) { return started; }
	void startIfIdle(void) { if (!isStarted()) start(); }
	void startIfIdle(int seconds,int usec=0) {
		setDelay(seconds,usec);
		return startIfIdle();
	}

	void cancel(void);
	int isCancelled(void) { return cancelled; }

	ExWatchdogTimer *getPrev(void) {
		return (ExWatchdogTimer *)base_getPrev();
	}
	ExWatchdogTimer *getNext(void) {
		return (ExWatchdogTimer *)base_getNext();
	}
	int isTimedOut(void) { return timer.isTimedOut(); }
	void callback(void);

	int isPeriodic(void) { return (!pMsgHandler); }
	void getRemainingTime(struct timeval *pTime) {
		if (isPeriodic()) {
			getDelay(pTime);
		} else {
			timer.getRemainingTime(pTime);
		}
	}
#if 0
	void getTimeoutTime(struct timeval *pTime) {
		if (isPeriodic()) {
			getDelay(pTime);
		} else {
			timer.getTimeoutTime(pTime);
		}
	}
#endif
};

class ExCountingWatchdogTimer: public ExWatchdogTimer {
    protected:
	int count;

    public:
	ExCountingWatchdogTimer(ExMsgHandler *pMsgHandler=0,ExMsg *pMsg=0,
	    char *mystring=0): ExWatchdogTimer(pMsgHandler,pMsg,mystring) {
		reset();
	}

	void setDelay(int seconds,int usec=0) {
		return ExWatchdogTimer::setDelay(seconds,usec);
	}

	void start(void) {
		count++;
		ExWatchdogTimer::start();
	}

	void stop(void) {
		if (count) {
			count--;
			if (!count) cancel();
		}
	}

	void reset(void) {
		count=0;
		ExWatchdogTimer::cancel();
	}

	int getCount(void) { return count; }
};

class ExWatchdogTimerQueue: public Queue {
    public:
	void add(ExWatchdogTimer *pWatchdogTimer);
	ExWatchdogTimer *peek(ExWatchdogTimer *pWatchdogTimer=0) {
		return (ExWatchdogTimer *)base_peek(pWatchdogTimer);
	}
	ExWatchdogTimer *pop(ExWatchdogTimer *pWatchdogTimer=0) {
		return (ExWatchdogTimer *)base_pop(pWatchdogTimer);
	}
};

class ExBdr: public QueueEntry {
    protected:
	ExBufferPool *pBufferPool;
	int bufferSize;
	unsigned char *startAddress;

	ExBdr *nextInChain;
	int dataOffset;
	int datalen;
	ExTransport *pTransport;
	TCD tcd;

    public:
	ExBdr(ExBufferPool *pBufferPool,int bufferSize);
	~ExBdr();
	int getSize(void) { return bufferSize; }
	void returnBuffer(void);
	ExBdr *getNext(void) { return nextInChain; }
	void setNext(ExBdr *nextInChain) { SETTHIS(nextInChain); }
	ExBdr *appendFromPool(void);
	unsigned char *getStartAddress(void) { return startAddress; }
	int getDataOffset(void) { return dataOffset; }
	void setDataOffset(int dataOffset) { SETTHIS(dataOffset); }
	void unprependData(int offset) {
		dataOffset+=offset;
		datalen-=offset;
	}
	int getDataLength(void) { return datalen; }
	void setDataLength(int datalen) { SETTHIS(datalen); }
	ExTransport *getTransportNotification(void) { return pTransport; }
	void setTransportNotification (ExTransport *pTransport) {
		SETTHIS(pTransport);
	}
	TCD getTCD(void) { return tcd; }
	void setTCD(TCD tcd) { SETTHIS(tcd); }
	int getTransportHeaderFlags(void) { return 0; }
	void setTransportHeaderFlags(int flags) { }
    protected:
	void reset(void) {
		setNext(0);
		dataOffset=datalen=0;
		pTransport=0;
		tcd=0;
	}
};

inline void exReturnPhysicalPortBuffer(ExBdr *pBdr) {
	pBdr->setTCD(0);
	pBdr->returnBuffer();
}

class ExBdrQueue: public Queue {
    public:
	ExBdrQueue(char *name=0): Queue(name) { }
	void add(ExBdr *pBdr) { base_add(pBdr); }
	void Add(ExBdr *pBdr) { add(pBdr); }
	ExBdr *peek(void) { return (ExBdr *)base_peek(); }
	ExBdr *Peek(void) { return peek(); }
	ExBdr *pop(void) { return (ExBdr *)base_pop(); }
	ExBdr *Pop(void) { return pop(); }
	void empty(ExBdrQueue *pDestQueue=0) {
		ExBdr *pBdr;
		while ((pBdr=pop())!=0) {
			if (pDestQueue) pDestQueue->add(pBdr);
			else pBdr->returnBuffer();
		}
	}
};

class ExBufferPool {
    protected:
	ExBdrQueue bdrQueue;
	int bufferSize;
	int requestedBufferCount;
	int bufferCount;
    public:
	ExBufferPool(void *owner,int bufferSize,int requestedBufferCount,
	    int port,int bufferManager);
	~ExBufferPool();
	ExBdr *getBuffer(int noCreate=0);
	void returnBuffer(ExBdr *pBdr) { bdrQueue.add(pBdr); }
#ifdef JD_DEBUGLITE
	void dump(void) {
		printf("available#=%d, size=%d, requested#=%d, allocated#=%d\n",
			bdrQueue.depth(),bufferSize,
			requestedBufferCount,bufferCount);
	}
#endif
};

#define EX_KNOB_TRANSPORT 0x300
#include "transport/ExTransportKnobs.h"

STATUS tknobGetWorkingValue(int port,int id,int *pValue);
STATUS exTknobGetInit(int id,int *pValue);

class ExMgr: public ExMsgHandler {
	// Major helper classes:
    protected:
	ExBufferPool *pBufferPool;
	ExTransport *pTransport;

	// Constructor, destructor, dump:
    public:
	ExMgr();
	~ExMgr();
#ifdef JD_DEBUGLITE
	void dump(void);
#endif

	// Runtime flow of control:
    protected:
	static const int port=0;
	int noFork;
	int stayActivated;
	int noDot4;
	int noPmlMultiplexing;
	int sleepBeforeOpen;
	int initialized;
	fd_set rset;
	fd_set wset;
	int fdCount;
	enum ExContext {
		EX_CONTEXT_NONE=0,
		EX_CONTEXT_SESSION_HANDLE_FORWARD_DATA,
		EX_CONTEXT_SESSION_WRITE,
		EX_CONTEXT_LLIO_SERVICE_OUTPUT,
		EX_CONTEXT_LLIO_SERVICE_INPUT
	} exContext;
	enum {
		EX_STATE_DOWN=0,
		EX_STATE_ACTIVATING,
		EX_STATE_UP,
		EX_STATE_DEACTIVATING
	} exState;
	int exActivateCount;
	int exCloseCount;
	int exCloseReason;
	int exReactivateFlag;
	int tryDot4;
	int tryMlc;
	int miser;
    public:
	int isInitialized(void) { return initialized; } 
	void signalHandler(int signum);
	int exMain(int argc,char **argv);
    protected:
	int fdSetBlocking(int fd) { return fcntl(fd,F_SETFL,0); }
	int fdSetNonblocking(int fd) { return fcntl(fd,F_SETFL,O_NONBLOCK); }
	STATUS _fdRegister(int fd,int r,int w,char *file,int line);
#define fdRegister(fd,r,w) _fdRegister(fd,r,w,__FILE__,__LINE__)
	int fdIsRegisteredForRead(int fd) { return FD_ISSET(fd,&rset); }
    public:
	int getPortNumber(void) { return port; }
	int getBufferPoolMgr(void) { return 0; }
    protected:
	STATUS exActivate(void);
	void parseDeviceID(void);
	virtual int getDefaultTryDot4(void) { return 1; }
    public:
	STATUS tknobGet(int id,int *pValue);
    protected:
	void exActivateComplete(void) {
		exState=EX_STATE_UP;
		LOG_SYSLOG(cEXBP,0,"ptal-mlcd successfully activated.\n");
	}
    public:
	void exClose(int reason=0);

	// Command-line processing:
    protected:
	int argcOriginal;
	char **argvOriginal;
	int argc;
	char **argv;
    public:
	static void syntaxError(char *arg=0);
    protected:
	void argInit(int argc,char **argv) {
		argcOriginal=SETTHIS(argc);
		argvOriginal=SETTHIS(argv);
	}
	virtual void argProcess(char *arg);
	void argProcess(void) {
		while (argc) argProcess(argGetString());
	}
	static void printOptions(void);
	char *argPeekString(void) {
		if (argc<=0) return 0;
		return *argv;
	}
	char *argGetString(char *arg=0) {
		if (argc<=0) syntaxError(arg);
		argc--;
		return *(argv++);
	}
	int argGetInt(char *arg=0) {
		return strtol(argGetString(arg),0,0);
	}
	void argDump(int fd=ERROR);	// Not JD_DEBUGLITE-only.

	// /dev/null management:
    protected:
	int fdNull;
    protected:
	void nullInit(void) {
		fdNull=open("/dev/null",O_RDWR);
		LOG_ASSERT(fdNull>=0,cEXBP,0,cCauseFuncFailed);
	}
	int nullDup(void) { return dup(fdNull); }
	int nullDup(int fd) { return dup2(fdNull,fd); }

	// Message and timer management:
    protected:
	ExMsgQueue *pFreeMsgPool;
	ExMsgQueue *pPendingMsgQueue;
	ExWatchdogTimerQueue *pActiveTimerQueue;
	ExWatchdogTimerQueue *pPeriodicTimerQueue;
    public:
	ExMsg *getFreeMsg(void);
	void msgEnqueue(ExMsg *pMsg) { pPendingMsgQueue->add(pMsg); }
    protected:
	void msgService(void);
    public:
	virtual void handleMsg(ExMsg *pMsg);
	void returnMsg(ExMsg *pMsg) { pFreeMsgPool->add(pMsg); }
    protected:
	void timerService(void);
	void timerGetSelectTimeout(struct timeval *timeout,
	    struct timeval **pTimeout);
    public:
	void timerCancel(ExWatchdogTimer *pWatchdogTimer) {
		(pWatchdogTimer->isPeriodic()?pPeriodicTimerQueue:pActiveTimerQueue)->pop(pWatchdogTimer);
	}
	void timerStart(ExWatchdogTimer *pWatchdogTimer) {
		timerCancel(pWatchdogTimer);
		(pWatchdogTimer->isPeriodic()?pPeriodicTimerQueue:pActiveTimerQueue)->add(pWatchdogTimer);
	}

	// Debug console management:
    public:
	static const int CONSOLE_STDIN=0;
	static const int CONSOLE_STDOUT=1;
	static const int CONSOLE_STDERR=2;
    protected:
	static const int CONSOLE_BUFFER_LEN=80;
	int consoleAllowRemote;
	int consoleOldStdin;
	int consoleOldStdout;
	int consoleIsRemote;
    protected:
	void consoleInit(void);
	STATUS consolePreopen(void);
	void consoleOpen(int fdStdin,int fdStdout);
	void consoleOpen(void) {
		consoleOpen(consoleOldStdin,consoleOldStdout);
	}
	void consoleOpen(int fd) { consoleOpen(fd,fd); }
	void consoleOpenQuiet(int fdStdin,int fdStdout);
	void consoleOpenQuiet(void) {
		consoleOpenQuiet(consoleOldStdin,consoleOldStdout);
	}
	void consolePrompt(void);
	void consoleService(void);
	void consolePostSelect(fd_set *rset,fd_set *wset) {
		if (FD_ISSET(CONSOLE_STDIN,rset)) consoleService();
	}

	// Socket management:
    protected:
	static const char socketPrefix[]="/dev/ptal-mlcd/";
	static const int SOCKET_BACKLOG=5;
	char *socketSuffix;
	char *socketAliasSuffix;
	char *socketName;
	char *socketAliasName;
	int socketFd;
    public:
	char *getSocketSuffix(void) { return socketSuffix; }
    protected:
	void socketPreInit(void);
	void socketInit(void);
	void socketDone(void);
	void socketService(void);
	void socketPostSelect(fd_set *rset,fd_set *wset) {
		if (FD_ISSET(socketFd,rset)) socketService();
	}

	// Session management:
    protected:
	static const int MAX_REMOTE_SOCKETS=32;
	static const int MAX_COMMAND_SESSIONS   =8;
	static const int MAX_TRANSPORT_SESSIONS =16;
	static const int MAX_PML_SESSIONS       =8;
	static const int FIRST_COMMAND_SESSION  =0;
	static const int FIRST_TRANSPORT_SESSION=(FIRST_COMMAND_SESSION+MAX_COMMAND_SESSIONS);
	static const int FIRST_PML_SESSION      =(FIRST_TRANSPORT_SESSION+MAX_TRANSPORT_SESSIONS);
	static const int MAX_SESSIONS           =(FIRST_PML_SESSION+MAX_PML_SESSIONS);
	static const int LAST_COMMAND_SESSION   =(FIRST_TRANSPORT_SESSION-1);
	static const int LAST_TRANSPORT_SESSION =(FIRST_PML_SESSION-1);
	static const int LAST_PML_SESSION       =(MAX_SESSIONS-1);
	static const int PML_TRANSPORT_SESSION  =FIRST_TRANSPORT_SESSION;
	static const int MLC_LEN_HEADER=6;
	static const int DEFAULT_MAX_DATALEN=4096;
	static const int BUFFER_SIZE=(DEFAULT_MAX_DATALEN+MLC_LEN_HEADER);
	static const int PML_MAX_FORWARD_DATALEN=DEFAULT_MAX_DATALEN;
	static const int PML_MAX_REVERSE_DATALEN=DEFAULT_MAX_DATALEN;
	static const int MAX_FORWARD_BUFFERS=50;		/* TODO: Adjust? */
	static const int SESSION_MIN_BUFFERS=10;		/* TODO: Adjust? */
	static const int SESSION_BENEFIT_OF_MORE_BUFFERS=1;	/* TODO: Adjust? */
	static const int EXTRA_REVERSE_BUFFERS=20;		/* TODO: Adjust? */
	static const int MAX_REVERSE_BUFFERS=((MAX_TRANSPORT_SESSIONS*SESSION_MIN_BUFFERS)+EXTRA_REVERSE_BUFFERS);
	static const int MAX_BUFFERS=(MAX_FORWARD_BUFFERS+MAX_REVERSE_BUFFERS);
	static const int MAX_BDRS_PER_TRANSACTION=4;		/* TODO: Adjust? */
	static const int SOCKET_CONSOLE=-1;
	static const int SOCKET_PID=-2;
	static const int SOCKET_CMDLINE=-3;
	static const int SOCKET_DEVNODE=-4;
	static const int SOCKET_DUMP=-5;
	enum SessionState {
		SESSION_STATE_SHUTDOWN=-2,
		SESSION_STATE_STARTUP,		// -1
		SESSION_STATE_AVAILABLE,	// 0
		SESSION_STATE_COMMAND,
		SESSION_STATE_ACTIVATE_PENDING,
		SESSION_STATE_LOOKUP_PENDING,
		SESSION_STATE_SET_REMOTE_SOCKET_PENDING,
		SESSION_STATE_OPEN_PENDING,
		SESSION_STATE_OPEN,
		SESSION_STATE_CLOSE_PENDING
	};
	SCD pmlTransportSession;
	SCD pmlCurrentSession;
	SCD pmlLastSession;
	struct {
		SessionState state;
		int fd;
		SCD scdlink;
		ExSessionLookup *pLookup;
		int outstandingForwardBdrCount;
		ExBdrQueue *pReverseBdrQueue;
		TCD tcd;
		ExBdr *pCommandBdr;
		int pmlTrapsRegistered;
	} session[MAX_SESSIONS];
    protected:
#ifdef JD_DEBUGLITE
	int sessionIsInteresting(SCD scd);
	void sessionDump(SCD scd);
#endif
	void sessionPmlEnable(void) {
		pmlTransportSession=PML_TRANSPORT_SESSION;
		pmlCurrentSession=ERROR;
		pmlLastSession=LAST_PML_SESSION;
	}
	void sessionPmlDisable(void) {
		if (pmlTransportSession!=ERROR) {
			sessionChangeState(pmlTransportSession,
				SESSION_STATE_AVAILABLE);
		}
		pmlTransportSession=ERROR;
	}
	int sessionPmlIsEngaged(void) { return (pmlCurrentSession!=ERROR); }
	int sessionPmlEngage(SCD scd) {
		if (sessionPmlIsEngaged()) return ERROR;
		LOG_INFO(cEXBP,0,"sessionPmlEngage(scd=%d)\n",scd);
		return (pmlCurrentSession=scd);
	}
	int sessionPmlDisengage(SCD scd) {
		if (scd!=pmlCurrentSession) return ERROR;
		LOG_INFO(cEXBP,0,"sessionPmlDisengage(scd=%d)\n",scd);
		pmlCurrentSession=ERROR;
		pmlLastSession=scd;
		return OK;
	}
	void sessionChangeState(SessionState state);
	void sessionChangeState(SCD scd,SessionState state,int param=ERROR);
	SCD sessionFindAvailable(SCD first);
	SCD sessionStart(int fd);
	void sessionPostSelect(fd_set *rset,fd_set *wset);
	void sessionReadCommand(SCD scd);
	void sessionProcessCommand(SCD scd,int status=MLCD_STATUS_SUCCESSFUL);
	STATUS sessionTryLocalLookup(SCD scd);
	void sessionActivate(void);
	void sessionInitialRequestsComplete(void);
	void sessionActivateResponseFromTransport(void);
	void sessionActivateResponse(void);
	void sessionLookupResponse(SCD scd,ExSessionLookup *pLookup);
	void sessionSetRemoteSocketResponse(SCD scd,int status);
	void sessionOpenChannelResponse(SCD scd,
		int maxForwardDatalen,int maxReverseDatalen,int status);
	void sessionStartClose(SCD scd);
	STATUS sessionFlushClose(SCD scd);
	void sessionCloseChannelResponse(SCD scd,int status);
	void sessionHandleForwardData(SCD scd);
    public:
	ExBdr *pullForwardData(SCD scd,int maxForwardDatalen);
	void forwardDataResponse(SCD scd,ExBdr *pBdr,int status);
	void reverseDataReceived(SCD scd,ExBdr *pBdr,int status);
    protected:
	void sessionServiceOutput(SCD scd);
	int sessionWrite(SCD scd,unsigned char *data,int datalen);
	void sessionDeactivate(void);

	// Low-level I/O (LLIO) (partially handled by derived class):
    protected:
	static const int LLIO_MAX_POSSIBLE_NAMES=20;	/* TODO: Adjust? */
	static const int LLIO_MAX_DEVICE_ID_MATCHES=5;	/* TODO: Adjust? */
	static const int MAX_TRANSACTIONS=(MAX_TRANSPORT_SESSIONS)*(cMaxWriteIrps);
	static const int LLIO_READ_TIMEOUT=1;
	static const int LLIO_READ_LOOP_TIMEOUT=5;
	enum LlioPollEvent {
		LLIO_POLL_EVENT_OPEN,
		LLIO_POLL_EVENT_CLOSE,
		LLIO_POLL_EVENT_HIT,
		LLIO_POLL_EVENT_MISS
	};      //                       1dcm..u
	static const int LLIO_POLL_RATE_HIGH=     50000;
	static const int LLIO_POLL_RATE_MEDIUM=  100000;
	static const int LLIO_POLL_RATE_LOW=     500000;
	static const int LLIO_POLL_THRESHOLD_MEDIUM=5;
	static const int LLIO_POLL_THRESHOLD_LOW=10;
	ExWatchdogTimer *llioPollTimer;
	char *llioPossibleName[LLIO_MAX_POSSIBLE_NAMES];
	int llioPossibleNameCount;
	char *llioName;
	char *llioOverrideDeviceID;
	char *llioMatchDeviceID[LLIO_MAX_DEVICE_ID_MATCHES];
	int llioMatchDeviceIDCount;
	int llioFd;
	int llioDummyFd;
	enum LlioPollState {
		LLIO_POLL_STATE_HIGH,
		LLIO_POLL_STATE_MEDIUM,
		LLIO_POLL_STATE_LOW
	} llioPollState;
	time_t llioLastHitTime;
	ExBdrQueue *llioForwardBdrQueue;
	unsigned char *llioDeviceID;
    protected:
#ifdef JD_DEBUGLITE
	virtual void llioDump(void);
#endif
	void llioPreInit(void);
	void llioAddPossibleName(char *name);
	void llioAddMatchDeviceID(char *s);
	virtual void llioInit(void) { }
	void llioDone(void);
	STATUS llioOpen(void);
	STATUS llioOpenOne(int *pOpened=0);
	char *llioGetDeviceNode(void) { return llioName; }
	virtual unsigned char *llioGetDeviceID(void)=0;
	virtual STATUS llioSetup(void) { return OK; }
	void llioClose(void);
	virtual void llioReset(void) { }
	void llioService(void);
	void llioSchedulePoll(LlioPollEvent event);
	virtual int llioReverseDataIsAvailable(void);
	virtual STATUS llioForwardToReverse(void) { return OK; }
	virtual STATUS llioReverseToForward(void) { return OK; }
	virtual int llioRead(unsigned char *data,int datalen,int noblock);
	virtual int llioWrite(unsigned char *data,int datalen);
    public:
	void sendForwardData(ExBdr *pBdr,int priority) {
		llioForwardBdrQueue->add(pBdr);
		if (llioFd!=ERROR) fdRegister(llioFd,0,1);
	}
	void resetTransport(ExTransport *pTransport);
};

#endif
