// ExMlcTransport.h

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#ifndef EXMLCTRANSPORT_H
#define EXMLCTRANSPORT_H

#include <bp/ex/transport/ExTransport.h>

class ExMlcTransport;
class ExMlcCommandChannel;

class ExMlcCreditCounter: public ExCreditCounter {
    public:
	ExMlcCreditCounter(int initial=0): ExCreditCounter(initial) {
		setMax(0xFFFF);
	}

	void reset(void) { ExCreditCounter::reset(0); }
};

class ExMlcTransportChannel: public ExTransportChannel {
    public:	// const
	// static const int FLAG_ALLOCATED		=0x0001; (base class)
	// static const int FLAG_OPEN			=0x0002; (base class)
	// static const int FLAG_DEACTIVATING		=0x0004; (base class)
	// static const int FLAG_????			=0x0008; (base class)
	static const int FLAG_LOCAL_OPEN_PENDING		=0x0010;
	static const int FLAG_LOCAL_CLOSE_PENDING		=0x0020;
	static const int FLAG_LOCAL_CREDIT_PENDING		=0x0040;
	static const int FLAG_LOCAL_CREDIT_REQUEST_PENDING	=0x0080;
	static const int FLAG_REMOTE_CLOSE_PENDING		=0x0100;
	static const int FLAG_LOCAL_CLOSE_REQUESTED		=0x0200;

	static const int DEFAULT_FORWARD_CREDIT_REQUEST=16;
	static const int DEFAULT_MAX_OUTSTANDING_CREDIT=0xFFFF;
	static const int DEFAULT_MUSHER_FIRST_CREDIT_REQUEST_DELAY=2;
	static const int DEFAULT_MUSHER_NEXT_CREDIT_REQUEST_DELAY=5;
	static const int DEFAULT_GUSHER_FIRST_CREDIT_REQUEST_DELAY=5;
	static const int DEFAULT_GUSHER_NEXT_CREDIT_REQUEST_DELAY=10;
	static const int DEFAULT_MISER_FIRST_CREDIT_REQUEST_DELAY=NO_WAIT;
	static const int DEFAULT_MISER_NEXT_CREDIT_REQUEST_DELAY=2;
	static const int DEFAULT_GUSHER_PIGGYBACK_CREDIT_COUNT=1;
	static const int DEFAULT_GUSHER_CREDIT_COUNT=2;
	static const int DEFAULT_MISER_CREDIT_REQUEST_COUNT=2;
	static const int DEFAULT_REVERSE_CREDIT_HEARTBEAT_DELAY=1;

    protected:	// const
	enum SourceEnum {
		SOURCE_HANDLE_MSG,
		SOURCE_CANCEL,

		SOURCE_GRAB_FORWARD_CREDIT,
		SOURCE_HANDLE_FORWARD_CREDIT,

		SOURCE_GRAB_REVERSE_CREDIT,
		SOURCE_HANDLE_CREDIT_REPLY,
		SOURCE_CREDIT_RETURNED_BUFFERS
	};

	enum {
		MSG_CREDIT_REQUEST_TIMEOUT=MSG_EX_TRANSPORT_CHANNEL_LAST,
		MSG_CREDIT_HEARTBEAT,
		MSG_EX_MLC_TRANSPORT_CHANNEL_LAST
	};

    protected:
	// Initialized at startup:

	ExMlcTransport *pMlcTransport;
	ExMlcCommandChannel *pCommandChannel;
	int disableCreditCommands;	/* Nonzero iff command channel. */

	ExWatchdogTimer *pForwardCreditRequestTimer;
	ExMsg *pForwardCreditRequestMsg;
	ExWatchdogTimer *pReverseCreditHeartbeatTimer;
	ExMsg *pReverseCreditHeartbeatMsg;

	// Initialized at allocation:

	int musherFirstCreditRequestDelay;
	int musherNextCreditRequestDelay;
	int gusherFirstCreditRequestDelay;
	int gusherNextCreditRequestDelay;
	int miserFirstCreditRequestDelay;
	int miserNextCreditRequestDelay;
	int gusherPiggybackCreditCount;
	int gusherCreditCount;
	int miserCreditRequestCount;
	int workaroundReverseCreditLoss;    /* Workaround DeskJet PML bug. */

	// Initialized at open:

	int maxForwardPacketSize;
	int maxReversePacketSize;

	/* "Forward" and "reverse" refer to the direction of the data,
	 * which is the opposite direction in which the credits travel. */
	int forwardCreditRequest;
	int forwardMaxOutstandingCredit;
	ExMlcCreditCounter forwardCredit;
	int lastCreditRequestGotUsNowhere;
	int reverseMaxOutstandingCredit;
	int reverseBuffersPerPacket;	/* Includes overhead for header. */
	int uncreditedBuffers;
	int reverseCreditToGrant;
	ExMlcCreditCounter reverseCredit;

	ExDebugCounter countSendPiggybackCredit;
	ExCounter      countHandlePiggybackCredit; /* *Not* ExDebugCounter!!! */
	ExDebugCounter countSendCredit;
	ExCounter      countHandleCredit;	/* *Not* ExDebugCounter!!! */
	ExDebugCounter countHandleCreditAfterCreditRequest;
	ExCounter      countSendCreditRequest;	/* *Not* ExDebugCounter!!! */
	ExDebugCounter countHandleCreditRequest;
	ExDebugCounter countSendEmptyCreditRequestReply;
	ExDebugCounter countHandleEmptyCreditRequestReply;

	// Constructor, destructor, reset, dump, handleMsg:
    public:
	ExMlcTransportChannel(ExMlcTransport *pMlcTransport,int channel,
	    int localSocket,ExMgr *pMgr,ExPhysicalPort *pPhysicalPort,
	    ExMlcCommandChannel *pCommandChannel,int final=1);
	virtual ~ExMlcTransportChannel(void);
protected:
	virtual void reset(int rtype);
	virtual void registerStuff(char *title);
    public:
#ifdef JD_DEBUGLITE
	virtual void dump(void);
#endif
    protected:
	virtual void handleMsg(ExMsg *pMsg);

	// Allocate channel:
	virtual void allocate(ExService *pService,SCD scd,
	    int forwardDataPriority,
	    int minBuffersRequired,int benefitOfMoreBuffers,
	    int reverseDataBufferSize);
	virtual void grantBuffers(int count);

	// Get/set local/remote socket:
    public:
	int getPrimarySocket(void);
	int getSecondarySocket(void);
    protected:
	virtual void setRemoteSocket_ts(int remoteSocket,
	    int maxForwardDatalen,int maxReverseDatalen);

	// Open:
    protected:
	virtual void open_ts(int maxForwardDatalen,int maxReverseDatalen);
    public:
	void handleOpenChannelReply(int result,int psid,int ssid,
	    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
	    int maxOutstandingCredit,int credit);
    protected:
	static int packetSizeIsValid(int packetSize);
	static int packetSizesAreValid(int ps1,int ps2) {
		return (packetSizeIsValid(ps1) &&
			packetSizeIsValid(ps2) &&
			(ps1 || ps2));
	}
	void setDatalensFromPacketSizes(void);

	// Close:
    protected:
	void flushCloseChannel(void) {
		if (testFlagsSetClear(FLAG_LOCAL_CLOSE_REQUESTED,0)) {
			close();
		}
	}
	virtual void close_ts(void);
    public:
	void handleCloseChannelReply(int result,int psid,int ssid);
	void handleCloseChannel(int psid,int ssid);
    protected:
	void flushCloseChannelReply(void);

	// BDR management for forward/reverse data flow:
    protected:
	static unsigned char *bdrGetData(ExBdr *pBdr) {
		if (!pBdr) return 0;
		return (unsigned char *)
			(pBdr->getStartAddress()+pBdr->getDataOffset());
	}

	// Forward data flow:
    protected:
	virtual void noMoreForwardData(void);
	enum IsGusher {
		IS_MUSHER=-1,	// Maybe gusher (default).
		IS_MISER=0,	// Probably miser.
		IS_GUSHER=1	// Definitely gusher.
	};
	IsGusher isGusher(void);
	virtual STATUS addOverhead(ExBdr **ppBdr);
	virtual void removeOverhead(ExBdr **ppBdr);
	virtual int grabForwardCredit(ExBdr *pBdr);
	void scheduleCreditRequest(SourceEnum source);
    public:
	void handleCreditRequestReply(int result,int psid,int ssid,
	    int credit,int cmdinfo);
	void handleCredit(int psid,int ssid,int credit,int cmdinfo);
    protected:
	virtual int handlePiggybackCreditPart1(int credit,
	    unsigned char *data,int datalen,int cmdinfo);
	STATUS handleForwardCredit(int credit,int cmdinfo) {
		return handleForwardCreditPart2(
			handleForwardCreditPart1(credit,cmdinfo));
	}
	virtual int handleForwardCreditPart1(int credit,int cmdinfo);
	STATUS handleForwardCreditPart2(int part1Result);

	// Reverse data flow:
    public:
	virtual void reverseDataReceived_ts(ExBdr *pBdr,int status);
    protected:
	virtual int grabReverseCredit(unsigned char *data,int datalen);
    public:
	virtual void returnBufferNotification(ExBdr *pBdr);
    protected:
	void creditReturnedBuffers(int returnedBuffers=0);
	void uncreditCreditedBuffers(void);
	virtual int grantReverseCredit(int max=0xFFFF,
	    unsigned char *data=0,int datalen=0);
	virtual int prepareToSendPiggybackCredit(int credit);
	void ungrantReverseCredit(void);
	void scheduleCredit(SourceEnum source);
    public:
	void handleCreditReply(int result,int psid,int ssid);
	void handleCreditRequest(int psid,int ssid,
	    int creditRequested,int maxOutstandingCredit);
	int getWorkaroundReverseCreditLoss(void) {
		return workaroundReverseCreditLoss;
	}
	/* The PML service may want to enable this for DeskJet 8xx printers. */
	void setWorkaroundReverseCreditLoss(int workaroundReverseCreditLoss) {
		SETTHIS(workaroundReverseCreditLoss);
	}
};


class ExMlcCommandChannel: public ExMlcTransportChannel {
    public:
	static const int COMMAND_SOCKET=0;

    protected:
	enum {
		MSG_COMMAND_REPLY_TIMEOUT=MSG_EX_MLC_TRANSPORT_CHANNEL_LAST,
		MSG_EX_MLC_COMMAND_CHANNEL_LAST
	};

	static const int COMMAND_REPLY_TIMEOUT=15;	/* P68 was 30 seconds. */
	static const int MAX_COMMAND_PACKET_LENGTH=64;
	static const int MAX_SERVICE_NAME_LENGTH=40;
	static const int COMMAND_MAX_OUTSTANDING_CREDIT=1;

	static const int CMD_INIT			=0x00;
	static const int CMD_INIT_REPLY				=0x80;
	static const int CMD_OPEN_CHANNEL		=0x01;
	static const int CMD_OPEN_CHANNEL_REPLY			=0x81;
	static const int CMD_CLOSE_CHANNEL		=0x02;
	static const int CMD_CLOSE_CHANNEL_REPLY		=0x82;
	static const int CMD_CREDIT			=0x03;
	static const int CMD_CREDIT_REPLY			=0x83;
	static const int CMD_CREDIT_REQUEST		=0x04;
	static const int CMD_CREDIT_REQUEST_REPLY		=0x84;
	static const int CMD_CONFIG_SOCKET		=0x07;
	static const int CMD_CONFIG_SOCKET_REPLY		=0x87;
	static const int CMD_EXIT			=0x08;
	static const int CMD_EXIT_REPLY				=0x88;
	static const int CMD_GET_SOCKET_ID		=0x09;
	static const int CMD_GET_SOCKET_ID_REPLY		=0x89;
	static const int CMD_GET_SERVICE_NAME		=0x0A;
	static const int CMD_GET_SERVICE_NAME_REPLY		=0x8A;
	static const int CMD_ERROR			=0x7F;
	static const int REPLY_BIT			=0x80;

	struct CmdRequest {
		unsigned char command;
	};
	static const int LEN_REQUEST			=0x01;
	struct CmdReply {
		unsigned char command;
		unsigned char result;
	};
	static const int LEN_REPLY			=0x02;

	struct CmdInit {
		unsigned char command;
		unsigned char revision;
	};
	static const int LEN_INIT			=0x02;
	struct CmdInitReply {
		unsigned char command;
		unsigned char result;
		unsigned char revision;
	};
	static const int LEN_INIT_REPLY			=0x03;

	struct CmdOpenChannel {
		unsigned char command;
		unsigned char psid;
		unsigned char ssid;
		union {
		    struct {
			unsigned char credit[2];
		    } mlc;
		    struct {
			unsigned char maxPriToSecPacketSize[2];
			unsigned char maxSecToPriPacketSize[2];
			unsigned char maxOutstandingCredit[2];
		    } dot4;
		};
	};
	static const int MLC_LEN_OPEN_CHANNEL		=0x05;
	static const int DOT4_LEN_OPEN_CHANNEL		=0x09;
	struct CmdOpenChannelReply {
		unsigned char command;
		unsigned char result;
		union {
		    struct {
			unsigned char credit[2];
		    } mlc;
		    struct {
			unsigned char psid;
			unsigned char ssid;
			unsigned char maxPriToSecPacketSize[2];
			unsigned char maxSecToPriPacketSize[2];
			unsigned char maxOutstandingCredit[2];
			unsigned char credit[2];
		    } dot4;
		};
	};
	static const int MLC_LEN_OPEN_CHANNEL_REPLY	=0x04;
	static const int DOT4_LEN_OPEN_CHANNEL_REPLY	=0x0C;

	struct CmdCloseChannel {
		unsigned char command;
		unsigned char psid;
		unsigned char ssid;
	};
	static const int LEN_CLOSE_CHANNEL		=0x03;
	struct CmdCloseChannelReply {
		unsigned char command;
		unsigned char result;
		union {
		    struct {
			unsigned char psid;
			unsigned char ssid;
		    } dot4;
		};
	};
	static const int MLC_LEN_CLOSE_CHANNEL_REPLY	=0x02;
	static const int DOT4_LEN_CLOSE_CHANNEL_REPLY	=0x04;

	struct CmdCredit {
		unsigned char command;
		unsigned char psid;
		unsigned char ssid;
		unsigned char credit[2];
	};
	static const int LEN_CREDIT			=0x05;
	struct CmdCreditReply {
		unsigned char command;
		unsigned char result;
		union {
		    struct {
			unsigned char psid;
			unsigned char ssid;
		    } dot4;
		};
	};
	static const int MLC_LEN_CREDIT_REPLY		=0x02;
	static const int DOT4_LEN_CREDIT_REPLY		=0x04;

	struct CmdCreditRequest {
		unsigned char command;
		unsigned char psid;
		unsigned char ssid;
		union {
		    struct {
			unsigned char creditRequested[2];
		    } mlc;
		    struct {
			unsigned char maxOutstandingCredit[2];
		    } dot4;
		};
	};
	static const int LEN_CREDIT_REQUEST		=0x05;
	struct CmdCreditRequestReply {
		unsigned char command;
		unsigned char result;
		union {
		    struct {
			unsigned char credit[2];
		    } mlc;
		    struct {
			unsigned char psid;
			unsigned char ssid;
			unsigned char credit[2];
		    } dot4;
		};
	};
	static const int MLC_LEN_CREDIT_REQUEST_REPLY	=0x04;
	static const int DOT4_LEN_CREDIT_REQUEST_REPLY	=0x06;

	struct CmdConfigSocket {
		unsigned char command;
		unsigned char socketID;
		unsigned char maxPriToSecPacketSize[2];
		unsigned char maxSecToPriPacketSize[2];
		unsigned char statusLevel;
	};
	static const int LEN_CONFIG_SOCKET		=0x07;
	struct CmdConfigSocketReply {
		unsigned char command;
		unsigned char result;
		unsigned char maxPriToSecPacketSize[2];
		unsigned char maxSecToPriPacketSize[2];
		unsigned char statusLevel;
	};
	static const int LEN_CONFIG_SOCKET_REPLY	=0x07;

	struct CmdExit {
		unsigned char command;
	};
	static const int LEN_EXIT			=0x01;
	struct CmdExitReply {
		unsigned char command;
		unsigned char result;
	};
	static const int LEN_EXIT_REPLY			=0x02;

	struct CmdGetSocketID {
		unsigned char command;
		/*unsigned*/ char serviceName[MAX_SERVICE_NAME_LENGTH+1];
	};
	static const int LEN_GET_SOCKET_ID		=0x01;
	struct CmdGetSocketIDReply {
		unsigned char command;
		unsigned char result;
		unsigned char socketID;
		/*unsigned*/ char serviceName[MAX_SERVICE_NAME_LENGTH+1];
	};
	static const int LEN_GET_SOCKET_ID_REPLY	=0x03;

	struct CmdGetServiceName {
		unsigned char command;
		unsigned char socketID;
	};
	static const int LEN_GET_SERVICE_NAME		=0x02;
	struct CmdGetServiceNameReply {
		unsigned char command;
		unsigned char result;
		unsigned char socketID;
		/*unsigned*/ char serviceName[MAX_SERVICE_NAME_LENGTH+1];
	};
	static const int LEN_GET_SERVICE_NAME_REPLY	 =0x03;

	struct CmdError {
		unsigned char command;
		union {
		    struct {
			unsigned char error;
		    } mlc;
		    struct {
			unsigned char psid;
			unsigned char ssid;
			unsigned char error;
		    } dot4;
		};
	};
	static const int MLC_LEN_ERROR			=0x02;
	static const int DOT4_LEN_ERROR			=0x04;

	union CommandUnion {
		CmdRequest			request;
		CmdReply			reply;
		struct {
			char pad[MAX_COMMAND_PACKET_LENGTH+1];
			int consumeCredit;
			int grantCredit;
			int psid;
			int ssid;
			int closeAfterSend;
		} options;

		CmdInit				init;
		CmdInitReply			initReply;
		CmdOpenChannel			openChannel;
		CmdOpenChannelReply		openChannelReply;
		CmdCloseChannel			closeChannel;
		CmdCloseChannelReply		closeChannelReply;
		CmdCredit			credit;
		CmdCreditReply			creditReply;
		CmdCreditRequest		creditRequest;
		CmdCreditRequestReply		creditRequestReply;
		CmdConfigSocket			configSocket;
		CmdConfigSocketReply		configSocketReply;
		CmdExit				exit;
		CmdExitReply			exitReply;
		CmdGetSocketID			getSocketID;
		CmdGetSocketIDReply		getSocketIDReply;
		CmdGetServiceName		getServiceName;
		CmdGetServiceNameReply		getServiceNameReply;
		CmdError			error;
	};

    protected:
	// Initialized at startup:

	ExBufferPool *pForwardCommandPool;
	ExBdrQueue *pForwardNonconsumingQueue;
	ExBdrQueue *pForwardReplyQueue;
	ExBdrQueue *pForwardRequestQueue;
	ExCountingWatchdogTimer *pCommandReplyTimer;
	ExMsg *pCommandReplyTimeoutMsg;

	// Initialized at activation (also open):

	ExMlcCreditCounter forwardRequestCredit;

	int allowErrorPackets;
	int lastPsid;
	int lastSsid;

	// Constructor, destructor, dump, handleMsg:
    public:
	ExMlcCommandChannel(ExMlcTransport *pMlcTransport,int channel,
	    ExMgr *pMgr,ExPhysicalPort *pPhysicalPort,
	    int channelCount,int final=1);
	int calculateForwardCommandPoolSize(int channelCount) {
		int factor=4;
		int size=(channelCount)*factor;
#ifndef JD_DEBUGLITE
		/* Add a few extra buffers for non-DEBUG, just in case. */
		int extra=factor;
		if (extra<channelCount) extra=channelCount;
		size+=extra;
#endif
		return size;
	}
	virtual ~ExMlcCommandChannel(void);
protected:
	void flushQueue(ExBdrQueue *pQueue);
	virtual void reset(int rtype);
	virtual void registerStuff(char *title);
    public:
#ifdef JD_DEBUGLITE
	virtual void dump(void);
#endif
protected:
	virtual void handleMsg(ExMsg *pMsg);

	// Allocate channel:
	virtual void allocate(ExService *pService,SCD scd,
	    int forwardDataPriority,
	    int minBuffersRequired,int benefitOfMoreBuffers,
	    int reverseDataBufferSize);

	// BDR management for forward/reverse data flow:
    protected:
	static CommandUnion *bdrGetCommandUnion(ExBdr *pBdr) {
		return (CommandUnion *)bdrGetData(pBdr);
	}

	// Routines to pack and enqueue forward commands:
    public:
	void sendInit(int revision);
	void sendInitReply(int result,int revision);
	void sendOpenChannel(int psid,int ssid,
	    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
	    int maxOutstandingCredit,int credit);
	void sendOpenChannelReply(int result,int psid,int ssid,
	    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
	    int maxOutstandingCredit,int credit);
	void sendCloseChannel(int psid,int ssid);
	void sendCloseChannelReply(int result,int psid,int ssid);
	void sendCredit(int psid,int ssid,int credit);
	void sendCreditReply(int result,int psid,int ssid);
	void sendCreditRequest(int psid,int ssid,int creditRequested,
	    int maxOutstandingCredit);
	void sendCreditRequestReply(int result,int psid,int ssid,int credit);
	void sendConfigSocket(int socketID,
	    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
	    int statusLevel=0);
	void sendConfigSocketReply(int result,int socketID,
	    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
	    int statusLevel=0);
	void sendExit(void);
	void sendExitReply(int result);
	void sendGetSocketID(char *serviceName);
	void sendGetSocketIDReply(int result,int socketID,char *serviceName);
	void sendGetServiceName(int socketID);
	void sendGetServiceNameReply(int result,int socketID,char *serviceName);
	void mlcSendError(int error);
	void dot4SendError(int psid,int ssid,int error);
	void sendError(int psid,int ssid,int mlcError,int dot4Error);
	void sendGenericReply(int command,int result);

	// Forward data flow:
    protected:
	void getEmptyBuffer(ExBdr **ppBdr,CommandUnion **pData);
	void putFullBuffer(ExBdr *pBdr,CommandUnion *data);
	virtual ExBdr *servicePullForwardData(void);
	virtual int grabForwardCredit(ExBdr *pBdr);
	virtual int handlePiggybackCreditPart1(int credit,
	    unsigned char *data,int datalen,int cmdinfo);
	virtual int handleForwardCreditPart1(int credit,int cmdinfo);
	virtual void serviceForwardDataResponse(ExBdr *pBdr,int status);

	// Reverse data flow:
    protected:
	virtual int grabReverseCredit(unsigned char *data,int datalen);
	virtual int grantReverseCredit(int max=0xFFFF,
	    unsigned char *data=0,int datalen=0);
	virtual int prepareToSendPiggybackCredit(int credit);
	virtual void serviceReverseDataReceived(ExBdr *pBdr,int status,
	    unsigned char *data,int datalen);
	STATUS checkDatalen(CommandUnion *data,int datalen,int expectedDatalen);

	// Routines to unpack and route reverse commands:
    protected:
	void handleInit			(CommandUnion *data,int datalen);
	STATUS handleInitReply		(CommandUnion *data,int datalen);
	void handleOpenChannel		(CommandUnion *data,int datalen);
	void handleOpenChannelReply	(CommandUnion *data,int datalen);
	void handleCloseChannel		(CommandUnion *data,int datalen);
	void handleCloseChannelReply	(CommandUnion *data,int datalen);
	void handleCredit		(CommandUnion *data,int datalen);
	void handleCreditReply		(CommandUnion *data,int datalen);
	void handleCreditRequest	(CommandUnion *data,int datalen);
	void handleCreditRequestReply	(CommandUnion *data,int datalen);
	void handleConfigSocket		(CommandUnion *data,int datalen);
	void handleConfigSocketReply	(CommandUnion *data,int datalen);
	void handleExit			(CommandUnion *data,int datalen);
	void handleExitReply		(CommandUnion *data,int datalen);
	void handleGetSocketID		(CommandUnion *data,int datalen);
	void handleGetSocketIDReply	(CommandUnion *data,int datalen);
	void handleGetServiceName	(CommandUnion *data,int datalen);
	void handleGetServiceNameReply	(CommandUnion *data,int datalen);
	void handleError		(CommandUnion *data,int datalen);
};

class ExMlcTransport: public ExTransport {
    public:	// static const
	static const int FIRST_DYNAMIC_SOCKET=0x80;

	/* The code depends on NONE>DOT4>PRE_DOT4>MLC. */
	static const int REVISION_NONE		=0xFFFF;
	static const int REVISION_DOT4		=0x20;
	static const int REVISION_PRE_DOT4	=0x10;
	static const int REVISION_MLC		=0x03;

	static const int RESULT_SUCCESS				=0x00;

	static const int MLC_RESULT_TOO_MANY_CONNECTIONS	=0x01;
	static const int MLC_RESULT_INSUFFICIENT_RESOURCES	=0x02;
	static const int MLC_RESULT_BAD_SOCKET			=0x01;
	static const int MLC_RESULT_UNKNOWN_COMMAND		=0xFF;

	static const int MLC_ERROR_BAD_COMMAND_PACKET_LENGTH	=0x00;
	static const int MLC_ERROR_RECEIVED_UNCREDITED_PACKET	=0x01;
	static const int MLC_ERROR_REPLY_WITHOUT_REQUEST	=0x02;
	static const int MLC_ERROR_CREDIT_OVERFLOW		=0x03;
	static const int MLC_ERROR_NOT_INITIALIZED		=0x04;
	static const int MLC_ERROR_PACKET_TOO_BIG		=0x05;
	static const int MLC_ERROR_DATA_LINK_ERROR		=0x06;
	static const int MLC_ERROR_UNOPENED_CHANNEL_DATA	=0x07;
	static const int MLC_ERROR_DUPLICATE_CONFIG_SOCKET	=0x08;
	static const int MLC_ERROR_UNOPENED_CHANNEL_CLOSE	=0x09;
	static const int MLC_ERROR_OPEN_UNCONFIG_SOCKET		=0x0A;
	static const int MLC_ERROR_UNOPENED_CHANNEL_CREDIT	=0x0B;
	static const int MLC_ERROR_UNKNOWN_RESULT_VALUE		=0x0C;
	static const int MLC_ERROR_SOCKET_DOMAIN_MISMATCH	=0x0D;
	static const int MLC_ERROR_INVALID_PACKET_SIZE		=0x0E;
	static const int MLC_ERROR_SOCKET_ALREADY_OPEN		=0x0F;
	static const int MLC_ERROR_INVALID_ENTER_COMMAND	=0x10;
	static const int MLC_ERROR_INVALID_REVISION		=0x20;

	static const int DOT4_RESULT_UNABLE_TO_INITIALIZE	=0x01;
	static const int DOT4_RESULT_INVALID_REVISION		=0x02;
	static const int DOT4_RESULT_COMMAND_CHANNEL_CLOSE	=0x03;
	static const int DOT4_RESULT_INSUFFICIENT_RESOURCES	=0x04;
	static const int DOT4_RESULT_CONNECTION_REFUSED		=0x05;
	static const int DOT4_RESULT_CHANNEL_ALREADY_OPEN	=0x06;
	static const int DOT4_RESULT_CREDIT_OVERFLOW		=0x07;
	static const int DOT4_RESULT_CHANNEL_NOT_OPEN		=0x08;
	static const int DOT4_RESULT_NO_SERVICE_ON_SOCKET	=0x09;
	static const int DOT4_RESULT_SERVICE_LOOKUP_FAILED	=0x0A;
	static const int DOT4_RESULT_SIMULTANEOUS_INIT		=0x0B;
	static const int DOT4_RESULT_INVALID_PACKET_SIZE	=0x0C;
	static const int DOT4_RESULT_BOTH_PACKET_SIZES_ZERO	=0x0D;
	static const int DOT4_RESULT_UNSUPPORTED_CREDIT_MODE	=0x0E;

	static const int DOT4_ERROR_MALFORMED_PACKET			=0x80;
	static const int DOT4_ERROR_RECEIVED_UNCREDITED_PACKET		=0x81;
	static const int DOT4_ERROR_REPLY_WITHOUT_REQUEST		=0x82;
	static const int DOT4_ERROR_PACKET_TOO_BIG			=0x83;
	static const int DOT4_ERROR_UNOPENED_CHANNEL_DATA		=0x84;
	static const int DOT4_ERROR_UNKNOWN_RESULT_VALUE		=0x85;
	static const int DOT4_ERROR_PIGGYBACKED_CREDIT_OVERFLOW		=0x86;
	static const int DOT4_ERROR_UNKNOWN_COMMAND			=0x87;
	static const int DOT4_ERROR_PACKET_DIRECTION_ZERO_DATALEN	=0x88;

	enum {
		REASON_REPLY_TIMEOUT=REASON_EX_TRANSPORT_LAST,
		REASON_INIT_REVISION_REFUSED,
		REASON_RECEIVED_UNSUCCESSFUL_REPLY,
		REASON_RECEIVED_EXIT_REPLY,
		REASON_RECEIVED_ERROR,
		REASON_SENT_ERROR,
		REASON_SENT_EXIT_REPLY,
		REASON_OPEN_CHANNEL_FAILED,
		REASON_EX_MLC_TRANSPORT_LAST
	};

	struct Header {
		unsigned char psid;
		unsigned char ssid;
		unsigned char length[2];
		unsigned char credit;
		unsigned char control;
	};
	static const int LEN_HEADER			=0x06;

	static const int REMSOCK_STATE_UNCONFIGURED=0;
	static const int REMSOCK_STATE_CONFIGURING=1;
	static const int REMSOCK_STATE_CONFIGURED=2;

    protected:	// static const
	static const int GRC_STATE_READY_FOR_HEADER=0;
	static const int GRC_STATE_READY_FOR_DATA=1;

	static const int DEFAULT_MLC_FORWARD_DATA_TIMEOUT=15;
	static const int DEFAULT_TRY_REVISION_MLC=1;
	static const int DEFAULT_TRY_REVISION_PRE_DOT4=0;
	static const int DEFAULT_TRY_REVISION_DOT4=1;
	static const int DEFAULT_MAX_REMOTE_SOCKETS=16;

	enum {
		MSG_EX_MLC_TRANSPORT_LAST=MSG_EX_TRANSPORT_LAST
	};

	struct RemoteSocket {
		unsigned char state;
		unsigned char socketID;
		unsigned short maxForwardPacketSize;
		unsigned short maxReversePacketSize;
	};

    protected:
	ExBufferPool *pForwardHeaderPool;

	ExMlcTransportChannel **mlcChannelArray;
	ExMlcCommandChannel *pCommandChannel;

	int grcState;
	int reverseDataStopped;

	int tryDot4;
	int tryPreDot4;
	int tryMlc;
	int requestedRevision;
	int revision;

	int maxRemoteSockets;
	RemoteSocket *remoteSocketArray;

	// Constructor, destructor, dump:
    public:
	ExMlcTransport(ExMgr *pMgr,ExPhysicalPort *pPhysicalPort,
	    int forwardTransactionCount,int channelCount,int final=1);
    protected:
	virtual void initChannelArray(void);
	virtual ExTransportChannel *newChannel(int channel);
    public:
	virtual ~ExMlcTransport(void);
    protected:
	virtual void reset(int rtype);
	virtual void registerStuff(char *title);
    public:
#ifdef JD_DEBUGLITE
	virtual void dump(void);
#endif

	// Get/set revision:
    protected:
	int getRevision(void) {
		return revision;
	}
	void setRevision(int value) {
		revision=value;
	}
	void unsetRevision(void) {
		requestedRevision=revision=REVISION_NONE;
	}
    public:
	int revisionIsSet(void) {
		return (revision!=REVISION_NONE);
	}
	int revisionIsMlc(void) {
		int r=revision;
		if (r==REVISION_NONE) r=requestedRevision;
		return (r==REVISION_MLC);
	}
	int revisionIsDot4(void) {
		int r=revision;
		if (r==REVISION_NONE) r=requestedRevision;
		return (r==REVISION_DOT4 || r==REVISION_PRE_DOT4);
	}

	// Convert between local/remote and primary/secondary sockets:
	/* Always call these routines when converting between
	 * local/remote and primary/secondary socket ID pairs.
	 * That way, it will be a lot easier to add support
	 * for allowing the peripheral to be the primary peer. */
	int pickLocalSocket(int psid,int ssid) {
		return psid;
	}
	int pickRemoteSocket(int psid,int ssid) {
		return ssid;
	}
	int pickPrimarySocket(int localSocket,int remoteSocket) {
		return localSocket;
	}
	int pickSecondarySocket(int localSocket,int remoteSocket) {
		return remoteSocket;
	}

	// Activate:
    protected:
	virtual void transportResetComplete_ts(void);
	void sendInit(int nextRevision=REVISION_NONE);
    public:
	STATUS handleInitReply(int result,int revision);
	void handleInit(int revision);

	// Deactivate:
    public:
	void handleExit(void);
	void handleExitReply(int result);
	void handleError(int psid,int ssid,int error);

	// Remote socket lookup:
    public:
	virtual void lookupRemoteSocket_ts(ExLookup *pLookupRequest);
	void handleGetSocketIDReply(int result,int socketID,
	    char *serviceName);
	void handleGetServiceNameReply(int result,int socketID,
	    char *serviceName);
    protected:
	void handleLookupReply(int lastSet,int result,
	    int socketID,char *serviceName);
    public:
	void handleGetSocketID(char *serviceName);
	void handleGetServiceName(int socketID);

	// Set remote socket:
    protected:
	void remsockReset(void);
    public:
	STATUS remsockLookup(int socketID,int *pState,
	    int *pMaxForwardPacketSize,int *pMaxForwardPacketSize);
	STATUS remsockAdd(int socketID);
	void handleConfigSocketReply(int result,int socketID,
	    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
	    int statusLevel);
    protected:
	STATUS remsockUpdate(int socketID,
	    int maxForwardPacketSize,int maxForwardPacketSize);
	STATUS remsockDelete(int socketID);

	// Peripheral-initiated opens (not supported):
    public:
	void handleOpenChannel(int psid,int ssid,
	    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
	    int maxOutstandingCredit,int credit);
	void handleConfigSocket(int socketID,
	    int maxPriToSecPacketSize,int maxSecToPriPacketSize,
	    int statusLevel);

	// Forward data flow:
    public:
	ExBdr *getHeaderBuffer(Header **ppHeader) {
		ExBdr *pBdr=pForwardHeaderPool->getBuffer();
		if (pBdr) *ppHeader=bdrGetHeader(pBdr);
		return pBdr;
	}
	void returnHeaderBuffer(ExBdr *pBdr) {
		pForwardHeaderPool->returnBuffer(pBdr);
	}

	// Reverse data flow:
    public:
	void stopReverseData(void) { reverseDataStopped=1; }
	virtual void resetReverseCount(void);
	virtual STATUS getReverseCount(char *lastData,int lastLength,
	    int *pCount,int *pFlags);

    protected:
	virtual ExTransportChannel *lookupChannel(ExBdr *pBdr);
    public:
	static Header *bdrGetHeader(ExBdr *pBdr) {
		return (Header *)
			(pBdr->getStartAddress()+pBdr->getDataOffset());
	}
	ExMlcTransportChannel *lookupChannel(int psid,int ssid);
};

inline int ExMlcTransportChannel::getPrimarySocket(void) {
	return pMlcTransport->pickPrimarySocket(localSocket,remoteSocket);
}

inline int ExMlcTransportChannel::getSecondarySocket(void) {
	return pMlcTransport->pickSecondarySocket(localSocket,remoteSocket);
}

inline int ExMlcTransportChannel::packetSizeIsValid(int packetSize) {
	return (packetSize==0 ||
		(packetSize>=ExMlcTransport::LEN_HEADER &&
		 packetSize<=0xFFFF));
}

#endif
