/* ptal-hpjd -- PTAL provider for HP JetDirect print servers */

/* Copyright (C) 2000-2001 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

/* Original author: David Paschal */

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>

#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <netdb.h>
#include <stdlib.h>

#ifdef HAVE_SNMP
#include <asn1.h>
#include <snmp.h>
#include <snmp_api.h>
#include <snmp_client.h>
#endif

#include "ptal-internal.h"

#define HPJD_TCPPORT_PRINT	9100
#define HPJD_TCPPORT_SCAN	9290
#define HPJD_TCPPORT_GENERIC	9220


typedef struct ptalHpjdDevice_s {
	struct ptalDevice_s base;

	/* Just hostname (no colon-portnum), dynamically allocated! */
	char *hostname;

	/* dev->saddr.sin_family=AF_INET;
	 * dev->saddr.sin_addr
	 * (dev->saddr.sin_port filled in for channel) */
	struct sockaddr_in saddr;

	/* JetDirect port number (1,2,3). */
	int port;

#ifdef HAVE_SNMP
	/* SNMP session state. */
	char snmpCommunityName[32];	/* TODO: Make dynamic! */
	struct snmp_session *snmpSession;
#endif
	ptalPmlObject_t objDeviceID;
} *ptalHpjdDevice_t;

typedef struct ptalHpjdProbe_s {
	char *hostname;

	struct sockaddr_in saddr;
	int port;

	ptalHpjdDevice_t dev;
} *ptalHpjdProbe_t;

typedef struct ptalHpjdChannel_s {
	struct ptalChannel_s base;

	/* ch->saddr.sin_port=htons(___); */
	struct sockaddr_in saddr;
} *ptalHpjdChannel_t;

extern struct ptalProvider_s ptalHpjdProvider;

int ptalHpjdSnmpOpen(ptalHpjdDevice_t dev) {
#ifndef HAVE_SNMP
	return PTAL_ERROR;
#else
	struct snmp_session preSession;
	snmp_sess_init(&preSession);
	preSession.version=SNMP_VERSION_1;

	preSession.peername=dev->hostname;
	sprintf(dev->snmpCommunityName,"public.%d",dev->port);
	preSession.community=dev->snmpCommunityName;
	preSession.community_len=strlen(dev->snmpCommunityName);

	dev->snmpSession=snmp_open(&preSession);
	if (!dev->snmpSession) {
		PTAL_LOG_ERROR("ptalHpjdSnmpOpen(dev=0x%8.8X): "
			"snmp_open failed!\n",dev);
		return PTAL_ERROR;
	}

	dev->objDeviceID=ptalPmlAllocateID(&dev->base,
		"\xFF\x01\x03\x06\x01\x04\x01\x0B\x02\x03\x09\x01\x01\x07");

	return PTAL_OK;
#endif
}

int ptalHpjdSnmpClose(ptalHpjdDevice_t dev) {
#ifndef HAVE_SNMP
	return PTAL_ERROR;
#else
	if (!dev->snmpSession) {
		return PTAL_ERROR;
	}

	snmp_close(dev->snmpSession);
	dev->snmpSession=0;
	return PTAL_OK;
#endif
}

int ptalHpjdHostToAddr(char *hostname,struct sockaddr_in *saddr) {
	struct hostent *phe;

	if ((phe=gethostbyname(hostname))!=0) {
		/* h_addr only gives us the first entry if there are more
		 * than one.  Oh well... */
		bcopy(phe->h_addr,(char *)(&saddr->sin_addr.s_addr),
			phe->h_length);

	} else if ((saddr->sin_addr.s_addr=inet_addr(hostname))==-1) {
		PTAL_LOG_ERROR("ptalHpjdHostToAddr(hostname=<%s>): "
			"lookup failed!\n",hostname);
		return PTAL_ERROR;
	}

	return PTAL_OK;
}

void ptalHpjdAddrCopy(struct sockaddr_in *dest,struct sockaddr_in *src) {
	memcpy(dest,src,sizeof(struct sockaddr_in));
}

int ptalHpjdAddrCompare(struct sockaddr_in *saddr1,struct sockaddr_in *saddr2) {
	return saddr1->sin_addr.s_addr-saddr2->sin_addr.s_addr;
}

int ptalHpjdDevCompareCallback(ptalDevice_t _dev,void *_cbd) {
	ptalHpjdDevice_t dev=(ptalHpjdDevice_t)_dev;
	ptalHpjdProbe_t cbd=(ptalHpjdProbe_t)_cbd;

	/* PTAL_LOG_DEBUG("ptalHpjdDevCompareCallback: "
		"dev=0x%8.8X, dev->saddr=0x%8.8X, dev->port=%d, "
		"cbd->saddr=0x%8.8X, cbd->port=%d.\n",
		dev,dev->saddr.sin_addr.s_addr,dev->port,
		cbd->saddr.sin_addr.s_addr,cbd->port); */

	if (!ptalHpjdAddrCompare(&dev->saddr,&cbd->saddr) &&
	    dev->port==cbd->port) {
		if (cbd->dev) {

			return 0;
		}
		cbd->dev=dev;
		return 1;
	}

	return 0;
}

ptalDevice_t ptalHpjdDeviceOpen(char *name) {
	ptalDevice_t dev=0;
	char *p;
	struct ptalHpjdProbe_s cbd;
	int size;

	if (!name) {
		/* TODO: Device discovery? */
		PTAL_LOG_ERROR("ptalHpjdDeviceOpen(name=NULL): "
			"discovery not supported!\n");
		goto done;
	}

	size=strlen(name)+1;
	cbd.hostname=malloc(size);
	if (!cbd.hostname) {
		PTAL_LOG_ERROR("ptalHpjdDeviceOpen(name=<%s>): "
			"malloc(name=%d) failed!\n",name,size);
		goto done;
	}
	memcpy(cbd.hostname,name,size);
	cbd.hostname[size-1]=0;
	p=strchr(cbd.hostname,':');
	cbd.port=1;
	if (p) {
		*p=0;
		cbd.port=atoi(p+1);
	}

	cbd.saddr.sin_family=AF_INET;
	if (ptalHpjdHostToAddr(cbd.hostname,&cbd.saddr)==PTAL_ERROR) {
		goto done;
	}

	/* TODO: If no port specified, then probe JetDirect for number
	 * of ports. */

	cbd.dev=0;

	ptalDeviceEnumerate(&ptalHpjdProvider,ptalHpjdDevCompareCallback,&cbd);

	if (cbd.dev) {
		PTAL_LOG_DEBUG("ptalHpjdDeviceOpen(name=<%s>): "
			"found matching dev=0x%8.8X.\n",name,cbd.dev);
		dev=&cbd.dev->base;
		goto done;
	}

	cbd.dev=(ptalHpjdDevice_t)ptalDeviceAdd(&ptalHpjdProvider,name,&cbd);
	if (!cbd.dev) {
		goto done;
	}

	dev=&cbd.dev->base;
done:
	if (cbd.hostname) free(cbd.hostname);
	return dev;
}

void ptalHpjdDeviceConstructor(ptalDevice_t _dev,void *_cbd) {
	ptalHpjdDevice_t dev=(ptalHpjdDevice_t)_dev;
	ptalHpjdProbe_t cbd=(ptalHpjdProbe_t)_cbd;

	dev->hostname=cbd->hostname;
	cbd->hostname=0;
	ptalHpjdAddrCopy(&dev->saddr,&cbd->saddr);
	dev->port=cbd->port;

	ptalHpjdSnmpOpen(dev);
}

void ptalHpjdDeviceDestructor(ptalDevice_t _dev) {
	ptalHpjdDevice_t dev=(ptalHpjdDevice_t)_dev;

	ptalHpjdSnmpClose(dev);
}

void ptalHpjdDeviceDump(ptalDevice_t _dev,int level) {
	ptalHpjdDevice_t dev=(ptalHpjdDevice_t)_dev;

	fprintf(stderr,"hostname=0x%8.8X <%s>\n",
		(int)dev->hostname,dev->hostname);
	fprintf(stderr,"saddr=0x%8.8X\n",
		dev->saddr.sin_addr.s_addr);
	fprintf(stderr,"port=%d\n",
		dev->port);
#ifdef HAVE_SNMP
	fprintf(stderr,"snmpCommunityName=<%s>\n",
		dev->snmpCommunityName);
	fprintf(stderr,"snmpSession=0x%8.8X\n",
		(int)dev->snmpSession);
#endif
}

int ptalHpjdDeviceGetDeviceIDString(ptalDevice_t _dev,char *buffer,int maxlen) {
	ptalHpjdDevice_t dev=(ptalHpjdDevice_t)_dev;
	if (!dev->objDeviceID) {

		return PTAL_ERROR;
	}
	if (ptalPmlRequestGet(dev->objDeviceID,0)!=PTAL_OK) {

		return PTAL_ERROR;
	}
	return ptalPmlGetValue(dev->objDeviceID,0,buffer,maxlen);
}

void ptalHpjdChannelConstructor(ptalChannel_t _chan) {
	ptalHpjdChannel_t chan=(ptalHpjdChannel_t)_chan;
	ptalHpjdDevice_t dev=(ptalHpjdDevice_t)chan->base.dev;

	ptalHpjdAddrCopy(&chan->saddr,&dev->saddr);
}

void ptalHpjdChannelDump(ptalChannel_t _chan,int level) {
	ptalHpjdChannel_t chan=(ptalHpjdChannel_t)_chan;

	fprintf(stderr,"saddr=0x%8.8X\n",
		chan->saddr.sin_addr.s_addr);
}

int ptalHpjdSgwParse(ptalChannel_t chan) {
	char buffer[2];
	struct timeval timeout;
	int r;

	timeout.tv_sec=1;
	timeout.tv_usec=0;

	r=ptalChannelReadTimeout(chan,buffer,2,&timeout,&timeout);
	if (r!=2) {
		PTAL_LOG_ERROR("ptalHpjdSgwParse(chan=0x%8.8X): "
			"ptalChannelReadTimeout returns %d, expected=2!\n",
			chan,r);
		return PTAL_ERROR;
	}

	if (buffer[0]!='0') {
		PTAL_LOG_ERROR("ptalHpjdSgwParse(chan=0x%8.8X): "
			"first character=%d, expected='0'!\n",
			chan,buffer[0]);
		return PTAL_ERROR;
	}

	return (unsigned)buffer[1];
}

#define STATE_BEFORE_NUM	0
#define STATE_IN_NUM		1
#define STATE_AFTER_NUM		2

int ptalHpjdGgwParse(ptalChannel_t chan,char *buffer,int lenBuffer) {
	struct timeval timeout;
	unsigned char c;
	int i=0,r;
	int accum=0,state=STATE_BEFORE_NUM;

	timeout.tv_sec=1;
	timeout.tv_usec=0;

	if (!buffer) lenBuffer=0;
	while (42) {
		r=ptalChannelReadTimeout(chan,(char *)&c,1,&timeout,&timeout);
		if (r!=1) {
			PTAL_LOG_ERROR("ptalHpjdGgwParse(chan=0x%8.8X): "
				"ptalChannelReadTimeout returns %d!\n",
				chan,r);
			return PTAL_ERROR;
		}

		switch (state) {
		   case STATE_BEFORE_NUM:
			if (c<=' ') break;
		   case STATE_IN_NUM:
			if (c<'0' || c>'9') {
				state=STATE_AFTER_NUM;
				break;
			}
			state=STATE_IN_NUM;
			accum=(accum*10)+(c-'0');
			break;

		   case STATE_AFTER_NUM:
			break;
		}

		if (i<lenBuffer) buffer[i++]=c;
		if (c=='\n') break;
	}

	if (i<lenBuffer) {
		buffer[i++]=0;
	} else if (lenBuffer>0) {
		PTAL_LOG_ERROR("ptalHpjdGgwParse(chan=0x%8.8X): "
			"read %d bytes, but lenBuffer=%d!\n",
			chan,i,lenBuffer);
		buffer[lenBuffer-1]=0;
	}

	PTAL_LOG_DEBUG("ptalHpjdGgwParse(chan=0x%8.8X) returns %d, "
		"buffer=<%s>.\n",
		chan,accum,lenBuffer>0?buffer:"");
	return accum;
}

#define LEN_BUFFER 100

/* TODO: Implement socket->service lookup. */
int ptalHpjdLookup(ptalHpjdChannel_t chan,char *serviceName,int socketID) {
	char buffer[LEN_BUFFER];
	int r,r250;

	r=snprintf(buffer,LEN_BUFFER,"serv %s\n",serviceName);
	if (r>=LEN_BUFFER) {
		PTAL_LOG_ERROR("ptalHpjdLookup(chan=0x%8.8X): "
			"snprintf returned %d, expected<%d (serv)!\n",
			chan,r,LEN_BUFFER);
		return PTAL_ERROR;
	}
	ptalChannelWrite(&chan->base,buffer,strlen(buffer));

	r=ptalHpjdGgwParse(&chan->base,buffer,LEN_BUFFER);
	if (r!=250) {
		PTAL_LOG_ERROR("ptalHpjdLookup(chan=0x%8.8X): "
			"GGW returned %d, expected=200 (open)!\n",
			chan,r);
		return PTAL_ERROR;
	}

	r=sscanf(buffer," %d %d",&r250,&socketID);
	if (r!=2 || r250!=250) {
		PTAL_LOG_ERROR("ptalHpjdLookup(chan=0x%8.8X): "
			"sscanf returned %d, first field=%d!\n",r,r250);
		return PTAL_ERROR;
	}

	return socketID;
}

int ptalHpjdChannelOpen(ptalChannel_t _chan) {
	ptalHpjdChannel_t chan=(ptalHpjdChannel_t)_chan;
	int tcpPort,*s=&chan->base.fd,socketID;
	int retries=chan->base.retryCount,retcode=PTAL_OK,r;
	char buffer[LEN_BUFFER];

	/* Figure out what TCP port to connect to. */
	switch (chan->base.serviceType) {
	   case PTAL_STYPE_PRINT:
		tcpPort=HPJD_TCPPORT_PRINT;
		break;

	   case PTAL_STYPE_SCAN:
		tcpPort=HPJD_TCPPORT_SCAN;
		break;

	   case PTAL_STYPE_GENERIC:
		if (chan->base.socketID<0) {
			tcpPort=-chan->base.socketID;
			break;
		}
		tcpPort=HPJD_TCPPORT_GENERIC;
		break;

	   default:
		PTAL_LOG_ERROR("ptalHpjdChannelOpen(chan=0x%8.8X): "
			"invalid serviceType=%d!\n",
			chan,chan->base.serviceType);
		goto abort;
	}
	tcpPort+=((ptalHpjdDevice_t)(chan->base.dev))->port-1;
	chan->saddr.sin_port=htons(tcpPort);

    while (42) {
	/* Create socket. */
	*s=socket(AF_INET,SOCK_STREAM,0);
	if (*s<0) {
		PTAL_LOG_ERROR("ptalHpjdChannelOpen(chan=0x%8.8X): "
			"error creating socket!\n",chan);
		*s=PTAL_NO_FD;
		goto abort;
	}

	/* Try connecting to the JetDirect. */
	if (connect(*s,(struct sockaddr *)(&chan->saddr),
	     sizeof(chan->saddr))<0) {
		PTAL_LOG_ERROR("ptalHpjdChannelOpen(chan=0x%8.8X): "
			"error connecting socket!\n",chan);
		goto abort;
	}

	/* Post-processing once we're connected. */
	switch (chan->base.serviceType) {
	   case PTAL_STYPE_PRINT:
		goto done;

	   case PTAL_STYPE_SCAN:
		r=ptalHpjdSgwParse(&chan->base);
		switch (r) {
		   case '0':
			PTAL_LOG_DEBUG("ptalHpjdChannelOpen(chan=0x%8.8X): "
				"scan connection successful ('00').\n",chan);
			goto done;

		   case '1':
			PTAL_LOG_WARN("ptalHpjdChannelOpen(chan=0x%8.8X): "
				"scan service busy ('01')!\n",chan);
			goto retry;

		   case '2':
			if (retries>0) goto short_retry;
			PTAL_LOG_ERROR("ptalHpjdChannelOpen(chan=0x%8.8X): "
				"scan service unavailable ('02')!\n",chan);
			break;

		   case PTAL_ERROR:
			break;

		   default:
			PTAL_LOG_ERROR("ptalHpjdChannelOpen(chan=0x%8.8X): "
				"ptalHpjdSgwParse returned unrecognized "
				"character=%d!\n",chan,r);
			break;
		}
		goto abort;

	   case PTAL_STYPE_GENERIC:
		if (chan->base.socketID<0) goto done;

		r=ptalHpjdGgwParse(&chan->base,buffer,LEN_BUFFER);
		if (r!=220) {
			PTAL_LOG_ERROR("ptalHpjdChannelOpen(chan=0x%8.8X): "
				"GGW returned %d, expected=220 (ready)!\n",
				chan,r);
			goto abort;
		}

		if (chan->base.serviceName && strlen(chan->base.serviceName)) {
			socketID=ptalHpjdLookup(chan,chan->base.serviceName,0);
			if (socketID==PTAL_ERROR) goto noServiceName;

		} else {
noServiceName:
			socketID=chan->base.socketID;
		}

		r=snprintf(buffer,LEN_BUFFER,"open %d\n",socketID);
		if (r>=LEN_BUFFER) {
			PTAL_LOG_ERROR("ptalHpjdChannelOpen(chan=0x%8.8X): "
				"snprintf returned %d, expected<%d (open)!\n",
				chan,r,LEN_BUFFER);
			goto abort;
		}
		ptalChannelWrite(&chan->base,buffer,strlen(buffer));

		r=ptalHpjdGgwParse(&chan->base,buffer,LEN_BUFFER);
		if (r!=200) {
			if (retries>0) goto short_retry;
			PTAL_LOG_ERROR("ptalHpjdChannelOpen(chan=0x%8.8X): "
				"GGW returned %d, expected=200 (open)!\n",
				chan,r);
			goto abort;
		}

		r=snprintf(buffer,LEN_BUFFER,"data\n");
		if (r>=LEN_BUFFER) {
			PTAL_LOG_ERROR("ptalHpjdChannelOpen(chan=0x%8.8X): "
				"snprintf returned %d, expected<%d (data)!\n",
				chan,r,LEN_BUFFER);
			goto abort;
		}
		ptalChannelWrite(&chan->base,buffer,strlen(buffer));

		r=ptalHpjdGgwParse(&chan->base,buffer,LEN_BUFFER);
		if (r!=200) {
			PTAL_LOG_ERROR("ptalHpjdChannelOpen(chan=0x%8.8X): "
				"GGW returned %d, expected=200 (data)!\n",
				chan,r);
			goto abort;
		}

		goto done;

	   default:
		PTAL_LOG_ERROR("ptalHpjdChannelOpen(chan=0x%8.8X): "
			"invalid serviceType=%d #2!\n",
			chan,chan->base.serviceType);
		goto abort;
	}

abort:
	retcode=PTAL_ERROR;
short_retry:
#ifdef PTAL_SHORT_RETRY_COUNT
	if (retries>PTAL_SHORT_RETRY_COUNT) retries=PTAL_SHORT_RETRY_COUNT;
#endif
retry:
	if (*s!=PTAL_NO_FD) close(*s);
	*s=PTAL_NO_FD;

	if (retcode==PTAL_ERROR || retries<=0) break;
	retries--;
	sleep(chan->base.retryDelay);
    }

done:
	return retcode;
}


int ptalHpjdPmlOpen(ptalDevice_t dev) {
	return PTAL_OK;
}

int ptalHpjdPmlClose(ptalDevice_t dev) {
	return PTAL_OK;
}

#ifdef HAVE_SNMP

oid ptalHpjdSnmpOidPml[]=     {1,3,6,1,4,1,11,2,3,9,4,2};
/* oid ptalHpjdSnmpOidNumPorts[]={1,3,6,1,4,1,11,2,4,3,13,1,0}; */

int ptalHpjdPmlToSnmpOid(ptalPmlObject_t obj,oid *snmpOid) {
	unsigned char *pmlOid=(unsigned char *)obj->oid;
	int len=0,max=sizeof(ptalHpjdSnmpOidPml)/sizeof(oid);

	if (*pmlOid==255) {
		pmlOid++;

	} else while (len<max) {
		snmpOid[len]=ptalHpjdSnmpOidPml[len];
		len++;
	}

	while (len<MAX_OID_LEN) {
		if ((snmpOid[len++]=*(pmlOid++))==0) break;
	}

	return len;
}

int ptalHpjdPmlFromSnmpOid(ptalPmlObject_t obj,oid *snmpOid,int snmpOidLen) {
	char pmlOid[PTAL_PML_MAX_OID_LEN+1];
	int len=sizeof(ptalHpjdSnmpOidPml)/sizeof(oid);

	if (snmpOidLen<=len ||
	    memcmp(ptalHpjdSnmpOidPml,snmpOid,len)) {
		pmlOid[0]=255;
		len=1;
	} else {
		snmpOid+=len;
		snmpOidLen-=len;
		len=0;
	}
	while (len<PTAL_PML_MAX_OID_LEN && snmpOidLen) {
		if ((pmlOid[len]=*(snmpOid++))==0) break;
		len++;
		snmpOidLen--;
	}
	return ptalPmlSetID(obj,pmlOid);
}

int ptalHpjdSetSnmpStatus(ptalPmlObject_t obj,int status) {
	switch (status) {
	   case SNMP_ERR_NOERROR:
		status=PTAL_PML_OK;
		break;
	   case SNMP_ERR_TOOBIG:
		status=PTAL_PML_ERROR_BUFFER_OVERFLOW;
		break;
	   case SNMP_ERR_NOSUCHNAME:
		status=PTAL_PML_ERROR_UNKNOWN_OBJECT_IDENTIFIER;
		break;
	   case SNMP_ERR_BADVALUE:
		status=PTAL_PML_ERROR_INVALID_OR_UNSUPPORTED_VALUE;
		break;
	   case SNMP_ERR_READONLY:
		status=PTAL_PML_ERROR_OBJECT_DOES_NOT_SUPPORT_REQUESTED_ACTION;
		break;
	   default:
		status=PTAL_PML_ERROR_COMMAND_EXECUTION_ERROR;
		break;
	}

	return ptalPmlSetStatus(obj,status);
}

#if 0
struct variable_list {
    struct variable_list *next_variable;    /* NULL for last variable */
    oid     *name;  /* Object identifier of variable */
    size_t  name_length;    /* number of subid's in name */
    u_char  type;   /* ASN type of variable */
    union { /* value of variable */
        long    *integer;
        u_char  *string;
        oid     *objid;
        u_char  *bitstring;
        struct counter64 *counter64;
#ifdef OPAQUE_SPECIAL_TYPES
        float   *floatVal;
        double  *doubleVal;
/*      t_union *unionVal; */
#endif /* OPAQUE_SPECIAL_TYPES */
    } val;
    size_t          val_len;
    oid name_loc[MAX_OID_LEN];  /* 90 percentile < 24. */
    u_char buf[40];             /* 90 percentile < 40. */
};
#endif

#endif

int ptalHpjdPmlGet(ptalPmlObject_t obj,ptalPmlObject_t next) {
#ifndef HAVE_SNMP
	return PTAL_ERROR;
#else
	int r=PTAL_ERROR,rr,type,len;
	int msg=next?SNMP_MSG_GETNEXT:SNMP_MSG_GET;
	struct snmp_pdu *response=0,*pdu;
	struct variable_list *v;
	oid name[MAX_OID_LEN];
	char buffer[PTAL_PML_MAX_VALUE_LEN+1],*value=buffer;
	size_t name_length;

	PTAL_LOG_DEBUG("ptalHpjdPmlGet(obj=0x%8.8X,next=0x%8.8X)\n",
		obj,next);

	if (!(pdu=snmp_pdu_create(msg))) {
		PTAL_LOG_ERROR("ptalHpjdPmlGet(obj=0x%8.8X): "
			"can't allocate pdu!\n");
		goto abort;
	}
	name_length=ptalHpjdPmlToSnmpOid(obj,name);
	snmp_add_null_var(pdu,name,name_length);

	if ((rr=snmp_synch_response(((ptalHpjdDevice_t)obj->dev)->snmpSession,
	     pdu,&response))!=STAT_SUCCESS) {
		snmp_error(((ptalHpjdDevice_t)obj->dev)->snmpSession,
			0,0,&value);
		PTAL_LOG_ERROR("ptalHpjdPmlGet(obj=0x%8.8X,next=0x%8.8X): "
			"snmp_synch_response returns %d (%s)!\n",
			obj,next,rr,(int)value);
		free(value);
		goto abort;
	}

	if (ptalHpjdSetSnmpStatus(obj,response->errstat)!=PTAL_PML_OK) {
		PTAL_LOG_DEBUG("ptalHpjdPmlGet(obj=0x%8.8X,next=0x%8.8X): "
			"errstat=%d (%s)!\n",
			obj,next,response->errstat,
			snmp_errstring(response->errstat));
		goto abort;
	}

	v=response->variables;

	if (next) {
		obj=next;
		ptalHpjdPmlFromSnmpOid(obj,v->name,v->name_length);
	}

	switch(v->type) {
#if 0
	   case ASN_OBJECT_ID:
		type=PTAL_PML_TYPE_OBJECT_IDENTIFIER;
		/* TODO */

		break;
#endif

	   case ASN_INTEGER:
#ifdef OPAQUE_SPECIAL_TYPES
	   case ASN_UINTEGER:
#endif
		type=PTAL_PML_TYPE_SIGNED_INTEGER;
		ptalPmlSetIntegerValue(obj,type,*v->val.integer);
		break;

	   case ASN_OCTET_STR:
		type=PTAL_PML_TYPE_BINARY;
		value=v->val.string;
		if (!value) value="";
		len=v->val_len;
		if (ptalPmlValueIsValidString(value,len)) {
			type=PTAL_PML_TYPE_STRING;
		}
		ptalPmlSetValue(obj,type,value,len);
		break;

	   case ASN_NULL:
		type=PTAL_PML_TYPE_NULL_VALUE;
		ptalPmlSetValue(obj,type,0,0);
		break;

#ifdef OPAQUE_SPECIAL_TYPES
	   case ASN_OPAQUE:
	   case ASN_TIMETICKS:
	   case ASN_GAUGE:
	   case ASN_COUNTER:
	   case ASN_IPADDRESS:
	   case ASN_COUNTER64:
	   case ASN_OPAQUE_U64:
	   case ASN_OPAQUE_I64:
	   case ASN_OPAQUE_COUNTER64:
	   case ASN_OPAQUE_FLOAT:
	   case ASN_OPAQUE_DOUBLE:
#endif
	   default:
		PTAL_LOG_ERROR("ptalHpjdPmlGet(obj=0x%8.8X,"
			"next=0x%8.8X): unsupported type=%d!\n",
			obj,next,v->type);
		goto abort;
	}

	r=PTAL_OK;

abort:
	if (response) {
		snmp_free_pdu(response);
		response=0;
	}

	PTAL_LOG_DEBUG("ptalHpjdPmlGet(obj=0x%8.8X,next=0x%8.8X) "
		"returns %d.\n",obj,next,r);
	return r;
#endif
}

int ptalHpjdPmlSet(ptalPmlObject_t obj) {
#ifndef HAVE_SNMP
	return PTAL_ERROR;
#else
	int r=PTAL_ERROR,rr,type,intValue,len;
	struct snmp_pdu *response=0,*pdu;
	oid name[MAX_OID_LEN];
	char stringValue[PTAL_PML_MAX_VALUE_LEN+1],*pValue;
	size_t name_length;

	PTAL_LOG_DEBUG("ptalHpjdPmlSet(obj=0x%8.8X)\n",obj);

	if (!(pdu=snmp_pdu_create(SNMP_MSG_SET))) {
		PTAL_LOG_ERROR("ptalHpjdPmlSet(obj=0x%8.8X): "
			"can't allocate pdu!\n",obj);
		goto abort;
	}
	name_length=ptalHpjdPmlToSnmpOid(obj,name);

	if (ptalPmlGetIntegerValue(obj,&type,&intValue)!=PTAL_ERROR) {
		type=ASN_INTEGER;
		pValue=(char *)(&intValue);
		len=sizeof(int);
	} else {
		len=ptalPmlGetValue(obj,&type,stringValue,
			PTAL_PML_MAX_VALUE_LEN);
	   	type=ASN_OCTET_STR;
		pValue=stringValue;
	}
	snmp_pdu_add_variable(pdu,name,name_length,type,pValue,len);

	if ((rr=snmp_synch_response(((ptalHpjdDevice_t)obj->dev)->snmpSession,
	     pdu,&response))!=STAT_SUCCESS) {
		snmp_error(((ptalHpjdDevice_t)obj->dev)->snmpSession,
			0,0,&pValue);
		PTAL_LOG_ERROR("ptalHpjdPmlSet(obj=0x%8.8X: "
			"snmp_synch_response returns %d (%s)!\n",
			obj,rr,pValue);
		free(pValue);

	} else if (ptalHpjdSetSnmpStatus(obj,response->errstat)!=PTAL_PML_OK) {
		PTAL_LOG_DEBUG("ptalHpjdPmlSet(obj=0x%8.8X): "
			"errstat=%d (%s)!\n",obj,response->errstat,
			snmp_errstring(response->errstat));

	} else {
		r=PTAL_OK;
	}

abort:
	if (response) {
		snmp_free_pdu(response);
		response=0;
	}

	PTAL_LOG_DEBUG("ptalHpjdPmlSet(obj=0x%8.8X) returns %d.\n",obj,r);
	return r;
#endif
}


struct ptalProvider_s ptalHpjdProvider={
	"hpjd",
	sizeof(struct ptalHpjdDevice_s),
	sizeof(struct ptalHpjdChannel_s),

	ptalHpjdDeviceOpen,
	ptalHpjdDeviceConstructor,
	ptalHpjdDeviceDestructor,
	ptalHpjdDeviceDump,
	ptalHpjdDeviceGetDeviceIDString,

	ptalHpjdChannelConstructor,
	0,
	ptalHpjdChannelDump,
	0,
	ptalHpjdChannelOpen,
	0,
	0,
	0,
	0,

	ptalHpjdPmlOpen,
	ptalHpjdPmlClose,
	ptalHpjdPmlGet,
	ptalHpjdPmlSet,
	0,
	0
};
