#include <stdarg.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <string.h>
#include <errno.h>

#define CMD_LSPCI "lspci"

static const char *Opt_lcd = NULL;
static const char *Opt_crt = NULL;

struct i810_par {
	unsigned char *mmio;
	int mem;
	int flag;
};

#define i810_readb(mmio_base, where)                     \
        *((volatile char *) (mmio_base + where))

#define i810_readw(mmio_base, where)                     \
        *((volatile short *) (mmio_base + where))

#define i810_readl(mmio_base, where)                     \
        *((volatile int *) (mmio_base + where))

#define i810_writeb(mmio_base, where, val)                              \
	*((volatile char *) (mmio_base + where)) = (volatile char) val

#define i810_writew(mmio_base, where, val)                              \
	*((volatile short *) (mmio_base + where)) = (volatile short) val

#define i810_writel(mmio_base, where, val)                              \
	*((volatile int *) (mmio_base + where)) = (volatile int) val

#define I810_MEMOPEN		1
#define I810_MMIO		2

/* chip types */
#define I810			1
#define I830			2

/* registers */
#define PWR_CLKC		0x06014
#define HVSYNC			0x05000 
#define LCDTV_C			0x60018
#define I830_ADPA		0x61100
#define I830_DSPACTRL		0x70180
#define I830_DSPBCTRL		0x71180

#define I810STR			"8086:7121"
#define I810ESTR		"8086:7123"
#define I810_DC100STR_1		"8086:7125"
#define I810_DC100STR_2		"8086:1102"
#define I810_IGSTR		"8086:1112"
#define I810_CFCSTR		"8086:1132"
#define I830STR			"8086:3577"
#define I845STR			"8086:2562"
#define I855STR			"8086:3582"
//#define I865STR			"8086:2572"
#define MEMSTR			"Memory at"
#define NONPRSTR		"32-bit, non-prefetchable"

static void release_resource(struct i810_par *par)
{
	if (par->flag & I810_MEMOPEN)
		close(par->mem);
	if (par->flag & I810_MMIO)
		munmap(par->mmio, 512 * 1024);
}

char *i810_chip(char **buff_ptr, int *len_ptr, FILE *pci_f, int* chiptype)
{
	int i;
	char *p;

	while (getline(buff_ptr, len_ptr, pci_f) > 0) {
		i = (p = strstr(*buff_ptr, I810STR)) != NULL ||
			(p = strstr(*buff_ptr, I810ESTR)) != NULL ||
			(p = strstr(*buff_ptr, I810_DC100STR_1)) != NULL ||
			(p = strstr(*buff_ptr, I810_DC100STR_2)) != NULL ||
			(p = strstr(*buff_ptr, I810_IGSTR)) != NULL ||
			(p = strstr(*buff_ptr, I810_CFCSTR)) != NULL;
		if (i)
		{
			*chiptype = I810;
			return p;
		}

		i = (p = strstr(*buff_ptr, I830STR)) != NULL ||
		    (p = strstr(*buff_ptr, I845STR)) != NULL ||

#if defined (I865STR)
		    (p = strstr(*buff_ptr, I865STR)) != NULL ||
#endif /*defined (I865STR)*/
		    (p = strstr(*buff_ptr, I855STR)) != NULL;
		if (i)
		{
			*chiptype = I830;
			return p;
		}
	}
	return NULL;
}
	
unsigned long i810_addr(char **buff_ptr, int *len_ptr, FILE *pci_f)
{
	char *p;

	while (getline(buff_ptr, len_ptr, pci_f) > 0)
		if (strstr(*buff_ptr, NONPRSTR) != NULL) {
			p = strstr(*buff_ptr, MEMSTR);
			if (p != NULL)
				return strtoul(p+sizeof(MEMSTR), NULL, 16);
		}
	return 0;
}

int i810_usage ()
{
	fprintf(stderr, "usage: i810switch [crt on/off] [lcd on/off]\n");
	fprintf(stderr, "  crt: enables/disables the output to the CRT display\n");
	fprintf(stderr, "  lcd: enables/disables the output to the LCD\n");
	fprintf(stderr, "  no options: displays the current output status\n");
	return 1;
}

int main (int argc, char *argv[])
{
	struct i810_par par;
	unsigned long addr;
	int i, crt = -1, lcd = -1, err = 0, count = 0, chiptype, len = 0;
	FILE *pci_f;
	char *buff = NULL;
	char lspcistr[] = CMD_LSPCI " -v -d xxxx:xxxx";
	char *chip;

	putenv("PATH=/sbin:/usr/sbin:/bin:/usr/bin");

	while (--argc > 0) {
		argv++;
		if (!strcmp(argv[0], "crt")) {
			if (--argc > 0) {
				argv++;
				if (!strcmp(argv[0], "on"))
					crt = 1;
				else if (!strcmp(argv[0], "off"))
					crt = 0;
				else
					exit(i810_usage());
			} else {
				exit(i810_usage());
			}
		} else if (!strcmp(argv[0], "lcd")) {
			argv++;
			if (--argc > 0) {
				if (!strcmp(argv[0], "on"))
					lcd = 1;
				else if (!strcmp(argv[0], "off"))
					lcd = 0;
				else
					exit(i810_usage());
			} else {
				exit(i810_usage());
			}
		} else {
			exit(i810_usage());
		}
	}

	pci_f = popen(CMD_LSPCI " -n", "r");
	if (!pci_f) {
		fprintf(stderr, "Something is wrong with lspci.\n");
		exit(1);
	}
	chip = i810_chip(&buff, &len, pci_f, &chiptype);
	if (chip == NULL) {
		fprintf(stderr, "PCI id of i810 is not recognized.\n");
		exit(1);
	}
	pclose(pci_f);

	{
		char *p = strstr(lspcistr, "xxxx:xxxx");
		if (p == 0) {
			fprintf(stderr, "CMD_LSPCI is wrong.\n");
			exit(1);
		}
		memcpy(p, chip, 9);
	}

	pci_f = popen(lspcistr, "r");
	if (!pci_f) {
		fprintf(stderr, "Something is wrong with lspci.\n");
		exit(1);
	}
	addr = i810_addr(&buff, &len, pci_f);
	if (addr == 0) {
		fprintf(stderr, "Something is wrong with lspci.\n");
		exit(1);
	}
	pclose(pci_f);

	memset(&par, 0, sizeof(struct i810_par));
	par.mem = open("/dev/mem", (crt+lcd == -2) ? O_RDONLY : O_RDWR);
	if (par.mem == -1) {
    	  	i = errno;
		perror("/dev/mem");
		if (i == EACCES && geteuid() != 0)
			fprintf(stderr, "Must have access to `/dev/mem'.\n");
		release_resource(&par);
		exit(1);
	}
	par.flag |= I810_MEMOPEN;

	par.mmio = mmap(NULL, 512 * 1024, (crt+lcd == -2) ? PROT_READ : PROT_WRITE | PROT_READ,
			MAP_SHARED, par.mem, addr);
	if (!par.mmio) {
		release_resource(&par);
		exit(1);
	}

	par.flag |= I810_MMIO;

	if (crt+lcd == -2) {
		if (chiptype == I810) {
			printf("CRT: ");
			if (i810_readl(par.mmio, PWR_CLKC) & 1)
				printf("on\n");
			else
				printf("off\n");

			printf("LCD: ");
			if (i810_readl(par.mmio, LCDTV_C) & 0xc00)
				printf("off\n");
			else
				printf("on\n");
		}
		else if (chiptype == I830) {
			printf("CRT: ");
			if (i810_readl(par.mmio, I830_ADPA) & 0x80000000)
				printf("on\n");
			else
				printf("off\n");

			printf("LCD: on\n");
		}
	}
	
	if (crt != -1) {
		if (chiptype == I810) {
			int pwr_clkc = i810_readl(par.mmio, PWR_CLKC);
			int hvsync = i810_readl(par.mmio, HVSYNC);

			if (crt == 0) {
				printf("Disabling CRT display...\n");
				pwr_clkc &= ~1;
				hvsync = 0xa0000;
			} else {
				printf("Enabling CRT display...\n");
				pwr_clkc |= 1;
				hvsync = 0;
			}
			i810_writel(par.mmio, PWR_CLKC, pwr_clkc);
			i810_writel(par.mmio, HVSYNC, hvsync);
		}
		else if (chiptype == I830)
		{
			int i830_adpa = i810_readl(par.mmio, I830_ADPA);
			int dspactrl = i810_readl(par.mmio, I830_DSPACTRL);
			int dspbctrl = i810_readl(par.mmio, I830_DSPBCTRL);
			int pipe = 0;

			/* Find active display plane, and use the same pipe it's using */
			if (dspactrl & (1<<31))
				pipe = (dspactrl & (1<<24)) >> 24;
			else if (dspbctrl & (1<<31))
				pipe = (dspbctrl & (1<<24)) >> 24;
			else
				fprintf(stderr, "Warning: No active display plane detected.  Trying pipe A.\n");
			if (crt == 0) {
				printf("Disabling CRT display...\n");
				i830_adpa = (i830_adpa & ~0x80000000 & ~(pipe<<30)) | 0xc00;
			} else {
				printf("Enabling CRT display...\n");
				i830_adpa = (i830_adpa | 0x80000000 | (pipe<<30)) & ~0xc00;
			}
			i810_writel(par.mmio, I830_ADPA, i830_adpa);
		}
	}
	
	if (lcd != -1) {
		if (chiptype == I810) {
			int lcdtv_c = i810_readl(par.mmio, LCDTV_C);
			if (lcd == 0) {
				printf("Disabling LCD...\n");
				lcdtv_c |= 0xc00;
			} else {
				printf("Enabling LCD...\n");
				lcdtv_c &= ~(0xc00); 
			}
			i810_writel(par.mmio, LCDTV_C, lcdtv_c);
		}
		else if (chiptype == I830) {
			fprintf(stderr, "No LCD support for this chipset.\n");
		}
	}

	release_resource(&par);
	exit(0);
}
