/***************************************************************************
 *   Copyright (C) 2004-2006 by Jim Campbell                               *
 *   ifpgui@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "ifpguiview.h"
#include "radiofreqdlg.h"
#include "playlist/playlistloader.h"

#include <qpainter.h>
#include <qlayout.h>
#include <qapplication.h>
#include <qmainwindow.h>

#include <qfileinfo.h>
#include <qdir.h>
#include <qmessagebox.h>
#include <qfiledialog.h>
#ifdef BSD4
   #include "libgen.h"
#endif

#include "progressdlg.h"
#include "iointerface.h"
#include "gui_user.h"

ifpGUIView::ifpGUIView( QMainWindow *parent )
      : QWidget((QWidget*) parent )
   {
   int w_width, w_height, ifp_width, dir_width;
   QRect rect;
   QString settings_path;

   parent_window = parent;

   settings_path = QDir::homeDirPath() + "/.ifpgui_rc";
   guiSettings = new Settings(settings_path);
   guiSettings->getWindowSize(&w_width, &w_height);

   //check widths for sane values
   rect = QApplication::desktop()->geometry();
   if ((w_width > 50) && (w_width <= rect.width()))
      parent->resize(w_width, w_height);

   // setup our layout manager to automatically add our widgets
   top_layout = new QBoxLayout(this, QBoxLayout::LeftToRight);

   top_layout->setAutoAdd( false );

   ifptoolbar = new ifpToolsWidget(parent, "ifptoolbar");

   split = new QSplitter(this);
   top_layout->addWidget(split);
   dirview = new fileManagerView(split, tr("Local File System").utf8().data(), TRUE, IO_INTF_TYPE_LOCAL);
   dirview->setMinimumSize(100, 100);
   dirview->setMaximumSize(2000, 2000);
   dirview->setDir(guiSettings->getDefaultFolder());
   dirview->show();

   ifpview = new fileManagerView(split, tr("iRiver File System").utf8().data(), FALSE, IO_INTF_TYPE_IFP);
   ifpview->setMinimumSize(100, 100);
   ifpview->setMaximumSize(2000, 2000);
   ifpview->show();

   connect(ifptoolbar, SIGNAL(UploadPlaylistClicked()),
           this, SLOT(uploadPlaylist()));
   connect(ifptoolbar, SIGNAL(UploadClicked()),
           this, SLOT(uploadSelectedFile()));
   connect(ifptoolbar, SIGNAL(DownloadClicked()),
           this, SLOT(downloadSelectedFile()));
   connect(ifptoolbar, SIGNAL(FormatClicked()),
           this, SLOT(formatIfpDevice()));
   connect(ifptoolbar, SIGNAL(UpdateFirwareClicked()),
           this, SLOT(UpdateFirware()));
   connect(ifptoolbar, SIGNAL(RadioFreqClicked()),
           this, SLOT(setRadioFrequency()));


   //check widths for sane values
   QValueList<int> list;
   dir_width = guiSettings->getDirViewWidth();
   if ((dir_width > 0) && (dir_width < parent->width() - 50))
      list.append(dir_width);
   else
      list.append(parent->width() / 2);

   ifp_width = guiSettings->getIfpViewWidth();
   if ((ifp_width > 0) && (ifp_width < parent->width() - 50))
      list.append(ifp_width);
   else
      list.append(parent->width() / 2);

   split->setSizes(list);

   // update the iRiver status
   updateStatus();
   }

ifpGUIView::~ifpGUIView()
   {
   int w_width, w_height, cnt;
   QString settings_path, destDir;

   w_width  = parent_window->width();
   w_height = parent_window->height();
   guiSettings->setWindowSize(w_width, w_height);
   destDir = dirview->currentDirectory();
   guiSettings->setDefaultFolder(destDir);

   cnt = 0;
   QValueList<int> list = split->sizes();
   QValueList<int>::Iterator it = list.begin();
   while( it != list.end() )
      {
      switch(cnt)
         {
         case 0:
            guiSettings->setDirViewWidth(*it);
            break;
         case 1:
            guiSettings->setIfpViewWidth(*it);
            break;
         }
      ++cnt;
      ++it;
      }

   settings_path = QDir::homeDirPath() + "/.ifpgui_rc";
   guiSettings->saveSettings(settings_path);

   release_ifp_device();
   }

int ifpGUIView::updateStatus(void)
   {
   int ret;

   if (isProcessing())
      return(0);

   ret = -1;
   if (ifp_dh != NULL)
      {
      ifptoolbar->sCapacity(getCapacity(), getFreeSpace());
      ret = getBatteryStatus();
      if (ret < 0)
         {
         release_ifp_device();
         ifpview->setDir("/");
         ifpview->fmRefreshFileItem();
         }
      }
   return(ret);
   }

void ifpGUIView::showToolbar()
   {
   ifptoolbar->show();
   }

void ifpGUIView::hideToolbar()
   {
   ifptoolbar->hide();
   }

void ifpGUIView::showToolbarText()
   {
   ifptoolbar->sShowToolbarText();
   }

void ifpGUIView::hideToolbarText()
   {
   ifptoolbar->sHideToolbarText();
   }

void ifpGUIView::slotSetTitle( const QString& title )
   {
   emit signalChangeCaption( title );
   }

void ifpGUIView::uploadPlaylist()
   {
   PlaylistLoader loader;
   QFile playlistfile;
   QFileInfo finfo;
   PlayList::iterator it;
   QString playlistname, destDir, Message, err, playlistDir;
   int ret, playlistcount, playlistsize, freespace;
   ifp_progress_info pginfo;
   QPtrList<FileInfos> file_info_list;

   if (ifp_dh == NULL)
      {
      showInfoMsgBox(tr("iFP GUI Upload Playlist - Error"), tr(iFPNotConnectedMsg));
      return;
      }

   playlistDir = guiSettings->getPlaylistFolder();
   playlistname = "";
   playlistname = QFileDialog::getOpenFileName(
                    playlistDir,
                    tr("PlayList (*.m3u)"),
                    this,
                    tr("Open Playlist Dialog"),
                    tr("Choose a playlist") );
   if (playlistname == "")
      return;

   finfo.setFile(playlistname);
   playlistDir = finfo.dirPath(true);
   guiSettings->setPlaylistFolder(playlistDir);
   playlistfile.setName(playlistname);
   loader.load(playlistfile, PlaylistLoader::M3U);
   destDir = ifpview->currentDirectory();

   playlistcount = loader.numFilesInPlaylist();
   playlistsize  = loader.sizeOfPlaylist();
   freespace     = getFreeSpace();

   if (freespace < playlistsize)
      {
      Message = tr("There is not enough free space to upload the entire playlist.");
      showInfoMsgBox(tr("iFP GUI Upload Playlist - Error"), Message);
      return;
      }

   //----------------------------------------------------------------
   // make a list of all selected files then call exec_TransferUpload
   //----------------------------------------------------------------
   for(it = loader.list.begin(); it != loader.list.end(); ++it)
      {
      finfo.setFile((*it).filename());
      pginfo.total_transfered += finfo.size();
      if (finfo.isFile())
         {
         file_info_list.append(new FileInfos(QString((*it).filename()), IO_INTF_FILE, finfo.size(), NULL));
         }
      else if (finfo.isDir())
         {
         file_info_list.append(new FileInfos(QString((*it).filename()), IO_INTF_DIR, 0, NULL));
         }
      }
   ret = dirview->fmUploadFileList(destDir, &file_info_list);

   ifpview->setDir(destDir);
   ifpview->fmRefreshFileItem();
   }

void ifpGUIView::unselectAllChildren(QListViewItem *parentItem)
   {
   QListViewItem *current;

   current = parentItem->firstChild();
   while (current != NULL)
      {
      current->setSelected(false);
      if (current->childCount() > 0)
         unselectAllChildren(current);
      current = current->nextSibling();
      }
   }

void ifpGUIView::uploadSelectedFile(void)
   {
   QString destDir;

   if (ifp_dh == NULL)
      {
      showInfoMsgBox(tr("iFP GUI Upload Selected Files - Error"), tr(iFPNotConnectedMsg));
      return;
      }

   // get the ifp viewer directory
   destDir = ifpview->currentDirectory();
   // upload selected files on local file system to destination
   dirview->fmTransferSelectedFiles(destDir);

   ifpview->setDir(destDir);
   ifpview->fmRefreshFileItem();
   }

void ifpGUIView::downloadSelectedFile(void)
   {
   QString destDir;

   if (ifp_dh == NULL)
      {
      showInfoMsgBox(tr("iFP GUI Download Selected Files - Error"), tr(iFPNotConnectedMsg));
      return;
      }

   // get the ifp viewer directory
   destDir = dirview->currentDirectory();
   // upload selected files on local file system to destination
   ifpview->fmTransferSelectedFiles(destDir);

   dirview->setDir(destDir);
   dirview->fmRefreshFileItem();
   }


void ifpGUIView::deleteSelectedFile(void)
   {
   if (ifpview->hasFocus())
      {
      if (ifp_dh == NULL)
         {
         showInfoMsgBox(tr("iFP GUI Delete Selected Files - Error"), tr(iFPNotConnectedMsg));
         return;
         }
      ifpview->fmDeleteFile();
      }
   else if (dirview->hasFocus())
      {
      dirview->fmDeleteFile();
      }
   }

int ifpGUIView::getBatteryStatus()
   {
   int ret;

   if (ifp_dh == NULL)
      return(-1);

   ret = ifpgui_ifp_battery(&ifp_dev);
   ifptoolbar->sBatteryStatus(ret);
   return(ret);
   }

QString ifpGUIView::getFirmwareRev()
   {
   char firm_str[20];
   int  firm_val;

   if (ifp_dh == NULL)
      return("");

   firm_val = ifpgui_ifp_firmware_version(&ifp_dev);
   sprintf(firm_str, "%x.%02x", firm_val/0x0100, firm_val%0x100);
   return(QString(firm_str));
   }

int ifpGUIView::getCapacity()
   {
   if (ifp_dh == NULL)
      return(-1);

   return(ifpgui_ifp_capacity(&ifp_dev));
   }

int ifpGUIView::getFreeSpace()
   {
   if (ifp_dh == NULL)
      return(-1);

   return(ifpgui_ifp_freespace(&ifp_dev));
   }

QString ifpGUIView::getifpType()
   {
   char model_str[20];

   if (ifp_dh == NULL)
      return("");

   ifpgui_ifp_model(&ifp_dev, model_str, 20);
   return(QString(model_str));
   }

void ifpGUIView::formatIfpDevice()
   {
   int mbret;
   QString str;
   ProgressWindow *pgw;

   if (ifp_dh == NULL)
      {
      showInfoMsgBox(tr("iFP GUI Format iRiver Device - Error"), tr(iFPNotConnectedMsg));
      return;
      }

   str = tr("Format IFP Device.\n\nWARNING---\n\nAnswering 'Yes' will erase all data on this device!\n\n");

   QMessageBox mb(tr("Format IFP Device?"),
                  str,
                  QMessageBox::Question,
                  QMessageBox::No  | QMessageBox::Escape,
                  QMessageBox::Yes,
                  QMessageBox::NoButton);
   mbret = mb.exec();
   if (mbret == QMessageBox::No)
      {
      return;
      }
   str = tr("Please Wait while formatting IFP device.");
   pgw = new ProgressWindow(str);
   pgw->show();
   pgw->raise();
   pgw->setFileName(str);
   pgw->setTotalProgressTarget(-1);
   pgw->setFileProgressTarget(100);
   pgw->setFileProgress(50);
   qApp->processEvents();
   pgw->show();
   pgw->raise();
   qApp->processEvents();

   ifpgui_ifp_format(&ifp_dev);
   ifp_read_file_tree();
   pgw->setFileProgress(100);
   qApp->processEvents();
   delete pgw;
   showInfoMsgBox(tr("Format Device Status"), tr("Format Done."));
   ifpview->setDir("/");
   ifpview->fmRefreshFileItem();
   }

void ifpGUIView::UpdateFirware()
   {
   QString str;
   QString msg;
   QString filename;
   QCString qcfilename;
   ifp_progress_info pginfo;

   if (ifp_dh == NULL)
      {
      showInfoMsgBox(tr("iFP GUI Update Firmware - Error"), tr(iFPNotConnectedMsg));
      return;
      }

   str = tr("Update Firmware");
   msg  = tr("Update firmware?\n\n");
   msg += tr(" Make sure your battery is full.\n");
   msg += tr(" DO NOT disconnect your USB cable until this process is finished!");

   filename = QFileDialog::getOpenFileName(
                    QDir::homeDirPath(),
                    tr("IFP Firmware Files  (*.hex; *.HEX)"),
                    this,
                    tr("Open Firmware File"),
                    tr("Choose a Firmware HEX file") );
   if (filename != "")
      {
      str = tr("Please Wait while updating firmware.");

      pginfo.pgw = new ProgressWindow(str);
      pginfo.pgw->setFileProgressTarget(0);
      pginfo.pgw->setTotalProgressTarget(0);
      pginfo.pgw->setFileProgress(0);
      pginfo.pgw->setTotalProgress(0);
      pginfo.pgw->hideCancelBtn();

      if (filename.length() > IFP_MAXPATHLEN)
         {
         showInfoMsgBox(tr("Firmware Update Status"), tr("Path to firmware file is too long."));
         delete pginfo.pgw;
         return;
         }
      qcfilename = filename.utf8();
      ifpgui_ifp_update_firmware(&ifp_dev, qcfilename, progress, NULL);
      usleep(10000000); // sleep 10 seconds
      delete pginfo.pgw;
      msg  = tr("Update Done\n\n");
      msg += tr(" DO NOT touch the device until it has turned itself off!\n\n");
      msg += tr(" Afterwards, you may have to turn the device back on.");
      showInfoMsgBox(tr("Firmware Update Status"), msg);
      }

   }

void ifpGUIView::setRadioFrequency()
   {
   RadioFreqWindow *rfdialog;

   if (ifp_dh == NULL)
      {
      showInfoMsgBox(tr("iFP GUI Tuner Settings - Error"), tr(iFPNotConnectedMsg));
      return;
      }

   rfdialog = new RadioFreqWindow();
   rfdialog->setModal(TRUE);
   rfdialog->exec();
   delete rfdialog;
   }

bool ifpGUIView::isProcessing()
{
   bool processingDirview = dirview->isProcessing();
   bool processingIfpview = ifpview->isProcessing();

   return(processingDirview || processingIfpview);
}

void ifpGUIView::create_directory()
   {
   if (ifpview->hasFocus())
      {
      ifpview->fmCreateDirectory();
      }
   else if (dirview->hasFocus())
      {
      dirview->fmCreateDirectory();
      }
   }

void ifpGUIView::rename_file()
   {
   if (ifpview->hasFocus())
      {
      ifpview->fmRenameFile();
      }
   else if (dirview->hasFocus())
      {
      dirview->fmRenameFile();
      }
   }

void ifpGUIView::delete_file()
   {
   if (ifpview->hasFocus())
      {
      ifpview->fmDeleteFile();
      }
   else if (dirview->hasFocus())
      {
      dirview->fmDeleteFile();
      }
   }

void ifpGUIView::file_properties()
   {
   }

void ifpGUIView::align_horizontal()
   {
   split->setOrientation(Qt::Horizontal);
   }

void ifpGUIView::align_vertical()
   {
   split->setOrientation(Qt::Vertical);
   }


