/*  Simple bitmap font rendering example.
 *
 *  font.cc
 *
 *  Inti Translation of the GtkGLExt bitmap font rendering example
 *  written by Naofumi Yasufuku <naofumi@users.sourceforge.net>.
 */

#include "font.h"
#include <inti/gdk/gl/drawable.h>
#include <inti/gdk/gl/font.h>
#include <inti/gdk/gl/query.h>
#include <inti/gtk/button.h>
#include <inti/gtk/gl/init.h>
#include <inti/pango/font.h>
#include <GL/gl.h>
#include <iostream>
#include <cstdlib>

namespace {

const String font_string("courier 12");

} // namespace

/*  GLDrawingArea
 */

GLDrawingArea::GLDrawingArea(Gdk::GL::Config *glconfig)
: Gtk::GL::DrawingArea(*glconfig, 640, 240)
{
	examine_attrib(*glconfig);
}

GLDrawingArea::~GLDrawingArea()
{
}

void
GLDrawingArea::on_realize()
{
	Gtk::GL::DrawingArea::on_realize();
	Gdk::GL::Drawable *gldrawable = get_gl_drawable();

	// OpenGL begin
	if (!gldrawable->gl_begin(*get_gl_context()))
		return;

	// Generate font display lists.
	font_list_base = glGenLists(128);
	Pointer<Pango::FontDescription> font_desc = new Pango::FontDescription(font_string);
	Pango::Font *font = Gdk::GL::font_use_pango_font(*font_desc, 0, 128, font_list_base);
	if (!font)
	{
		std::cout << "*** Can't load font " << font_string << std::endl;
		Main::quit();
	}

	Pointer<Pango::FontMetrics> font_metrics = font->get_metrics();
	font_height = PANGO_PIXELS(font_metrics->get_ascent() + font_metrics->get_descent());

	glClearColor(1.0, 1.0, 1.0, 1.0);
	glClearDepth(1.0);
	glViewport(0, 0, get_allocation().width(), get_allocation().height());
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0.0, get_allocation().width(), 0.0, get_allocation().height(), -1.0, 1.0);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	gldrawable->gl_end();
	// OpenGL end
}

bool
GLDrawingArea::on_configure_event(const Gdk::EventConfigure& event)
{
	Gdk::GL::Drawable *gldrawable = get_gl_drawable();

	// OpenGL begin
	if (!gldrawable->gl_begin(*get_gl_context()))
		return false;

	glViewport(0, 0, get_allocation().width(), get_allocation().height());
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0.0, get_allocation().width(), 0.0, get_allocation().height(), -1.0, 1.0);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	gldrawable->gl_end();
	// OpenGL end

	return true;
}

bool
GLDrawingArea::on_expose_event(const Gdk::EventExpose& event)
{
	Gdk::GL::Drawable *gldrawable = get_gl_drawable();

	// OpenGL begin
	if (!gldrawable->gl_begin(*get_gl_context()))
		return false;

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	// Draw some text.
	glColor3f(0.0, 0.0, 0.0);
	for (int i = 2; i >= -2; i--)
	{
		glRasterPos2f(10.0, 0.5 * get_allocation().height() + i * font_height);
		for (int j = ' '; j <= 'Z'; j++)
			glCallList(font_list_base + j);
	}

	// Show font description string.
	glColor3f(1.0, 0.0, 0.0);
	glRasterPos2f(10.0, 10.0);
	glListBase(font_list_base);
	glCallLists(font_string.length(), GL_UNSIGNED_BYTE, font_string.c_str());

	if (gldrawable->is_double_buffered())
		gldrawable->swap_buffers();
	else
		glFlush();

	gldrawable->gl_end();
	// OpenGL end

	return true;
}

void
GLDrawingArea::examine_attrib(Gdk::GL::Config& glconfig)
{
	using namespace std;

	cout << "\nOpenGL visual configurations :\n\n";
	cout << "Gdk::GL::Config::is_rgba() = " << (glconfig.is_rgba() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::is_double_buffered() = " << (glconfig.is_double_buffered() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::is_stereo() = " << (glconfig.is_stereo() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::has_alpha() = " << (glconfig.has_alpha() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::has_depth_buffer() = " << (glconfig.has_depth_buffer() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::has_stencil_buffer() = " << (glconfig.has_stencil_buffer() ? "true" : "false") << endl;
	cout << "Gdk::GL::Config::has_accum_buffer() = " << (glconfig.has_accum_buffer() ? "true" : "false") << endl;
	cout << endl;

	print_attrib(glconfig, "GDK_GL_USE_GL", GDK_GL_USE_GL, true);
	print_attrib(glconfig, "GDK_GL_BUFFER_SIZE", GDK_GL_BUFFER_SIZE, false);
	print_attrib(glconfig, "GDK_GL_LEVEL", GDK_GL_LEVEL, false);
	print_attrib(glconfig, "GDK_GL_RGBA", GDK_GL_RGBA, true);
	print_attrib(glconfig, "GDK_GL_DOUBLEBUFFER",  GDK_GL_DOUBLEBUFFER, true);
	print_attrib(glconfig, "GDK_GL_STEREO", GDK_GL_STEREO, true);
	print_attrib(glconfig, "GDK_GL_AUX_BUFFERS", GDK_GL_AUX_BUFFERS, false);
	print_attrib(glconfig, "GDK_GL_RED_SIZE", GDK_GL_RED_SIZE, false);
	print_attrib(glconfig, "GDK_GL_GREEN_SIZE", GDK_GL_GREEN_SIZE, false);
	print_attrib(glconfig, "GDK_GL_BLUE_SIZE",  GDK_GL_BLUE_SIZE, false);
	print_attrib(glconfig, "GDK_GL_ALPHA_SIZE",  GDK_GL_ALPHA_SIZE, false);
	print_attrib(glconfig, "GDK_GL_DEPTH_SIZE",  GDK_GL_DEPTH_SIZE, false);
	print_attrib(glconfig, "GDK_GL_STENCIL_SIZE", GDK_GL_STENCIL_SIZE, false);
	print_attrib(glconfig, "GDK_GL_ACCUM_RED_SIZE", GDK_GL_ACCUM_RED_SIZE, false);
	print_attrib(glconfig, "GDK_GL_ACCUM_GREEN_SIZE", GDK_GL_ACCUM_GREEN_SIZE, false);
	print_attrib(glconfig, "GDK_GL_ACCUM_BLUE_SIZE", GDK_GL_ACCUM_BLUE_SIZE, false);
	print_attrib(glconfig, "GDK_GL_ACCUM_ALPHA_SIZE", GDK_GL_ACCUM_ALPHA_SIZE, false);

	cout << endl;
}

void
GLDrawingArea::print_attrib(Gdk::GL::Config& glconfig, const char *attrib_str, int attrib, bool is_boolean)
{
	using namespace std;

	cout << attrib_str << " = ";

	int value;
	if (glconfig.get_attrib(attrib, value))
	{
		if (is_boolean)
			cout << (value ? "true" : "false") << endl;
		else
			cout << value << endl;
	}
	else
		cout << "*** Cannot get " << attrib_str << " attribute value" << endl;
}

/*  Window
 */

Window::Window(Gdk::GL::Config *glconfig)
{
	set_title("Font Example");

	// Perform the resizes immediately.
	set_resize_mode(Gtk::RESIZE_IMMEDIATE);

	// Get automatically redrawn if any of their children changed allocation.
	set_reallocate_redraws(true);

	Gtk::VBox *vbox = new Gtk::VBox;
	add(*vbox);
	vbox->show();

	// Create an OpenGL-capable drawing area.
	GLDrawingArea *drawing_area = new GLDrawingArea(glconfig);
	vbox->pack_start(*drawing_area);
	drawing_area->show();

	// Simple quit button.
	Gtk::Button *button = new Gtk::Button("Quit");
	button->sig_clicked().connect(slot(&Main::quit));
	vbox->pack_start(*button, false, false);
	button->show();
	show();
}

Window::~Window()
{
}

int main (int argc, char *argv[])
{
	using namespace Main;

	init(&argc, &argv);

	// Initialize GtkGLExt
	Gtk::GL::init(&argc, &argv);

	// Query OpenGL extension version.
	int major, minor;
	Gdk::GL::query_version(&major, &minor);
	std::cout << "\nOpenGL extension version - " << major << "." << minor << std::endl;

	// Configure OpenGL-capable visual.
	Pointer<Gdk::GL::Config> glconfig = Gdk::GL::Config::create(Gdk::GL::MODE_RGB | Gdk::GL::MODE_DOUBLE);
	if (!glconfig)
	{
		std::cout << "*** Cannot find the double-buffered visual." << std::endl;
		std::cout << "*** Trying single-buffered visual." << std::endl;

		// Try single-buffered visual
		glconfig = Gdk::GL::Config::create(Gdk::GL::MODE_RGB);
		if (!glconfig)
		{
			std::cout << "*** No appropriate OpenGL-capable visual found." << std::endl;
			return 1;
		}
	}

	// Top-level window.
	Window window(glconfig);
	window.sig_destroy().connect(slot(&Inti::Main::quit));

	run();
	return 0;
}

